// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_NET_NETWORK_DIAGNOSTICS_ARC_DNS_RESOLUTION_ROUTINE_H_
#define CHROME_BROWSER_CHROMEOS_NET_NETWORK_DIAGNOSTICS_ARC_DNS_RESOLUTION_ROUTINE_H_

#include <string>
#include <vector>

#include "base/memory/weak_ptr.h"
#include "chrome/browser/chromeos/net/network_diagnostics/network_diagnostics_routine.h"
#include "components/arc/mojom/net.mojom.h"
#include "components/arc/session/arc_bridge_service.h"

namespace chromeos {
namespace network_diagnostics {

// Performs DNS queries from within ARC to resolve a hardcoded set of hostnames
// and returns the result.
class ArcDnsResolutionRoutine : public NetworkDiagnosticsRoutine {
 public:
  using RunArcDnsResolutionCallback =
      mojom::NetworkDiagnosticsRoutines::RunArcDnsResolutionCallback;

  ArcDnsResolutionRoutine();
  ArcDnsResolutionRoutine(const ArcDnsResolutionRoutine&) = delete;
  ArcDnsResolutionRoutine& operator=(const ArcDnsResolutionRoutine&) = delete;
  ~ArcDnsResolutionRoutine() override;

  // NetworkDiagnosticsRoutine:
  mojom::RoutineType Type() override;
  void Run() override;
  void AnalyzeResultsAndExecuteCallback() override;

  void set_net_instance_for_testing(arc::mojom::NetInstance* net_instance) {
    net_instance_ = net_instance;
  }

 private:
  // Attempts the next DNS query.
  void AttemptNextQuery();

  // Gets NetInstance service if it is not already set for testing
  // purposes as |net_instance_|.
  arc::mojom::NetInstance* GetNetInstance();

  // Processes the |result| of an DnsResolutionTest API request.
  void OnQueryComplete(arc::mojom::ArcDnsResolutionTestResultPtr result);

  std::vector<std::string> hostnames_to_resolve_dns_;
  std::vector<mojom::ArcDnsResolutionProblem> problems_;
  bool successfully_resolved_hostnames_ = true;
  bool failed_to_get_arc_service_manager_ = false;
  bool failed_to_get_net_instance_service_for_dns_resolution_test_ = false;
  arc::mojom::NetInstance* net_instance_ = nullptr;
  int64_t max_latency_ = 0;
  base::WeakPtrFactory<ArcDnsResolutionRoutine> weak_ptr_factory_{this};
};

}  // namespace network_diagnostics
}  // namespace chromeos

#endif  //  CHROME_BROWSER_CHROMEOS_NET_NETWORK_DIAGNOSTICS_ARC_DNS_RESOLUTION_ROUTINE_H_
