// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <vector>

#include "base/base_switches.h"
#include "base/files/file_path.h"
#include "base/files/file_util.h"
#include "base/json/json_writer.h"
#include "base/test/scoped_feature_list.h"
#include "base/values.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/policy/policy_test_utils.h"
#include "chrome/browser/search/search.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/tabs/tab_strip_model.h"
#include "chrome/browser/ui/ui_features.h"
#include "chrome/browser/ui/webui/whats_new/whats_new_ui.h"
#include "chrome/browser/ui/webui/whats_new/whats_new_util.h"
#include "chrome/common/chrome_constants.h"
#include "chrome/common/chrome_switches.h"
#include "chrome/common/chrome_version.h"
#include "chrome/common/pref_names.h"
#include "chrome/common/webui_url_constants.h"
#include "chrome/test/base/in_process_browser_test.h"
#include "chrome/test/base/testing_profile.h"
#include "components/policy/core/common/policy_map.h"
#include "components/policy/core/common/policy_types.h"
#include "components/policy/policy_constants.h"
#include "components/prefs/pref_service.h"
#include "content/public/test/browser_test.h"
#include "net/base/url_util.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "url/gurl.h"

#if !defined(OS_CHROMEOS)
#include "chrome/browser/ui/webui/welcome/helpers.h"
#endif

namespace policy {

// Base class for testing the policy.
class PromotionalTabsEnabledPolicyTest
    : public PolicyTest,
      public testing::WithParamInterface<PolicyTest::BooleanPolicy> {
 public:
  PromotionalTabsEnabledPolicyTest(const PromotionalTabsEnabledPolicyTest&) =
      delete;
  PromotionalTabsEnabledPolicyTest& operator=(
      const PromotionalTabsEnabledPolicyTest&) = delete;

 protected:
  static std::string GetWhatsNewAutoURL() {
    GURL url(chrome::kChromeUIWhatsNewURL);
    return net::AppendQueryParameter(url, "auto", "true")
        .possibly_invalid_spec();
  }

  PromotionalTabsEnabledPolicyTest() {
    const std::vector<base::Feature> kEnabledFeatures = {
      features::kChromeWhatsNewUI,
#if !defined(OS_CHROMEOS)
      welcome::kForceEnabled,
#endif
    };
    whats_new::g_force_enable_for_tests = true;
    scoped_feature_list_.InitWithFeatures(kEnabledFeatures, {});
  }
  ~PromotionalTabsEnabledPolicyTest() override = default;

  void SetUp() override {
    // Ordinarily, browser tests include chrome://blank on the command line to
    // suppress any onboarding or promotional tabs. This test, on the other
    // hand, must evaluate startup with nothing on the command line so that a
    // default launch takes place.
    set_open_about_blank_on_browser_launch(false);
    PolicyTest::SetUp();
  }

  void CreatedBrowserMainParts(
      content::BrowserMainParts* browser_main_parts) override {
    // Set policies before the browser starts up.
    PolicyMap policies;

    // Suppress the first-run dialog by disabling metrics reporting.
    policies.Set(key::kMetricsReportingEnabled, POLICY_LEVEL_MANDATORY,
                 POLICY_SCOPE_MACHINE, POLICY_SOURCE_CLOUD, base::Value(false),
                 nullptr);

    // Apply the policy setting under test.
    if (GetParam() != BooleanPolicy::kNotConfigured) {
      policies.Set(key::kPromotionalTabsEnabled, POLICY_LEVEL_MANDATORY,
                   POLICY_SCOPE_MACHINE, POLICY_SOURCE_CLOUD,
                   base::Value(GetParam() == BooleanPolicy::kTrue), nullptr);
    }

    UpdateProviderPolicy(policies);
    PolicyTest::CreatedBrowserMainParts(browser_main_parts);
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;
};

#if !defined(OS_CHROMEOS)
// Tests that the PromotionalTabsEnabled policy properly suppresses the welcome
// page for browser first-runs.
class PromotionalTabsEnabledPolicyWelcomeTest
    : public PromotionalTabsEnabledPolicyTest {
 public:
  PromotionalTabsEnabledPolicyWelcomeTest(
      const PromotionalTabsEnabledPolicyWelcomeTest&) = delete;
  PromotionalTabsEnabledPolicyWelcomeTest& operator=(
      const PromotionalTabsEnabledPolicyWelcomeTest&) = delete;

 protected:
  PromotionalTabsEnabledPolicyWelcomeTest() = default;
  ~PromotionalTabsEnabledPolicyWelcomeTest() override = default;

  void SetUpCommandLine(base::CommandLine* command_line) override {
    command_line->AppendSwitch(switches::kForceFirstRun);
  }
};

IN_PROC_BROWSER_TEST_P(PromotionalTabsEnabledPolicyWelcomeTest, RunTest) {
  TabStripModel* tab_strip = browser()->tab_strip_model();
  ASSERT_GE(tab_strip->count(), 1);
  const auto& url = tab_strip->GetWebContentsAt(0)->GetURL();
  switch (GetParam()) {
    case BooleanPolicy::kFalse:
      // Only the NTP should show.
      EXPECT_EQ(tab_strip->count(), 1);
      if (url.possibly_invalid_spec() != chrome::kChromeUINewTabURL)
        EXPECT_PRED2(search::IsNTPOrRelatedURL, url, browser()->profile());
      break;
    case BooleanPolicy::kNotConfigured:
    case BooleanPolicy::kTrue:
      // One or more onboarding tabs should show.
      EXPECT_NE(url.possibly_invalid_spec(), chrome::kChromeUINewTabURL);
      // Welcome should override What's New.
      EXPECT_NE(url.possibly_invalid_spec(), GetWhatsNewAutoURL());
      EXPECT_FALSE(search::IsNTPOrRelatedURL(url, browser()->profile())) << url;
      break;
  }
}

INSTANTIATE_TEST_SUITE_P(
    All,
    PromotionalTabsEnabledPolicyWelcomeTest,
    ::testing::Values(PolicyTest::BooleanPolicy::kNotConfigured,
                      PolicyTest::BooleanPolicy::kFalse,
                      PolicyTest::BooleanPolicy::kTrue));
#endif  // !defined(OS_CHROMEOS)

// Tests that the PromotionalTabsEnabled policy properly suppresses the What's
// New page.
class PromotionalTabsEnabledPolicyWhatsNewTest
    : public PromotionalTabsEnabledPolicyTest {
 public:
  PromotionalTabsEnabledPolicyWhatsNewTest(
      const PromotionalTabsEnabledPolicyWhatsNewTest&) = delete;
  PromotionalTabsEnabledPolicyWhatsNewTest& operator=(
      const PromotionalTabsEnabledPolicyWhatsNewTest&) = delete;

 protected:
  PromotionalTabsEnabledPolicyWhatsNewTest() = default;
  ~PromotionalTabsEnabledPolicyWhatsNewTest() override = default;

  virtual int WhatsNewVersionForPref() { return CHROME_VERSION_MAJOR - 1; }

  void SetUpCommandLine(base::CommandLine* command_line) override {
    ASSERT_TRUE(temp_dir_.CreateUniqueTempDir());
    command_line->RemoveSwitch(switches::kForceFirstRun);
    command_line->AppendSwitch(switches::kNoFirstRun);
    command_line->AppendSwitchPath(switches::kUserDataDir, temp_dir_.GetPath());

    // Suppress the welcome page by setting the pref indicating that it has
    // already been seen. This is necessary because welcome/onboarding takes
    // precedence over What's New.
    std::string json;
    base::DictionaryValue prefs;
    prefs.SetBoolean(prefs::kHasSeenWelcomePage, true);
    base::JSONWriter::Write(prefs, &json);

    base::FilePath default_dir =
        temp_dir_.GetPath().AppendASCII(chrome::kInitialProfile);
    ASSERT_TRUE(base::CreateDirectory(default_dir));
    base::FilePath preferences_path =
        default_dir.Append(chrome::kPreferencesFilename);

    ASSERT_TRUE(base::WriteFile(
        default_dir.Append(chrome::kPreferencesFilename), json));

    // Also set the version for What's New in the local state.
    base::DictionaryValue local_state;
    local_state.SetInteger(prefs::kLastWhatsNewVersion,
                           WhatsNewVersionForPref());
    std::string local_state_string;
    base::JSONWriter::Write(local_state, &local_state_string);
    ASSERT_TRUE(
        base::WriteFile(temp_dir_.GetPath().Append(chrome::kLocalStateFilename),
                        local_state_string));
  }

 private:
  base::ScopedTempDir temp_dir_;
};

IN_PROC_BROWSER_TEST_P(PromotionalTabsEnabledPolicyWhatsNewTest, RunTest) {
  TabStripModel* tab_strip = browser()->tab_strip_model();
  ASSERT_GE(tab_strip->count(), 1);
  const auto& url = tab_strip->GetWebContentsAt(0)->GetURL();
  switch (GetParam()) {
    case BooleanPolicy::kFalse:
      // Only the NTP should show.
      EXPECT_EQ(tab_strip->count(), 1);
      if (url.possibly_invalid_spec() != chrome::kChromeUINewTabURL)
        EXPECT_PRED2(search::IsNTPOrRelatedURL, url, browser()->profile());
      break;
    case BooleanPolicy::kNotConfigured:
    case BooleanPolicy::kTrue:
      // Whats's New should show with auto=true query param.
      EXPECT_EQ(url.possibly_invalid_spec(), GetWhatsNewAutoURL());
      break;
  }
}

INSTANTIATE_TEST_SUITE_P(
    All,
    PromotionalTabsEnabledPolicyWhatsNewTest,
    ::testing::Values(PolicyTest::BooleanPolicy::kNotConfigured,
                      PolicyTest::BooleanPolicy::kFalse,
                      PolicyTest::BooleanPolicy::kTrue));

// Tests that What's New doesn't show up regardless of the policy if the version
// is not greater than the one in |prefs::kLastWhatsNewVersion|.
class PromotionalTabsEnabledPolicyWhatsNewInvalidTest
    : public PromotionalTabsEnabledPolicyWhatsNewTest {
 public:
  PromotionalTabsEnabledPolicyWhatsNewInvalidTest(
      const PromotionalTabsEnabledPolicyWhatsNewInvalidTest&) = delete;
  PromotionalTabsEnabledPolicyWhatsNewTest& operator=(
      const PromotionalTabsEnabledPolicyWhatsNewInvalidTest&) = delete;

 protected:
  PromotionalTabsEnabledPolicyWhatsNewInvalidTest() = default;
  ~PromotionalTabsEnabledPolicyWhatsNewInvalidTest() override = default;

  int WhatsNewVersionForPref() override { return CHROME_VERSION_MAJOR; }
};

IN_PROC_BROWSER_TEST_P(PromotionalTabsEnabledPolicyWhatsNewInvalidTest,
                       RunTest) {
  TabStripModel* tab_strip = browser()->tab_strip_model();
  ASSERT_GE(tab_strip->count(), 1);
  const auto& url = tab_strip->GetWebContentsAt(0)->GetURL();
  // Only the NTP should show. There are no other relevant tabs since
  // welcome and What's New have both already been shown.
  EXPECT_EQ(tab_strip->count(), 1);
  if (url.possibly_invalid_spec() != chrome::kChromeUINewTabURL)
    EXPECT_PRED2(search::IsNTPOrRelatedURL, url, browser()->profile());
}

INSTANTIATE_TEST_SUITE_P(
    All,
    PromotionalTabsEnabledPolicyWhatsNewInvalidTest,
    ::testing::Values(PolicyTest::BooleanPolicy::kNotConfigured,
                      PolicyTest::BooleanPolicy::kFalse,
                      PolicyTest::BooleanPolicy::kTrue));
}  // namespace policy
