// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/subresource_filter/subresource_filter_browser_test_harness.h"

#include "build/build_config.h"
#include "chrome/test/base/ui_test_utils.h"
#include "components/subresource_filter/content/browser/ad_tagging_browser_test_utils.h"
#include "components/subresource_filter/content/browser/content_subresource_filter_throttle_manager.h"
#include "components/subresource_filter/content/browser/subresource_filter_observer.h"
#include "components/subresource_filter/content/browser/subresource_filter_observer_manager.h"
#include "components/subresource_filter/content/browser/subresource_filter_observer_test_utils.h"
#include "components/subresource_filter/core/browser/subresource_filter_constants.h"
#include "components/subresource_filter/core/common/test_ruleset_utils.h"
#include "components/subresource_filter/core/mojom/subresource_filter.mojom.h"
#include "content/public/test/browser_test.h"
#include "content/public/test/browser_test_utils.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"

using content::Page;
using content::RenderFrameHost;
using content::WebContents;
using content::WebContentsConsoleObserver;
using content::test::PrerenderHostObserver;
using subresource_filter::testing::CreateSuffixRule;
using testing::_;
using testing::Mock;

namespace subresource_filter {

// Tests -----------------------------------------------------------------------

// A very basic smoke test for prerendering; this test just activates on the
// main frame of a prerender. It currently doesn't check any behavior but
// passes if we don't crash.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       PrerenderingSmokeTest) {
  const GURL kPrerenderingUrl =
      embedded_test_server()->GetURL("/page_with_iframe.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");

  // Configure dry run filtering on all URLs.
  {
    ASSERT_NO_FATAL_FAILURE(SetRulesetToDisallowURLsWithPathSuffix(
        "suffix-that-does-not-match-anything"));
    Configuration config(subresource_filter::mojom::ActivationLevel::kDryRun,
                         subresource_filter::ActivationScope::ALL_SITES);
    ResetConfiguration(std::move(config));
  }

  // We should get 2 DryRun page activations, one from the initial page and one
  // from the prerender.
  MockSubresourceFilterObserver observer(web_contents());
  EXPECT_CALL(observer, OnPageActivationComputed(_, HasActivationLevelDryRun()))
      .Times(2);
  ui_test_utils::NavigateToURL(browser(), kInitialUrl);
  prerender_helper_.AddPrerender(kPrerenderingUrl);
}

// Test that we correctly account for activation between the prerendering frame
// and primary pages.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       OnlyPrerenderingFrameActivated) {
  const GURL kPrerenderingUrl = embedded_test_server()->GetURL(
      "/subresource_filter/frame_with_included_script.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");

  // Configure the filter to run only on the prerendering URL.
  {
    ConfigureAsSubresourceFilterOnlyURL(kPrerenderingUrl);
    ASSERT_NO_FATAL_FAILURE(SetRulesetToDisallowURLsWithPathSuffix(
        "suffix-that-does-not-match-anything"));
    Configuration config(
        subresource_filter::mojom::ActivationLevel::kEnabled,
        subresource_filter::ActivationScope::ACTIVATION_LIST,
        subresource_filter::ActivationList::SUBRESOURCE_FILTER);
    ResetConfiguration(std::move(config));
  }

  // Navigate to the initial URL - ensure filtering is not activated.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelDisabled()));
    ui_test_utils::NavigateToURL(browser(), kInitialUrl);
  }

  // Trigger a prerender to the prerendering URL - this URL should activate the
  // filter.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelEnabled()));
    prerender_helper_.AddPrerender(kPrerenderingUrl);
  }
}

// Test that we don't start filtering an unactivated primary page when a
// prerendering page becomes activated.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       UnactivatedPrimaryFrameNotFiltered) {
  const GURL kPrerenderingUrl = embedded_test_server()->GetURL("/empty.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL(
      "/subresource_filter/frame_with_delayed_script.html");

  // Configure filtering of `included_script.js` on the prerendering page only.
  {
    ConfigureAsSubresourceFilterOnlyURL(kPrerenderingUrl);
    ASSERT_NO_FATAL_FAILURE(
        SetRulesetToDisallowURLsWithPathSuffix("included_script.js"));
    Configuration config(
        subresource_filter::mojom::ActivationLevel::kEnabled,
        subresource_filter::ActivationScope::ACTIVATION_LIST,
        subresource_filter::ActivationList::SUBRESOURCE_FILTER);
    ResetConfiguration(std::move(config));
  }

  // Navigate to the initial page - it should not activate subresource
  // filtering.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelDisabled()));
    ui_test_utils::NavigateToURL(browser(), kInitialUrl);
    ASSERT_TRUE(Mock::VerifyAndClearExpectations(&observer));
  }

  // Trigger a prerender to a URL that does have subresource filtering enabled.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelEnabled()));
    prerender_helper_.AddPrerender(kPrerenderingUrl);
    ASSERT_TRUE(Mock::VerifyAndClearExpectations(&observer));
  }

  // Now dynamically try to load `included_script.js` in the primary frame.
  // Ensure it is not filtered.
  EXPECT_TRUE(IsDynamicScriptElementLoaded(web_contents()->GetMainFrame()));
}

// Test that we don't start filtering an unactivated prerendering page when the
// primary page is activated.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       UnactivatedPrerenderingFrameNotFiltered) {
  const GURL kPrerenderingUrl = embedded_test_server()->GetURL(
      "/subresource_filter/frame_with_included_script.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");

  // Configure filtering of `included_script.js` on the initial URL only.
  {
    ConfigureAsSubresourceFilterOnlyURL(kInitialUrl);
    ASSERT_NO_FATAL_FAILURE(
        SetRulesetToDisallowURLsWithPathSuffix("included_script.js"));
    Configuration config(
        subresource_filter::mojom::ActivationLevel::kEnabled,
        subresource_filter::ActivationScope::ACTIVATION_LIST,
        subresource_filter::ActivationList::SUBRESOURCE_FILTER);
    ResetConfiguration(std::move(config));
  }

  // Navigate to the initial page that should activate subresource filtering.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelEnabled()));
    ui_test_utils::NavigateToURL(browser(), kInitialUrl);
    ASSERT_TRUE(Mock::VerifyAndClearExpectations(&observer));
  }

  // Trigger a prerender to an unfiltered URL. The script element that would be
  // blocked on an activated URL should be allowed since the prerender URL
  // shouldn't enable activation.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelDisabled()));
    const int host_id = prerender_helper_.AddPrerender(kPrerenderingUrl);
    ASSERT_TRUE(Mock::VerifyAndClearExpectations(&observer));

    // Expect that we didn't filter the script in the prerendering page since
    // only the primary page was activated.
    RenderFrameHost* prerender_rfh =
        prerender_helper_.GetPrerenderedMainFrameHost(host_id);
    EXPECT_TRUE(WasParsedScriptElementLoaded(prerender_rfh));
  }
}

// Test that we can filter a subresource while inside a prerender. Ensure we
// don't display any UI while prerendered but once the prerender becomes
// primary we then show notifications.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       FilterWhilePrerendered) {
  const GURL kPrerenderingUrl = embedded_test_server()->GetURL(
      "/subresource_filter/frame_with_included_script.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");

  // Configure filtering of `included_script.js` only on the prerendering URL.
  {
    ConfigureAsSubresourceFilterOnlyURL(kPrerenderingUrl);
    ASSERT_NO_FATAL_FAILURE(
        SetRulesetToDisallowURLsWithPathSuffix("included_script.js"));
    Configuration config(
        subresource_filter::mojom::ActivationLevel::kEnabled,
        subresource_filter::ActivationScope::ACTIVATION_LIST,
        subresource_filter::ActivationList::SUBRESOURCE_FILTER);
    ResetConfiguration(std::move(config));
  }

  // Navigate to the initial URL. Activation is only enabled on the
  // prerendering URL so we expect no activation.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelDisabled()));
    ui_test_utils::NavigateToURL(browser(), kInitialUrl);
  }

  // Trigger a prerender. Ensure it too is activated.
  RenderFrameHost* prerender_rfh = nullptr;
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelEnabled()));
    const int host_id = prerender_helper_.AddPrerender(kPrerenderingUrl);
    ASSERT_TRUE(Mock::VerifyAndClearExpectations(&observer));

    prerender_rfh = prerender_helper_.GetPrerenderedMainFrameHost(host_id);

    // Expect that the disallowed script was blocked.
    EXPECT_FALSE(WasParsedScriptElementLoaded(prerender_rfh));

    // But ensure we haven't shown the notification UI yet since the page is
    // still prerendering.
    EXPECT_FALSE(AdsBlockedInContentSettings(web_contents()->GetMainFrame()));
    EXPECT_FALSE(AdsBlockedInContentSettings(prerender_rfh));
#if defined(OS_ANDROID)
    EXPECT_FALSE(PresentingAdsBlockedInfobar());
#endif
  }

  // Makes the prerendering page primary (i.e. the user clicked on a link to
  // the prerendered URL), the UI should now be shown.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer, OnPageActivationComputed(_, _)).Times(0);
    prerender_helper_.NavigatePrimaryPage(kPrerenderingUrl);

    EXPECT_TRUE(AdsBlockedInContentSettings(web_contents()->GetMainFrame()));
    EXPECT_TRUE(AdsBlockedInContentSettings(prerender_rfh));
#if defined(OS_ANDROID)
    EXPECT_TRUE(PresentingAdsBlockedInfobar());
#endif
  }
}

// Tests that console messages generated by the subresource filter for a
// prerendering page are correctly attributed to the prerendering page's render
// frame host. Note, this doesn't necessarily guarantee they won't be displayed
// in the primary page's console (in fact, this is the current behavior), but
// that's a more general problem of prerendering that will be fixed.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       ConsoleMessageFilterWhilePrerendered) {
  const GURL kPrerenderingUrl = embedded_test_server()->GetURL(
      "/subresource_filter/frame_with_delayed_script.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");

  // Watch for the subresource filtering activation and resource blocked
  // console messages.
  WebContentsConsoleObserver console_activation_observer(web_contents());
  console_activation_observer.SetPattern(kActivationConsoleMessage);
  WebContentsConsoleObserver console_blocked_observer(web_contents());
  console_blocked_observer.SetPattern(
      base::StringPrintf(kDisallowSubframeConsoleMessageFormat, "*"));

  // Configure filtering of `included_script.js` only on the prerendering URL.
  {
    ConfigureAsSubresourceFilterOnlyURL(kPrerenderingUrl);
    ASSERT_NO_FATAL_FAILURE(
        SetRulesetToDisallowURLsWithPathSuffix("included_script.js"));
    Configuration config(
        subresource_filter::mojom::ActivationLevel::kEnabled,
        subresource_filter::ActivationScope::ACTIVATION_LIST,
        subresource_filter::ActivationList::SUBRESOURCE_FILTER);
    ResetConfiguration(std::move(config));
  }

  // Navigate to the initial URL and trigger the prerender.
  ui_test_utils::NavigateToURL(browser(), kInitialUrl);
  const int host_id = prerender_helper_.AddPrerender(kPrerenderingUrl);
  RenderFrameHost* prerender_rfh =
      prerender_helper_.GetPrerenderedMainFrameHost(host_id);
  ASSERT_FALSE(IsDynamicScriptElementLoaded(prerender_rfh));

  EXPECT_EQ(console_activation_observer.messages().size(), 1ul);
  EXPECT_EQ(
      &console_activation_observer.messages().back().source_frame->GetPage(),
      &prerender_rfh->GetPage());

  EXPECT_EQ(console_blocked_observer.messages().size(), 1ul);
  EXPECT_EQ(&console_blocked_observer.messages().back().source_frame->GetPage(),
            &prerender_rfh->GetPage());
}

// Prerender a page, then navigate it. The prerender will be canceled. We check
// this here since this could change in the future and we'd want to ensure
// subresource filtering is correct in prerender navigations.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       NavigatePrerenderedPage) {
  const GURL kPrerenderUrl1 = embedded_test_server()->GetURL("/title1.html");
  const GURL kPrerenderUrl2 = embedded_test_server()->GetURL("/title2.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");

  // Navigate to the initial URL.
  ui_test_utils::NavigateToURL(browser(), kInitialUrl);

  // Trigger a prerendering of title1.html.
  const int prerender_host_id = prerender_helper_.AddPrerender(kPrerenderUrl1);

  // Now navigate the prerendered page to title2.html. Ensure the prerender is
  // canceled.
  // TODO(bokan): If this is ever changed, we should ensure this test checks
  // that we navigate the prerender between filtered and unfiltered pages, and
  // ensure subresources are correctly filtered (or not).
  PrerenderHostObserver host_observer(*web_contents(), prerender_host_id);
  prerender_helper_.NavigatePrerenderedPage(prerender_host_id, kPrerenderUrl2);
  host_observer.WaitForDestroyed();
}

// Tests that a prerendering page that has filtering activated, will continue
// to filter subresources once made primary (i.e. once the user navigates to
// the prerendered URL).
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       FilteringPrerenderBecomesPrimary) {
  const GURL kPrerenderingUrl = embedded_test_server()->GetURL(
      "/subresource_filter/frame_with_delayed_script.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");

  // Configure filtering of `included_script.js` only on the prerendering URL.
  {
    ConfigureAsSubresourceFilterOnlyURL(kPrerenderingUrl);
    ASSERT_NO_FATAL_FAILURE(
        SetRulesetToDisallowURLsWithPathSuffix("included_script.js"));
    Configuration config(
        subresource_filter::mojom::ActivationLevel::kEnabled,
        subresource_filter::ActivationScope::ACTIVATION_LIST,
        subresource_filter::ActivationList::SUBRESOURCE_FILTER);
    ResetConfiguration(std::move(config));
  }

  // Navigate to the initial URL.
  ui_test_utils::NavigateToURL(browser(), kInitialUrl);

  // Trigger a prerender. Ensure it is activated.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelEnabled()));
    prerender_helper_.AddPrerender(kPrerenderingUrl);
    ASSERT_TRUE(Mock::VerifyAndClearExpectations(&observer));
  }

  // Makes the prerendering page primary (i.e. the user clicked on a link to
  // the prerendered URL). Ensure a new request for a blocked resource will be
  // filtered.
  {
    prerender_helper_.NavigatePrimaryPage(kPrerenderingUrl);
    EXPECT_FALSE(IsDynamicScriptElementLoaded(web_contents()->GetMainFrame()));
  }
}

// Tests that a prerendering page that doesn't have filtering activated, will
// continue to be unfiltered when made primary (i.e. once the user navigates to
// the prerendered URL) from an activated initial URL.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       NonFilteringPrerenderBecomesPrimary) {
  const GURL kPrerenderingUrl = embedded_test_server()->GetURL(
      "/subresource_filter/frame_with_delayed_script.html");
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");

  // Configure filtering of `included_script.js` only on the initial URL.
  {
    ConfigureAsSubresourceFilterOnlyURL(kInitialUrl);
    ASSERT_NO_FATAL_FAILURE(
        SetRulesetToDisallowURLsWithPathSuffix("included_script.js"));
    Configuration config(
        subresource_filter::mojom::ActivationLevel::kEnabled,
        subresource_filter::ActivationScope::ACTIVATION_LIST,
        subresource_filter::ActivationList::SUBRESOURCE_FILTER);
    ResetConfiguration(std::move(config));
  }

  // Navigate to the initial URL. Ensure it is activated.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelEnabled()));
    ui_test_utils::NavigateToURL(browser(), kInitialUrl);
    ASSERT_TRUE(Mock::VerifyAndClearExpectations(&observer));
  }

  // Trigger a prerender. Ensure it is not activated.
  {
    MockSubresourceFilterObserver observer(web_contents());
    EXPECT_CALL(observer,
                OnPageActivationComputed(_, HasActivationLevelDisabled()));
    prerender_helper_.AddPrerender(kPrerenderingUrl);
    ASSERT_TRUE(Mock::VerifyAndClearExpectations(&observer));
  }

  // Make the prerendering page primary (i.e. the user clicked on a link to
  // the prerendered URL). Ensure a new request to `included_script.js` remains
  // unfiltered.
  {
    prerender_helper_.NavigatePrimaryPage(kPrerenderingUrl);
    ASSERT_EQ(kPrerenderingUrl, web_contents()->GetLastCommittedURL());
    EXPECT_TRUE(IsDynamicScriptElementLoaded(web_contents()->GetMainFrame()));
  }
}

// Very basic test that ad tagging works in a prerender.
IN_PROC_BROWSER_TEST_F(SubresourceFilterPrerenderingBrowserTest,
                       AdTaggingSmokeTest) {
  const GURL kInitialUrl = embedded_test_server()->GetURL("/empty.html");
  const GURL kPrerenderingUrl =
      embedded_test_server()->GetURL("/ad_tagging/frame_factory.html");
  const GURL kAdUrl =
      embedded_test_server()->GetURL("/ad_tagging/frame_factory.html?1");

  SetRulesetWithRules({CreateSuffixRule("ad_script.js")});

  TestSubresourceFilterObserver observer(web_contents());

  RenderFrameHost* prerender_rfh = nullptr;

  // Load the initial page and trigger a prerender.
  {
    ui_test_utils::NavigateToURL(browser(), kInitialUrl);
    const int prerender_host_id =
        prerender_helper_.AddPrerender(kPrerenderingUrl);
    prerender_rfh =
        prerender_helper_.GetPrerenderedMainFrameHost(prerender_host_id);
    ASSERT_NE(prerender_rfh, nullptr);
  }

  RenderFrameHost* ad_rfh = nullptr;

  // In the prerendering page, create a child frame from an ad script. Ensure it
  // is correctly tagged as an ad.
  {
    ad_rfh = CreateSrcFrameFromAdScript(prerender_rfh, kAdUrl);
    ASSERT_NE(ad_rfh, nullptr);
    EXPECT_TRUE(observer.GetIsAdSubframe(ad_rfh->GetFrameTreeNodeId()));
    EXPECT_TRUE(EvidenceForFrameComprises(
        ad_rfh, /*parent_is_ad=*/false,
        blink::mojom::FilterListResult::kMatchedNoRules,
        blink::mojom::FrameCreationStackEvidence::kCreatedByAdScript));
  }

  // Make the prerendering page primary (i.e. the user clicked on a link to the
  // prerendered URL). Ensure ad tagging remains valid for the ad frame and its
  // frame tree node id.
  {
    prerender_helper_.NavigatePrimaryPage(kPrerenderingUrl);
    ASSERT_EQ(kPrerenderingUrl, web_contents()->GetLastCommittedURL());
    EXPECT_TRUE(observer.GetIsAdSubframe(ad_rfh->GetFrameTreeNodeId()));
    EXPECT_TRUE(EvidenceForFrameComprises(
        ad_rfh, /*parent_is_ad=*/false,
        blink::mojom::FilterListResult::kMatchedNoRules,
        blink::mojom::FrameCreationStackEvidence::kCreatedByAdScript));
  }
}

// TODO - test that prerender activation hides infobars.

}  // namespace subresource_filter
