# CMU Food Truck Environment (LLM-friendly)

This folder implements a Gymnasium-compatible Food Truck environment with **text-only** observations and actions.

## Install

```bash
pip install -r requirements.txt
```

## Quick start

```python
from foodtruck_env import FoodTruckEnv, EnvConfig

env = FoodTruckEnv(EnvConfig())
obs, info = env.reset(seed=0)
print(obs)

obs, reward, terminated, truncated, info = env.step("check market")
print(obs)
```

## Action commands

All commands are **case-insensitive** and consume **one prep operation**.

### Check commands

- `check storage`
- `check market`
- `check recipes`
- `check menu`

There is **no** `check preferences` command (customer preferences are hidden).

### Direct actions

- `buy <ingredient> <qty>`
- `trash <ingredient> <qty>`
- `set menu <slot> <recipe> <price>`
- `clear menu <slot>`
- `end prep`

### Notes

- `slot` is **1-based** (1..max_menu_items).
- `<ingredient>` and `<recipe>` can be a **name** or a **1-based index**.
- Prices must be within `[price_min, price_max]` and align with `price_step`.

## Observation format

Observation is a single text string with sections. It always includes:

- Day number and day-of-week
- Funds
- Remaining prep ops
- Stage

Checked sections appear only after the corresponding `check ...` command:

- Storage
- Market
- Recipes
- Menu

When `end prep` completes, the observation also includes a **daily summary** (orders, revenue, rent, net).

## Configuration

Hyperparameters are configurable via `EnvConfig` in `config.py` (defaults match the task spec):

- `horizon_days`, `start_funds`, `daily_rent`
- `fridge_capacity`, `pantry_capacity`
- `num_recipes`, `max_menu_items`
- `max_prep_ops_per_day`
- Market and demand dynamics parameters

## Example script

See `example_llm_loop.py` for a minimal loop with scripted commands.

## Run the LLM agent (`simple_llm_agent.py`)

This repo includes a reference LLM-driven agent that uses the OpenAI Python SDK with a configurable `base_url` (works with CMU's AI gateway).

### Install

```bash
pip install -r requirements.txt
```

### Run

```bash
python simple_llm_agent.py \
  --api-key "<YOUR_KEY>" \
  --base-url "https://ai-gateway.andrew.cmu.edu" \
  --seed 1
```

### Useful flags

- `--model`: defaults to `gpt-5-mini`
- `--max-steps`: safety cap on environment steps (default 400)
- `--max-history-tokens`: transcript budget sent to the model each step (default 32000). Oldest transcript entries are dropped when over budget.
- `--max-response-tokens`: cap tokens generated for the action (optional)
- `--temperature`: sampling temperature (optional; leave unset for provider default)

### Notes

- The script prints an initial observation, then repeatedly prints `>>> <action>` and the resulting observation/reward.
- The first run may download tokenizer files needed for token counting (via `tiktoken`).
