/*
 * Copyright (c) 2014, the Dart project authors.  Please see the AUTHORS file
 * for details. All rights reserved. Use of this source code is governed by a
 * BSD-style license that can be found in the LICENSE file.
 */
/** 
 * @description Test that rules in an inner scoped stylesheet don't override
 * !important rules declared in an outer scoped stylesheet.
 */
import "dart:html";
import "../../../testcommon.dart";
import "../../../../Utils/async_utils.dart";
import "pwd.dart";

main() {
  var style = new Element.html('''
      <style>
      span { border-color: lime ! important; }
      div { border-color: orange; }
      div#id2 { border-color: yellow; }
      </style>
      ''', treeSanitizer: new NullTreeSanitizer());
  document.head.append(style);

  document.body.setInnerHtml('''
      <div>
        <style scoped>
            div { border-color: green ! important; }
            div { border-color: red; }
            div#id2 {border-color: purple; }
        </style>
        <div>
          <style scoped>
              div { border-color: blue; }
          </style>
          <div id="id1"></div>
        </div>
        <div>
          <style scoped>
              div#id2 { border-color: blue; }
          </style>
          <div id="id2"></div>
        </div>
        <div>
          <style scoped>
              div { border-color: blue; }
          </style>
          <div id="id3" style="border-color: red;"></div>
        </div>
        <div>
          <style scoped>
            div { border-color: blue !important; }
          </style>
          <div id="id4"></div>
        </div>
      </div>
      <div>
        <style scoped>
            span#id5 { border-color: blue; }
        </style>
        <span id="id5"></span>
      </div>
      <div>
        <style scoped>
            span { border-color: blue !important; }
        </style>
        <span id="id6"></span>
      </div>
      ''', treeSanitizer: new NullTreeSanitizer());

  var target1 = document.getElementById("id1");
  debug("Case1: The target element has any matched important rule declared in an outer scoped stylesheet, and the element also has any matched normal rule declared in an inner scoped stylesheet.");
  shouldBe(getComputedStyle(target1).borderColor, "rgb(0, 128, 0)");

  debug("Case2: The target element has any matched important rule declared in an outer scoped stylesheet, and the element also has a matched normal ID rule declared in an inner scoped stylesheet.");
  var target2 = document.getElementById("id2");
  shouldBe(getComputedStyle(target2).borderColor, "rgb(0, 128, 0)");

  debug("Case3: The target element has any matched important rule declared in an outer scoped stylesheet, and the element also has matched normal rules declared in an inner scoped stylesheet and in a STYLE attribute.");
  var target3 = document.getElementById("id3");
  shouldBe(getComputedStyle(target3).borderColor, "rgb(0, 128, 0)");

  debug("Case4: The target element has matched important rules. One is declared in an outer scoped stylesheet and the other is declared in an inner scoped stylesheet.");
  var target4 = document.getElementById("id4");
  shouldBe(getComputedStyle(target4).borderColor, "rgb(0, 0, 255)");

  debug("Case5: The target element has any matched important rule declared in an author stylesheet, and the element also has matched normal rules declared in an inner scoped stylesheet.");
  var target5 = document.getElementById("id5");
  shouldBe(getComputedStyle(target5).borderColor, "rgb(0, 255, 0)");

  debug("Case6: The target element has matched important rules. One is declared in an author stylesheet (not scoped) and the other is declared in a scoped stylesheet.");
  var target6 = document.getElementById("id6");
  shouldBe(getComputedStyle(target6).borderColor, "rgb(0, 0, 255)");
}
