/*
 * Copyright (c) 2014, the Dart project authors.  Please see the AUTHORS file
 * for details. All rights reserved. Use of this source code is governed by a
 * BSD-style license that can be found in the LICENSE file.
 */
/**
 * @assertion 
 * @description IDNA2008 handling in domain name labels.
 * of this test are set to the behavior in IDNA2008.
 */
import "../../testharness.dart";
import "resources/utilities.dart";

List cases = [ 
  // For IDNA Compatibility test material see
  // http://www.unicode.org/reports/tr46/
  // http://www.unicode.org/Public/idna/latest/IdnaMappingTable.txt 
  // We are testing disallowed, ignored, mapped, deviant, and valid cases.
  // 1) Deviant character tests (deviant processing behavior from IDNA2003)
  ["B\u00FCcher.de","xn--bcher-kva.de"],
  // The ß U+00DF LATIN SMALL LETTER SHARP S does NOT normalize to "ss" like it does during IDNA2003's mapping phase
  ["fa\u00DF.de","xn--fa-hia.de"],
  // The ς U+03C2 GREEK SMALL LETTER FINAL SIGMA using βόλος.com
  ["\u03B2\u03CC\u03BB\u03BF\u03C2.com","xn--nxasmm1c.com"],
  // The ZWJ U+200D ZERO WIDTH JOINER
  ["\u0DC1\u0DCA\u200D\u0DBB\u0DD3.com","xn--10cl1a0b660p.com"],
  // The ZWNJ U+200C ZERO WIDTH NON-JOINER
  ["\u0646\u0627\u0645\u0647\u200C\u0627\u06CC.com","xn--mgba3gch31f060k.com"],
  // 2) Disallowed characters in IDNA2008
  // U+2665 BLACK HEART SUIT
  ["\u2665.net","\uFFFD.net"],
  // U+0378 <reserved>
  ["\u0378.net","\uFFFD.net"],
  ["\u04C0.com","\uFFFD.com"],
//  ["\uD87E\uDC68.com","\uFFFD.com"],
  ["\u2183.com","\uFFFD.com"],
  // 3) Ignored characters should be removed * security risk
  // U+034F COMBINING GRAPHEME JOINER
  ["look\u034Fout.net","lookout.net"],
  // 4) Mapped characters; four cases belows are actually UTS 46 tests.
  ["gOoGle.com","google.com"],
  // U+09DC is normalized to U+09A1, U+09BC before being turned to punycode.
  ["\u09dc.com","xn--15b8c.com"],
  // 1E9E; mapped; 0073 0073
  ["\u1E9E.com","ss.com"],
  // 5) Validity FAIL cases - these should each cause an error.
  ["-foo.bar.com",""],
  ["foo-.bar.com",""],
  ["ab--cd.com",""],
  ["xn--0.com",""],
  ["foo\u0300.bar.com","foo%CC%80.bar.com"]
];

void main() {
    
   for (var i = 0; i < cases.length; ++i) {
      String test_vector = cases[i][0];;
      String actual=canonicalize('http://$test_vector/');
      String expected='http://${cases[i][1]}/';
      test((){
          Expect.equals(expected, actual);
      }, "$i");
    }
    checkTestFailures();
}
