/*
 * Copyright (c) 2014, the Dart project authors.  Please see the AUTHORS file
 * for details. All rights reserved. Use of this source code is governed by a
 * BSD-style license that can be found in the LICENSE file.
 */
/**
 * @assertion <a href="http://bugs.webkit.org/show_bug.cgi?id=50366">bug 50366</a>:
 * XPath lexer misinterprets expression starting with "div"
 * @description Test that an NCName and * are interpreted as an operator when in
 * binary operator context, and as a NameTest otherwise.
 */
import "dart:html";
import "xpath-test-pre.dart";

const String htmlEL = r'''
<div id="context">
  <div id="two" div="x">2</div>
</div>
''';

void main() {
    document.body.appendHtml(htmlEL);
    var context = document.getElementById('context');
    var div = document.getElementById('two');

    // =========== div ===========

    test(document, context, "div", [div], null);  // div with no preceding token
    test(document, context, "  div", [div], null);
    // div following every possible [28] ExprToken http://www.w3.org/TR/xpath/
    test(document, context, "(div) div (div)", 2/2, null);  // div after ( and )
    test(document, context, "self::node()[div] div 1", 2/1, null);  // div after [ and ]
    test(document, context, ". div .", 2/2, null);  // div after .
    test(document, div, ".. div ..", 2/2, null);  // div after ..
    test(document, context, "string(div/@div)", "x", null);  // div after @
    test(document, context, "substring-before('1992', div)", "199", null);  // div after ,
    test(document, div, "self::div", [div], null);  // div after ::
    test(document, context, "* div 4", 2/4, null);  // div after NameTest *
    test(document, context, "'3' div 4", 3/4, null);  // div after Literal
    test(document, context, "\"3\" div 4", 3/4, null);  // div after Literal
    test(document, context, "12 div 4", 12/4, null);  // div after Number

    // div following every [32] Operator
    test(document, context, "true() and div", true, null);  // div after OperatorName and
    test(document, context, "false() or div", true, null);  // div after OperatorName or
    test(document, context, "div mod div", 0, null);  // div after OperatorName mod
    test(document, context, "div div div", 1, null);  // div after OperatorName div
    test(document, context, "3 * div", 6, null);  // div after MultiplyOperator
    test(document, context, "div/div", [], null);  // div after /
    test(document, context, "div//div", [], null);  // div after //
    test(document, context, "zz|div", [div], null);  // div after |
    test(document, context, "div+div", 4, null);  // div after +
    test(document, context, "- - div", 2, null);  // div after unary -
    test(document, context, "5 -div", 3, null);  // div after binary -
    test(document, context, "div=div", true, null);  // div after =
    test(document, context, "div!=div", false, null);  // div after =
    test(document, context, "div<div", false, null);  // div after <
    test(document, context, "div<=div", true, null);  // div after <=
    test(document, context, "div>div", false, null);  // div after >
    test(document, context, "div>=div", true, null);  // div after >=


    // =========== * ===========

    test(document, context, "*", [div], null);  // * with no preceding token
    test(document, context, "  *", [div], null);
    // * following every possible [28] ExprToken http://www.w3.org/TR/xpath/
    test(document, context, "(*) * (*)", 2*2, null);  // * after ( and )
    test(document, context, "self::node()[*] * 1", 2*1, null);  // * after [ and ]
    test(document, context, ". * .", 2*2, null);  // * after .
    test(document, div, ".. * ..", 2*2, null);  // * after ..
    // (* can't follow @)
    test(document, context, "substring-before('1992', *)", "199", null);  // * after ,
    test(document, div, "self::*", [div], null);  // * after ::
    test(document, context, "* * 4", 2*4, null);  // * after NameTest *
    test(document, context, "'3' * 4", 3*4, null);  // * after Literal
    test(document, context, "\"3\" * 4", 3*4, null);  // * after Literal
    test(document, context, "12 * 4", 12*4, null);  // * after Number

    // * following every [32] Operator
    test(document, context, "true() and *", true, null);  // * after OperatorName and
    test(document, context, "false() or *", true, null);  // * after OperatorName or
    test(document, context, "* mod *", 0, null);  // * after OperatorName mod
    test(document, context, "* div *", 1, null);  // * after OperatorName div
    test(document, context, "3 * *", 6, null);  // * after MultiplyOperator
    test(document, context, "*/*", [], null);  // * after /
    test(document, context, "*//*", [], null);  // * after //
    test(document, context, "zz|*", [div], null);  // * after |
    test(document, context, "*+*", 4, null);  // * after +
    test(document, context, "- - *", 2, null);  // * after unary -
    test(document, context, "5 -*", 3, null);  // * after binary -
    test(document, context, "*=*", true, null);  // * after =
    test(document, context, "*!=*", false, null);  // * after =
    test(document, context, "*<*", false, null);  // * after <
    test(document, context, "*<=*", true, null);  // * after <=
    test(document, context, "*>*", false, null);  // * after >
    test(document, context, "*>=*", true, null);  // * after >=

    var xmlDoc = (new DomParser()).parseFromString("<or xmlns='uri'>5</or>", "text/xml");
    var nsResolver = (prefix) {return 'uri';};
    // div and * after :
    test(xmlDoc, xmlDoc, "or:or", [xmlDoc.documentElement], nsResolver);
    test(xmlDoc, xmlDoc, "or:*", [xmlDoc.documentElement], nsResolver);

    // A few tests with mod as well
    var xmlDoc2 = (new DomParser()).parseFromString("<mod and='x'>8</mod>", "text/xml");
    test(xmlDoc2, xmlDoc2, "mod", [xmlDoc2.documentElement], null);
    test(xmlDoc2, xmlDoc2, "mod mod mod", 0, null);
    test(xmlDoc2, xmlDoc2, "(mod) mod 5", 3, null);
    test(xmlDoc2, xmlDoc2, "string(mod/@and)", 'x', null);
    
    checkTestFailures();
}