/*
 * Copyright (c) 2011, the Dart project authors.  Please see the AUTHORS file
 * for details. All rights reserved. Use of this source code is governed by a
 * BSD-style license that can be found in the LICENSE file.
 */
/**
 * @assertion abstract void sort([int compare(E a, E b)])
 * Sorts the list according to the order specified by the compare function.
 * The compare function must act as a Comparator.
 * The default List implementations use Comparable.compare if compare is omitted.
 * @description Checks sorting various integer arrays (lengths: 8, 15, 16, 17, 31, 32, 33, 57, 79, 123, 127, 128, 129).
 * @author iefremov
 * @reviewer msyabro
 */
library sort_A01_t04;

import "../../../Utils/expect.dart";

test(List create([int length])) {

  void check(var actual0, var expected) {
    List actual=create();
    actual.addAll(actual0);
    int c(var a, var b) {
      return a < b ? -1 : (a == b ? 0 : 1);
    }
    actual.sort(c);
    Expect.listEquals(expected, actual);
  }

//length == 8
  check([ 00, -01, -01,  01,  00,  00, -01,  00], [-01, -01, -01,  00,  00,  00,  00,  01]);
  check([ 01,  00,  00, -01,  00,  01, -01,  00], [-01, -01,  00,  00,  00,  00,  01,  01]);
  check([ 01,  00,  00,  00, -01,  00, -01, -01], [-01, -01, -01,  00,  00,  00,  00,  01]);
  check([-01, -01, -01, -01, -01, -01, -01,  01], [-01, -01, -01, -01, -01, -01, -01,  01]);
  check([-01,  00,  01,  01,  01,  01, -01,  00], [-01, -01,  00,  00,  01,  01,  01,  01]);
  check([ 00,  01,  01,  01,  01, -01,  01,  01], [-01,  00,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  00,  00,  01,  00, -01, -01], [-01, -01,  00,  00,  00,  00,  00,  01]);
  check([ 01, -01,  00,  01,  00,  00,  01,  01], [-01,  00,  00,  00,  01,  01,  01,  01]);
  check([-01,  01,  01, -01,  01,  01,  01, -01], [-01, -01, -01,  01,  01,  01,  01,  01]);
  check([ 01,  00, -01,  01,  01, -01,  00,  00], [-01, -01,  00,  00,  00,  01,  01,  01]);


//length == 8
  check([ 23, -20,  08, -10,  12,  11, -16,  03], [-20, -16, -10,  03,  08,  11,  12,  23]);
  check([-19,  00, -06,  02,  21,  14,  15,  03], [-19, -06,  00,  02,  03,  14,  15,  21]);
  check([-14,  17,  22,  06,  06, -25,  04, -21], [-25, -21, -14,  04,  06,  06,  17,  22]);
  check([ 05,  01, -24, -07,  15,  23, -17,  15], [-24, -17, -07,  01,  05,  15,  15,  23]);
  check([-22,  18, -19,  17,  04, -10, -05,  10], [-22, -19, -10, -05,  04,  10,  17,  18]);
  check([ 17,  15, -11,  13,  06, -20, -06,  19], [-20, -11, -06,  06,  13,  15,  17,  19]);
  check([ 22, -07,  01,  03, -07, -20,  10, -25], [-25, -20, -07, -07,  01,  03,  10,  22]);
  check([ 08,  11, -05, -02,  10,  03, -11,  15], [-11, -05, -02,  03,  08,  10,  11,  15]);
  check([-03, -03,  08, -22,  14, -22,  14,  06], [-22, -22, -03, -03,  06,  08,  14,  14]);
  check([ 20, -22, -05,  02, -15,  16, -04, -16], [-22, -16, -15, -05, -04,  02,  16,  20]);


//length == 15
  check([ 00, -01, -01,  01,  00,  00, -01,  00,  01,  00,  00, -01,  00,  01, -01],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01]);
  check([ 00,  01,  00,  00,  00, -01,  00, -01, -01, -01, -01, -01, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  01]);
  check([-01,  01, -01,  00,  01,  01,  01,  01, -01,  00,  00,  01,  01,  01,  01],
          [-01, -01, -01,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01,  01,  00,  00,  00,  00,  01,  00, -01, -01,  01, -01,  00,  01],
          [-01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01]);
  check([ 00,  00,  01,  01, -01,  01,  01, -01,  01,  01,  01, -01,  01,  00, -01],
          [-01, -01, -01, -01,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01, -01,  00,  00,  00,  01,  00,  00,  00,  00,  01, -01,  00, -01],
          [-01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01]);
  check([ 00, -01, -01, -01, -01,  00,  01, -01,  00,  00,  01,  00,  01, -01,  00],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  01,  01,  01]);
  check([ 01, -01, -01, -01,  00,  01,  01,  01, -01,  00,  00,  01, -01,  01,  00],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  01, -01,  00, -01,  00,  00, -01, -01, -01, -01,  01,  01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  01,  01,  01,  01]);
  check([-01, -01, -01,  00,  00, -01, -01,  00, -01,  00,  01,  00,  00,  00,  00],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01]);


//length == 15
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15],
          [-20, -19, -16, -10, -06,  00,  02,  03,  08,  11,  12,  14,  15,  21,  23]);
  check([ 03, -14,  17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23],
          [-25, -24, -21, -14, -07,  01,  03,  04,  05,  06,  06,  15,  17,  22,  23]);
  check([-17,  15, -22,  18, -19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06],
          [-22, -19, -17, -11, -10, -05,  04,  06,  10,  13,  15,  15,  17,  17,  18]);
  check([-20, -06,  19,  22, -07,  01,  03, -07, -20,  10, -25,  08,  11, -05, -02],
          [-25, -20, -20, -07, -07, -06, -05, -02,  01,  03,  08,  10,  11,  19,  22]);
  check([ 10,  03, -11,  15, -03, -03,  08, -22,  14, -22,  14,  06,  20, -22, -05],
          [-22, -22, -22, -11, -05, -03, -03,  03,  06,  08,  10,  14,  14,  15,  20]);
  check([ 02, -15,  16, -04, -16, -16,  22,  15,  04, -21,  00,  06,  12, -14, -24],
          [-24, -21, -16, -16, -15, -14, -04,  00,  02,  04,  06,  12,  15,  16,  22]);
  check([-12,  23,  07,  02,  14, -19,  24, -01, -16,  13,  02, -25, -04,  00, -19],
          [-25, -19, -19, -16, -12, -04, -01,  00,  02,  02,  07,  13,  14,  23,  24]);
  check([ 16, -21, -09,  09,  00,  01,  20,  00, -09,  01,  06,  18, -18, -06, -19],
          [-21, -19, -18, -09, -09, -06,  00,  00,  01,  01,  06,  09,  16,  18,  20]);
  check([-14,  07, -20,  20, -14,  21, -24, -13, -05, -13, -23,  24, -12, -02, -01],
          [-24, -23, -20, -14, -14, -13, -13, -12, -05, -02, -01,  07,  20,  21,  24]);
  check([-06, -08,  03,  12, -22, -19, -12,  24,  08,  06,  00, -11,  01,  10,  10],
          [-22, -19, -12, -11, -08, -06,  00,  01,  03,  06,  08,  10,  10,  12,  24]);


//length == 16
  check([ 00, -01, -01,  01,  00,  00, -01,  00,  01,  00,  00, -01,  00,  01, -01,  00],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01]);
  check([ 01,  00,  00,  00, -01,  00, -01, -01, -01, -01, -01, -01, -01, -01, -01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  01,  01]);
  check([-01,  00,  01,  01,  01,  01, -01,  00,  00,  01,  01,  01,  01, -01,  01,  01],
          [-01, -01, -01,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  00,  00,  01,  00, -01, -01,  01, -01,  00,  01,  00,  00,  01,  01],
          [-01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01]);
  check([-01,  01,  01, -01,  01,  01,  01, -01,  01,  00, -01,  01,  01, -01,  00,  00],
          [-01, -01, -01, -01, -01,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  00,  00,  00,  00,  01, -01,  00, -01,  00, -01, -01, -01, -01,  00],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01]);
  check([ 01, -01,  00,  00,  01,  00,  01, -01,  00,  01, -01, -01, -01,  00,  01,  01],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  00,  00,  01, -01,  01,  00,  01, -01,  01, -01,  00, -01,  00,  00],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01]);
  check([-01, -01, -01, -01,  01,  01, -01, -01, -01, -01,  00,  00, -01, -01,  00, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  01,  01]);
  check([ 00,  01,  00,  00,  00,  00, -01,  00,  01, -01,  01,  00,  00, -01,  01,  01],
          [-01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01]);


//length == 16
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15,  03],
          [-20, -19, -16, -10, -06,  00,  02,  03,  03,  08,  11,  12,  14,  15,  21,  23]);
  check([-14,  17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23, -17,  15],
          [-25, -24, -21, -17, -14, -07,  01,  04,  05,  06,  06,  15,  15,  17,  22,  23]);
  check([-22,  18, -19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06, -20, -06,  19],
          [-22, -20, -19, -11, -10, -06, -05,  04,  06,  10,  13,  15,  17,  17,  18,  19]);
  check([ 22, -07,  01,  03, -07, -20,  10, -25,  08,  11, -05, -02,  10,  03, -11,  15],
          [-25, -20, -11, -07, -07, -05, -02,  01,  03,  03,  08,  10,  10,  11,  15,  22]);
  check([-03, -03,  08, -22,  14, -22,  14,  06,  20, -22, -05,  02, -15,  16, -04, -16],
          [-22, -22, -22, -16, -15, -05, -04, -03, -03,  02,  06,  08,  14,  14,  16,  20]);
  check([-16,  22,  15,  04, -21,  00,  06,  12, -14, -24, -12,  23,  07,  02,  14, -19],
          [-24, -21, -19, -16, -14, -12,  00,  02,  04,  06,  07,  12,  14,  15,  22,  23]);
  check([ 24, -01, -16,  13,  02, -25, -04,  00, -19,  16, -21, -09,  09,  00,  01,  20],
          [-25, -21, -19, -16, -09, -04, -01,  00,  00,  01,  02,  09,  13,  16,  20,  24]);
  check([ 00, -09,  01,  06,  18, -18, -06, -19, -14,  07, -20,  20, -14,  21, -24, -13],
          [-24, -20, -19, -18, -14, -14, -13, -09, -06,  00,  01,  06,  07,  18,  20,  21]);
  check([-05, -13, -23,  24, -12, -02, -01, -06, -08,  03,  12, -22, -19, -12,  24,  08],
          [-23, -22, -19, -13, -12, -12, -08, -06, -05, -02, -01,  03,  08,  12,  24,  24]);
  check([ 06,  00, -11,  01,  10,  10,  08,  23, -06,  15,  20,  07,  13,  21,  19, -15],
          [-15, -11, -06,  00,  01,  06,  07,  08,  10,  10,  13,  15,  19,  20,  21,  23]);


//length == 17
  check([ 00, -01, -01,  01,  00,  00, -01,  00,  01,  00,  00, -01,  00,  01, -01,  00,  01],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01]);
  check([ 00,  00,  00, -01,  00, -01, -01, -01, -01, -01, -01, -01, -01, -01,  01, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  01]);
  check([ 01,  01,  01,  01, -01,  00,  00,  01,  01,  01,  01, -01,  01,  01,  00,  00,  00],
          [-01, -01,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  00, -01, -01,  01, -01,  00,  01,  00,  00,  01,  01, -01,  01,  01, -01],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01,  01, -01,  01,  00, -01,  01,  01, -01,  00,  00,  00,  01,  00,  00,  00],
          [-01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01, -01,  00, -01,  00, -01, -01, -01, -01,  00,  01, -01,  00,  00,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01]);
  check([ 01, -01,  00,  01, -01, -01, -01,  00,  01,  01,  01, -01,  00,  00,  01, -01,  01],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01, -01,  01, -01,  00, -01,  00,  00, -01, -01, -01, -01,  01,  01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  01,  01,  01,  01]);
  check([-01, -01,  00,  00, -01, -01,  00, -01,  00,  01,  00,  00,  00,  00, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01]);
  check([-01,  01,  00,  00, -01,  01,  01,  00, -01,  01, -01, -01,  00,  00,  00,  01, -01],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01]);


//length == 17
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15,  03, -14],
          [-20, -19, -16, -14, -10, -06,  00,  02,  03,  03,  08,  11,  12,  14,  15,  21,  23]);
  check([ 17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23, -17,  15, -22,  18],
          [-25, -24, -22, -21, -17, -07,  01,  04,  05,  06,  06,  15,  15,  17,  18,  22,  23]);
  check([-19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06, -20, -06,  19,  22, -07,  01],
          [-20, -19, -11, -10, -07, -06, -05,  01,  04,  06,  10,  13,  15,  17,  17,  19,  22]);
  check([ 03, -07, -20,  10, -25,  08,  11, -05, -02,  10,  03, -11,  15, -03, -03,  08, -22],
          [-25, -22, -20, -11, -07, -05, -03, -03, -02,  03,  03,  08,  08,  10,  10,  11,  15]);
  check([ 14, -22,  14,  06,  20, -22, -05,  02, -15,  16, -04, -16, -16,  22,  15,  04, -21],
          [-22, -22, -21, -16, -16, -15, -05, -04,  02,  04,  06,  14,  14,  15,  16,  20,  22]);
  check([ 00,  06,  12, -14, -24, -12,  23,  07,  02,  14, -19,  24, -01, -16,  13,  02, -25],
          [-25, -24, -19, -16, -14, -12, -01,  00,  02,  02,  06,  07,  12,  13,  14,  23,  24]);
  check([-04,  00, -19,  16, -21, -09,  09,  00,  01,  20,  00, -09,  01,  06,  18, -18, -06],
          [-21, -19, -18, -09, -09, -06, -04,  00,  00,  00,  01,  01,  06,  09,  16,  18,  20]);
  check([-19, -14,  07, -20,  20, -14,  21, -24, -13, -05, -13, -23,  24, -12, -02, -01, -06],
          [-24, -23, -20, -19, -14, -14, -13, -13, -12, -06, -05, -02, -01,  07,  20,  21,  24]);
  check([-08,  03,  12, -22, -19, -12,  24,  08,  06,  00, -11,  01,  10,  10,  08,  23, -06],
          [-22, -19, -12, -11, -08, -06,  00,  01,  03,  06,  08,  08,  10,  10,  12,  23,  24]);
  check([ 15,  20,  07,  13,  21,  19, -15, -15,  23, -16, -02, -03,  11,  19,  16, -09,  07],
          [-16, -15, -15, -09, -03, -02,  07,  07,  11,  13,  15,  16,  19,  19,  20,  21,  23]);


//length == 31
  check([ 00, -01, -01,  01,  00,  00, -01,  00,  01,  00,  00, -01,  00,  01, -01,  00,  01,  00,  00,  00, -01,  00, -01, -01, -01, -01, -01, -01, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01]);
  check([ 01, -01,  00,  01,  01,  01,  01, -01,  00,  00,  01,  01,  01,  01, -01,  01,  01,  00,  00,  00,  00,  01,  00, -01, -01,  01, -01,  00,  01,  00,  00],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01, -01,  01,  01, -01,  01,  01,  01, -01,  01,  00, -01,  01,  01, -01,  00,  00,  00,  01,  00,  00,  00,  00,  01, -01,  00, -01,  00, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01,  00,  01, -01,  00,  00,  01,  00,  01, -01,  00,  01, -01, -01, -01,  00,  01,  01,  01, -01,  00,  00,  01, -01,  01,  00,  01, -01,  01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00, -01,  00,  00, -01, -01, -01, -01,  01,  01, -01, -01, -01, -01,  00,  00, -01, -01,  00, -01,  00,  01,  00,  00,  00,  00, -01,  00,  01, -01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01]);
  check([ 00,  00, -01,  01,  01,  00, -01,  01, -01, -01,  00,  00,  00,  01, -01,  01, -01,  00,  00, -01,  00,  00, -01,  01, -01,  01, -01,  01, -01,  00,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  01,  01,  00, -01,  01,  01,  01, -01, -01, -01,  00,  01, -01,  01,  00, -01, -01,  00,  00,  01,  01,  00, -01, -01,  01,  01, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00, -01, -01,  01,  00, -01,  00, -01, -01,  01,  00, -01, -01,  01,  01,  00, -01, -01, -01,  00, -01, -01, -01, -01,  00, -01, -01,  00,  00,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01]);
  check([ 01, -01,  01, -01,  01,  00,  00,  01,  00,  01,  00,  01, -01,  00,  01,  01, -01,  01,  00,  00,  00,  00,  01, -01,  00, -01, -01,  01,  00,  01,  01],
          [-01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01,  00,  00, -01, -01, -01,  01,  00, -01,  00,  00,  00,  00,  01,  01,  00,  01, -01,  01,  01,  00,  01,  01, -01,  01,  01,  00, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01,  01,  00,  01,  01,  01,  01,  00, -01,  01,  01,  00, -01,  01, -01,  01,  01, -01,  01,  00,  01,  00,  00, -01,  01,  00,  00,  01,  00,  01],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  00,  00,  00,  00, -01,  00,  01,  01,  00,  00,  00, -01,  00,  01,  01, -01,  00,  01, -01, -01, -01, -01,  00, -01,  01,  01,  01,  01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01, -01,  00,  01, -01,  01,  00,  01,  00,  01,  00,  01,  00,  00,  00,  01,  01,  01, -01, -01,  01, -01,  01, -01, -01, -01, -01, -01,  01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01,  01,  01,  01, -01,  00,  00, -01, -01, -01,  01,  01,  01,  00,  00,  01, -01,  00, -01,  01, -01,  01,  00,  00,  01,  01,  00, -01,  00, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  01, -01,  00, -01,  01, -01, -01,  01, -01,  01, -01, -01, -01,  01, -01,  01, -01,  00,  00,  00,  00, -01,  01,  00, -01,  00,  01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  00, -01, -01,  00,  01, -01,  00, -01,  00,  00,  00,  01,  01, -01, -01,  00, -01,  01, -01,  01, -01,  00,  00,  01,  00,  01,  01,  00,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00, -01, -01,  00, -01,  01,  00,  00, -01,  01,  01,  00, -01,  01,  00, -01, -01,  01,  00, -01, -01, -01,  01,  01,  01,  00,  00,  01,  00, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  00,  00,  00, -01, -01,  01,  00,  01, -01, -01, -01,  01,  01, -01, -01,  01,  01,  01,  00, -01,  00, -01,  01,  01,  00,  01,  00, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01,  01,  01, -01,  01, -01, -01,  00,  01, -01,  00,  01,  00,  00, -01,  00,  01, -01, -01,  00, -01,  00,  00, -01,  00, -01, -01, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01, -01,  00,  01,  01, -01, -01,  00,  01,  00,  00, -01,  01, -01,  00,  00,  01,  01, -01, -01, -01,  01,  01,  01,  00,  00,  01,  01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  00,  00,  00,  01,  00,  01, -01,  00,  00,  00,  01, -01,  01,  00,  01,  01,  00,  01,  00,  00,  01, -01, -01,  00,  01,  00, -01,  01,  01],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  00, -01,  00, -01,  01, -01, -01, -01,  01,  00,  01,  01,  01,  00,  00,  00,  01,  01,  01, -01,  00,  00,  00, -01,  01,  00,  00, -01,  00, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01,  00,  01,  01,  00, -01, -01,  01, -01,  01,  00,  01,  00,  00, -01,  00, -01, -01, -01,  00,  00,  01, -01,  01,  00,  00, -01,  01, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01, -01, -01,  01,  00,  00,  00, -01,  00, -01,  01,  00, -01, -01,  01,  00,  00,  01,  01, -01,  00,  01,  00,  00,  00, -01,  00, -01,  00,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  00,  01,  00, -01, -01, -01,  00,  01,  00, -01, -01,  00,  01,  01, -01,  00,  00, -01, -01,  01,  01,  00,  00, -01,  01, -01,  00,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01, -01,  01, -01, -01, -01, -01,  00,  01,  01,  01,  01,  00,  00,  01,  01,  00,  00,  01,  00,  00,  01, -01, -01, -01,  01,  00,  01,  01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01, -01,  01,  01,  00,  01,  00,  01,  00, -01,  01,  00,  01,  01, -01,  00, -01,  01,  01,  01, -01,  01,  01, -01, -01,  01,  00, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  00,  01,  01,  00, -01,  00,  01, -01, -01,  00,  00,  00,  00, -01,  01,  01,  00,  00,  01, -01,  00,  01,  01, -01, -01,  00,  01, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01,  01, -01,  01,  00,  00, -01,  01,  00,  00, -01, -01,  01, -01, -01,  01,  00,  01,  01,  00,  00,  01,  00,  00, -01,  00,  01,  00,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01, -01,  01,  01,  00, -01,  01,  00,  01, -01,  00, -01, -01, -01,  00, -01,  01, -01, -01,  00,  00, -01, -01,  00, -01,  00, -01,  00,  00,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01]);


//length == 31
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15,  03, -14,  17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23, -17],
          [-25, -24, -21, -20, -19, -17, -16, -14, -10, -07, -06,  00,  01,  02,  03,  03,  04,  05,  06,  06,  08,  11,  12,  14,  15,  15,  17,  21,  22,  23,  23]);
  check([ 15, -22,  18, -19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06, -20, -06,  19,  22, -07,  01,  03, -07, -20,  10, -25,  08,  11, -05, -02,  10,  03],
          [-25, -22, -20, -20, -19, -11, -10, -07, -07, -06, -05, -05, -02,  01,  03,  03,  04,  06,  08,  10,  10,  10,  11,  13,  15,  15,  17,  17,  18,  19,  22]);
  check([-11,  15, -03, -03,  08, -22,  14, -22,  14,  06,  20, -22, -05,  02, -15,  16, -04, -16, -16,  22,  15,  04, -21,  00,  06,  12, -14, -24, -12,  23,  07],
          [-24, -22, -22, -22, -21, -16, -16, -15, -14, -12, -11, -05, -04, -03, -03,  00,  02,  04,  06,  06,  07,  08,  12,  14,  14,  15,  15,  16,  20,  22,  23]);
  check([ 02,  14, -19,  24, -01, -16,  13,  02, -25, -04,  00, -19,  16, -21, -09,  09,  00,  01,  20,  00, -09,  01,  06,  18, -18, -06, -19, -14,  07, -20,  20],
          [-25, -21, -20, -19, -19, -19, -18, -16, -14, -09, -09, -06, -04, -01,  00,  00,  00,  01,  01,  02,  02,  06,  07,  09,  13,  14,  16,  18,  20,  20,  24]);
  check([-14,  21, -24, -13, -05, -13, -23,  24, -12, -02, -01, -06, -08,  03,  12, -22, -19, -12,  24,  08,  06,  00, -11,  01,  10,  10,  08,  23, -06,  15,  20],
          [-24, -23, -22, -19, -14, -13, -13, -12, -12, -11, -08, -06, -06, -05, -02, -01,  00,  01,  03,  06,  08,  08,  10,  10,  12,  15,  20,  21,  23,  24,  24]);
  check([ 07,  13,  21,  19, -15, -15,  23, -16, -02, -03,  11,  19,  16, -09,  07,  19, -01,  22,  20, -18,  06, -04, -01, -18, -17, -14,  17, -17,  08,  09, -22],
          [-22, -18, -18, -17, -17, -16, -15, -15, -14, -09, -04, -03, -02, -01, -01,  06,  07,  07,  08,  09,  11,  13,  16,  17,  19,  19,  19,  20,  21,  22,  23]);
  check([-10, -03, -24, -13,  07, -14, -15,  19,  12,  09,  05,  06,  00,  21, -12, -03, -02, -12, -08,  05,  19,  15, -19, -22, -25, -05,  21, -17, -22,  05, -14],
          [-25, -24, -22, -22, -19, -17, -15, -14, -14, -13, -12, -12, -10, -08, -05, -03, -03, -02,  00,  05,  05,  05,  06,  07,  09,  12,  15,  19,  19,  21,  21]);
  check([-07, -20, -11,  05,  12,  01,  16,  06, -12,  00, -12,  21, -22, -13, -15,  00,  10,  00,  19, -08, -06,  10, -01, -01,  10,  21, -03,  19,  24, -22, -18],
          [-22, -22, -20, -18, -15, -13, -12, -12, -11, -08, -07, -06, -03, -01, -01,  00,  00,  00,  01,  05,  06,  10,  10,  10,  12,  16,  19,  19,  21,  21,  24]);
  check([-08, -15, -03,  23,  22, -25,  16,  06, -12, -07,  19, -16, -02, -19, -04,  23,  18,  21,  20, -14, -08,  05,  12,  19, -10,  08, -09, -14,  07, -04, -04],
          [-25, -19, -16, -15, -14, -14, -12, -10, -09, -08, -08, -07, -04, -04, -04, -03, -02,  05,  06,  07,  08,  12,  16,  18,  19,  19,  20,  21,  22,  23,  23]);
  check([ 24,  06,  18,  24,  06,  20,  17, -13, -17, -14, -17, -06,  11, -08,  18,  12, -15,  16,  07, -04, -14, -13, -15, -20,  04,  18, -04,  18,  00,  20, -11],
          [-20, -17, -17, -15, -15, -14, -14, -13, -13, -11, -08, -06, -04, -04,  00,  04,  06,  06,  07,  11,  12,  16,  17,  18,  18,  18,  18,  20,  20,  24,  24]);
  check([ 02,  01, -16,  03, -18, -21, -04, -04, -11,  09,  07,  08, -05,  24,  03, -18, -14, -05,  14,  08,  06, -22,  18,  13,  08,  14, -14,  01, -11, -19,  17],
          [-22, -21, -19, -18, -18, -16, -14, -14, -11, -11, -05, -05, -04, -04,  01,  01,  02,  03,  03,  06,  07,  08,  08,  08,  09,  13,  14,  14,  17,  18,  24]);
  check([ 18,  08, -24, -03,  17, -20,  20, -09, -04,  04,  23, -21, -24,  22,  10, -14, -17, -20, -25,  18,  13, -19,  14,  01,  14,  03,  12, -08,  19,  21, -16],
          [-25, -24, -24, -21, -20, -20, -19, -17, -16, -14, -09, -08, -04, -03,  01,  03,  04,  08,  10,  12,  13,  14,  14,  17,  18,  18,  19,  20,  21,  22,  23]);
  check([ 13,  04, -13, -13, -02, -08, -18,  14,  13,  13,  12,  19,  16,  11,  06, -23,  20,  11, -20,  15,  01, -14,  04, -21, -23, -16,  19, -04, -21,  15,  07],
          [-23, -23, -21, -21, -20, -18, -16, -14, -13, -13, -08, -04, -02,  01,  04,  04,  06,  07,  11,  11,  12,  13,  13,  13,  14,  15,  15,  16,  19,  19,  20]);
  check([ 19, -04,  19, -17,  19, -12, -10,  09, -22, -20, -04,  24,  21, -17,  06,  24, -20, -08, -19,  22,  21, -08,  04, -25, -04,  13,  21,  17,  19,  11, -01],
          [-25, -22, -20, -20, -19, -17, -17, -12, -10, -08, -08, -04, -04, -04, -01,  04,  06,  09,  11,  13,  17,  19,  19,  19,  19,  21,  21,  21,  22,  24,  24]);
  check([ 15, -18, -07,  23, -23,  08, -10,  13,  11, -05, -16,  10, -06, -06, -09, -05,  01,  11,  01, -01,  07,  20, -20,  09, -09,  20,  06, -15,  15, -06,  09],
          [-23, -20, -18, -16, -15, -10, -09, -09, -07, -06, -06, -06, -05, -05, -01,  01,  01,  06,  07,  08,  09,  09,  10,  11,  11,  13,  15,  15,  20,  20,  23]);
  check([ 07,  02, -21,  08,  06,  12,  23, -06, -02, -04,  05, -15,  17, -25,  02, -13,  03, -12,  13, -21,  22,  10, -16,  08, -24, -20, -11, -14,  22,  09,  22],
          [-25, -24, -21, -21, -20, -16, -15, -14, -13, -12, -11, -06, -04, -02,  02,  02,  03,  05,  06,  07,  08,  08,  09,  10,  12,  13,  17,  22,  22,  22,  23]);
  check([ 06, -12, -22, -11,  19,  17, -12, -10, -10,  11,  20,  00,  03,  22, -21,  15,  01, -08,  05,  07, -09, -10,  19, -25, -07,  24, -09,  06,  23, -23,  05],
          [-25, -23, -22, -21, -12, -12, -11, -10, -10, -10, -09, -09, -08, -07,  00,  01,  03,  05,  05,  06,  06,  07,  11,  15,  17,  19,  19,  20,  22,  23,  24]);
  check([ 04, -10,  08,  19, -14,  00, -16,  01,  17,  20, -01, -06,  00, -04, -01, -10,  24,  18,  22,  09, -13, -11,  03, -13,  07,  02,  03, -10,  02,  08,  20],
          [-16, -14, -13, -13, -11, -10, -10, -10, -06, -04, -01, -01,  00,  00,  01,  02,  02,  03,  03,  04,  07,  08,  08,  09,  17,  18,  19,  20,  20,  22,  24]);
  check([-19,  23,  05, -23, -13, -20, -12,  15,  24, -17, -11, -07,  08,  13,  19, -25,  12,  15,  24, -04, -21, -12,  24, -09,  22,  03,  21, -11, -18,  04, -16],
          [-25, -23, -21, -20, -19, -18, -17, -16, -13, -12, -12, -11, -11, -09, -07, -04,  03,  04,  05,  08,  12,  13,  15,  15,  19,  21,  22,  23,  24,  24,  24]);
  check([-09,  05,  16, -07,  17,  21,  09,  09,  22,  17,  24, -07,  03,  12, -11,  05,  01, -21,  05, -25, -17,  20, -24,  01,  20,  05,  00, -16,  14, -19, -04],
          [-25, -24, -21, -19, -17, -16, -11, -09, -07, -07, -04,  00,  01,  01,  03,  05,  05,  05,  05,  09,  09,  12,  14,  16,  17,  17,  20,  20,  21,  22,  24]);
  check([-20,  11, -13,  01,  05, -14, -15, -10, -17, -21, -09,  01,  09, -22,  16, -10,  06,  22,  22,  06, -17,  17,  08,  09,  12, -10, -14,  24, -21, -07, -05],
          [-22, -21, -21, -20, -17, -17, -15, -14, -14, -13, -10, -10, -10, -09, -07, -05,  01,  01,  05,  06,  06,  08,  09,  09,  11,  12,  16,  17,  22,  22,  24]);
  check([-13, -19,  07,  13,  12,  18, -25, -21, -23, -19, -05,  05,  16,  00, -04, -17, -19, -04, -20,  15,  04,  24, -02, -10,  14,  13,  02,  13,  19,  22, -17],
          [-25, -23, -21, -20, -19, -19, -19, -17, -17, -13, -10, -05, -04, -04, -02,  00,  02,  04,  05,  07,  12,  13,  13,  13,  14,  15,  16,  18,  19,  22,  24]);
  check([-19, -22,  17,  21,  15,  11,  23,  21,  15, -20, -07, -05,  21,  18,  19, -21, -23, -10, -14,  17,  21, -15, -08, -12, -24, -20,  15,  14, -24,  14,  24],
          [-24, -24, -23, -22, -21, -20, -20, -19, -15, -14, -12, -10, -08, -07, -05,  11,  14,  14,  15,  15,  15,  17,  17,  18,  19,  21,  21,  21,  21,  23,  24]);
  check([-17,  18,  17, -19,  08,  03, -20,  07, -07, -15, -25,  15, -17,  21,  09, -11,  23, -24,  00,  17,  24,  12, -14, -12,  14, -07, -20,  03, -06,  20,  05],
          [-25, -24, -20, -20, -19, -17, -17, -15, -14, -12, -11, -07, -07, -06,  00,  03,  03,  05,  07,  08,  09,  12,  14,  15,  17,  17,  18,  20,  21,  23,  24]);
  check([ 04,  13, -03,  11, -02, -23,  16, -18, -03, -22, -17, -13, -14, -21,  24,  00, -21, -23, -25,  23, -21,  12, -16, -08,  03,  02, -01, -18,  23, -04,  14],
          [-25, -23, -23, -22, -21, -21, -21, -18, -18, -17, -16, -14, -13, -08, -04, -03, -03, -02, -01,  00,  02,  03,  04,  11,  12,  13,  14,  16,  23,  23,  24]);
  check([ 03, -14, -14, -11,  12, -12, -18,  19,  10, -15, -21,  24, -04, -15,  23,  23, -09, -22, -25, -11, -18,  14,  00,  01, -07, -21, -25,  02, -22, -01, -09],
          [-25, -25, -22, -22, -21, -21, -18, -18, -15, -15, -14, -14, -12, -11, -11, -09, -09, -07, -04, -01,  00,  01,  02,  03,  10,  12,  14,  19,  23,  23,  24]);
  check([ 06,  10,  02,  22, -03,  17, -21, -06,  02, -11, -02,  03,  12,  09,  04,  10, -25,  07,  12, -08,  16,  01,  19, -08,  19,  24, -06, -04, -21,  18,  14],
          [-25, -21, -21, -11, -08, -08, -06, -06, -04, -03, -02,  01,  02,  02,  03,  04,  06,  07,  09,  10,  10,  12,  12,  14,  16,  17,  18,  19,  19,  22,  24]);
  check([ 10,  04, -09,  09, -22, -17,  13, -03,  12, -21,  23, -07,  16,  09,  22,  03,  09,  06, -10, -22, -03,  16,  23,  14,  13,  24, -17, -14, -22, -21, -24],
          [-24, -22, -22, -22, -21, -21, -17, -17, -14, -10, -09, -07, -03, -03,  03,  04,  06,  09,  09,  09,  10,  12,  13,  13,  14,  16,  16,  22,  23,  23,  24]);
  check([ 15,  10, -06,  01,  15,  05, -11, -10, -06, -07, -12,  12, -14,  22,  09,  16,  08, -10, -17,  12,  12, -24,  12,  03,  14,  13,  14, -22,  18,  18, -21],
          [-24, -22, -21, -17, -14, -12, -11, -10, -10, -07, -06, -06,  01,  03,  05,  08,  09,  10,  12,  12,  12,  12,  13,  14,  14,  15,  15,  16,  18,  18,  22]);
  check([ 10,  05, -02, -14, -05, -22,  00,  10, -02,  20,  23, -15, -17,  22,  22, -24,  06, -11, -16, -05, -23, -15, -16,  07, -23,  22, -04, -20,  15, -09, -16],
          [-24, -23, -23, -22, -20, -17, -16, -16, -16, -15, -15, -14, -11, -09, -05, -05, -04, -02, -02,  00,  05,  06,  07,  10,  10,  15,  20,  22,  22,  22,  23]);


//length == 32
  check([ 00, -01, -01,  01,  00,  00, -01,  00,  01,  00,  00, -01,  00,  01, -01,  00,  01,  00,  00,  00, -01,  00, -01, -01, -01, -01, -01, -01, -01, -01, -01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01]);
  check([-01,  00,  01,  01,  01,  01, -01,  00,  00,  01,  01,  01,  01, -01,  01,  01,  00,  00,  00,  00,  01,  00, -01, -01,  01, -01,  00,  01,  00,  00,  01,  01],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01,  01, -01,  01,  01,  01, -01,  01,  00, -01,  01,  01, -01,  00,  00,  00,  01,  00,  00,  00,  00,  01, -01,  00, -01,  00, -01, -01, -01, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  00,  00,  01,  00,  01, -01,  00,  01, -01, -01, -01,  00,  01,  01,  01, -01,  00,  00,  01, -01,  01,  00,  01, -01,  01, -01,  00, -01,  00,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01, -01, -01,  01,  01, -01, -01, -01, -01,  00,  00, -01, -01,  00, -01,  00,  01,  00,  00,  00,  00, -01,  00,  01, -01,  01,  00,  00, -01,  01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00, -01,  01, -01, -01,  00,  00,  00,  01, -01,  01, -01,  00,  00, -01,  00,  00, -01,  01, -01,  01, -01,  01, -01,  00,  00,  00,  01,  01,  01,  00, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01,  01, -01, -01, -01,  00,  01, -01,  01,  00, -01, -01,  00,  00,  01,  01,  00, -01, -01,  01,  01, -01, -01, -01,  00,  00, -01, -01,  01,  00, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00, -01, -01,  01,  00, -01, -01,  01,  01,  00, -01, -01, -01,  00, -01, -01, -01, -01,  00, -01, -01,  00,  00,  00,  01, -01,  01, -01,  01,  00,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  00,  01, -01,  00,  01,  01, -01,  01,  00,  00,  00,  00,  01, -01,  00, -01, -01,  01,  00,  01,  01, -01,  01,  00,  00, -01, -01, -01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  00,  00,  00,  00,  01,  01,  00,  01, -01,  01,  01,  00,  01,  01, -01,  01,  01,  00, -01, -01, -01,  01,  01,  01,  00,  01,  01,  01,  01,  00, -01],
          [-01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01,  00, -01,  01, -01,  01,  01, -01,  01,  00,  01,  00,  00, -01,  01,  00,  00,  01,  00,  01,  01,  00,  00,  00,  00, -01,  00,  01,  01,  00,  00],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00, -01,  00,  01,  01, -01,  00,  01, -01, -01, -01, -01,  00, -01,  01,  01,  01,  01,  01,  00,  01, -01, -01,  00,  01, -01,  01,  00,  01,  00,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  00,  00,  00,  01,  01,  01, -01, -01,  01, -01,  01, -01, -01, -01, -01, -01,  01,  01, -01, -01,  01,  01,  01, -01,  00,  00, -01, -01, -01,  01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  00,  00,  01, -01,  00, -01,  01, -01,  01,  00,  00,  01,  01,  00, -01,  00, -01,  00,  00,  01, -01,  00, -01,  01, -01, -01,  01, -01,  01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01, -01,  01, -01,  00,  00,  00,  00, -01,  01,  00, -01,  00,  01,  01,  00,  00,  00,  00, -01, -01,  00,  01, -01,  00, -01,  00,  00,  00,  01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01,  00, -01,  01, -01,  01, -01,  00,  00,  01,  00,  01,  01,  00,  00,  00, -01, -01,  00, -01,  01,  00,  00, -01,  01,  01,  00, -01,  01,  00, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01,  00, -01, -01, -01,  01,  01,  01,  00,  00,  01,  00, -01,  00,  01, -01,  00,  00,  00, -01, -01,  01,  00,  01, -01, -01, -01,  01,  01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01,  01,  00, -01,  00, -01,  01,  01,  00,  01,  00, -01, -01, -01, -01,  01,  01, -01,  01, -01, -01,  00,  01, -01,  00,  01,  00,  00, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01,  00, -01,  00,  00, -01,  00, -01, -01, -01, -01, -01,  01,  01, -01,  00,  01,  01, -01, -01,  00,  01,  00,  00, -01,  01, -01,  00,  00,  01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01, -01,  01,  01,  01,  00,  00,  01,  01,  01,  00,  00,  00,  00,  00,  00,  01,  00,  01, -01,  00,  00,  00,  01, -01,  01,  00,  01,  01,  00,  01],
          [-01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  01, -01, -01,  00,  01,  00, -01,  01,  01, -01,  00, -01,  00, -01,  01, -01, -01, -01,  01,  00,  01,  01,  01,  00,  00,  00,  01,  01,  01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  00, -01,  01,  00,  00, -01,  00, -01, -01,  01,  00,  01,  01,  00, -01, -01,  01, -01,  01,  00,  01,  00,  00, -01,  00, -01, -01, -01,  00,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  01,  00,  00, -01,  01, -01,  00, -01, -01, -01, -01,  01,  00,  00,  00, -01,  00, -01,  01,  00, -01, -01,  01,  00,  00,  01,  01, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  00, -01,  00, -01,  00,  00,  00,  00,  00,  01,  00, -01, -01, -01,  00,  01,  00, -01, -01,  00,  01,  01, -01,  00,  00, -01, -01,  01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01]);
  check([ 00, -01,  01, -01,  00,  00,  01, -01, -01, -01,  01, -01, -01, -01, -01,  00,  01,  01,  01,  01,  00,  00,  01,  01,  00,  00,  01,  00,  00,  01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01,  00,  01,  01, -01,  01,  01, -01,  01,  01,  00,  01,  00,  01,  00, -01,  01,  00,  01,  01, -01,  00, -01,  01,  01,  01, -01,  01,  01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  00, -01,  00,  01,  01,  00,  01,  01,  00, -01,  00,  01, -01, -01,  00,  00,  00,  00, -01,  01,  01,  00,  00,  01, -01,  00,  01,  01, -01, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  00,  01, -01, -01,  01, -01,  01,  00,  00, -01,  01,  00,  00, -01, -01,  01, -01, -01,  01,  00,  01,  01,  00,  00,  01,  00,  00, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  00,  01, -01, -01,  01,  01,  00, -01,  01,  00,  01, -01,  00, -01, -01, -01,  00, -01,  01, -01, -01,  00,  00, -01, -01,  00, -01,  00, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00, -01,  00,  01, -01,  00,  00, -01,  00, -01, -01, -01,  01,  00, -01, -01,  01,  00, -01,  01,  00,  01,  00,  00,  01,  01,  01,  00,  01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);


//length == 32
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15,  03, -14,  17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23, -17,  15],
          [-25, -24, -21, -20, -19, -17, -16, -14, -10, -07, -06,  00,  01,  02,  03,  03,  04,  05,  06,  06,  08,  11,  12,  14,  15,  15,  15,  17,  21,  22,  23,  23]);
  check([-22,  18, -19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06, -20, -06,  19,  22, -07,  01,  03, -07, -20,  10, -25,  08,  11, -05, -02,  10,  03, -11,  15],
          [-25, -22, -20, -20, -19, -11, -11, -10, -07, -07, -06, -05, -05, -02,  01,  03,  03,  04,  06,  08,  10,  10,  10,  11,  13,  15,  15,  17,  17,  18,  19,  22]);
  check([-03, -03,  08, -22,  14, -22,  14,  06,  20, -22, -05,  02, -15,  16, -04, -16, -16,  22,  15,  04, -21,  00,  06,  12, -14, -24, -12,  23,  07,  02,  14, -19],
          [-24, -22, -22, -22, -21, -19, -16, -16, -15, -14, -12, -05, -04, -03, -03,  00,  02,  02,  04,  06,  06,  07,  08,  12,  14,  14,  14,  15,  16,  20,  22,  23]);
  check([ 24, -01, -16,  13,  02, -25, -04,  00, -19,  16, -21, -09,  09,  00,  01,  20,  00, -09,  01,  06,  18, -18, -06, -19, -14,  07, -20,  20, -14,  21, -24, -13],
          [-25, -24, -21, -20, -19, -19, -18, -16, -14, -14, -13, -09, -09, -06, -04, -01,  00,  00,  00,  01,  01,  02,  06,  07,  09,  13,  16,  18,  20,  20,  21,  24]);
  check([-05, -13, -23,  24, -12, -02, -01, -06, -08,  03,  12, -22, -19, -12,  24,  08,  06,  00, -11,  01,  10,  10,  08,  23, -06,  15,  20,  07,  13,  21,  19, -15],
          [-23, -22, -19, -15, -13, -12, -12, -11, -08, -06, -06, -05, -02, -01,  00,  01,  03,  06,  07,  08,  08,  10,  10,  12,  13,  15,  19,  20,  21,  23,  24,  24]);
  check([-15,  23, -16, -02, -03,  11,  19,  16, -09,  07,  19, -01,  22,  20, -18,  06, -04, -01, -18, -17, -14,  17, -17,  08,  09, -22, -10, -03, -24, -13,  07, -14],
          [-24, -22, -18, -18, -17, -17, -16, -15, -14, -14, -13, -10, -09, -04, -03, -03, -02, -01, -01,  06,  07,  07,  08,  09,  11,  16,  17,  19,  19,  20,  22,  23]);
  check([-15,  19,  12,  09,  05,  06,  00,  21, -12, -03, -02, -12, -08,  05,  19,  15, -19, -22, -25, -05,  21, -17, -22,  05, -14, -07, -20, -11,  05,  12,  01,  16],
          [-25, -22, -22, -20, -19, -17, -15, -14, -12, -12, -11, -08, -07, -05, -03, -02,  00,  01,  05,  05,  05,  05,  06,  09,  12,  12,  15,  16,  19,  19,  21,  21]);
  check([ 06, -12,  00, -12,  21, -22, -13, -15,  00,  10,  00,  19, -08, -06,  10, -01, -01,  10,  21, -03,  19,  24, -22, -18, -08, -15, -03,  23,  22, -25,  16,  06],
          [-25, -22, -22, -18, -15, -15, -13, -12, -12, -08, -08, -06, -03, -03, -01, -01,  00,  00,  00,  06,  06,  10,  10,  10,  16,  19,  19,  21,  21,  22,  23,  24]);
  check([-12, -07,  19, -16, -02, -19, -04,  23,  18,  21,  20, -14, -08,  05,  12,  19, -10,  08, -09, -14,  07, -04, -04,  24,  06,  18,  24,  06,  20,  17, -13, -17],
          [-19, -17, -16, -14, -14, -13, -12, -10, -09, -08, -07, -04, -04, -04, -02,  05,  06,  06,  07,  08,  12,  17,  18,  18,  19,  19,  20,  20,  21,  23,  24,  24]);
  check([-14, -17, -06,  11, -08,  18,  12, -15,  16,  07, -04, -14, -13, -15, -20,  04,  18, -04,  18,  00,  20, -11,  02,  01, -16,  03, -18, -21, -04, -04, -11,  09],
          [-21, -20, -18, -17, -16, -15, -15, -14, -14, -13, -11, -11, -08, -06, -04, -04, -04, -04,  00,  01,  02,  03,  04,  07,  09,  11,  12,  16,  18,  18,  18,  20]);
  check([ 07,  08, -05,  24,  03, -18, -14, -05,  14,  08,  06, -22,  18,  13,  08,  14, -14,  01, -11, -19,  17,  18,  08, -24, -03,  17, -20,  20, -09, -04,  04,  23],
          [-24, -22, -20, -19, -18, -14, -14, -11, -09, -05, -05, -04, -03,  01,  03,  04,  06,  07,  08,  08,  08,  08,  13,  14,  14,  17,  17,  18,  18,  20,  23,  24]);
  check([-21, -24,  22,  10, -14, -17, -20, -25,  18,  13, -19,  14,  01,  14,  03,  12, -08,  19,  21, -16,  13,  04, -13, -13, -02, -08, -18,  14,  13,  13,  12,  19],
          [-25, -24, -21, -20, -19, -18, -17, -16, -14, -13, -13, -08, -08, -02,  01,  03,  04,  10,  12,  12,  13,  13,  13,  13,  14,  14,  14,  18,  19,  19,  21,  22]);
  check([ 16,  11,  06, -23,  20,  11, -20,  15,  01, -14,  04, -21, -23, -16,  19, -04, -21,  15,  07,  19, -04,  19, -17,  19, -12, -10,  09, -22, -20, -04,  24,  21],
          [-23, -23, -22, -21, -21, -20, -20, -17, -16, -14, -12, -10, -04, -04, -04,  01,  04,  06,  07,  09,  11,  11,  15,  15,  16,  19,  19,  19,  19,  20,  21,  24]);
  check([-17,  06,  24, -20, -08, -19,  22,  21, -08,  04, -25, -04,  13,  21,  17,  19,  11, -01,  15, -18, -07,  23, -23,  08, -10,  13,  11, -05, -16,  10, -06, -06],
          [-25, -23, -20, -19, -18, -17, -16, -10, -08, -08, -07, -06, -06, -05, -04, -01,  04,  06,  08,  10,  11,  11,  13,  13,  15,  17,  19,  21,  21,  22,  23,  24]);
  check([-09, -05,  01,  11,  01, -01,  07,  20, -20,  09, -09,  20,  06, -15,  15, -06,  09,  07,  02, -21,  08,  06,  12,  23, -06, -02, -04,  05, -15,  17, -25,  02],
          [-25, -21, -20, -15, -15, -09, -09, -06, -06, -05, -04, -02, -01,  01,  01,  02,  02,  05,  06,  06,  07,  07,  08,  09,  09,  11,  12,  15,  17,  20,  20,  23]);
  check([-13,  03, -12,  13, -21,  22,  10, -16,  08, -24, -20, -11, -14,  22,  09,  22,  06, -12, -22, -11,  19,  17, -12, -10, -10,  11,  20,  00,  03,  22, -21,  15],
          [-24, -22, -21, -21, -20, -16, -14, -13, -12, -12, -12, -11, -11, -10, -10,  00,  03,  03,  06,  08,  09,  10,  11,  13,  15,  17,  19,  20,  22,  22,  22,  22]);
  check([ 01, -08,  05,  07, -09, -10,  19, -25, -07,  24, -09,  06,  23, -23,  05,  04, -10,  08,  19, -14,  00, -16,  01,  17,  20, -01, -06,  00, -04, -01, -10,  24],
          [-25, -23, -16, -14, -10, -10, -10, -09, -09, -08, -07, -06, -04, -01, -01,  00,  00,  01,  01,  04,  05,  05,  06,  07,  08,  17,  19,  19,  20,  23,  24,  24]);
  check([ 18,  22,  09, -13, -11,  03, -13,  07,  02,  03, -10,  02,  08,  20, -19,  23,  05, -23, -13, -20, -12,  15,  24, -17, -11, -07,  08,  13,  19, -25,  12,  15],
          [-25, -23, -20, -19, -17, -13, -13, -13, -12, -11, -11, -10, -07,  02,  02,  03,  03,  05,  07,  08,  08,  09,  12,  13,  15,  15,  18,  19,  20,  22,  23,  24]);
  check([ 24, -04, -21, -12,  24, -09,  22,  03,  21, -11, -18,  04, -16, -09,  05,  16, -07,  17,  21,  09,  09,  22,  17,  24, -07,  03,  12, -11,  05,  01, -21,  05],
          [-21, -21, -18, -16, -12, -11, -11, -09, -09, -07, -07, -04,  01,  03,  03,  04,  05,  05,  05,  09,  09,  12,  16,  17,  17,  21,  21,  22,  22,  24,  24,  24]);
  check([-25, -17,  20, -24,  01,  20,  05,  00, -16,  14, -19, -04, -20,  11, -13,  01,  05, -14, -15, -10, -17, -21, -09,  01,  09, -22,  16, -10,  06,  22,  22,  06],
          [-25, -24, -22, -21, -20, -19, -17, -17, -16, -15, -14, -13, -10, -10, -09, -04,  00,  01,  01,  01,  05,  05,  06,  06,  09,  11,  14,  16,  20,  20,  22,  22]);
  check([-17,  17,  08,  09,  12, -10, -14,  24, -21, -07, -05, -13, -19,  07,  13,  12,  18, -25, -21, -23, -19, -05,  05,  16,  00, -04, -17, -19, -04, -20,  15,  04],
          [-25, -23, -21, -21, -20, -19, -19, -19, -17, -17, -14, -13, -10, -07, -05, -05, -04, -04,  00,  04,  05,  07,  08,  09,  12,  12,  13,  15,  16,  17,  18,  24]);
  check([ 24, -02, -10,  14,  13,  02,  13,  19,  22, -17, -19, -22,  17,  21,  15,  11,  23,  21,  15, -20, -07, -05,  21,  18,  19, -21, -23, -10, -14,  17,  21, -15],
          [-23, -22, -21, -20, -19, -17, -15, -14, -10, -10, -07, -05, -02,  02,  11,  13,  13,  14,  15,  15,  17,  17,  18,  19,  19,  21,  21,  21,  21,  22,  23,  24]);
  check([-08, -12, -24, -20,  15,  14, -24,  14,  24, -17,  18,  17, -19,  08,  03, -20,  07, -07, -15, -25,  15, -17,  21,  09, -11,  23, -24,  00,  17,  24,  12, -14],
          [-25, -24, -24, -24, -20, -20, -19, -17, -17, -15, -14, -12, -11, -08, -07,  00,  03,  07,  08,  09,  12,  14,  14,  15,  15,  17,  17,  18,  21,  23,  24,  24]);
  check([-12,  14, -07, -20,  03, -06,  20,  05,  04,  13, -03,  11, -02, -23,  16, -18, -03, -22, -17, -13, -14, -21,  24,  00, -21, -23, -25,  23, -21,  12, -16, -08],
          [-25, -23, -23, -22, -21, -21, -21, -20, -18, -17, -16, -14, -13, -12, -08, -07, -06, -03, -03, -02,  00,  03,  04,  05,  11,  12,  13,  14,  16,  20,  23,  24]);
  check([ 03,  02, -01, -18,  23, -04,  14,  03, -14, -14, -11,  12, -12, -18,  19,  10, -15, -21,  24, -04, -15,  23,  23, -09, -22, -25, -11, -18,  14,  00,  01, -07],
          [-25, -22, -21, -18, -18, -18, -15, -15, -14, -14, -12, -11, -11, -09, -07, -04, -04, -01,  00,  01,  02,  03,  03,  10,  12,  14,  14,  19,  23,  23,  23,  24]);
  check([-21, -25,  02, -22, -01, -09,  06,  10,  02,  22, -03,  17, -21, -06,  02, -11, -02,  03,  12,  09,  04,  10, -25,  07,  12, -08,  16,  01,  19, -08,  19,  24],
          [-25, -25, -22, -21, -21, -11, -09, -08, -08, -06, -03, -02, -01,  01,  02,  02,  02,  03,  04,  06,  07,  09,  10,  10,  12,  12,  16,  17,  19,  19,  22,  24]);
  check([-06, -04, -21,  18,  14,  10,  04, -09,  09, -22, -17,  13, -03,  12, -21,  23, -07,  16,  09,  22,  03,  09,  06, -10, -22, -03,  16,  23,  14,  13,  24, -17],
          [-22, -22, -21, -21, -17, -17, -10, -09, -07, -06, -04, -03, -03,  03,  04,  06,  09,  09,  09,  10,  12,  13,  13,  14,  14,  16,  16,  18,  22,  23,  23,  24]);
  check([-14, -22, -21, -24,  15,  10, -06,  01,  15,  05, -11, -10, -06, -07, -12,  12, -14,  22,  09,  16,  08, -10, -17,  12,  12, -24,  12,  03,  14,  13,  14, -22],
          [-24, -24, -22, -22, -21, -17, -14, -14, -12, -11, -10, -10, -07, -06, -06,  01,  03,  05,  08,  09,  10,  12,  12,  12,  12,  13,  14,  14,  15,  15,  16,  22]);
  check([ 18,  18, -21,  10,  05, -02, -14, -05, -22,  00,  10, -02,  20,  23, -15, -17,  22,  22, -24,  06, -11, -16, -05, -23, -15, -16,  07, -23,  22, -04, -20,  15],
          [-24, -23, -23, -22, -21, -20, -17, -16, -16, -15, -15, -14, -11, -05, -05, -04, -02, -02,  00,  05,  06,  07,  10,  10,  15,  18,  18,  20,  22,  22,  22,  23]);
  check([-09, -16,  00,  21,  09,  13, -08,  13, -12, -21, -12, -15, -20,  00, -05, -23, -01, -04,  10,  14,  05, -20,  18,  17, -11,  00,  19, -12,  22,  24, -22, -10],
          [-23, -22, -21, -20, -20, -16, -15, -12, -12, -12, -11, -10, -09, -08, -05, -04, -01,  00,  00,  00,  05,  09,  10,  13,  13,  14,  17,  18,  19,  21,  22,  24]);


//length == 33
  check([ 00, -01, -01,  01,  00,  00, -01,  00,  01,  00,  00, -01,  00,  01, -01,  00,  01,  00,  00,  00, -01,  00, -01, -01, -01, -01, -01, -01, -01, -01, -01,  01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01]);
  check([ 00,  01,  01,  01,  01, -01,  00,  00,  01,  01,  01,  01, -01,  01,  01,  00,  00,  00,  00,  01,  00, -01, -01,  01, -01,  00,  01,  00,  00,  01,  01, -01,  01],
          [-01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  01,  01,  01, -01,  01,  00, -01,  01,  01, -01,  00,  00,  00,  01,  00,  00,  00,  00,  01, -01,  00, -01,  00, -01, -01, -01, -01,  00,  01, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  00,  01, -01,  00,  01, -01, -01, -01,  00,  01,  01,  01, -01,  00,  00,  01, -01,  01,  00,  01, -01,  01, -01,  00, -01,  00,  00, -01, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01, -01, -01, -01, -01,  00,  00, -01, -01,  00, -01,  00,  01,  00,  00,  00,  00, -01,  00,  01, -01,  01,  00,  00, -01,  01,  01,  00, -01,  01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  00,  01, -01,  01, -01,  00,  00, -01,  00,  00, -01,  01, -01,  01, -01,  01, -01,  00,  00,  00,  01,  01,  01,  00, -01,  01,  01,  01, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01, -01,  01,  00, -01, -01,  00,  00,  01,  01,  00, -01, -01,  01,  01, -01, -01, -01,  00,  00, -01, -01,  01,  00, -01,  00, -01, -01,  01,  00, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01,  00, -01, -01, -01,  00, -01, -01, -01, -01,  00, -01, -01,  00,  00,  00,  01, -01,  01, -01,  01,  00,  00,  01,  00,  01,  00,  01, -01,  00,  01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01,  00,  00,  00,  00,  01, -01,  00, -01, -01,  01,  00,  01,  01, -01,  01,  00,  00, -01, -01, -01,  01,  00, -01,  00,  00,  00,  00,  01,  01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  01,  01,  00,  01,  01, -01,  01,  01,  00, -01, -01, -01,  01,  01,  01,  00,  01,  01,  01,  01,  00, -01,  01,  01,  00, -01,  01, -01,  01,  01, -01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  00,  00, -01,  01,  00,  00,  01,  00,  01,  01,  00,  00,  00,  00, -01,  00,  01,  01,  00,  00,  00, -01,  00,  01,  01, -01,  00,  01, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  00, -01,  01,  01,  01,  01,  01,  00,  01, -01, -01,  00,  01, -01,  01,  00,  01,  00,  01,  00,  01,  00,  00,  00,  01,  01,  01, -01, -01,  01, -01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01, -01, -01, -01, -01,  01,  01, -01, -01,  01,  01,  01, -01,  00,  00, -01, -01, -01,  01,  01,  01,  00,  00,  01, -01,  00, -01,  01, -01,  01,  00,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  00, -01,  00, -01,  00,  00,  01, -01,  00, -01,  01, -01, -01,  01, -01,  01, -01, -01, -01,  01, -01,  01, -01,  00,  00,  00,  00, -01,  01,  00, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01,  00,  00,  00,  00, -01, -01,  00,  01, -01,  00, -01,  00,  00,  00,  01,  01, -01, -01,  00, -01,  01, -01,  01, -01,  00,  00,  01,  00,  01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00, -01, -01,  00, -01,  01,  00,  00, -01,  01,  01,  00, -01,  01,  00, -01, -01,  01,  00, -01, -01, -01,  01,  01,  01,  00,  00,  01,  00, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  00,  00,  00, -01, -01,  01,  00,  01, -01, -01, -01,  01,  01, -01, -01,  01,  01,  01,  00, -01,  00, -01,  01,  01,  00,  01,  00, -01, -01, -01, -01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  01, -01, -01,  00,  01, -01,  00,  01,  00,  00, -01,  00,  01, -01, -01,  00, -01,  00,  00, -01,  00, -01, -01, -01, -01, -01,  01,  01, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01, -01,  00,  01,  00,  00, -01,  01, -01,  00,  00,  01,  01, -01, -01, -01,  01,  01,  01,  00,  00,  01,  01,  01,  00,  00,  00,  00,  00,  00,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  00,  00,  00,  01, -01,  01,  00,  01,  01,  00,  01,  00,  00,  01, -01, -01,  00,  01,  00, -01,  01,  01, -01,  00, -01,  00, -01,  01, -01, -01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  00,  01,  01,  01,  00,  00,  00,  01,  01,  01, -01,  00,  00,  00, -01,  01,  00,  00, -01,  00, -01, -01,  01,  00,  01,  01,  00, -01, -01,  01, -01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  00,  00, -01,  00, -01, -01, -01,  00,  00,  01, -01,  01,  00,  00, -01,  01, -01,  00, -01, -01, -01, -01,  01,  00,  00,  00, -01,  00, -01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01]);
  check([-01, -01,  01,  00,  00,  01,  01, -01,  00,  01,  00,  00,  00, -01,  00, -01,  00,  00,  00,  00,  00,  01,  00, -01, -01, -01,  00,  01,  00, -01, -01,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01,  00,  00, -01, -01,  01,  01,  00,  00, -01,  01, -01,  00,  00,  01, -01, -01, -01,  01, -01, -01, -01, -01,  00,  01,  01,  01,  01,  00,  00,  01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00,  01,  00,  00,  01, -01, -01, -01,  01,  00,  01,  01, -01,  01,  01, -01,  01,  01,  00,  01,  00,  01,  00, -01,  01,  00,  01,  01, -01,  00, -01,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01,  01, -01,  01,  01, -01, -01,  01,  00, -01,  00,  01,  01,  00,  01,  01,  00, -01,  00,  01, -01, -01,  00,  00,  00,  00, -01,  01,  01,  00,  00,  01, -01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  01,  01, -01, -01,  00,  01, -01,  00,  01, -01, -01,  01, -01,  01,  00,  00, -01,  01,  00,  00, -01, -01,  01, -01, -01,  01,  00,  01,  01,  00,  00,  01],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 00,  00, -01,  00,  01,  00,  01,  00,  01, -01, -01,  01,  01,  00, -01,  01,  00,  01, -01,  00, -01, -01, -01,  00, -01,  01, -01, -01,  00,  00, -01, -01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([-01,  00, -01,  00,  00,  00, -01,  00,  01, -01,  00,  00, -01,  00, -01, -01, -01,  01,  00, -01, -01,  01,  00, -01,  01,  00,  01,  00,  00,  01,  01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01]);
  check([ 01, -01, -01,  00,  00,  01,  00,  00,  00, -01,  01,  01,  00, -01, -01, -01,  00,  00,  00, -01,  01,  01, -01,  00,  00,  01, -01, -01,  00, -01,  01,  01,  00],
          [-01, -01, -01, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  00,  01,  01,  01,  01,  01,  01,  01,  01,  01]);


//length == 33
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15,  03, -14,  17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23, -17,  15, -22],
          [-25, -24, -22, -21, -20, -19, -17, -16, -14, -10, -07, -06,  00,  01,  02,  03,  03,  04,  05,  06,  06,  08,  11,  12,  14,  15,  15,  15,  17,  21,  22,  23,  23]);
  check([ 18, -19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06, -20, -06,  19,  22, -07,  01,  03, -07, -20,  10, -25,  08,  11, -05, -02,  10,  03, -11,  15, -03, -03],
          [-25, -20, -20, -19, -11, -11, -10, -07, -07, -06, -05, -05, -03, -03, -02,  01,  03,  03,  04,  06,  08,  10,  10,  10,  11,  13,  15,  15,  17,  17,  18,  19,  22]);
  check([ 08, -22,  14, -22,  14,  06,  20, -22, -05,  02, -15,  16, -04, -16, -16,  22,  15,  04, -21,  00,  06,  12, -14, -24, -12,  23,  07,  02,  14, -19,  24, -01, -16],
          [-24, -22, -22, -22, -21, -19, -16, -16, -16, -15, -14, -12, -05, -04, -01,  00,  02,  02,  04,  06,  06,  07,  08,  12,  14,  14,  14,  15,  16,  20,  22,  23,  24]);
  check([ 13,  02, -25, -04,  00, -19,  16, -21, -09,  09,  00,  01,  20,  00, -09,  01,  06,  18, -18, -06, -19, -14,  07, -20,  20, -14,  21, -24, -13, -05, -13, -23,  24],
          [-25, -24, -23, -21, -20, -19, -19, -18, -14, -14, -13, -13, -09, -09, -06, -05, -04,  00,  00,  00,  01,  01,  02,  06,  07,  09,  13,  16,  18,  20,  20,  21,  24]);
  check([-12, -02, -01, -06, -08,  03,  12, -22, -19, -12,  24,  08,  06,  00, -11,  01,  10,  10,  08,  23, -06,  15,  20,  07,  13,  21,  19, -15, -15,  23, -16, -02, -03],
          [-22, -19, -16, -15, -15, -12, -12, -11, -08, -06, -06, -03, -02, -02, -01,  00,  01,  03,  06,  07,  08,  08,  10,  10,  12,  13,  15,  19,  20,  21,  23,  23,  24]);
  check([ 11,  19,  16, -09,  07,  19, -01,  22,  20, -18,  06, -04, -01, -18, -17, -14,  17, -17,  08,  09, -22, -10, -03, -24, -13,  07, -14, -15,  19,  12,  09,  05,  06],
          [-24, -22, -18, -18, -17, -17, -15, -14, -14, -13, -10, -09, -04, -03, -01, -01,  05,  06,  06,  07,  07,  08,  09,  09,  11,  12,  16,  17,  19,  19,  19,  20,  22]);
  check([ 00,  21, -12, -03, -02, -12, -08,  05,  19,  15, -19, -22, -25, -05,  21, -17, -22,  05, -14, -07, -20, -11,  05,  12,  01,  16,  06, -12,  00, -12,  21, -22, -13],
          [-25, -22, -22, -22, -20, -19, -17, -14, -13, -12, -12, -12, -12, -11, -08, -07, -05, -03, -02,  00,  00,  01,  05,  05,  05,  06,  12,  15,  16,  19,  21,  21,  21]);
  check([-15,  00,  10,  00,  19, -08, -06,  10, -01, -01,  10,  21, -03,  19,  24, -22, -18, -08, -15, -03,  23,  22, -25,  16,  06, -12, -07,  19, -16, -02, -19, -04,  23],
          [-25, -22, -19, -18, -16, -15, -15, -12, -08, -08, -07, -06, -04, -03, -03, -02, -01, -01,  00,  00,  06,  10,  10,  10,  16,  19,  19,  19,  21,  22,  23,  23,  24]);
  check([ 18,  21,  20, -14, -08,  05,  12,  19, -10,  08, -09, -14,  07, -04, -04,  24,  06,  18,  24,  06,  20,  17, -13, -17, -14, -17, -06,  11, -08,  18,  12, -15,  16],
          [-17, -17, -15, -14, -14, -14, -13, -10, -09, -08, -08, -06, -04, -04,  05,  06,  06,  07,  08,  11,  12,  12,  16,  17,  18,  18,  18,  19,  20,  20,  21,  24,  24]);
  check([ 07, -04, -14, -13, -15, -20,  04,  18, -04,  18,  00,  20, -11,  02,  01, -16,  03, -18, -21, -04, -04, -11,  09,  07,  08, -05,  24,  03, -18, -14, -05,  14,  08],
          [-21, -20, -18, -18, -16, -15, -14, -14, -13, -11, -11, -05, -05, -04, -04, -04, -04,  00,  01,  02,  03,  03,  04,  07,  07,  08,  08,  09,  14,  18,  18,  20,  24]);
  check([ 06, -22,  18,  13,  08,  14, -14,  01, -11, -19,  17,  18,  08, -24, -03,  17, -20,  20, -09, -04,  04,  23, -21, -24,  22,  10, -14, -17, -20, -25,  18,  13, -19],
          [-25, -24, -24, -22, -21, -20, -20, -19, -19, -17, -14, -14, -11, -09, -04, -03,  01,  04,  06,  08,  08,  10,  13,  13,  14,  17,  17,  18,  18,  18,  20,  22,  23]);
  check([ 14,  01,  14,  03,  12, -08,  19,  21, -16,  13,  04, -13, -13, -02, -08, -18,  14,  13,  13,  12,  19,  16,  11,  06, -23,  20,  11, -20,  15,  01, -14,  04, -21],
          [-23, -21, -20, -18, -16, -14, -13, -13, -08, -08, -02,  01,  01,  03,  04,  04,  06,  11,  11,  12,  12,  13,  13,  13,  14,  14,  14,  15,  16,  19,  19,  20,  21]);
  check([-23, -16,  19, -04, -21,  15,  07,  19, -04,  19, -17,  19, -12, -10,  09, -22, -20, -04,  24,  21, -17,  06,  24, -20, -08, -19,  22,  21, -08,  04, -25, -04,  13],
          [-25, -23, -22, -21, -20, -20, -19, -17, -17, -16, -12, -10, -08, -08, -04, -04, -04, -04,  04,  06,  07,  09,  13,  15,  19,  19,  19,  19,  21,  21,  22,  24,  24]);
  check([ 21,  17,  19,  11, -01,  15, -18, -07,  23, -23,  08, -10,  13,  11, -05, -16,  10, -06, -06, -09, -05,  01,  11,  01, -01,  07,  20, -20,  09, -09,  20,  06, -15],
          [-23, -20, -18, -16, -15, -10, -09, -09, -07, -06, -06, -05, -05, -01, -01,  01,  01,  06,  07,  08,  09,  10,  11,  11,  11,  13,  15,  17,  19,  20,  20,  21,  23]);
  check([ 15, -06,  09,  07,  02, -21,  08,  06,  12,  23, -06, -02, -04,  05, -15,  17, -25,  02, -13,  03, -12,  13, -21,  22,  10, -16,  08, -24, -20, -11, -14,  22,  09],
          [-25, -24, -21, -21, -20, -16, -15, -14, -13, -12, -11, -06, -06, -04, -02,  02,  02,  03,  05,  06,  07,  08,  08,  09,  09,  10,  12,  13,  15,  17,  22,  22,  23]);
  check([ 22,  06, -12, -22, -11,  19,  17, -12, -10, -10,  11,  20,  00,  03,  22, -21,  15,  01, -08,  05,  07, -09, -10,  19, -25, -07,  24, -09,  06,  23, -23,  05,  04],
          [-25, -23, -22, -21, -12, -12, -11, -10, -10, -10, -09, -09, -08, -07,  00,  01,  03,  04,  05,  05,  06,  06,  07,  11,  15,  17,  19,  19,  20,  22,  22,  23,  24]);
  check([-10,  08,  19, -14,  00, -16,  01,  17,  20, -01, -06,  00, -04, -01, -10,  24,  18,  22,  09, -13, -11,  03, -13,  07,  02,  03, -10,  02,  08,  20, -19,  23,  05],
          [-19, -16, -14, -13, -13, -11, -10, -10, -10, -06, -04, -01, -01,  00,  00,  01,  02,  02,  03,  03,  05,  07,  08,  08,  09,  17,  18,  19,  20,  20,  22,  23,  24]);
  check([-23, -13, -20, -12,  15,  24, -17, -11, -07,  08,  13,  19, -25,  12,  15,  24, -04, -21, -12,  24, -09,  22,  03,  21, -11, -18,  04, -16, -09,  05,  16, -07,  17],
          [-25, -23, -21, -20, -18, -17, -16, -13, -12, -12, -11, -11, -09, -09, -07, -07, -04,  03,  04,  05,  08,  12,  13,  15,  15,  16,  17,  19,  21,  22,  24,  24,  24]);
  check([ 21,  09,  09,  22,  17,  24, -07,  03,  12, -11,  05,  01, -21,  05, -25, -17,  20, -24,  01,  20,  05,  00, -16,  14, -19, -04, -20,  11, -13,  01,  05, -14, -15],
          [-25, -24, -21, -20, -19, -17, -16, -15, -14, -13, -11, -07, -04,  00,  01,  01,  01,  03,  05,  05,  05,  05,  09,  09,  11,  12,  14,  17,  20,  20,  21,  22,  24]);
  check([-10, -17, -21, -09,  01,  09, -22,  16, -10,  06,  22,  22,  06, -17,  17,  08,  09,  12, -10, -14,  24, -21, -07, -05, -13, -19,  07,  13,  12,  18, -25, -21, -23],
          [-25, -23, -22, -21, -21, -21, -19, -17, -17, -14, -13, -10, -10, -10, -09, -07, -05,  01,  06,  06,  07,  08,  09,  09,  12,  12,  13,  16,  17,  18,  22,  22,  24]);
  check([-19, -05,  05,  16,  00, -04, -17, -19, -04, -20,  15,  04,  24, -02, -10,  14,  13,  02,  13,  19,  22, -17, -19, -22,  17,  21,  15,  11,  23,  21,  15, -20, -07],
          [-22, -20, -20, -19, -19, -19, -17, -17, -10, -07, -05, -04, -04, -02,  00,  02,  04,  05,  11,  13,  13,  14,  15,  15,  15,  16,  17,  19,  21,  21,  22,  23,  24]);
  check([-05,  21,  18,  19, -21, -23, -10, -14,  17,  21, -15, -08, -12, -24, -20,  15,  14, -24,  14,  24, -17,  18,  17, -19,  08,  03, -20,  07, -07, -15, -25,  15, -17],
          [-25, -24, -24, -23, -21, -20, -20, -19, -17, -17, -15, -15, -14, -12, -10, -08, -07, -05,  03,  07,  08,  14,  14,  15,  15,  17,  17,  18,  18,  19,  21,  21,  24]);
  check([ 21,  09, -11,  23, -24,  00,  17,  24,  12, -14, -12,  14, -07, -20,  03, -06,  20,  05,  04,  13, -03,  11, -02, -23,  16, -18, -03, -22, -17, -13, -14, -21,  24],
          [-24, -23, -22, -21, -20, -18, -17, -14, -14, -13, -12, -11, -07, -06, -03, -03, -02,  00,  03,  04,  05,  09,  11,  12,  13,  14,  16,  17,  20,  21,  23,  24,  24]);
  check([ 00, -21, -23, -25,  23, -21,  12, -16, -08,  03,  02, -01, -18,  23, -04,  14,  03, -14, -14, -11,  12, -12, -18,  19,  10, -15, -21,  24, -04, -15,  23,  23, -09],
          [-25, -23, -21, -21, -21, -18, -18, -16, -15, -15, -14, -14, -12, -11, -09, -08, -04, -04, -01,  00,  02,  03,  03,  10,  12,  12,  14,  19,  23,  23,  23,  23,  24]);
  check([-22, -25, -11, -18,  14,  00,  01, -07, -21, -25,  02, -22, -01, -09,  06,  10,  02,  22, -03,  17, -21, -06,  02, -11, -02,  03,  12,  09,  04,  10, -25,  07,  12],
          [-25, -25, -25, -22, -22, -21, -21, -18, -11, -11, -09, -07, -06, -03, -02, -01,  00,  01,  02,  02,  02,  03,  04,  06,  07,  09,  10,  10,  12,  12,  14,  17,  22]);
  check([-08,  16,  01,  19, -08,  19,  24, -06, -04, -21,  18,  14,  10,  04, -09,  09, -22, -17,  13, -03,  12, -21,  23, -07,  16,  09,  22,  03,  09,  06, -10, -22, -03],
          [-22, -22, -21, -21, -17, -10, -09, -08, -08, -07, -06, -04, -03, -03,  01,  03,  04,  06,  09,  09,  09,  10,  12,  13,  14,  16,  16,  18,  19,  19,  22,  23,  24]);
  check([ 16,  23,  14,  13,  24, -17, -14, -22, -21, -24,  15,  10, -06,  01,  15,  05, -11, -10, -06, -07, -12,  12, -14,  22,  09,  16,  08, -10, -17,  12,  12, -24,  12],
          [-24, -24, -22, -21, -17, -17, -14, -14, -12, -11, -10, -10, -07, -06, -06,  01,  05,  08,  09,  10,  12,  12,  12,  12,  13,  14,  15,  15,  16,  16,  22,  23,  24]);
  check([ 03,  14,  13,  14, -22,  18,  18, -21,  10,  05, -02, -14, -05, -22,  00,  10, -02,  20,  23, -15, -17,  22,  22, -24,  06, -11, -16, -05, -23, -15, -16,  07, -23],
          [-24, -23, -23, -22, -22, -21, -17, -16, -16, -15, -15, -14, -11, -05, -05, -02, -02,  00,  03,  05,  06,  07,  10,  10,  13,  14,  14,  18,  18,  20,  22,  22,  23]);
  check([ 22, -04, -20,  15, -09, -16,  00,  21,  09,  13, -08,  13, -12, -21, -12, -15, -20,  00, -05, -23, -01, -04,  10,  14,  05, -20,  18,  17, -11,  00,  19, -12,  22],
          [-23, -21, -20, -20, -20, -16, -15, -12, -12, -12, -11, -09, -08, -05, -04, -04, -01,  00,  00,  00,  05,  09,  10,  13,  13,  14,  15,  17,  18,  19,  21,  22,  22]);
  check([ 24, -22, -10, -15,  05, -13,  20, -07,  06,  10,  08,  10, -25,  20,  17,  00, -10,  20, -25,  11,  05,  16, -07,  11,  09, -14, -23, -16, -20, -09, -17, -18, -04],
          [-25, -25, -23, -22, -20, -18, -17, -16, -15, -14, -13, -10, -10, -09, -07, -07, -04,  00,  05,  05,  06,  08,  09,  10,  10,  11,  11,  16,  17,  20,  20,  20,  24]);


//length == 57
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15,  03, -14,  17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23, -17,  15, -22,  18, -19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06, -20, -06,  19,  22, -07,  01,  03, -07, -20,  10, -25,  08],
          [-25, -25, -24, -22, -21, -20, -20, -20, -19, -19, -17, -16, -14, -11, -10, -10, -07, -07, -07, -06, -06, -05,  00,  01,  01,  02,  03,  03,  03,  04,  04,  05,  06,  06,  06,  08,  08,  10,  10,  11,  12,  13,  14,  15,  15,  15,  15,  17,  17,  17,  18,  19,  21,  22,  22,  23,  23]);
  check([ 11, -05, -02,  10,  03, -11,  15, -03, -03,  08, -22,  14, -22,  14,  06,  20, -22, -05,  02, -15,  16, -04, -16, -16,  22,  15,  04, -21,  00,  06,  12, -14, -24, -12,  23,  07,  02,  14, -19,  24, -01, -16,  13,  02, -25, -04,  00, -19,  16, -21, -09,  09,  00,  01,  20,  00, -09],
          [-25, -24, -22, -22, -22, -21, -21, -19, -19, -16, -16, -16, -15, -14, -12, -11, -09, -09, -05, -05, -04, -04, -03, -03, -02, -01,  00,  00,  00,  00,  01,  02,  02,  02,  03,  04,  06,  06,  07,  08,  09,  10,  11,  12,  13,  14,  14,  14,  15,  15,  16,  16,  20,  20,  22,  23,  24]);
  check([ 01,  06,  18, -18, -06, -19, -14,  07, -20,  20, -14,  21, -24, -13, -05, -13, -23,  24, -12, -02, -01, -06, -08,  03,  12, -22, -19, -12,  24,  08,  06,  00, -11,  01,  10,  10,  08,  23, -06,  15,  20,  07,  13,  21,  19, -15, -15,  23, -16, -02, -03,  11,  19,  16, -09,  07,  19],
          [-24, -23, -22, -20, -19, -19, -18, -16, -15, -15, -14, -14, -13, -13, -12, -12, -11, -09, -08, -06, -06, -06, -05, -03, -02, -02, -01,  00,  01,  01,  03,  06,  06,  07,  07,  07,  08,  08,  10,  10,  11,  12,  13,  15,  16,  18,  19,  19,  19,  20,  20,  21,  21,  23,  23,  24,  24]);
  check([-01,  22,  20, -18,  06, -04, -01, -18, -17, -14,  17, -17,  08,  09, -22, -10, -03, -24, -13,  07, -14, -15,  19,  12,  09,  05,  06,  00,  21, -12, -03, -02, -12, -08,  05,  19,  15, -19, -22, -25, -05,  21, -17, -22,  05, -14, -07, -20, -11,  05,  12,  01,  16,  06, -12,  00, -12],
          [-25, -24, -22, -22, -22, -20, -19, -18, -18, -17, -17, -17, -15, -14, -14, -14, -13, -12, -12, -12, -12, -11, -10, -08, -07, -05, -04, -03, -03, -02, -01, -01,  00,  00,  01,  05,  05,  05,  05,  06,  06,  06,  07,  08,  09,  09,  12,  12,  15,  16,  17,  19,  19,  20,  21,  21,  22]);
  check([ 21, -22, -13, -15,  00,  10,  00,  19, -08, -06,  10, -01, -01,  10,  21, -03,  19,  24, -22, -18, -08, -15, -03,  23,  22, -25,  16,  06, -12, -07,  19, -16, -02, -19, -04,  23,  18,  21,  20, -14, -08,  05,  12,  19, -10,  08, -09, -14,  07, -04, -04,  24,  06,  18,  24,  06,  20],
          [-25, -22, -22, -19, -18, -16, -15, -15, -14, -14, -13, -12, -10, -09, -08, -08, -08, -07, -06, -04, -04, -04, -03, -03, -02, -01, -01,  00,  00,  05,  06,  06,  06,  07,  08,  10,  10,  10,  12,  16,  18,  18,  19,  19,  19,  19,  20,  20,  21,  21,  21,  22,  23,  23,  24,  24,  24]);
  check([ 17, -13, -17, -14, -17, -06,  11, -08,  18,  12, -15,  16,  07, -04, -14, -13, -15, -20,  04,  18, -04,  18,  00,  20, -11,  02,  01, -16,  03, -18, -21, -04, -04, -11,  09,  07,  08, -05,  24,  03, -18, -14, -05,  14,  08,  06, -22,  18,  13,  08,  14, -14,  01, -11, -19,  17,  18],
          [-22, -21, -20, -19, -18, -18, -17, -17, -16, -15, -15, -14, -14, -14, -14, -13, -13, -11, -11, -11, -08, -06, -05, -05, -04, -04, -04, -04,  00,  01,  01,  02,  03,  03,  04,  06,  07,  07,  08,  08,  08,  09,  11,  12,  13,  14,  14,  16,  17,  17,  18,  18,  18,  18,  18,  20,  24]);
  check([ 08, -24, -03,  17, -20,  20, -09, -04,  04,  23, -21, -24,  22,  10, -14, -17, -20, -25,  18,  13, -19,  14,  01,  14,  03,  12, -08,  19,  21, -16,  13,  04, -13, -13, -02, -08, -18,  14,  13,  13,  12,  19,  16,  11,  06, -23,  20,  11, -20,  15,  01, -14,  04, -21, -23, -16,  19],
          [-25, -24, -24, -23, -23, -21, -21, -20, -20, -20, -19, -18, -17, -16, -16, -14, -14, -13, -13, -09, -08, -08, -04, -03, -02,  01,  01,  03,  04,  04,  04,  06,  08,  10,  11,  11,  12,  12,  13,  13,  13,  13,  14,  14,  14,  15,  16,  17,  18,  19,  19,  19,  20,  20,  21,  22,  23]);
  check([-04, -21,  15,  07,  19, -04,  19, -17,  19, -12, -10,  09, -22, -20, -04,  24,  21, -17,  06,  24, -20, -08, -19,  22,  21, -08,  04, -25, -04,  13,  21,  17,  19,  11, -01,  15, -18, -07,  23, -23,  08, -10,  13,  11, -05, -16,  10, -06, -06, -09, -05,  01,  11,  01, -01,  07,  20],
          [-25, -23, -22, -21, -20, -20, -19, -18, -17, -17, -16, -12, -10, -10, -09, -08, -08, -07, -06, -06, -05, -05, -04, -04, -04, -04, -01, -01,  01,  01,  04,  06,  07,  07,  08,  09,  10,  11,  11,  11,  13,  13,  15,  15,  17,  19,  19,  19,  19,  20,  21,  21,  21,  22,  23,  24,  24]);
  check([-20,  09, -09,  20,  06, -15,  15, -06,  09,  07,  02, -21,  08,  06,  12,  23, -06, -02, -04,  05, -15,  17, -25,  02, -13,  03, -12,  13, -21,  22,  10, -16,  08, -24, -20, -11, -14,  22,  09,  22,  06, -12, -22, -11,  19,  17, -12, -10, -10,  11,  20,  00,  03,  22, -21,  15,  01],
          [-25, -24, -22, -21, -21, -21, -20, -20, -16, -15, -15, -14, -13, -12, -12, -12, -11, -11, -10, -10, -09, -06, -06, -04, -02,  00,  01,  02,  02,  03,  03,  05,  06,  06,  06,  07,  08,  08,  09,  09,  09,  10,  11,  12,  13,  15,  15,  17,  17,  19,  20,  20,  22,  22,  22,  22,  23]);
  check([-08,  05,  07, -09, -10,  19, -25, -07,  24, -09,  06,  23, -23,  05,  04, -10,  08,  19, -14,  00, -16,  01,  17,  20, -01, -06,  00, -04, -01, -10,  24,  18,  22,  09, -13, -11,  03, -13,  07,  02,  03, -10,  02,  08,  20, -19,  23,  05, -23, -13, -20, -12,  15,  24, -17, -11, -07],
          [-25, -23, -23, -20, -19, -17, -16, -14, -13, -13, -13, -12, -11, -11, -10, -10, -10, -10, -09, -09, -08, -07, -07, -06, -04, -01, -01,  00,  00,  01,  02,  02,  03,  03,  04,  05,  05,  05,  06,  07,  07,  08,  08,  09,  15,  17,  18,  19,  19,  20,  20,  22,  23,  23,  24,  24,  24]);
  check([ 08,  13,  19, -25,  12,  15,  24, -04, -21, -12,  24, -09,  22,  03,  21, -11, -18,  04, -16, -09,  05,  16, -07,  17,  21,  09,  09,  22,  17,  24, -07,  03,  12, -11,  05,  01, -21,  05, -25, -17,  20, -24,  01,  20,  05,  00, -16,  14, -19, -04, -20,  11, -13,  01,  05, -14, -15],
          [-25, -25, -24, -21, -21, -20, -19, -18, -17, -16, -16, -15, -14, -13, -12, -11, -11, -09, -09, -07, -07, -04, -04,  00,  01,  01,  01,  03,  03,  04,  05,  05,  05,  05,  05,  08,  09,  09,  11,  12,  12,  13,  14,  15,  16,  17,  17,  19,  20,  20,  21,  21,  22,  22,  24,  24,  24]);
  check([-10, -17, -21, -09,  01,  09, -22,  16, -10,  06,  22,  22,  06, -17,  17,  08,  09,  12, -10, -14,  24, -21, -07, -05, -13, -19,  07,  13,  12,  18, -25, -21, -23, -19, -05,  05,  16,  00, -04, -17, -19, -04, -20,  15,  04,  24, -02, -10,  14,  13,  02,  13,  19,  22, -17, -19, -22],
          [-25, -23, -22, -22, -21, -21, -21, -20, -19, -19, -19, -19, -17, -17, -17, -17, -14, -13, -10, -10, -10, -10, -09, -07, -05, -05, -04, -04, -02,  00,  01,  02,  04,  05,  06,  06,  07,  08,  09,  09,  12,  12,  13,  13,  13,  14,  15,  16,  16,  17,  18,  19,  22,  22,  22,  24,  24]);
  check([ 17,  21,  15,  11,  23,  21,  15, -20, -07, -05,  21,  18,  19, -21, -23, -10, -14,  17,  21, -15, -08, -12, -24, -20,  15,  14, -24,  14,  24, -17,  18,  17, -19,  08,  03, -20,  07, -07, -15, -25,  15, -17,  21,  09, -11,  23, -24,  00,  17,  24,  12, -14, -12,  14, -07, -20,  03],
          [-25, -24, -24, -24, -23, -21, -20, -20, -20, -20, -19, -17, -17, -15, -15, -14, -14, -12, -12, -11, -10, -08, -07, -07, -07, -05,  00,  03,  03,  07,  08,  09,  11,  12,  14,  14,  14,  15,  15,  15,  15,  17,  17,  17,  17,  18,  18,  19,  21,  21,  21,  21,  21,  23,  23,  24,  24]);
  check([-06,  20,  05,  04,  13, -03,  11, -02, -23,  16, -18, -03, -22, -17, -13, -14, -21,  24,  00, -21, -23, -25,  23, -21,  12, -16, -08,  03,  02, -01, -18,  23, -04,  14,  03, -14, -14, -11,  12, -12, -18,  19,  10, -15, -21,  24, -04, -15,  23,  23, -09, -22, -25, -11, -18,  14,  00],
          [-25, -25, -23, -23, -22, -22, -21, -21, -21, -21, -18, -18, -18, -18, -17, -16, -15, -15, -14, -14, -14, -13, -12, -11, -11, -09, -08, -06, -04, -04, -03, -03, -02, -01,  00,  00,  02,  03,  03,  04,  05,  10,  11,  12,  12,  13,  14,  14,  16,  19,  20,  23,  23,  23,  23,  24,  24]);
  check([ 01, -07, -21, -25,  02, -22, -01, -09,  06,  10,  02,  22, -03,  17, -21, -06,  02, -11, -02,  03,  12,  09,  04,  10, -25,  07,  12, -08,  16,  01,  19, -08,  19,  24, -06, -04, -21,  18,  14,  10,  04, -09,  09, -22, -17,  13, -03,  12, -21,  23, -07,  16,  09,  22,  03,  09,  06],
          [-25, -25, -22, -22, -21, -21, -21, -21, -17, -11, -09, -09, -08, -08, -07, -07, -06, -06, -04, -03, -03, -02, -01,  01,  01,  02,  02,  02,  03,  03,  04,  04,  06,  06,  07,  09,  09,  09,  09,  10,  10,  10,  12,  12,  12,  13,  14,  16,  16,  17,  18,  19,  19,  22,  22,  23,  24]);
  check([-10, -22, -03,  16,  23,  14,  13,  24, -17, -14, -22, -21, -24,  15,  10, -06,  01,  15,  05, -11, -10, -06, -07, -12,  12, -14,  22,  09,  16,  08, -10, -17,  12,  12, -24,  12,  03,  14,  13,  14, -22,  18,  18, -21,  10,  05, -02, -14, -05, -22,  00,  10, -02,  20,  23, -15, -17],
          [-24, -24, -22, -22, -22, -22, -21, -21, -17, -17, -17, -15, -14, -14, -14, -12, -11, -10, -10, -10, -07, -06, -06, -05, -03, -02, -02,  00,  01,  03,  05,  05,  08,  09,  10,  10,  10,  12,  12,  12,  12,  13,  13,  14,  14,  14,  15,  15,  16,  16,  18,  18,  20,  22,  23,  23,  24]);
  check([ 22,  22, -24,  06, -11, -16, -05, -23, -15, -16,  07, -23,  22, -04, -20,  15, -09, -16,  00,  21,  09,  13, -08,  13, -12, -21, -12, -15, -20,  00, -05, -23, -01, -04,  10,  14,  05, -20,  18,  17, -11,  00,  19, -12,  22,  24, -22, -10, -15,  05, -13,  20, -07,  06,  10,  08,  10],
          [-24, -23, -23, -23, -22, -21, -20, -20, -20, -16, -16, -16, -15, -15, -15, -13, -12, -12, -12, -11, -11, -10, -09, -08, -07, -05, -05, -04, -04, -01,  00,  00,  00,  05,  05,  06,  06,  07,  08,  09,  10,  10,  10,  13,  13,  14,  15,  17,  18,  19,  20,  21,  22,  22,  22,  22,  24]);
  check([-25,  20,  17,  00, -10,  20, -25,  11,  05,  16, -07,  11,  09, -14, -23, -16, -20, -09, -17, -18, -04, -01, -06, -21,  13, -11, -01, -06, -24, -15, -19, -24, -20,  24,  04, -02,  21,  06, -14,  01, -03,  05, -11, -19,  18, -06, -08, -25,  10,  01, -18, -17, -25,  02, -11,  15,  18],
          [-25, -25, -25, -25, -24, -24, -23, -21, -20, -20, -19, -19, -18, -18, -17, -17, -16, -15, -14, -14, -11, -11, -11, -10, -09, -08, -07, -06, -06, -06, -04, -03, -02, -01, -01,  00,  01,  01,  02,  04,  05,  05,  06,  09,  10,  11,  11,  13,  15,  16,  17,  18,  18,  20,  20,  21,  24]);
  check([ 14, -16,  20,  24, -08,  23, -21, -09,  02,  04, -13, -17,  16,  16,  05, -04, -18,  13, -11,  01, -19, -11, -12,  09, -01, -03,  09, -22,  11,  01,  21,  02,  10,  18,  01, -21,  17,  08, -04, -06, -13,  10,  05, -20,  03, -15,  01,  11, -24,  17, -13, -18, -16,  03,  16,  10, -25],
          [-25, -24, -22, -21, -21, -20, -19, -18, -18, -17, -16, -16, -15, -13, -13, -13, -12, -11, -11, -09, -08, -06, -04, -04, -03, -01,  01,  01,  01,  01,  02,  02,  03,  03,  04,  05,  05,  08,  09,  09,  10,  10,  10,  11,  11,  13,  14,  16,  16,  16,  17,  17,  18,  20,  21,  23,  24]);
  check([ 02,  13,  13, -22,  11, -07,  15,  07,  19,  19, -01,  04, -08,  20,  19, -20,  00,  24,  08,  13,  03, -04,  16, -03,  11,  23,  06, -11,  16, -07, -11, -05, -19, -21,  00,  20, -03, -08,  02, -06, -12, -22, -25,  06,  23,  19,  13,  01,  21, -04, -09, -01,  20, -18,  21,  06, -18],
          [-25, -22, -22, -21, -20, -19, -18, -18, -12, -11, -11, -09, -08, -08, -07, -07, -06, -05, -04, -04, -03, -03, -01, -01,  00,  00,  01,  02,  02,  03,  04,  06,  06,  06,  07,  08,  11,  11,  13,  13,  13,  13,  15,  16,  16,  19,  19,  19,  19,  20,  20,  20,  21,  21,  23,  23,  24]);
  check([ 05,  20,  23,  23, -14, -05, -20, -08,  22, -23,  15, -11,  04, -14,  02,  09, -12, -15,  09, -17,  23, -15, -21, -03,  03,  05, -08,  10,  01,  23,  17, -19,  18,  17, -20, -19, -11, -13, -02, -14, -09, -10,  02,  22,  01, -20,  06,  15, -08, -10, -25, -09,  03, -19,  13, -19,  11],
          [-25, -23, -21, -20, -20, -20, -19, -19, -19, -19, -17, -15, -15, -14, -14, -14, -13, -12, -11, -11, -10, -10, -09, -09, -08, -08, -08, -05, -03, -02,  01,  01,  02,  02,  03,  03,  04,  05,  05,  06,  09,  09,  10,  11,  13,  15,  15,  17,  17,  18,  20,  22,  22,  23,  23,  23,  23]);
  check([-20,  17, -13, -20,  11, -04, -25,  06,  01, -19,  20,  15,  04, -16, -19,  20,  11, -22, -02,  18,  11, -12, -14, -24, -10,  04,  04, -04, -08,  13, -18, -01,  07, -03,  06, -07,  18,  06, -24, -04,  12,  24, -12, -07, -17, -06, -12,  21, -01,  12,  17, -15, -23, -20, -14, -05,  09],
          [-25, -24, -24, -23, -22, -20, -20, -20, -19, -19, -18, -17, -16, -15, -14, -14, -13, -12, -12, -12, -10, -08, -07, -07, -06, -05, -04, -04, -04, -03, -02, -01, -01,  01,  04,  04,  04,  06,  06,  06,  07,  09,  11,  11,  11,  12,  12,  13,  15,  17,  17,  18,  18,  20,  20,  21,  24]);
  check([ 18,  16, -22,  06, -24,  04, -12, -02, -15,  08, -09,  18,  12,  12,  05,  11, -25, -25,  19, -04, -09,  17,  22, -20,  09, -18, -18,  14, -05,  04, -02, -10, -04,  03,  23, -03, -18, -12,  22, -08,  22, -12, -15,  09, -23,  17, -05, -21,  20, -09,  02, -14, -17, -01, -09,  20,  08],
          [-25, -25, -24, -23, -22, -21, -20, -18, -18, -18, -17, -15, -15, -14, -12, -12, -12, -10, -09, -09, -09, -09, -08, -05, -05, -04, -04, -03, -02, -02, -01,  02,  03,  04,  04,  05,  06,  08,  08,  09,  09,  11,  12,  12,  14,  16,  17,  17,  18,  18,  19,  20,  20,  22,  22,  22,  23]);
  check([ 00,  09, -22, -20, -15, -04,  03,  13, -06, -23,  23,  08,  24, -10,  07, -13,  03, -09, -09, -03,  13, -03, -08, -19,  24,  03, -09,  00,  21, -14, -17, -03,  21, -12,  02, -17,  09, -20,  21, -19, -18,  21,  14, -17, -11, -04, -05,  17,  14,  13, -09,  04, -15,  11,  10, -14, -11],
          [-23, -22, -20, -20, -19, -19, -18, -17, -17, -17, -15, -15, -14, -14, -13, -12, -11, -11, -10, -09, -09, -09, -09, -08, -06, -05, -04, -04, -03, -03, -03,  00,  00,  02,  03,  03,  03,  04,  07,  08,  09,  09,  10,  11,  13,  13,  13,  14,  14,  17,  21,  21,  21,  21,  23,  24,  24]);
  check([-24,  11, -14, -12,  19,  10, -14, -16, -11, -06,  21, -06, -08, -23,  03, -11,  16,  13,  03, -11, -17, -03, -22,  23,  13,  09, -17, -01, -06, -04,  16, -05, -18, -21,  10, -22,  14,  21, -13, -22, -08, -15, -01,  10, -13, -21, -24, -20,  17,  04, -04, -23, -24, -01, -23,  16, -15],
          [-24, -24, -24, -23, -23, -23, -22, -22, -22, -21, -21, -20, -18, -17, -17, -16, -15, -15, -14, -14, -13, -13, -12, -11, -11, -11, -08, -08, -06, -06, -06, -05, -04, -04, -03, -01, -01, -01,  03,  03,  04,  09,  10,  10,  10,  11,  13,  13,  14,  16,  16,  16,  17,  19,  21,  21,  23]);
  check([-13, -07,  04,  08, -14, -23,  15, -10,  12,  20, -19,  11, -16, -14, -22, -05,  12,  15,  07,  18,  16,  15,  10, -05, -14,  14, -02,  13,  16, -08,  23, -22,  10,  05,  13,  21,  07,  03, -12, -04, -02, -04, -18,  09,  07, -12, -21, -06, -22,  14, -13,  20,  04,  24, -10,  17,  15],
          [-23, -22, -22, -22, -21, -19, -18, -16, -14, -14, -14, -13, -13, -12, -12, -10, -10, -08, -07, -06, -05, -05, -04, -04, -02, -02,  03,  04,  04,  05,  07,  07,  07,  08,  09,  10,  10,  11,  12,  12,  13,  13,  14,  14,  15,  15,  15,  15,  16,  16,  17,  18,  20,  20,  21,  23,  24]);
  check([ 16,  07,  08, -17,  06,  13,  20, -14,  01,  18,  18, -19, -19, -09,  06,  02, -01, -10, -14,  12, -05,  05,  15, -14,  19,  12,  17,  20, -20,  09,  10,  21, -08, -05, -19, -25, -17, -24, -14,  11,  21, -19,  17, -21, -03, -02,  06,  23,  15,  19,  12, -13,  01,  05, -02, -05, -08],
          [-25, -24, -21, -20, -19, -19, -19, -19, -17, -17, -14, -14, -14, -14, -13, -10, -09, -08, -08, -05, -05, -05, -03, -02, -02, -01,  01,  01,  02,  05,  05,  06,  06,  06,  07,  08,  09,  10,  11,  12,  12,  12,  13,  15,  15,  16,  17,  17,  18,  18,  19,  19,  20,  20,  21,  21,  23]);
  check([-10, -08, -03, -23, -21, -05, -04, -01,  01, -04,  09,  04,  09, -05,  00,  17, -11,  04, -09,  14, -13, -10,  04, -19, -23,  17,  07,  07, -08, -21, -25,  08, -02, -01,  12,  02,  20, -17, -22, -02,  06,  12, -22, -10, -16,  05, -18, -02, -13, -02, -13, -01,  15,  17,  08,  18,  11],
          [-25, -23, -23, -22, -22, -21, -21, -19, -18, -17, -16, -13, -13, -13, -11, -10, -10, -10, -09, -08, -08, -05, -05, -04, -04, -03, -02, -02, -02, -02, -01, -01, -01,  00,  01,  02,  04,  04,  04,  05,  06,  07,  07,  08,  08,  09,  09,  11,  12,  12,  14,  15,  17,  17,  17,  18,  20]);
  check([-08,  02, -22, -01,  04,  13,  22, -19,  00,  02, -24,  10,  05, -01, -09, -05,  04,  06,  04, -15,  15, -20, -01, -09, -08,  23, -19, -14,  08, -24,  24, -22,  04, -22,  04, -15,  18,  01, -09, -06, -20, -06, -21,  13,  19, -02, -17, -02, -19,  14,  10,  22,  19, -16, -12, -11, -15],
          [-24, -24, -22, -22, -22, -21, -20, -20, -19, -19, -19, -17, -16, -15, -15, -15, -14, -12, -11, -09, -09, -09, -08, -08, -06, -06, -05, -02, -02, -01, -01, -01,  00,  01,  02,  02,  04,  04,  04,  04,  04,  05,  06,  08,  10,  10,  13,  13,  14,  15,  18,  19,  19,  22,  22,  23,  24]);
  check([-06,  00,  20, -02,  00, -25, -21,  05,  04, -11, -02, -19,  08,  19, -12, -23, -24, -24,  21, -01, -16, -03,  05,  24, -16,  02,  20, -04,  17, -16,  08, -11,  12,  03,  12, -11,  04,  16,  19, -17, -18, -06, -09,  15, -11,  07,  18, -08,  08,  14,  16,  18, -12, -02,  19,  00, -22],
          [-25, -24, -24, -23, -22, -21, -19, -18, -17, -16, -16, -16, -12, -12, -11, -11, -11, -11, -09, -08, -06, -06, -04, -03, -02, -02, -02, -01,  00,  00,  00,  02,  03,  04,  04,  05,  05,  07,  08,  08,  08,  12,  12,  14,  15,  16,  16,  17,  18,  18,  19,  19,  19,  20,  20,  21,  24]);


//length == 79
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15,  03, -14,  17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23, -17,  15, -22,  18, -19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06, -20, -06,  19,  22, -07,  01,  03, -07, -20,  10, -25,  08,  11, -05, -02,  10,  03, -11,  15, -03, -03,  08, -22,  14, -22,  14,  06,  20, -22, -05,  02, -15,  16, -04],
          [-25, -25, -24, -22, -22, -22, -22, -21, -20, -20, -20, -19, -19, -17, -16, -15, -14, -11, -11, -10, -10, -07, -07, -07, -06, -06, -05, -05, -05, -04, -03, -03, -02,  00,  01,  01,  02,  02,  03,  03,  03,  03,  04,  04,  05,  06,  06,  06,  06,  08,  08,  08,  10,  10,  10,  11,  11,  12,  13,  14,  14,  14,  15,  15,  15,  15,  15,  16,  17,  17,  17,  18,  19,  20,  21,  22,  22,  23,  23]);
  check([-16, -16,  22,  15,  04, -21,  00,  06,  12, -14, -24, -12,  23,  07,  02,  14, -19,  24, -01, -16,  13,  02, -25, -04,  00, -19,  16, -21, -09,  09,  00,  01,  20,  00, -09,  01,  06,  18, -18, -06, -19, -14,  07, -20,  20, -14,  21, -24, -13, -05, -13, -23,  24, -12, -02, -01, -06, -08,  03,  12, -22, -19, -12,  24,  08,  06,  00, -11,  01,  10,  10,  08,  23, -06,  15,  20,  07,  13,  21],
          [-25, -24, -24, -23, -22, -21, -21, -20, -19, -19, -19, -19, -18, -16, -16, -16, -14, -14, -14, -13, -13, -12, -12, -12, -11, -09, -09, -08, -06, -06, -06, -05, -04, -02, -01, -01,  00,  00,  00,  00,  00,  01,  01,  01,  02,  02,  03,  04,  06,  06,  06,  07,  07,  07,  08,  08,  09,  10,  10,  12,  12,  13,  13,  14,  15,  15,  16,  18,  20,  20,  20,  21,  21,  22,  23,  23,  24,  24,  24]);
  check([ 19, -15, -15,  23, -16, -02, -03,  11,  19,  16, -09,  07,  19, -01,  22,  20, -18,  06, -04, -01, -18, -17, -14,  17, -17,  08,  09, -22, -10, -03, -24, -13,  07, -14, -15,  19,  12,  09,  05,  06,  00,  21, -12, -03, -02, -12, -08,  05,  19,  15, -19, -22, -25, -05,  21, -17, -22,  05, -14, -07, -20, -11,  05,  12,  01,  16,  06, -12,  00, -12,  21, -22, -13, -15,  00,  10,  00,  19, -08],
          [-25, -24, -22, -22, -22, -22, -20, -19, -18, -18, -17, -17, -17, -16, -15, -15, -15, -15, -14, -14, -14, -13, -13, -12, -12, -12, -12, -11, -10, -09, -08, -08, -07, -05, -04, -03, -03, -03, -02, -02, -01, -01,  00,  00,  00,  00,  01,  05,  05,  05,  05,  06,  06,  06,  07,  07,  08,  09,  09,  10,  11,  12,  12,  15,  16,  16,  17,  19,  19,  19,  19,  19,  19,  20,  21,  21,  21,  22,  23]);
  check([-06,  10, -01, -01,  10,  21, -03,  19,  24, -22, -18, -08, -15, -03,  23,  22, -25,  16,  06, -12, -07,  19, -16, -02, -19, -04,  23,  18,  21,  20, -14, -08,  05,  12,  19, -10,  08, -09, -14,  07, -04, -04,  24,  06,  18,  24,  06,  20,  17, -13, -17, -14, -17, -06,  11, -08,  18,  12, -15,  16,  07, -04, -14, -13, -15, -20,  04,  18, -04,  18,  00,  20, -11,  02,  01, -16,  03, -18, -21],
          [-25, -22, -21, -20, -19, -18, -18, -17, -17, -16, -16, -15, -15, -15, -14, -14, -14, -14, -13, -13, -12, -11, -10, -09, -08, -08, -08, -07, -06, -06, -04, -04, -04, -04, -04, -03, -03, -02, -01, -01,  00,  01,  02,  03,  04,  05,  06,  06,  06,  07,  07,  08,  10,  10,  11,  12,  12,  16,  16,  17,  18,  18,  18,  18,  18,  19,  19,  19,  20,  20,  20,  21,  21,  22,  23,  23,  24,  24,  24]);
  check([-04, -04, -11,  09,  07,  08, -05,  24,  03, -18, -14, -05,  14,  08,  06, -22,  18,  13,  08,  14, -14,  01, -11, -19,  17,  18,  08, -24, -03,  17, -20,  20, -09, -04,  04,  23, -21, -24,  22,  10, -14, -17, -20, -25,  18,  13, -19,  14,  01,  14,  03,  12, -08,  19,  21, -16,  13,  04, -13, -13, -02, -08, -18,  14,  13,  13,  12,  19,  16,  11,  06, -23,  20,  11, -20,  15,  01, -14,  04],
          [-25, -24, -24, -23, -22, -21, -20, -20, -20, -19, -19, -18, -18, -17, -16, -14, -14, -14, -14, -13, -13, -11, -11, -09, -08, -08, -05, -05, -04, -04, -04, -03, -02,  01,  01,  01,  03,  03,  04,  04,  04,  06,  06,  07,  08,  08,  08,  08,  09,  10,  11,  11,  12,  12,  13,  13,  13,  13,  13,  14,  14,  14,  14,  14,  15,  16,  17,  17,  18,  18,  18,  19,  19,  20,  20,  21,  22,  23,  24]);
  check([-21, -23, -16,  19, -04, -21,  15,  07,  19, -04,  19, -17,  19, -12, -10,  09, -22, -20, -04,  24,  21, -17,  06,  24, -20, -08, -19,  22,  21, -08,  04, -25, -04,  13,  21,  17,  19,  11, -01,  15, -18, -07,  23, -23,  08, -10,  13,  11, -05, -16,  10, -06, -06, -09, -05,  01,  11,  01, -01,  07,  20, -20,  09, -09,  20,  06, -15,  15, -06,  09,  07,  02, -21,  08,  06,  12,  23, -06, -02],
          [-25, -23, -23, -22, -21, -21, -21, -20, -20, -20, -19, -18, -17, -17, -16, -16, -15, -12, -10, -10, -09, -09, -08, -08, -07, -06, -06, -06, -06, -05, -05, -04, -04, -04, -04, -02, -01, -01,  01,  01,  02,  04,  06,  06,  06,  07,  07,  07,  08,  08,  09,  09,  09,  10,  11,  11,  11,  12,  13,  13,  15,  15,  15,  17,  19,  19,  19,  19,  19,  20,  20,  21,  21,  21,  22,  23,  23,  24,  24]);
  check([-04,  05, -15,  17, -25,  02, -13,  03, -12,  13, -21,  22,  10, -16,  08, -24, -20, -11, -14,  22,  09,  22,  06, -12, -22, -11,  19,  17, -12, -10, -10,  11,  20,  00,  03,  22, -21,  15,  01, -08,  05,  07, -09, -10,  19, -25, -07,  24, -09,  06,  23, -23,  05,  04, -10,  08,  19, -14,  00, -16,  01,  17,  20, -01, -06,  00, -04, -01, -10,  24,  18,  22,  09, -13, -11,  03, -13,  07,  02],
          [-25, -25, -24, -23, -22, -21, -21, -20, -16, -16, -15, -14, -14, -13, -13, -13, -12, -12, -12, -11, -11, -11, -10, -10, -10, -10, -10, -09, -09, -08, -07, -06, -04, -04, -01, -01,  00,  00,  00,  01,  01,  02,  02,  03,  03,  03,  04,  05,  05,  05,  06,  06,  07,  07,  08,  08,  09,  09,  10,  11,  13,  15,  17,  17,  17,  18,  19,  19,  19,  20,  20,  22,  22,  22,  22,  22,  23,  24,  24]);
  check([ 03, -10,  02,  08,  20, -19,  23,  05, -23, -13, -20, -12,  15,  24, -17, -11, -07,  08,  13,  19, -25,  12,  15,  24, -04, -21, -12,  24, -09,  22,  03,  21, -11, -18,  04, -16, -09,  05,  16, -07,  17,  21,  09,  09,  22,  17,  24, -07,  03,  12, -11,  05,  01, -21,  05, -25, -17,  20, -24,  01,  20,  05,  00, -16,  14, -19, -04, -20,  11, -13,  01,  05, -14, -15, -10, -17, -21, -09,  01],
          [-25, -25, -24, -23, -21, -21, -21, -20, -20, -19, -19, -18, -17, -17, -17, -16, -16, -15, -14, -13, -13, -12, -12, -11, -11, -11, -10, -10, -09, -09, -09, -07, -07, -07, -04, -04,  00,  01,  01,  01,  01,  02,  03,  03,  03,  04,  05,  05,  05,  05,  05,  05,  08,  08,  09,  09,  11,  12,  12,  13,  14,  15,  15,  16,  17,  17,  19,  20,  20,  20,  21,  21,  22,  22,  23,  24,  24,  24,  24]);
  check([ 09, -22,  16, -10,  06,  22,  22,  06, -17,  17,  08,  09,  12, -10, -14,  24, -21, -07, -05, -13, -19,  07,  13,  12,  18, -25, -21, -23, -19, -05,  05,  16,  00, -04, -17, -19, -04, -20,  15,  04,  24, -02, -10,  14,  13,  02,  13,  19,  22, -17, -19, -22,  17,  21,  15,  11,  23,  21,  15, -20, -07, -05,  21,  18,  19, -21, -23, -10, -14,  17,  21, -15, -08, -12, -24, -20,  15,  14, -24],
          [-25, -24, -24, -23, -23, -22, -22, -21, -21, -21, -20, -20, -20, -19, -19, -19, -19, -17, -17, -17, -15, -14, -14, -13, -12, -10, -10, -10, -10, -08, -07, -07, -05, -05, -05, -04, -04, -02,  00,  02,  04,  05,  06,  06,  07,  08,  09,  09,  11,  12,  12,  13,  13,  13,  14,  14,  15,  15,  15,  15,  16,  16,  17,  17,  17,  18,  18,  19,  19,  21,  21,  21,  21,  22,  22,  22,  23,  24,  24]);
  check([ 14,  24, -17,  18,  17, -19,  08,  03, -20,  07, -07, -15, -25,  15, -17,  21,  09, -11,  23, -24,  00,  17,  24,  12, -14, -12,  14, -07, -20,  03, -06,  20,  05,  04,  13, -03,  11, -02, -23,  16, -18, -03, -22, -17, -13, -14, -21,  24,  00, -21, -23, -25,  23, -21,  12, -16, -08,  03,  02, -01, -18,  23, -04,  14,  03, -14, -14, -11,  12, -12, -18,  19,  10, -15, -21,  24, -04, -15,  23],
          [-25, -25, -24, -23, -23, -22, -21, -21, -21, -21, -20, -20, -19, -18, -18, -18, -17, -17, -17, -16, -15, -15, -15, -14, -14, -14, -14, -13, -12, -12, -11, -11, -08, -07, -07, -06, -04, -04, -03, -03, -02, -01,  00,  00,  02,  03,  03,  03,  03,  04,  05,  07,  08,  09,  10,  11,  12,  12,  12,  13,  14,  14,  14,  15,  16,  17,  17,  18,  19,  20,  21,  23,  23,  23,  23,  24,  24,  24,  24]);
  check([ 23, -09, -22, -25, -11, -18,  14,  00,  01, -07, -21, -25,  02, -22, -01, -09,  06,  10,  02,  22, -03,  17, -21, -06,  02, -11, -02,  03,  12,  09,  04,  10, -25,  07,  12, -08,  16,  01,  19, -08,  19,  24, -06, -04, -21,  18,  14,  10,  04, -09,  09, -22, -17,  13, -03,  12, -21,  23, -07,  16,  09,  22,  03,  09,  06, -10, -22, -03,  16,  23,  14,  13,  24, -17, -14, -22, -21, -24,  15],
          [-25, -25, -25, -24, -22, -22, -22, -22, -22, -21, -21, -21, -21, -21, -18, -17, -17, -14, -11, -11, -10, -09, -09, -09, -08, -08, -07, -07, -06, -06, -04, -03, -03, -03, -02, -01,  00,  01,  01,  02,  02,  02,  03,  03,  04,  04,  06,  06,  07,  09,  09,  09,  09,  10,  10,  10,  12,  12,  12,  13,  13,  14,  14,  14,  15,  16,  16,  16,  17,  18,  19,  19,  22,  22,  23,  23,  23,  24,  24]);
  check([ 10, -06,  01,  15,  05, -11, -10, -06, -07, -12,  12, -14,  22,  09,  16,  08, -10, -17,  12,  12, -24,  12,  03,  14,  13,  14, -22,  18,  18, -21,  10,  05, -02, -14, -05, -22,  00,  10, -02,  20,  23, -15, -17,  22,  22, -24,  06, -11, -16, -05, -23, -15, -16,  07, -23,  22, -04, -20,  15, -09, -16,  00,  21,  09,  13, -08,  13, -12, -21, -12, -15, -20,  00, -05, -23, -01, -04,  10,  14],
          [-24, -24, -23, -23, -23, -22, -22, -21, -21, -20, -20, -17, -17, -16, -16, -16, -15, -15, -15, -14, -14, -12, -12, -12, -11, -11, -10, -10, -09, -08, -07, -06, -06, -05, -05, -05, -04, -04, -02, -02, -01,  00,  00,  00,  01,  03,  05,  05,  06,  07,  08,  09,  09,  10,  10,  10,  10,  12,  12,  12,  12,  13,  13,  13,  14,  14,  14,  15,  15,  16,  18,  18,  20,  21,  22,  22,  22,  22,  23]);
  check([ 05, -20,  18,  17, -11,  00,  19, -12,  22,  24, -22, -10, -15,  05, -13,  20, -07,  06,  10,  08,  10, -25,  20,  17,  00, -10,  20, -25,  11,  05,  16, -07,  11,  09, -14, -23, -16, -20, -09, -17, -18, -04, -01, -06, -21,  13, -11, -01, -06, -24, -15, -19, -24, -20,  24,  04, -02,  21,  06, -14,  01, -03,  05, -11, -19,  18, -06, -08, -25,  10,  01, -18, -17, -25,  02, -11,  15,  18,  14],
          [-25, -25, -25, -25, -24, -24, -23, -22, -21, -20, -20, -20, -19, -19, -18, -18, -17, -17, -16, -15, -15, -14, -14, -13, -12, -11, -11, -11, -11, -10, -10, -09, -08, -07, -07, -06, -06, -06, -04, -03, -02, -01, -01,  00,  00,  01,  01,  02,  04,  05,  05,  05,  05,  06,  06,  08,  09,  10,  10,  10,  11,  11,  13,  14,  15,  16,  17,  17,  18,  18,  18,  19,  20,  20,  20,  21,  22,  24,  24]);
  check([-16,  20,  24, -08,  23, -21, -09,  02,  04, -13, -17,  16,  16,  05, -04, -18,  13, -11,  01, -19, -11, -12,  09, -01, -03,  09, -22,  11,  01,  21,  02,  10,  18,  01, -21,  17,  08, -04, -06, -13,  10,  05, -20,  03, -15,  01,  11, -24,  17, -13, -18, -16,  03,  16,  10, -25,  02,  13,  13, -22,  11, -07,  15,  07,  19,  19, -01,  04, -08,  20,  19, -20,  00,  24,  08,  13,  03, -04,  16],
          [-25, -24, -22, -22, -21, -21, -20, -20, -19, -18, -18, -17, -16, -16, -15, -13, -13, -13, -12, -11, -11, -09, -08, -08, -07, -06, -04, -04, -04, -03, -01, -01,  00,  01,  01,  01,  01,  02,  02,  02,  03,  03,  03,  04,  04,  05,  05,  07,  08,  08,  09,  09,  10,  10,  10,  11,  11,  11,  13,  13,  13,  13,  15,  16,  16,  16,  16,  17,  17,  18,  19,  19,  19,  20,  20,  21,  23,  24,  24]);
  check([-03,  11,  23,  06, -11,  16, -07, -11, -05, -19, -21,  00,  20, -03, -08,  02, -06, -12, -22, -25,  06,  23,  19,  13,  01,  21, -04, -09, -01,  20, -18,  21,  06, -18,  05,  20,  23,  23, -14, -05, -20, -08,  22, -23,  15, -11,  04, -14,  02,  09, -12, -15,  09, -17,  23, -15, -21, -03,  03,  05, -08,  10,  01,  23,  17, -19,  18,  17, -20, -19, -11, -13, -02, -14, -09, -10,  02,  22,  01],
          [-25, -23, -22, -21, -21, -20, -20, -19, -19, -19, -18, -18, -17, -15, -15, -14, -14, -14, -13, -12, -12, -11, -11, -11, -11, -10, -09, -09, -08, -08, -08, -07, -06, -05, -05, -04, -03, -03, -03, -02, -01,  00,  01,  01,  01,  02,  02,  02,  03,  04,  05,  05,  06,  06,  06,  09,  09,  10,  11,  13,  15,  16,  17,  17,  18,  19,  20,  20,  20,  21,  21,  22,  22,  23,  23,  23,  23,  23,  23]);
  check([-20,  06,  15, -08, -10, -25, -09,  03, -19,  13, -19,  11, -20,  17, -13, -20,  11, -04, -25,  06,  01, -19,  20,  15,  04, -16, -19,  20,  11, -22, -02,  18,  11, -12, -14, -24, -10,  04,  04, -04, -08,  13, -18, -01,  07, -03,  06, -07,  18,  06, -24, -04,  12,  24, -12, -07, -17, -06, -12,  21, -01,  12,  17, -15, -23, -20, -14, -05,  09,  18,  16, -22,  06, -24,  04, -12, -02, -15,  08],
          [-25, -25, -24, -24, -24, -23, -22, -22, -20, -20, -20, -20, -19, -19, -19, -19, -18, -17, -16, -15, -15, -14, -14, -13, -12, -12, -12, -12, -10, -10, -09, -08, -08, -07, -07, -06, -05, -04, -04, -04, -03, -02, -02, -01, -01,  01,  03,  04,  04,  04,  04,  06,  06,  06,  06,  06,  07,  08,  09,  11,  11,  11,  11,  12,  12,  13,  13,  15,  15,  16,  17,  17,  18,  18,  18,  20,  20,  21,  24]);
  check([-09,  18,  12,  12,  05,  11, -25, -25,  19, -04, -09,  17,  22, -20,  09, -18, -18,  14, -05,  04, -02, -10, -04,  03,  23, -03, -18, -12,  22, -08,  22, -12, -15,  09, -23,  17, -05, -21,  20, -09,  02, -14, -17, -01, -09,  20,  08,  00,  09, -22, -20, -15, -04,  03,  13, -06, -23,  23,  08,  24, -10,  07, -13,  03, -09, -09, -03,  13, -03, -08, -19,  24,  03, -09,  00,  21, -14, -17, -03],
          [-25, -25, -23, -23, -22, -21, -20, -20, -19, -18, -18, -18, -17, -17, -15, -15, -14, -14, -13, -12, -12, -10, -10, -09, -09, -09, -09, -09, -09, -09, -08, -08, -06, -05, -05, -04, -04, -04, -03, -03, -03, -03, -02, -01,  00,  00,  02,  03,  03,  03,  03,  04,  05,  07,  08,  08,  09,  09,  09,  11,  12,  12,  13,  13,  14,  17,  17,  18,  19,  20,  20,  21,  22,  22,  22,  23,  23,  24,  24]);
  check([ 21, -12,  02, -17,  09, -20,  21, -19, -18,  21,  14, -17, -11, -04, -05,  17,  14,  13, -09,  04, -15,  11,  10, -14, -11, -24,  11, -14, -12,  19,  10, -14, -16, -11, -06,  21, -06, -08, -23,  03, -11,  16,  13,  03, -11, -17, -03, -22,  23,  13,  09, -17, -01, -06, -04,  16, -05, -18, -21,  10, -22,  14,  21, -13, -22, -08, -15, -01,  10, -13, -21, -24, -20,  17,  04, -04, -23, -24, -01],
          [-24, -24, -24, -23, -23, -22, -22, -22, -21, -21, -20, -20, -19, -18, -18, -17, -17, -17, -17, -16, -15, -15, -14, -14, -14, -13, -13, -12, -12, -11, -11, -11, -11, -11, -09, -08, -08, -06, -06, -06, -05, -05, -04, -04, -04, -03, -01, -01, -01,  02,  03,  03,  04,  04,  09,  09,  10,  10,  10,  10,  11,  11,  13,  13,  13,  14,  14,  14,  16,  16,  17,  17,  19,  21,  21,  21,  21,  21,  23]);
  check([-23,  16, -15, -13, -07,  04,  08, -14, -23,  15, -10,  12,  20, -19,  11, -16, -14, -22, -05,  12,  15,  07,  18,  16,  15,  10, -05, -14,  14, -02,  13,  16, -08,  23, -22,  10,  05,  13,  21,  07,  03, -12, -04, -02, -04, -18,  09,  07, -12, -21, -06, -22,  14, -13,  20,  04,  24, -10,  17,  15,  16,  07,  08, -17,  06,  13,  20, -14,  01,  18,  18, -19, -19, -09,  06,  02, -01, -10, -14],
          [-23, -23, -22, -22, -22, -21, -19, -19, -19, -18, -17, -16, -15, -14, -14, -14, -14, -14, -13, -13, -12, -12, -10, -10, -10, -09, -08, -07, -06, -05, -05, -04, -04, -02, -02, -01,  01,  02,  03,  04,  04,  05,  06,  06,  07,  07,  07,  07,  08,  08,  09,  10,  10,  11,  12,  12,  13,  13,  13,  14,  14,  15,  15,  15,  15,  16,  16,  16,  16,  17,  18,  18,  18,  20,  20,  20,  21,  23,  24]);
  check([ 12, -05,  05,  15, -14,  19,  12,  17,  20, -20,  09,  10,  21, -08, -05, -19, -25, -17, -24, -14,  11,  21, -19,  17, -21, -03, -02,  06,  23,  15,  19,  12, -13,  01,  05, -02, -05, -08, -10, -08, -03, -23, -21, -05, -04, -01,  01, -04,  09,  04,  09, -05,  00,  17, -11,  04, -09,  14, -13, -10,  04, -19, -23,  17,  07,  07, -08, -21, -25,  08, -02, -01,  12,  02,  20, -17, -22, -02,  06],
          [-25, -25, -24, -23, -23, -22, -21, -21, -21, -20, -19, -19, -19, -17, -17, -14, -14, -13, -13, -11, -10, -10, -09, -08, -08, -08, -08, -05, -05, -05, -05, -05, -04, -04, -03, -03, -02, -02, -02, -02, -01, -01,  00,  01,  01,  02,  04,  04,  04,  05,  05,  06,  06,  07,  07,  08,  09,  09,  09,  10,  11,  12,  12,  12,  12,  14,  15,  15,  17,  17,  17,  17,  19,  19,  20,  20,  21,  21,  23]);
  check([ 12, -22, -10, -16,  05, -18, -02, -13, -02, -13, -01,  15,  17,  08,  18,  11, -08,  02, -22, -01,  04,  13,  22, -19,  00,  02, -24,  10,  05, -01, -09, -05,  04,  06,  04, -15,  15, -20, -01, -09, -08,  23, -19, -14,  08, -24,  24, -22,  04, -22,  04, -15,  18,  01, -09, -06, -20, -06, -21,  13,  19, -02, -17, -02, -19,  14,  10,  22,  19, -16, -12, -11, -15, -06,  00,  20, -02,  00, -25],
          [-25, -24, -24, -22, -22, -22, -22, -21, -20, -20, -19, -19, -19, -18, -17, -16, -16, -15, -15, -15, -14, -13, -13, -12, -11, -10, -09, -09, -09, -08, -08, -06, -06, -06, -05, -02, -02, -02, -02, -02, -01, -01, -01, -01,  00,  00,  00,  01,  02,  02,  04,  04,  04,  04,  04,  05,  05,  06,  08,  08,  10,  10,  11,  12,  13,  13,  14,  15,  15,  17,  18,  18,  19,  19,  20,  22,  22,  23,  24]);
  check([-21,  05,  04, -11, -02, -19,  08,  19, -12, -23, -24, -24,  21, -01, -16, -03,  05,  24, -16,  02,  20, -04,  17, -16,  08, -11,  12,  03,  12, -11,  04,  16,  19, -17, -18, -06, -09,  15, -11,  07,  18, -08,  08,  14,  16,  18, -12, -02,  19,  00, -22,  16,  21,  20, -24,  04, -14,  15, -16,  23, -21,  13,  16, -25,  24,  24, -04, -08,  14,  12,  24,  07, -21,  08, -01,  20,  03,  12, -04],
          [-25, -24, -24, -24, -23, -22, -21, -21, -21, -19, -18, -17, -16, -16, -16, -16, -14, -12, -12, -11, -11, -11, -11, -09, -08, -08, -06, -04, -04, -04, -03, -02, -02, -01, -01,  00,  02,  03,  03,  04,  04,  04,  05,  05,  07,  07,  08,  08,  08,  08,  12,  12,  12,  12,  13,  14,  14,  15,  15,  16,  16,  16,  16,  17,  18,  18,  19,  19,  19,  20,  20,  20,  21,  21,  23,  24,  24,  24,  24]);
  check([-01, -13, -01, -10, -15, -06, -07,  14,  06, -17,  24,  06, -11,  14, -02, -09,  15, -03,  13, -17, -12,  00, -18,  21,  05,  17, -05,  02, -03, -16,  23,  21, -01, -03, -11,  09,  19,  09, -01,  02,  20,  00, -17, -14, -11,  08,  03, -18, -18, -09, -10, -04,  16, -01, -06, -02, -06,  14, -24,  16, -25,  24,  15, -01, -01, -19, -16, -07,  17,  10,  22,  12, -13, -18,  24,  03,  16,  02,  10],
          [-25, -24, -19, -18, -18, -18, -18, -17, -17, -17, -16, -16, -15, -14, -13, -13, -12, -11, -11, -11, -10, -10, -09, -09, -07, -07, -06, -06, -06, -05, -04, -03, -03, -03, -02, -02, -01, -01, -01, -01, -01, -01, -01,  00,  00,  02,  02,  02,  03,  03,  05,  06,  06,  08,  09,  09,  10,  10,  12,  13,  14,  14,  14,  15,  15,  16,  16,  16,  17,  17,  19,  20,  21,  21,  22,  23,  24,  24,  24]);
  check([-25,  20, -25, -04,  13,  02,  15, -13,  21,  06, -12,  14,  07, -11,  06, -17,  13,  12, -06, -17,  05, -21, -20, -06, -09, -12, -05,  22, -19,  24,  07, -19,  19,  10,  05,  08, -13, -03,  22, -15, -19, -13, -24,  15,  01,  08,  23, -08, -03, -07,  00, -21, -01,  08, -01,  18,  23,  19,  15, -21,  19, -03, -15,  15, -18,  17,  00, -04, -10, -03,  08, -04,  09,  10, -14, -12, -07, -14,  07],
          [-25, -25, -24, -21, -21, -21, -20, -19, -19, -19, -18, -17, -17, -15, -15, -14, -14, -13, -13, -13, -12, -12, -12, -11, -10, -09, -08, -07, -07, -06, -06, -05, -04, -04, -04, -03, -03, -03, -03, -01, -01,  00,  00,  01,  02,  05,  05,  06,  06,  07,  07,  07,  08,  08,  08,  08,  09,  10,  10,  12,  13,  13,  14,  15,  15,  15,  15,  17,  18,  19,  19,  19,  20,  21,  22,  22,  23,  23,  24]);
  check([ 15,  04, -16,  20, -21,  17, -04,  24,  15, -08,  14,  19, -12, -12, -18, -21, -02,  24,  06,  19, -09, -21,  03,  14,  15, -12, -25, -20,  08, -13,  12,  00,  18,  22, -03,  24,  16,  18,  23,  07, -12,  14,  03,  01, -22,  12,  05,  01,  14, -11, -03, -20, -05, -25,  20, -15, -10,  22, -07, -25, -14, -20,  01, -18, -21,  23, -19,  21,  19, -18,  05, -18,  21, -17,  08, -24,  21, -09,  04],
          [-25, -25, -25, -24, -22, -21, -21, -21, -21, -20, -20, -20, -19, -18, -18, -18, -18, -17, -16, -15, -14, -13, -12, -12, -12, -12, -11, -10, -09, -09, -08, -07, -05, -04, -03, -03, -02,  00,  01,  01,  01,  03,  03,  04,  04,  05,  05,  06,  07,  08,  08,  12,  12,  14,  14,  14,  14,  15,  15,  15,  16,  17,  18,  18,  19,  19,  19,  20,  20,  21,  21,  21,  22,  22,  23,  23,  24,  24,  24]);
  check([ 12,  05, -23,  17, -25, -21,  14, -13, -05,  14,  07, -03, -25,  15,  23, -18,  19,  24, -09,  17,  20, -02, -03, -23, -04,  06,  12, -02,  04, -20, -21, -09,  12, -19, -15,  14, -12, -23, -21,  10,  16,  11, -18,  16,  01, -19, -24, -02, -18, -08, -10, -23,  17,  13, -19, -12, -04,  20,  13, -23, -24,  20, -05,  15, -24,  07,  05, -09,  09,  09, -24,  00, -05, -16, -06,  24, -08, -05, -03],
          [-25, -25, -24, -24, -24, -24, -23, -23, -23, -23, -23, -21, -21, -21, -20, -19, -19, -19, -18, -18, -18, -16, -15, -13, -12, -12, -10, -09, -09, -09, -08, -08, -06, -05, -05, -05, -05, -04, -04, -03, -03, -03, -02, -02, -02,  00,  01,  04,  05,  05,  06,  07,  07,  09,  09,  10,  11,  12,  12,  12,  13,  13,  14,  14,  14,  15,  15,  16,  16,  17,  17,  17,  19,  20,  20,  20,  23,  24,  24]);
  check([-01,  14,  12,  01,  06,  02,  09,  21,  23,  04,  12, -23,  07,  07, -03, -02,  10, -20, -22, -23,  14,  14, -22, -08, -14, -13,  11, -13,  04, -17,  09, -20,  24, -01,  06,  07, -24, -08,  03, -23,  22, -08, -21,  04,  24,  02, -21,  10,  09, -16,  14,  00,  23,  17,  17, -13, -18,  05, -01,  13,  13, -14,  19,  12,  10,  02, -06,  13,  20, -25,  15,  19, -08,  22, -02, -06,  01,  05, -19],
          [-25, -24, -23, -23, -23, -22, -22, -21, -21, -20, -20, -19, -18, -17, -16, -14, -14, -13, -13, -13, -08, -08, -08, -08, -06, -06, -03, -02, -02, -01, -01, -01,  00,  01,  01,  02,  02,  02,  03,  04,  04,  04,  05,  05,  06,  06,  07,  07,  07,  09,  09,  09,  10,  10,  10,  11,  12,  12,  12,  13,  13,  13,  14,  14,  14,  14,  15,  17,  17,  19,  19,  20,  21,  22,  22,  23,  23,  24,  24]);
  check([-15,  14,  20,  10,  15,  14,  05, -23,  21, -15,  03,  10, -24,  14,  06,  15,  01, -17, -15, -10, -20, -13, -18,  24,  06, -21, -02, -25,  05, -20, -17,  15,  19, -22,  03,  09,  18, -15,  13,  16, -05, -06,  01, -04, -17, -18, -13,  12, -07, -01, -23, -02,  13, -14, -02, -06, -09,  23, -03,  21, -22,  07,  14,  22,  11, -06,  09,  06,  04, -01, -03, -24,  18,  24, -02, -21, -17,  12,  18],
          [-25, -24, -24, -23, -23, -22, -22, -21, -21, -20, -20, -18, -18, -17, -17, -17, -17, -15, -15, -15, -15, -14, -13, -13, -10, -09, -07, -06, -06, -06, -05, -04, -03, -03, -02, -02, -02, -02, -01, -01,  01,  01,  03,  03,  04,  05,  05,  06,  06,  06,  07,  09,  09,  10,  10,  11,  12,  12,  13,  13,  14,  14,  14,  14,  15,  15,  15,  16,  18,  18,  18,  19,  20,  21,  21,  22,  23,  24,  24]);
  check([ 01, -12,  20, -23, -22, -19,  00, -01, -01,  00,  23, -04,  03,  06, -13,  02, -08,  06, -12,  23, -15,  13, -03, -14,  08, -02,  11,  14,  07,  00,  07, -15,  15,  04, -11,  18,  13,  14, -07, -13, -11, -09,  10,  19,  22,  22, -03, -09,  05,  12, -09,  17,  00,  14, -21, -14, -13,  17, -23,  21, -05,  10, -16, -15, -09, -02, -19, -19, -12, -01, -06,  04,  17, -19, -01,  17, -19,  23, -17],
          [-23, -23, -22, -21, -19, -19, -19, -19, -19, -17, -16, -15, -15, -15, -14, -14, -13, -13, -13, -12, -12, -12, -11, -11, -09, -09, -09, -09, -08, -07, -06, -05, -04, -03, -03, -02, -02, -01, -01, -01, -01,  00,  00,  00,  00,  01,  02,  03,  04,  04,  05,  06,  06,  07,  07,  08,  10,  10,  11,  12,  13,  13,  14,  14,  14,  15,  17,  17,  17,  17,  18,  19,  20,  21,  22,  22,  23,  23,  23]);
  check([ 11,  10,  02,  04, -12, -09,  10,  01,  05,  04,  03,  02,  24, -10,  11, -13,  07, -16, -07,  13, -01,  19, -16, -21,  11, -07,  05,  03, -01,  03,  14, -15, -10, -07,  16,  03,  09,  01, -21, -09, -19,  10,  20, -20, -25,  06, -08,  09,  18,  12,  00, -08,  06,  11, -02, -07, -21, -20,  23,  03,  10,  14,  16, -24, -18,  07,  06,  18, -16,  13, -15, -10, -02, -20, -03,  00,  14,  17, -15],
          [-25, -24, -21, -21, -21, -20, -20, -20, -19, -18, -16, -16, -16, -15, -15, -15, -13, -12, -10, -10, -10, -09, -09, -08, -08, -07, -07, -07, -07, -03, -02, -02, -01, -01,  00,  00,  01,  01,  02,  02,  03,  03,  03,  03,  03,  04,  04,  05,  05,  06,  06,  06,  07,  07,  09,  09,  10,  10,  10,  10,  11,  11,  11,  11,  12,  13,  13,  14,  14,  14,  16,  16,  17,  18,  18,  19,  20,  23,  24]);


//length == 123
  check([ 23, -20,  08, -10,  12,  11, -16,  03, -19,  00, -06,  02,  21,  14,  15,  03, -14,  17,  22,  06,  06, -25,  04, -21,  05,  01, -24, -07,  15,  23, -17,  15, -22,  18, -19,  17,  04, -10, -05,  10,  17,  15, -11,  13,  06, -20, -06,  19,  22, -07,  01,  03, -07, -20,  10, -25,  08,  11, -05, -02,  10,  03, -11,  15, -03, -03,  08, -22,  14, -22,  14,  06,  20, -22, -05,  02, -15,  16, -04, -16, -16,  22,  15,  04, -21,  00,  06,  12, -14, -24, -12,  23,  07,  02,  14, -19,  24, -01, -16,  13,  02, -25, -04,  00, -19,  16, -21, -09,  09,  00,  01,  20,  00, -09,  01,  06,  18, -18, -06, -19, -14,  07, -20],
        [-25, -25, -25, -24, -24, -22, -22, -22, -22, -21, -21, -21, -20, -20, -20, -20, -19, -19, -19, -19, -19, -18, -17, -16, -16, -16, -16, -15, -14, -14, -14, -12, -11, -11, -10, -10, -09, -09, -07, -07, -07, -06, -06, -06, -05, -05, -05, -04, -04, -03, -03, -02, -01,  00,  00,  00,  00,  00,  01,  01,  01,  01,  02,  02,  02,  02,  03,  03,  03,  03,  04,  04,  04,  05,  06,  06,  06,  06,  06,  06,  07,  07,  08,  08,  08,  09,  10,  10,  10,  11,  11,  12,  12,  13,  13,  14,  14,  14,  14,  15,  15,  15,  15,  15,  15,  16,  16,  17,  17,  17,  18,  18,  19,  20,  20,  21,  22,  22,  22,  23,  23,  23,  24]);
  check([ 20, -14,  21, -24, -13, -05, -13, -23,  24, -12, -02, -01, -06, -08,  03,  12, -22, -19, -12,  24,  08,  06,  00, -11,  01,  10,  10,  08,  23, -06,  15,  20,  07,  13,  21,  19, -15, -15,  23, -16, -02, -03,  11,  19,  16, -09,  07,  19, -01,  22,  20, -18,  06, -04, -01, -18, -17, -14,  17, -17,  08,  09, -22, -10, -03, -24, -13,  07, -14, -15,  19,  12,  09,  05,  06,  00,  21, -12, -03, -02, -12, -08,  05,  19,  15, -19, -22, -25, -05,  21, -17, -22,  05, -14, -07, -20, -11,  05,  12,  01,  16,  06, -12,  00, -12,  21, -22, -13, -15,  00,  10,  00,  19, -08, -06,  10, -01, -01,  10,  21, -03,  19,  24],
        [-25, -24, -24, -23, -22, -22, -22, -22, -22, -20, -19, -19, -18, -18, -17, -17, -17, -16, -15, -15, -15, -15, -14, -14, -14, -14, -13, -13, -13, -13, -12, -12, -12, -12, -12, -12, -11, -11, -10, -09, -08, -08, -08, -07, -06, -06, -06, -05, -05, -04, -03, -03, -03, -03, -02, -02, -02, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  01,  01,  03,  05,  05,  05,  05,  06,  06,  06,  06,  07,  07,  07,  08,  08,  08,  09,  09,  10,  10,  10,  10,  10,  11,  12,  12,  12,  13,  15,  15,  16,  16,  17,  19,  19,  19,  19,  19,  19,  19,  20,  20,  20,  21,  21,  21,  21,  21,  21,  22,  23,  23,  24,  24,  24]);
  check([-22, -18, -08, -15, -03,  23,  22, -25,  16,  06, -12, -07,  19, -16, -02, -19, -04,  23,  18,  21,  20, -14, -08,  05,  12,  19, -10,  08, -09, -14,  07, -04, -04,  24,  06,  18,  24,  06,  20,  17, -13, -17, -14, -17, -06,  11, -08,  18,  12, -15,  16,  07, -04, -14, -13, -15, -20,  04,  18, -04,  18,  00,  20, -11,  02,  01, -16,  03, -18, -21, -04, -04, -11,  09,  07,  08, -05,  24,  03, -18, -14, -05,  14,  08,  06, -22,  18,  13,  08,  14, -14,  01, -11, -19,  17,  18,  08, -24, -03,  17, -20,  20, -09, -04,  04,  23, -21, -24,  22,  10, -14, -17, -20, -25,  18,  13, -19,  14,  01,  14,  03,  12, -08],
        [-25, -25, -24, -24, -22, -22, -21, -21, -20, -20, -20, -19, -19, -19, -18, -18, -18, -17, -17, -17, -16, -16, -15, -15, -15, -14, -14, -14, -14, -14, -14, -14, -13, -13, -12, -11, -11, -11, -10, -09, -09, -08, -08, -08, -08, -07, -06, -05, -05, -04, -04, -04, -04, -04, -04, -04, -04, -03, -03, -02,  00,  01,  01,  01,  02,  03,  03,  03,  04,  04,  05,  06,  06,  06,  06,  07,  07,  07,  08,  08,  08,  08,  08,  09,  10,  11,  12,  12,  12,  13,  13,  14,  14,  14,  14,  16,  16,  17,  17,  17,  18,  18,  18,  18,  18,  18,  18,  18,  19,  19,  20,  20,  20,  20,  21,  22,  22,  23,  23,  23,  24,  24,  24]);
  check([ 19,  21, -16,  13,  04, -13, -13, -02, -08, -18,  14,  13,  13,  12,  19,  16,  11,  06, -23,  20,  11, -20,  15,  01, -14,  04, -21, -23, -16,  19, -04, -21,  15,  07,  19, -04,  19, -17,  19, -12, -10,  09, -22, -20, -04,  24,  21, -17,  06,  24, -20, -08, -19,  22,  21, -08,  04, -25, -04,  13,  21,  17,  19,  11, -01,  15, -18, -07,  23, -23,  08, -10,  13,  11, -05, -16,  10, -06, -06, -09, -05,  01,  11,  01, -01,  07,  20, -20,  09, -09,  20,  06, -15,  15, -06,  09,  07,  02, -21,  08,  06,  12,  23, -06, -02, -04,  05, -15,  17, -25,  02, -13,  03, -12,  13, -21,  22,  10, -16,  08, -24, -20, -11],
        [-25, -25, -24, -23, -23, -23, -22, -21, -21, -21, -21, -20, -20, -20, -20, -20, -19, -18, -18, -17, -17, -16, -16, -16, -16, -15, -15, -14, -13, -13, -13, -12, -12, -11, -10, -10, -09, -09, -08, -08, -08, -07, -06, -06, -06, -06, -05, -05, -04, -04, -04, -04, -04, -02, -02, -01, -01,  01,  01,  01,  02,  02,  03,  04,  04,  04,  05,  06,  06,  06,  06,  07,  07,  07,  08,  08,  08,  09,  09,  09,  10,  10,  11,  11,  11,  11,  11,  12,  12,  13,  13,  13,  13,  13,  13,  14,  15,  15,  15,  15,  16,  17,  17,  19,  19,  19,  19,  19,  19,  19,  20,  20,  20,  21,  21,  21,  21,  22,  22,  23,  23,  24,  24]);
  check([-14,  22,  09,  22,  06, -12, -22, -11,  19,  17, -12, -10, -10,  11,  20,  00,  03,  22, -21,  15,  01, -08,  05,  07, -09, -10,  19, -25, -07,  24, -09,  06,  23, -23,  05,  04, -10,  08,  19, -14,  00, -16,  01,  17,  20, -01, -06,  00, -04, -01, -10,  24,  18,  22,  09, -13, -11,  03, -13,  07,  02,  03, -10,  02,  08,  20, -19,  23,  05, -23, -13, -20, -12,  15,  24, -17, -11, -07,  08,  13,  19, -25,  12,  15,  24, -04, -21, -12,  24, -09,  22,  03,  21, -11, -18,  04, -16, -09,  05,  16, -07,  17,  21,  09,  09,  22,  17,  24, -07,  03,  12, -11,  05,  01, -21,  05, -25, -17,  20, -24,  01,  20,  05],
        [-25, -25, -25, -24, -23, -23, -22, -21, -21, -21, -20, -19, -18, -17, -17, -16, -16, -14, -14, -13, -13, -13, -12, -12, -12, -12, -11, -11, -11, -11, -11, -10, -10, -10, -10, -10, -10, -09, -09, -09, -09, -08, -07, -07, -07, -07, -06, -04, -04, -01, -01,  00,  00,  00,  01,  01,  01,  01,  02,  02,  03,  03,  03,  03,  03,  04,  04,  05,  05,  05,  05,  05,  05,  05,  06,  06,  07,  07,  08,  08,  08,  09,  09,  09,  09,  11,  12,  12,  13,  15,  15,  15,  16,  17,  17,  17,  17,  18,  19,  19,  19,  19,  20,  20,  20,  20,  20,  21,  21,  22,  22,  22,  22,  22,  22,  23,  23,  24,  24,  24,  24,  24,  24]);
  check([ 00, -16,  14, -19, -04, -20,  11, -13,  01,  05, -14, -15, -10, -17, -21, -09,  01,  09, -22,  16, -10,  06,  22,  22,  06, -17,  17,  08,  09,  12, -10, -14,  24, -21, -07, -05, -13, -19,  07,  13,  12,  18, -25, -21, -23, -19, -05,  05,  16,  00, -04, -17, -19, -04, -20,  15,  04,  24, -02, -10,  14,  13,  02,  13,  19,  22, -17, -19, -22,  17,  21,  15,  11,  23,  21,  15, -20, -07, -05,  21,  18,  19, -21, -23, -10, -14,  17,  21, -15, -08, -12, -24, -20,  15,  14, -24,  14,  24, -17,  18,  17, -19,  08,  03, -20,  07, -07, -15, -25,  15, -17,  21,  09, -11,  23, -24,  00,  17,  24,  12, -14, -12,  14],
        [-25, -25, -24, -24, -24, -23, -23, -22, -22, -21, -21, -21, -21, -20, -20, -20, -20, -20, -19, -19, -19, -19, -19, -19, -17, -17, -17, -17, -17, -17, -16, -15, -15, -15, -14, -14, -14, -14, -13, -13, -12, -12, -11, -10, -10, -10, -10, -10, -09, -08, -07, -07, -07, -05, -05, -05, -04, -04, -04, -02,  00,  00,  00,  01,  01,  02,  03,  04,  05,  05,  06,  06,  07,  07,  08,  08,  09,  09,  09,  11,  11,  12,  12,  12,  13,  13,  13,  14,  14,  14,  14,  14,  15,  15,  15,  15,  15,  16,  16,  17,  17,  17,  17,  17,  18,  18,  18,  19,  19,  21,  21,  21,  21,  21,  22,  22,  22,  23,  23,  24,  24,  24,  24]);
  check([-07, -20,  03, -06,  20,  05,  04,  13, -03,  11, -02, -23,  16, -18, -03, -22, -17, -13, -14, -21,  24,  00, -21, -23, -25,  23, -21,  12, -16, -08,  03,  02, -01, -18,  23, -04,  14,  03, -14, -14, -11,  12, -12, -18,  19,  10, -15, -21,  24, -04, -15,  23,  23, -09, -22, -25, -11, -18,  14,  00,  01, -07, -21, -25,  02, -22, -01, -09,  06,  10,  02,  22, -03,  17, -21, -06,  02, -11, -02,  03,  12,  09,  04,  10, -25,  07,  12, -08,  16,  01,  19, -08,  19,  24, -06, -04, -21,  18,  14,  10,  04, -09,  09, -22, -17,  13, -03,  12, -21,  23, -07,  16,  09,  22,  03,  09,  06, -10, -22, -03,  16,  23,  14],
        [-25, -25, -25, -25, -23, -23, -22, -22, -22, -22, -22, -21, -21, -21, -21, -21, -21, -21, -21, -20, -18, -18, -18, -18, -17, -17, -16, -15, -15, -14, -14, -14, -13, -12, -11, -11, -11, -10, -09, -09, -09, -08, -08, -08, -07, -07, -07, -06, -06, -06, -04, -04, -04, -03, -03, -03, -03, -03, -02, -02, -01, -01,  00,  00,  01,  01,  02,  02,  02,  02,  03,  03,  03,  03,  03,  04,  04,  04,  05,  06,  06,  07,  09,  09,  09,  09,  10,  10,  10,  10,  11,  12,  12,  12,  12,  12,  13,  13,  14,  14,  14,  14,  16,  16,  16,  16,  17,  18,  19,  19,  19,  20,  22,  22,  23,  23,  23,  23,  23,  23,  24,  24,  24]);
  check([ 13,  24, -17, -14, -22, -21, -24,  15,  10, -06,  01,  15,  05, -11, -10, -06, -07, -12,  12, -14,  22,  09,  16,  08, -10, -17,  12,  12, -24,  12,  03,  14,  13,  14, -22,  18,  18, -21,  10,  05, -02, -14, -05, -22,  00,  10, -02,  20,  23, -15, -17,  22,  22, -24,  06, -11, -16, -05, -23, -15, -16,  07, -23,  22, -04, -20,  15, -09, -16,  00,  21,  09,  13, -08,  13, -12, -21, -12, -15, -20,  00, -05, -23, -01, -04,  10,  14,  05, -20,  18,  17, -11,  00,  19, -12,  22,  24, -22, -10, -15,  05, -13,  20, -07,  06,  10,  08,  10, -25,  20,  17,  00, -10,  20, -25,  11,  05,  16, -07,  11,  09, -14, -23],
        [-25, -25, -24, -24, -24, -23, -23, -23, -23, -22, -22, -22, -22, -21, -21, -21, -20, -20, -20, -17, -17, -17, -16, -16, -16, -15, -15, -15, -15, -14, -14, -14, -14, -13, -12, -12, -12, -12, -11, -11, -11, -10, -10, -10, -10, -09, -08, -07, -07, -07, -06, -06, -05, -05, -05, -04, -04, -02, -02, -01,  00,  00,  00,  00,  00,  01,  03,  05,  05,  05,  05,  05,  06,  06,  07,  08,  08,  09,  09,  09,  10,  10,  10,  10,  10,  10,  11,  11,  12,  12,  12,  12,  13,  13,  13,  13,  14,  14,  14,  15,  15,  15,  16,  16,  17,  17,  18,  18,  18,  19,  20,  20,  20,  20,  21,  22,  22,  22,  22,  22,  23,  24,  24]);
  check([-16, -20, -09, -17, -18, -04, -01, -06, -21,  13, -11, -01, -06, -24, -15, -19, -24, -20,  24,  04, -02,  21,  06, -14,  01, -03,  05, -11, -19,  18, -06, -08, -25,  10,  01, -18, -17, -25,  02, -11,  15,  18,  14, -16,  20,  24, -08,  23, -21, -09,  02,  04, -13, -17,  16,  16,  05, -04, -18,  13, -11,  01, -19, -11, -12,  09, -01, -03,  09, -22,  11,  01,  21,  02,  10,  18,  01, -21,  17,  08, -04, -06, -13,  10,  05, -20,  03, -15,  01,  11, -24,  17, -13, -18, -16,  03,  16,  10, -25,  02,  13,  13, -22,  11, -07,  15,  07,  19,  19, -01,  04, -08,  20,  19, -20,  00,  24,  08,  13,  03, -04,  16, -03],
        [-25, -25, -25, -24, -24, -24, -22, -22, -21, -21, -21, -20, -20, -20, -20, -19, -19, -19, -18, -18, -18, -18, -17, -17, -17, -16, -16, -16, -15, -15, -14, -13, -13, -13, -12, -11, -11, -11, -11, -11, -09, -09, -08, -08, -08, -07, -06, -06, -06, -06, -04, -04, -04, -04, -03, -03, -03, -02, -01, -01, -01, -01,  00,  01,  01,  01,  01,  01,  01,  02,  02,  02,  02,  03,  03,  03,  04,  04,  04,  05,  05,  05,  06,  07,  08,  08,  09,  09,  10,  10,  10,  10,  11,  11,  11,  13,  13,  13,  13,  13,  14,  15,  15,  16,  16,  16,  16,  17,  17,  18,  18,  18,  19,  19,  19,  20,  20,  21,  21,  23,  24,  24,  24]);
  check([ 11,  23,  06, -11,  16, -07, -11, -05, -19, -21,  00,  20, -03, -08,  02, -06, -12, -22, -25,  06,  23,  19,  13,  01,  21, -04, -09, -01,  20, -18,  21,  06, -18,  05,  20,  23,  23, -14, -05, -20, -08,  22, -23,  15, -11,  04, -14,  02,  09, -12, -15,  09, -17,  23, -15, -21, -03,  03,  05, -08,  10,  01,  23,  17, -19,  18,  17, -20, -19, -11, -13, -02, -14, -09, -10,  02,  22,  01, -20,  06,  15, -08, -10, -25, -09,  03, -19,  13, -19,  11, -20,  17, -13, -20,  11, -04, -25,  06,  01, -19,  20,  15,  04, -16, -19,  20,  11, -22, -02,  18,  11, -12, -14, -24, -10,  04,  04, -04, -08,  13, -18, -01,  07],
        [-25, -25, -25, -24, -23, -22, -22, -21, -21, -20, -20, -20, -20, -20, -19, -19, -19, -19, -19, -19, -19, -18, -18, -18, -17, -16, -15, -15, -14, -14, -14, -14, -13, -13, -12, -12, -12, -11, -11, -11, -11, -10, -10, -10, -09, -09, -09, -08, -08, -08, -08, -08, -07, -06, -05, -05, -04, -04, -04, -03, -03, -02, -02, -01, -01,  00,  01,  01,  01,  01,  02,  02,  02,  03,  03,  04,  04,  04,  04,  05,  05,  06,  06,  06,  06,  06,  07,  09,  09,  10,  11,  11,  11,  11,  11,  13,  13,  13,  15,  15,  15,  16,  17,  17,  17,  18,  18,  19,  20,  20,  20,  20,  20,  21,  21,  22,  22,  23,  23,  23,  23,  23,  23]);
  check([-03,  06, -07,  18,  06, -24, -04,  12,  24, -12, -07, -17, -06, -12,  21, -01,  12,  17, -15, -23, -20, -14, -05,  09,  18,  16, -22,  06, -24,  04, -12, -02, -15,  08, -09,  18,  12,  12,  05,  11, -25, -25,  19, -04, -09,  17,  22, -20,  09, -18, -18,  14, -05,  04, -02, -10, -04,  03,  23, -03, -18, -12,  22, -08,  22, -12, -15,  09, -23,  17, -05, -21,  20, -09,  02, -14, -17, -01, -09,  20,  08,  00,  09, -22, -20, -15, -04,  03,  13, -06, -23,  23,  08,  24, -10,  07, -13,  03, -09, -09, -03,  13, -03, -08, -19,  24,  03, -09,  00,  21, -14, -17, -03,  21, -12,  02, -17,  09, -20,  21, -19, -18,  21],
        [-25, -25, -24, -24, -23, -23, -23, -22, -22, -21, -20, -20, -20, -20, -19, -19, -18, -18, -18, -18, -17, -17, -17, -17, -15, -15, -15, -15, -14, -14, -14, -13, -12, -12, -12, -12, -12, -12, -10, -10, -09, -09, -09, -09, -09, -09, -09, -08, -08, -07, -07, -06, -06, -05, -05, -05, -04, -04, -04, -04, -03, -03, -03, -03, -03, -02, -02, -01, -01,  00,  00,  02,  02,  03,  03,  03,  03,  04,  04,  05,  06,  06,  06,  07,  08,  08,  08,  09,  09,  09,  09,  09,  11,  12,  12,  12,  12,  13,  13,  14,  16,  17,  17,  17,  18,  18,  18,  19,  20,  20,  21,  21,  21,  21,  21,  22,  22,  22,  23,  23,  24,  24,  24]);
  check([ 14, -17, -11, -04, -05,  17,  14,  13, -09,  04, -15,  11,  10, -14, -11, -24,  11, -14, -12,  19,  10, -14, -16, -11, -06,  21, -06, -08, -23,  03, -11,  16,  13,  03, -11, -17, -03, -22,  23,  13,  09, -17, -01, -06, -04,  16, -05, -18, -21,  10, -22,  14,  21, -13, -22, -08, -15, -01,  10, -13, -21, -24, -20,  17,  04, -04, -23, -24, -01, -23,  16, -15, -13, -07,  04,  08, -14, -23,  15, -10,  12,  20, -19,  11, -16, -14, -22, -05,  12,  15,  07,  18,  16,  15,  10, -05, -14,  14, -02,  13,  16, -08,  23, -22,  10,  05,  13,  21,  07,  03, -12, -04, -02, -04, -18,  09,  07, -12, -21, -06, -22,  14, -13],
        [-24, -24, -24, -23, -23, -23, -23, -22, -22, -22, -22, -22, -22, -21, -21, -21, -20, -19, -18, -18, -17, -17, -17, -16, -16, -15, -15, -15, -14, -14, -14, -14, -14, -14, -13, -13, -13, -13, -12, -12, -12, -11, -11, -11, -11, -11, -10, -09, -08, -08, -08, -07, -06, -06, -06, -06, -05, -05, -05, -05, -04, -04, -04, -04, -04, -03, -02, -02, -01, -01, -01,  03,  03,  03,  04,  04,  04,  05,  07,  07,  07,  08,  09,  09,  10,  10,  10,  10,  10,  10,  11,  11,  11,  12,  12,  13,  13,  13,  13,  13,  14,  14,  14,  14,  14,  15,  15,  15,  16,  16,  16,  16,  16,  17,  17,  18,  19,  20,  21,  21,  21,  23,  23]);
  check([ 20,  04,  24, -10,  17,  15,  16,  07,  08, -17,  06,  13,  20, -14,  01,  18,  18, -19, -19, -09,  06,  02, -01, -10, -14,  12, -05,  05,  15, -14,  19,  12,  17,  20, -20,  09,  10,  21, -08, -05, -19, -25, -17, -24, -14,  11,  21, -19,  17, -21, -03, -02,  06,  23,  15,  19,  12, -13,  01,  05, -02, -05, -08, -10, -08, -03, -23, -21, -05, -04, -01,  01, -04,  09,  04,  09, -05,  00,  17, -11,  04, -09,  14, -13, -10,  04, -19, -23,  17,  07,  07, -08, -21, -25,  08, -02, -01,  12,  02,  20, -17, -22, -02,  06,  12, -22, -10, -16,  05, -18, -02, -13, -02, -13, -01,  15,  17,  08,  18,  11, -08,  02, -22],
        [-25, -25, -24, -23, -23, -22, -22, -22, -21, -21, -21, -20, -19, -19, -19, -19, -19, -18, -17, -17, -17, -16, -14, -14, -14, -14, -13, -13, -13, -13, -11, -10, -10, -10, -10, -10, -09, -09, -08, -08, -08, -08, -08, -05, -05, -05, -05, -05, -04, -04, -03, -03, -02, -02, -02, -02, -02, -02, -01, -01, -01, -01,  00,  01,  01,  01,  02,  02,  02,  04,  04,  04,  04,  05,  05,  05,  06,  06,  06,  06,  07,  07,  07,  08,  08,  08,  09,  09,  09,  10,  11,  11,  12,  12,  12,  12,  12,  13,  14,  15,  15,  15,  15,  16,  17,  17,  17,  17,  17,  17,  18,  18,  18,  19,  19,  20,  20,  20,  20,  21,  21,  23,  24]);
  check([-01,  04,  13,  22, -19,  00,  02, -24,  10,  05, -01, -09, -05,  04,  06,  04, -15,  15, -20, -01, -09, -08,  23, -19, -14,  08, -24,  24, -22,  04, -22,  04, -15,  18,  01, -09, -06, -20, -06, -21,  13,  19, -02, -17, -02, -19,  14,  10,  22,  19, -16, -12, -11, -15, -06,  00,  20, -02,  00, -25, -21,  05,  04, -11, -02, -19,  08,  19, -12, -23, -24, -24,  21, -01, -16, -03,  05,  24, -16,  02,  20, -04,  17, -16,  08, -11,  12,  03,  12, -11,  04,  16,  19, -17, -18, -06, -09,  15, -11,  07,  18, -08,  08,  14,  16,  18, -12, -02,  19,  00, -22,  16,  21,  20, -24,  04, -14,  15, -16,  23, -21,  13,  16],
        [-25, -24, -24, -24, -24, -24, -23, -22, -22, -22, -21, -21, -21, -20, -20, -19, -19, -19, -19, -18, -17, -17, -16, -16, -16, -16, -16, -15, -15, -15, -14, -14, -12, -12, -12, -11, -11, -11, -11, -11, -09, -09, -09, -09, -08, -08, -06, -06, -06, -06, -05, -04, -03, -02, -02, -02, -02, -02, -01, -01, -01, -01,  00,  00,  00,  00,  01,  02,  02,  03,  04,  04,  04,  04,  04,  04,  04,  04,  05,  05,  05,  06,  07,  08,  08,  08,  08,  10,  10,  12,  12,  13,  13,  13,  14,  14,  15,  15,  15,  16,  16,  16,  16,  17,  18,  18,  18,  19,  19,  19,  19,  19,  20,  20,  20,  21,  21,  22,  22,  23,  23,  24,  24]);
  check([-25,  24,  24, -04, -08,  14,  12,  24,  07, -21,  08, -01,  20,  03,  12, -04, -01, -13, -01, -10, -15, -06, -07,  14,  06, -17,  24,  06, -11,  14, -02, -09,  15, -03,  13, -17, -12,  00, -18,  21,  05,  17, -05,  02, -03, -16,  23,  21, -01, -03, -11,  09,  19,  09, -01,  02,  20,  00, -17, -14, -11,  08,  03, -18, -18, -09, -10, -04,  16, -01, -06, -02, -06,  14, -24,  16, -25,  24,  15, -01, -01, -19, -16, -07,  17,  10,  22,  12, -13, -18,  24,  03,  16,  02,  10, -25,  20, -25, -04,  13,  02,  15, -13,  21,  06, -12,  14,  07, -11,  06, -17,  13,  12, -06, -17,  05, -21, -20, -06, -09, -12, -05,  22],
        [-25, -25, -25, -25, -24, -21, -21, -20, -19, -18, -18, -18, -18, -17, -17, -17, -17, -17, -16, -16, -15, -14, -13, -13, -13, -12, -12, -12, -11, -11, -11, -11, -10, -10, -09, -09, -09, -08, -07, -07, -06, -06, -06, -06, -06, -05, -05, -04, -04, -04, -04, -03, -03, -03, -02, -02, -01, -01, -01, -01, -01, -01, -01, -01,  00,  00,  02,  02,  02,  02,  03,  03,  03,  05,  05,  06,  06,  06,  06,  07,  07,  08,  08,  09,  09,  10,  10,  12,  12,  12,  12,  13,  13,  13,  14,  14,  14,  14,  14,  15,  15,  15,  16,  16,  16,  17,  17,  19,  20,  20,  20,  21,  21,  21,  22,  22,  23,  24,  24,  24,  24,  24,  24]);
  check([-19,  24,  07, -19,  19,  10,  05,  08, -13, -03,  22, -15, -19, -13, -24,  15,  01,  08,  23, -08, -03, -07,  00, -21, -01,  08, -01,  18,  23,  19,  15, -21,  19, -03, -15,  15, -18,  17,  00, -04, -10, -03,  08, -04,  09,  10, -14, -12, -07, -14,  07,  15,  04, -16,  20, -21,  17, -04,  24,  15, -08,  14,  19, -12, -12, -18, -21, -02,  24,  06,  19, -09, -21,  03,  14,  15, -12, -25, -20,  08, -13,  12,  00,  18,  22, -03,  24,  16,  18,  23,  07, -12,  14,  03,  01, -22,  12,  05,  01,  14, -11, -03, -20, -05, -25,  20, -15, -10,  22, -07, -25, -14, -20,  01, -18, -21,  23, -19,  21,  19, -18,  05, -18],
        [-25, -25, -25, -24, -22, -21, -21, -21, -21, -21, -21, -20, -20, -20, -19, -19, -19, -19, -18, -18, -18, -18, -18, -16, -15, -15, -15, -14, -14, -14, -13, -13, -13, -12, -12, -12, -12, -12, -11, -10, -10, -09, -08, -08, -07, -07, -07, -05, -04, -04, -04, -03, -03, -03, -03, -03, -03, -02, -01, -01,  00,  00,  00,  01,  01,  01,  01,  03,  03,  04,  05,  05,  05,  06,  07,  07,  07,  08,  08,  08,  08,  08,  09,  10,  10,  12,  12,  14,  14,  14,  14,  15,  15,  15,  15,  15,  15,  16,  17,  17,  18,  18,  18,  19,  19,  19,  19,  19,  19,  20,  20,  21,  22,  22,  22,  23,  23,  23,  23,  24,  24,  24,  24]);
  check([ 21, -17,  08, -24,  21, -09,  04,  12,  05, -23,  17, -25, -21,  14, -13, -05,  14,  07, -03, -25,  15,  23, -18,  19,  24, -09,  17,  20, -02, -03, -23, -04,  06,  12, -02,  04, -20, -21, -09,  12, -19, -15,  14, -12, -23, -21,  10,  16,  11, -18,  16,  01, -19, -24, -02, -18, -08, -10, -23,  17,  13, -19, -12, -04,  20,  13, -23, -24,  20, -05,  15, -24,  07,  05, -09,  09,  09, -24,  00, -05, -16, -06,  24, -08, -05, -03, -01,  14,  12,  01,  06,  02,  09,  21,  23,  04,  12, -23,  07,  07, -03, -02,  10, -20, -22, -23,  14,  14, -22, -08, -14, -13,  11, -13,  04, -17,  09, -20,  24, -01,  06,  07, -24],
        [-25, -25, -24, -24, -24, -24, -24, -24, -23, -23, -23, -23, -23, -23, -23, -22, -22, -21, -21, -21, -20, -20, -20, -19, -19, -19, -18, -18, -18, -17, -17, -16, -15, -14, -13, -13, -13, -12, -12, -10, -09, -09, -09, -09, -08, -08, -08, -06, -05, -05, -05, -05, -04, -04, -03, -03, -03, -03, -02, -02, -02, -02, -01, -01,  00,  01,  01,  02,  04,  04,  04,  04,  05,  05,  06,  06,  06,  07,  07,  07,  07,  07,  08,  09,  09,  09,  09,  10,  10,  11,  11,  12,  12,  12,  12,  12,  13,  13,  14,  14,  14,  14,  14,  14,  15,  15,  16,  16,  17,  17,  17,  19,  20,  20,  20,  21,  21,  21,  23,  23,  24,  24,  24]);
  check([-08,  03, -23,  22, -08, -21,  04,  24,  02, -21,  10,  09, -16,  14,  00,  23,  17,  17, -13, -18,  05, -01,  13,  13, -14,  19,  12,  10,  02, -06,  13,  20, -25,  15,  19, -08,  22, -02, -06,  01,  05, -19, -15,  14,  20,  10,  15,  14,  05, -23,  21, -15,  03,  10, -24,  14,  06,  15,  01, -17, -15, -10, -20, -13, -18,  24,  06, -21, -02, -25,  05, -20, -17,  15,  19, -22,  03,  09,  18, -15,  13,  16, -05, -06,  01, -04, -17, -18, -13,  12, -07, -01, -23, -02,  13, -14, -02, -06, -09,  23, -03,  21, -22,  07,  14,  22,  11, -06,  09,  06,  04, -01, -03, -24,  18,  24, -02, -21, -17,  12,  18,  01, -12],
        [-25, -25, -24, -24, -23, -23, -23, -22, -22, -21, -21, -21, -21, -20, -20, -19, -18, -18, -18, -17, -17, -17, -17, -16, -15, -15, -15, -15, -14, -14, -13, -13, -13, -12, -10, -09, -08, -08, -08, -07, -06, -06, -06, -06, -06, -05, -04, -03, -03, -02, -02, -02, -02, -02, -01, -01, -01,  00,  01,  01,  01,  01,  02,  02,  03,  03,  03,  04,  04,  05,  05,  05,  05,  06,  06,  06,  07,  09,  09,  09,  10,  10,  10,  10,  11,  12,  12,  12,  13,  13,  13,  13,  13,  14,  14,  14,  14,  14,  15,  15,  15,  15,  16,  17,  17,  18,  18,  18,  19,  19,  19,  20,  20,  21,  21,  22,  22,  22,  23,  23,  24,  24,  24]);
  check([ 20, -23, -22, -19,  00, -01, -01,  00,  23, -04,  03,  06, -13,  02, -08,  06, -12,  23, -15,  13, -03, -14,  08, -02,  11,  14,  07,  00,  07, -15,  15,  04, -11,  18,  13,  14, -07, -13, -11, -09,  10,  19,  22,  22, -03, -09,  05,  12, -09,  17,  00,  14, -21, -14, -13,  17, -23,  21, -05,  10, -16, -15, -09, -02, -19, -19, -12, -01, -06,  04,  17, -19, -01,  17, -19,  23, -17,  11,  10,  02,  04, -12, -09,  10,  01,  05,  04,  03,  02,  24, -10,  11, -13,  07, -16, -07,  13, -01,  19, -16, -21,  11, -07,  05,  03, -01,  03,  14, -15, -10, -07,  16,  03,  09,  01, -21, -09, -19,  10,  20, -20, -25,  06],
        [-25, -23, -23, -22, -21, -21, -21, -20, -19, -19, -19, -19, -19, -19, -17, -16, -16, -16, -15, -15, -15, -15, -14, -14, -13, -13, -13, -13, -12, -12, -12, -11, -11, -10, -10, -09, -09, -09, -09, -09, -09, -08, -07, -07, -07, -07, -06, -05, -04, -03, -03, -02, -02, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  01,  01,  02,  02,  02,  03,  03,  03,  03,  03,  04,  04,  04,  04,  05,  05,  05,  06,  06,  06,  07,  07,  07,  08,  09,  10,  10,  10,  10,  10,  11,  11,  11,  11,  12,  13,  13,  13,  14,  14,  14,  14,  15,  16,  17,  17,  17,  17,  18,  19,  19,  20,  20,  21,  22,  22,  23,  23,  23,  24]);
  check([-08,  09,  18,  12,  00, -08,  06,  11, -02, -07, -21, -20,  23,  03,  10,  14,  16, -24, -18,  07,  06,  18, -16,  13, -15, -10, -02, -20, -03,  00,  14,  17, -15,  09,  04,  12, -22, -14, -02,  02,  06,  03,  09,  06, -17, -05, -04,  24, -02,  05,  07, -21, -01,  16,  17,  11, -17, -10,  18,  07,  18,  07,  01, -22, -09, -19,  15, -03, -06, -10, -24, -23,  20,  10,  08, -21, -18, -19, -22,  05,  12,  12,  12, -14,  05,  04,  24,  15,  22,  17, -02,  15,  02, -24,  20,  20, -18,  12, -08,  03, -23, -07,  05,  23, -21, -09, -21, -12, -03, -16,  19, -16,  22,  06, -03,  02, -13, -04, -07, -14, -09,  18, -22],
        [-24, -24, -24, -23, -23, -22, -22, -22, -22, -21, -21, -21, -21, -21, -20, -20, -19, -19, -18, -18, -18, -17, -17, -16, -16, -16, -15, -15, -14, -14, -14, -13, -12, -10, -10, -10, -09, -09, -09, -08, -08, -08, -07, -07, -07, -06, -05, -04, -04, -03, -03, -03, -03, -02, -02, -02, -02, -02, -01,  00,  00,  01,  02,  02,  02,  03,  03,  03,  04,  04,  05,  05,  05,  05,  06,  06,  06,  06,  06,  07,  07,  07,  07,  08,  09,  09,  09,  10,  10,  11,  11,  12,  12,  12,  12,  12,  12,  13,  14,  14,  15,  15,  15,  16,  16,  17,  17,  17,  18,  18,  18,  18,  18,  19,  20,  20,  20,  22,  22,  23,  23,  24,  24]);
  check([ 18,  19,  23,  15, -21,  10, -17,  09,  15,  03, -08,  15,  09,  08,  19,  23, -18, -22,  19, -08, -25,  00,  16,  05,  14, -10, -25, -24,  06,  20, -21,  01,  14, -20, -07,  20,  17,  03,  07,  07, -18,  24, -03,  16,  09, -07,  16,  16, -01,  10, -15, -01, -13, -23, -19, -21, -06, -19, -20, -25, -22, -14,  03,  20, -09,  21,  15, -16,  00, -03,  18,  09, -02, -07,  02, -18,  11, -06, -24, -15, -19, -14,  12, -04, -10,  20,  00,  09, -21,  07,  11, -18,  18, -09, -23, -13, -12,  20, -02,  15, -06, -09,  01,  18,  11, -22, -23, -02, -01, -22,  10,  06, -08, -01, -21,  09, -05,  04, -04, -01, -14, -18,  08],
        [-25, -25, -25, -24, -24, -23, -23, -23, -22, -22, -22, -22, -21, -21, -21, -21, -21, -20, -20, -19, -19, -19, -18, -18, -18, -18, -18, -17, -16, -15, -15, -14, -14, -14, -13, -13, -12, -10, -10, -09, -09, -09, -08, -08, -08, -07, -07, -07, -06, -06, -06, -05, -04, -04, -03, -03, -02, -02, -02, -01, -01, -01, -01, -01,  00,  00,  00,  01,  01,  02,  03,  03,  03,  04,  05,  06,  06,  07,  07,  07,  08,  08,  09,  09,  09,  09,  09,  09,  10,  10,  10,  11,  11,  11,  12,  14,  14,  15,  15,  15,  15,  15,  16,  16,  16,  16,  17,  18,  18,  18,  18,  19,  19,  19,  20,  20,  20,  20,  20,  21,  23,  23,  24]);
  check([-19, -01,  11, -07,  14,  06,  18,  06, -23, -13, -18,  20,  23, -13, -25, -02,  12, -20, -16, -05, -01,  10, -01, -16, -18, -20,  05,  08, -07,  14, -08, -01, -12, -22,  20, -21,  11,  13,  10,  13, -23,  19,  11, -24, -18,  11, -01,  21,  18,  10, -07, -07, -04,  19,  02,  05,  24, -16, -11, -08,  23,  06,  18, -13,  11,  13, -07, -03,  04, -19, -13,  06, -25,  23,  07, -16,  11, -16, -18,  05,  21,  00,  23, -08, -06,  02, -25, -05,  11, -11,  14,  09,  20,  08,  23,  06, -04, -08, -20, -23, -02, -08,  11,  00, -07, -05,  10, -19,  04,  19,  11,  01, -05,  11,  20,  16, -12,  20, -13, -24, -16, -24,  11],
        [-25, -25, -25, -24, -24, -24, -23, -23, -23, -22, -21, -20, -20, -20, -19, -19, -19, -18, -18, -18, -18, -16, -16, -16, -16, -16, -16, -13, -13, -13, -13, -13, -12, -12, -11, -11, -08, -08, -08, -08, -08, -07, -07, -07, -07, -07, -07, -06, -05, -05, -05, -05, -04, -04, -03, -02, -02, -01, -01, -01, -01, -01,  00,  00,  01,  02,  02,  04,  04,  05,  05,  05,  06,  06,  06,  06,  06,  07,  08,  08,  09,  10,  10,  10,  10,  11,  11,  11,  11,  11,  11,  11,  11,  11,  11,  11,  12,  13,  13,  13,  14,  14,  14,  16,  18,  18,  18,  19,  19,  19,  20,  20,  20,  20,  20,  21,  21,  23,  23,  23,  23,  23,  24]);
  check([-19,  09,  11,  12, -17, -22,  17, -13,  03, -13,  23, -19,  07, -06,  18,  13, -25,  12,  00,  01, -18, -14, -01,  24,  02, -06, -12,  03,  06, -11, -09,  12, -24, -22,  00, -14, -19, -06,  00,  12,  08, -01,  18, -10,  20,  11, -21,  20, -02,  06, -03,  08,  19,  23,  09, -04, -08,  22, -23,  23, -12, -07,  13, -11, -04, -10,  02,  05,  09, -22, -06, -05,  04, -13,  10, -01, -25,  16, -06,  00, -03,  18, -17, -08,  16,  17,  15, -14,  14,  17, -14, -22, -14,  24, -06,  09, -11,  24, -11, -24, -23,  08, -04,  08,  22, -17, -18,  22, -25,  03,  00, -03, -01,  08,  16, -10,  03,  07,  03, -06, -01,  15, -01],
        [-25, -25, -25, -24, -24, -23, -23, -22, -22, -22, -22, -21, -19, -19, -19, -18, -18, -17, -17, -17, -14, -14, -14, -14, -14, -13, -13, -13, -12, -12, -11, -11, -11, -11, -10, -10, -10, -09, -08, -08, -07, -06, -06, -06, -06, -06, -06, -06, -05, -04, -04, -04, -03, -03, -03, -02, -01, -01, -01, -01, -01, -01,  00,  00,  00,  00,  00,  01,  02,  02,  03,  03,  03,  03,  03,  04,  05,  06,  06,  07,  07,  08,  08,  08,  08,  08,  09,  09,  09,  09,  10,  11,  11,  12,  12,  12,  12,  13,  13,  14,  15,  15,  16,  16,  16,  17,  17,  17,  18,  18,  18,  19,  20,  20,  22,  22,  22,  23,  23,  23,  24,  24,  24]);
  check([ 10,  16,  19, -05, -19,  20,  11, -16,  24, -03,  07,  07, -06,  15,  16, -06,  17, -06,  19, -08,  20,  04, -17, -14, -16,  15,  16,  04, -08,  06, -22, -21,  00,  24,  01,  08,  19, -12,  17,  18,  10,  01,  02, -19, -07, -07,  02, -14,  15, -02,  05,  10, -22,  13,  23, -11,  06,  15,  18,  23, -02,  24, -23,  23,  23,  04,  06,  20,  19,  00,  15,  06, -24, -07,  14, -05,  13, -08,  08,  03,  15, -12, -09,  20, -24, -11, -15,  09, -19, -20,  07,  05, -21,  12,  05, -22, -07, -11, -25, -11,  14,  15,  20,  16, -15,  11, -12,  24, -22,  21,  04,  21, -16,  22,  16, -13, -13, -22,  22, -05, -14,  06, -23],
        [-25, -24, -24, -23, -23, -22, -22, -22, -22, -22, -21, -21, -20, -19, -19, -19, -17, -16, -16, -16, -15, -15, -14, -14, -14, -13, -13, -12, -12, -12, -11, -11, -11, -11, -09, -08, -08, -08, -07, -07, -07, -07, -06, -06, -06, -05, -05, -05, -03, -02, -02,  00,  00,  01,  01,  02,  02,  03,  04,  04,  04,  04,  05,  05,  05,  06,  06,  06,  06,  06,  07,  07,  07,  08,  08,  09,  10,  10,  10,  11,  11,  12,  13,  13,  14,  14,  15,  15,  15,  15,  15,  15,  15,  16,  16,  16,  16,  16,  17,  17,  18,  18,  19,  19,  19,  19,  20,  20,  20,  20,  20,  21,  21,  22,  22,  23,  23,  23,  23,  24,  24,  24,  24]);
  check([-10, -05,  08, -05,  15,  24, -05, -21,  13, -12, -24,  06, -02,  13,  19, -01,  18,  17, -19,  14, -22, -22,  08, -09, -10,  13, -10,  13,  24,  23,  15, -10, -06,  00,  12, -16, -01, -17, -11, -12, -02, -08,  21,  21, -18,  18, -02, -24,  12,  04,  17,  16,  07,  00, -18, -25, -11, -01,  13, -10, -03,  03,  07,  16, -21, -05, -22,  05,  05, -06,  20, -22,  11,  17, -24,  19,  12, -01,  22, -01, -22,  14, -10,  13, -10, -01,  13,  06,  23,  01,  21, -04, -19,  04, -11, -13, -24, -08,  18,  06,  11,  13,  11,  00,  05,  12, -04, -06, -13, -07,  19, -08, -18, -16, -20, -03,  09,  20, -20,  09, -04, -21, -18],
        [-25, -24, -24, -24, -24, -22, -22, -22, -22, -22, -21, -21, -21, -20, -20, -19, -19, -18, -18, -18, -18, -17, -16, -16, -13, -13, -12, -12, -11, -11, -11, -10, -10, -10, -10, -10, -10, -10, -09, -08, -08, -08, -07, -06, -06, -06, -05, -05, -05, -05, -04, -04, -04, -03, -03, -02, -02, -02, -01, -01, -01, -01, -01, -01,  00,  00,  00,  01,  03,  04,  04,  05,  05,  05,  06,  06,  06,  07,  07,  08,  08,  09,  09,  11,  11,  11,  12,  12,  12,  12,  13,  13,  13,  13,  13,  13,  13,  13,  14,  14,  15,  15,  16,  16,  17,  17,  17,  18,  18,  18,  19,  19,  19,  20,  20,  21,  21,  21,  22,  23,  23,  24,  24]);
  check([ 05,  10, -03,  17,  11,  16,  10, -08,  05, -01, -20, -20, -19,  17,  01,  03, -19,  19, -03, -01, -22,  06,  06,  03, -08,  02,  08, -21, -25,  12, -14,  05, -01,  10, -02, -15,  02, -17,  04, -18,  09,  09, -11,  18,  02,  17, -04,  10,  13,  18, -14,  16,  01,  18, -06,  21, -03, -20, -23, -03,  19, -12, -22, -06,  24,  01,  04,  03,  11, -14,  12, -02,  20, -24, -07, -01,  20,  14, -13,  08,  09, -02,  01, -13, -09,  21, -15,  13, -24, -13, -14,  22,  01, -09, -09,  02,  19,  23, -18,  05, -16,  19, -20, -19,  22, -02,  08,  17, -13,  20,  00,  23, -07, -24, -14,  12,  24, -02,  00, -23,  11,  13, -25],
        [-25, -25, -24, -24, -24, -23, -23, -22, -22, -21, -20, -20, -20, -20, -19, -19, -19, -18, -18, -17, -16, -15, -15, -14, -14, -14, -14, -14, -13, -13, -13, -13, -12, -11, -09, -09, -09, -08, -08, -07, -07, -06, -06, -04, -03, -03, -03, -03, -02, -02, -02, -02, -02, -01, -01, -01, -01,  00,  00,  01,  01,  01,  01,  01,  02,  02,  02,  02,  03,  03,  03,  04,  04,  05,  05,  05,  05,  06,  06,  08,  08,  08,  09,  09,  09,  10,  10,  10,  10,  11,  11,  11,  12,  12,  12,  13,  13,  13,  14,  16,  16,  17,  17,  17,  17,  18,  18,  18,  19,  19,  19,  19,  20,  20,  20,  21,  21,  22,  22,  23,  23,  24,  24]);
  check([-11, -21, -09,  18,  23, -11, -25,  06,  00,  19,  11,  07,  16, -13, -10,  08,  01, -15, -15,  00,  05, -12,  13, -08, -12, -14,  20, -10,  24,  08, -08, -12,  15,  09, -19,  15, -25, -17, -02,  01, -23, -13, -15,  20, -01,  00,  03, -25,  12,  15,  02, -06, -21, -10,  12, -06,  04,  07,  11,  03, -08, -21,  19, -16,  15, -23, -25,  15, -14,  00, -07, -12,  12,  03, -14, -12, -20,  16, -11,  19,  07,  18, -11,  13, -16, -22, -18,  15,  10,  18, -07, -21,  24, -13, -11,  14, -08, -09,  07,  03,  18, -25,  18,  06,  06, -21,  21,  11,  21, -13,  08,  05, -19,  22, -07, -08, -25,  00, -18,  12, -05,  00,  16],
        [-25, -25, -25, -25, -25, -25, -23, -23, -22, -21, -21, -21, -21, -21, -20, -19, -19, -18, -18, -17, -16, -16, -15, -15, -15, -14, -14, -14, -13, -13, -13, -13, -12, -12, -12, -12, -12, -11, -11, -11, -11, -11, -10, -10, -10, -09, -09, -08, -08, -08, -08, -08, -07, -07, -07, -06, -06, -05, -02, -01,  00,  00,  00,  00,  00,  00,  01,  01,  02,  03,  03,  03,  03,  04,  05,  05,  06,  06,  06,  07,  07,  07,  07,  08,  08,  08,  09,  10,  11,  11,  11,  12,  12,  12,  12,  13,  13,  14,  15,  15,  15,  15,  15,  15,  16,  16,  16,  18,  18,  18,  18,  18,  19,  19,  19,  20,  20,  21,  21,  22,  23,  24,  24]);
  check([-05,  15, -18, -16, -18,  00,  16,  12, -06,  19,  07, -23,  00,  12,  23, -12,  10, -14,  21, -10, -06,  20,  08,  11,  20, -17,  18,  07,  05, -05,  01, -25, -13,  10, -15, -04, -14, -22, -17,  05,  22,  16,  07, -01,  05,  07,  15, -10,  20,  11,  05, -11,  07, -10, -25,  04,  00,  20, -11, -20, -07,  15, -19,  05,  00, -07, -23,  11, -04, -13, -07, -04, -20, -23,  22,  10,  10,  12, -25,  05, -01,  07,  20, -17,  22,  22,  14, -03,  20, -22,  05, -12,  18,  11,  18, -04, -19,  22, -16,  02, -15,  05, -25, -10,  07,  23, -22, -08,  10, -20, -25, -14,  13,  22, -04,  12,  21, -14, -13,  16, -11,  17, -21],
        [-25, -25, -25, -25, -25, -23, -23, -23, -22, -22, -22, -21, -20, -20, -20, -19, -19, -18, -18, -17, -17, -17, -16, -16, -15, -15, -14, -14, -14, -14, -13, -13, -13, -12, -12, -11, -11, -11, -10, -10, -10, -10, -08, -07, -07, -07, -06, -06, -05, -05, -04, -04, -04, -04, -04, -03, -01, -01,  00,  00,  00,  00,  01,  02,  04,  05,  05,  05,  05,  05,  05,  05,  05,  07,  07,  07,  07,  07,  07,  07,  08,  10,  10,  10,  10,  10,  11,  11,  11,  11,  12,  12,  12,  12,  13,  14,  15,  15,  15,  16,  16,  16,  17,  18,  18,  18,  19,  20,  20,  20,  20,  20,  20,  21,  21,  22,  22,  22,  22,  22,  22,  23,  23]);
  check([-15,  05, -25,  08,  11,  22,  19, -10, -16, -01, -07,  00, -18, -09,  03,  01, -22,  08,  01, -08, -02, -02,  15, -12, -05, -24,  02, -12, -07, -06, -07,  05, -24, -05, -12,  14, -08, -18, -21,  02,  07, -01, -21,  16,  17,  09, -08,  21, -06,  21, -10,  18, -04, -20, -17,  18, -16,  10, -18,  02, -19,  02, -18, -18,  22, -03,  21, -09,  04, -23,  20, -12,  02,  24, -19, -06,  08, -01, -08, -20, -03,  09, -25,  18,  15, -17,  12,  24,  18,  19,  01,  24, -04,  10, -17, -05, -18, -19,  11,  13, -15,  07, -21,  14,  08, -15, -16, -06,  11,  03, -01, -17, -12, -01, -21, -22,  07,  16, -21,  02,  12,  07,  03],
        [-25, -25, -24, -24, -23, -22, -22, -21, -21, -21, -21, -21, -20, -20, -19, -19, -19, -18, -18, -18, -18, -18, -18, -17, -17, -17, -17, -16, -16, -16, -15, -15, -15, -12, -12, -12, -12, -12, -10, -10, -09, -09, -08, -08, -08, -08, -07, -07, -07, -06, -06, -06, -06, -05, -05, -05, -04, -04, -03, -03, -02, -02, -01, -01, -01, -01, -01,  00,  01,  01,  01,  02,  02,  02,  02,  02,  02,  03,  03,  03,  04,  05,  05,  07,  07,  07,  07,  08,  08,  08,  08,  09,  09,  10,  10,  11,  11,  11,  12,  12,  13,  14,  14,  15,  15,  16,  16,  17,  18,  18,  18,  18,  19,  19,  20,  21,  21,  21,  22,  22,  24,  24,  24]);
  check([-15, -08,  11,  05, -01,  19, -07, -11, -21, -25, -07,  19,  09,  04, -20, -22, -10,  08,  04,  01,  23, -22,  05, -22,  12, -04, -18, -11, -15,  14,  17, -05, -17,  05, -25,  09, -01, -07,  24,  05, -04, -08, -01, -18,  23,  04, -15, -09, -10,  14,  17, -10,  19, -03, -06,  06,  20,  03,  20, -20, -05,  14,  02,  03, -06,  02, -12,  20,  22, -11,  00, -07,  08, -23,  00,  07,  06,  12,  23,  23,  03,  15, -11, -03, -11,  10, -22, -14, -12, -25, -07,  08,  14,  20, -13, -15, -01,  02, -18, -04,  18,  08,  17,  01,  10, -08, -17, -07, -18, -19, -06,  10,  23,  08, -15, -11, -07, -10,  02,  06, -09,  20, -08],
        [-25, -25, -25, -23, -22, -22, -22, -22, -21, -20, -20, -19, -18, -18, -18, -18, -17, -17, -15, -15, -15, -15, -15, -14, -13, -12, -12, -11, -11, -11, -11, -11, -11, -10, -10, -10, -10, -09, -09, -08, -08, -08, -08, -07, -07, -07, -07, -07, -07, -07, -06, -06, -06, -05, -05, -04, -04, -04, -03, -03, -01, -01, -01, -01,  00,  00,  01,  01,  02,  02,  02,  02,  03,  03,  03,  04,  04,  04,  05,  05,  05,  05,  06,  06,  06,  07,  08,  08,  08,  08,  08,  09,  09,  10,  10,  10,  11,  12,  12,  14,  14,  14,  14,  15,  17,  17,  17,  18,  19,  19,  19,  20,  20,  20,  20,  20,  22,  23,  23,  23,  23,  23,  24]);

//length == 127
  check([-14,  14, -25,  16, -05, -01, -16,  11, -09,  00,  11,  24,  15, -10, -05,  03,  08,  15, -16,  21, -18,  13, -15, -16, -20, -23, -03,  05,  05, -12, -02,  16, -21, -02,  09, -01,  24,  20, -12, -07, -05, -24, -06, -12, -09,  14,  18, -24,  04, -23,  22,  11,  17, -18,  22, -25, -14, -04,  05,  16,  10, -20,  08,  14,  03, -06, -09,  04, -10,  06,  22,  12,  07, -09, -25, -25, -20,  20, -23,  11,  22,  24, -01,  15, -18, -04,  17, -07,  18, -03, -13,  05,  02,  20, -04, -18, -09,  14,  13,  06, -05,  11, -06, -23, -21, -04, -20, -13, -08, -18, -25, -09, -19,  00, -17, -10,  21, -25,  09,  16, -03,  21, -04, -24,  18,  18, -15],
        [-25, -25, -25, -25, -25, -25, -24, -24, -24, -23, -23, -23, -23, -21, -21, -20, -20, -20, -20, -19, -18, -18, -18, -18, -18, -17, -16, -16, -16, -15, -15, -14, -14, -13, -13, -12, -12, -12, -10, -10, -10, -09, -09, -09, -09, -09, -09, -08, -07, -07, -06, -06, -06, -05, -05, -05, -05, -04, -04, -04, -04, -04, -03, -03, -03, -02, -02, -01, -01, -01,  00,  00,  02,  03,  03,  04,  04,  05,  05,  05,  05,  06,  06,  07,  08,  08,  09,  09,  10,  11,  11,  11,  11,  11,  12,  13,  13,  14,  14,  14,  14,  15,  15,  15,  16,  16,  16,  16,  17,  17,  18,  18,  18,  18,  20,  20,  20,  21,  21,  21,  22,  22,  22,  22,  24,  24,  24]);
  check([-16,  07,  24,  18, -20,  12, -11, -16,  18,  12, -11, -18, -21, -04, -17, -02,  03,  08,  06,  18,  04,  07,  04, -02, -19,  02,  19, -15, -03,  14, -05,  09, -01, -04,  02,  04, -17,  18,  13, -21,  05, -22, -14,  10, -01, -06, -17, -21, -23,  16,  23,  09, -02,  04, -18,  07, -18, -22,  17,  06,  18, -13, -10, -08,  09,  19,  21,  19,  12,  09,  23, -07,  12, -13, -20, -11,  06, -10, -07,  11, -19, -09, -05,  07,  23,  04, -11, -20,  07, -19,  11,  00, -05, -21,  17, -19,  23,  15, -24,  11,  02, -24, -21, -11, -12, -14,  03,  22,  01,  24,  08,  07, -08, -20, -11, -10,  09,  03, -03, -09,  11, -16,  19, -18, -10,  13, -12],
        [-24, -24, -23, -22, -22, -21, -21, -21, -21, -21, -20, -20, -20, -20, -19, -19, -19, -19, -18, -18, -18, -18, -17, -17, -17, -16, -16, -16, -15, -14, -14, -13, -13, -12, -12, -11, -11, -11, -11, -11, -11, -10, -10, -10, -10, -09, -09, -08, -08, -07, -07, -06, -05, -05, -05, -04, -04, -03, -03, -02, -02, -02, -01, -01,  00,  01,  02,  02,  02,  03,  03,  03,  04,  04,  04,  04,  04,  05,  06,  06,  06,  07,  07,  07,  07,  07,  07,  08,  08,  09,  09,  09,  09,  09,  10,  11,  11,  11,  11,  12,  12,  12,  12,  13,  13,  14,  15,  16,  17,  17,  18,  18,  18,  18,  18,  19,  19,  19,  19,  21,  22,  23,  23,  23,  23,  24,  24]);
  check([-12,  04, -09,  24, -19, -05, -20, -03,  10, -07, -22,  07,  19, -23, -08,  04, -06, -03,  18,  10, -19, -01, -16,  00, -13, -07, -06, -04,  08, -16,  10,  24,  13, -22, -25,  21,  00, -17, -04, -14,  01, -01,  20, -04,  03, -12, -25,  24,  10,  20,  11,  19, -04,  21, -06,  09, -09,  15, -18, -23,  24,  17, -24,  15,  23, -24,  13,  00, -16, -16,  11,  13,  10,  09, -16, -12,  22, -14, -10,  09, -19, -24,  03,  05,  22,  24, -09, -11,  14, -02, -07,  14, -07, -06,  06, -07, -03, -06,  18,  09,  04,  07, -03, -09, -09,  06,  05, -12,  19,  20,  24, -23,  23,  03,  07,  21,  02, -25, -13, -06, -01,  07, -17,  19, -24,  16, -13],
        [-25, -25, -25, -24, -24, -24, -24, -23, -23, -23, -22, -22, -20, -19, -19, -19, -18, -17, -17, -16, -16, -16, -16, -16, -14, -14, -13, -13, -13, -12, -12, -12, -12, -11, -10, -09, -09, -09, -09, -09, -08, -07, -07, -07, -07, -07, -06, -06, -06, -06, -06, -06, -05, -04, -04, -04, -04, -03, -03, -03, -03, -02, -01, -01, -01,  00,  00,  00,  01,  02,  03,  03,  03,  04,  04,  04,  05,  05,  06,  06,  07,  07,  07,  07,  08,  09,  09,  09,  09,  10,  10,  10,  10,  10,  11,  11,  13,  13,  13,  14,  14,  15,  15,  16,  17,  18,  18,  19,  19,  19,  19,  20,  20,  20,  21,  21,  21,  22,  22,  23,  23,  24,  24,  24,  24,  24,  24]);
  check([ 00, -15, -20, -16,  16,  14,  06, -17, -20, -11,  15, -05, -17,  12, -05, -14,  10, -25,  20,  08,  02,  21,  22,  21, -03,  04, -19, -09,  05,  22,  05, -19, -15,  12, -08, -24,  02,  24, -16,  09, -10,  24, -20, -01,  11,  02,  12, -01,  02,  07, -18, -19,  05, -20, -20, -23,  09, -12, -05, -08, -12, -25,  00, -02,  15,  17,  01, -08,  18,  11, -24, -16,  12, -17,  10,  01,  10, -03, -25, -11, -19, -16, -04,  12, -09,  03,  16, -24,  16, -13, -05, -21, -11,  22,  04, -21,  16, -19, -04,  12,  19,  00,  21,  08,  08,  06, -14, -04, -20, -12,  10, -14, -02, -17,  23,  14,  11,  15,  17,  05, -21, -11,  09, -07, -14, -09, -02],
        [-25, -25, -25, -24, -24, -24, -23, -21, -21, -21, -20, -20, -20, -20, -20, -20, -19, -19, -19, -19, -19, -18, -17, -17, -17, -17, -16, -16, -16, -16, -15, -15, -14, -14, -14, -14, -13, -12, -12, -12, -11, -11, -11, -11, -10, -09, -09, -09, -08, -08, -08, -07, -05, -05, -05, -05, -04, -04, -04, -03, -03, -02, -02, -02, -01, -01,  00,  00,  00,  01,  01,  02,  02,  02,  02,  03,  04,  04,  05,  05,  05,  05,  06,  06,  07,  08,  08,  08,  09,  09,  09,  10,  10,  10,  10,  11,  11,  11,  12,  12,  12,  12,  12,  12,  14,  14,  15,  15,  15,  16,  16,  16,  16,  17,  17,  18,  19,  20,  21,  21,  21,  22,  22,  22,  23,  24,  24]);
  check([-20, -03,  21,  17, -07, -02,  15, -24, -16, -02, -10,  05,  03,  03, -08,  16, -24,  01,  17,  18, -11,  07,  12,  19,  13, -23,  04, -19, -10,  22,  06, -05, -04,  05, -11,  14, -20, -21,  17, -11,  02, -18,  21, -18,  11, -11,  24,  14,  17,  16,  07, -19,  00, -03, -24, -12, -01,  07, -04,  16,  04, -20, -11, -25,  12,  03,  16,  19,  10,  10, -16, -13,  18, -18, -05,  06, -02, -04, -05, -08, -13, -20, -01,  14,  02,  02, -23, -22, -16,  00,  21,  15,  07, -15,  17, -06,  16,  08, -11,  01, -07,  00,  15, -12,  09, -13,  19, -17,  08, -08,  00,  22, -01, -24,  11, -22,  03,  15, -19,  14, -09, -23,  04,  23, -10, -02, -05],
        [-25, -24, -24, -24, -24, -23, -23, -23, -22, -22, -21, -20, -20, -20, -20, -19, -19, -19, -18, -18, -18, -17, -16, -16, -16, -15, -13, -13, -13, -12, -12, -11, -11, -11, -11, -11, -11, -10, -10, -10, -09, -08, -08, -08, -07, -07, -06, -05, -05, -05, -05, -04, -04, -04, -03, -03, -02, -02, -02, -02, -01, -01, -01,  00,  00,  00,  00,  01,  01,  02,  02,  02,  03,  03,  03,  03,  04,  04,  04,  05,  05,  06,  06,  07,  07,  07,  07,  08,  08,  09,  10,  10,  11,  11,  12,  12,  13,  14,  14,  14,  14,  15,  15,  15,  15,  16,  16,  16,  16,  16,  17,  17,  17,  17,  17,  18,  18,  19,  19,  19,  21,  21,  21,  22,  22,  23,  24]);
  check([-19, -17,  09,  07,  04, -14, -01,  17,  20,  12,  14, -22, -05, -19,  06, -07,  07,  09, -19,  10, -12,  22,  16, -21, -12,  20,  09, -12, -15, -16,  08, -07, -07, -08, -25,  22,  03,  00,  16,  01, -11,  05,  06,  09,  13, -13, -21, -05,  24, -14, -20, -13, -17,  23, -09, -02,  21, -23,  11, -19, -13, -05,  00,  05,  14,  02,  04, -05, -21, -05,  21, -07, -22,  02, -20,  16,  17, -14, -11,  16, -03, -04, -20,  07, -05, -02, -18, -09,  01,  19, -01,  15, -09, -24, -03, -20,  06, -24,  02,  10, -02,  00, -19,  01, -20, -12, -05,  24, -01,  09,  15,  24, -20,  22,  06,  02, -05,  16,  20,  23,  10, -05,  15, -22, -02, -13, -17],
        [-25, -24, -24, -23, -22, -22, -22, -21, -21, -21, -20, -20, -20, -20, -20, -20, -19, -19, -19, -19, -19, -18, -17, -17, -17, -16, -15, -14, -14, -14, -13, -13, -13, -13, -12, -12, -12, -12, -11, -11, -09, -09, -09, -08, -07, -07, -07, -07, -05, -05, -05, -05, -05, -05, -05, -05, -05, -04, -03, -03, -02, -02, -02, -02, -01, -01, -01,  00,  00,  00,  01,  01,  01,  02,  02,  02,  02,  03,  04,  04,  05,  05,  06,  06,  06,  06,  07,  07,  07,  08,  09,  09,  09,  09,  09,  10,  10,  10,  11,  12,  13,  14,  14,  15,  15,  15,  16,  16,  16,  16,  16,  17,  17,  19,  20,  20,  20,  21,  21,  22,  22,  22,  23,  23,  24,  24,  24]);
  check([-21, -10,  13,  17,  14, -12,  23, -10, -07, -12,  12, -08,  12, -04, -16,  11,  04, -17, -05, -17,  04, -14, -21,  04,  23,  01, -05, -24,  24,  07, -14, -19, -25,  24,  23,  14, -10,  23, -19,  08, -14,  19, -22,  23, -08, -11,  12,  23, -02, -16, -18, -23, -03, -14,  08, -05,  12, -22, -02,  13,  13,  09,  19,  13,  11,  17,  04, -24,  15,  12,  11, -22,  06,  16, -21, -24, -19,  16,  24,  04, -23, -19,  08, -01, -08, -09,  19, -19, -03, -06,  20, -13, -22,  14, -23,  14,  09,  08,  17,  01, -05,  04,  05, -21, -05,  09, -20,  01,  02, -21, -18,  04, -12,  15, -20,  07, -16,  24,  14,  08, -05,  11,  20, -02,  02,  24, -10],
        [-25, -24, -24, -24, -23, -23, -23, -22, -22, -22, -22, -21, -21, -21, -21, -21, -20, -20, -19, -19, -19, -19, -19, -18, -18, -17, -17, -16, -16, -16, -14, -14, -14, -14, -13, -12, -12, -12, -11, -10, -10, -10, -10, -09, -08, -08, -08, -07, -06, -05, -05, -05, -05, -05, -05, -04, -03, -03, -02, -02, -02, -01,  01,  01,  01,  02,  02,  04,  04,  04,  04,  04,  04,  04,  05,  06,  07,  07,  08,  08,  08,  08,  08,  09,  09,  09,  11,  11,  11,  11,  12,  12,  12,  12,  12,  13,  13,  13,  13,  14,  14,  14,  14,  14,  15,  15,  16,  16,  17,  17,  17,  19,  19,  19,  20,  20,  23,  23,  23,  23,  23,  23,  24,  24,  24,  24,  24]);
  check([-14,  07, -18,  13, -23,  11, -07, -19, -16, -21, -12,  12,  06, -07,  20, -13,  08,  12, -08, -10,  21, -07, -21,  06,  13,  17,  03, -12, -05,  02,  03,  06, -14,  11, -04, -11, -01,  16, -03,  08,  20,  11, -04,  03, -19, -07,  17,  14, -20, -14, -19, -21,  04, -12,  10, -06, -20, -09,  08,  00,  20, -14, -16, -18,  24,  07, -02, -01, -01,  20, -16, -04,  08,  07, -23,  16, -25,  19,  07, -17, -19,  14, -13,  12, -21,  24, -18, -16, -10,  15,  12, -14, -22,  23, -05, -22,  05,  18,  04, -19,  15,  15,  03,  01, -02,  05, -08,  00,  01, -23,  10,  07,  18,  22, -03,  22,  24,  04, -19, -09, -04,  20,  04, -01,  18,  24,  04],
        [-25, -23, -23, -23, -22, -22, -21, -21, -21, -21, -20, -20, -19, -19, -19, -19, -19, -19, -18, -18, -18, -17, -16, -16, -16, -16, -14, -14, -14, -14, -14, -13, -13, -12, -12, -12, -11, -10, -10, -09, -09, -08, -08, -07, -07, -07, -07, -06, -05, -05, -04, -04, -04, -04, -03, -03, -02, -02, -01, -01, -01, -01,  00,  00,  01,  01,  02,  03,  03,  03,  03,  04,  04,  04,  04,  04,  05,  05,  06,  06,  06,  07,  07,  07,  07,  07,  08,  08,  08,  08,  10,  10,  11,  11,  11,  12,  12,  12,  12,  13,  13,  14,  14,  15,  15,  15,  16,  16,  17,  17,  18,  18,  18,  19,  20,  20,  20,  20,  20,  21,  22,  22,  23,  24,  24,  24,  24]);
  check([ 01,  19, -15,  07,  10, -24, -13, -14,  01,  17,  05, -22, -04,  09,  14, -22,  02,  13,  00, -01,  14, -19,  08, -19,  02,  05,  10, -21,  01,  12,  10, -23,  06,  21,  11, -07,  24,  24,  06,  02,  18, -13,  07, -11,  23, -02, -05,  01, -13,  22, -23, -24, -21,  12, -16,  08, -05,  22,  14,  21,  09,  00, -25, -10, -02,  13,  09, -01,  12, -10, -22,  08,  04,  10,  22,  05, -16, -06, -19, -02, -08, -15,  01, -02, -25,  11, -19, -03,  08,  21, -07, -08, -02, -05,  09,  23, -15, -05, -03, -25,  13,  00,  08, -06, -13,  07,  24, -02, -23, -18,  21, -06, -05, -02,  17, -05, -14, -25,  17,  19,  23, -13, -12, -04,  09,  24, -04],
        [-25, -25, -25, -25, -24, -24, -23, -23, -23, -22, -22, -22, -21, -21, -19, -19, -19, -19, -18, -16, -16, -15, -15, -15, -14, -14, -13, -13, -13, -13, -13, -12, -11, -10, -10, -08, -08, -07, -07, -06, -06, -06, -05, -05, -05, -05, -05, -05, -04, -04, -04, -03, -03, -02, -02, -02, -02, -02, -02, -02, -01, -01,  00,  00,  00,  01,  01,  01,  01,  01,  02,  02,  02,  04,  05,  05,  05,  06,  06,  07,  07,  07,  08,  08,  08,  08,  08,  09,  09,  09,  09,  09,  10,  10,  10,  10,  11,  11,  12,  12,  12,  13,  13,  13,  14,  14,  14,  17,  17,  17,  18,  19,  19,  21,  21,  21,  21,  22,  22,  22,  23,  23,  23,  24,  24,  24,  24]);
  check([ 19, -03,  18,  19, -15, -05,  02,  06,  10, -15,  06, -17, -11,  13, -18,  08, -15,  05,  00,  05,  18,  02, -01,  14,  01,  13, -23,  24, -03, -22, -04, -06,  00, -09, -10,  10,  12,  18, -08, -01, -20, -25,  07, -06,  15,  14, -21, -24, -04,  06,  08, -11, -17, -15, -22,  11,  23, -18,  11, -02, -14, -16,  19,  11,  00, -16, -01, -11, -21,  18,  13, -14,  18, -04,  07,  10, -13,  11,  11,  11, -08, -03, -25,  03,  07, -19, -11,  05, -10, -23, -20,  03, -14,  24, -08,  12, -14,  16, -24, -10,  11,  17,  02,  06, -10, -16, -09,  02,  21, -20, -12, -10,  02, -09,  20, -16, -01, -15,  16,  14, -11,  24, -05,  01, -25,  12, -10],
        [-25, -25, -25, -24, -24, -23, -23, -22, -22, -21, -21, -20, -20, -20, -19, -18, -18, -17, -17, -16, -16, -16, -16, -15, -15, -15, -15, -15, -14, -14, -14, -14, -13, -12, -11, -11, -11, -11, -11, -10, -10, -10, -10, -10, -10, -09, -09, -09, -08, -08, -08, -06, -06, -05, -05, -04, -04, -04, -03, -03, -03, -02, -01, -01, -01, -01,  00,  00,  00,  01,  01,  02,  02,  02,  02,  02,  03,  03,  05,  05,  05,  06,  06,  06,  06,  07,  07,  07,  08,  08,  10,  10,  10,  11,  11,  11,  11,  11,  11,  11,  12,  12,  12,  13,  13,  13,  14,  14,  14,  15,  16,  16,  17,  18,  18,  18,  18,  18,  19,  19,  19,  20,  21,  23,  24,  24,  24]);


//length == 128
  check([-12, -11, -17,  00, -21, -15, -11, -09,  16,  09, -17, -05, -25,  05,  19, -18, -10,  13,  20, -23,  09, -20,  01, -04,  15,  08,  14, -03,  16, -19,  12, -18, -05,  22,  09,  01, -18,  24,  17,  23,  08, -22, -05,  10,  08, -09,  18, -25, -20,  13, -23,  16,  20,  05, -10,  10, -11, -21, -16, -18, -13,  21, -11,  07,  19, -24, -15, -24, -25, -22, -23,  10, -19, -01, -05,  16,  16, -12,  18,  23, -22,  20,  16,  23,  03, -17,  11,  19, -12,  20, -24,  00,  19, -07, -15,  13, -06, -05,  16, -06,  00,  20, -21,  08, -03, -01, -01, -10,  15, -06, -10,  18, -08, -19,  17,  20, -08,  05,  14,  05,  00,  17, -18, -06, -15, -08, -16,  06],
        [-25, -25, -25, -24, -24, -24, -23, -23, -23, -22, -22, -22, -21, -21, -21, -20, -20, -19, -19, -19, -18, -18, -18, -18, -18, -17, -17, -17, -16, -16, -15, -15, -15, -15, -13, -12, -12, -12, -11, -11, -11, -11, -10, -10, -10, -10, -09, -09, -08, -08, -08, -07, -06, -06, -06, -06, -05, -05, -05, -05, -05, -04, -03, -03, -01, -01, -01,  00,  00,  00,  00,  01,  01,  03,  05,  05,  05,  05,  06,  07,  08,  08,  08,  08,  09,  09,  09,  10,  10,  10,  11,  12,  13,  13,  13,  14,  14,  15,  15,  16,  16,  16,  16,  16,  16,  16,  17,  17,  17,  18,  18,  18,  19,  19,  19,  19,  20,  20,  20,  20,  20,  20,  21,  22,  23,  23,  23,  24]);
  check([ 15, -22, -23, -08, -25, -19, -24, -03,  08,  02,  12, -02,  22, -23, -07, -11, -14, -15, -14,  03,  15, -25, -15, -07,  17, -08,  14, -20,  12,  24,  13,  04, -21,  16,  21, -21,  24,  24,  02,  09,  02, -09, -18,  01, -06,  03,  15,  07,  13, -24, -13,  06, -22, -03, -24,  22,  16,  15, -21,  03,  16,  18, -16,  20,  11, -19, -23,  10, -18,  04, -05,  09,  22,  04, -15, -09, -18, -23, -25,  23, -20, -13,  04, -15,  11,  07, -17,  03, -03, -11, -17, -11, -16, -07, -14,  20,  01, -12,  06,  08,  19, -24, -07,  17,  05,  05, -15,  13,  08, -14,  11,  15,  00, -08, -24, -13,  24, -14,  17, -02,  00, -25,  14,  10, -07, -24,  05,  19],
        [-25, -25, -25, -25, -24, -24, -24, -24, -24, -24, -23, -23, -23, -23, -22, -22, -21, -21, -21, -20, -20, -19, -19, -18, -18, -18, -17, -17, -16, -16, -15, -15, -15, -15, -15, -14, -14, -14, -14, -14, -13, -13, -13, -12, -11, -11, -11, -09, -09, -08, -08, -08, -07, -07, -07, -07, -07, -06, -05, -03, -03, -03, -02, -02,  00,  00,  01,  01,  02,  02,  02,  03,  03,  03,  03,  04,  04,  04,  04,  05,  05,  05,  06,  06,  07,  07,  08,  08,  08,  09,  09,  10,  10,  11,  11,  11,  12,  12,  13,  13,  13,  14,  14,  15,  15,  15,  15,  15,  16,  16,  16,  17,  17,  17,  18,  19,  19,  20,  20,  21,  22,  22,  22,  23,  24,  24,  24,  24]);
  check([-09, -14,  03, -14, -11,  23, -20,  20,  05, -10,  10, -10,  03, -02, -19, -21,  17, -18, -07,  18, -07, -15, -07,  20, -15, -17,  05,  06, -14, -12,  02,  02,  01, -18,  15,  16, -20,  20,  13,  13, -14,  00, -22,  14,  23, -16,  20,  17, -07, -10,  10,  13,  00, -20,  11,  13, -12, -07, -04, -01,  06,  23, -21, -17, -19,  19, -24, -12,  15,  14, -24, -24, -11, -20,  17, -11, -09,  15, -19,  10, -20,  18,  00,  06,  23, -14, -04, -11,  05,  17,  13, -12,  17,  19, -04,  23,  14, -01, -13,  04, -12, -12,  07,  04, -05,  01,  20,  12, -09, -22, -01, -03,  23, -24, -20,  24, -12,  03, -12,  20, -05, -22, -17, -11,  23,  07, -12,  12],
        [-24, -24, -24, -24, -22, -22, -22, -21, -21, -20, -20, -20, -20, -20, -20, -19, -19, -19, -18, -18, -17, -17, -17, -16, -15, -15, -14, -14, -14, -14, -14, -13, -12, -12, -12, -12, -12, -12, -12, -12, -12, -11, -11, -11, -11, -11, -10, -10, -10, -09, -09, -09, -07, -07, -07, -07, -07, -05, -05, -04, -04, -04, -03, -02, -01, -01, -01,  00,  00,  00,  01,  01,  02,  02,  03,  03,  03,  04,  04,  05,  05,  05,  06,  06,  06,  07,  07,  10,  10,  10,  11,  12,  12,  13,  13,  13,  13,  13,  14,  14,  14,  15,  15,  15,  16,  17,  17,  17,  17,  17,  18,  18,  19,  19,  20,  20,  20,  20,  20,  20,  23,  23,  23,  23,  23,  23,  23,  24]);
  check([-19,  02, -07, -03,  15, -25, -22, -12,  01,  24, -23,  20, -21,  01, -08, -22,  04, -03, -23,  19, -25, -10,  14, -03, -07, -25,  11, -07,  07, -24, -20,  15,  03, -02, -11, -04,  00, -05,  09, -21, -04,  11,  24,  00, -11, -07,  03,  18,  15,  05,  13,  17,  22,  02, -09, -09,  04, -21,  11, -14, -19,  17, -21,  09, -08, -05, -20,  18,  15,  14,  24, -14,  02,  23,  12,  16, -07, -08,  10, -17,  23,  00, -23,  20, -21, -05, -12,  09, -01, -25,  22,  05,  19, -22, -08, -14, -01, -03, -19, -09, -11, -18,  03,  16, -20, -08,  10, -02,  09, -03,  08,  09,  22,  12,  07, -22, -18,  22,  12,  09,  24,  12, -11,  18,  15,  08, -18, -11],
        [-25, -25, -25, -25, -24, -23, -23, -23, -22, -22, -22, -22, -21, -21, -21, -21, -21, -20, -20, -20, -19, -19, -19, -18, -18, -18, -17, -14, -14, -14, -12, -12, -11, -11, -11, -11, -11, -10, -09, -09, -09, -08, -08, -08, -08, -08, -07, -07, -07, -07, -07, -05, -05, -05, -04, -04, -03, -03, -03, -03, -03, -02, -02, -01, -01,  00,  00,  00,  01,  01,  02,  02,  02,  03,  03,  03,  04,  04,  05,  05,  07,  07,  08,  08,  09,  09,  09,  09,  09,  09,  10,  10,  11,  11,  11,  12,  12,  12,  12,  13,  14,  14,  15,  15,  15,  15,  15,  16,  16,  17,  17,  18,  18,  18,  19,  19,  20,  20,  22,  22,  22,  22,  23,  23,  24,  24,  24,  24]);
  check([-19, -10,  08, -03, -02, -14, -10,  05,  05, -23, -21, -11, -01,  14, -24, -02,  04,  08,  02,  13,  05, -09, -03,  05, -22,  14,  00,  19, -03,  07, -15,  05,  23,  18, -23, -04, -19, -05, -24,  11, -03, -18, -22,  22,  24, -19, -03,  03,  14,  24, -07, -04, -09,  16, -24, -04,  05,  02, -10, -23, -14,  03,  08, -16, -04,  12,  07,  05, -18,  11, -07,  05,  20, -04,  04,  19,  02, -24, -01, -07, -22,  18,  15, -04,  09,  18,  17, -09, -05, -15, -07,  09,  13, -22,  18, -14,  16,  03,  16, -25, -11, -15,  05, -14,  08, -14, -19, -12, -10,  07,  06, -05,  02, -04,  18, -14, -08,  10,  04,  12,  20,  23, -02,  10, -24, -08,  22,  19],
        [-25, -24, -24, -24, -24, -24, -23, -23, -23, -22, -22, -22, -22, -21, -19, -19, -19, -19, -18, -18, -16, -15, -15, -15, -14, -14, -14, -14, -14, -14, -12, -11, -11, -10, -10, -10, -10, -09, -09, -09, -08, -08, -07, -07, -07, -07, -05, -05, -05, -04, -04, -04, -04, -04, -04, -04, -03, -03, -03, -03, -03, -02, -02, -02, -01, -01,  00,  02,  02,  02,  02,  03,  03,  03,  04,  04,  04,  05,  05,  05,  05,  05,  05,  05,  05,  05,  06,  07,  07,  07,  08,  08,  08,  08,  09,  09,  10,  10,  11,  11,  12,  12,  13,  13,  14,  14,  14,  15,  16,  16,  16,  17,  18,  18,  18,  18,  18,  19,  19,  19,  20,  20,  22,  22,  23,  23,  24,  24]);
  check([ 22,  15,  20, -12, -25,  02, -01,  09,  16,  07,  22, -19, -10,  05,  03,  19, -21, -04, -19, -04, -17,  12, -15, -19,  10,  09,  16,  14, -22,  15,  10, -25,  06,  05, -10,  06, -17,  14, -08, -24, -01, -11, -16,  16,  22,  12,  10, -24, -17,  18, -03, -09,  06,  07, -01, -07, -07, -09, -18, -04, -19,  18, -02,  12,  00,  13, -06,  10,  05,  13,  11, -21, -20,  20,  22, -23,  07,  07, -22,  17,  03,  02, -14, -16, -14,  10,  04,  05, -22,  12, -22, -15,  07,  02,  24, -16, -10, -05,  20,  22, -16,  06, -22, -11,  04, -25, -09, -12,  10, -04, -19, -12,  23, -06, -01, -14, -19, -22,  18, -15,  17,  22, -03,  01, -01, -04,  11,  16],
        [-25, -25, -25, -24, -24, -23, -22, -22, -22, -22, -22, -22, -21, -21, -20, -19, -19, -19, -19, -19, -19, -18, -17, -17, -17, -16, -16, -16, -16, -15, -15, -15, -14, -14, -14, -12, -12, -12, -11, -11, -10, -10, -10, -09, -09, -09, -08, -07, -07, -06, -06, -05, -04, -04, -04, -04, -04, -03, -03, -02, -01, -01, -01, -01, -01,  00,  01,  02,  02,  02,  03,  03,  04,  04,  05,  05,  05,  05,  06,  06,  06,  06,  07,  07,  07,  07,  07,  09,  09,  10,  10,  10,  10,  10,  10,  11,  11,  12,  12,  12,  12,  13,  13,  14,  14,  15,  15,  16,  16,  16,  16,  17,  17,  18,  18,  18,  19,  20,  20,  20,  22,  22,  22,  22,  22,  22,  23,  24]);
  check([ 17,  06,  14, -24, -11,  19, -10,  18,  20,  08, -18,  07, -21, -10,  20, -21,  11, -04, -10,  17, -01, -16, -21, -06, -19,  01,  20,  07, -25,  06, -02,  17, -11, -11,  18,  06, -16, -15, -01, -19,  18,  08,  13,  24, -25,  10, -22,  11, -19, -07,  06,  07,  02,  10, -24,  10, -11,  22, -08, -11,  05,  18, -19,  22, -16, -24,  03, -05, -12, -21,  01, -19,  13, -09, -18,  13, -24, -15,  02, -16,  03, -17,  17, -17,  18,  18,  20, -16,  17, -10, -02,  00, -15,  06, -03, -06,  07, -25,  17,  20, -21, -07, -22,  19,  12, -15,  10,  15, -03, -13, -25, -24, -03,  17, -14, -08,  12, -17,  02,  05, -02, -25, -20,  08,  07,  04, -20, -09],
        [-25, -25, -25, -25, -25, -24, -24, -24, -24, -24, -22, -22, -21, -21, -21, -21, -21, -20, -20, -19, -19, -19, -19, -19, -18, -18, -17, -17, -17, -16, -16, -16, -16, -16, -15, -15, -15, -15, -14, -13, -12, -11, -11, -11, -11, -11, -10, -10, -10, -10, -09, -09, -08, -08, -07, -07, -06, -06, -05, -04, -03, -03, -03, -02, -02, -02, -01, -01,  00,  01,  01,  02,  02,  02,  03,  03,  04,  05,  05,  06,  06,  06,  06,  06,  07,  07,  07,  07,  07,  08,  08,  08,  10,  10,  10,  10,  11,  11,  12,  12,  13,  13,  13,  14,  15,  17,  17,  17,  17,  17,  17,  17,  18,  18,  18,  18,  18,  18,  19,  19,  20,  20,  20,  20,  20,  22,  22,  24]);
  check([ 04,  22, -11,  10, -08, -06,  07, -21,  05, -08,  20, -21,  06,  20, -20, -22,  14, -07, -05,  01,  02,  22, -17, -23, -25, -10,  11,  09,  19,  18, -25,  00,  15, -11, -12, -18,  09,  20, -11, -09, -11, -16, -05,  20, -19,  03,  23,  22,  23, -05,  00,  00, -05,  09,  05, -05, -24, -07, -19,  21, -14, -19, -02, -24, -02,  11, -15, -16,  08, -01,  02,  22,  10,  22,  17,  18,  02,  17,  17,  01, -10, -07, -22,  10, -21,  08, -18, -20, -22, -12, -22, -11, -04,  02, -10,  21, -10,  03, -20, -24, -21,  07,  23,  15,  07,  18,  10, -14,  10,  03,  14, -25,  23,  20,  12, -23,  03,  19, -18,  07, -16, -12,  23,  06,  17, -09,  02, -16],
        [-25, -25, -25, -24, -24, -24, -23, -23, -22, -22, -22, -22, -21, -21, -21, -21, -20, -20, -20, -19, -19, -19, -18, -18, -18, -17, -16, -16, -16, -16, -15, -14, -14, -12, -12, -12, -11, -11, -11, -11, -11, -10, -10, -10, -10, -09, -09, -08, -08, -07, -07, -07, -06, -05, -05, -05, -05, -05, -04, -02, -02, -01,  00,  00,  00,  01,  01,  02,  02,  02,  02,  02,  03,  03,  03,  03,  04,  05,  05,  06,  06,  07,  07,  07,  07,  08,  08,  09,  09,  09,  10,  10,  10,  10,  10,  11,  11,  12,  14,  14,  15,  15,  17,  17,  17,  17,  18,  18,  18,  19,  19,  20,  20,  20,  20,  20,  21,  21,  22,  22,  22,  22,  22,  23,  23,  23,  23,  23]);
  check([ 19,  10, -15,  23, -06, -16,  15, -24, -23,  03, -12,  14, -19, -21,  17, -21,  24,  06, -17,  05,  03, -08, -11,  12,  05, -13, -05, -03,  03, -25,  09, -03,  10,  19, -02, -21, -22, -12, -17, -18,  18, -02,  24,  24,  02,  18, -20,  02, -01, -10, -18, -23,  08, -04,  17, -12,  10, -13,  13, -11, -13, -03,  13,  24, -07, -12, -21, -01,  02, -11,  08, -05,  12,  07, -03, -11,  02,  04,  18, -23,  20, -23, -19,  03,  00,  23,  18, -14, -14,  08,  02,  00, -18, -10,  00,  03,  04,  06, -23, -17,  20,  10,  05,  07, -05, -21,  23,  22,  09,  19, -24,  06,  21, -17, -14, -03, -17, -19,  10, -06,  17, -13,  22,  24,  02, -03,  02, -17],
        [-25, -24, -24, -23, -23, -23, -23, -23, -22, -21, -21, -21, -21, -21, -20, -19, -19, -19, -18, -18, -18, -17, -17, -17, -17, -17, -17, -16, -15, -14, -14, -14, -13, -13, -13, -13, -12, -12, -12, -12, -11, -11, -11, -11, -10, -10, -08, -07, -06, -06, -05, -05, -05, -04, -03, -03, -03, -03, -03, -03, -02, -02, -01, -01,  00,  00,  00,  02,  02,  02,  02,  02,  02,  02,  03,  03,  03,  03,  03,  04,  04,  05,  05,  05,  06,  06,  06,  07,  07,  08,  08,  08,  09,  09,  10,  10,  10,  10,  10,  12,  12,  13,  13,  14,  15,  17,  17,  17,  18,  18,  18,  18,  19,  19,  19,  20,  20,  21,  22,  22,  23,  23,  23,  24,  24,  24,  24,  24]);
  check([-22,  06, -09, -25, -08,  24,  09,  14, -22,  07,  11,  14,  03,  15, -03,  00,  23,  08,  24, -19,  17,  09,  03,  09,  21,  02,  10,  00,  24, -10,  09, -21,  21, -23, -19, -10, -22,  15, -21, -18, -01,  16,  23, -22,  06, -04,  03,  06, -19,  04,  14,  23, -12, -06,  09, -14,  21, -05,  11,  22,  10, -03, -24,  08,  02, -18,  24,  05,  24, -22,  14, -01,  21,  13,  02,  04, -14, -18, -13, -06,  11, -23, -05, -24, -02, -21, -13, -05, -01, -25, -06, -14, -02, -02,  20, -25,  07,  19,  05,  07,  24, -03, -19,  21, -13,  08,  02, -02,  17,  15,  19,  03,  19, -11,  04, -08, -06,  18,  14,  18,  18, -16, -20, -09,  07, -23, -07, -11],
        [-25, -25, -25, -24, -24, -23, -23, -23, -22, -22, -22, -22, -22, -21, -21, -21, -20, -19, -19, -19, -19, -18, -18, -18, -16, -14, -14, -14, -13, -13, -13, -12, -11, -11, -10, -10, -09, -09, -08, -08, -07, -06, -06, -06, -06, -05, -05, -05, -04, -03, -03, -03, -02, -02, -02, -02, -01, -01, -01,  00,  00,  02,  02,  02,  02,  03,  03,  03,  03,  04,  04,  04,  05,  05,  06,  06,  06,  07,  07,  07,  07,  08,  08,  08,  09,  09,  09,  09,  09,  10,  10,  11,  11,  11,  13,  14,  14,  14,  14,  14,  15,  15,  15,  16,  17,  17,  18,  18,  18,  19,  19,  19,  20,  21,  21,  21,  21,  21,  22,  23,  23,  23,  24,  24,  24,  24,  24,  24]);


//length == 129
  check([ 22, -05,  04,  13, -11,  02,  05,  10,  24, -19,  18,  07,  05, -19, -06, -04, -19,  05, -16, -08,  18, -23,  03, -16,  15, -09, -02, -04,  09, -22,  20,  08,  01,  01, -04,  17, -22, -23,  04, -23, -15, -02,  10,  15,  04, -19, -14,  11,  14, -05, -20,  07, -02,  08,  16, -10, -25, -11,  13,  11, -05,  08, -04,  21, -16,  20,  13, -13,  24, -06, -08, -16,  17, -21,  24, -03, -13, -13, -15, -24,  09, -08,  11, -18, -24,  02,  00, -22,  19, -12,  16, -11,  24, -11, -13, -17, -16, -23, -02, -17, -04,  17, -08, -11,  21, -07,  13, -15,  05,  23, -13, -10, -10, -25, -01, -07,  02, -24, -02, -02, -08, -09,  12, -09,  06, -24,  01,  15, -20],
        [-25, -25, -24, -24, -24, -24, -23, -23, -23, -23, -22, -22, -22, -21, -20, -20, -19, -19, -19, -19, -18, -17, -17, -16, -16, -16, -16, -16, -15, -15, -15, -14, -13, -13, -13, -13, -13, -12, -11, -11, -11, -11, -11, -10, -10, -10, -09, -09, -09, -08, -08, -08, -08, -08, -07, -07, -06, -06, -05, -05, -05, -04, -04, -04, -04, -04, -03, -02, -02, -02, -02, -02, -02, -01,  00,  01,  01,  01,  02,  02,  02,  03,  04,  04,  04,  05,  05,  05,  05,  06,  07,  07,  08,  08,  08,  09,  09,  10,  10,  11,  11,  11,  12,  13,  13,  13,  13,  14,  15,  15,  15,  16,  16,  17,  17,  17,  18,  18,  19,  20,  20,  21,  21,  22,  23,  24,  24,  24,  24]);
  check([-24,  24,  02,  18, -07,  18,  16,  12,  06, -23, -06,  06, -09,  09,  23, -09, -16, -08,  20, -13,  17, -06,  04, -16, -17,  20,  15, -13, -01,  07, -08,  00,  08,  21, -06, -21,  14, -15,  16, -03, -11, -15, -20,  05,  22, -21,  23, -17, -02,  21, -05, -10, -10, -25,  01,  00,  22, -07,  12, -04, -24, -20,  22,  11, -22, -07,  15,  18,  03,  08, -08,  20,  21, -02,  02,  18,  04,  01, -22, -23, -01, -02, -06,  14,  00,  21, -09,  00, -11, -21,  23, -08, -14,  22, -21, -11, -10,  21, -16,  21,  05,  04,  16,  01, -21, -07, -04,  08,  21, -01,  12, -05,  24, -19, -14,  00, -23,  03, -23, -06,  07, -25,  13,  20,  23,  19, -16, -12,  16],
        [-25, -25, -24, -24, -23, -23, -23, -23, -22, -22, -21, -21, -21, -21, -21, -20, -20, -19, -17, -17, -16, -16, -16, -16, -15, -15, -14, -14, -13, -13, -12, -11, -11, -11, -10, -10, -10, -09, -09, -09, -08, -08, -08, -08, -07, -07, -07, -07, -06, -06, -06, -06, -06, -05, -05, -04, -04, -03, -02, -02, -02, -01, -01, -01,  00,  00,  00,  00,  00,  01,  01,  01,  02,  02,  03,  03,  04,  04,  04,  05,  05,  06,  06,  07,  07,  08,  08,  08,  09,  11,  12,  12,  12,  13,  14,  14,  15,  15,  16,  16,  16,  16,  17,  18,  18,  18,  18,  19,  20,  20,  20,  20,  21,  21,  21,  21,  21,  21,  21,  22,  22,  22,  22,  23,  23,  23,  23,  24,  24]);
  check([-04, -16, -04, -25, -23,  24, -21, -04, -05,  14, -06,  21,  01,  15,  20,  07, -24, -03,  12,  04, -01, -17, -12,  02,  21, -15,  02,  16, -03,  15,  09,  18, -23, -20,  20, -21, -19, -24,  02,  03,  15,  22,  01, -09,  14, -04, -25,  17,  21,  14, -03, -03,  22,  12,  24,  21,  23,  01,  12, -05, -06, -04, -10, -02,  03, -15,  04,  11, -14, -18, -11, -22, -19,  15, -04,  22, -12, -02,  14, -16, -12, -12,  09, -15, -24,  08, -19, -24, -13,  20, -02,  06, -07,  13, -19,  21, -25,  10,  09, -12,  19, -25, -07, -23,  17,  14,  24, -19, -13,  14, -08,  00, -21, -24,  13, -18,  12,  21, -17,  24,  17,  08,  07, -15, -02,  15, -16, -02,  00],
        [-25, -25, -25, -25, -24, -24, -24, -24, -24, -23, -23, -23, -22, -21, -21, -21, -20, -19, -19, -19, -19, -19, -18, -18, -17, -17, -16, -16, -16, -15, -15, -15, -15, -14, -13, -13, -12, -12, -12, -12, -12, -11, -10, -09, -08, -07, -07, -06, -06, -05, -05, -04, -04, -04, -04, -04, -04, -03, -03, -03, -03, -02, -02, -02, -02, -02, -01,  00,  00,  01,  01,  01,  02,  02,  02,  03,  03,  04,  04,  06,  07,  07,  08,  08,  09,  09,  09,  10,  11,  12,  12,  12,  12,  13,  13,  14,  14,  14,  14,  14,  14,  15,  15,  15,  15,  15,  16,  17,  17,  17,  18,  19,  20,  20,  20,  21,  21,  21,  21,  21,  21,  22,  22,  22,  23,  24,  24,  24,  24]);
  check([ 18,  13, -05,  21, -19, -01,  13,  20, -01,  21, -15, -10, -11,  12, -06, -08,  00,  02, -19, -03,  12, -20, -09, -04,  14,  03,  19,  06,  12, -07, -18,  08, -17,  04,  06, -10, -22, -06, -15,  04, -09, -03,  19,  07, -15, -09,  24,  10,  18, -19, -16,  05, -12,  02, -22, -20, -19,  23,  11,  20, -07,  20,  05,  01, -01, -14,  16,  05,  06, -21, -16,  24,  01, -19,  06,  13, -03,  05,  01, -10,  13,  12,  22, -23, -10, -24, -16, -04, -24,  20, -07, -04,  16, -24,  22, -08, -13,  16,  22,  20,  22, -18,  19,  00, -12,  02, -11,  12, -17,  15, -21,  23,  02, -24, -23,  19, -21, -14, -08, -18, -16,  11,  03, -23,  12,  03, -06, -24, -04],
        [-24, -24, -24, -24, -24, -23, -23, -23, -22, -22, -21, -21, -21, -20, -20, -19, -19, -19, -19, -19, -18, -18, -18, -17, -17, -16, -16, -16, -16, -15, -15, -15, -14, -14, -13, -12, -12, -11, -11, -10, -10, -10, -10, -09, -09, -09, -08, -08, -08, -07, -07, -07, -06, -06, -06, -05, -04, -04, -04, -04, -03, -03, -03, -01, -01, -01,  00,  00,  01,  01,  01,  02,  02,  02,  02,  03,  03,  03,  04,  04,  05,  05,  05,  05,  06,  06,  06,  06,  07,  08,  10,  11,  11,  12,  12,  12,  12,  12,  12,  13,  13,  13,  13,  14,  15,  16,  16,  16,  18,  18,  19,  19,  19,  19,  20,  20,  20,  20,  20,  21,  21,  22,  22,  22,  22,  23,  23,  24,  24]);
  check([-06,  22, -07,  03,  18,  20,  18, -04, -16,  05,  06, -24,  09,  04,  06,  12,  09,  00,  19,  20,  20, -24, -19,  06,  05, -17, -05, -15,  05, -02,  06, -24, -05, -24,  04, -09,  21, -03,  14, -19, -21, -05, -16,  13, -24,  17,  02,  10, -07, -04,  08, -12, -02,  14,  21, -20, -25, -07, -10,  07,  16,  23,  08, -11,  24, -11,  07,  20,  13, -04, -22,  17,  18, -12,  07,  19, -20, -16,  07,  00,  06, -08,  15, -21, -19,  13, -14, -17, -19,  01,  17, -25, -01,  02, -11, -02,  16,  23, -05,  04, -06, -01, -04, -13,  14, -20, -17,  21, -10,  15, -01,  21, -18, -11, -23, -09, -20, -10,  01, -12,  16, -06, -12, -08,  21,  04,  17,  15,  02],
        [-25, -25, -24, -24, -24, -24, -24, -23, -22, -21, -21, -20, -20, -20, -20, -19, -19, -19, -19, -18, -17, -17, -17, -16, -16, -16, -15, -14, -13, -12, -12, -12, -12, -11, -11, -11, -11, -10, -10, -10, -09, -09, -08, -08, -07, -07, -07, -06, -06, -06, -05, -05, -05, -05, -04, -04, -04, -04, -03, -02, -02, -02, -01, -01, -01,  00,  00,  01,  01,  02,  02,  02,  03,  04,  04,  04,  04,  05,  05,  05,  06,  06,  06,  06,  06,  07,  07,  07,  07,  08,  08,  09,  09,  10,  12,  13,  13,  13,  14,  14,  14,  15,  15,  15,  16,  16,  16,  17,  17,  17,  17,  18,  18,  18,  19,  19,  20,  20,  20,  20,  21,  21,  21,  21,  21,  22,  23,  23,  24]);
  check([-13, -06,  21,  13,  18, -15,  02, -25, -06,  01, -10, -14, -25, -12, -06, -11, -08,  12, -04,  07, -10,  10,  00,  09, -25,  17,  08,  05,  11,  00, -16,  24,  21, -17,  14,  16, -07, -08,  17,  14,  18, -18, -24,  18, -02, -03, -16,  15, -14,  08, -02,  01, -07,  23, -12, -07,  18,  21, -25,  04, -02, -15,  05, -06, -07, -05, -14,  11,  14, -22,  01,  09, -13,  04,  04,  10, -24,  15,  03, -13, -02, -24,  13,  18, -24, -24, -13,  21,  24, -13,  03, -01, -01, -17,  19,  17,  05, -18,  06, -06, -13, -16, -20,  01,  13,  09, -13,  16,  00,  15, -20, -25,  18,  18,  19,  21,  22, -17,  18,  23, -05, -02, -02,  22,  06, -06,  14, -11,  01],
        [-25, -25, -25, -25, -25, -24, -24, -24, -24, -24, -22, -20, -20, -18, -18, -17, -17, -17, -16, -16, -16, -15, -15, -14, -14, -14, -13, -13, -13, -13, -13, -13, -13, -12, -12, -11, -11, -10, -10, -08, -08, -07, -07, -07, -07, -06, -06, -06, -06, -06, -06, -05, -05, -04, -03, -02, -02, -02, -02, -02, -02, -01, -01,  00,  00,  00,  01,  01,  01,  01,  01,  02,  03,  03,  04,  04,  04,  05,  05,  05,  06,  06,  07,  08,  08,  09,  09,  09,  10,  10,  11,  11,  12,  13,  13,  13,  14,  14,  14,  14,  15,  15,  15,  16,  16,  17,  17,  17,  18,  18,  18,  18,  18,  18,  18,  18,  19,  19,  21,  21,  21,  21,  21,  22,  22,  23,  23,  24,  24]);
  check([-03,  08,  13,  06,  16, -11, -04,  00,  03, -11, -23, -05, -06, -20, -10, -10,  24, -13, -13, -18, -18, -15,  04,  05,  10,  01, -12, -21, -07,  02,  05,  17, -12, -05,  01, -21,  12,  24,  06, -08, -11,  11,  13,  08,  16, -22,  23,  17, -08, -13, -24,  01,  00,  05, -19, -15, -16, -05, -08,  04,  24,  24, -04, -13, -05,  24, -07, -16,  24, -25,  01, -12,  13, -09,  23,  04, -03,  24, -04,  14, -14, -01, -09,  11, -21, -01,  24, -10,  21, -09,  21,  21, -10, -07, -15,  12, -06,  04,  23, -07,  06,  00,  08, -06,  18,  07, -25, -10,  06, -02, -18,  17,  22,  00,  06, -22,  24,  07, -06,  23, -25, -08,  21, -10,  10, -19, -20, -18,  12],
        [-25, -25, -25, -24, -23, -22, -22, -21, -21, -21, -20, -20, -19, -19, -18, -18, -18, -18, -16, -16, -15, -15, -15, -14, -13, -13, -13, -13, -12, -12, -12, -11, -11, -11, -10, -10, -10, -10, -10, -10, -09, -09, -09, -08, -08, -08, -08, -07, -07, -07, -07, -06, -06, -06, -06, -05, -05, -05, -05, -04, -04, -04, -03, -03, -02, -01, -01,  00,  00,  00,  00,  01,  01,  01,  01,  02,  03,  04,  04,  04,  04,  05,  05,  05,  06,  06,  06,  06,  06,  07,  07,  08,  08,  08,  10,  10,  11,  11,  12,  12,  12,  13,  13,  13,  14,  16,  16,  17,  17,  17,  18,  21,  21,  21,  21,  22,  23,  23,  23,  23,  24,  24,  24,  24,  24,  24,  24,  24,  24]);
  check([-22,  00, -05,  05, -16,  16, -01,  18,  18,  14, -01, -09,  21, -07, -10, -04, -24, -04, -02,  08,  17, -04,  10, -16, -06, -24,  22,  00, -17, -21, -10, -14,  04,  12,  19,  15,  04, -07,  08, -03, -16, -16,  16, -17,  05, -17,  06,  06,  04, -21, -08, -04,  02, -23,  06,  21, -20,  05, -01, -10,  09,  14,  02, -10,  03, -04, -19, -18,  16,  16,  07, -25, -22,  00, -17,  08,  10,  15, -09, -10,  21,  10,  13, -01,  15, -04, -03,  20, -24,  21, -14,  12,  12,  15, -22, -09, -12, -14,  00, -19, -23, -16, -19, -18,  09, -08,  15, -05, -18, -16,  12, -22,  19,  02,  04,  09,  24, -21,  07, -23, -25,  20,  15,  15,  12,  20, -17, -25, -19],
        [-25, -25, -25, -24, -24, -24, -23, -23, -23, -22, -22, -22, -22, -21, -21, -21, -20, -19, -19, -19, -19, -18, -18, -18, -17, -17, -17, -17, -17, -16, -16, -16, -16, -16, -16, -14, -14, -14, -12, -10, -10, -10, -10, -10, -09, -09, -09, -08, -08, -07, -07, -06, -05, -05, -04, -04, -04, -04, -04, -04, -03, -03, -02, -01, -01, -01, -01,  00,  00,  00,  00,  02,  02,  02,  03,  04,  04,  04,  04,  05,  05,  05,  06,  06,  06,  07,  07,  08,  08,  08,  09,  09,  09,  10,  10,  10,  12,  12,  12,  12,  12,  13,  14,  14,  15,  15,  15,  15,  15,  15,  15,  16,  16,  16,  16,  17,  18,  18,  19,  19,  20,  20,  20,  21,  21,  21,  21,  22,  24]);
  check([ 10, -19, -15,  20, -11, -07,  06,  06, -15, -22,  15, -06,  17,  21, -11, -05,  00,  23, -04,  06,  07,  00,  09,  02, -08, -01, -11, -13,  09, -09, -05, -06, -01,  06, -11,  14, -24,  23, -05, -14, -24, -14,  06,  19, -18,  20, -09,  09,  20,  12, -09,  03, -11, -25, -18,  07,  01, -03,  19, -15,  13, -10,  06, -10,  23,  23, -21,  24,  21,  01, -15,  24,  14,  16,  20,  21,  13, -14,  08,  09, -25, -24, -11, -10, -22, -04,  22,  04,  20,  18,  16,  11, -15, -01, -24, -17, -03, -20, -16, -05,  06, -05, -05, -04, -12, -10, -06, -21,  04, -21,  13,  04, -20, -21,  21, -15,  00,  20,  14, -02,  16,  07, -16, -24, -18, -15, -13,  06, -08],
        [-25, -25, -24, -24, -24, -24, -24, -22, -22, -21, -21, -21, -21, -20, -20, -19, -18, -18, -18, -17, -16, -16, -15, -15, -15, -15, -15, -15, -15, -14, -14, -14, -13, -13, -12, -11, -11, -11, -11, -11, -11, -10, -10, -10, -10, -09, -09, -09, -08, -08, -07, -06, -06, -06, -05, -05, -05, -05, -05, -05, -04, -04, -04, -03, -03, -02, -01, -01, -01,  00,  00,  00,  01,  01,  02,  03,  04,  04,  04,  06,  06,  06,  06,  06,  06,  06,  06,  07,  07,  07,  08,  09,  09,  09,  09,  10,  11,  12,  13,  13,  13,  14,  14,  14,  15,  16,  16,  16,  17,  18,  19,  19,  20,  20,  20,  20,  20,  20,  21,  21,  21,  21,  22,  23,  23,  23,  23,  24,  24]);
  check([-04, -23, -25,  18, -03, -02, -16,  14,  20, -12, -07,  24, -24, -25, -18, -18,  23, -08,  09,  19, -16, -18,  10,  18, -07,  13, -25,  03, -25,  07,  22, -01,  09,  23, -08, -17, -02,  01,  24, -07,  14, -05, -05,  17, -05,  04, -24, -05,  21,  11, -11, -18,  20, -24, -24, -11,  17, -22,  19,  17,  12,  17, -09, -02,  17,  11,  08, -10, -13,  08,  11, -21, -22, -17,  23,  00,  12, -25,  20,  10,  11, -13,  20,  08, -10,  21,  22, -18, -24,  17, -25,  16,  09, -07, -11,  03, -21, -25,  20, -06,  08,  06,  00,  11,  14, -02, -14,  03,  00, -17, -11, -12, -05, -16,  22,  11, -18,  19,  20, -15,  11,  20, -24, -03, -09, -07, -25, -03, -07],
        [-25, -25, -25, -25, -25, -25, -25, -25, -24, -24, -24, -24, -24, -24, -23, -22, -22, -21, -21, -18, -18, -18, -18, -18, -18, -17, -17, -17, -16, -16, -16, -15, -14, -13, -13, -12, -12, -11, -11, -11, -11, -10, -10, -09, -09, -08, -08, -07, -07, -07, -07, -07, -07, -06, -05, -05, -05, -05, -05, -04, -03, -03, -03, -02, -02, -02, -02, -01,  00,  00,  00,  01,  03,  03,  03,  04,  06,  07,  08,  08,  08,  08,  09,  09,  09,  10,  10,  11,  11,  11,  11,  11,  11,  11,  12,  12,  13,  14,  14,  14,  16,  17,  17,  17,  17,  17,  17,  18,  18,  19,  19,  19,  20,  20,  20,  20,  20,  20,  20,  21,  21,  22,  22,  22,  23,  23,  23,  24,  24]);
}
