// Copyright (c) 2014 the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import "dart:math" show pow;
import "package:expect/expect.dart";

const whiteSpace = const [
  "",
  "\x09",
  "\x0a",
  "\x0b",
  "\x0c",
  "\x0d",
  "\x85",    /// 01: ok
  "\xa0",
  "\u1680",
  "\u180e",
  "\u2000",
  "\u2001",
  "\u2002",
  "\u2003",
  "\u2004",
  "\u2005",
  "\u2006",
  "\u2007",
  "\u2008",
  "\u2009",
  "\u200a",
  "\u2028",
  "\u2029",
  "\u202f",
  "\u205f",
  "\u3000",
  "\uFEFF"
];

void expectNumEquals(double expected, var actual, String message) {
  if (expected.isNaN) {
    Expect.isTrue(actual is double && actual.isNaN, "isNaN: $message");
  } else {
    Expect.identical(expected, actual, message);
  }
}

// Test source surrounded by any combination of whitespace.
void testParseAllWhitespace(String source, double result) {
  for (String ws1 in whiteSpace) {
    for (String ws2 in whiteSpace) {
      String padded = "$ws1$source$ws2";
      // Use Expect.identical because it also handles NaN and 0.0/-0.0.
      // Except on dart2js: http://dartbug.com/11551
      expectNumEquals(result, double.parse(padded), "parse '$padded'");
      padded = "$ws1$ws2$source";
      expectNumEquals(result, double.parse(padded), "parse '$padded'");
      padded = "$source$ws1$ws2";
      expectNumEquals(result, double.parse(padded), "parse '$padded'");
    }
  }
}

// Test source and -source surrounded by any combination of whitespace.
void testParseWhitespace(String source, double result) {
  assert(result >= 0);
  testParseAllWhitespace(source, result);
  testParseAllWhitespace("-$source", -result);
}

// Test parsing source, optionally preceeded and/or followed by whitespace.
void testParse(String source, double result, [name = ""]) {
  expectNumEquals(result, double.parse(source), "parse '$source:$name");
  expectNumEquals(result, double.parse(" $source"),
                  "parse ' $source':$name");
  expectNumEquals(result, double.parse("$source "),
                  "parse '$source ':$name");
  expectNumEquals(result, double.parse(" $source "),
                  "parse ' $source ':$name");

  expectNumEquals(result, double.parse("+$source"),
                  "parse '+$source:$name");
  expectNumEquals(result, double.parse(" +$source"),
                  "parse ' +$source':$name");
  expectNumEquals(result, double.parse("+$source "),
                  "parse '+$source ':$name");
  expectNumEquals(result, double.parse(" +$source "),
                  "parse ' +$source ':$name");

  expectNumEquals(-result, double.parse("-$source"),
                  "parse '-$source:$name");
  expectNumEquals(-result, double.parse(" -$source"),
                  "parse ' -$source':$name");
  expectNumEquals(-result, double.parse("-$source "),
                  "parse '-$source ':$name");
  expectNumEquals(-result, double.parse(" -$source "),
                  "parse ' -$source ':$name");
}

void testDouble(double value) {
  testParse("$value", value);
  if (value.isFinite) {
    String exp = value.toStringAsExponential();
    String lcexp = exp.toLowerCase();
    testParse(lcexp, value);
    String ucexp = exp.toUpperCase();
    testParse(ucexp, value);
  }
}

void testFail(String source) {
  var object = new Object();
  Expect.throws(() {
    double.parse(source, (s) {
      Expect.equals(source, s);
      throw object;
    });
  }, (e) => identical(object, e), "Fail: '$source'");
  Expect.equals(1.5, double.parse(source, (s) => 1.5));
}

void main() {
  testDouble(0.0);
  testDouble(5e-324);
  testDouble(2.225073858507201e-308);
  testDouble(2.2250738585072014e-308);
  testDouble(0.49999999999999994);
  testDouble(0.5);
  testDouble(0.50000000000000006);
  testDouble(0.9999999999999999);
  testDouble(1.0);
  testDouble(1.0000000000000002);
  testDouble(4294967295.0);
  testDouble(4294967296.0);
  testDouble(4503599627370495.5);
  testDouble(4503599627370497.0);
  testDouble(9007199254740991.0);
  testDouble(9007199254740992.0);
  testDouble(1.7976931348623157e+308);
  testDouble(double.INFINITY);
  testDouble(double.NAN);

  // Strings that cannot occur from toString of a number.
  testParse("000000000000", 0.0);
  testParse("000000000001", 1.0);
  testParse("000000000000.0000000000000", 0.0);
  testParse("000000000001.0000000000000", 1.0);
  testParse("0e0", 0.0);
  testParse("0e+0", 0.0);
  testParse("0e-0", 0.0);
  testParse("1e0", 1.0);
  testParse("1e+0", 1.0);
  testParse("1e-0", 1.0);
  testParse("1.", 1.0);
  testParse(".1", 0.1);
  testParse("1.e1", 10.0);
  testParse(".1e1", 1.0);
  testParse("Infinity", double.INFINITY);
  testParse("NaN", double.NAN);

  // Cases where mantissa and 10^exponent are representable as a double.
  for (int i = -22; i <= 22; i++) {
    for (double j in [1.0, 9007199254740991.0, 9007199254740992.0]) {
      var value = (i >= 0) ? j * pow(10.0, i) : j / pow(10.0, -i);
      testParse("${j}e$i", value, "$i/$j");
      testParse("${j}E$i", value, "$i/$j");
      if (i >= 0) {
        testParse("${j}e+$i", value, "$i/$j");
        testParse("${j}E+$i", value, "$i/$j");
      }
    }
  }
  for (int i = 0; i <= 22; i++) {
    var digits = "9007199254740991";
    for (int i = 0; i < digits.length; i++) {
      int dotIndex = digits.length - i;
      var string = "${digits.substring(0, dotIndex)}."
                   "${digits.substring(dotIndex)}e$i";
      testParse(string, 9007199254740991.0);
    }
  }

  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000012351641146031163604414219821705534309126495065358119110639"
            "6420625168876817552187966324959090408998094949141173861429432731"
            "6641775889849490996936990026954695315751782975778511319614542919"
            "6224552592217965901424968268076250159685228839124609682811834931"
            "8292403785007928846349518531559641397792756664639171692046759890"
            "0776562329863178978731138323263641361002818700324274998854829973"
            "5227010414083113118928696725368169503983880965288753370088162336"
            "8004844756702677687292583305671118833393020810798402309572336459"
            "2015026502876542452438269585569329582311976245631182694093981811"
            "9686640211945509336174248834117544931694293962814151377997828762"
            "2277536275946568454181273895934743339974841620248529105142565927"
            "256981069188614130727188467062660492956638336181640625",
            0.0);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000024703282292062327208828439643411068618252990130716238221279"
            "2841250337753635104375932649918180817996189898282347722858865463"
            "3283551779698981993873980053909390631503565951557022639229085839"
            "2449105184435931802849936536152500319370457678249219365623669863"
            "6584807570015857692699037063119282795585513329278343384093519780"
            "1553124659726357957462276646527282722005637400648549997709659947"
            "0454020828166226237857393450736339007967761930577506740176324673"
            "6009689513405355374585166611342237666786041621596804619144672918"
            "4030053005753084904876539171138659164623952491262365388187963623"
            "9373280423891018672348497668235089863388587925628302755995657524"
            "4555072551893136908362547791869486679949683240497058210285131854"
            "51396213837722826145437693412532098591327667236328125",
            0.0);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000024703282292062327208828439643411068618252990130716238221279"
            "2841250337753635104375932649918180817996189898282347722858865463"
            "3283551779698981993873980053909390631503565951557022639229085839"
            "2449105184435931802849936536152500319370457678249219365623669863"
            "6584807570015857692699037063119282795585513329278343384093519780"
            "1553124659726357957462276646527282722005637400648549997709659947"
            "0454020828166226237857393450736339007967761930577506740176324673"
            "6009689513405355374585166611342237666786041621596804619144672918"
            "4030053005753084904876539171138659164623952491262365388187963623"
            "9373280423891018672348497668235089863388587925628302755995657524"
            "4555072551893136908362547791869486679949683240497058210285131854"
            "51396213837722826145437693412532098591327667236328125",
            0.0);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000024703282292062327208828439643411068627545332140664243314532"
            "8041234170109088178685677591650492652607243027730579814636067699"
            "1112238669661707327453443265068702897439863329200619332642599205"
            "1806252781222000513169502627641523911022607448403553068808609405"
            "1727798181294290864842608522062097649849550765341204993205100587"
            "2127469658709242016690593998242808606978027857019419997429604579"
            "7572623273334010723772922131119806567715298322567005234345331218"
            "5169920860031716486480793611343761679481328431956040281530986197"
            "8304604971452253283193290744072288902141724247846767401941767720"
            "8561650585989659548591956327689896895290365125294085321852619688"
            "9863888974446146846024033780172178553364579041996676675092137151"
            "9705456298034409473812692774776868254618683783877327369245051207"
            "5931578479504396230612962142122846982018227555473696607567828620"
            "5497859173707553281928994692862033843994140625",
            5e-324);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000037054923438093490813242659465116602918087143186126352238665"
            "4061891674274999582254154033144959392383231717975289492511095959"
            "2096640779585747657231506869704773681319051549691937265430115392"
            "9316510179867828993955338712739726887403536747219495345250565253"
            "9734220743745353366904984135736109339114232557854653467028698863"
            "1755341990606652876965097618075398198036065644601954996844545287"
            "8562428797081554870870561495721040952204106503876761616095480465"
            "4854302923481671949982122917011832487483775622035971266330696098"
            "1770527542930458978998057183774359009418156980309146068528141338"
            "9871550473737887132279287842897827763181104688776671568136524122"
            "1523792405286695424882335699501538146509629059245968850620692484"
            "3143597161396297062546730578356156393213233025204703880754948792"
            "4068421520495603769387037857877153017981772444526303392432171379"
            "4502140826292446718071005307137966156005859375",
            5e-324);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000037054923438093490813242659465116602927379485196074357331918"
            "9261875506630452656563898974877271226994284847423521584288298194"
            "9925327669548472990810970080864085947255348927335533958843628758"
            "8673657776653897704274904804228750479055686517373829048435504795"
            "4877211355023786539048555594678924193378269993917515076140279670"
            "2329686989589536936193414969790924083008456100972824996564489920"
            "5681031242249339356786090176104508511951642895866260110264487010"
            "4014534270108033061877749917013356500179062432395206928717009377"
            "6045079508629627357314808756707988746935928736893548082281945435"
            "9059920635836528008522746502352634795082881888442454133993486286"
            "6832608827839705362543821687804230019924524860745587315427697781"
            "770943207565842392181565401187981478869915008544921875",
            5e-324);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000037054923438093490813242659465116602936671827206022362425172"
            "4461859338985905730873643916609583061605337976871753676065500430"
            "7754014559511198324390433292023398213191646304979130652257142124"
            "8030805373439966414594470895717774070707836287528162751620444337"
            "0020201966302219711192127053621739047642307429980376685251860477"
            "2904031988572420995421732321506449967980846557343694996284434553"
            "2799633687417123842701618856487976071699179287855758604433493555"
            "3174765616734394173773376917014880512874349242754442591103322657"
            "0319631474328795735631560329641618484453700493477950096035749532"
            "8248290797935168884766205161807441826984659088108236699850448451"
            "2141425250392715300205307676106921893339420662245205780234703079"
            "2275266989920550781084577445403473184185067145693733619245051207"
            "5931578479504396230612962142122846982018227555473696607567828620"
            "5497859173707553281928994692862033843994140625",
            5e-324);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000049406564584124654417656879286822137236505980261432476442558"
            "5682500675507270208751865299836361635992379796564695445717730926"
            "6567103559397963987747960107818781263007131903114045278458171678"
            "4898210368871863605699873072305000638740915356498438731247339727"
            "3169615140031715385398074126238565591171026658556686768187039560"
            "3106249319452715914924553293054565444011274801297099995419319894"
            "0908041656332452475714786901472678015935523861155013480352649347"
            "2019379026810710749170333222684475333572083243193609238289345836"
            "8060106011506169809753078342277318329247904982524730776375927247"
            "8746560847782037344696995336470179726777175851256605511991315048"
            "9110145103786273816725095583738973359899366480994116420570263709"
            "0279242767544565229087538682506419718265533447265625",
            5e-324);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000049406564584124654417656879286822137255090664281328486629065"
            "6082468340218176357371355183300985305214486055461159629272135398"
            "2224477339323414654906886530137405794879726658401238665285198410"
            "3612505562444001026339005255283047822045214896807106137617218810"
            "3455596362588581729685217044124195299699101530682409986410201174"
            "4254939317418484033381187996485617213956055714038839994859209159"
            "5145246546668021447545844262239613135430596645134010468690662437"
            "0339841720063432972961587222687523358962656863912080563061972395"
            "6609209942904506566386581488144577804283448495693534803883535441"
            "7123301171979319097183912655379793790580730250588170643705239377"
            "9727777948892293692048067560344357106729158083993353350184274303"
            "9410912596068818947625385549553736509237367567754654738490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            5e-324);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000074109846876186981626485318930233205836174286372252704477330"
            "8123783348549999164508308066289918784766463435950578985022191918"
            "4193281559171495314463013739409547362638103099383874530860230785"
            "8633020359735657987910677425479453774807073494438990690501130507"
            "9468441487490706733809968271472218678228465115709306934057397726"
            "3510683981213305753930195236150796396072131289203909993689090575"
            "7124857594163109741741122991442081904408213007753523232190960930"
            "9708605846963343899964245834023664974967551244071942532661392196"
            "3541055085860917957996114367548718018836313960618292137056282677"
            "9743100947475774264558575685795655526362209377553343136273048244"
            "3047584810573390849764671399003076293019258118491937701241384968"
            "6287194322792594125093461156712312786426466050409407761509897584"
            "8136843040991207538774075715754306035963544889052606784864342758"
            "900428165258489343614201061427593231201171875",
            5e-324);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000074109846876186981626485318930233205854758970392148714663837"
            "8523751013260905313127797949754542453988569694847043168576596389"
            "9850655339096945981621940161728171894510697854671067917687257517"
            "7347315553307795408549809608457500958111373034747658096871009590"
            "9754422710047573078097111189357848386756539987835030152280559340"
            "4659373979179073872386829939581848166016912201945649993128979841"
            "1362062484498678713572180352209017023903285791732520220528974020"
            "8029068540216066123755499834026713000358124864790413857434018755"
            "2090159017259254714629617513415977493871857473787096164563890871"
            "8119841271673056017045493004705269590165763776884908267986972573"
            "3665217655679410725087643375608460039849049721491174630855395563"
            "54188641513168478436313080237596295773983001708984375",
            1e-323);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000074109846876186981626485318930233205873343654412044724850344"
            "8923718677971811461747287833219166123210675953743507352131000861"
            "5508029119022396648780866584046796426383292609958261304514284249"
            "6061610746879932829188941791435548141415672575056325503240888674"
            "0040403932604439422384254107243478095284614859960753370503720954"
            "5808063977144841990843464643012899935961693114687389992568869106"
            "5599267374834247685403237712975952143398358575711517208866987110"
            "6349531233468788347546753834029761025748698485508885182206645314"
            "0639262948657591471263120659283236968907400986955900192071499065"
            "6496581595870337769532410323614883653969318176216473399700896902"
            "4282850500785430600410615352213843786678841324490411560469406158"
            "4550533979841101562169154890806946368370134291387467238490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            1e-323);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360069154511635866620203210960799023116591527666370844360221740"
            "6959097927141579506255510282033669865517905502576217080776730054"
            "4280061926888594105653889967660011652398050737212918180359607825"
            "2347125186710418762540332530832907947436024558998429581982425031"
            "7954385059152437399890443876874974725790225802525457699928291235"
            "4093225567689679024960579905428830259962166760571761950743978498"
            "0479564444580149632075553173315669683173879325651468588102366281"
            "5890742832175436061414318821022423405703806955738531400844926622"
            "0550120807237108092835830752700771425423583764509515806613894483"
            "6485368656166704349449158753391942346304638698898642932982747054"
            "5684547703068233784351199339157645340492308605462312698364257812"
            "5",
            1.1125369292536007e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360069154511635866620203210960799023116610112350390740370408247"
            "7359065591852485654875000165498293534740011761472681264331134525"
            "9937435706814044772812816389978636184270645492500111567186634557"
            "1061420380282556183179464713810955130740324099307096988352304114"
            "8240366281709303744177586794760604434318300674651180918151452849"
            "5241915565655447143417214608859882029906947673313501950183867763"
            "4716769334915718603906610534082604802668952109630465576440379371"
            "4211205525428158285205572821025471431094380576457002725617553180"
            "9099224738635444849469333898568030900459127277678319834121502677"
            "4862108980363986101936076072301556410108193098230208064696671383"
            "6302180548174253659674171315763029087322100208461549627978268407"
            "4131669828524253718537846867047316790971834120489029738490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            1.1125369292536007e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360093857793927928947412039400442434185191195972481664588256512"
            "9400380600184308462011953048487227014291989141962100620081191046"
            "1906239926662125432368943599250777752029021933482747432761666932"
            "6081935177574213144751136884007361083502182696938981541236215812"
            "4253211406611428748302338022108627812847664259678077865798649401"
            "4497660229450268863966221848525061212023023248478571949013749179"
            "6696380382410806898101889263285073571646568472249978339940677865"
            "3579969652328069212208231432361613047099274956616864695216972981"
            "6031069881591856241078866777972171115011992742603077167294249913"
            "7481908755860441269310739102717418145889672225195380557264480249"
            "9621987409855350817390775154421748273612200242960133979035379072"
            "1007951555248028896005922474205893068160932603143782761509897584"
            "8136843040991207538774075715754306035963544889052606784864342758"
            "900428165258489343614201061427593231201171875",
            1.1125369292536007e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360093857793927928947412039400442434185209780656501560598443019"
            "9800348264895214610631442931951850683514095400858564803635595517"
            "7563613706587576099527870021569402283901616688769940819588693664"
            "4796230371146350565390269066985408266806482237247648947606094895"
            "4539192629168295092589480939994257521375739131803801084021811015"
            "5646350227416036982422856551956112981967804161220311948453638445"
            "0933585272746375869932946624052008691141641256228975328278690955"
            "1900432345580791435999485432364661072489848577335336019989599540"
            "4580173812990192997712369923839430590047536255771881194801858107"
            "5858649080057723021797656421627032209693226624526945688978404579"
            "0239620254961370692713747131027132020441991845959370908649389667"
            "01396213837722826145437693412532098591327667236328125",
            1.1125369292536007e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360093857793927928947412039400442434185228365340521456608629527"
            "0200315929606120759250932815416474352736201659755028987189999989"
            "3220987486513026766686796443888026815774211444057134206415720396"
            "3510525564718487986029401249963455450110781777556316353975973978"
            "4825173851725161436876623857879887229903814003929524302244972629"
            "6795040225381805100879491255387164751912585073962051947893527710"
            "5170790163081944841764003984818943810636714040207972316616704045"
            "0220895038833513659790739432367709097880422198053807344762226099"
            "3129277744388529754345873069706690065083079768940685222309466301"
            "4235389404255004774284573740536646273496781023858510820692328908"
            "0857253100067390568036719107632515767271783448958607838263400261"
            "9271291212296536333081616208300526650104600844121842238490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            1.112536929253601e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360118561076219991274620867840085845253828033646632276836664299"
            "2641598602648849715007375581870031501510285299140912526494460981"
            "0847165486286558093401850075478792915405182640326963458817779503"
            "7245335555582282368240205603137908586176939915496868313229764759"
            "1124000199184152785288518003113540316961252461082144468115330795"
            "7199474887142394939885133198483395703973441561868861946163298392"
            "1387606100912602107790340074788347699109403186806482068455015628"
            "7910121858986146810584652043706898739275890198932140639134272458"
            "8610226818743277902588909094978089754671488747034246582989821731"
            "5231929503948741694146154089862122073081814550155248444974062103"
            "4794692806854507601076294922896618700391675086456429118934521521"
            "5279242767544565229087538682506419718265533447265625",
            1.112536929253601e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360118561076219991274620867840085845253846618330652172846850806"
            "3041566267359755863626865465334655170732391558037376710048865452"
            "6504539266212008760560776497797417447277777395614156845644806235"
            "5959630749154419788879337786115955769481239455805535719599643842"
            "1409981421741019129575660920999170025489327333207867686338492409"
            "8348164885108163058341767901914447473918222474610601945603187657"
            "5624810991248171079621397435555282818604475970785479056793028718"
            "6230584552238869034375906043709946764666463819650611963906899017"
            "7159330750141614659222412240845349229707032260203050610497429925"
            "3608669828146023446633071408771736136885368949486813576687986432"
            "5412325651960527476399266899502002447221466689455666048548532116"
            "4410912596068818947625385549553736509237367567754654738490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            1.112536929253601e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360143264358512053601829696279729256322427701952743097064699071"
            "5082881275691578670763818348323588650284368938526796065798921972"
            "8473343486060089420116903707069559015036153836596792711219838611"
            "0980145546446076750451009956312361722243098053437420272483555539"
            "7422826546643144133700412148347193404018690918234764633985688961"
            "7603909548902984778890775141579626656034298049775671944433069073"
            "7604422038743259373816676164757751587582092333404991820293327212"
            "5599348679138779961378564655046088380671358199810473933506318818"
            "4091175893098026050831945120249489444259897725127807943670177161"
            "6228469603642478614007734439187597872666848076451986069255795298"
            "8732132513641624634115870738160721633511566723954250399605642781"
            "1287194322792594125093461156712312786426466050409407761509897584"
            "8136843040991207538774075715754306035963544889052606784864342758"
            "900428165258489343614201061427593231201171875",
            1.112536929253601e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360143264358512053601829696279729256322446286636762993074885578"
            "5482848940402484819383308231788212319506475197423260249353326444"
            "4130717265985540087275830129388183546908748591883986098046865342"
            "9694440740018214171090142139290408905547397593746087678853434622"
            "7708807769200010477987555066232823112546765790360487852208850575"
            "8752599546868752897347409845010678425979078962517411943872958339"
            "1841626929078828345647733525524686707077165117383988808631340302"
            "3919811372391502185169818655049136406061931820528945258278945377"
            "2640279824496362807465448266116748919295441238296611971177785355"
            "4605209927839760366494651758097211936470402475783551200969719627"
            "9349765358747644509438842714766105380341358326953487329219653376"
            "04188641513168478436313080237596295773983001708984375",
            1.1125369292536017e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000011125369292"
            "5360143264358512053601829696279729256322464871320782889085072085"
            "5882816605113390968002798115252835988728581456319724432907730915"
            "9788091045910990754434756551706808078781343347171179484873892074"
            "8408735933590351591729274322268456088851697134054755085223313705"
            "7994788991756876822274697984118452821074840662486211070432012189"
            "9901289544834521015804044548441730195923859875259151943312847604"
            "6078831819414397317478790886291621826572237901362985796969353392"
            "2240274065644224408961072655052184431452505441247416583051571936"
            "1189383755894699564098951411984008394330984751465415998685393549"
            "2981950252037042118981569077006826000273956875115116332683643956"
            "9967398203853664384761814691371489127171149929952724258833663970"
            "9550533979841101562169154890806946368370134291387467238490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            1.1125369292536017e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720138309023271733240406421921598046233183055332741688720443481"
            "3918195854283159012511020564067339731035811005152434161553460108"
            "8560123853777188211307779935320023304796101474425836360719215650"
            "4694250373420837525080665061665815894872049117996859163964850063"
            "5908770118304874799780887753749949451580451605050915399856582470"
            "8186451135379358049921159810857660519924333521143523901487956996"
            "0959128889160299264151106346631339366347758651302937176204732563"
            "1781485664350872122828637642044846811407613911477062801689853244"
            "1100241614474216185671661505401542850847167529019031613227788967"
            "2970737312333408698898317506783884692609277397797285865965494109"
            "1369095406136467568702398678315290680984617210924625396728515625",
            2.2250738585072014e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720138309023271733240406421921598046233201640016761584730629988"
            "4318163518994065161130510447531963400257917264048898345107864580"
            "4217497633702638878466706357638647836668696229713029747546242382"
            "3408545566992974945719797244643863078176348658305526570334729146"
            "6194751340861741144068030671635579160108526477176638618079744084"
            "9335141133345126168377794514288712289869114433885263900927846261"
            "5196333779495868235982163707398274485842831435281934164542745653"
            "0101948357603594346619891642047894836798187532195534126462479802"
            "9649345545872552942305164651268802325882711042187835640735397161"
            "1347477636530690451385234825693498756412831797128850997679418438"
            "1986728251242487444025370654920674427814408813923862326342526219"
            "9131669828524253718537846867047316790971834120489029738490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            2.2250738585072014e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720163012305563795567615250361241457301782723638852508948478253"
            "6359478527325887968267463330520896879809894644538317700857921100"
            "6186301853550719538022833566910789404427072670695665613121274757"
            "8429060364284631907291469414840269030938207255937411123218640844"
            "2207596465763866148192781898983602538637890062203535565726940636"
            "8590885797139947888926801753953891471985190009050333899757727677"
            "7175944826990956530177442436600743254820447797901446928043044146"
            "9470712484503505273622550253384036452803081912355396096061899603"
            "6581190688828964333914697530672942540435576507112592973908144397"
            "3967277412027145618759897856109360492194310924094023490247227304"
            "5306535112923584601741974493579393614104508848422446677399636884"
            "6007951555248028896005922474205893068160932603143782761509897584"
            "8136843040991207538774075715754306035963544889052606784864342758"
            "900428165258489343614201061427593231201171875",
            2.2250738585072014e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720163012305563795567615250361241457301801308322872404958664760"
            "6759446192036794116886953213985520549032000903434781884412325572"
            "1843675633476170205181759989229413936299667425982858999948301489"
            "7143355557856769327930601597818316214242506796246078529588519927"
            "2493577688320732492479924816869232247165964934329258783950102250"
            "9739575795105716007383436457384943241929970921792073899197616943"
            "1413149717326525502008499797367678374315520581880443916381057236"
            "7791175177756227497413804253387084478193655533073867420834526162"
            "5130294620227301090548200676540202015471120020281397001415752591"
            "2344017736224427371246815175018974555997865323425588621961151633"
            "5924167958029604477064946470184777360934300451421683607013647479"
            "51396213837722826145437693412532098591327667236328125",
            2.2250738585072014e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720163012305563795567615250361241457301819893006892300968851267"
            "7159413856747700265506443097450144218254107162331246067966730043"
            "7501049413401620872340686411548038468172262181270052386775328221"
            "5857650751428906748569733780796363397546806336554745935958399010"
            "2779558910877598836767067734754861955694039806454982002173263865"
            "0888265793071484125840071160815995011874751834533813898637506208"
            "5650354607662094473839557158134613493810593365859440904719070326"
            "6111637871008949721205058253390132503584229153792338745607152721"
            "3679398551625637847181703822407461490506663533450201028923360785"
            "0720758060421709123733732493928588619801419722757153753675075962"
            "6541800803135624352387918446790161107764092054420920536627658074"
            "4271291212296536333081616208300526650104600844121842238490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            2.225073858507202e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720187715587855857894824078800884868370419561313003121196886039"
            "9600696529790429221262885863903701367028190801717129607271191035"
            "5127227413175152199055740043138804567803233377539881639177387328"
            "9592460742292701130780538133970816533612964474495297895212189790"
            "9078385258336590185178961879988515042751478263607602168043622031"
            "1292700454832073964845713103912225963935608322440623896907276890"
            "1867170545492751739865893248104017382283282512457950656557381910"
            "3800864691161582871998970864729322144979697154670672039979199080"
            "9160347625980385995424739847678861180095072511543762389603716215"
            "1717298160115446043595312843254064419386453249053891377956809158"
            "0479240509922741385427494262054264040883983691918741817298779334"
            "0279242767544565229087538682506419718265533447265625",
            2.225073858507202e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720187715587855857894824078800884868370438145997023017207072547"
            "0000664194501335369882375747368325036250297060613593790825595507"
            "0784601193100602866214666465457429099675828132827075026004414060"
            "8306755935864838551419670316948863716917264014803965301582068873"
            "9364366480893456529466104797874144751279553135733325386266783645"
            "2441390452797842083302347807343277733880389235182363896347166155"
            "6104375435828320711696950608870952501778355296436947644895395000"
            "2121327384414305095790224864732370170370270775389143364751825639"
            "7709451557378722752058242993546120655130616024712566417111324409"
            "0094038484312727796082230162163678483190007648385456509670733487"
            "1096873355028761260750466238659647787713775294917978746912789928"
            "9410912596068818947625385549553736509237367567754654738490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            2.225073858507202e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720212418870147920222032907240528279439019229619113941424920812"
            "2041979202833158177019328630357258515802274441103013146575652027"
            "2753405412948683525770793674729570667434204573809710891579446436"
            "3327270733156495512991342487145269669679122612435849854465980571"
            "5377211605795581533590856025222168129808916720760222333913980197"
            "1697135116592663803851355047008456915996464810347433895177047571"
            "8083986483323409005892229338073421270755971659056460408395693494"
            "1490091511314216022792883476068511786375165155549005334351245440"
            "4641296700335134143667775872950260869683481489637323750284071645"
            "2713838259809182963456893192579540218971486775350629002238542353"
            "4416680216709858418467070077318366974003875329416563097969900593"
            "6287194322792594125093461156712312786426466050409407761509897584"
            "8136843040991207538774075715754306035963544889052606784864342758"
            "900428165258489343614201061427593231201171875",
            2.225073858507202e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720212418870147920222032907240528279439037814303133837435107319"
            "2441946867544064325638818513821882185024380699999477330130056498"
            "8410779192874134192929720097048195199306799329096904278406473168"
            "2041565926728632933630474670123316852983422152744517260835859654"
            "5663192828352447877877998943107797838336991592885945552137141811"
            "2845825114558431922307989750439508685941245723089173894616936837"
            "2321191373658977977723286698840356390251044443035457396733706583"
            "9810554204566938246584137476071559811765738776267476659123871999"
            "3190400631733470900301279018817520344719025002806127777791679839"
            "1090578584006464715943810511489154282775041174682194133952466682"
            "5034313061815878293790042053923750720833666932415800027583911188"
            "54188641513168478436313080237596295773983001708984375",
            2.2250738585072024e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000022250738585"
            "0720212418870147920222032907240528279439056398987153733445293826"
            "2841914532254970474258308397286505854246486958895941513684460970"
            "4068152972799584860088646519366819731179394084384097665233499900"
            "0755861120300770354269606853101364036287721693053184667205738737"
            "5949174050909314222165141860993427546865066465011668770360303425"
            "3994515112524200040764624453870560455886026635830913894056826102"
            "6558396263994546949554344059607291509746117227014454385071719673"
            "8131016897819660470375391476074607837156312396985947983896498558"
            "1739504563131807656934782164684779819754568515974931805299288032"
            "9467318908203746468430727830398768346578595574013759265666391011"
            "5651945906921898169113014030529134467663458535415036957197921783"
            "4550533979841101562169154890806946368370134291387467238490102415"
            "1863156959008792461225924284245693964036455110947393215135657241"
            "099571834741510656385798938572406768798828125",
            2.2250738585072024e-308);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000100208418000448638899805402"
            "5675081023947173151201955956352756465219477298834774949553868001"
            "7172354813827222283890479851867436841402279176933922227531985191"
            "9016883548022442028316027357265612083085166971723487691494714589"
            "5578331812936322386545483369746322848628024994920311121080242823"
            "5714326388663336759990619917681011604908667604621058852897083245"
            "0816977893343649540492907374306035456728456338904015275857972276"
            "1526157631250497160973795706924168310120033199859199012414074894"
            "8202195091387650850548954516920927386376492603899633528627985383"
            "2932104496700123254769405070337764094630065313795728259355401668"
            "8207267296623786504114239553488532354086348874488976434804499149"
            "322509765625",
            1.0020841800044864e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000100208418000448638899805402"
            "5675081023947173151201956040050732492021560444211472901925540702"
            "1586014286915762763038748010206109436411449756840422987132589595"
            "3085274649978096343836685185731606312366412920027708756586099963"
            "7975552653362983131506665648352778744386705639822064232334712197"
            "7867779247842315182877112028658635902319185766818606103345104073"
            "9923514010434512590195330770391813338086311968111531328662618487"
            "3315578521649250253839821623536466419018719481428679013982985346"
            "1571995354000952600076951720089726239270930511481845755671197870"
            "4747866889418540057514218596081876158765839665014133210631738825"
            "4165335079883777074977685931654819161553596657523084699300803458"
            "3609637179288099118793858900152420545987239088654699259390735847"
            "0152049332041520812604025151794328168042160636342216519487980314"
            "421878240614097350935640662328296457417309284210205078125",
            1.0020841800044864e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000100208418000448650025174695"
            "1035150178458809017822159083615579461533985681124447841542417041"
            "9717793267880261310997721975562434111911014099603638548708110842"
            "9228554372955381818449259496459629506201971760632184806762937040"
            "5528236159220080404124633621972774900305368909016987591808198481"
            "1515258588636795736994571683578362033288375244948969302377353651"
            "5803667343942465515751063883649087835332767470268261173797304563"
            "0216301185431893700183322963627539884395226243941187598947530725"
            "0723137660949785162435276134774551939185861652055952702429699518"
            "1666462911218814544860422297294423455917874727086839114692958995"
            "8734568169530500282699951928714187892923739790353511103290941894"
            "8525105836280134665557340439005224794505069516807613438973521965"
            "4847950667958479187395974848205671831957839363657783480512019685"
            "578121759385902649064359337671703542582690715789794921875",
            1.0020841800044864e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000100208418000448650025174695"
            "1035150178458809017822159167313555488336068826501145793914089742"
            "4131452740968801790145990133901106706920184679510139308308715246"
            "3296945474911036133969917324925623735483217708936405871854322414"
            "7925456999646741149085815900579230796064049553918740703062667855"
            "3668711447815774159881063794555986330698893407146516552825374480"
            "4910203461033328565453487279734865716690623099475777226601950774"
            "2005722075830646793049348880239837993293912525510667600516441176"
            "4092937923563086911963273337943350792080299559638164929472912005"
            "3482225303937231347605235823038535520053649078305244065969296152"
            "4692635952790490853563398306880474700390987573387619367787246203"
            "8909645359318233784351199339157645340492308605462312698364257812"
            "5",
            1.0020841800044864e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000100208418000448650025174695"
            "1035150178458809017822159251011531515138151971877843746285762442"
            "8545112214057342269294258292239779301929355259416640067909319649"
            "7365336576866690449490575153391617964764463657240626936945707789"
            "0322677840073401894046998179185686691822730198820493814317137229"
            "5822164306994752582767555905533610628109411569344063803273395309"
            "4016739578124191615155910675820643598048478728683293279406596985"
            "3795142966229399885915374796852136102192598807080147602085351627"
            "7462738186176388661491270541112149644974737467220377156516124492"
            "5297987696655648150350049348782647584189423429523649017245633309"
            "0650703736050481424426844685046761507858235356421727632283550512"
            "9294184882356332903145058239310065886479547694117011957754993659"
            "5152049332041520812604025151794328168042160636342216519487980314"
            "421878240614097350935640662328296457417309284210205078125",
            1.0020841800044866e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000200416836000897255548872220"
            "0630023738871074569163505490783914884205771542336808210387292522"
            "0426513773371285555269939139667533951768747348715410293510510274"
            "9473643242267695845324274779211200861374232469021139022270213528"
            "6462413252451807248010301677826829802384000871843763078195635583"
            "5519882659021798720200352081612073758236883604191202305937584019"
            "3447504651307941031064654937754410393532579156664506650227987556"
            "2093186373340695057796485067216997253892307748415460848623417226"
            "4622904518424429578269271391797007961345371296322204255566117522"
            "4763967378926030323867148635273985338412963098572424905483014370"
            "3443797280914164309330161600193180015563420351180667003643504189"
            "5081099906363532431297601321684709319015382789075374603271484375",
            2.0041683600089726e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000200416836000897255548872220"
            "0630023738871074569163505574481890911007854687713506162758965222"
            "4840173246459826034418207298006206546777917928621911053111114678"
            "3542034344223350160844932607677195090655478417325360087361598902"
            "8859634092878467992971483956433285698142681516745516189450104957"
            "7673335518200777143086844192589698055647401766388749556385604848"
            "2554040768398804080767078333840188274890434785872022703032633767"
            "3882607263739448150662510983829295362790994029984940850192327677"
            "7992704781037731327797268594965806814239809203904416482609330009"
            "6579729771644447126611962161018097402548737449790829856759351526"
            "9401865064174154880193607978359466823030668134214775268139808498"
            "5465639429401631550091460221837129865002621877730073862662220222"
            "0152049332041520812604025151794328168042160636342216519487980314"
            "421878240614097350935640662328296457417309284210205078125",
            2.0041683600089726e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000200416836000897266674241512"
            "5990092893382710435783708618046737880520279924626481102375841562"
            "2971952227424324582377181263362531222277482271385126614686635925"
            "9685314067200635635457506918405218284491037257929836137538435979"
            "6412317598735565265589451930053281854061344785940439548923591241"
            "1320814858995257697204303847509424186616591244519112755417854425"
            "8434194101906757006322811447097462772136890288028752548167319843"
            "0783329927522091597006012323920368828167500792497449435156873056"
            "7143847087986563890155593009650632514154740344478523429367831657"
            "3498325793444721613958165862230644699700772511863535760820571697"
            "3971098153820878087915873975418835554400811267045201672129946935"
            "0381108086393667096854941760689934113520452305882988042245006340"
            "4847950667958479187395974848205671831957839363657783480512019685"
            "578121759385902649064359337671703542582690715789794921875",
            2.0041683600089726e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000200416836000897266674241512"
            "5990092893382710435783708701744713907322363070003179054747514262"
            "7385611700512865061525449421701203817286652851291627374287240329"
            "3753705169156289950978164746871212513772283206234057202629821353"
            "8809538439162226010550634208659737749820025430842192660178060615"
            "3474267718174236120090795958487048484027109406716660005865875254"
            "7540730218997620056025234843183240653494745917236268600971966054"
            "2572750817920844689872038240532666937066187074066929436725783508"
            "0513647350599865639683590212819431367049178252060735656411044144"
            "5314088186163138416702979387974756763836546863081940712096908853"
            "9929165937080868658779320353585122361868059050079309936626251244"
            "0765647609431766215648800660842354659507691394537687301635742187"
            "5",
            2.004168360008973e-292);
  testParse("0.00000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000200416836000897266674241512"
            "5990092893382710435783708785442689934124446215379877007119186963"
            "1799271173601405540673717580039876412295823431198128133887844732"
            "7822096271111944266498822575337206743053529154538278267721206728"
            "1206759279588886755511816487266193645578706075743945771432529989"
            "5627720577353214542977288069464672781437627568914207256313896083"
            "6647266336088483105727658239269018534852601546443784653776612265"
            "4362171708319597782738064157144965045964873355636409438294693959"
            "3883447613213167389211587415988230219943616159642947883454256631"
            "7129850578881555219447792913718868827972321214300345663373246010"
            "5887233720340859229642766731751409169335306833113418201122555553"
            "1150187132469865334442659560994775205494930483192386561026478034"
            "5152049332041520812604025151794328168042160636342216519487980314"
            "421878240614097350935640662328296457417309284210205078125",
            2.004168360008973e-292);
  testParse("0.99999999999999988897769753748434595763683319091796875",
            0.9999999999999999);
  testParse("0.99999999999999988897769753748434595763683319091796879176194859"
            "5190556970945882299241904356487451251641385905655273315006228765"
            "423911854226535211864757002331316471099853515625",
            0.9999999999999999);
  testParse("0.99999999999999994448884876874217297881841659545898433323805140"
            "4809443029054117700758095643512548748358614094344726684993771234"
            "576088145773464788135242997668683528900146484375",
            0.9999999999999999);
  testParse("0.999999999999999944488848768742172978818416595458984375",
            1.0);
  testParse("0.99999999999999994448884876874217297881841659545898441676194859"
            "5190556970945882299241904356487451251641385905655273315006228765"
            "423911854226535211864757002331316471099853515625",
            1.0);
  testParse("0.499999999999999944488848768742172978818416595458984375",
            0.49999999999999994);
  testParse("0.49999999999999994448884876874217297881841659545898439588097429"
            "7595278485472941149620952178243725625820692952827636657503114382"
            "7119559271132676059323785011656582355499267578125",
            0.49999999999999994);
  testParse("0.49999999999999997224442438437108648940920829772949216661902570"
            "2404721514527058850379047821756274374179307047172363342496885617"
            "2880440728867323940676214988343417644500732421875",
            0.49999999999999994);
  testParse("0.4999999999999999722444243843710864894092082977294921875",
            0.5);
  testParse("0.49999999999999997224442438437108648940920829772949220838097429"
            "7595278485472941149620952178243725625820692952827636657503114382"
            "7119559271132676059323785011656582355499267578125",
            0.5);
  testParse("1.9999999999999997779553950749686919152736663818359375",
            1.9999999999999998);
  testParse("1.99999999999999977795539507496869191527366638183593758352389719"
            "0381113941891764598483808712974902503282771811310546630012457530"
            "84782370845307042372951400466263294219970703125",
            1.9999999999999998);
  testParse("1.99999999999999988897769753748434595763683319091796866647610280"
            "9618886058108235401516191287025097496717228188689453369987542469"
            "15217629154692957627048599533736705780029296875",
            1.9999999999999998);
  testParse("1.99999999999999988897769753748434595763683319091796875",
            2.0);
  testParse("1.99999999999999988897769753748434595763683319091796883352389719"
            "0381113941891764598483808712974902503282771811310546630012457530"
            "84782370845307042372951400466263294219970703125",
            2.0);
  testParse("4503599627370495.5",
            4503599627370495.5);
  testParse("4503599627370495.50000000000000000000000000000000000018807909613"
            "1566001274997845955559308450986489083534003441400273004546761512"
            "75634765625",
            4503599627370495.5);
  testParse("4503599627370495.74999999999999999999999999999999999981192090386"
            "8433998725002154044440691549013510916465996558599726995453238487"
            "24365234375",
            4503599627370495.5);
  testParse("4503599627370495.75",
            4503599627370496.0);
  testParse("4503599627370495.75000000000000000000000000000000000018807909613"
            "1566001274997845955559308450986489083534003441400273004546761512"
            "75634765625",
            4503599627370496.0);
  testParse("4503599627370496",
            4503599627370496.0);
  testParse("4503599627370496.00000000000000000000000000000000000037615819226"
            "3132002549995691911118616901972978167068006882800546009093523025"
            "5126953125",
            4503599627370496.0);
  testParse("4503599627370496.49999999999999999999999999999999999962384180773"
            "6867997450004308088881383098027021832931993117199453990906476974"
            "4873046875",
            4503599627370496.0);
  testParse("4503599627370496.5",
            4503599627370496.0);
  testParse("4503599627370496.50000000000000000000000000000000000037615819226"
            "3132002549995691911118616901972978167068006882800546009093523025"
            "5126953125",
            4503599627370497.0);
  testParse("9007199254740991",
            9007199254740991.0);
  testParse("9007199254740991.00000000000000000000000000000000000037615819226"
            "3132002549995691911118616901972978167068006882800546009093523025"
            "5126953125",
            9007199254740991.0);
  testParse("9007199254740991.49999999999999999999999999999999999962384180773"
            "6867997450004308088881383098027021832931993117199453990906476974"
            "4873046875",
            9007199254740991.0);
  testParse("9007199254740991.5",
            9007199254740992.0);
  testParse("9007199254740991.50000000000000000000000000000000000037615819226"
            "3132002549995691911118616901972978167068006882800546009093523025"
            "5126953125",
            9007199254740992.0);
  testParse("1797693134862315708145274237317043567980705675258449965989174768"
            "0315726078002853876058955863276687817154045895351438246423432132"
            "6889464182768467546703537516986049910576551282076245490090389328"
            "9440758685084551339423045832369032229481658085593321233482747978"
            "26204144723168738177180919299881250404026184124858368",
            1.7976931348623157e+308);
  testParse("1797693134862315708145274237317043567980705675258450041064343056"
            "0785749075118623426984641994452647172104399563141377322686588394"
            "6121172130178666127034918365540069095282013901258936156392632590"
            "7055367751483602939820315058278058002847941584484775356682545657"
            "44106770877499077221865536419145864291265781790932992",
            1.7976931348623157e+308);
  testParse("1797693134862315807937289714053034150799341327100378194286569501"
            "7574473832160705543739215666582761354682674973879349714831499292"
            "8620232455220458908340124972127889717295245764493583188543574449"
            "7703035690903649098158444443687202655781915109457910629727393062"
            "50541739356374003666875082388828428992938306508423168",
            1.7976931348623157e+308);
  testParse("1797693134862315807937289714053034150799341327100378269361737789"
            "8044496829276475094664901797758720709633028641669288791094655554"
            "7851940402630657488671505820681908902000708383676273854845817711"
            "5317644757302700698555713669596228429148198608349364752927190741"
            "68444365510704342711559699508093042880177904174497792",
            double.INFINITY);
  testParse("1797693134862315807937289714053034150799341327100378344436906077"
            "8514519826392244645590587928934680064583382309459227867357811816"
            "7083648350040856069002886669235928086706171002858964521148060973"
            "2932253823701752298952982895505254202514482107240818876126988420"
            "86346991665034681756244316627357656767417501840572416",
            double.INFINITY);

  // Edge cases of algorithm (e+-22/23).
  testParse("1e22", 1e22);
  testParse("1e23", 1e23);
  testParse("1e-22", 1e-22);
  testParse("1e-23", 1e-23);

  testParseWhitespace("1", 1.0);
  testParseWhitespace("1.0", 1.0);
  testParseWhitespace("1e1", 10.0);
  testParseWhitespace(".1e1", 1.0);
  testParseWhitespace("1.e1", 10.0);
  testParseWhitespace("1e+1", 10.0);
  testParseWhitespace("1e-1", 0.1);

  // Negative tests - things not to allow.

  // Spaces inside the numeral.
  testFail("- 1");
  testFail("+ 1");
  testFail("2 2");
  testFail("1 .");
  testFail(". 1");
  testFail("1e 2");
  testFail("1 e2");
  // Invalid characters.
  testFail("0x0");
  testFail("0x1H");
  testFail("12H");
  testFail("1x2");
  testFail("00x2");
  testFail("0x2.2");
  // Double exponent without value.
  testFail(".e1");
  testFail("e1");
  testFail("e+1");
  testFail("e-1");
  testFail("-e1");
  testFail("-e+1");
  testFail("-e-1");
  // Too many signs.
  testFail("--1");
  testFail("-+1");
  testFail("+-1");
  testFail("++1");
  // Incorrect ways to write NaN/Infinity.
  testFail("infinity");
  testFail("INFINITY");
  testFail("1.#INF");
  testFail("inf");
  testFail("nan");
  testFail("NAN");
  testFail("1.#IND");
  testFail("indef");
  testFail("qnan");
  testFail("snan");
}
