#!/usr/bin/bash
# =============================================================================
# install-jlink.sh — Install SEGGER J-Link from a compressed tarball on AlmaLinux 9 (ARM64 or x86_64)
#
# Usage:
#   sudo ./install-jlink.sh /path/to/JLink_Linux_*.tar.{gz,xz} [--with-pylink]
#
# Description:
#   - Installs SEGGER J-Link into /opt/SEGGER/JLink
#   - Configures loader and udev rules
#   - Creates handy symlinks in /usr/local/bin
#   - Optionally installs pylink-square for Python use
#   - Automatically detects and removes any mismatched (32-bit) ARM libs
#
# Author: Niels Steyaert
# =============================================================================
set -euo pipefail

# -------- Configuration --------
PREFIX="/opt/SEGGER/JLink"
UDEV_RULE="/etc/udev/rules.d/99-jlink.rules"
LD_CONF="/etc/ld.so.conf.d/segger-jlink.conf"
PROFILE_SNIPPET="/etc/profile.d/segger-jlink.sh"
SEGGER_VENDOR_HEX="1366"

# -------- Args --------
if [[ $# -lt 1 ]]; then
  echo "Usage: sudo $0 /path/to/JLink_Linux_*.tar.{gz,xz} [--with-pylink]"
  exit 1
fi

ARCHIVE="$1"
WITH_PYLINK="${2-}"

if [[ $EUID -ne 0 ]]; then
  echo "Please run as root (sudo)."
  exit 1
fi

if [[ ! -f "$ARCHIVE" ]]; then
  echo "File not found: $ARCHIVE"
  exit 1
fi

ARCH="$(uname -m)"
echo ">>> Detected system architecture: $ARCH"
if [[ "$ARCH" != "aarch64" && "$ARCH" != "x86_64" ]]; then
  echo "Warning: Architecture '$ARCH' not officially supported; proceeding anyway..."
fi

echo ">>> Installing SEGGER J-Link from: $ARCHIVE"
echo ">>> Target directory: $PREFIX"
mkdir -p "$PREFIX"

# -------- Extract and copy --------
WORKDIR="$(mktemp -d)"
trap 'rm -rf "$WORKDIR"' EXIT
tar --auto-compress -xf "$ARCHIVE" -C "$WORKDIR"

# Find top-level folder inside the archive
TOPDIR="$(find "$WORKDIR" -mindepth 1 -maxdepth 1 -type d | head -n1 || true)"
SRC_DIR="${TOPDIR:-$WORKDIR}"

echo ">>> Copying files into $PREFIX"
rsync -a --delete "$SRC_DIR"/ "$PREFIX"/

# -------- Remove 32-bit ARM libs if present --------
echo ">>> Checking for 32-bit ARM libraries that conflict with 64-bit..."
if find "$PREFIX" -type f -name "libjlinkarm_arm.so.*" | grep -q .; then
  echo "Found 32-bit (ARMhf) J-Link libs; removing them to avoid ELFCLASS32 conflicts."
  find "$PREFIX" -type f -name "libjlinkarm_arm.so.*" -delete
  find "$PREFIX" -type l -name "libjlinkarm_arm.so*" -delete
fi

# -------- Dynamic linker configuration --------
echo ">>> Configuring loader paths"
echo "$PREFIX" > "$LD_CONF"
ldconfig

# Ensure symlinks for the main library
cd "$PREFIX"
if ls libjlinkarm.so.* >/dev/null 2>&1; then
  LATEST="$(ls -1 libjlinkarm.so.* | sort -V | tail -1)"
  ln -sf "$LATEST" libjlinkarm.so
  ln -sf "$LATEST" libjlinkarm.so.8
  echo "Linked libjlinkarm.so and libjlinkarm.so.8 → $LATEST"
fi
ldconfig

# -------- Environment profile snippet --------
cat > "$PROFILE_SNIPPET" <<'EOF'
# SEGGER J-Link environment
export PATH="/opt/SEGGER/JLink:$PATH"
if [ -z "${LD_LIBRARY_PATH:-}" ]; then
  export LD_LIBRARY_PATH="/opt/SEGGER/JLink"
else
  export LD_LIBRARY_PATH="/opt/SEGGER/JLink:$LD_LIBRARY_PATH"
fi
EOF

# -------- Symlink executables --------
echo ">>> Creating convenience symlinks in /usr/local/bin"
BIN_DIR="/usr/local/bin"
mkdir -p "$BIN_DIR"
for exe in JLinkExe JLinkGDBServer JLinkGDBServerCLExe JLinkRTTClient \
           JLinkConfig JLinkSWOViewer JLinkRemoteServer JFlashExe JFlashLite; do
  if [[ -x "$PREFIX/$exe" ]]; then
    ln -sf "$PREFIX/$exe" "$BIN_DIR/$exe"
    echo " - Linked: $exe"
  fi
done

# -------- udev rules --------
echo ">>> Installing udev rule for SEGGER J-Link"
cat > "$UDEV_RULE" <<EOF
SUBSYSTEM=="usb", ATTR{idVendor}=="$SEGGER_VENDOR_HEX", MODE="0666"
EOF
udevadm control --reload-rules
udevadm trigger

# -------- Optional: disable ModemManager interference --------
if systemctl is-enabled ModemManager.service >/dev/null 2>&1; then
  echo "Note: ModemManager may interfere with J-Link VCOM devices."
  echo "      You can disable it via: sudo systemctl disable --now ModemManager.service"
fi

# -------- Optional: pylink-square install --------
if [[ "$WITH_PYLINK" == "--with-pylink" ]]; then
  echo ">>> Installing pylink-square for Python integration"
  if command -v python3 >/dev/null 2>&1; then
    python3 -m pip install --upgrade pip >/dev/null
    python3 -m pip install --upgrade pylink-square
  else
    echo "python3 not found; skipping pylink-square install."
  fi
fi

# -------- Post-install sanity check --------
echo ">>> Verifying architecture of installed libjlinkarm.so"
file "$PREFIX"/libjlinkarm.so* | grep -E "ELF|symbolic"

echo
echo ">>> Installation complete!"
echo " - Replug your J-Link device so udev rules apply."
echo " - Test the connection with:  JLinkExe"
echo
echo "Python (pylink) test snippet:"
cat <<'PYHELP'
    python3 - <<'PY'
import pylink
j = pylink.JLink()
j.open()
print(j.product_name, j.hardware_version)
j.close()
PY
PYHELP
