#!/bin/sh

ROOT_DIR=$(dirname "$0")
IMAGES_DIR="${ROOT_DIR}/images"

# Abort the script (and wait for a key to be pressed).
abort_now() {
  echo
  read -p "Aborting now (press Enter to terminate)." a
  exit 1
}

# Validate checksums. Abort on failure.
# Arguments: <validation command>
validate_checksums() {
  echo "Validating files..."

  # The digest files feature relative paths
  (cd "$IMAGES_DIR" && ${1}) \
  || {
    echo "ERROR: Checksums do not match."
    echo "Please download the manual update image again."
    abort_now
  }

  echo "Validation complete."
}

# Flash an image to a partition. Abort on failure.
# Arguments: <device serial number> <partition name> <image file>
flash_image_or_abort() {
  ($FASTBOOT_BIN -s "${1}" flash "${2}" "${3}") \
  || {
    echo
    echo "ERROR: Could not flash the ${2} partition on device ${1}."
    echo
    echo "ERROR: Please unplug the phone, take the battery out, boot the device into"
    echo "ERROR: fastboot mode, plug in the phone, and start this script again."
    echo "ERROR: (To get to fastboot mode, press Volume-Down and Power until the)"
    echo "ERROR: (Fairphone logo appears.)"
    abort_now
  }
}


echo "** Fairphone Open 17.10.1 Manual Flashing Script **"
echo

case "$(uname -s 2> /dev/null)" in
  Darwin)
    FASTBOOT_BIN="${ROOT_DIR}/bin-darwin/fastboot"
    SHA256SUM_BIN=$(which shasum)
    SHA256SUM_CMD="${SHA256SUM_BIN} --algorithm 256 --portable --status --check SHA256SUMS"
    # The 'md5' tool available on Mac OS X cannot validate a digest file…
    MD5SUM_BIN=
    MD5SUM_CMD=
    ;;
  Linux|GNU/Linux)
    FASTBOOT_BIN=$(which fastboot)
    SHA256SUM_BIN=$(which sha256sum)
    SHA256SUM_CMD="${SHA256SUM_BIN} --status --check SHA256SUMS"
    MD5SUM_BIN=$(which md5sum)
    MD5SUM_CMD="${MD5SUM_BIN} --status --check MD5SUMS"
    ;;
  *)
    echo "ERROR: Unsupported operating system (${OSTYPE})."
    echo "ERROR: Only GNU/Linux and Mac OS X systems are supported."
    abort_now
    ;;
esac

if [ ! -x "${FASTBOOT_BIN}" ]
then
  echo $FASTBOOT_BIN
  echo "ERROR: The 'fastboot' command is not available."
  echo "ERROR: Please install the 'fastboot' tool from your package manager."

  abort_now
fi

if [ -x "${SHA256SUM_BIN}" ]
then
  validate_checksums "${SHA256SUM_CMD}"
elif [ -x "${MD5SUM_BIN}" ]
then
  validate_checksums "${MD5SUM_CMD}"
else
  echo "WARNING: No SHA-256 or MD5 checksum program is available."
  echo "WARNING: Cannot validate integrity of the Fairphone Open update."
  echo

  while [ 1 = 1 ]
  do
    read -p "Do you still want to continue? [(y)es/(N)o]: " a
    if [ -z "${a}" -o "${a}" = 'n' -o "${a}" = 'N' ]
    then
      exit 0
    elif [ "${a}" = 'y' -o "${a}" = 'Y' ]
    then
      break
    fi
  done
fi

sn=
look_for_a_device=1

while [ 1 = 1 ]
do
  serial_numbers=

  for sn in $(${FASTBOOT_BIN} devices | grep fastboot | grep -oE '^[[:alnum:]]+')
  do
    product=$($FASTBOOT_BIN -s ${sn} getvar product 2>&1 | grep -oE 'product:\s+FP2')
    if [ $? -eq 0 ]
    then
      serial_numbers="${serial_numbers} $sn"
    fi
  done

  case $(echo $serial_numbers | wc -w | grep -oE '[0-9]+') in
    0)
      echo
      echo "WARNING: No Fairphone 2 found in fastboot mode."
      echo "WARNING: Make sure that a Fairphone 2 is connected."
      echo "WARNING: To check for devices, type '${FASTBOOT_BIN} devices'."
      ;;
    1)
      echo
      echo "One Fairphone 2 in fastboot mode found (serial number: ${sn})."

      sn=${serial_numbers}
      break
      ;;
    *)
      echo
      echo "WARNING: Several Fairphone 2's in fastboot mode connected."
      echo "WARNING: Please connect only one Fairphone 2."
      ;;
  esac

  echo
  while [ 1 = 1 ]
  do
    read -p "Do you want to look for a Fairphone 2 again? [(Y)es/(n)o]: " a
    if [ -z "${a}" -o "${a}" = 'y' -o "${a}" = 'Y' ]
    then
      break
    elif [ "${a}" = 'n' -o "${a}" = 'N' ]
    then
      exit 0
    fi
  done
done

echo
echo "Proceeding to flash the device."
echo
flash_image_or_abort ${sn} rpm "${IMAGES_DIR}/rpm.mbn"
flash_image_or_abort ${sn} sbl1 "${IMAGES_DIR}/sbl1.mbn"
flash_image_or_abort ${sn} tz "${IMAGES_DIR}/tz.mbn"
flash_image_or_abort ${sn} modem "${IMAGES_DIR}/NON-HLOS.bin"
flash_image_or_abort ${sn} splash "${IMAGES_DIR}/splash.img"
flash_image_or_abort ${sn} aboot "${IMAGES_DIR}/emmc_appsboot.mbn"
flash_image_or_abort ${sn} boot "${IMAGES_DIR}/boot.img"
flash_image_or_abort ${sn} recovery "${IMAGES_DIR}/recovery.img"
flash_image_or_abort ${sn} system "${IMAGES_DIR}/system.img"

echo
echo "Flashing successful!"
echo "Your Fairphone 2 will now run **Fairphone Open 17.10.1**."
echo
read -p "Press Enter to reboot the device and complete the installation..." a

${FASTBOOT_BIN} -s ${sn} reboot
