<?php
/**
 * Handles plugin settings with tabbed theme configuration
 * Refactored to use modular partial classes
 */

// Include all partial classes
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-settings-partial-base.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-api-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-firebase-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-leads-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-languages-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-misc-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-theme-general-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-theme-colors-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-theme-typography-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-theme-layout-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-theme-effects-settings.php';
require_once plugin_dir_path(__FILE__) . 'partials/settings/class-theme-legacy-settings.php';

class FansUnitedSettings {
    private $options;
    private $partials = array();
    const OPTION_NAME = 'fans_united_core_options';

    private $config_tabs = array(
        'api' => array(
            'title' => '🔑 API Configuration',
            'description' => 'Configure your Fans United API settings'
        ),
        'firebase' => array(
            'title' => '🔥 Firebase Configuration',
            'description' => 'Configure your Firebase settings'
        ),
        'leads' => array(
            'title' => '📋 Leads',
            'description' => 'Configure leads collection options'
        ),
        'languages' => array(
            'title' => '🌍 Languages',
            'description' => 'Configure language settings'
        ),
        'misc' => array(
            'title' => '⚙️ Misc',
            'description' => 'Miscellaneous widget configuration options'
        )
    );

    private $theme_tabs = array(
        'general' => array(
            'title' => '🎨 General',
            'description' => 'Basic theme configuration options'
        ),
        'colors' => array(
            'title' => '🌈 Colors',
            'description' => 'Color palettes for light and dark modes'
        ),
        'typography' => array(
            'title' => '📝 Typography',
            'description' => 'Font family settings'
        ),
        'layout' => array(
            'title' => '📐 Layout & Spacing',
            'description' => 'Spacing, breakpoints, and borders'
        ),
        'effects' => array(
            'title' => '✨ Effects',
            'description' => 'Gradients and visual effects'
        ),
        'legacy' => array(
            'title' => '⚠️ Legacy',
            'description' => 'Deprecated settings (backward compatibility)'
        )
    );

    public function __construct() {
        $this->options = get_option(self::OPTION_NAME, array());
        $this->initializePartials();
        add_action('admin_menu', array($this, 'addSettingsPage'));
        add_action('admin_init', array($this, 'registerSettings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueueAdminScripts'));
    }

    private function initializePartials() {
        $this->partials = array(
            'api' => new FansUnitedApiSettings($this->options, self::OPTION_NAME),
            'firebase' => new FansUnitedFirebaseSettings($this->options, self::OPTION_NAME),
            'leads' => new FansUnitedLeadsSettings($this->options, self::OPTION_NAME),
            'languages' => new FansUnitedLanguagesSettings($this->options, self::OPTION_NAME),
            'misc' => new FansUnitedMiscSettings($this->options, self::OPTION_NAME),
            'theme_general' => new FansUnitedThemeGeneralSettings($this->options, self::OPTION_NAME),
            'theme_colors' => new FansUnitedThemeColorsSettings($this->options, self::OPTION_NAME),
            'theme_typography' => new FansUnitedThemeTypographySettings($this->options, self::OPTION_NAME),
            'theme_layout' => new FansUnitedThemeLayoutSettings($this->options, self::OPTION_NAME),
            'theme_effects' => new FansUnitedThemeEffectsSettings($this->options, self::OPTION_NAME),
            'theme_legacy' => new FansUnitedThemeLegacySettings($this->options, self::OPTION_NAME),
        );
    }

    public function enqueueAdminScripts($hook) {
        if ('settings_page_fans-united-core' !== $hook) {
            return;
        }

        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        // Enqueue tab switching functionality
        wp_add_inline_script('wp-color-picker', '
            jQuery(document).ready(function($) {
                $(".color-picker").wpColorPicker();
                
                // Tab functionality for both config and theme tabs
                $(".nav-tab").click(function(e) {
                    e.preventDefault();
                    
                    var tabGroup = $(this).closest(".fans-united-tabs");
                    var tabId = $(this).data("tab");
                    var tabType = $(this).data("type");
                    
                    // Remove active class from all tabs in this group
                    tabGroup.find(".nav-tab").removeClass("nav-tab-active");
                    tabGroup.find(".tab-content").hide();
                    
                    // Add active class to clicked tab
                    $(this).addClass("nav-tab-active");
                    
                    // Show corresponding content
                    $("#tab-" + tabType + "-" + tabId).show();
                    
                    // Store active tab in localStorage for persistence
                    localStorage.setItem("fans_united_active_" + tabType + "_tab", tabId);
                });
                
                // Restore active tabs from localStorage or default to first tab
                var activeConfigTab = localStorage.getItem("fans_united_active_config_tab") || "api";
                var activeThemeTab = localStorage.getItem("fans_united_active_theme_tab") || "general";
                
                $(".nav-tab[data-type=\"config\"][data-tab=\"" + activeConfigTab + "\"]").click();
                $(".nav-tab[data-type=\"theme\"][data-tab=\"" + activeThemeTab + "\"]").click();
            });
        ');

        // Add custom admin styles
        wp_add_inline_style('wp-admin', '
            .fans-united-tabs {
                margin: 20px 0;
            }
            .tab-content {
                display: none;
                background: #fff;
                border: 1px solid #c3c4c7;
                border-top: none;
                padding: 20px;
                margin: 0;
            }
            .tab-content.active {
                display: block;
            }
            .tab-description {
                color: #666;
                font-style: italic;
                margin-bottom: 20px;
                padding: 10px;
                background: #f9f9f9;
                border-left: 4px solid #0073aa;
            }
            .section-divider {
                margin: 30px 0 20px 0;
                padding-bottom: 8px;
                border-bottom: 1px solid #ddd;
                color: #666;
                font-size: 14px;
                font-weight: 600;
                text-transform: uppercase;
                letter-spacing: 0.5px;
                width: 100%;
                margin-left: -20px;
                margin-right: -20px;
                padding-left: 20px;
                padding-right: 20px;
            }
            .section-divider:first-of-type {
                margin-top: 15px;
            }
            .deprecated-notice {
                background: #fff2f2;
                border-left: 4px solid #dc3232;
                padding: 15px;
                margin: 0 0 20px 0;
            }
            .deprecated-notice h4 {
                color: #dc3232;
                margin: 0 0 10px 0;
            }
            .warning-badge {
                background: #dc3232;
                color: white;
                padding: 2px 6px;
                border-radius: 3px;
                font-size: 10px;
                font-weight: bold;
                margin-left: 8px;
            }
        ');
    }

    public function addSettingsPage() {
        add_options_page(
            __('Fans United Core Settings', 'fans-united-core'),
            __('Fans United Core', 'fans-united-core'),
            'manage_options',
            'fans-united-core',
            array($this, 'renderSettingsPage')
        );
    }

    public function renderSettingsPage() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }


        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form action="options.php" method="post">
                <?php settings_fields('fans_united_core_options'); ?>
                
                <!-- Fans United Configuration with Tabs -->
                <div class="config-configuration">
                    <h2><?php _e('Fans United Configuration', 'fans-united-core'); ?></h2>
                    <p><?php _e('Configure your Fans United plugin settings, API connections, and basic options.', 'fans-united-core'); ?></p>
                    
                    <div class="fans-united-tabs">
                        <!-- Config Tab Navigation -->
                        <h2 class="nav-tab-wrapper">
                            <?php foreach ($this->config_tabs as $tab_key => $tab_config): ?>
                                <a href="#" 
                                   class="nav-tab" 
                                   data-tab="<?php echo esc_attr($tab_key); ?>"
                                   data-type="config">
                                    <?php echo esc_html($tab_config['title']); ?>
                                </a>
                            <?php endforeach; ?>
                        </h2>

                        <!-- Config Tab Content -->
                        <?php foreach ($this->config_tabs as $tab_key => $tab_config): ?>
                            <div id="tab-config-<?php echo esc_attr($tab_key); ?>" class="tab-content">
                                <div class="tab-description">
                                    <?php echo esc_html($tab_config['description']); ?>
                                </div>
                                
                                <?php do_settings_sections("fans-united-config-{$tab_key}"); ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <!-- Theme Configuration with Tabs -->
                <div class="theme-configuration">
                    <h2><?php _e('Theme Configuration', 'fans-united-core'); ?></h2>
                    <p><?php _e('Configure theme settings for your Fans United widgets. The new theme system provides comprehensive styling options for both light and dark modes.', 'fans-united-core'); ?></p>
                    
                    <div class="fans-united-tabs">
                        <!-- Theme Tab Navigation -->
                        <h2 class="nav-tab-wrapper">
                            <?php foreach ($this->theme_tabs as $tab_key => $tab_config): ?>
                                <a href="#" 
                                   class="nav-tab" 
                                   data-tab="<?php echo esc_attr($tab_key); ?>"
                                   data-type="theme">
                                    <?php echo esc_html($tab_config['title']); ?>
                                </a>
                            <?php endforeach; ?>
                        </h2>

                        <!-- Theme Tab Content -->
                        <?php foreach ($this->theme_tabs as $tab_key => $tab_config): ?>
                            <div id="tab-theme-<?php echo esc_attr($tab_key); ?>" class="tab-content">
                                <div class="tab-description">
                                    <?php echo esc_html($tab_config['description']); ?>
                                </div>
                                
                                <?php do_settings_sections("fans-united-theme-{$tab_key}"); ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    public function registerSettings() {
        register_setting(
            'fans_united_core_options',
            self::OPTION_NAME,
            array($this, 'sanitizeOptions')
        );

        // Register all partial settings
        foreach ($this->partials as $partial) {
            $partial->registerSettings();
        }
    }

    // Section render methods - no longer needed since we use renderEmptySection
    public function renderEmptySection() {
        // Empty section callback for tabs
    }

    public function sanitizeOptions($options) {
        $sanitized = array();
        
        foreach ($options as $key => $value) {
            if (is_array($value)) {
                $sanitized[$key] = array_map('sanitize_text_field', $value);
            } else {
                $sanitized[$key] = sanitize_text_field($value);
            }
        }

        return $sanitized;
    }

    public function getOption($key, $default = null) {
        return isset($this->options[$key]) ? $this->options[$key] : $default;
    }

    public function getAllOptions() {
        return $this->options;
    }

    /**
     * Get theme options in the new structured format
     * @return array
     */
    public function getThemeOptions() {
        $mode = $this->getOption('theme_mode', 'light');

        $themeOptions = array(
            'mode' => $mode,
            'colorSchemes' => array(),
            'customBreakpoints' => array(
                'values' => array()
            ),
            'spacingScale' => array(),
            'customFontFamily' => array(),
            'customRadius' => array(),
            'border' => array(),
            'imageBackgroundGradient' => array()
        );

        // Build color schemes for both light and dark modes
        foreach (array('light', 'dark') as $themeMode) {
            $themeOptions['colorSchemes'][$themeMode] = array(
                'palette' => array(
                    'success' => array(
                        'plainColor' => $this->getOption("theme_{$themeMode}_success_plain_color", '#4CAF50'),
                        'outlinedBorder' => $this->getOption("theme_{$themeMode}_success_outlined_border", '#4CAF50'),
                    ),
                    'danger' => array(
                        'softBg' => $this->getOption("theme_{$themeMode}_danger_soft_bg", '#FEE4E2'),
                        'plainColor' => $this->getOption("theme_{$themeMode}_danger_plain_color", '#F44336'),
                        'outlinedBorder' => $this->getOption("theme_{$themeMode}_danger_outlined_border", '#F44336'),
                    ),
                    'primary' => array(
                        'plainColor' => $this->getOption("theme_{$themeMode}_primary_plain_color", '#1A77D2'),
                        'outlinedBorder' => $this->getOption("theme_{$themeMode}_primary_outlined_border", '#1A77D2'),
                        'onPrimary' => $this->getOption("theme_{$themeMode}_primary_on_primary", '#FAFAFA'),
                        'primaryContainer' => $this->getOption("theme_{$themeMode}_primary_container", '#2397F3'),
                    ),
                    'warning' => array(
                        'softBg' => $this->getOption("theme_{$themeMode}_warning_soft_bg", '#FEF0C7'),
                        'plainColor' => $this->getOption("theme_{$themeMode}_warning_plain_color", '#DC6803'),
                    ),
                ),
                'textPrimary' => $this->getOption("theme_{$themeMode}_text_primary", $themeMode === 'light' ? '#212121' : '#FAFAFA'),
                'textSecondary' => $this->getOption("theme_{$themeMode}_text_secondary", $themeMode === 'light' ? '#212121' : '#FAFAFA'),
                'textColor' => $this->getOption("theme_{$themeMode}_text_color", $themeMode === 'light' ? '#212121' : '#FAFAFA'),
                'textDisabled' => $this->getOption("theme_{$themeMode}_text_disabled", $themeMode === 'light' ? '#212121' : '#FAFAFA'),
                'surface' => $this->getOption("theme_{$themeMode}_surface", $themeMode === 'light' ? '#FFFFFF' : '#424242'),
                'onSurface' => $this->getOption("theme_{$themeMode}_on_surface", $themeMode === 'light' ? '#F5F5F5' : '#212121'),
                'surfaceVariant' => $this->getOption("theme_{$themeMode}_surface_variant", $themeMode === 'light' ? '#EEEEEE' : '#616161'),
                'surfaceTintDim' => $this->getOption("theme_{$themeMode}_surface_tint_dim", $themeMode === 'light' ? '#212121' : '#FAFAFA'),
                'surfaceInverse' => $this->getOption("theme_{$themeMode}_surface_inverse", $themeMode === 'light' ? '#F5F5F5' : '#FAFAFA'),
                'outlineEnabledBorder' => $this->getOption("theme_{$themeMode}_outline_enabled_border", $themeMode === 'light' ? '#E0E0E0' : '#757575'),
                'secondaryContainer' => $this->getOption("theme_{$themeMode}_secondary_container", $themeMode === 'light' ? '#BDBDBD' : '#757575'),
            );
        }

        // Build breakpoints
        $breakpoints = array('xs', 'sm', 'md', 'lg', 'xl', 'xxl');
        $defaultBreakpoints = array('xs' => 0, 'sm' => 444, 'md' => 600, 'lg' => 900, 'xl' => 1200, 'xxl' => 1536);
        foreach ($breakpoints as $bp) {
            $themeOptions['customBreakpoints']['values'][$bp] = (int) $this->getOption("theme_breakpoint_{$bp}", $defaultBreakpoints[$bp]);
        }

        // Build spacing scale
        $spacings = array('3xs', '2xs', 'xs', 'sm', 'md', 'lg', 'xl', '2xl', '3xl');
        $defaultSpacings = array('3xs' => '2px', '2xs' => '4px', 'xs' => '8px', 'sm' => '12px', 'md' => '16px', 'lg' => '24px', 'xl' => '32px', '2xl' => '40px', '3xl' => '48px');
        foreach ($spacings as $spacing) {
            $themeOptions['spacingScale'][$spacing] = $this->getOption("theme_spacing_{$spacing}", $defaultSpacings[$spacing]);
        }

        // Build font families
        foreach (array('light', 'dark') as $themeMode) {
            $themeOptions['customFontFamily'][$themeMode] = array(
                'primary' => $this->getOption("theme_font_{$themeMode}_primary", 'Ubuntu, sans-serif'),
                'secondary' => $this->getOption("theme_font_{$themeMode}_secondary", 'Roboto, sans-serif'),
            );
        }

        // Build radius values
        $radiusValues = array('none', '2xs', 'xs', 'sm', 'md', 'lg', 'xl', '2xl', 'full');
        $defaultRadius = array('none' => '0px', '2xs' => '2px', 'xs' => '4px', 'sm' => '8px', 'md' => '12px', 'lg' => '16px', 'xl' => '24px', '2xl' => '232px', 'full' => '1000px');
        foreach (array('light', 'dark') as $themeMode) {
            $themeOptions['customRadius'][$themeMode] = array();
            foreach ($radiusValues as $radius) {
                $themeOptions['customRadius'][$themeMode][$radius] = $this->getOption("theme_radius_{$themeMode}_{$radius}", $defaultRadius[$radius]);
            }
        }

        // Build border settings
        foreach (array('light', 'dark') as $themeMode) {
            $themeOptions['border'][$themeMode] = array(
                'size' => $this->getOption("theme_border_{$themeMode}_size", $themeMode === 'light' ? '1px' : '2px'),
            );
        }

        // Build gradients
        foreach (array('light', 'dark') as $themeMode) {
            $themeOptions['imageBackgroundGradient'][$themeMode] = array();
            $gradientTypes = $themeMode === 'light' ? array('standard', 'split') : array('standard', 'split', 'overlay');
            foreach ($gradientTypes as $type) {
                $defaultGradient = 'linear-gradient(270deg, rgba(255, 255, 255, 0) 0%, rgba(18, 18, 18, 0.8) 100%)';
                $themeOptions['imageBackgroundGradient'][$themeMode][$type] = $this->getOption("theme_gradient_{$themeMode}_{$type}", $defaultGradient);
            }
        }

        return $themeOptions;
    }
}