<?php
/**
 * Base class for settings partials
 */
abstract class FansUnitedSettingsPartialBase {
    protected $options;
    protected $option_name;

    public function __construct($options, $option_name) {
        $this->options = $options;
        $this->option_name = $option_name;
    }

    /**
     * Register settings sections and fields for this partial
     */
    abstract public function registerSettings();

    /**
     * Get option value with default fallback
     */
    protected function getOption($key, $default = null) {
        return isset($this->options[$key]) ? $this->options[$key] : $default;
    }

    /**
     * Add a settings field
     */
    protected function addField($id, $title, $section, $args = array()) {
        $defaults = array(
            'type' => 'text',
            'description' => '',
            'options' => array(),
            'class' => '',
            'default' => '',
            'page' => 'fans-united-core'
        );
        $args = array_merge($defaults, $args);

        add_settings_field(
            $id,
            $title,
            array($this, 'renderField'),
            $args['page'],
            $section,
            array_merge($args, array('id' => $id))
        );
    }

    /**
     * Render a settings field
     */
    public function renderField($args) {
        $id = $args['id'];
        $type = $args['type'];
        $class = $args['class'];
        $value = $this->getOption($id, $args['default']);
        $name = $this->option_name . "[{$id}]";

        switch ($type) {
            case 'text':
                printf(
                    '<input type="text" id="%s" name="%s" value="%s" class="regular-text %s">',
                    esc_attr($id),
                    esc_attr($name),
                    esc_attr($value),
                    esc_attr($class)
                );
                break;

            case 'textarea':
                printf(
                    '<textarea id="%s" name="%s" class="large-text %s" rows="3">%s</textarea>',
                    esc_attr($id),
                    esc_attr($name),
                    esc_attr($class),
                    esc_textarea($value)
                );
                break;

            case 'html':
                if (isset($args['content'])) {
                    echo $args['content'];
                }
                break;

            case 'color':
                printf(
                    '<input type="text" id="%s" name="%s" value="%s" class="regular-text %s">',
                    esc_attr($id),
                    esc_attr($name),
                    esc_attr($value),
                    esc_attr($class)
                );
                break;

            case 'checkbox':
                printf(
                    '<input type="checkbox" id="%s" name="%s" value="1" %s>',
                    esc_attr($id),
                    esc_attr($name),
                    checked(1, $value, false)
                );
                break;

            case 'select':
                printf('<select id="%s" name="%s" class="%s">',
                    esc_attr($id),
                    esc_attr($name),
                    esc_attr($class)
                );
                foreach ($args['options'] as $key => $label) {
                    printf(
                        '<option value="%s" %s>%s</option>',
                        esc_attr($key),
                        selected($value, $key, false),
                        esc_html($label)
                    );
                }
                echo '</select>';
                break;

            case 'multicheck':
                echo '<fieldset>';
                $selected = (array) $value;
                foreach ($args['options'] as $key => $label) {
                    $checked = in_array($key, $selected) ? 'checked' : '';
                    printf(
                        '<label style="display:block;margin-bottom:8px;"><input type="checkbox" name="%s[]" value="%s" %s> %s</label>',
                        esc_attr($name),
                        esc_attr($key),
                        $checked,
                        esc_html($label)
                    );
                }
                echo '</fieldset>';
                break;
        }

        if (!empty($args['description']) && $type !== 'html') {
            printf('<p class="description">%s</p>', esc_html($args['description']));
        }
    }

    /**
     * Empty section callback for tabs
     */
    public function renderEmptySection() {
        // Empty section callback for tabs
    }
}
