/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.datastore.core.index;

import com.google.apphosting.datastore.shared.Config;
import com.google.cloud.datastore.core.common.DatastoreCoreConfig;
import com.google.cloud.datastore.core.index.IndexValueHelper;
import com.google.cloud.datastore.core.rep.EntitySize;
import com.google.cloud.datastore.core.rep.IndexEntry;
import com.google.cloud.datastore.core.rep.IndexValueAndRemainder;
import com.google.cloud.datastore.core.rep.UnifiedIndexValue;
import com.google.cloud.datastore.core.rep.proto.IndexValue;
import com.google.cloud.datastore.core.rep.proto.IndexValueOrBuilder;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.storage.onestore.v3.OnestoreEntity;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;

public class IndexValueSize {
    private static final int SIZE_NULL = 1;
    private static final int SIZE_BOOLEAN = 1;
    private static final int SIZE_NUMBER = 8;
    private static final int SIZE_TIMESTAMP = 8;
    private static final int SIZE_GEO_POINT = 16;
    private static final int SIZE_RESOURCE_NAME_DATABASE = 16;
    private static final int BASE_STRING_SIZE = 1;
    private static final int TRUNCATION_THRESHOLD = Config.FirestoreV1Constants.getDefaultInstance().getIndexTruncationThresholdBytes();

    public static int indexValueSize(IndexValueOrBuilder value) {
        switch (value.getValueTypeCase()) {
            case NULL_VALUE: {
                return 1;
            }
            case BOOLEAN_VALUE: {
                return 1;
            }
            case NAN_VALUE: 
            case NUMBER_VALUE: {
                return 8;
            }
            case TIMESTAMP_VALUE: {
                return 8;
            }
            case STRING_VALUE: {
                return IndexValueSize.stringSize(value.getStringValue());
            }
            case BYTES_VALUE: {
                return value.getBytesValue().getValue().size();
            }
            case RESOURCE_NAME_VALUE: {
                return IndexValueSize.resourceNameSize(value.getResourceNameValue());
            }
            case GEO_POINT_VALUE: {
                return 16;
            }
            case MAP_VALUE: {
                return IndexValueSize.mapSize(value.getMapValue());
            }
            case ARRAY_VALUE: {
                return IndexValueSize.arraySize(value.getArrayValue());
            }
        }
        String string = String.valueOf(value.getValueTypeCase());
        throw new IllegalArgumentException(new StringBuilder(19 + String.valueOf(string).length()).append("unknown value type ").append(string).toString());
    }

    public static IndexValue truncate(IndexValue value) {
        return IndexValueSize.truncateWithThreshold(value, TRUNCATION_THRESHOLD);
    }

    public static boolean isUnderTruncationThreshold(IndexValue value) {
        return IndexValueSize.indexValueSize(value) <= TRUNCATION_THRESHOLD;
    }

    @VisibleForTesting
    static IndexValue truncateWithThreshold(IndexValue value, int threshold) {
        IndexValueHelper.checkTruncation(value);
        IndexValue.Builder builder = value.toBuilder();
        new IndexValueTruncater(threshold).truncateIndexValue(builder);
        return builder.build();
    }

    private static int stringSize(IndexValue.StringValueOrBuilder string) {
        return 1 + string.getValueBytes().size();
    }

    private static int segmentSize(IndexValue.DocumentResourceName.Segment segment) {
        switch (segment.getIdTypeCase()) {
            case INTEGER_ID: {
                return 8;
            }
            case STRING_ID: {
                return IndexValueSize.stringSize(segment.getStringId());
            }
        }
        String string = String.valueOf(segment.getIdTypeCase());
        throw new IllegalArgumentException(new StringBuilder(21 + String.valueOf(string).length()).append("unknown segment type ").append(string).toString());
    }

    private static int resourceNameSize(IndexValue.ResourceNameOrBuilder resourceName) {
        switch (resourceName.getResourceTypeCase()) {
            case DOCUMENT_RESOURCE_NAME: {
                return IndexValueSize.documentResourceNameSize(resourceName.getDocumentResourceName());
            }
        }
        String string = String.valueOf(resourceName.getResourceTypeCase());
        throw new IllegalArgumentException(new StringBuilder(27 + String.valueOf(string).length()).append("unknown resource name type ").append(string).toString());
    }

    private static int documentResourceNameSize(IndexValue.DocumentResourceNameOrBuilder documentResourceName) {
        int size = 16;
        if (documentResourceName.hasDatabaseId() && !documentResourceName.getNamespaceId().getValue().isEmpty()) {
            size += IndexValueSize.stringSize(documentResourceName.getNamespaceId());
        }
        for (int i = 0; i < documentResourceName.getSegmentsCount(); ++i) {
            size += IndexValueSize.segmentSize(documentResourceName.getSegments(i));
        }
        return size;
    }

    private static int arraySize(IndexValue.ArrayValueOrBuilder array) {
        int sum = 0;
        for (int i = 0; i < array.getValuesCount(); ++i) {
            sum += IndexValueSize.indexValueSize(array.getValues(i));
        }
        return sum;
    }

    private static int mapEntrySize(IndexValue.MapValue.MapEntryOrBuilder entry) {
        int size = IndexValueSize.stringSize(entry.getKey());
        if (entry.hasValue()) {
            size += IndexValueSize.indexValueSize(entry.getValue());
        }
        return size;
    }

    private static int mapSize(IndexValue.MapValueOrBuilder map) {
        int sum = 0;
        for (int i = 0; i < map.getEntriesCount(); ++i) {
            sum += IndexValueSize.mapEntrySize(map.getEntries(i));
        }
        return sum;
    }

    public static int indexValueAndRemainderNaturalSize(IndexValueAndRemainder indexValueAndRemainder) {
        UnifiedIndexValue unifiedIndexValue = indexValueAndRemainder.unifiedIndexValue();
        if (unifiedIndexValue.isDatastore()) {
            return EntitySize.propertyValueSize(unifiedIndexValue.datastoreIndexValue(), OnestoreEntity.Property.Meaning.NO_MEANING);
        }
        int indexValueSize = -1;
        int jndexValueSize = -1;
        if (DatastoreCoreConfig.compareIndexValueJndexValue || !DatastoreCoreConfig.useJndexValue) {
            indexValueSize = IndexValueSize.indexValueSize(unifiedIndexValue.firestoreIndexValue().xjProto());
        }
        if (DatastoreCoreConfig.compareIndexValueJndexValue || DatastoreCoreConfig.useJndexValue) {
            jndexValueSize = unifiedIndexValue.firestoreIndexValue().naturalSize();
        }
        if (DatastoreCoreConfig.compareIndexValueJndexValue && indexValueSize != jndexValueSize) {
            DatastoreCoreConfig.hackReportProblem(DatastoreCoreConfig.ProblemType.INDEX_JNDEX_FIELD_VALUE_SIZE_MISMATCH, null, new Exception(indexValueAndRemainder.toString()));
        }
        return DatastoreCoreConfig.useJndexValue ? jndexValueSize : indexValueSize;
    }

    public static int indexEntryNaturalSize(IndexEntry indexEntry) {
        int sum = 0;
        ImmutableList<IndexValueAndRemainder> billedIndexValues = indexEntry.getSingleFieldIndexValue();
        if (billedIndexValues == null) {
            billedIndexValues = indexEntry.values();
        }
        for (IndexValueAndRemainder value : billedIndexValues) {
            sum += IndexValueSize.indexValueAndRemainderNaturalSize(value);
        }
        return sum;
    }

    private static class IndexValueTruncater {
        private int threshold;

        IndexValueTruncater(int threshold) {
            Preconditions.checkArgument(threshold > 0);
            this.threshold = threshold;
        }

        boolean truncateIndexValue(IndexValue.Builder value) {
            Preconditions.checkArgument(this.threshold > 0);
            switch (value.getValueTypeCase()) {
                case NULL_VALUE: 
                case BOOLEAN_VALUE: 
                case NAN_VALUE: 
                case NUMBER_VALUE: 
                case TIMESTAMP_VALUE: 
                case GEO_POINT_VALUE: {
                    this.threshold -= IndexValueSize.indexValueSize(value);
                    return false;
                }
                case STRING_VALUE: {
                    return this.truncateStringValue(value.getStringValueBuilder());
                }
                case BYTES_VALUE: {
                    return this.truncateBytesValue(value.getBytesValueBuilder());
                }
                case RESOURCE_NAME_VALUE: {
                    return this.truncateResourceName(value.getResourceNameValueBuilder());
                }
                case MAP_VALUE: {
                    return this.truncateMapValue(value.getMapValueBuilder());
                }
                case ARRAY_VALUE: {
                    return this.truncateArrayValue(value.getArrayValueBuilder());
                }
            }
            String string = String.valueOf(value.getValueTypeCase());
            throw new IllegalArgumentException(new StringBuilder(19 + String.valueOf(string).length()).append("unknown value type ").append(string).toString());
        }

        private boolean truncateStringValue(IndexValue.StringValue.Builder string) {
            int i;
            Preconditions.checkArgument(this.threshold > 0);
            --this.threshold;
            int originalSize = string.getValue().length();
            for (i = 0; i < originalSize && this.threshold > 0; ++i) {
                char c = string.getValue().charAt(i);
                if (Character.isHighSurrogate(c)) {
                    this.threshold -= 4;
                    Preconditions.checkArgument(++i < originalSize && Character.isLowSurrogate(string.getValue().charAt(i)));
                    continue;
                }
                if (c <= '\u007f') {
                    --this.threshold;
                    continue;
                }
                if (c <= '\u07ff') {
                    this.threshold -= 2;
                    continue;
                }
                Preconditions.checkArgument(!Character.isSurrogate(c));
                this.threshold -= 3;
            }
            if (i < originalSize) {
                string.setValue(string.getValue().substring(0, i)).setTruncated(true);
            }
            return string.getTruncated();
        }

        private boolean truncateBytesValue(IndexValue.BytesValue.Builder bytes) {
            Preconditions.checkArgument(this.threshold > 0);
            if (bytes.getValue().size() > this.threshold) {
                bytes.setValue(bytes.getValue().substring(0, this.threshold));
                bytes.setTruncated(true);
            }
            this.threshold -= bytes.getValue().size();
            return bytes.getTruncated();
        }

        private boolean truncateResourceName(IndexValue.ResourceName.Builder resourceName) {
            Preconditions.checkArgument(this.threshold > 0);
            switch (resourceName.getResourceTypeCase()) {
                case DOCUMENT_RESOURCE_NAME: {
                    return this.truncateDocumentResourceName(resourceName.getDocumentResourceNameBuilder());
                }
            }
            String string = String.valueOf(resourceName.getResourceTypeCase());
            throw new IllegalArgumentException(new StringBuilder(27 + String.valueOf(string).length()).append("unknown resource name type ").append(string).toString());
        }

        private boolean truncateDocumentResourceName(IndexValue.DocumentResourceName.Builder documentResourceName) {
            Preconditions.checkArgument(this.threshold > 0);
            this.threshold -= 16;
            if (this.threshold <= 0) {
                documentResourceName.clearNamespaceId();
                documentResourceName.clearSegments();
                return true;
            }
            if (!documentResourceName.getNamespaceId().getValue().isEmpty() && this.truncateStringValue(documentResourceName.getNamespaceIdBuilder())) {
                documentResourceName.clearSegments();
                return true;
            }
            return this.truncateList(documentResourceName.getSegmentsBuilderList(), this::truncateSegment, documentResourceName::removeSegments, () -> documentResourceName.setTruncated(true));
        }

        private boolean truncateSegment(IndexValue.DocumentResourceName.Segment.Builder segment) {
            Preconditions.checkArgument(this.threshold > 0);
            switch (segment.getIdTypeCase()) {
                case INTEGER_ID: {
                    this.threshold -= 8;
                    return false;
                }
                case STRING_ID: {
                    return this.truncateStringValue(segment.getStringIdBuilder());
                }
            }
            String string = String.valueOf(segment.getIdTypeCase());
            throw new IllegalArgumentException(new StringBuilder(21 + String.valueOf(string).length()).append("unknown segment type ").append(string).toString());
        }

        private boolean truncateArrayValue(IndexValue.ArrayValue.Builder array) {
            Preconditions.checkArgument(this.threshold > 0);
            return this.truncateList(array.getValuesBuilderList(), this::truncateIndexValue, array::removeValues, () -> array.setTruncated(true));
        }

        private boolean truncateMapValue(IndexValue.MapValue.Builder map) {
            Preconditions.checkArgument(this.threshold > 0);
            return this.truncateList(map.getEntriesBuilderList(), this::truncateMapEntry, map::removeEntries, () -> map.setTruncated(true));
        }

        private boolean truncateMapEntry(IndexValue.MapValue.MapEntry.Builder entry) {
            Preconditions.checkArgument(this.threshold > 0);
            if (this.truncateStringValue(entry.getKeyBuilder()) || this.threshold <= 0) {
                entry.clearValue();
                return true;
            }
            if (entry.hasValue()) {
                return this.truncateIndexValue(entry.getValueBuilder());
            }
            return true;
        }

        private <T> boolean truncateList(List<T> list, Predicate<T> truncationPredicate, Consumer<Integer> removeItem, Runnable setTruncated) {
            boolean truncated;
            int i;
            int originalSize = list.size();
            boolean hasTruncatedElement = false;
            for (i = 0; i < originalSize && this.threshold > 0 && !hasTruncatedElement; ++i) {
                hasTruncatedElement = truncationPredicate.test(list.get(i));
            }
            for (int j = originalSize; j > i; --j) {
                removeItem.accept(j - 1);
            }
            boolean bl = truncated = !hasTruncatedElement && i < originalSize;
            if (truncated) {
                setTruncated.run();
            }
            return hasTruncatedElement || truncated;
        }
    }
}

