/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.datastore.core.rep.converter;

import com.google.cloud.datastore.core.exception.InvalidConversionException;
import com.google.common.base.Preconditions;
import com.google.common.base.Utf8;
import com.google.common.io.BaseEncoding;
import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;

public final class ConverterHelper {
    public static final int MAX_BYTES_PER_UTF16_CHARACTER = 3;
    private static final int NANOSECONDS_PER_MICROSECOND = 1000;
    private static final String NON_UTF8_BYTES_STRING_PREFIX = "__bytes";
    private static final Pattern NON_UTF8_BYTES_STRING_REGEX = Pattern.compile("__bytes(.*)__", 32);
    private static final String NON_UTF8_BYTES_STRING_FORMAT = "__bytes%s__";

    private ConverterHelper() {
    }

    public static String convertUtf8(String description, String descriptionSuffix, byte[] byteArray) throws InvalidConversionException {
        InvalidConversionException.checkConversion(Utf8.isWellFormed(byteArray), "%s%s is not valid UTF-8.", description, descriptionSuffix);
        return new String(byteArray, StandardCharsets.UTF_8);
    }

    public static String convertBytesToString(String description, byte[] byteArray) throws InvalidConversionException {
        if (Utf8.isWellFormed(byteArray)) {
            String string = new String(byteArray, StandardCharsets.UTF_8);
            InvalidConversionException.checkConversion(!string.startsWith(NON_UTF8_BYTES_STRING_PREFIX) || !NON_UTF8_BYTES_STRING_REGEX.matcher(string).matches(), "Invalid %s \"%s\".", description, string);
            return string;
        }
        return String.format(NON_UTF8_BYTES_STRING_FORMAT, BaseEncoding.base64Url().encode(byteArray));
    }

    public static String convertBytesToStringUnchecked(byte[] byteArray, int offset, int length) {
        if (Utf8.isWellFormed(byteArray, offset, length)) {
            return new String(byteArray, offset, length, StandardCharsets.UTF_8);
        }
        return String.format(NON_UTF8_BYTES_STRING_FORMAT, BaseEncoding.base64Url().encode(byteArray));
    }

    public static byte[] convertStringToBytes(String description, String string) {
        byte[] bytes = ConverterHelper.convertStringToBytesIfNonUtf8(description, string);
        if (bytes != null) {
            return bytes;
        }
        return string.getBytes(StandardCharsets.UTF_8);
    }

    @Nullable
    public static byte[] convertStringToBytesIfNonUtf8(String description, String string) {
        if (!string.startsWith(NON_UTF8_BYTES_STRING_PREFIX)) {
            return null;
        }
        Matcher patternMatch = NON_UTF8_BYTES_STRING_REGEX.matcher(string);
        if (!patternMatch.matches()) {
            return null;
        }
        String encodedText = patternMatch.group(1);
        byte[] byteArray = BaseEncoding.base64Url().decode(encodedText);
        Preconditions.checkArgument(!Utf8.isWellFormed(byteArray), "Invalid %s \"%s\".", (Object)description, (Object)string);
        return byteArray;
    }

    public static Timestamp microsecondsToTimestamp(long microseconds) throws InvalidConversionException {
        long seconds = microseconds / 1000000L;
        int microsecondsRemainder = (int)(microseconds % 1000000L);
        if (microsecondsRemainder < 0) {
            microsecondsRemainder += 1000000;
            --seconds;
        }
        Timestamp timestamp = Timestamp.newBuilder().setSeconds(seconds).setNanos(microsecondsRemainder * 1000).build();
        try {
            Timestamps.checkValid(timestamp);
        }
        catch (IllegalArgumentException exception) {
            throw new InvalidConversionException(exception);
        }
        return timestamp;
    }

    public static long timestampToMicrosecondsLossy(Timestamp timestamp) {
        long nanoseconds = timestamp.getNanos();
        return timestamp.getSeconds() * 1000000L + nanoseconds / 1000L;
    }

    public static long timestampsToMicroseconds(Timestamp timestamp) throws InvalidConversionException {
        InvalidConversionException.checkConversion(Timestamps.isValid(timestamp), "invalid timestamp", new Object[0]);
        InvalidConversionException.checkConversion(timestamp.getNanos() % 1000 == 0, "timestamp cannot have more than microseconds precision", new Object[0]);
        InvalidConversionException.checkConversion(timestamp.getSeconds() >= 0L, "Only timestamps past epoch are supported.", new Object[0]);
        return ConverterHelper.timestampToMicrosecondsLossy(timestamp);
    }
}

