/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.time;

import com.google.common.annotations.GwtCompatible;
import com.google.common.base.Preconditions;
import com.google.common.collect.DiscreteDomain;
import com.google.common.math.DoubleMath;
import com.google.common.math.LongMath;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import com.google.errorprone.annotations.DoNotCall;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.TimeUnit;

@GwtCompatible
public final class Durations {
    static final Duration MIN = Duration.ofSeconds(Long.MIN_VALUE);
    private static final double MIN_SECONDS = MIN.getSeconds();
    public static final Duration MAX = Duration.ofSeconds(Long.MAX_VALUE, 999999999L);
    private static final double MAX_SECONDS = 9.223372036854776E18;
    private static final int NANOS_PER_SECOND = 1000000000;
    private static final int NANOS_PER_MICRO = 1000;
    private static final int MICROS_PER_SECOND = 1000000;

    private Durations() {
    }

    public static double toSecondsAsDouble(Duration duration) {
        return (double)duration.getSeconds() + (double)duration.getNano() / 1.0E9;
    }

    public static Duration ofSeconds(double seconds) {
        if (seconds >= 9.223372036854776E18) {
            return MAX;
        }
        if (seconds <= MIN_SECONDS) {
            return MIN;
        }
        long wholeSeconds = DoubleMath.roundToLong(seconds, RoundingMode.FLOOR);
        long nanos = DoubleMath.roundToLong((seconds - (double)wholeSeconds) * 1.0E9, RoundingMode.FLOOR);
        return Duration.ofSeconds(wholeSeconds, nanos);
    }

    @Deprecated
    @DoNotCall(value="Use Duration.ofSeconds(long) instead.")
    public static Duration ofSeconds(long seconds) {
        return Duration.ofSeconds(seconds);
    }

    public static long toMicros(Duration duration) {
        long micros = LongMath.checkedMultiply(duration.getSeconds(), 1000000L);
        return LongMath.checkedAdd(micros, duration.getNano() / 1000);
    }

    public static Duration ofMicros(long micros) {
        long seconds = LongMath.divide(micros, 1000000L, RoundingMode.FLOOR);
        int microsOfSecond = LongMath.mod(micros, 1000000);
        return Duration.ofSeconds(seconds, microsOfSecond * 1000);
    }

    public static Duration of(long value, TimeUnit timeUnit) {
        return Duration.of(value, Durations.convertTimeUnitToChronoUnit(timeUnit));
    }

    private static ChronoUnit convertTimeUnitToChronoUnit(TimeUnit timeUnit) {
        Preconditions.checkNotNull(timeUnit, "timeUnit");
        switch (timeUnit) {
            case NANOSECONDS: {
                return ChronoUnit.NANOS;
            }
            case MICROSECONDS: {
                return ChronoUnit.MICROS;
            }
            case MILLISECONDS: {
                return ChronoUnit.MILLIS;
            }
            case SECONDS: {
                return ChronoUnit.SECONDS;
            }
            case MINUTES: {
                return ChronoUnit.MINUTES;
            }
            case HOURS: {
                return ChronoUnit.HOURS;
            }
            case DAYS: {
                return ChronoUnit.DAYS;
            }
        }
        throw new AssertionError((Object)"Unknown TimeUnit enum constant");
    }

    @CanIgnoreReturnValue
    public static Duration checkNotNegative(Duration duration) {
        Preconditions.checkArgument(!duration.isNegative(), "duration (%s) must not be negative", (Object)duration);
        return duration;
    }

    @CanIgnoreReturnValue
    public static Duration checkPositive(Duration duration) {
        Preconditions.checkArgument(!duration.isNegative() && !duration.isZero(), "duration (%s) must be positive", (Object)duration);
        return duration;
    }

    public static DiscreteDomain<Duration> domain() {
        return DurationDomain.INSTANCE;
    }

    private static final class DurationDomain
    extends DiscreteDomain<Duration> {
        private static final DurationDomain INSTANCE = new DurationDomain();

        private DurationDomain() {
        }

        @Override
        public Duration minValue() {
            return MIN;
        }

        @Override
        public Duration maxValue() {
            return MAX;
        }

        @Override
        public Duration next(Duration value) {
            return value.equals(this.maxValue()) ? null : value.plusNanos(1L);
        }

        @Override
        public Duration previous(Duration value) {
            return value.equals(this.minValue()) ? null : value.minusNanos(1L);
        }

        @Override
        public long distance(Duration start, Duration end) {
            long seconds = LongMath.saturatedSubtract(end.getSeconds(), start.getSeconds());
            long nanos = LongMath.saturatedMultiply(seconds, 1000000000L);
            return LongMath.saturatedAdd(nanos, end.getNano() - start.getNano());
        }

        public String toString() {
            return "Durations.domain()";
        }
    }
}

