/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.flags.ext;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Ascii;
import com.google.common.base.CharMatcher;
import com.google.common.base.Converter;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.flags.Flag;
import com.google.common.flags.InvalidFlagValueException;
import com.google.common.flags.ext.FlagBuilder;
import com.google.common.math.LongMath;
import com.google.common.time.Durations;
import java.time.Duration;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;

public final class DurationFlag
extends Flag<Duration> {
    private static final long NANOS_PER_DAY = 86400000000000L;
    private static final long NANOS_PER_HOUR = 3600000000000L;
    private static final long NANOS_PER_MINUTE = 60000000000L;
    private static final long NANOS_PER_SECOND = 1000000000L;
    private static final long NANOS_PER_MILLI = 1000000L;
    private static final long NANOS_PER_MICRO = 1000L;
    private static final Converter<String, Duration> DURATION_CONVERTER = new Converter<String, Duration>(){

        @Override
        protected Duration doForward(String durationString) {
            return DurationFlag.parseDuration(durationString);
        }

        @Override
        protected String doBackward(Duration duration) {
            return DurationFlag.formatDuration(duration);
        }
    };
    private static final String UNIT_PATTERN_STRING = "(?x) (?<whole>[0-9]+)? (?<frac>\\.[0-9]*)? (?<unit>d|h|ms?|s|us|ns)";
    private static final Pattern UNIT_PATTERN = Pattern.compile("(?x) (?<whole>[0-9]+)? (?<frac>\\.[0-9]*)? (?<unit>d|h|ms?|s|us|ns)");
    private static final CharMatcher ASCII_DIGIT = CharMatcher.inRange('0', '9');

    public static Converter<String, Duration> converter() {
        return DURATION_CONVERTER;
    }

    public static Flag<Duration> value(Duration defaultValue) {
        return new DurationFlag(Preconditions.checkNotNull(defaultValue));
    }

    public static Flag<Duration> nullValue() {
        return new DurationFlag(null);
    }

    public static Flag<Duration> zero() {
        return DurationFlag.value(Duration.ZERO);
    }

    public static Flag<Duration> nonNegativeValue(@Nullable Duration defaultValue) {
        return FlagBuilder.newBuilder(DURATION_CONVERTER).addConstraint(d -> !d.isNegative(), "duration must not be negative").withDefaultValue(defaultValue).build();
    }

    public static Flag<Duration> positiveValue(@Nullable Duration defaultValue) {
        return FlagBuilder.newBuilder(DURATION_CONVERTER).addConstraint(d -> !d.isNegative() && !d.isZero(), "duration must be positive").withDefaultValue(defaultValue).build();
    }

    public static Flag<Duration> interval(@Nullable Duration defaultValue, Duration min, Duration max) {
        Preconditions.checkNotNull(min, "minimum value cannot be null");
        Preconditions.checkNotNull(max, "maximum value cannot be null");
        Preconditions.checkArgument(min.compareTo(max) <= 0, "the minimum value (%s) must be <= to the maximum value (%s)", (Object)min, (Object)max);
        String string = String.valueOf(min);
        FlagBuilder<Duration> flagBuilder = FlagBuilder.newBuilder(DURATION_CONVERTER).addConstraint(d -> d.compareTo(min) >= 0, new StringBuilder(29 + String.valueOf(string).length()).append("duration cannot be less than ").append(string).toString());
        string = String.valueOf(max);
        return flagBuilder.addConstraint(d -> d.compareTo(max) <= 0, new StringBuilder(32 + String.valueOf(string).length()).append("duration cannot be greater than ").append(string).toString()).withDefaultValue(defaultValue).build();
    }

    public static Flag<Duration> days(long defaultValue) {
        return DurationFlag.value(Duration.ofDays(defaultValue));
    }

    public static Flag<Duration> hours(long defaultValue) {
        return DurationFlag.value(Duration.ofHours(defaultValue));
    }

    public static Flag<Duration> minutes(long defaultValue) {
        return DurationFlag.value(Duration.ofMinutes(defaultValue));
    }

    public static Flag<Duration> seconds(long defaultValue) {
        return DurationFlag.value(Duration.ofSeconds(defaultValue));
    }

    public static Flag<Duration> millis(long defaultValue) {
        return DurationFlag.value(Duration.ofMillis(defaultValue));
    }

    public static Flag<Duration> micros(long defaultValue) {
        return DurationFlag.value(Durations.ofMicros(defaultValue));
    }

    public static Flag<Duration> nanos(long defaultValue) {
        return DurationFlag.value(Duration.ofNanos(defaultValue));
    }

    private DurationFlag(@Nullable Duration defaultValue) {
        super(defaultValue);
    }

    @Override
    protected String parsableStringValue(Duration value) {
        return DurationFlag.formatDuration(value);
    }

    @Override
    protected Duration parse(String text) throws InvalidFlagValueException {
        try {
            return DurationFlag.parseDuration(text);
        }
        catch (IllegalArgumentException e) {
            throw new InvalidFlagValueException("Invalid syntax for Duration", e);
        }
    }

    @VisibleForTesting
    static Duration parseDuration(String value) {
        Preconditions.checkArgument(value != null, "input value cannot be null");
        Preconditions.checkArgument(!value.isEmpty(), "input value cannot be empty");
        Preconditions.checkArgument(!value.equals("-"), "input value cannot be '-'");
        Preconditions.checkArgument(!value.equals("+"), "input value cannot be '+'");
        value = CharMatcher.whitespace().trimFrom(value);
        if ("0".equals(value)) {
            return Duration.ZERO;
        }
        long nanos = 0L;
        boolean negative = value.startsWith("-");
        boolean explicitlyPositive = value.startsWith("+");
        int index = negative || explicitlyPositive ? 1 : 0;
        Matcher matcher = UNIT_PATTERN.matcher(value);
        while (matcher.find(index) && matcher.start() == index) {
            Preconditions.checkArgument(ASCII_DIGIT.matchesAnyOf(matcher.group(0)));
            try {
                long whole = Long.parseLong(MoreObjects.firstNonNull(matcher.group("whole"), "0"));
                String string = String.valueOf(MoreObjects.firstNonNull(matcher.group("frac"), ""));
                double frac = Double.parseDouble(string.length() != 0 ? "0".concat(string) : new String("0"));
                long nanosPerUnit = DurationFlag.nanosPerUnit(matcher.group("unit"));
                nanos = LongMath.checkedAdd(nanos, LongMath.checkedMultiply(nanosPerUnit, whole));
                nanos = LongMath.checkedAdd(nanos, (long)((double)nanosPerUnit * frac));
            }
            catch (ArithmeticException e) {
                throw new IllegalArgumentException(e);
            }
            index = matcher.end();
        }
        if (index < value.length()) {
            String string = String.valueOf(value);
            throw new IllegalArgumentException(string.length() != 0 ? "Could not parse entire duration: ".concat(string) : new String("Could not parse entire duration: "));
        }
        if (negative) {
            nanos = -nanos;
        }
        return Duration.ofNanos(nanos);
    }

    private static long nanosPerUnit(String unit) {
        switch (Ascii.toLowerCase(unit)) {
            case "d": {
                return 86400000000000L;
            }
            case "h": {
                return 3600000000000L;
            }
            case "m": {
                return 60000000000L;
            }
            case "s": {
                return 1000000000L;
            }
            case "ms": {
                return 1000000L;
            }
            case "us": {
                return 1000L;
            }
            case "ns": {
                return 1L;
            }
        }
        String string = String.valueOf(unit);
        throw new IllegalArgumentException(string.length() != 0 ? "Unknown duration unit: ".concat(string) : new String("Unknown duration unit: "));
    }

    static String formatDuration(Duration duration) {
        long nanos;
        long micros;
        long millis;
        long seconds;
        long minutes;
        long hours;
        long days;
        if (duration.equals(Duration.ZERO)) {
            return "0";
        }
        StringBuilder sb = new StringBuilder();
        if (duration.isNegative()) {
            sb.append('-');
            duration = duration.negated();
        }
        if ((days = duration.toDays()) > 0L) {
            sb.append(days).append("d");
            duration = duration.minusDays(days);
        }
        if ((hours = duration.toHours()) > 0L) {
            sb.append(hours).append("h");
            duration = duration.minusHours(hours);
        }
        if ((minutes = duration.toMinutes()) > 0L) {
            sb.append(minutes).append("m");
            duration = duration.minusMinutes(minutes);
        }
        if ((seconds = duration.getSeconds()) > 0L) {
            sb.append(seconds).append("s");
            duration = duration.minusSeconds(seconds);
        }
        if ((millis = duration.toMillis()) > 0L) {
            sb.append(millis).append("ms");
            duration = duration.minusMillis(millis);
        }
        if ((micros = Durations.toMicros(duration)) > 0L) {
            sb.append(micros).append("us");
            duration = duration.minus(Durations.ofMicros(micros));
        }
        if ((nanos = duration.toNanos()) > 0L) {
            sb.append(nanos).append("ns");
            duration = duration.minusNanos(nanos);
        }
        Verify.verify(duration.equals(Duration.ZERO), "problem formatting duration. partial format: %s; leftovers: %s", (Object)sb, (Object)duration);
        Verify.verify(sb.length() > 0, "problem formatting duration. leftovers: %s", (Object)duration);
        return sb.toString();
    }

    public static final class DurationFactory {
        public static Duration valueOf(String durationString) {
            return DurationFlag.parseDuration(durationString);
        }

        private DurationFactory() {
        }
    }
}

