/*
 * Decompiled with CFR 0.152.
 */
package com.mythx.wrapper.service;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.mythx.wrapper.Launch;
import com.mythx.wrapper.config.Config;
import com.mythx.wrapper.config.HttpClientConfig;
import com.mythx.wrapper.controller.ErrorController;
import com.mythx.wrapper.model.LauncherDataResponse;
import com.mythx.wrapper.model.Version;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UpdateService {
    private static final Logger log = LoggerFactory.getLogger(UpdateService.class);
    private static final String BASE_DIR = System.getProperty("user.home") + File.separator + ".mythx" + File.separator;
    private static final String LAUNCHER_DIR = BASE_DIR + "launcher" + File.separator;
    private static final String LAUNCHER_VERSION_PATH = LAUNCHER_DIR + "launcher_version.json";
    private static File file = new File(LAUNCHER_VERSION_PATH);
    private static final int MAX_RETRIES = 3;
    private static final int RETRY_DELAY_MS = 1500;
    private final CloseableHttpClient httpClient = HttpClientConfig.getHttpClient();

    public void init() {
        this.createDirectory();
        this.loadLocalVersion();
        this.loadRemoteVersionWithRetry();
        this.checkUpdates();
    }

    private void loadRemoteVersionWithRetry() {
        for (int attempt = 1; attempt <= 3; ++attempt) {
            this.loadRemoteVersion();
            if (Config.get().getRemoteVersion() != null) {
                return;
            }
            if (attempt >= 3) continue;
            log.info("Retry attempt {} of {} in {}ms...", attempt, 3, 1500);
            try {
                Thread.sleep(1500L);
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
        log.warn("Failed to load remote version after {} attempts", (Object)3);
    }

    private void createDirectory() {
        File parentDir = file.getParentFile();
        if (!parentDir.exists()) {
            if (parentDir.mkdirs()) {
                log.info("Directory created: " + parentDir.getAbsolutePath());
            } else {
                log.warn("Failed to create directory: " + parentDir.getAbsolutePath());
            }
        }
    }

    private void loadLocalVersion() {
        Gson gson = new Gson();
        if (file.exists()) {
            try (FileReader reader = new FileReader(file);){
                Type type = new TypeToken<Version>(){}.getType();
                Config.get().setLocalVersion((Version)gson.fromJson((Reader)reader, type));
            }
            catch (IOException e) {
                log.warn(e.getMessage(), e);
                ErrorController.sendErrorAsync("unknown", e);
            }
        }
    }

    public void loadRemoteVersion() {
        Gson gson = new Gson();
        String versionUrl = Config.get().getLauncherVersionUrl();
        log.info("Fetching launcher version from: {}", (Object)versionUrl);
        HttpGet httpGet = new HttpGet(versionUrl);
        try (CloseableHttpResponse response = this.httpClient.execute(httpGet);){
            String responseString = EntityUtils.toString(response.getEntity());
            LauncherDataResponse dataResponse = gson.fromJson(responseString, LauncherDataResponse.class);
            Config.get().setRemoteVersion(dataResponse.getData());
            log.info("Remote version loaded: {}", (Object)dataResponse.getData());
        }
        catch (IOException e) {
            log.warn("Error retrieving data from API", e);
            ErrorController.sendErrorAsync("unknown", e);
        }
    }

    public void checkUpdates() {
        Version localVersion = Config.get().getLocalVersion();
        Version remoteVersion = Config.get().getRemoteVersion();
        log.info("=== Update Check ===");
        log.info("Local version: {}", (Object)localVersion);
        log.info("Remote version: {}", (Object)remoteVersion);
        if (remoteVersion == null) {
            if (localVersion != null && this.hasCachedLauncher(localVersion)) {
                log.info("Cannot reach update server. Using cached launcher: {}", (Object)localVersion.getFilename());
                return;
            }
            throw new RuntimeException("Cannot reach update server and no cached launcher available. Please check your internet connection.");
        }
        if (localVersion == null) {
            log.info("Local version is missing. Starting fresh download...");
            log.info("Download URL: {}", (Object)remoteVersion.getUrl());
            log.info("Filename: {}", (Object)remoteVersion.getFilename());
            this.downloadJar(remoteVersion.getUrl(), remoteVersion.getFilename());
        } else {
            int locVersion = Integer.parseInt(localVersion.getVersion());
            int remVersion = Integer.parseInt(remoteVersion.getVersion());
            log.info("Version comparison: local={} vs remote={}", (Object)locVersion, (Object)remVersion);
            if (locVersion < remVersion) {
                log.info("Update available! Downloading new version...");
                log.info("Download URL: {}", (Object)remoteVersion.getUrl());
                log.info("Filename: {}", (Object)remoteVersion.getFilename());
                this.downloadJar(remoteVersion.getUrl(), remoteVersion.getFilename());
            } else {
                log.info("Launcher is up to date (version {})", (Object)locVersion);
            }
        }
    }

    private boolean hasCachedLauncher(Version version) {
        if (version == null || version.getFilename() == null) {
            return false;
        }
        File cachedJar = new File(LAUNCHER_DIR + version.getFilename() + ".jar");
        return cachedJar.exists();
    }

    public void downloadJar(String fileURL, String fileName) {
        block53: {
            String savePath = LAUNCHER_DIR + fileName + ".jar";
            log.info("Starting download - URL: {}", (Object)fileURL);
            log.info("Save path: {}", (Object)savePath);
            File existingFile = new File(savePath);
            if (existingFile.exists()) {
                log.info("Deleting existing file before fresh download: {}", (Object)savePath);
                if (existingFile.delete()) {
                    log.info("Existing file deleted successfully");
                } else {
                    log.warn("Failed to delete existing file - may cause download issues");
                }
            }
            CloseableHttpClient client = HttpClientConfig.getHttpClient();
            String separator = fileURL.contains("?") ? "&" : "?";
            String cacheBustedUrl = fileURL + separator + "t=" + System.currentTimeMillis();
            log.info("Cache-busted URL: {}", (Object)cacheBustedUrl);
            HttpGet request = new HttpGet(cacheBustedUrl);
            request.setHeader("Cache-Control", "no-cache, no-store, must-revalidate");
            request.setHeader("Pragma", "no-cache");
            try (CloseableHttpResponse response = client.execute(request);){
                int statusCode = response.getStatusLine().getStatusCode();
                log.info("Download response status: {}", (Object)statusCode);
                if (statusCode == 200) {
                    HttpEntity entity = response.getEntity();
                    if (entity != null) {
                        long contentLength = entity.getContentLength();
                        log.info("Content length: {} bytes", (Object)contentLength);
                        try (InputStream inputStream = entity.getContent();){
                            Path outputPath = Paths.get(savePath, new String[0]);
                            Files.createDirectories(outputPath.getParent(), new FileAttribute[0]);
                            try (FileOutputStream fileOutputStream = new FileOutputStream(savePath);){
                                int bytesRead;
                                byte[] buffer = new byte[1024];
                                long totalBytesRead = 0L;
                                while ((bytesRead = inputStream.read(buffer)) != -1) {
                                    fileOutputStream.write(buffer, 0, bytesRead);
                                    totalBytesRead += (long)bytesRead;
                                    if (contentLength <= 0L) continue;
                                    int progress = (int)(totalBytesRead * 100L / contentLength);
                                    this.showProgress(progress);
                                }
                                log.info("Total bytes written: {}", (Object)totalBytesRead);
                            }
                            File downloadedFile = new File(savePath);
                            if (!downloadedFile.exists()) {
                                log.error("Download appeared to complete but file does not exist: {}", (Object)savePath);
                                return;
                            }
                            log.info("Download completed and verified: {} ({} bytes)", (Object)savePath, (Object)downloadedFile.length());
                            if (this.isOldJarExist()) {
                                this.deleteOldJar();
                            }
                            this.saveVersion();
                            break block53;
                        }
                    }
                    log.error("Response entity is null - no content received");
                    break block53;
                }
                log.warn("Failed to download file: {} - Response: {}", (Object)fileURL, (Object)response.getStatusLine());
            }
            catch (IOException e) {
                log.error("Error while downloading file from {}: {}", fileURL, e.getMessage(), e);
                ErrorController.sendErrorAsync("unknown", e);
            }
        }
    }

    private void showProgress(int progress) {
        if (Launch.getSplash() != null) {
            Launch.getSplash().setProgress(progress);
        }
    }

    public boolean isOldJarExist() {
        Version localVersion = Config.get().getLocalVersion();
        Version remoteVersion = Config.get().getRemoteVersion();
        if (localVersion != null && !localVersion.getFilename().equals(remoteVersion.getFilename())) {
            return new File(LAUNCHER_DIR + localVersion.getFilename() + ".jar").exists();
        }
        return false;
    }

    private void deleteOldJar() {
        String oldJar = LAUNCHER_DIR + Config.get().getLocalVersion().getFilename() + ".jar";
        File oldFile = new File(oldJar);
        if (oldFile.exists()) {
            if (oldFile.delete()) {
                log.info("Old JAR file deleted successfully: " + oldJar);
            } else {
                log.warn("Failed to delete old JAR file: " + oldJar);
            }
        } else {
            log.info("No old JAR file found to delete at: " + oldJar);
        }
    }

    private void saveVersion() {
        Gson gson = new Gson();
        try (FileWriter writer = new FileWriter(file);){
            Config.get().setLocalVersion(Config.get().getRemoteVersion());
            gson.toJson((Object)Config.get().getLocalVersion(), (Appendable)writer);
        }
        catch (IOException e) {
            log.warn(e.getMessage(), e);
            ErrorController.sendErrorAsync("unknown", e);
        }
    }
}

