let codeViewEditor;
let fileLanguage = "dart";

/**
 * Initializes the Monaco editors for a 3-way merge view and sets up various
 * configurations and listeners.
 *
 * @param {string} content - The content of the file.
 * @param {boolean} isLightMode - A flag indicating whether the light mode is enabled.
 */
function initializeEditor(content, isLightMode) {
  updateWebviewTheme(isLightMode);

  require.config({
    paths: {
      vs: "https://cdnjs.cloudflare.com/ajax/libs/monaco-editor/0.34.0/min/vs",
    },
  });
  require(["vs/editor/editor.main"], function () {
    setCustomTheme(isLightMode);

    // Decode the content from base64 to string
    let contentDecoded;
    try {
      contentDecoded = atob(content);
    } catch (error) {
      console.error(`Error decoding file content: ${error}`);
      return;
    }

    initializeCodeView(contentDecoded);
  });
}

// Modified window resize handler
window.addEventListener("resize", () => {
  codeViewEditor.layout();
});

function initializeCodeView(content) {
  try {
    // Create the editor to render the file content
    // Show and setup single editor container
    const container = ensureContainerExists("single-container");
    // container.style.width = "100%";
    // container.style.height = "100%";
    // container.style.padding = "0 0 0 8px";
    // container.style.margin = "0";
    // container.style.boxSizing = "border-box";

    codeViewEditor = monaco.editor.create(container, {
      value: content,
      language: fileLanguage,
      readOnly: true,
      unusualLineTerminators: "off",
      minimap: { enabled: false },
    });
  } catch (error) {
    console.error(`Error initializing single editor: ${error}`);
  }
}

// Apply FF theme to the Monaco Editor
function setCustomTheme(isLightMode) {
  const themeName = isLightMode ? "custom-light-theme" : "custom-dark-theme";

  const themeConfig = isLightMode
    ? {
        base: "vs", // Light theme base
        inherit: true,
        rules: [{ token: "", background: "FFFFFF" }],
        colors: {
          "editor.background": "#FFFFFF", // Editor background color
          "editor.lineHighlightBackground": "#F1F4F8",
          "editor.selectionBackground": "#4B39EF4C",
          "editor.selectionHighlightBackground": "#4B39EF4C",
          "diffEditor.insertedLineBackground": "#ceefeb", // Light Green
          "diffEditor.insertedTextBackground": "#aae3dc",
          "diffEditor.removedLineBackground": "#f6d0d0", // Light Red
          "diffEditor.removedTextBackground": "#efadad",
          "editorOverviewRuler.addedForeground": "#ceefeb", // Green in minimap
          "editorOverviewRuler.deletedForeground": "#f6d0d0", // Red in minimap
        },
      }
    : {
        base: "vs-dark", // Dark theme base
        inherit: true,
        rules: [{ token: "", background: "14181B" }],
        colors: {
          "editor.background": "#14181B", // Editor background color
          "editor.lineHighlightBackground": "#1D2428",
          "editor.selectionBackground": "#4B39EF4C",
          "editor.selectionHighlightBackground": "#4B39EF4C",
          "diffEditor.insertedLineBackground": "#264642", // Light Green
          "diffEditor.insertedTextBackground": "#26615c",
          "diffEditor.removedLineBackground": "#4e2827", // Light Red
          "diffEditor.removedTextBackground": "#6c2b2d",
          "editorOverviewRuler.addedForeground": "#264642", // Green in minimap
          "editorOverviewRuler.deletedForeground": "#4e2827", // Red in minimap
        },
      };

  // Define the custom theme
  monaco.editor.defineTheme(themeName, themeConfig);

  // Apply the custom theme
  monaco.editor.setTheme(themeName);
}

/**
 * Updates the webview theme based on the provided mode.
 *
 * This function changes the background color of the body and the document element
 * based on whether the light mode is enabled or not. It also updates the colors
 * of the dividers with the appropriate color for the selected mode.
 *
 * @param {boolean} isLightMode - A boolean indicating if the light mode is enabled.
 * If true, the light mode is enabled; otherwise, the dark mode is enabled.
 */
function updateWebviewTheme(isLightMode) {
  const backgroundColor = isLightMode ? "#FFFFFF" : "#14181B";
  document.body.style.backgroundColor = backgroundColor;
  document.documentElement.style.backgroundColor = backgroundColor;

  const dividerColor = isLightMode ? "#E0E3E7" : "#323B45";

  // Update all possible dividers
  const dividers = ["divider", "horizontal-divider", "diff-divider"];
  dividers.forEach((id) => {
    const element = document.getElementById(id);
    if (element) {
      element.style.backgroundColor = dividerColor;
    }
  });
}

// Utility functions for DOM manipulation
function ensureContainerExists(id) {
  let container = document.getElementById(id);
  if (!container) {
    container = document.createElement("div");
    container.id = id;
    container.className = "editor-pane";
    document.getElementById("top-editors-container").appendChild(container);
  }
  container.style.display = "block";
  return container;
}

/**
 * Updates the editor content with new text.
 *
 * @param {string} newContent - The new content to set in the editor.
 */
function updateEditorContent(newContent) {
  if (!codeViewEditor) return;

  // Decode the content from base64 to string
  let contentDecoded;
  try {
    contentDecoded = atob(newContent);
  } catch (error) {
    console.error(`Error decoding file content: ${error}`);
    return;
  }

  try {
    // Update the content
    codeViewEditor.setValue(contentDecoded);

    // Reset scroll position to top
    codeViewEditor.revealPositionInCenter({ lineNumber: 1, column: 1 });
  } catch (error) {
    console.error(`Error updating editor content: ${error}`);
  }
}
