// Editors for the merge view
let baseYamlEditor, headYamlEditor, resolvedYamlEditor;

// Editor for the project refactor view
let singleEditor;

// Editor for the commit view
let commitDiffEditor;

let resolvedYamlModel;
let isSyncingScroll = false;
let conflictDecorations = [];
let errorDecorations = [];
let fileLanguage = "yaml";
let isMac;
let currentFileKey;

let currentView;

const extensionToLanguageMap = {
  js: "javascript",
  ts: "typescript",
  json: "json",
  yaml: "yaml",
  yml: "yaml",
  css: "css",
  scss: "scss",
  html: "html",
  xml: "xml",
  md: "markdown",
  markdown: "markdown",
  cpp: "cpp",
  c: "c",
  h: "cpp",
  java: "java",
  py: "python",
  sql: "sql",
  swift: "swift",
  kotlin: "kotlin",
  dart: "dart",
  sh: "shell",
  bash: "shell",
  txt: "plaintext",
  log: "plaintext",
};

// Store docs decorations for each editor,
// helps to add the documentation tooltips
let commitOriginalDocDecorations = [];
let commitModifiedDocDecorations = [];
let baseDocDecorations = [];
let headDocDecorations = [];
let resolvedDocDecorations = [];
let singleDocDecorations = [];

let conflictPositions = []; // Stores all conflict line numbers
let currentConflictIndex = -1; // Tracks the current conflict

// Store the current documentation data for all the editors
let documentationMap = {};

// Currently only used for the single view code editor
// Stores fileKey -> Monaco Model
const modelCache = new Map();
// Caches caret positions for files
// Stores fileKey -> {lineNumber, column}
const caretPositionCache = new Map();

// Add these at the top with other handlers
const DIVIDER_POSITION_HANDLER = "dividerPositionChanged";

// Add this at the top with other global variables
let mouseDownDisposables = new Map();

/**
 * Shows the specified view and hides others
 * @param {string} viewType - Type of view to show ('merge', 'diff', or 'unchanged')
 */
function showView(viewType) {
  // Hide all views first
  document.getElementById("merge-view").style.display = "none";
  document.getElementById("commit-view").style.display = "none";
  document.getElementById("single-view").style.display = "none";

  // Show the selected view
  const viewElement = document.getElementById(`${viewType}-view`);
  viewElement.style.display = "flex";
  viewElement.style.flexDirection = "column";

  // Update current view tracker
  currentView = viewType;

  // Trigger layout update for visible editors
  requestAnimationFrame(() => {
    switch (viewType) {
      case "merge":
        if (baseYamlEditor && headYamlEditor && resolvedYamlEditor) {
          baseYamlEditor.layout();
          headYamlEditor.layout();
          resolvedYamlEditor.layout();
        }
        break;
      case "commit":
        if (commitDiffEditor) {
          commitDiffEditor.layout();
        }
        break;
      case "single":
        if (singleEditor) {
          singleEditor.layout();
        }
        break;
    }
  });
}

function inferLanguageFromExtension(extension) {
  return extensionToLanguageMap[extension] || "yaml";
}

/**
 * Initializes the Monaco editors for a 3-way merge view and sets up various
 * configurations and listeners.
 *
 * @param {string} initialYamlContent - The initial YAML content.
 * @param {string} baseYamlContent - The base YAML content.
 * @param {string} headYamlContent - The head YAML content.
 * @param {string} resolvedYamlContent - The resolved YAML content.
 * @param {Object} docsMap - A map containing documentation tooltips.
 * @param {boolean} isLightMode - A flag indicating whether the light mode is enabled.
 * @param {string} fileType - The type of file ('merge', 'commit', or 'single').
 * @param {string} fileExtension - The file extension (e.g., 'yaml', 'dart').
 * @param {boolean} isFileEditable - A flag indicating whether the file is editable.
 * @param {number} highlightedErrorLineNumber - The line number to highlight in the editor.
 * @param {Object} keyToTreeMetaData - A map containing key to tree metadata.
 * @param {boolean} isMacOS - A flag indicating whether the platform is macOS.
 * @param {string} filePath - The full path of the file being edited.
 */
function initializeEditors(
  initialYamlContent,
  baseYamlContent,
  headYamlContent,
  resolvedYamlContent,
  docsMap,
  isLightMode,
  fileType,
  fileExtension,
  isFileEditable,
  highlightedErrorLineNumber,
  keyToTreeMetaData,
  isMacOS,
  filePath
) {
  updateWebviewTheme(isLightMode);

  require.config({
    paths: {
      vs: "https://cdnjs.cloudflare.com/ajax/libs/monaco-editor/0.34.0/min/vs",
    },
  });
  require(["vs/editor/editor.main"], function () {
    setCustomTheme(isLightMode);
    documentationMap = docsMap;
    isMac = isMacOS;
    fileLanguage = inferLanguageFromExtension(fileExtension);

    // Decode the content from base64 to string
    let initialYamlContentDecoded;
    let baseYamlContentDecoded;
    let headYamlContentDecoded;
    let resolvedYamlContentDecoded;
    try {
      initialYamlContentDecoded = atob(initialYamlContent);
      baseYamlContentDecoded = atob(baseYamlContent);
      headYamlContentDecoded = atob(headYamlContent);
      resolvedYamlContentDecoded = atob(resolvedYamlContent);
    } catch (error) {
      console.error(`Error decoding file content: ${error}`);
      return;
    }

    // Initialize the appropriate view based on fileType
    switch (fileType) {
      case "merge":
        initializeMergeView(
          initialYamlContentDecoded,
          baseYamlContentDecoded,
          headYamlContentDecoded,
          resolvedYamlContentDecoded,
          isFileEditable
        );

        // Set up merge view specific features
        updateConflictPositions();
        addResolvedEditorChangeListener();
        applyConflictDecorations(resolvedYamlEditor, resolvedYamlModel);
        monitorModelChanges(resolvedYamlEditor, resolvedYamlModel);
        addClickableKeyDecorations(keyToTreeMetaData);
        updateDocsTooltips(documentationMap);
        syncScrolling(baseYamlEditor, headYamlEditor, resolvedYamlEditor);
        addAcceptChangeCommands();
        addConflictHighlighting();
        if (highlightedErrorLineNumber !== 0) {
          highlightErrorLine(resolvedYamlEditor, highlightedErrorLineNumber);
        }
        resolvedYamlEditor.focus();
        break;

      case "commit":
        initializeCommitView(baseYamlContentDecoded, headYamlContentDecoded);
        addClickableKeyDecorations(keyToTreeMetaData);
        updateDocsTooltips(documentationMap);
        break;

      case "single":
        initializeSingleView(filePath, resolvedYamlContentDecoded);
        addClickableKeyDecorations(keyToTreeMetaData);
        updateDocsTooltips(documentationMap);
        break;

      default:
        console.error("Invalid file type:", fileType);
        return;
    }

    showView(fileType);
  });
}

// Modified window resize handler
window.addEventListener("resize", () => {
  switch (currentView) {
    case "merge":
      if (baseYamlEditor && headYamlEditor && resolvedYamlEditor) {
        baseYamlEditor.layout();
        headYamlEditor.layout();
        resolvedYamlEditor.layout();
      }
      break;
    case "commit":
      if (commitDiffEditor) {
        commitDiffEditor.layout();
      }
      break;
    case "single":
      if (singleEditor) {
        singleEditor.layout();
      }
      break;
  }
});

// Add scroll sync for diff view
function setupDiffScrollSync() {
  let isScrolling = false;

  const syncScroll = (sourceEditor, targetEditor, e) => {
    if (isScrolling) return;
    isScrolling = true;

    const targetModified = targetEditor.getModifiedEditor();
    targetModified.setScrollPosition({
      scrollTop: e.scrollTop,
      scrollLeft: e.scrollLeft,
    });

    isScrolling = false;
  };

  diffBaseEditor.getModifiedEditor().onDidScrollChange((e) => {
    syncScroll(diffBaseEditor, diffHeadEditor, e);
  });

  diffHeadEditor.getModifiedEditor().onDidScrollChange((e) => {
    syncScroll(diffHeadEditor, diffBaseEditor, e);
  });
}

function initializeCommitView(baseContent, headContent) {
  // Hide all merge/diff related containers
  hideElement("base-container");
  hideElement("head-container");
  hideElement("divider");
  hideElement("horizontal-divider");

  // Create diff editors using the correct containers
  commitDiffEditor = monaco.editor.createDiffEditor(
    document.getElementById("commit-diff-container"),
    {
      readOnly: true,
      renderSideBySide: true,
      enableSplitViewResizing: false,
      hover: { above: false },
      unusualLineTerminators: "off",
    }
  );
  const baseModel = monaco.editor.createModel(baseContent, fileLanguage);
  const headModel = monaco.editor.createModel(headContent, fileLanguage);

  commitDiffEditor.setModel({
    original: baseModel,
    modified: headModel,
  });

  // Ensure focus is applied after rendering is complete
  setTimeout(() => {
    if (commitDiffEditor) {
      commitDiffEditor.focus();
    }
  }, 100); // Delay to allow rendering
}

function initializeSingleView(fileKey, content) {
  clearAllFileCache();
  // Hide all merge/diff related containers
  hideElement("base-container");
  hideElement("head-container");
  hideElement("divider");
  hideElement("horizontal-divider");

  // Show and setup single editor container
  const container = ensureContainerExists("single-container");
  container.style.width = "100%";
  container.style.height = "100%";

  // Create the editor to render the file content
  singleEditor = monaco.editor.create(container, {
    value: content,
    language: fileLanguage,
    lineNumbers: "on",
    folding: true,
    hover: { above: false },
    unusualLineTerminators: "off",
  });

  // Create a new model for the file content
  resolvedYamlModel = monaco.editor.createModel(
    content,
    fileLanguage,
    monaco.Uri.parse(fileKey)
  );

  // Cache the model for the file path
  modelCache.set(fileKey, resolvedYamlModel);

  // Load the model in the editor
  singleEditor.setModel(resolvedYamlModel);

  singleEditor.focus();

  addResolvedEditorChangeListener();

  currentFileKey = fileKey;

  singleEditor.onDidFocusEditorWidget(() => {
    focusSingleEditor();
  });

  registerSaveShortcut();
}

function registerSaveShortcut() {
  if (!singleEditor) return;

  // Detects Cmd+S (Mac) or Ctrl+S (Windows/Linux)
  singleEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyS, () => {
    const content = resolvedYamlModel.getValue();

    if (isWeb()) {
      // Notify Flutter (Web)
      onSingleEditorSave(content);
    } else if (window.flutter_inappwebview) {
      // Notify Flutter (Desktop)
      window.flutter_inappwebview.callHandler("saveFile", content);
    }
  });
  // Detects Cmd+Z (Mac) or Ctrl+Z (Windows/Linux)
  singleEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyZ, () => {
    let canUndo = resolvedYamlModel.canUndo();
    if (canUndo) {
      resolvedYamlModel.undo();
    } else {
      if (isWeb()) {
        // Notify Flutter (Web)
        maybeResetFile(true);
      } else if (window.flutter_inappwebview) {
        // Notify Flutter (Desktop)
        window.flutter_inappwebview.callHandler("maybeResetFile");
      }
    }
  });
}

/// Makes sure any other focus is removed from the Flutter
function focusSingleEditor() {
  if (isWeb()) {
    // Notify Flutter (Web)
    onSingleEditorFocusChanged(true);
  }

  // On desktop webview, this is handled by the `onWindowFocus` event.
}

/**
 * Switches the file being edited in the single editor view.
 *
 * @param {string} fileKey - The unique key of the file being edited.
 * @param {string} fileContent - The content of the file being edited. Only used if the file is not cached.
 * @param {string} fileExtension - The extension of the file being edited.
 * @param {Object} documentationMap - The documentation map for the file being edited.
 * @param {Object} keyToTreeMetaData - The key to tree metadata for the file being edited.
 */
function switchFile(
  fileKey,
  fileContent,
  fileExtension,
  documentationMap,
  keyToTreeMetaData
) {
  // Save caret position for the previous file
  if (currentFileKey) saveCaretPosition(currentFileKey);

  // Make sure the correct language is set (for proper syntax highlighting)
  fileLanguage = inferLanguageFromExtension(fileExtension);

  // Check if the file is already cached, if so, switch to that model, otherwise
  // create a new model.
  if (modelCache.has(fileKey)) {
    resolvedYamlModel = modelCache.get(fileKey);
    singleEditor.setModel(resolvedYamlModel);
  } else {
    // Decode the new file content from base64 to string
    let fileContentDecoded;
    try {
      fileContentDecoded = atob(fileContent);
    } catch (error) {
      console.error(`Error decoding file content: ${error}`);
      return;
    }
    // Create a new model for the file content
    resolvedYamlModel = monaco.editor.createModel(
      fileContentDecoded,
      fileLanguage,
      monaco.Uri.parse(fileKey)
    );
    // Add the new model to the cache
    modelCache.set(fileKey, resolvedYamlModel);
    // Load the new model in the editor
    singleEditor.setModel(resolvedYamlModel);
    // Make sure to listen for changes in the new model
    addResolvedEditorChangeListener();
  }

  // Restore caret position if available
  restoreCaretPosition(fileKey);

  currentFileKey = fileKey;

  // Update the docs tooltips for the new file
  updateDocsTooltips(documentationMap);

  // Update the clickable key decorations for the new file
  addClickableKeyDecorations(keyToTreeMetaData);
}

function saveCaretPosition(fileKey) {
  // Only store if it was focused
  if (!singleEditor) return;

  const position = singleEditor.getPosition();
  if (position) {
    caretPositionCache.set(fileKey, position);
  }
}

function restoreCaretPosition(fileKey) {
  if (!singleEditor) return;

  const position = caretPositionCache.get(fileKey);
  if (position) {
    requestAnimationFrame(() => {
      // Set the caret position
      singleEditor.setPosition(position);
      // Ensure it's visible
      singleEditor.revealLineInCenter(position.lineNumber);
      // Focus the editor otherwise the caret won't be visible
      singleEditor.focus();
    });
  }
}

/// Clears the cached model for the specified file key.
function clearFileCache(fileKey) {
  modelCache.delete(fileKey);
  caretPositionCache.delete(fileKey);
}

/// Clears all cached models from the model cache.
function clearAllFileCache() {
  modelCache.clear();
  caretPositionCache.clear();
}

function highlightSearchedLine(lineNumber) {
  if (!singleEditor) return;

  scrollToAndHighlightLine(
    singleEditor,
    lineNumber,
    "line-highlight-decoration"
  );
}

function highlightSingleEditorErrorLine(lineNumber) {
  if (!singleEditor) return;

  highlightErrorLine(singleEditor, lineNumber);
}

function highlightErrorLine(editor, lineNumber) {
  if (!editor) return;

  scrollToAndHighlightLine(
    editor,
    lineNumber,
    "error-line-highlight-decoration"
  );
}

function updateFileCache(fileKey, fileContent) {
  if (modelCache.has(fileKey)) {
    // Decode the new file content from base64 to string
    let fileContentDecoded;
    try {
      fileContentDecoded = atob(fileContent);
    } catch (error) {
      console.error(`Error decoding file content: ${error}`);
      return;
    }
    let model = modelCache.get(fileKey);
    const fullRange = model.getFullModelRange();
    model.pushEditOperations(
      [],
      [{ range: fullRange, text: fileContentDecoded }],
      () => null
    );
  }
}

// Utility functions for DOM manipulation
function ensureContainerExists(id) {
  let container = document.getElementById(id);
  if (!container) {
    container = document.createElement("div");
    container.id = id;
    container.className = "editor-pane";
    document.getElementById("top-editors-container").appendChild(container);
  }
  container.style.display = "block";
  return container;
}

function ensureDividerExists(id) {
  let divider = document.getElementById(id);
  if (!divider) {
    divider = document.createElement("div");
    divider.id = id;
    divider.className = "divider";
    document.getElementById("top-editors-container").appendChild(divider);
  }
  divider.style.display = "block";
  return divider;
}

function hideElement(id) {
  const element = document.getElementById(id);
  if (element) {
    element.style.display = "none";
  }
}

// Function to find all conflict markers and store their positions
function updateConflictPositions() {
  conflictPositions = []; // Reset conflicts
  const model = resolvedYamlEditor.getModel();
  const lines = model.getLinesContent();

  lines.forEach((lineContent, lineNumber) => {
    if (lineContent.startsWith("<<<<<<<")) {
      conflictPositions.push(lineNumber + 1); // Store line number (1-based)
    }
  });
}

// Scroll to a specific conflict line
function scrollToConflict(index) {
  if (conflictPositions.length === 0) return;

  const lineNumber = conflictPositions[index];
  resolvedYamlEditor.revealLineInCenter(lineNumber);

  // Highlight the conflict line briefly
  const highlightOptions = {
    range: new monaco.Range(lineNumber, 1, lineNumber, 1),
    options: {
      isWholeLine: true,
      className: "line-highlight-decoration",
    },
  };
  blinkHighlight(resolvedYamlEditor, highlightOptions);
}

// Navigate to the next conflict
function goToNextConflict() {
  if (conflictPositions.length === 0) return;

  currentConflictIndex = (currentConflictIndex + 1) % conflictPositions.length;
  scrollToConflict(currentConflictIndex);
}

// Navigate to the previous conflict
function goToPreviousConflict() {
  if (conflictPositions.length === 0) return;

  currentConflictIndex =
    (currentConflictIndex - 1 + conflictPositions.length) %
    conflictPositions.length;
  scrollToConflict(currentConflictIndex);
}

// Function to listen for changes in the resolved editor
function addResolvedEditorChangeListener() {
  resolvedYamlModel.onDidChangeContent(() => {
    if (currentView === "merge") {
      updateConflictPositions();
    }

    const content = resolvedYamlModel.getValue();

    if (isWeb()) {
      // Notify Flutter (Web)
      onResolvedDartCallback(content);
    } else if (window.flutter_inappwebview) {
      // Notify Flutter (Desktop)
      window.flutter_inappwebview.callHandler("resolvedEditorChanged", content);
    } else {
      console.error("No valid platform found to send resolved content to.");
    }
  });
}

// Utility function to detect if the platform is web
function isWeb() {
  return typeof window.flutter_inappwebview === "undefined";
}

// Apply FF theme to the Monaco Editor
function setCustomTheme(isLightMode) {
  const themeName = isLightMode ? "custom-light-theme" : "custom-dark-theme";

  const themeConfig = isLightMode
    ? {
        base: "vs", // Light theme base
        inherit: true,
        rules: [{ token: "", background: "FFFFFF" }],
        colors: {
          "editor.background": "#FFFFFF", // Editor background color
          "editor.lineHighlightBackground": "#F1F4F8",
          "editor.selectionBackground": "#4B39EF4C",
          "editor.selectionHighlightBackground": "#4B39EF4C",
          "diffEditor.insertedLineBackground": "#ceefeb", // Light Green
          "diffEditor.insertedTextBackground": "#aae3dc",
          "diffEditor.removedLineBackground": "#f6d0d0", // Light Red
          "diffEditor.removedTextBackground": "#efadad",
          "editorOverviewRuler.addedForeground": "#ceefeb", // Green in minimap
          "editorOverviewRuler.deletedForeground": "#f6d0d0", // Red in minimap
        },
      }
    : {
        base: "vs-dark", // Dark theme base
        inherit: true,
        rules: [{ token: "", background: "14181B" }],
        colors: {
          "editor.background": "#14181B", // Editor background color
          "editor.lineHighlightBackground": "#1D2428",
          "editor.selectionBackground": "#4B39EF4C",
          "editor.selectionHighlightBackground": "#4B39EF4C",
          "diffEditor.insertedLineBackground": "#264642", // Light Green
          "diffEditor.insertedTextBackground": "#26615c",
          "diffEditor.removedLineBackground": "#4e2827", // Light Red
          "diffEditor.removedTextBackground": "#6c2b2d",
          "editorOverviewRuler.addedForeground": "#264642", // Green in minimap
          "editorOverviewRuler.deletedForeground": "#4e2827", // Red in minimap
        },
      };

  // Define the custom theme
  monaco.editor.defineTheme(themeName, themeConfig);

  // Apply the custom theme
  monaco.editor.setTheme(themeName);
}

/**
 * Initializes the Monaco Editor merge view with the provided YAML content.
 *
 * @param {string} initialYamlContent - The initial YAML content to be used as the base for comparison.
 * @param {string} baseYamlContent - The base YAML content to compare against the initial content.
 * @param {string} headYamlContent - The head YAML content to compare against the initial content.
 * @param {string} resolvedYamlContent - The resolved YAML content to be displayed in the resolved editor.
 */
function initializeMergeView(
  initialYamlContent,
  baseYamlContent,
  headYamlContent,
  resolvedYamlContent,
  isFileEditable
) {
  const initialYamlModel = monaco.editor.createModel(
    initialYamlContent,
    fileLanguage
  );
  const baseYamlModel = monaco.editor.createModel(
    baseYamlContent,
    fileLanguage
  );
  const headYamlModel = monaco.editor.createModel(
    headYamlContent,
    fileLanguage
  );
  resolvedYamlModel = monaco.editor.createModel(
    resolvedYamlContent,
    fileLanguage
  );

  // Create Monaco Editors
  baseYamlEditor = monaco.editor.createDiffEditor(
    document.getElementById("base-container"),
    {
      readOnly: true,
      renderSideBySide: false,
      unusualLineTerminators: "off",
      hover: { above: false },
    }
  );
  headYamlEditor = monaco.editor.createDiffEditor(
    document.getElementById("head-container"),
    {
      readOnly: true,
      renderSideBySide: false,
      unusualLineTerminators: "off",
      hover: { above: false },
    }
  );
  resolvedYamlEditor = monaco.editor.create(
    document.getElementById("resolved-container"),
    {
      language: fileLanguage,
      lineNumbers: "on",
      folding: true,
      codeLens: true,
      readOnly: !isFileEditable,
      hover: { above: false },
      unusualLineTerminators: "off",
    }
  );

  // Set models for the editors
  baseYamlEditor.setModel({
    original: initialYamlModel,
    modified: baseYamlModel,
  });
  headYamlEditor.setModel({
    original: initialYamlModel,
    modified: headYamlModel,
  });
  resolvedYamlEditor.setModel(resolvedYamlModel);

  // Ensure focus is applied after rendering is complete
  setTimeout(() => {
    if (resolvedYamlEditor) {
      resolvedYamlEditor.focus();
    }
  }, 100); // Delay to allow rendering
}

/**
 * Registers a CodeLens provider to add conflict highlighting with dynamic branch names.
 */
function addConflictHighlighting() {
  monaco.languages.registerCodeLensProvider(fileLanguage, {
    provideCodeLenses: (model) => {
      const matches = model.findMatches(
        "^<<<<<<< .*|^=======|^>>>>>>> .*",
        false,
        true,
        false,
        null,
        true
      );

      const lenses = [];
      let currentConflictStart = null;
      let currentConflictEnd = null;

      for (let i = 0; i < matches.length; i++) {
        const match = matches[i];
        const line = match.range.startLineNumber;
        const lineContent = model.getLineContent(line);

        if (lineContent.startsWith("<<<<<<<")) {
          currentConflictStart = line;
          const currentBranch = lineContent.replace("<<<<<<< ", "").trim();

          // Find the corresponding end marker
          let j = i + 1;
          while (j < matches.length) {
            const endMatch = matches[j];
            const endLine = endMatch.range.startLineNumber;
            const endContent = model.getLineContent(endLine);

            if (endContent.startsWith(">>>>>>>")) {
              currentConflictEnd = endLine;
              const incomingBranch = endContent.replace(">>>>>>> ", "").trim();

              // Add CodeLens for this specific conflict
              lenses.push(
                {
                  range: {
                    startLineNumber: line,
                    endLineNumber: line,
                    startColumn: 1,
                    endColumn: 1,
                  },
                  command: {
                    id: "merge.accept.current",
                    title: `Accept change in ${currentBranch}`,
                    arguments: [
                      {
                        startLine: currentConflictStart,
                        endLine: currentConflictEnd,
                      },
                    ],
                  },
                },
                {
                  range: {
                    startLineNumber: line,
                    endLineNumber: line,
                    startColumn: 1,
                    endColumn: 1,
                  },
                  command: {
                    id: "merge.accept.incoming",
                    title: `Accept change in ${incomingBranch}`,
                    arguments: [
                      {
                        startLine: currentConflictStart,
                        endLine: currentConflictEnd,
                      },
                    ],
                  },
                }
              );
              break;
            }
            j++;
          }
        }
      }

      return { lenses, dispose: () => {} };
    },
  });
}

/**
 * Notifies Flutter about divider position changes
 * @param {Object} positions - Object containing vertical and/or horizontal divider positions as percentages
 */
function notifyDividerPositions(positions) {
  if (isWeb()) {
    // Notify Flutter (Web)
    window[DIVIDER_POSITION_HANDLER](JSON.stringify(positions));
  } else {
    // Notify Flutter (Desktop)
    window.flutter_inappwebview.callHandler(
      DIVIDER_POSITION_HANDLER,
      JSON.stringify(positions)
    );
  }
}

// Add Resizable Divider Handlers
document.addEventListener("DOMContentLoaded", function () {
  const verticalDivider = document.getElementById("divider");
  const horizontalDivider = document.getElementById("horizontal-divider");
  let isDraggingVertical = false;
  let isDraggingHorizontal = false;

  // Function to start dragging the divider
  function startDrag(e, type) {
    e.preventDefault();
    if (type === "vertical") {
      isDraggingVertical = true;
      document.body.style.cursor = "ew-resize";
    } else if (type === "horizontal") {
      isDraggingHorizontal = true;
      document.body.style.cursor = "ns-resize";
    }
    document.body.classList.add("disable-selection");
  }

  // Stop dragging when the mouse is released
  document.addEventListener("mouseup", () => {
    if (isDraggingVertical || isDraggingHorizontal) {
      isDraggingVertical = false;
      isDraggingHorizontal = false;
      document.body.style.cursor = "default";
      document.body.classList.remove("disable-selection");
    }
  });

  // Update the divider position while dragging
  document.addEventListener("mousemove", (e) => {
    const positions = {};

    if (isDraggingVertical) {
      const container = document.getElementById("top-editors-container");
      const leftPane = document.getElementById("base-container");
      const rightPane = document.getElementById("head-container");
      const containerRect = container.getBoundingClientRect();
      const offsetX = e.clientX - containerRect.left;

      if (offsetX > 50 && offsetX < containerRect.width - 50) {
        const leftWidth = offsetX; // New width for the left pane
        const rightWidth = containerRect.width - offsetX - 5; // New width for the right pane, accounting for divider

        // Update left and right pane sizes
        leftPane.style.width = `${leftWidth}px`;
        rightPane.style.width = `${rightWidth}px`;

        // Calculate vertical divider position as percentage
        positions.vertical = (offsetX / containerRect.width) * 100;

        // Call layout to adjust Monaco editors to their new sizes
        baseYamlEditor.layout();
        headYamlEditor.layout();
      }
    }

    // Update the divider position for horizontal resizing
    if (isDraggingHorizontal) {
      const topContainer = document.getElementById("top-editors-container");
      const bottomContainer = document.getElementById(
        "bottom-editor-container"
      );
      const containerRect = topContainer.parentElement.getBoundingClientRect();
      const offsetY = e.clientY - containerRect.top;

      if (offsetY > 50 && offsetY < containerRect.height - 50) {
        const topHeight = (offsetY / containerRect.height) * 100;
        const bottomHeight = 100 - topHeight;

        topContainer.style.height = `${topHeight}%`;
        bottomContainer.style.height = `${bottomHeight}%`;

        // Calculate horizontal divider position as percentage
        positions.horizontal = topHeight;

        baseYamlEditor.layout();
        headYamlEditor.layout();
        resolvedYamlEditor.layout();
      }
    }

    // Only notify if positions changed
    if (Object.keys(positions).length > 0) {
      notifyDividerPositions(positions);
    }
  });

  if (verticalDivider) {
    verticalDivider.addEventListener("mousedown", (e) =>
      startDrag(e, "vertical")
    );
  }
  if (horizontalDivider) {
    horizontalDivider.addEventListener("mousedown", (e) =>
      startDrag(e, "horizontal")
    );
  }

  // Send initial positions
  const container = document.getElementById("top-editors-container");
  const containerRect = container.getBoundingClientRect();
  const initialPositions = {
    vertical: 50, // Default split is 50%
    horizontal: 50, // Default split is 50%
  };
  notifyDividerPositions(initialPositions);
});

// Add Clickable Key Decorations to all editors
function addClickableKeyDecorations(keyToTreeMetaData) {
  const editors = [
    commitDiffEditor?.getOriginalEditor(),
    commitDiffEditor?.getModifiedEditor(),
    baseYamlEditor?.getOriginalEditor(),
    baseYamlEditor?.getModifiedEditor(),
    headYamlEditor?.getOriginalEditor(),
    headYamlEditor?.getModifiedEditor(),
    resolvedYamlEditor,
    singleEditor,
  ];

  const keyboardModifierKey = isMac ? "⌘" : "Ctrl";

  editors.forEach((editor) => {
    if (!editor) {
      return;
    }

    try {
      const model = editor?.getModel();
      if (!model) {
        return;
      }

      // For single editor, remove existing clickable key decorations and event listeners first
      if (editor === singleEditor) {
        // Remove existing decorations
        const existingDecorations = editor
          .getDecorationsInRange(
            new monaco.Range(1, 1, model.getLineCount(), 1)
          )
          .filter(
            (decoration) =>
              decoration.options.inlineClassName === "clickable-key-decoration"
          );
        if (existingDecorations.length > 0) {
          editor.deltaDecorations(
            existingDecorations.map((d) => d.id),
            []
          );
        }

        // Remove existing event listener if any
        const disposable = mouseDownDisposables.get(model.uri.path);
        if (disposable) {
          disposable.dispose();
          mouseDownDisposables.delete(model.uri.path);
        }
      }

      const decorations = [];
      const matchesMap = {};

      // Regular expression to match `key: value`
      const keyPattern = /key:\s*(\w+)/g;
      const lines = model.getLinesContent();

      lines.forEach((lineContent, lineNumber) => {
        let match;

        while ((match = keyPattern.exec(lineContent)) !== null) {
          const startColumn = match.index + match[0].indexOf(match[1]) + 1;
          const endColumn = startColumn + match[1].length - 1;
          const matchedValue = match[1];

          // Check if matchedValue exists in keyToTreeMetaData
          const nodeKey = `id-${matchedValue}`;
          if (!keyToTreeMetaData[nodeKey]) continue;

          // Use 1-index for the line numbers
          const range = new monaco.Range(
            lineNumber + 1,
            startColumn,
            lineNumber + 1,
            endColumn + 1
          );

          const nodeMetaData = keyToTreeMetaData[nodeKey];
          const nodeName = nodeMetaData.name || matchedValue;

          decorations.push({
            range,
            options: {
              inlineClassName: "clickable-key-decoration",
              hoverMessage: {
                value: `**${nodeName}** (${keyboardModifierKey} + Click to go to the definition)`,
              },
            },
          });

          matchesMap[rangeToString(range)] = {
            type: "key",
            value: matchedValue,
          };
        }
      });

      const decorationIds = editor.deltaDecorations([], decorations);

      // Ctrl/Cmd key detection for cursor style
      document.addEventListener("keydown", (e) => {
        if (e.ctrlKey || e.metaKey) {
          document.body.classList.add("ctrl-hover");
        }
      });

      document.addEventListener("keyup", (e) => {
        if (!e.ctrlKey && !e.metaKey) {
          document.body.classList.remove("ctrl-hover");
        }
      });

      // Handle click events, ensuring Ctrl/Cmd key is pressed
      const disposable = editor.onMouseDown((event) => {
        if (
          event.target.element &&
          event.target.type === monaco.editor.MouseTargetType.CONTENT_TEXT &&
          (event.event.ctrlKey || event.event.metaKey) // Ensure Ctrl/Cmd is pressed
        ) {
          const position = event.target.position;
          const clickedRange = new monaco.Range(
            position.lineNumber,
            position.column,
            position.lineNumber,
            position.column
          );

          for (const [rangeKey, match] of Object.entries(matchesMap)) {
            if (rangesIntersect(stringToRange(rangeKey), clickedRange)) {
              const jsonPayload = JSON.stringify({
                type: match.type,
                value: match.value,
              });

              if (isWeb()) {
                // Notify Flutter (Web)
                onLinkClickDartCallback(jsonPayload);
              } else {
                // Notify Flutter (Desktop)
                window.flutter_inappwebview.callHandler(
                  "linkClicked",
                  jsonPayload
                );
              }
              break;
            }
          }
        }
      });

      // Store the disposable for the single editor
      if (editor === singleEditor) {
        mouseDownDisposables.set(model.uri.path, disposable);
      }
    } catch (error) {
      console.error(`Error in addClickableKeyDecorations: ${error}`);
    }
  });
}

// Utility functions for range conversion and intersection
function rangeToString(range) {
  return `${range.startLineNumber}:${range.startColumn}-${range.endLineNumber}:${range.endColumn}`;
}

function stringToRange(rangeStr) {
  const [start, end] = rangeStr.split("-");
  const [startLine, startColumn] = start.split(":").map(Number);
  const [endLine, endColumn] = end.split(":").map(Number);
  return new monaco.Range(startLine, startColumn, endLine, endColumn);
}

function rangesIntersect(range1, range2) {
  return (
    range1.startLineNumber === range2.startLineNumber &&
    range1.endLineNumber === range2.endLineNumber &&
    range1.startColumn <= range2.endColumn &&
    range1.endColumn >= range2.startColumn
  );
}

/**
 * Monitors changes in the content of the given model and checks for conflict markers.
 * If conflict markers are found, it reapplies conflict decorations.
 * If no conflict markers are present, it clears the conflict highlights.
 *
 * @param {object} editor - The Monaco editor instance.
 * @param {object} model - The Monaco editor model whose content is being monitored.
 */
function monitorModelChanges(editor, model) {
  model.onDidChangeContent(() => {
    const content = model.getValue();

    // Update documentation tooltips
    updateDocsTooltips(documentationMap);

    // Check for conflict markers
    const hasConflicts =
      content.includes("<<<<<<<") &&
      content.includes("=======") &&
      content.includes(">>>>>>>");

    if (hasConflicts) {
      // Reapply conflict decorations
      applyConflictDecorations(editor, model);
    } else {
      // Clear decorations if no conflicts are present
      clearConflictHighlights();
    }
  });
}

/**
 * Clears all conflict-related highlights in the Monaco editor.
 *
 * This function removes all decorations related to conflicts by updating the
 * `conflictDecorations` array with an empty array, effectively clearing any
 * visual indicators of conflicts in the editor.
 */
function clearConflictHighlights() {
  // Remove all conflict-related decorations
  conflictDecorations = resolvedYamlEditor.deltaDecorations(
    conflictDecorations,
    []
  );
}

/**
 * Applies conflict decorations to a Monaco editor instance based on the conflict
 * markers in the model.
 *
 * This function scans through the lines of the model to identify conflict markers
 * (<<<<<<<, =======, >>>>>>>) and applies decorations to highlight the current
 * change, the divider, and the incoming change.
 *
 * @param {monaco.editor.IStandaloneCodeEditor} editor - The Monaco editor instance
 * to apply decorations to.
 * @param {monaco.editor.ITextModel} model - The text model of the editor containing
 * the content to be scanned for conflict markers.
 */
function applyConflictDecorations(editor, model) {
  const decorations = [];
  const lines = model.getLinesContent();
  let inCurrentChange = false;
  let inIncomingChange = false;
  let startLine = 0;

  lines.forEach((lineContent, lineNumber) => {
    const lineIdx = lineNumber + 1;

    if (lineContent.startsWith("<<<<<<<")) {
      inCurrentChange = true;
      startLine = lineIdx;
      decorations.push({
        range: new monaco.Range(lineIdx, 1, lineIdx, 1),
        options: { isWholeLine: true, className: "currentLineDecoration" },
      });
    } else if (lineContent.startsWith("=======") && inCurrentChange) {
      decorations.push({
        range: new monaco.Range(startLine, 1, lineIdx, 1),
        options: { isWholeLine: true, className: "currentLineDecoration" },
      });
      inCurrentChange = false;
      inIncomingChange = true;
      startLine = lineIdx;
      decorations.push({
        range: new monaco.Range(lineIdx, 1, lineIdx, 1),
        options: { isWholeLine: true, className: "dividerLineDecoration" },
      });
    } else if (lineContent.startsWith(">>>>>>>") && inIncomingChange) {
      decorations.push({
        range: new monaco.Range(startLine, 1, lineIdx, 1),
        options: { isWholeLine: true, className: "incomingLineDecoration" },
      });
      inIncomingChange = false;
    }
  });

  // Apply decorations and store their identifiers
  conflictDecorations = editor.deltaDecorations(
    conflictDecorations,
    decorations
  );
}

/**
 * Synchronizes the scrolling of three Monaco Editor instances.
 *
 * @param {Object} baseEditor - The first Monaco Editor instance.
 * @param {Object} headEditor - The second Monaco Editor instance.
 * @param {Object} resolvedEditor - The third Monaco Editor instance.
 */
function syncScrolling(baseEditor, headEditor, resolvedEditor) {
  const editors = [baseEditor, headEditor, resolvedEditor];

  editors.forEach((sourceEditor, index) => {
    const sourceInstance = sourceEditor.getModifiedEditor
      ? sourceEditor.getModifiedEditor()
      : sourceEditor;

    sourceInstance.onDidScrollChange((e) => {
      if (isSyncingScroll) return;
      isSyncingScroll = true;

      editors.forEach((targetEditor, targetIndex) => {
        if (index !== targetIndex) {
          const targetInstance = targetEditor.getModifiedEditor
            ? targetEditor.getModifiedEditor()
            : targetEditor;
          targetInstance.setScrollTop(e.scrollTop);
        }
      });

      isSyncingScroll = false;
    });
  });
}

/**
 * Registers commands for accepting changes in the Monaco editor.
 *
 * This function registers two commands:
 * - "merge.accept.current": Accepts the current change.
 * - "merge.accept.incoming": Accepts the incoming change.
 *
 * These commands are used in the context of merging changes in the editor.
 */
function addAcceptChangeCommands() {
  monaco.editor.registerCommand("merge.accept.current", (_, range) => {
    resolveConflict("current", range);
  });

  monaco.editor.registerCommand("merge.accept.incoming", (_, range) => {
    resolveConflict("incoming", range);
  });
}

/**
 * Resolves a conflict in a YAML model based on the specified type.
 *
 * @param {string} type - The type of resolution to apply ("current" or "incoming").
 * @param {Object} range - The range of lines where the conflict is located.
 * @param {number} range.startLine - The starting line number of the conflict.
 * @param {number} range.endLine - The ending line number of the conflict.
 */
function resolveConflict(type, range) {
  const content = resolvedYamlModel.getValue();
  const lines = content.split("\n");

  // Get the specific conflict content
  const conflictLines = lines.slice(range.startLine - 1, range.endLine);
  const conflictContent = conflictLines.join("\n");

  // Find the conflict parts using regex
  const conflictRegex =
    /^( *)(<<<<<<< .*?\n)([\s\S]*?)^( *)=======\n([\s\S]*?)^( *)>>>>>>> .*$/m;
  const match = conflictContent.match(conflictRegex);

  if (match) {
    const outerIndentation = match[1]; // Leading spaces before the conflict
    const current = match[3].trimEnd(); // Current branch content
    const incoming = match[5].trimEnd(); // Incoming branch content

    // Resolve based on type and preserve indentation
    const resolved =
      type === "current"
        ? `${outerIndentation}${current}`
        : `${outerIndentation}${incoming}`;

    // Create the edit operation for this specific conflict
    const edit = {
      range: new monaco.Range(
        range.startLine,
        1,
        range.endLine,
        lines[range.endLine - 1].length + 1
      ),
      text: resolved,
    };

    // Apply the edit
    resolvedYamlModel.pushEditOperations([], [edit], () => null);

    // Reapply decorations to handle remaining conflicts
    applyConflictDecorations(resolvedYamlEditor, resolvedYamlModel);
  }
}

/**
 * Updates the webview theme based on the provided mode.
 *
 * This function changes the background color of the body and the document element
 * based on whether the light mode is enabled or not. It also updates the colors
 * of the dividers with the appropriate color for the selected mode.
 *
 * @param {boolean} isLightMode - A boolean indicating if the light mode is enabled.
 * If true, the light mode is enabled; otherwise, the dark mode is enabled.
 */
function updateWebviewTheme(isLightMode) {
  const backgroundColor = isLightMode ? "#FFFFFF" : "#14181B";
  document.body.style.backgroundColor = backgroundColor;
  document.documentElement.style.backgroundColor = backgroundColor;

  const dividerColor = isLightMode ? "#E0E3E7" : "#323B45";

  // Update all possible dividers
  const dividers = ["divider", "horizontal-divider", "diff-divider"];
  dividers.forEach((id) => {
    const element = document.getElementById(id);
    if (element) {
      element.style.backgroundColor = dividerColor;
    }
  });
}

/**
 * Updates error decorations in the resolved editor.
 * @param {Array} errors - Array of error objects with line, startColumn, endColumn,
 * and message.
 */
function updateErrorDecorations(errors) {
  // Parse JSON string if running in web (because the
  let jsonFromString = isWeb() ? JSON.parse(errors) : errors;
  // Clear existing decorations
  errorDecorations = resolvedYamlEditor.deltaDecorations(errorDecorations, []);

  // Create new decorations
  const decorations = jsonFromString.map((error) => ({
    range: new monaco.Range(
      error.line,
      error.startColumn,
      error.line,
      error.endColumn
    ),
    options: {
      inlineClassName: "squiggly-error", // Use squiggly line styling
      hoverMessage: [{ value: `**Error**: ${error.message}` }], // Tooltip message
      overviewRuler: {
        color: "rgba(255, 0, 0, 0.8)", // Color for the minimap
        position: monaco.editor.OverviewRulerLane.Right,
      },
    },
  }));

  // Apply new decorations
  errorDecorations = resolvedYamlEditor.deltaDecorations(
    errorDecorations,
    decorations
  );
}

/**
 * Updates the tooltips for documentation in the Monaco editors.
 *
 * This function takes a JSON object containing documentation ranges for
 * base, head, and resolved YAML editors, and updates the tooltips in the
 * respective Monaco editors.
 *
 * @param {Object} docsJson - The JSON object containing documentation ranges.
 * @param {Array} docsJson.base - Array of documentation ranges for the base editor.
 * @param {Array} docsJson.head - Array of documentation ranges for the head editor.
 * @param {Array} docsJson.resolved - Array of documentation ranges for the resolved editor.
 */
function updateDocsTooltips(docsJson) {
  // Helper function to create decorations from doc ranges
  const createDecorations = (docs) =>
    docs.map((doc) => ({
      range: new monaco.Range(
        doc.range.lineNumber,
        doc.range.startColumnNumber,
        doc.range.lineNumber,
        doc.range.endColumnNumber
      ),
      options: {
        hoverMessage: [{ value: doc.documentation }],
      },
    }));

  // General function to update editor doc decorations
  const updateEditorDocDecorations = (
    editor,
    docData,
    currentDocDecorations
  ) => {
    if (!editor || !docData) return [];
    const modifiedEditor = editor.getModifiedEditor
      ? editor.getModifiedEditor()
      : editor;
    return modifiedEditor.deltaDecorations(
      currentDocDecorations,
      createDecorations(docData)
    );
  };

  // Update all relevant editors dynamically
  if (commitDiffEditor) {
    const commitOriginalEditor = commitDiffEditor.getOriginalEditor();
    const commitModifiedEditor = commitDiffEditor.getModifiedEditor();
    commitOriginalDocDecorations = updateEditorDocDecorations(
      commitOriginalEditor,
      docsJson.base,
      commitOriginalDocDecorations
    );
    commitModifiedDocDecorations = updateEditorDocDecorations(
      commitModifiedEditor,
      docsJson.head,
      commitModifiedDocDecorations
    );
  }

  // Base editor
  if (baseYamlEditor) {
    baseDocDecorations = updateEditorDocDecorations(
      baseYamlEditor,
      docsJson.base,
      baseDocDecorations
    );
  }

  // Head editor
  if (headYamlEditor) {
    headDocDecorations = updateEditorDocDecorations(
      headYamlEditor,
      docsJson.head,
      headDocDecorations
    );
  }

  // Resolved editor
  if (resolvedYamlEditor) {
    resolvedDocDecorations = updateEditorDocDecorations(
      resolvedYamlEditor,
      docsJson.resolved,
      resolvedDocDecorations
    );
  }

  // Single editor (refactor view)
  if (singleEditor) {
    singleDocDecorations = updateEditorDocDecorations(
      singleEditor,
      docsJson.resolved,
      singleDocDecorations
    );
  }
}

/**
 * Blinks a specific range in the resolved editor twice to grab attention.
 *
 * @param {Object} highlightOptions - Monaco editor decoration options.
 */
function blinkHighlight(editor, highlightOptions) {
  if (!editor) {
    console.error("Resolved editor is not initialized.");
    return;
  }

  try {
    let decorations = []; // Track active decorations

    // Blink twice by default
    function blink(timesLeft) {
      if (timesLeft > 0) {
        // Apply the highlight
        decorations = editor.deltaDecorations(decorations, [highlightOptions]);

        setTimeout(() => {
          // Remove the highlight
          decorations = editor.deltaDecorations(decorations, []);
          setTimeout(() => {
            blink(timesLeft - 1); // Repeat
          }, 200); // Time the highlight stays off
        }, 300); // Time the highlight stays on
      }
    }

    blink(2);
  } catch (e) {
    console.error("Error during blink highlight:", e);
  }
}

function scrollToAndHighlightLine(editor, lineNumber, decoration) {
  if (!editor) {
    return;
  }

  try {
    const model = editor.getModel();

    // Scroll to the specified line number and center it
    editor.revealLineInCenter(lineNumber);

    // Define the highlight color and options
    const highlightOptions = {
      range: new monaco.Range(
        lineNumber,
        1,
        lineNumber,
        model.getLineMaxColumn(lineNumber)
      ),
      options: {
        isWholeLine: true,
        className: decoration,
      },
    };
    blinkHighlight(editor, highlightOptions); // Blink twice
  } catch (e) {
    console.error("Error scrolling to line:", e);
  }
}
