let singleEditor;
let contentModel;
let overlayOpenContext;
let fileLanguage = "dart";
let isOverlayOpen = false;

const _resolvedEditorDartCallbackName = "DartResolvedEditorChanged";
const _editorLinesChangedName = "DartEditorLinesChanged";
const _editorFocusChangedName = "DartEditorFocusChanged";
const _editorSuggestionsRequestedName = "DartEditorSuggestionsRequested";
const _editorCursorPositionChangedName = "DartEditorCursorPositionChanged";
const _editorKeyUpName = "DartEditorKeyUp";
const _editorKeyDownName = "DartEditorKeyDown";
const _editorKeyEnterName = "DartEditorKeyEnter";
const _editorKeyEscapeName = "DartEditorKeyEscape";
const _editorCursorLineChangedName = "DartEditorCursorLineChanged";

/**
 * Properly decodes base64 encoded UTF-8 string including emoji characters
 * @param {string} base64 - Base64 encoded string
 * @returns {string} - Decoded UTF-8 string
 */
function decodeBase64UTF8(base64) {
  try {
    // First decode base64 to binary string
    const binaryString = atob(base64);

    // Convert binary string to Uint8Array
    const bytes = new Uint8Array(binaryString.length);
    for (let i = 0; i < binaryString.length; i++) {
      bytes[i] = binaryString.charCodeAt(i);
    }

    // Decode UTF-8 bytes to string
    const decoder = new TextDecoder("utf-8");
    return decoder.decode(bytes);
  } catch (error) {
    console.error("Error in UTF-8 decoding:", error);
    // Fallback to simple atob if TextDecoder fails
    return atob(base64);
  }
}

/**
 * Initializes the Monaco editors for a 3-way merge view and sets up various
 * configurations and listeners.
 *
 * @param {string} content - The content of the file.
 * @param {boolean} isLightMode - A flag indicating whether the light mode is enabled.
 * @param {boolean} isInlineEditor - A flag indicating whether the editor is an inline editor.
 * @param {boolean} isMacOS - A flag indicating whether the platform is macOS.
 * @param {boolean} isReadOnly - A flag indicating whether the editor is read-only.
 */
function initializeEditor(
  content,
  isLightMode,
  isInlineEditor,
  isMacOS,
  isReadOnly
) {
  updateWebviewTheme(isLightMode);

  require.config({
    paths: {
      vs: "https://cdnjs.cloudflare.com/ajax/libs/monaco-editor/0.34.0/min/vs",
    },
  });
  require(["vs/editor/editor.main"], function () {
    setCustomTheme(isLightMode);

    // Decode the content from base64 to string with proper UTF-8 handling
    let contentDecoded;
    try {
      contentDecoded = decodeBase64UTF8(content);
    } catch (error) {
      console.error(`Error decoding file content: ${error}`);
      return;
    }

    initializeSingleView(contentDecoded, isInlineEditor, isMacOS, isReadOnly);
  });
}

// Modified window resize handler
window.addEventListener("resize", () => {
  singleEditor.layout();
});

/**
 * Shows or hides the editor scrollbars
 * @param {boolean} show - Whether to show the scrollbars
 */
function setScrollbarVisibility(show) {
  if (!singleEditor) return;

  singleEditor.updateOptions({
    scrollbar: {
      horizontal: show ? "auto" : "hidden",
      vertical: show ? "auto" : "hidden",
      useShadows: show,
      verticalScrollbarSize: show ? 6 : 0,
      horizontalScrollbarSize: show ? 6 : 0,
      alwaysConsumeMouseWheel: false,
      verticalHasArrows: false,
      horizontalHasArrows: false,
    },
  });
}

function initializeSingleView(content, isInlineEditor, isMacOS, isReadOnly) {
  try {
    // Create the editor to render the file content
    // Show and setup single editor container
    const container = ensureContainerExists("single-container");
    container.style.width = "100%";
    container.style.height = "100%";
    container.style.padding = "0 0 0 8px";
    container.style.margin = "0";
    container.style.boxSizing = "border-box";

    singleEditor = monaco.editor.create(container, {
      value: content,
      language: fileLanguage,
      lineNumbers: "off",
      folding: false,
      padding: {
        top: 4,
        bottom: 4,
        left: 0,
        right: 0,
      },
      wordWrap: "on",
      minimap: { enabled: false },
      scrollBeyondLastLine: false,
      scrollbar: {
        horizontal: "hidden",
        vertical: "hidden",
        useShadows: false,
        verticalScrollbarSize: 0,
        horizontalScrollbarSize: 0,
        alwaysConsumeMouseWheel: false,
        verticalHasArrows: false,
        horizontalHasArrows: false,
      },
      renderLineHighlight: "none",
      overviewRulerLanes: 0,
      overviewRulerBorder: false,
      hideCursorInOverviewRuler: true,
      lineDecorationsWidth: 0,
      lineNumbersMinChars: 0,
      glyphMargin: false,
      fixedOverflowWidgets: true,
      contextmenu: false,
      quickSuggestions: false,
      suggestOnTriggerCharacters: false,
      parameterHints: { enabled: false },
      hover: { enabled: false },
      lightbulb: { enabled: false },
      codeLens: false,
      suggest: { enabled: false },
      tabCompletion: "off",
      snippetSuggestions: "none",
      acceptSuggestionOnEnter: "off",
      acceptSuggestionOnCommitCharacter: false,
      // Disable all editor actions
      actions: {
        enabled: false,
      },
      // Disable all editor commands
      commands: {
        enabled: false,
      },
      automaticLayout: true,
      readOnly: isReadOnly,
    });

    // Disable certain keyboard shortcuts for inline editor
    const disabledKeyCombinations = [
      // Disable search shortcut
      monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyF,
      // Disable F1 and F3 shortcuts
      monaco.KeyCode.F1,
      monaco.KeyCode.F3,

      monaco.KeyMod.Shift | monaco.KeyCode.Enter,
      monaco.KeyMod.CtrlCmd | monaco.KeyCode.Enter,
      monaco.KeyMod.Alt | monaco.KeyCode.Enter,
      monaco.KeyMod.Alt | monaco.KeyMod.WinCtrl | monaco.KeyCode.Enter,
      monaco.KeyMod.WinCtrl | monaco.KeyCode.Enter,
    ];

    // Only block Enter key if it's an inline editor
    if (isInlineEditor && !isReadOnly) {
      disabledKeyCombinations.push(monaco.KeyCode.Enter);
    }

    // To block monaco built-in suggestions on Windows
    if (!isMacOS) {
      disabledKeyCombinations.push(
        monaco.KeyMod.WinCtrl | monaco.KeyCode.Space
      );
    }

    disabledKeyCombinations.forEach((key) => {
      singleEditor.addCommand(key, function () {
        return null;
      });
    });

    const suggestionsKey = isMacOS
      ? monaco.KeyMod.WinCtrl | monaco.KeyCode.Space
      : monaco.KeyMod.CtrlCmd | monaco.KeyCode.Space;

    singleEditor.addCommand(suggestionsKey, function () {
      const position = singleEditor.getPosition();
      const offset = singleEditor.getModel().getOffsetAt(position);

      if (isWeb()) {
        onEditorSuggestionsRequested(offset);
      } else if (window.flutter_inappwebview) {
        window.flutter_inappwebview.callHandler(
          "editorSuggestionsRequested",
          offset
        );
      }
      return null;
    });

    // Create context keys for overlay and inline editor states
    overlayOpenContext = singleEditor.createContextKey(
      "overlayOpen",
      isOverlayOpen
    );
    const inlineEditorContext = singleEditor.createContextKey(
      "inlineEditor",
      isInlineEditor
    );

    // Add command for up arrow key
    singleEditor.addCommand(
      monaco.KeyCode.UpArrow,
      function () {
        if (isWeb()) {
          onEditorKeyUp(true);
        } else if (window.flutter_inappwebview) {
          window.flutter_inappwebview.callHandler("editorKeyUp");
        }
        return null;
      },
      "overlayOpen"
    );

    // Add command for down arrow key
    singleEditor.addCommand(
      monaco.KeyCode.DownArrow,
      function () {
        if (isWeb()) {
          onEditorKeyDown(true);
        } else if (window.flutter_inappwebview) {
          window.flutter_inappwebview.callHandler("editorKeyDown");
        }
        return null;
      },
      "overlayOpen"
    );

    // Add command for enter key
    singleEditor.addCommand(
      monaco.KeyCode.Enter,
      function () {
        if (isWeb()) {
          onEditorKeyEnter(true);
        } else if (window.flutter_inappwebview) {
          window.flutter_inappwebview.callHandler("editorKeyEnter");
        }
      },
      "overlayOpen || inlineEditor"
    );

    // Add command for escape key
    singleEditor.addCommand(
      monaco.KeyCode.Escape,
      function () {
        if (isWeb()) {
          onEditorKeyEscape(true);
        } else if (window.flutter_inappwebview) {
          window.flutter_inappwebview.callHandler("editorKeyEscape");
        }
      },
      "overlayOpen || inlineEditor"
    );

    // Create a new model for the file content
    contentModel = monaco.editor.createModel(content, fileLanguage);
    // Load the model in the editor
    singleEditor.setModel(contentModel);

    // Calculate and notify initial line count
    const initialLines = getTotalLinesIncludingSoftWraps();
    if (isWeb()) {
      onEditorLinesChanged(initialLines);
    } else if (window.flutter_inappwebview) {
      window.flutter_inappwebview.callHandler(
        "editorLinesChanged",
        initialLines
      );
    }

    addResolvedEditorChangeListener();

    // Set cursor position to the end of the text
    const lineCount = contentModel.getLineCount();
    const lastLine = contentModel.getLineContent(lineCount);
    singleEditor.setPosition({
      lineNumber: lineCount,
      column: lastLine.length + 1,
    });

    // Automatically focus the editor when it's loaded
    singleEditor.focus();

    // Scroll to the bottom of the editor
    // with 200ms delay (to ensure the editor is fully loaded)
    setTimeout(() => {
      singleEditor.setScrollPosition({
        scrollTop: singleEditor.getScrollHeight(),
        scrollLeft: 0,
      });
    }, 200);

    singleEditor.onDidFocusEditorWidget(() => {
      onSingleEditorFocusChanged(true);
    });
    singleEditor.onDidBlurEditorWidget(() => {
      onSingleEditorFocusChanged(false);
    });

    // Add cursor position change listener
    singleEditor.onDidChangeCursorPosition((e) => {
      const offset = singleEditor.getModel().getOffsetAt(e.position);
      const lineNumber = e.position.lineNumber;

      if (isWeb()) {
        onEditorCursorPositionChanged(offset);
        onEditorCursorLineChanged(lineNumber);
      } else if (window.flutter_inappwebview) {
        window.flutter_inappwebview.callHandler(
          "editorCursorPositionChanged",
          offset
        );
        window.flutter_inappwebview.callHandler(
          "editorCursorLineChanged",
          lineNumber
        );
      }
    });
  } catch (error) {
    console.error(`Error initializing single editor: ${error}`);
  }
}

function onSingleEditorFocusChanged(hasFocus) {
  if (isWeb()) {
    onEditorFocusChanged(hasFocus);
  } else if (window.flutter_inappwebview) {
    window.flutter_inappwebview.callHandler("editorFocusChanged", hasFocus);
  }
}

/**
 * Focuses the single editor.
 * This function can be called from Flutter to focus the editor.
 */
function focusSingleEditor() {
  if (singleEditor) {
    singleEditor.focus();
  }
}

/**
 * Applies completion text at the specified position in the editor.
 * @param {string} completionText - The text to insert
 * @param {number} replacementOffset - The offset where to start replacement
 * @param {number} replacementLength - The length of text to replace
 */
function applyCompletion(completionText, replacementOffset, replacementLength) {
  if (!singleEditor) return;

  const model = singleEditor.getModel();
  if (!model) return;

  // Convert offset to position
  const startPosition = model.getPositionAt(replacementOffset);
  const endPosition = model.getPositionAt(
    replacementOffset + replacementLength
  );

  // Create edit operation
  const edit = {
    range: new monaco.Range(
      startPosition.lineNumber,
      startPosition.column,
      endPosition.lineNumber,
      endPosition.column
    ),
    text: completionText,
    forceMoveMarkers: true,
  };

  // Apply the edit
  model.pushEditOperations([], [edit], () => []);

  // Move cursor to end of inserted text
  const newPosition = model.getPositionAt(
    replacementOffset + completionText.length
  );
  singleEditor.setPosition(newPosition);

  // Ensure editor is focused
  singleEditor.focus();
}

// Function to listen for changes in the resolved editor
function addResolvedEditorChangeListener() {
  contentModel.onDidChangeContent(() => {
    const content = contentModel.getValue();
    const totalLines = getTotalLinesIncludingSoftWraps();

    const position = singleEditor.getPosition();
    const offset = singleEditor.getModel().getOffsetAt(position);

    const data = JSON.stringify({
      content: content,
      offset: offset,
    });

    if (isWeb()) {
      // Notify Flutter (Web)
      onResolvedDartCallback(data);
      onEditorLinesChanged(totalLines);
    } else if (window.flutter_inappwebview) {
      // Notify Flutter (Desktop)
      window.flutter_inappwebview.callHandler("resolvedEditorChanged", data);
      window.flutter_inappwebview.callHandler("editorLinesChanged", totalLines);
    } else {
      console.error("No valid platform found to send resolved content to.");
    }
  });
}

// Utility function to detect if the platform is web
function isWeb() {
  return typeof window.flutter_inappwebview === "undefined";
}

// Apply FF theme to the Monaco Editor
function setCustomTheme(isLightMode) {
  const themeName = isLightMode ? "custom-light-theme" : "custom-dark-theme";

  const themeConfig = isLightMode
    ? {
        base: "vs", // Light theme base
        inherit: true,
        rules: [{ token: "", background: "FFFFFF" }],
        colors: {
          "editor.background": "#FFFFFF", // Editor background color
          "editor.lineHighlightBackground": "#F1F4F8",
          "editor.selectionBackground": "#4B39EF4C",
          "editor.selectionHighlightBackground": "#4B39EF4C",
          "diffEditor.insertedLineBackground": "#ceefeb", // Light Green
          "diffEditor.insertedTextBackground": "#aae3dc",
          "diffEditor.removedLineBackground": "#f6d0d0", // Light Red
          "diffEditor.removedTextBackground": "#efadad",
          "editorOverviewRuler.addedForeground": "#ceefeb", // Green in minimap
          "editorOverviewRuler.deletedForeground": "#f6d0d0", // Red in minimap
        },
      }
    : {
        base: "vs-dark", // Dark theme base
        inherit: true,
        rules: [{ token: "", background: "14181B" }],
        colors: {
          "editor.background": "#14181B", // Editor background color
          "editor.lineHighlightBackground": "#1D2428",
          "editor.selectionBackground": "#4B39EF4C",
          "editor.selectionHighlightBackground": "#4B39EF4C",
          "diffEditor.insertedLineBackground": "#264642", // Light Green
          "diffEditor.insertedTextBackground": "#26615c",
          "diffEditor.removedLineBackground": "#4e2827", // Light Red
          "diffEditor.removedTextBackground": "#6c2b2d",
          "editorOverviewRuler.addedForeground": "#264642", // Green in minimap
          "editorOverviewRuler.deletedForeground": "#4e2827", // Red in minimap
        },
      };

  // Define the custom theme
  monaco.editor.defineTheme(themeName, themeConfig);

  // Apply the custom theme
  monaco.editor.setTheme(themeName);
}

/**
 * Updates the webview theme based on the provided mode.
 *
 * This function changes the background color of the body and the document element
 * based on whether the light mode is enabled or not. It also updates the colors
 * of the dividers with the appropriate color for the selected mode.
 *
 * @param {boolean} isLightMode - A boolean indicating if the light mode is enabled.
 * If true, the light mode is enabled; otherwise, the dark mode is enabled.
 */
function updateWebviewTheme(isLightMode) {
  const backgroundColor = isLightMode ? "#FFFFFF" : "#14181B";
  document.body.style.backgroundColor = backgroundColor;
  document.documentElement.style.backgroundColor = backgroundColor;

  const dividerColor = isLightMode ? "#E0E3E7" : "#323B45";

  // Update all possible dividers
  const dividers = ["divider", "horizontal-divider", "diff-divider"];
  dividers.forEach((id) => {
    const element = document.getElementById(id);
    if (element) {
      element.style.backgroundColor = dividerColor;
    }
  });
}

// Utility functions for DOM manipulation
function ensureContainerExists(id) {
  let container = document.getElementById(id);
  if (!container) {
    container = document.createElement("div");
    container.id = id;
    container.className = "editor-pane";
    document.getElementById("top-editors-container").appendChild(container);
  }
  container.style.display = "block";
  return container;
}

/**
 * Calculates the total number of lines including soft wraps in the editor.
 * @returns {number} The total number of lines including soft wraps
 */
function getTotalLinesIncludingSoftWraps() {
  if (!singleEditor) return 0;

  const model = singleEditor.getModel();
  if (!model) return 0;

  const viewModel = singleEditor._getViewModel();
  if (!viewModel) return 0;

  let totalLines = 0;
  const lineCount = model.getLineCount();

  for (let lineNumber = 1; lineNumber <= lineCount; lineNumber++) {
    const viewLineNumber =
      viewModel.coordinatesConverter.convertModelPositionToViewPosition({
        lineNumber: lineNumber,
        column: 1,
      }).lineNumber;

    const nextViewLineNumber =
      lineNumber < lineCount
        ? viewModel.coordinatesConverter.convertModelPositionToViewPosition({
            lineNumber: lineNumber + 1,
            column: 1,
          }).lineNumber
        : viewModel.getLineCount() + 1;

    totalLines += nextViewLineNumber - viewLineNumber;
  }

  return totalLines;
}

/**
 * Updates the overlay open state
 * @param {boolean} open - Whether the overlay is open
 */
function updateOverlayState(open) {
  isOverlayOpen = open;
  if (singleEditor) {
    overlayOpenContext.set(open);
  }
}
