//
//	MorseEncoder.js
//
//	Created by Nikolas Hansen on 02.07.13.
//	Copyright (c) 2013 Nikolas Hansen. All rights reserved.
//

//  This code is Loosely based on:
/*
 Morse.js
A jQuery Plugin that annotates text with Morse Code
http://github.com/mattt/Morse.js

Copyright (c) 2010-2012 Mattt Thompson
Licensed under the MIT licenses.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */

function MorseEncoder() {
  this.morse_code = {
		"a": "._",    "b": "_...",  "c": "_._.",  "d": "_..",
		"e": ".",     "f": ".._.",  "g": "__.",   "h": "....",
		"i": "..",    "j": ".___",  "k": "_._",   "l": "._..",
		"m": "__",    "n": "_.",    "o": "___",   "p": ".__.",
		"q": "__._",  "r": "._.",   "s": "...",   "t": "_",
		"u": ".._",   "v": "..._",  "w": ".__",   "x": "_.._",
		"y": "_.__",  "z": "__..",  " ": " ",

		"1": ".____", "2": "..___", "3": "...__", "4": "...._", "5": ".....",
		"6": "_....", "7": "__...", "8": "___..", "9": "____.", "0": "_____",
		
		/*
		 * Note: Some operators prefer "!" as "___." and others as "_._.__"
		 * ARRL message format has most punctuation spelled out, as many symbols'
		 * encodings conflict with procedural signals (e.g. "=" and "BT").
		 */
		".": "._._._", ",": "__..__", "?": "..__..",  "'": ".____.",
		"/": "_.._.",  "(": "_.__.",  ")": "_.__._",  "&": "._...",
		":": "___...", ";": "_._._.", "=": "_..._",   "+": "._._.",
		"-": "_...._", "_": "..__._", "\"": "._.._.", "$": "..._.._",
		"!": "_._.__", "@": ".__._."
  };
}

MorseEncoder.prototype.encode = function(message) {
  var tokens = message.split(/\s+/);
  var symbols = [];

  for (i in tokens) {
    var token = tokens[i];
    var letters = token.split('');
		
    for (j in letters) {
      var letter = letters[j];
      var symbol = this.morse_code[letter.toLowerCase()];
      if (symbol) {
				symbols += symbol;
      }
    }
		
		if (i < tokens.length - 1) { 
			symbols += " ";
		}
  }
  return symbols;
}

MorseEncoder.prototype.getDuration = function(symbols) {
  buffer = [];
  for (var i in symbols) {
    var length;
    var symbol = symbols[i];

		if (symbol == ' ') {
			this.pause(5);
		
		} else {
			if (i > 0 && symbols[i-1] != ' ') {
				this.pause(1);
			}

			if (symbol == '.') {
				this.tone(1);
			} else if (symbol == '_') {
				this.tone(3);
			} 
		}
  }
  return buffer;
}

MorseEncoder.prototype.tone = function(duration) {
  for (i=0; i<duration; i++) {
		buffer.push(1);
  }
  return buffer;
}

MorseEncoder.prototype.pause = function(duration) {
  for (i=0; i<duration; i++) {
		buffer.push(0);
  }
  return buffer;
}

MorseEncoder.prototype.morse = function(durations, unit_duration) {
  this.buffer = durations;
  this.repeat_buffer = durations.slice(); //save copy of buffer

  this.blink_interval = setInterval(Zepto.proxy(function() {this.morseSample()}, this), unit_duration);
} 

MorseEncoder.prototype.startMorse = function(message) {
  var symbols = this.encode(message);
  var durations = this.getDuration(symbols);
  this.morse(durations, 200);
}

MorseEncoder.prototype.stopMorse = function() {
  clearInterval(this.blink_interval);
}


MorseEncoder.prototype.morseSample = function() {
  if (this.buffer.length <= 0) {
		if (this.repeat) {
			this.buffer = this.repeat_buffer.slice(); //copy back
			//TODO: pause a bit?
		} else {
			clearInterval(this.blink_interval);
			Zepto(document).trigger('morse_finished');
			return;
		}
  }
  var value = this.buffer.shift() //=> pop front
  if (value) {
		blinker.lightOn();
  } else {
		blinker.lightOff();
  }
}
