create or replace package body apps.graphite_sync as
  patch_key constant varchar2(5) := 'patch';
  ack_key constant varchar2(11) := 'acknowledge';
  connection_metadata_key constant varchar2(18) := 'connectionMetadata';
  metadata_keys_key constant varchar2(12) := 'metadataKeys';
  supplier_key constant varchar2(8) := 'supplier';
  sites_key constant varchar2(5) := 'sites';
  post_site_batch_key constant varchar2(15) := 'post_site_batch';
  contacts_key constant varchar2(8) := 'contacts';
  banks_key constant varchar2(5) := 'banks';
  ignore_missing_key constant varchar(14) := 'ignore_missing';

  procedure raise_if_error_status(
    name in varchar2,
    result in out nocopy api_result_t
  ) is
    idx number;
    idx_out number;
    debug_msg_data varchar2(4000);
    err_msg varchar(4000);
  begin
    if result.status = 'S' then
      return;
    end if;

      result.msg_data := 'Method: ' || name || chr(10) || result.msg_data;
      if result.msg_count >= 1 then
        result.msg_data := result.msg_data || chr(10) || 'FND_MSG_PUB details: ';
        for idx in 1..result.msg_count
          loop
            debug_msg_data := '';
            fnd_msg_pub.get(
              p_msg_index => idx,
              p_encoded => 'F',
              p_data => debug_msg_data,
              p_msg_index_out => idx_out
            );
            result.msg_data := result.msg_data || chr(10) || 'Message ' || idx_out || ': ' || debug_msg_data;
        end loop;
    end if;

    result.msg_data := replace(result.msg_data, chr(0), '');

    raise_application_error(-20002, result.msg_data);
  end raise_if_error_status;

  function convert_metadata_keys_json(
    metadata_keys_json in pljson
  ) return r_return_keys is
    parser graphite_sync_json_parser;
    keys r_return_keys;
  begin
    parser := graphite_sync_json_parser(json_data => metadata_keys_json);

    -- vendor keys
    keys.vendor_id_key := parser.get_string('vendorIdKey');
    keys.party_id_key := parser.get_string('partyIdKey');
    keys.segment1_key := parser.get_string('segment1Key');
    -- site keys
    keys.vendor_site_id_key := parser.get_string('vendorSiteIdKey');
    keys.party_site_id_key := parser.get_string('partySiteIdKey');
    keys.location_id_key := parser.get_string('locationIdKey');
    --bank keys
    keys.bank_id_key := parser.get_string('bankIdKey');
    keys.branch_id_key := parser.get_string('branchIdKey');
    keys.bank_account_id_key := parser.get_string('bankAccountIdKey');
    keys.intermediary_bank_id_key := parser.get_string('intermediaryBankIdKey');
    keys.ext_payment_assign_id_key := parser.get_string('paymentAssignIdKey');
    -- graphite return object keys
    keys.site_group_key := parser.get_string('siteGroupKey');
    keys.supplier_group_key := parser.get_string('supplierGroupKey');
    keys.bank_group_key := parser.get_string('bankGroupKey');
    keys.address_book_group_key := parser.get_string('addressBookGroupKey');

    return keys;
  end;

  procedure init_output_json(
    output_json in out pljson,
    ack_json in out pljson
  ) is
  begin
    output_json.put(patch_key, pljson());
    output_json.put(ack_key, ack_json);
  end init_output_json;

  function extract_json_elements(
    profile_json in pljson
  ) return vendor_elements_json_t is
    parser graphite_sync_json_parser;
    elements vendor_elements_json_t;
  begin
    parser := graphite_sync_json_parser(json_data => profile_json);
    elements.supplier := parser.get_object(supplier_key);

    elements.connection_metadata := parser.get_object(connection_metadata_key);
    elements.connection_metadata_keys := elements.connection_metadata.get_pljson(metadata_keys_key);
    elements.connection_metadata.remove(metadata_keys_key);

    elements.site_arr := parser.get_array(sites_key);
    elements.post_site_batch := parser.get_object(post_site_batch_key);

    elements.ignore_missing := parser.get_bool(ignore_missing_key);
    return elements;
  end extract_json_elements;

  procedure sync_site_bank(
    bank_json in pljson,
    supplier_output in out graphite_sync_supplier.supplier_output_t,
    site_output in out graphite_sync_site.site_output_t,
    bank_output in out graphite_sync_bank.bank_result_t,
    keys in r_return_keys,
    patch_json in out pljson,
    site_out_obj in out pljson
  ) is
    out_json pljson;
    bank_out_obj pljson := pljson();
    bank_out_arr pljson_list;
  begin
    out_json := case when keys.bank_group_key is not null then bank_out_obj else site_out_obj end;

    graphite_sync_bank.process_bank_objects_from_json(
      bank_obj_json => bank_json,
      bank_output => bank_output,
      supplier_output => supplier_output,
      site_output => site_output,
      keys => keys,
      return_json => bank_out_obj
    );

    if keys.bank_group_key is not null then
      if not patch_json.exist(keys.bank_group_key) then
        patch_json.put(keys.bank_group_key, pljson_list());
      end if;

      bank_out_arr := patch_json.get_pljson_list(keys.bank_group_key);
      bank_out_arr.append(bank_out_obj);
      patch_json.put(keys.bank_group_key, bank_out_arr);
    end if;

  end sync_site_bank;

  procedure sync_site_contact(
    contact_json in pljson,
    supplier_output in out graphite_sync_supplier.supplier_output_t,
    site_output in out graphite_sync_site.site_output_t,
    contact_output in out graphite_sync_contact.contact_output_t
  ) is
    begin
      graphite_sync_contact.process_contact_from_json(
        contact_json => contact_json,
        supplier_output => supplier_output,
        site_output => site_output,
        contact_output => contact_output
      );
  end sync_site_contact;

  procedure sync_site(
    site_json in out pljson,
    patch_json in out pljson,
    site_output in out graphite_sync_site.site_output_t,
    supplier_output in out graphite_sync_supplier.supplier_output_t,
    keys in r_return_keys,
    ignore_missing in boolean
  ) is
    site_out_arr pljson_list;
    site_out_obj pljson := pljson();
    ab_out_arr pljson_list;
    ab_out_obj pljson := pljson();
    bank_arr pljson_list;
    bank_obj pljson;
    bank_output graphite_sync_bank.bank_result_t;
    contact_arr pljson_list;
    contact_output graphite_sync_contact.contact_output_t;
    contact_obj pljson;
  begin
    site_out_arr := patch_json.get_pljson_list(keys.site_group_key);
    ab_out_arr := patch_json.get_pljson_list(keys.address_book_group_key);

    graphite_sync_site.process_site_from_json(
      site_json => site_json,
      site_output_obj => site_out_obj,
      address_book_output_obj => ab_out_obj,
      vendor_id => supplier_output.vendor_id,
      keys => keys,
      ignore_missing => ignore_missing,
      supplier_party_id => supplier_output.party_id,
      site_output => site_output
    );

    if site_json.exist(banks_key) and not ignore_missing then
      bank_arr := site_json.get_pljson_list(banks_key);
      for i in 1 .. bank_arr.count
        loop
          bank_obj := pljson(bank_arr.get(i));
          sync_site_bank(
            bank_json => bank_obj,
            supplier_output => supplier_output,
            site_output => site_output,
            bank_output => bank_output,
            keys => keys,
            patch_json => patch_json,
            site_out_obj => site_out_obj
          );
      end loop;
    end if;


    if site_json.exist(contacts_key) and not ignore_missing then
      contact_arr := site_json.get_pljson_list(contacts_key);
      for i in 1 .. contact_arr.count
        loop
          contact_obj := pljson(contact_arr.get(i));
          sync_site_contact(
            contact_json => contact_obj,
            supplier_output => supplier_output,
            site_output => site_output,
            contact_output => contact_output
          );
      end loop;
    end if;

    site_out_arr.append(site_out_obj);
    patch_json.put(keys.site_group_key, site_out_arr);
    ab_out_arr.append(ab_out_obj);
    patch_json.put(keys.address_book_group_key, ab_out_arr);
  end sync_site;

  procedure syncronize(
    user_id in number,
    resp_id in number,
    app_id in number,
    in_clob in clob,
    out_clob in out clob,
    log_level in number
  ) is
    output_json pljson := pljson();
    profile_json pljson;
    ack_json pljson;
    patch_json pljson;
    elements vendor_elements_json_t;
    keys r_return_keys;
    supplier_output graphite_sync_supplier.supplier_output_t;
    site_obj pljson;
    errors_arr pljson_list := pljson_list();
    processed_sites graphite_sync_site.site_output_table_t := graphite_sync_site.site_output_table_t();
  begin
    fnd_global.apps_initialize(user_id, resp_id, app_id);

    profile_json := pljson(in_clob);
    elements := extract_json_elements(profile_json);
    keys := convert_metadata_keys_json(elements.connection_metadata_keys);

    init_output_json(output_json => output_json, ack_json => elements.connection_metadata);

    ack_json := output_json.get_pljson(ack_key);
    patch_json := output_json.get_pljson(patch_key);

    graphite_sync_supplier.process_supplier_from_json(
      supplier_json => elements.supplier,
      output_json => patch_json,
      keys => keys,
      ignore_missing => elements.ignore_missing,
      supplier_output => supplier_output
    );

    patch_json.put(keys.site_group_key, pljson_list());
    patch_json.put(keys.address_book_group_key, pljson_list());

    for i in 1 .. elements.site_arr.count
      loop
        site_obj := pljson(elements.site_arr.get(i));
        processed_sites.extend(1);

        sync_site(
          site_json => site_obj,
          site_output => processed_sites(processed_sites.last),
          patch_json => patch_json,
          supplier_output => supplier_output,
          keys => keys,
          ignore_missing => elements.ignore_missing
        );
    end loop;

    if elements.post_site_batch is not null then
      graphite_sync_site.process_post_site_batch(
        post_site_batch_json => elements.post_site_batch,
        site_output_table => processed_sites,
        vendor_id => supplier_output.vendor_id
      );
    end if;

    output_json.put(patch_key, patch_json);
    output_json.to_clob(out_clob);

    exception
      when api_error_status then
        errors_arr.append('Oracle API Error: ' || sqlerrm);
        ack_json.put('errors', errors_arr);
        output_json.put(ack_key, ack_json);
        output_json.to_clob(out_clob);
      when others then
        errors_arr.append('Unexpected error occurred: ' || sqlerrm || '. ' || dbms_utility.format_error_backtrace());
        ack_json.put('errors', errors_arr);
        output_json.put(ack_key, ack_json);
        output_json.to_clob(out_clob);

  end syncronize;

end;
/
show err
