create or replace package body apps.graphite_sync_site as
  function party_id_for_vendor(
    vendor_id number
  ) return number is
    party_id number;
    v_vendor_id number;
  begin
    v_vendor_id := vendor_id;
    select party_id into party_id from ap_suppliers supp where supp.vendor_id = v_vendor_id;
    return party_id;

    exception
      when no_data_found then
        raise_application_error(-20002, 'The vendor id: ' || vendor_id || ' does not exist when attempting to find the party id for the vendor');
      when others then
        raise;
  end party_id_for_vendor;

  -- This function should only be called in an update
  -- Determines if the provided site code is the same as the site that we are updating. If it is, then we need to set the field in the record to null
  function is_updated_site_code(
    site_rec in ap_vendor_pub_pkg.r_vendor_site_rec_type
  ) return boolean is
    existing_vendor_code varchar2(15);
  begin
    select vendor_site_code into existing_vendor_code from ap_supplier_sites_all where vendor_id = site_rec.vendor_id and vendor_site_id = site_rec.vendor_site_id;
    if existing_vendor_code != site_rec.vendor_site_code then
      return true;
    end if;
    return false;

  exception
    when no_data_found then
      raise;
  end is_updated_site_code;

 procedure create_site(
    site_rec in ap_vendor_pub_pkg.r_vendor_site_rec_type,
    output out site_output_t,
    result out nocopy graphite_sync.api_result_t
 ) is
  begin
    ap_vendor_pub_pkg.create_vendor_site(
      p_api_version => 1.0,
      p_init_msg_list => fnd_api.g_false,
      p_commit => fnd_api.g_false,
      p_validation_level => fnd_api.g_valid_level_full,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data,
      p_vendor_site_rec => site_rec,
      x_vendor_site_id => output.vendor_site_id,
      x_party_site_id => output.party_site_id,
      x_location_id => output.location_id
    );

    graphite_sync.raise_if_error_status(name => 'create_site', result => result);
  end create_site;

 procedure update_site(
    site_rec in out ap_vendor_pub_pkg.r_vendor_site_rec_type,
    output out site_output_t,
    result out nocopy graphite_sync.api_result_t
 ) is
  begin
    if not is_updated_site_code(site_rec) then
      -- This needs to be set to null if the site code is not being updated
      site_rec.vendor_site_code := null;
    end if;

    ap_vendor_pub_pkg.update_vendor_site_public(
      p_api_version => 1.0,
      p_init_msg_list => fnd_api.g_true,
      p_commit => fnd_api.g_false,
      p_validation_level => fnd_api.g_valid_level_full,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data,
      p_vendor_site_rec => site_rec,
      p_vendor_site_id => site_rec.vendor_site_id
    );

    graphite_sync.raise_if_error_status(name => 'update_site', result => result);

    output.vendor_site_id := site_rec.vendor_site_id;
    output.party_site_id := site_rec.party_site_id;
    output.location_id := site_rec.location_id;
  end update_site;

  procedure update_location(
    address_rec in out hz_location_v2pub.location_rec_type,
    result out nocopy graphite_sync.api_result_t
  ) is
    object_version_number number;
  begin
    select object_version_number into object_version_number from hz_locations where location_id = address_rec.location_id;

    hz_location_v2pub.update_location(
      p_init_msg_list => fnd_api.g_true,
      p_location_rec => address_rec,
      p_object_version_number => object_version_number,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data
    );
    graphite_sync.raise_if_error_status(name => 'update_location', result => result);

  end update_location;

  procedure update_party_site(
    party_site_rec in out hz_party_site_v2pub.party_site_rec_type,
    result out nocopy graphite_sync.api_result_t
  ) is
    object_version_number number;
  begin
    select object_version_number into object_version_number from hz_party_sites where party_site_id = party_site_rec.party_site_id;

    hz_party_site_v2pub.update_party_site(
      p_init_msg_list => fnd_api.g_true,
      p_party_site_rec => party_site_rec,
      p_object_version_number => object_version_number,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data
    );

    graphite_sync.raise_if_error_status(name => 'update_party_site', result => result);
  end update_party_site;

-- Determines whether the provided json object contains an update or create for a given site
  function is_update (
    site_obj in pljson,
    vendor_id in number
  ) return boolean is
    already_exists number;
    site_id        number;
  begin
    site_id := graphite_sync_json_parser(site_obj).get_number('VENDOR_SITE_ID');
    if site_id is null then
       return false;
    end if;

  -- Confirm that the site id exists for the given vendor
    select 1 into already_exists from ap_supplier_sites_all where vendor_site_id = site_id and vendor_id = vendor_id;
  -- No exception, so the supplier exists
    return true;

  exception
    when no_data_found then
      return false;
  end is_update;

  procedure update_return_object(
    site_return_obj    in out pljson,
    ab_return_obj in out pljson,
    keys          in graphite_sync.r_return_keys,
    instance_id   in varchar2,
    ab_instance_id in varchar2,
    site_output   in site_output_t
  ) is
  begin
    site_return_obj.put(keys.vendor_site_id_key, site_output.vendor_site_id);
    site_return_obj.put(graphite_sync.instance_id_key, instance_id);

    ab_return_obj.put(keys.location_id_key, site_output.location_id);
    ab_return_obj.put(keys.party_site_id_key, site_output.party_site_id);
    ab_return_obj.put(graphite_sync.instance_id_key, ab_instance_id);
  end update_return_object;

  procedure process_post_site_batch(
    site_output_table in out site_output_table_t,
    post_site_batch_json in out pljson,
    vendor_id in number
  ) is
    site_rec ap_vendor_pub_pkg.r_vendor_site_rec_type;
    api_result graphite_sync.api_result_t;
  begin
    site_rec := graphite_sync_converter.convert_site_json(parser => graphite_sync_json_parser(json_data => post_site_batch_json));

    for i in 1 .. site_output_table.count loop
      site_rec.vendor_id := vendor_id;
      site_rec.vendor_site_id := site_output_table(i).vendor_site_id;
      site_rec.party_site_id := site_output_table(i).party_site_id;
      site_rec.location_id := site_output_table(i).location_id;

      update_site(site_rec => site_rec, output => site_output_table(i), result => api_result);
    end loop;
  end process_post_site_batch;

  procedure process_site_from_json(
    site_json in out pljson,
    site_output_obj in out nocopy pljson,
    address_book_output_obj in out nocopy pljson,
    vendor_id in number,
    keys in graphite_sync.r_return_keys,
    ignore_missing in boolean,
    supplier_party_id in number,
    site_output out site_output_t
  ) is
    payload_is_update boolean;
    parser graphite_sync_json_parser;
    update_parser graphite_sync_json_parser;
    api_result graphite_sync.api_result_t;
    site_rec ap_vendor_pub_pkg.r_vendor_site_rec_type;
    update_site_rec ap_vendor_pub_pkg.r_vendor_site_rec_type;
    address_rec hz_location_v2pub.location_rec_type;
    party_site_rec hz_party_site_v2pub.party_site_rec_type;
    instance_id varchar2(50);
    ab_instance_id varchar2(50);
  begin
    payload_is_update := is_update(site_obj => site_json, vendor_id => vendor_id);
    parser := graphite_sync_json_parser(json_data => site_json, is_update => payload_is_update and not ignore_missing);
    site_rec := graphite_sync_converter.convert_site_json(parser => parser);
    site_rec.vendor_id := vendor_id;

    if payload_is_update then
      update_site(site_rec => site_rec, output => site_output, result => api_result);
    else
      create_site(site_rec => site_rec, output => site_output, result => api_result);
    end if;

    if parser.get_object(address_key) is not null then
      address_rec := graphite_sync_converter.convert_address_json(address_json => parser.get_object(address_key));
      address_rec.location_id := site_output.location_id;
      update_location(address_rec => address_rec, result => api_result);
    end if;

    if parser.get_object(party_site_key) is not null then
      party_site_rec := graphite_sync_converter.convert_party_site_json(party_site_json => parser.get_object(party_site_key));
      party_site_rec.location_id := site_output.location_id;
      party_site_rec.party_site_id := site_output.party_site_id;

      -- Use supplier_party_id from payload if party_id is not set
      if party_site_rec.party_id is null then
        party_site_rec.party_id := supplier_party_id;
      end if;

      -- Only query database if still null (fallback for backward compatibility)
      if party_site_rec.party_id is null then
        party_site_rec.party_id := party_id_for_vendor(vendor_id => vendor_id);
      end if;

      update_party_site(party_site_rec => party_site_rec, result => api_result);
    end if;

    -- This allows us to make an update to the site after it was created.
    if parser.get_object(site_second_pass_key) is not null then
      update_parser := graphite_sync_json_parser(json_data => parser.get_object(site_second_pass_key));
      update_site_rec := graphite_sync_converter.convert_site_json(parser => update_parser);
      update_site_rec.vendor_id := vendor_id;
      update_site_rec.location_id := site_output.location_id;
      update_site_rec.party_site_id := site_output.party_site_id;
      update_site_rec.vendor_site_id := site_output.vendor_site_id;

      update_site(site_rec => update_site_rec, output => site_output, result => api_result);
    end if;

    instance_id := parser.get_string(graphite_sync.instance_id_key);
    ab_instance_id := parser.get_string(address_book_instance_id_key);

    update_return_object(
      site_return_obj => site_output_obj,
      ab_return_obj => address_book_output_obj,
      keys => keys,
      instance_id => instance_id,
      ab_instance_id => ab_instance_id,
      site_output => site_output
    );

  end process_site_from_json;

end graphite_sync_site;
/
show err
