create or replace package body apps.graphite_sync_supplier as
  -- Procedure to update the return pljson with the created/updated vendor ids
   procedure update_return_obj (
      return_obj    in out pljson,
      keys          in graphite_sync.r_return_keys,
      instance_id   in varchar2,
      supplier_output in supplier_output_t
   ) is
      list pljson_list;
      obj  pljson;
   begin
      -- Sometimes, customers allow only once supplier per graphite vendor, others can have multiple.
      -- If they have multiple then they get stored in a group.

      -- If this key is not set, then we are just dropping this right on the return object since this is stored at the root of the supplier
      if keys.supplier_group_key is null then
         return_obj.put(keys.vendor_id_key, supplier_output.vendor_id);
         return_obj.put(keys.party_id_key, supplier_output.party_id);
         return_obj.put(keys.segment1_key, supplier_output.segment1);
         return;
      end if;

      -- If the group key is set, we need to make the group.
      -- First we need to check if the group already exists from a previous supplier
      -- If it doesnt, create it
      if not return_obj.exist(keys.supplier_group_key) then
         return_obj.put(keys.supplier_group_key, pljson_list());
      end if;

      list := return_obj.get_pljson_list(keys.supplier_group_key);
      obj  := pljson();
      obj.put(keys.vendor_id_key, supplier_output.vendor_id);
      obj.put(keys.party_id_key, supplier_output.party_id);
      obj.put(keys.segment1_key, supplier_output.segment1);

      -- TODO - Link to the global package for the instance id key
      obj.put(graphite_sync.instance_id_key, instance_id);
      list.append(obj);
      return_obj.put(keys.supplier_group_key, list);
   end update_return_obj;

  -- TODO - Think about handling exceptions here instead of outside
   procedure create_supplier (
      supplier_rec    in ap_vendor_pub_pkg.r_vendor_rec_type,
      supplier_output out supplier_output_t,
      api_result      out nocopy graphite_sync.api_result_t
   ) is
   begin
      ap_vendor_pub_pkg.create_vendor(
                                     p_api_version => 1.0,
                                     p_init_msg_list => fnd_api.g_true,
                                     p_commit => fnd_api.g_false,
                                     p_validation_level => fnd_api.g_valid_level_full,
                                     x_return_status => api_result.status,
                                     x_msg_count => api_result.msg_count,
                                     x_msg_data => api_result.msg_data,
                                     p_vendor_rec => supplier_rec,
                                     x_vendor_id => supplier_output.vendor_id,
                                     x_party_id => supplier_output.party_id
      );

      graphite_sync.raise_if_error_status('create_supplier', api_result);

      select segment1 into supplier_output.segment1 from ap_suppliers where vendor_id = supplier_output.vendor_id;

   end create_supplier;

   procedure update_supplier (
      supplier_rec    in ap_vendor_pub_pkg.r_vendor_rec_type,
      supplier_output out supplier_output_t,
      api_result      out nocopy graphite_sync.api_result_t
   ) is
   begin
      ap_vendor_pub_pkg.update_vendor_public(
                                            p_api_version => 1.0,
                                            p_init_msg_list => fnd_api.g_true,
                                            p_commit => fnd_api.g_false,
                                            p_validation_level => fnd_api.g_valid_level_full,
                                            x_return_status => api_result.status,
                                            x_msg_count => api_result.msg_count,
                                            x_msg_data => api_result.msg_data,
                                            p_vendor_rec => supplier_rec,
                                            p_vendor_id => supplier_rec.vendor_id
      );

      -- Sometimes there is an error in the API that does not raise an exception. We need to check for that here
      -- even worse - sometimes it gives us no status at all... we know its a failure in this case

      dbms_output.put_line('API Result: ' || api_result.status);
      dbms_output.put_line('API Result: ' || api_result.msg_count);
      dbms_output.put_line('API Result: ' || api_result.msg_data);

      graphite_sync.raise_if_error_status('update_supplier', api_result);

      -- Use party_id from payload if available, otherwise query database
      if supplier_rec.party_id is not null then
         supplier_output.vendor_id := supplier_rec.vendor_id;
         supplier_output.party_id := supplier_rec.party_id;
         select segment1 into supplier_output.segment1 from ap_suppliers where vendor_id = supplier_rec.vendor_id;
      else
         -- Fallback to database query for backward compatibility
         select vendor_id,
                party_id,
                segment1
           into
            supplier_output.vendor_id,
            supplier_output.party_id,
            supplier_output.segment1
           from ap_suppliers
          where vendor_id = supplier_rec.vendor_id;
      end if;

   end update_supplier;

    procedure update_organization (
      org_rec         in hz_party_v2pub.organization_rec_type,
      api_result      out nocopy graphite_sync.api_result_t
    ) is
      version_num number;
      profile_id number;
    begin
      select object_version_number
      into
        version_num
      from hz_parties
      where party_id = org_rec.party_rec.party_id;

      dbms_output.put_line('Duns: ' || org_rec.duns_number_c);

      hz_party_v2pub.update_organization(
        p_init_msg_list => fnd_api.g_true,
        p_organization_rec => org_rec,
        p_party_object_version_number => version_num,
        x_profile_id => profile_id,
        x_return_status => api_result.status,
        x_msg_count => api_result.msg_count,
        x_msg_data => api_result.msg_data
      );

      graphite_sync.raise_if_error_status('update_organization', api_result);
    end update_organization;


-- Determines whether the provided json object contains an update or create payload
   function is_update (
      supplier_obj in pljson
   ) return boolean is
      already_exists number;
      vid            number;
   begin
    -- If there is a vendor id, then likely it alrady exists
      vid := graphite_sync_json_parser(json_data => supplier_obj).get_number('VENDOR_ID');
      if vid is null then
         return false;
      end if;

    -- Confirm that the vendor id exists and references a supplier
      select 1 into already_exists from ap_suppliers where vendor_id = vid;
    -- No exception, so the supplier exists
      return true;

    -- If the vendor id does not exist, then we have a data issue. Raise an exception and deal with it later
   exception
      when no_data_found then
         raise no_data_found;
   end is_update;


  procedure process_supplier_from_json(
    supplier_json in out pljson,
    output_json in out nocopy pljson,
    keys in graphite_sync.r_return_keys,
    ignore_missing in boolean,
    supplier_output out supplier_output_t
  ) is
    payload_is_update boolean;
    vendor_rec ap_vendor_pub_pkg.r_vendor_rec_type;
    org_json pljson;
    org_rec hz_party_v2pub.organization_rec_type;
    instance_id varchar2(50);
    parser graphite_sync_json_parser;
    result graphite_sync.api_result_t;
  begin
    payload_is_update := is_update(supplier_obj => supplier_json);
    parser := graphite_sync_json_parser(json_data => supplier_json, is_update => payload_is_update and not ignore_missing);
    vendor_rec := graphite_sync_converter.convert_supplier_json(supplier_json => supplier_json, parser => parser);
    if not payload_is_update then
      create_supplier(
        supplier_rec => vendor_rec,
        supplier_output => supplier_output,
        api_result => result
      );
    else
      update_supplier(
        supplier_rec => vendor_rec,
        supplier_output => supplier_output,
        api_result => result
      );
    end if;

    if supplier_json.exist(graphite_sync.organization_key) and supplier_json.get(graphite_sync.organization_key).count > 0 then
      org_json := parser.get_object(graphite_sync.organization_key);
      org_rec := graphite_sync_converter.convert_organization_json(org_json => org_json, party_id => supplier_output.party_id);
      update_organization(
        org_rec => org_rec,
        api_result => result
      );
    end if;

    instance_id := parser.get_string(graphite_sync.instance_id_key);
    update_return_obj(
      return_obj => output_json,
      keys => keys,
      instance_id => instance_id,
      supplier_output => supplier_output
    );

  end process_supplier_from_json;

end graphite_sync_supplier;
/
show err
