create or replace package body apps.graphite_sync_bank as

  procedure unset_intermediary_keys(
    inter_bank_rec in out iby_ext_bankacct_pub.intermediaryacct_rec_type
  ) is
    begin
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.intermediary_acct_id);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.bank_account_id);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.country_code);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.bank_name);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.city);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.bank_code);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.branch_number);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.bic);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.account_number);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.check_digits);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.iban);
      graphite_sync_json_utils.set_to_null_value(inter_bank_rec.comments);
    end unset_intermediary_keys;

  procedure set_existing_bank_id(
    bank_rec in out iby_ext_bankacct_pub.extbank_rec_type,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    end_date_ date;
  begin
    iby_ext_bankacct_pub.check_bank_exist(
      p_api_version => 1.0,
      p_init_msg_list => fnd_api.g_true,
      p_country_code => bank_rec.country_code,
      p_bank_name => bank_rec.bank_name,
      p_bank_number => bank_rec.bank_number,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data,
      x_bank_id => bank_rec.bank_id,
      x_end_date => end_date_,
      x_response => response_
    );

    graphite_sync.raise_if_error_status('set_existing_bank_id', result);

    if bank_rec.bank_id is not null then
      select object_version_number
      into bank_rec.object_version_number
      from hz_parties
      where party_id = bank_rec.bank_id;
    end if;

  end set_existing_bank_id;

  procedure set_existing_branch_id(
    branch_rec in out iby_ext_bankacct_pub.extbankbranch_rec_type,
    result in out nocopy graphite_sync.api_result_t
  ) is
      response_ iby_fndcpt_common_pub.result_rec_type;
      end_date_ date;
  begin
    iby_ext_bankacct_pub.check_ext_bank_branch_exist(
      p_api_version => 1.0,
      p_init_msg_list => fnd_api.g_true,
      p_bank_id => branch_rec.bank_party_id,
      p_branch_name => branch_rec.branch_name,
      p_branch_number => branch_rec.branch_number,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data,
      x_branch_id => branch_rec.branch_party_id,
      x_end_date => end_date_,
      x_response => response_
    );

    graphite_sync.raise_if_error_status('set_existing_branch_id', result);

    if branch_rec.branch_party_id is not null then
      -- don't ask... I have no idea what this does
      select
        nvl(hp_bch.object_version_number, 1),
        nvl(hca_rfc.object_version_number, 1),
        nvl(hcp_eft.object_version_number, 1),
        nvl(hca_typ.object_version_number, 1)
      into
        branch_rec.bch_object_version_number,
        branch_rec.rfc_object_version_number,
        branch_rec.eft_object_version_number,
        branch_rec.typ_object_version_number
      from
        hz_parties hp_bch,
        hz_code_assignments hca_rfc,
        hz_contact_points hcp_eft,
        hz_code_assignments hca_typ
      where hp_bch.party_id = branch_rec.branch_party_id
        and hca_rfc.class_category (+) = 'RFC_IDENTIFIER'
        and hca_rfc.owner_table_name (+) = 'HZ_PARTIES'
        and hca_rfc.owner_table_id (+) = hp_bch.party_id
        and hca_rfc.primary_flag (+) = 'Y'
        and hca_rfc.status (+) = 'A'
        and sysdate between hca_rfc.start_date_active (+) and nvl(hca_rfc.end_date_active(+), sysdate + 1)
        and hcp_eft.owner_table_name(+) = 'HZ_PARTIES'
        and hcp_eft.owner_table_id(+) = hp_bch.party_id
        and hcp_eft.contact_point_type(+) = 'EFT'
        and hcp_eft.status(+) = 'A'
        and hca_typ.class_category (+) = 'BANK_BRANCH_TYPE'
        and hca_typ.owner_table_name (+) = 'HZ_PARTIES'
        and hca_typ.owner_table_id (+) = hp_bch.party_id
        and hca_typ.status (+) = 'A'
        and hca_typ.primary_flag (+) = 'Y'
        and sysdate between hca_typ.start_date_active (+) and nvl(hca_typ.end_date_active(+), sysdate + 1);
      end if;
  end set_existing_branch_id;

  procedure set_existing_bank_acct_id(
    bank_acct_rec in out iby_ext_bankacct_pub.extbankacct_rec_type,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    start_date_ date;
    end_date_ date;
  begin
    iby_ext_bankacct_pub.check_ext_acct_exist(
      p_api_version => 1.0,
      p_init_msg_list => fnd_api.g_true,
      p_bank_id => bank_acct_rec.bank_id,
      p_branch_id => bank_acct_rec.branch_id,
      p_acct_number => bank_acct_rec.bank_account_num,
      p_acct_name => bank_acct_rec.bank_account_name,
      p_currency => bank_acct_rec.currency,
      p_country_code => bank_acct_rec.country_code,
      x_acct_id => bank_acct_rec.bank_account_id,
      x_start_date => start_date_,
      x_end_date => end_date_,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data,
      x_response => response_
    );

    if bank_acct_rec.bank_account_id is not null then
      select object_version_number
      into bank_acct_rec.object_version_number
      from iby_ext_bank_accounts
      where ext_bank_account_id = bank_acct_rec.bank_account_id;
    end if;

    graphite_sync.raise_if_error_status('set_existing_bank_acct_id', result);
  end set_existing_bank_acct_id;

  procedure set_existing_inter_bank_id(
    inter_bank_rec in out iby_ext_bankacct_pub.intermediaryacct_rec_type,
    result in out nocopy graphite_sync.api_result_t
  ) is
    begin
      select intermediary_acct_id
      into inter_bank_rec.intermediary_acct_id
      from iby_intermediary_accts
      where bic = inter_bank_rec.bic
        and bank_acct_id = inter_bank_rec.bank_account_id;

      exception
        when no_data_found then
          inter_bank_rec.intermediary_acct_id := null;
        when others then
          raise;
      
  end set_existing_inter_bank_id;

  function bank_is_update(
    bank_rec in iby_ext_bankacct_pub.extbank_rec_type
  ) return boolean is
    begin
      return bank_rec.bank_id is not null;
  end bank_is_update;

  function branch_is_update(
    branch_rec in iby_ext_bankacct_pub.extbankbranch_rec_type
  ) return boolean is
    begin
      return branch_rec.branch_party_id is not null;
  end branch_is_update;

  function bank_acct_is_update(
    bank_acct_rec in iby_ext_bankacct_pub.extbankacct_rec_type
  ) return boolean is
    begin
      return bank_acct_rec.bank_account_id is not null;
  end bank_acct_is_update;

  function inter_bank_is_update(
    inter_bank_rec in iby_ext_bankacct_pub.intermediaryacct_rec_type
  ) return boolean is
    begin
      return inter_bank_rec.intermediary_acct_id is not null;
  end inter_bank_is_update;

  procedure create_bank(
    bank_rec in iby_ext_bankacct_pub.extbank_rec_type,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    begin
      iby_ext_bankacct_pub.create_ext_bank(
        p_api_version => 1.0,
        p_init_msg_list => fnd_api.g_true,
        p_ext_bank_rec => bank_rec,
        x_bank_id => bank_output.bank_id,
        x_return_status => result.status,
        x_msg_count => result.msg_count,
        x_msg_data => result.msg_data,
        x_response => response_
      );

      graphite_sync.raise_if_error_status('create_bank', result);
    end create_bank;

  procedure update_bank(
    bank_rec in iby_ext_bankacct_pub.extbank_rec_type,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    begin
      iby_ext_bankacct_pub.update_ext_bank(
        p_api_version => 1.0,
        p_init_msg_list => fnd_api.g_true,
        p_ext_bank_rec => bank_rec,
        x_return_status => result.status,
        x_msg_count => result.msg_count,
        x_msg_data => result.msg_data,
        x_response => response_
      );

      graphite_sync.raise_if_error_status('update_bank', result);

      bank_output.bank_id := bank_rec.bank_id;
    end update_bank;

  procedure create_bank_branch(
    branch_rec in iby_ext_bankacct_pub.extbankbranch_rec_type,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is 
    response_ iby_fndcpt_common_pub.result_rec_type;
    begin
      iby_ext_bankacct_pub.create_ext_bank_branch(
        p_api_version => 1.0,
        p_init_msg_list => fnd_api.g_true,
        p_ext_bank_branch_rec => branch_rec,
        x_branch_id => bank_output.branch_id,
        x_return_status => result.status,
        x_msg_count => result.msg_count,
        x_msg_data => result.msg_data,
        x_response => response_
      );

      graphite_sync.raise_if_error_status('create_bank_branch', result);
  end create_bank_branch;

  procedure update_bank_branch(
    branch_rec in out iby_ext_bankacct_pub.extbankbranch_rec_type,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    begin
      iby_ext_bankacct_pub.update_ext_bank_branch(
        p_api_version => 1.0,
        p_init_msg_list => fnd_api.g_true,
        p_ext_bank_branch_rec => branch_rec,
        x_return_status => result.status,
        x_msg_count => result.msg_count,
        x_msg_data => result.msg_data,
        x_response => response_
      );

      graphite_sync.raise_if_error_status('update_bank_branch', result);

      bank_output.branch_id := branch_rec.branch_party_id;
  end update_bank_branch;

  procedure create_bank_acct(
    bank_acct_rec in out iby_ext_bankacct_pub.extbankacct_rec_type,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    begin
      iby_ext_bankacct_pub.create_ext_bank_acct(
        p_api_version => 1.0,
        p_init_msg_list => fnd_api.g_true,
        p_ext_bank_acct_rec => bank_acct_rec,
        x_acct_id => bank_output.bank_acct_id,
        x_return_status => result.status,
        x_msg_count => result.msg_count,
        x_msg_data => result.msg_data,
        x_response => response_
      );

      graphite_sync.raise_if_error_status('create_bank_acct', result);
  end create_bank_acct;

  procedure update_bank_acct(
    bank_acct_rec in out iby_ext_bankacct_pub.extbankacct_rec_type,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    begin
      iby_ext_bankacct_pub.update_ext_bank_acct(
        p_api_version => 1.0,
        p_init_msg_list => fnd_api.g_true,
        p_ext_bank_acct_rec => bank_acct_rec,
        x_return_status => result.status,
        x_msg_count => result.msg_count,
        x_msg_data => result.msg_data,
        x_response => response_
      );

      graphite_sync.raise_if_error_status('update_bank_acct', result);

      bank_output.bank_acct_id := bank_acct_rec.bank_account_id;
  end update_bank_acct;

  procedure create_inter_bank_acct(
    intermediary_bank_rec in out iby_ext_bankacct_pub.intermediaryacct_rec_type,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    begin
      iby_ext_bankacct_pub.create_intermediary_acct(
        p_api_version => 1.0,
        p_init_msg_list => fnd_api.g_true,
        p_intermed_acct_rec => intermediary_bank_rec,
        x_intermediary_acct_id => bank_output.intermediary_acct_id,
        x_return_status => result.status,
        x_msg_count => result.msg_count,
        x_msg_data => result.msg_data,
        x_response => response_
      );

      graphite_sync.raise_if_error_status('create_inter_bank_acct', result);
  end create_inter_bank_acct;

  procedure update_inter_bank_acct(
    inter_bank_rec in out iby_ext_bankacct_pub.intermediaryacct_rec_type,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    response_ iby_fndcpt_common_pub.result_rec_type;
    begin
      iby_ext_bankacct_pub.update_intermediary_acct(
        p_api_version => 1.0,
        p_init_msg_list => fnd_api.g_true,
        p_intermed_acct_rec => inter_bank_rec,
        x_return_status => result.status,
        x_msg_count => result.msg_count,
        x_msg_data => result.msg_data,
        x_response => response_
      );

      graphite_sync.raise_if_error_status('update_inter_bank_acct', result);

      bank_output.intermediary_acct_id := inter_bank_rec.intermediary_acct_id;
  end update_inter_bank_acct;


  -- Determines whether or not a given bank account needs to be linked to the site
  function bank_acct_needs_to_be_linked(
    site_id in number,
    bank_output in bank_result_t
  ) return boolean is
    acct_id number;
    begin
      select a.bank_account_num
      into acct_id
      from iby_ext_bank_accounts a
             join iby_account_owners b
                  on a.ext_bank_account_id = b.ext_bank_account_id
             join hz_party_sites c
                  on b.account_owner_party_id = c.party_id
      where c.party_site_id = site_id
            and a.bank_id = bank_output.bank_id
            and a.branch_id = bank_output.branch_id
        and a.ext_bank_account_id = bank_output.bank_acct_id;

    return false;

    exception
      when no_data_found then
        return true;
      when others then
        raise;

  end bank_acct_needs_to_be_linked;

function bank_acct_needs_joint_owner(
    bank_acct_rec in out iby_ext_bankacct_pub.extbankacct_rec_type,
    vendor_party_id in number
) return boolean is
    api_result graphite_sync.api_result_t;
    response iby_fndcpt_common_pub.result_rec_type;
  begin
    iby_ext_bankacct_pub.check_bank_acct_owner(
      p_api_version        => 1.0,
      p_init_msg_list       => fnd_api.g_true,
      p_bank_acct_id        => bank_acct_rec.bank_account_id,
      p_acct_owner_party_id => vendor_party_id,
      x_return_status       => api_result.status,
      x_msg_count           => api_result.msg_count,
      x_msg_data            => api_result.msg_data,
      x_response            => response
   );
    -- Dont raise if error status because this API returns an error if the bank account owner is not the current party
    -- Not really sure how to catch a potetntial error from this API call, but I will just pray that it doesnt return an error
    return api_result.status = fnd_api.g_ret_sts_error;
  end bank_acct_needs_joint_owner;

--  procedure process_end_date_old_banks(
--    supplier_output in out graphite_sync_supplier.supplier_output_t,
--    site_output in graphite_sync_site.site_output_t,
--    result in out nocopy graphite_sync.api_result_t
--  ) is
--    cursor old_banks is
--      
--  begin
--    -- Find all banks that are connected to the site
--    -- If the bank that we just created has a different id then the one we just created, we need to end date them
--    -- Once they are end dated, we need to take those and then see if there are any existing invoices on them
--    -- If there are, they need to be moved to the new bank.
--  end process_end_date_old_banks;


  -- Gonna be honest... this proc is kinda magic so good luck debugging
  procedure process_assign_bank_to_site(
    supplier_output in out graphite_sync_supplier.supplier_output_t,
    site_output in graphite_sync_site.site_output_t,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    payee_rec               iby_disbursement_setup_pub.payeecontext_rec_type;
    assignment_attrs  iby_fndcpt_setup_pub.pmtinstrassignment_rec_type;
    bank_rec  iby_ext_bank_accounts%rowtype;
    response_            iby_fndcpt_common_pub.result_rec_type;
    org_id              varchar2(100);
    party_site_id       number;
  begin
    select assa.org_id
      into org_id
    from ap_suppliers aps,
         ap_supplier_sites_all assa
    where aps.vendor_id = assa.vendor_id
      and aps.vendor_id = supplier_output.vendor_id
      and assa.vendor_site_id = site_output.vendor_site_id;

    payee_rec.supplier_site_id := site_output.vendor_site_id;
    payee_rec.party_id := supplier_output.party_id;
    payee_rec.party_site_id := site_output.party_site_id;
    payee_rec.payment_function := 'PAYABLES_DISB';
    payee_rec.org_id := org_id;
    payee_rec.org_type := 'OPERATING_UNIT';
    assignment_attrs.instrument.instrument_type := 'BANKACCOUNT';
    assignment_attrs.instrument.instrument_id := bank_output.bank_acct_id;
    assignment_attrs.priority := 1;
    assignment_attrs.start_date := sysdate;

    iby_disbursement_setup_pub.set_payee_instr_assignment(
      p_api_version => 1.0,
      p_init_msg_list => fnd_api.g_true,
      p_commit => fnd_api.g_false,
      p_payee => payee_rec,
      p_assignment_attribs => assignment_attrs,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data,
      x_assign_id => bank_output.external_payee_id,
      x_response => response_
    );

    graphite_sync.raise_if_error_status('process_assign_bank_to_site', result);
  end;

  procedure process_bank_acct_joint_owner(
    bank_acct_rec in out iby_ext_bankacct_pub.extbankacct_rec_type,
    vendor_party_id in number,
    result in out nocopy graphite_sync.api_result_t
  ) is
    joint_owner_id number;
    response iby_fndcpt_common_pub.result_rec_type;
  begin
    iby_ext_bankacct_pub.add_joint_account_owner(
      p_api_version => 1.0,
      p_init_msg_list => fnd_api.g_true,
      p_bank_account_id => bank_acct_rec.bank_account_id,
      p_acct_owner_party_id => vendor_party_id,
      x_joint_acct_owner_id => joint_owner_id,
      x_return_status => result.status,
      x_msg_count => result.msg_count,
      x_msg_data => result.msg_data,
      x_response => response
    );

    graphite_sync.raise_if_error_status('process_bank_acct_joint_owner', result);
  end;

  procedure process_bank_from_json(
    bank_json in pljson,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    bank_rec iby_ext_bankacct_pub.extbank_rec_type;
    begin
      bank_rec := graphite_sync_converter.convert_bank_json(bank_json);
      set_existing_bank_id(bank_rec, result);

      if bank_is_update(bank_rec) then
        -- unset_bank_keys(bank_rec);
        update_bank(bank_rec, bank_output, result);
      else
        create_bank(bank_rec, bank_output, result);
      end if;
    end process_bank_from_json; 

  procedure process_bank_branch_from_json(
    bank_branch_json in pljson,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    branch_rec iby_ext_bankacct_pub.extbankbranch_rec_type;
    begin
      branch_rec := graphite_sync_converter.convert_bank_branch_json(bank_branch_json);
      branch_rec.bank_party_id := bank_output.bank_id;
      set_existing_branch_id(branch_rec, result);

      if branch_is_update(branch_rec) then
        -- unset_branch_keys(branch_rec);
        branch_rec.bank_party_id := bank_output.bank_id;
        update_bank_branch(branch_rec, bank_output, result);
      else
        create_bank_branch(branch_rec, bank_output, result);
      end if;
    end process_bank_branch_from_json;

  procedure process_bank_acct_from_json(
    bank_acct_json in pljson,
    bank_output in out bank_result_t,
    acct_owner_id in number,
    result in out nocopy graphite_sync.api_result_t
  ) is
    bank_acct_rec iby_ext_bankacct_pub.extbankacct_rec_type;
    begin
      bank_acct_rec := graphite_sync_converter.convert_bank_acct_json(bank_acct_json);
      bank_acct_rec.bank_id := bank_output.bank_id;
      bank_acct_rec.branch_id := bank_output.branch_id;
      set_existing_bank_acct_id(bank_acct_rec, result);

      if bank_acct_is_update(bank_acct_rec) then
        if bank_acct_needs_joint_owner(bank_acct_rec => bank_acct_rec, vendor_party_id => acct_owner_id) then
          process_bank_acct_joint_owner(
            bank_acct_rec => bank_acct_rec,
            vendor_party_id => acct_owner_id,
            result => result
          );
        end if;
        bank_acct_rec.bank_id := bank_output.bank_id;
        bank_acct_rec.branch_id := bank_output.branch_id;
        update_bank_acct(bank_acct_rec, bank_output, result);
      else
        bank_acct_rec.acct_owner_party_id := acct_owner_id;
        create_bank_acct(bank_acct_rec, bank_output, result);
      end if;
    end process_bank_acct_from_json;

  procedure process_intermed_from_json(
    inter_bank_json in pljson,
    bank_output in out bank_result_t,
    result in out nocopy graphite_sync.api_result_t
  ) is
    inter_bank_rec iby_ext_bankacct_pub.intermediaryacct_rec_type;
    begin
      inter_bank_rec := graphite_sync_converter.convert_inter_bank_json(inter_bank_json);
      set_existing_inter_bank_id(inter_bank_rec, result);

      if inter_bank_is_update(inter_bank_rec) then
        -- unset_intermediary_keys(inter_bank_rec);
        update_inter_bank_acct(inter_bank_rec, bank_output, result);
      else
        create_inter_bank_acct(inter_bank_rec, bank_output, result);
      end if;
    end process_intermed_from_json;

  procedure update_return_json(
    return_json in out pljson,
    bank_output in bank_result_t,
    keys in graphite_sync.r_return_keys,
    instance_id in varchar2
  ) is
    begin
      return_json.put(keys.bank_id_key, bank_output.bank_id);
      return_json.put(keys.branch_id_key, bank_output.branch_id);
      return_json.put(keys.bank_account_id_key, bank_output.bank_acct_id);
      return_json.put(keys.intermediary_bank_id_key, bank_output.intermediary_acct_id);
      return_json.put(keys.ext_payment_assign_id_key, bank_output.external_payee_id);

      if instance_id is not null then
        return_json.put(graphite_sync.instance_id_key, instance_id);
      end if;

    end update_return_json;

  procedure process_bank_objects_from_json(
    bank_obj_json in pljson,
    bank_output in out bank_result_t,
    supplier_output in out graphite_sync_supplier.supplier_output_t,
    site_output in out graphite_sync_site.site_output_t,
    keys in graphite_sync.r_return_keys,
    return_json in out pljson
  ) is
    parser graphite_sync_json_parser;
    bank_json pljson;
    branch_json pljson;
    bank_acct_json pljson;
    inter_bank_json pljson;
    instance_id varchar2(100);
    result graphite_sync.api_result_t;
  begin
    parser := graphite_sync_json_parser(json_data => bank_obj_json);
    bank_json := parser.get_object('bank');
    branch_json := parser.get_object('branch');
    bank_acct_json := parser.get_object('bank_acct');

    process_bank_from_json(
      bank_json => bank_json, 
      bank_output => bank_output, 
      result => result
    );
    process_bank_branch_from_json(
      bank_branch_json => branch_json, 
      bank_output => bank_output, 
      result => result
    );
    process_bank_acct_from_json(
      bank_acct_json => bank_acct_json, 
      bank_output => bank_output, 
      acct_owner_id => supplier_output.party_id,
      result => result
    );

    if bank_obj_json.exist('inter_bank') then
      inter_bank_json := parser.get_object('inter_bank');
      process_intermed_from_json(
        inter_bank_json => inter_bank_json, 
        bank_output => bank_output, 
        result => result
      );
    end if;

    if bank_acct_needs_to_be_linked(site_id => site_output.vendor_site_id, bank_output => bank_output) then
      process_assign_bank_to_site(
        supplier_output => supplier_output, 
        site_output => site_output, 
        bank_output => bank_output, 
        result => result
      );
    end if;

    if bank_obj_json.exist(graphite_sync.instance_id_key) then
      instance_id := parser.get_string(graphite_sync.instance_id_key);
    end if;
    
    update_return_json(
      return_json => return_json,
      bank_output => bank_output, 
      keys => keys, 
      instance_id => instance_id
    );

  end process_bank_objects_from_json;


end graphite_sync_bank;
/
show err
