create or replace type body apps.graphite_sync_json_parser as
    constructor function graphite_sync_json_parser(
        json_data in pljson,
        is_update boolean default false
    ) return self as result is
    begin
        self.json_data := json_data;
        self.is_update_ := case is_update when true then 1 else 0 end;
        return;
    end;

    member procedure set_update_flag is
    begin
        self.is_update_ := 1;
    end;
    
    member function get_string(k varchar2, nullable boolean default false) return varchar2 is
        e pljson_element;
    begin
        e := self.json_data.get(k);

        if not self.json_data.exist(k) or e is null or e.is_null then
          return case when self.is_update and nullable then fnd_api.g_null_char else null end;
        end if;

        if e.get_type != 'string' then
            graphite_sync_json_utils.raise_wrong_type(e.get_type, 'string', k);
        end if;

        if e.get_string = graphite_sync_json_utils.null_char_placeholder then
            return fnd_api.g_null_char;
        end if;

        return e.get_string;
    end;
    
    member function get_number(k varchar2, nullable boolean default false) return number is
        e pljson_element;
    begin
        e := self.json_data.get(k);

        if not self.json_data.exist(k) or e is null or e.is_null then
          return case when self.is_update and nullable then fnd_api.g_null_num else null end;
        end if;
        
        if e.get_type = 'number' then
            return e.get_number;
        elsif e.get_type = 'string' then
            return to_number(e.get_string);
        else
            graphite_sync_json_utils.raise_wrong_type(e.get_type, 'number (string or number)', k);
        end if;

        exception
        when others then
          if sqlcode = -6502 then
            raise_application_error(-20001, 'Key "' || k || '": Wanted number but got invalid number string: ' || e.get_string);
            return null;
          else
            raise;
          end if;
    end;
    
    member function get_date(k varchar2, nullable boolean default false) return date is
        e pljson_element;
        val varchar2(50);
    begin
        e := self.json_data.get(k);

        if not self.json_data.exist(k) or e is null or e.is_null then
          return case when self.is_update and nullable then fnd_api.g_null_date else null end;
        end if;
        
        if e.get_type != 'string' then
            graphite_sync_json_utils.raise_wrong_type(e.get_type, 'date (string)', k);
        end if;
        
        val := e.get_string;
        return to_date(val, 'MM-DD-YYYY');
    exception
        when others then
            graphite_sync_json_utils.raise_wrong_type('invalid date string: ' || val, 'date format mm-dd-yyyy', k);
            return null;
    end;

    member function get_object(k varchar2) return pljson is
        e pljson_element;
    begin
        if not self.json_data.exist(k) then
          return null;
        end if;

        e := self.json_data.get(k);
        
        if e.get_type != 'object' then
            graphite_sync_json_utils.raise_wrong_type(e.get_type, 'object', k);
        end if;
        
        return pljson(e);
    end;

    member function get_bool(k varchar2) return boolean is
      e pljson_element;
    begin
      e := self.json_data.get(k);
      if not self.json_data.exist(k) or e is null or e.is_null then
        return false;
      end if;
      
      if e.get_type != 'bool' then
          graphite_sync_json_utils.raise_wrong_type(e.get_type, 'bool', k);
      end if;
      
      return e.get_bool;
    end;

    member function get_array(k varchar2) return pljson_list is
        e pljson_element;
    begin
        if not self.json_data.exist(k) then
          return pljson_list();
        end if;

        e := self.json_data.get(k);
        
        if e is null or e.get_type != 'array' then
            graphite_sync_json_utils.raise_wrong_type(e.get_type, 'array', k);
        end if;
        
        return pljson_list(e);
    end;

    member function is_update return boolean is
    begin
      return case self.is_update_ when 1 then true else false end;
    end;

end;
/
show err
