import { webcrypto as crypto } from 'node:crypto';
import { AUTH_KEY_OFFSET_BYTES, CONTENT_KEY_OFFSET_BYTES, KEY_LENGTH_BYTES, META_KEY_OFFSET_BYTES, } from './constants.js';
import * as u8 from './u8.js';
export class Keychain {
    salt;
    fragment;
    password;
    passwordBits;
    privateKey;
    publicKey;
    peerPublicKey;
    initialKey;
    intermediateKey;
    rawBits;
    metaKey;
    authKey;
    rawContentKey;
    async keygen() {
        if (this.password) {
            await this.genPasswordBits();
        }
        await this.genInitialKey();
        await this.genIntermediateKey();
        await this.genRawBits();
        await this.genMetaKey();
        await this.genAuthKey();
        this.genContentKey();
    }
    reset() {
        this.passwordBits = undefined;
        this.initialKey = undefined;
        this.intermediateKey = undefined;
        this.rawBits = undefined;
        this.metaKey = undefined;
        this.authKey = undefined;
        this.rawContentKey = undefined;
    }
    setPassword(password) {
        this.password = password;
        this.reset();
    }
    setFragment(fragment) {
        this.fragment = u8.fromBasedString('base64-url', fragment);
    }
    async setPeerPublicKeyHex(pubKey) {
        this.peerPublicKey = await crypto.subtle.importKey('raw', u8.fromBasedString('hex', pubKey), {
            name: 'ECDH',
            namedCurve: 'P-384',
        }, false, []);
    }
    async setPrivateKeyJwk(privKey) {
        this.privateKey = await crypto.subtle.importKey('jwk', privKey, {
            name: 'ECDH',
            namedCurve: 'P-384',
        }, false, ['deriveBits']);
    }
    async genKeyPair() {
        const kp = await crypto.subtle.generateKey({
            name: 'ECDH',
            namedCurve: 'P-384',
        }, true, ['deriveBits']);
        this.privateKey = kp.privateKey;
        this.publicKey = kp.publicKey;
    }
    async getPublicKeyHex() {
        if (!this.publicKey)
            throw new Error('Public key not available');
        const raw = await crypto.subtle.exportKey('raw', this.publicKey);
        return u8.toBasedString('hex', raw);
    }
    async getPrivateKeyJwk() {
        if (!this.privateKey)
            throw new Error('Private key not available');
        const raw = await crypto.subtle.exportKey('jwk', this.privateKey);
        return raw;
    }
    async genSharedSecret() {
        if (!this.privateKey)
            throw new Error('Private key not available');
        if (!this.peerPublicKey)
            throw new Error('Peer public key not available');
        return await crypto.subtle.deriveBits({
            name: 'ECDH',
            public: this.peerPublicKey,
        }, this.privateKey, 256);
    }
    async genPasswordBits() {
        if (!this.salt)
            throw new Error('Salt not set');
        const passwordKey = await crypto.subtle.importKey('raw', new TextEncoder().encode(this.password), 'PBKDF2', false, [
            'deriveBits',
        ]);
        this.passwordBits = await crypto.subtle.deriveBits({
            name: 'PBKDF2',
            salt: this.salt,
            iterations: 100000,
            hash: 'SHA-256',
        }, passwordKey, 256);
    }
    async genInitialKey() {
        const usageAlgorithm = this.passwordBits
            ? {
                name: 'HMAC',
                hash: { name: 'SHA-256' },
                length: 256,
            }
            : 'HKDF';
        const usage = this.passwordBits ? ['sign'] : ['deriveBits'];
        if (this.fragment) {
            this.initialKey = await crypto.subtle.importKey('raw', this.fragment, usageAlgorithm, false, usage);
        }
        else {
            if (!this.privateKey || !this.peerPublicKey)
                throw new Error('Shared Secret mode but keys not supplied');
            this.initialKey = await crypto.subtle.importKey('raw', await this.genSharedSecret(), usageAlgorithm, false, usage);
        }
    }
    async genIntermediateKey() {
        if (!this.initialKey)
            throw new Error('Initial key not available');
        if (this.passwordBits) {
            const signedPassword = await crypto.subtle.sign('HMAC', this.initialKey, this.passwordBits);
            this.intermediateKey = await crypto.subtle.importKey('raw', signedPassword, 'HKDF', false, ['deriveBits']);
        }
        else {
            this.intermediateKey = this.initialKey;
        }
    }
    async genRawBits() {
        if (!this.salt)
            throw new Error('Salt not set');
        if (!this.intermediateKey)
            throw new Error('Initial key not available');
        this.rawBits = await crypto.subtle.deriveBits({
            name: 'HKDF',
            salt: this.salt,
            info: new TextEncoder().encode('trebuchet'),
            hash: 'SHA-256',
        }, this.intermediateKey, 3 * KEY_LENGTH_BYTES * 8);
    }
    async genMetaKey() {
        if (!this.rawBits)
            throw new Error('Raw bits not available');
        this.metaKey = await crypto.subtle.importKey('raw', this.rawBits.slice(META_KEY_OFFSET_BYTES, META_KEY_OFFSET_BYTES + KEY_LENGTH_BYTES), {
            name: 'AES-GCM',
            length: KEY_LENGTH_BYTES * 8,
        }, false, ['encrypt', 'decrypt']);
    }
    async genAuthKey() {
        if (!this.rawBits)
            throw new Error('Raw bits not available');
        this.authKey = await crypto.subtle.importKey('raw', this.rawBits.slice(AUTH_KEY_OFFSET_BYTES, AUTH_KEY_OFFSET_BYTES + KEY_LENGTH_BYTES), {
            name: 'HMAC',
            hash: { name: 'SHA-256' },
            length: KEY_LENGTH_BYTES * 8,
        }, true, ['sign']);
    }
    genContentKey() {
        if (!this.rawBits)
            throw new Error('Raw bits not available');
        this.rawContentKey = new Uint8Array(this.rawBits.slice(CONTENT_KEY_OFFSET_BYTES, CONTENT_KEY_OFFSET_BYTES + KEY_LENGTH_BYTES));
    }
    async getAuthKeyHex() {
        if (!this.authKey)
            throw new Error('Auth key not available');
        const raw = await crypto.subtle.exportKey('raw', this.authKey);
        return u8.toBasedString('hex', raw);
    }
}
