import { getRandomValues } from 'node:crypto';
const hex = '0123456789abcdef'; // 1 char = 4 bits, 1 byte -> 2 chars
const b32 = 'abcdefghijklmnopqrstuvwxyz234567'; // 1 char = 5 bits, 5 chars -> 8 bytes
const b32js = '0123456789abcdefghjkmnpqrtuvwxyz'; // from github.com/agnoster/base32-js
const b64 = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/'; // 1 char = 6 bits, 3 bytes -> 4 chars
const b64url = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_'; // 1 char = 6 bits, 3 bytes -> 4 chars
const defs = {
    hex: { bytes: 1, chars: 2, bits: 4, alpha: hex, caseInsensitive: true },
    base32: { bytes: 5, chars: 8, bits: 5, alpha: b32, pad: '=', caseInsensitive: true },
    'base32-js': { bytes: 5, chars: 8, bits: 5, alpha: b32js, caseInsensitive: true },
    base64: { bytes: 3, chars: 4, bits: 6, alpha: b64, pad: '=' },
    'base64-url': { bytes: 3, chars: 4, bits: 6, alpha: b64url },
};
export function fromRandom(len) {
    const u8a = new Uint8Array(len);
    return getRandomValues(u8a);
}
export function fromBasedString(typ, input) {
    const c = defs[typ];
    const inStr = c.caseInsensitive ? input.toLowerCase() : input;
    const len = Math.ceil(inStr.length / c.chars) * c.bytes; // Round up to next block
    const res = new Uint8Array(len);
    let bitCount = 0;
    // Process each block
    for (let i = 0; i < inStr.length / c.chars; i++) {
        let val = 0n;
        const s = inStr.substring(i * c.chars, (i + 1) * c.chars);
        for (let j = 0; j < c.chars; j++) {
            if (s[j] && s[j] != c.pad) {
                bitCount += c.bits;
                const idx = c.alpha.indexOf(s[j]);
                if (idx < 0) {
                    throw new Error(`Unexpected character for ${typ}: ${s[j]}`);
                }
                val += BigInt(idx) << BigInt(c.bits * (c.chars - 1 - j));
            }
        }
        const bytes = bigIntToBytes(val, 8);
        while (bytes.length < c.bytes) {
            bytes.unshift(0);
        }
        res.set(bytes, i * c.bytes);
    }
    return res.slice(0, Math.floor(bitCount / 8));
}
export function toBasedString(typ, input, suppressPadding) {
    const c = defs[typ];
    const u8 = input instanceof Uint8Array ? input : new Uint8Array(input);
    const blocks = Math.ceil(u8.byteLength / c.bytes);
    let res = '';
    let bitCount = 0;
    for (let i = 0; i < blocks; i++) {
        let val = 0n;
        for (let j = 0; j < c.bytes; j++) {
            const offset = i * c.bytes + j;
            if (offset < u8.length) {
                bitCount += 8;
                val += BigInt(u8[offset]) << BigInt(8 * (c.bytes - 1 - j));
            }
        }
        const indexes = bigIntToBytes(val, c.bits);
        const chars = indexes.map((idx) => c.alpha[idx]);
        while (chars.length < c.chars) {
            chars.unshift(c.alpha[0]);
        }
        res += chars.join('');
    }
    res = res.substring(0, Math.ceil(bitCount / c.bits));
    if (c.pad && !suppressPadding) {
        while (res.length % c.chars) {
            res += c.pad;
        }
    }
    if (res.match(/\0/g)) {
        throw new Error('Null check fail');
    }
    return res;
}
function bigIntToBytes(num, bits) {
    const arr = [];
    const mask = BigInt(2 ** bits);
    while (num > 0n) {
        const v = num & (mask - 1n);
        arr.unshift(Number(v));
        num = (num - v) / mask;
    }
    return arr;
}
