import { webcrypto as crypto } from 'node:crypto';
import { encryptStream as encryptStreamTransform, encryptedSize } from './ece.js';
import { fileStream, transformStream, StreamSlicer, stringStream } from './streams.js';
import { ZipStreamController } from './zip.js';
import { Keychain } from './keychain.js';
import retryWrap from 'fetch-retry';
import * as u8 from './u8.js';
import { ResponseError, } from './api/index.js';
import { FileOrgQuotaError, FileTransferLimitError, FileUserQuotaError } from './errors.js';
import { ReadableStream } from 'node:stream/web';
export class BaseUpload extends Keychain {
    sdk;
    progressCb;
    files = [];
    archiveName = `shared_files_${new Date()
        .toISOString()
        .replace(/T/g, '_')
        .replace(/[:.]/g, '-')}.zip`;
    uploadMetadata;
    abortController = new AbortController();
    encryptedSize = 0;
    bytesDone = 0;
    startTime = 0;
    uploadData = {};
    fetch = retryWrap(globalThis.fetch, {
        retries: 30,
        retryOn: function (attempt, error, response) {
            if (error !== null) {
                if (error.name === 'AbortError') {
                    return false;
                }
                return true;
            }
            else if ([429, 503].includes(response?.status)) {
                return true;
            }
            return false;
        },
        retryDelay: function (attempt) {
            return Math.min(Math.pow(2, attempt) * 125, 2000);
        },
    });
    constructor(sdk, progressCb) {
        super();
        this.sdk = sdk;
        this.progressCb = progressCb;
    }
    abort() {
        this.abortController.abort();
    }
    addFile(file) {
        if (Array.isArray(file)) {
            this.files.push(...file);
        }
        else {
            this.files.push(file);
        }
    }
    setArchiveName(name) {
        this.archiveName = name;
    }
    async encryptMetadata() {
        if (!this.salt)
            throw new Error('Salt not set');
        if (!this.metaKey)
            throw new Error('Meta key not available');
        const data = await crypto.subtle.encrypt({
            name: 'AES-GCM',
            iv: this.salt.slice(0, 12),
            tagLength: 128,
        }, this.metaKey, new TextEncoder().encode(JSON.stringify(this.uploadMetadata)));
        return data;
    }
    async uploadMetaGCS(config) {
        await this.fetch(config.meta, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/trebuchet-encrypted-meta',
            },
            body: await this.encryptMetadata(),
        });
    }
    async uploadMetaS3(config) {
        await this.fetch(config.meta, {
            method: 'PUT',
            body: await this.encryptMetadata(),
        });
    }
    async uploadStartGCS(config) {
        const resp = await this.fetch(config.content[0], {
            method: 'POST',
            headers: {
                'Content-Length': '0',
                'Content-Type': 'application/trebuchet-encrypted-content',
                'x-goog-resumable': 'start',
            },
        });
        const url = resp.headers.get('Location');
        if (!url)
            throw new Error('No upload location returned from GCS');
        return url;
    }
    async uploadChunkGCS(reader, url) {
        const streamState = await reader.read();
        if (streamState.done) {
            return true;
        }
        // Calculate header values
        const buf = streamState.value;
        const chunkSize = streamState.value.length;
        const chunkFirstByte = this.bytesDone;
        const chunkLastByte = this.bytesDone + chunkSize - 1;
        const contentRange = `bytes ${chunkFirstByte}-${chunkLastByte}/${this.encryptedSize}`;
        const res = await this.fetch(url, {
            method: 'PUT',
            headers: {
                'Content-Length': chunkSize,
                'Content-Range': contentRange,
            },
            body: buf,
            signal: this.abortController.signal,
        });
        this.bytesDone += chunkSize;
        if (this.progressCb) {
            const ms = performance.now() - this.startTime;
            const rate = (this.bytesDone * 8) / (ms * 1000);
            this.progressCb(this.bytesDone, this.encryptedSize, rate);
        }
        // Check response
        if (res.status === 308) {
            // Successful but incomplete
            const remoteBytesSent = parseInt(res.headers.get('range').split('-')[1]);
            if (remoteBytesSent + 1 === this.bytesDone) {
                return false;
            }
            else {
                throw new Error('GCS byte count mismatch');
            }
        }
        else if (res.status === 200) {
            return true;
        }
        throw new Error('GCS unexpected response');
    }
}
// Either file or message
export class FileUpload extends BaseUpload {
    sdk;
    file;
    progressCb;
    constructor(sdk, file, progressCb) {
        super(sdk, progressCb);
        this.sdk = sdk;
        this.file = file;
        this.progressCb = progressCb;
    }
    static async createNew(sdk, params, progressCb) {
        const res = await sdk.filesApi.createFile(params);
        return new FileUpload(sdk, res, progressCb);
    }
    get fileId() {
        return this.file.publicId;
    }
    get metadata() {
        if (!this.uploadMetadata)
            throw new Error('Metadata not available');
        return this.uploadMetadata;
    }
    static async create(sdk, params, progressCb) {
        const file = await sdk.filesApi.createFile(params);
        return new FileUpload(sdk, file, progressCb);
    }
    get totalSizeBytes() {
        return this.files.reduce((acc, val) => acc + val.size, 0) || 0;
    }
    get fileCount() {
        return this.files.length;
    }
    addMessage(subject, message, type = 'application/trebuchet-message') {
        if (this.files.length > 0)
            throw new Error('Cannot add message with files');
        const file = {
            data: message,
            name: subject,
            type: type,
            size: new TextEncoder().encode(message).length, // Account for unicode length
        };
        this.files.push(file);
    }
    async upload() {
        let stream;
        if (this.files.length === 1) {
            this.uploadMetadata = {
                name: this.files[0].name,
                type: this.files[0].type || 'application/octet-stream',
                size: this.files[0].size,
            };
            stream =
                typeof this.files[0].data === 'string' ? stringStream(this.files[0].data) : fileStream(this.files[0].data);
        }
        else {
            this.uploadMetadata = {
                name: this.archiveName,
                type: 'application/zip',
                size: 0,
                files: this.files.map((f) => ({
                    name: f.name,
                    type: f.type || 'application/octet-stream',
                    size: f.size,
                })),
            };
            const zipCtrl = new ZipStreamController(this.files);
            this.uploadMetadata.size = zipCtrl.getSize();
            stream = new ReadableStream(zipCtrl);
        }
        let uploadConfig;
        try {
            uploadConfig = await this.sdk.filesApi.startFileUpload({
                fileId: this.file.publicId,
                fileStartUploadRequest: { size: this.uploadMetadata.size },
            });
        }
        catch (err) {
            if (err instanceof ResponseError && [400, 406].includes(err.response.status)) {
                const body = await err.response.json();
                switch (body.code) {
                    case 'file_org_quota':
                        throw new FileOrgQuotaError();
                    case 'file_user_quota':
                        throw new FileUserQuotaError();
                    case 'file_single_transfer_limit':
                        throw new FileTransferLimitError();
                    default:
                        throw err;
                }
            }
            throw err;
        }
        this.salt = u8.fromBasedString('hex', uploadConfig.salt);
        await this.keygen();
        if (!this.rawContentKey)
            throw new Error('Content key not available');
        const encryptStream = encryptStreamTransform(stream, this.rawContentKey);
        const uploadStream = transformStream(encryptStream, new StreamSlicer(uploadConfig.block));
        this.encryptedSize = encryptedSize(this.uploadMetadata.size);
        const reader = uploadStream.getReader();
        let url;
        let done = false;
        switch (uploadConfig.type) {
            case 'gcs':
                await this.uploadMetaGCS(uploadConfig);
                url = await this.uploadStartGCS(uploadConfig);
                this.startTime = performance.now();
                while (!done) {
                    done = await this.uploadChunkGCS(reader, url);
                }
                break;
            case 's3':
                await this.uploadMetaS3(uploadConfig);
                throw new Error('Not implemented');
        }
        await this.uploadComplete();
    }
    async uploadComplete() {
        await this.sdk.filesApi.verifyFileUpload({
            fileId: this.file.publicId,
            fileVerifyUploadRequest: {
                authKey: await this.getAuthKeyHex(),
                data: this.uploadData,
            },
        });
    }
}
export class InviteUpload extends BaseUpload {
    sdk;
    invite;
    progressCb;
    constructor(sdk, invite, progressCb) {
        super(sdk, progressCb);
        this.sdk = sdk;
        this.invite = invite;
        this.progressCb = progressCb;
    }
    async upload() {
        let stream;
        if (this.files.length === 1) {
            this.uploadMetadata = {
                name: this.files[0].name,
                type: this.files[0].type || 'application/octet-stream',
                size: this.files[0].size,
            };
            stream = fileStream(this.files[0].data);
        }
        else {
            this.uploadMetadata = {
                name: this.archiveName,
                type: 'application/zip',
                size: 0,
                files: this.files.map((f) => ({
                    name: f.name,
                    type: f.type || 'application/octet-stream',
                    size: f.size,
                })),
            };
            const zipCtrl = new ZipStreamController(this.files);
            this.uploadMetadata.size = zipCtrl.getSize();
            stream = new ReadableStream(zipCtrl);
        }
        let uploadConfig;
        try {
            uploadConfig = await this.sdk.invitesApi.startUpload({
                inviteId: this.invite.publicId,
                inviteStartUploadRequest: {
                    size: this.uploadMetadata.size,
                },
            });
        }
        catch (err) {
            if (err instanceof ResponseError && [400, 406].includes(err.response.status)) {
                const body = await err.response.json();
                switch (body.code) {
                    case 'file_org_quota':
                        throw new FileOrgQuotaError();
                    case 'file_user_quota':
                        throw new FileUserQuotaError();
                    case 'file_single_transfer_limit':
                        throw new FileTransferLimitError();
                    default:
                        throw err;
                }
            }
            throw err;
        }
        this.salt = u8.fromBasedString('hex', uploadConfig.salt);
        await this.keygen();
        if (!this.rawContentKey)
            throw new Error('Content key not available');
        const encryptStream = encryptStreamTransform(stream, this.rawContentKey);
        const uploadStream = transformStream(encryptStream, new StreamSlicer(uploadConfig.block));
        this.encryptedSize = encryptedSize(this.uploadMetadata.size);
        const reader = uploadStream.getReader();
        let url;
        let done = false;
        switch (uploadConfig.type) {
            case 'gcs':
                await this.uploadMetaGCS(uploadConfig);
                url = await this.uploadStartGCS(uploadConfig);
                this.startTime = performance.now();
                while (!done) {
                    done = await this.uploadChunkGCS(reader, url);
                }
                break;
            case 's3':
                await this.uploadMetaS3(uploadConfig);
                throw new Error('Not implemented');
        }
        await this.uploadComplete(uploadConfig.fileId);
    }
    async uploadComplete(id) {
        await this.sdk.invitesApi.verifyInviteUpload({
            inviteId: this.invite.publicId,
            inviteVerifyUploadRequest: {
                fileId: id,
                authKey: await this.getAuthKeyHex(),
                publicKey: await this.getPublicKeyHex(),
                data: this.uploadData,
            },
        });
    }
}
