/* Copyright 2018 The Bazel Authors. All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package language

import (
	"github.com/bazelbuild/bazel-gazelle/internal/config"
	"github.com/bazelbuild/bazel-gazelle/internal/resolve"
	"github.com/bazelbuild/bazel-gazelle/internal/rule"
)

// Language describes an extension for Gazelle that provides support for
// a set of Bazel rules.
//
// Languages are used primarily by the fix and update commands. The order
// in which languages are used matters, since languages may depend on
// one another. For example, go depends on proto, since go_proto_libraries
// are generated from metadata stored in proto_libraries.
//
// A single instance of Language is created for each fix / update run. Some
// state may be stored in this instance, but stateless behavior is encouraged,
// especially since some operations may be concurrent in the future.
//
// Tasks languages are used for
//
// * Configuration (embedded interface config.Configurer). Languages may
// define command line flags and alter the configuration in a directory
// based on directives in build files.
//
// * Fixing deprecated usage of rules in build files.
//
// * Generating rules from source files in a directory.
//
// * Resolving library imports (embedded interface resolve.Resolver). For
// example, import strings like "github.com/foo/bar" in Go can be resolved
// into Bazel labels like "@com_github_foo_bar//:go_default_library".
//
// Tasks languages support
//
// * Generating load statements: languages list files and symbols that may
// be loaded.
//
// * Merging generated rules into existing rules: languages provide metadata
// that helps with rule matching, merging, and deletion.
type Language interface {
	config.Configurer
	resolve.Resolver

	// Kinds returns a map of maps rule names (kinds) and information on how to
	// match and merge attributes that may be found in rules of those kinds. All
	// kinds of rules generated for this language may be found here.
	Kinds() map[string]rule.KindInfo

	// Loads returns .bzl files and symbols they define. Every rule generated by
	// GenerateRules, now or in the past, should be loadable from one of these
	// files.
	Loads() []rule.LoadInfo

	// GenerateRules extracts build metadata from source files in a directory.
	// GenerateRules is called in each directory where an update is requested
	// in depth-first post-order.
	//
	// c is the configuration for the current directory.
	// dir is the absolute path to the directory to scan.
	// rel is the relative path to the directory from the repository root.
	// f is the build file. It may be nil. It should not be modified.
	// subdirs is a list of subdirectory names.
	// regularFiles is a list of normal files in the directory.
	// genFiles is a list of generated files, found in outputs of rules.
	// otherEmpty and otherGen are lists of empty and generated rules created
	// by other languages processed before this language.
	//
	// empty is a list of empty rules that may be deleted after merge.
	// gen is a list of generated rules that may be updated or added.
	//
	// Any non-fatal errors this function encounters should be logged using
	// log.Print.
	GenerateRules(c *config.Config, dir, rel string, f *rule.File, subdirs, regularFiles, genFiles []string, otherEmpty, otherGen []*rule.Rule) (empty, gen []*rule.Rule)

	// Fix repairs deprecated usage of language-specific rules in f. This is
	// called before the file is indexed. Unless c.ShouldFix is true, fixes
	// that delete or rename rules should not be performed.
	Fix(c *config.Config, f *rule.File)
}
