const path = require('path');
const morgan = require('morgan');
const express = require('express');
const { createWriteStream } = require('fs');
const cookieParser = require('cookie-parser');
const { get, dbSession, dbAccount } = require('./db');
const { rand } = require('./helper');
const session = require('express-session');
const crypto = require('crypto');

const app = express();
const port = 61111;

app.use(session({
  secret: crypto.randomBytes(20).toString("hex"),
  cookie: {}
}));

const format = '[:date[clf]] :remote-addr ":method :url" :status :response-time ms';
app.use(
  morgan(format),
);
app.use((req, res, next) => {
  const nonce = rand();
  res.locals = { nonce };
  res.set('X-Frame-Options', 'SAMEORIGIN');
  res.set('Cross-Origin-Opener-Policy', 'same-origin');
  res.set('Cross-Origin-Resource-Policy', 'same-origin');
  res.set('Content-Security-Policy', `script-src 'nonce-${nonce}'`);
  next();
});
app.use(cookieParser());
app.use(express.urlencoded({ extended: false }));

app.set('view engine', 'ejs');
app.set('views', './views');

app.use((req, res, next) => {
  function clean(obj) {
    if (!obj) return;
    for (const [key] of Object.entries(obj)) {
      if (obj[key].includes('\'')) return next('Hack detected');
      obj[key] = Buffer.from(obj[key], 'utf-8').toString('ascii');
    }
  }
  clean(req.body);
  clean(req.query);
  clean(req.params);
  next();
});

app.use(async (req, res, next) => {
  const id = req.cookies.id || '';
  const username = await get(dbSession, id);
  if (username) {
    req.user = await get(dbAccount, username);
    if (req.user) req.user = JSON.parse(req.user);
  }
  else req.user = null;
  next();
});

app.use('/static', express.static(path.join(__dirname, 'static')))
app.use('/register', require('./routes/register'));
app.use('/bot', require('./routes/bot'));
app.use('/', require('./routes/index'));

app.use((err, req, res, next) => {
  console.log(err);
  res.status(500).send(err);
});

app.listen(port, () => {
  console.log(`Example app listening on port ${port}`)
})