<?php

use Illuminate\Database\Capsule\Manager as DB;
use MailerSend\MailerSend;
use MailerSend\Helpers\Builder\Recipient;
use MailerSend\Helpers\Builder\EmailParams;
use MailerSend\Helpers\Builder\Variable;
use MailerSend\Helpers\Builder\Attachment;

class EventReminder {

    /**
     * Checks and sends reminders to the event guests
     * @throws JsonException
     * @throws \MailerSend\Exceptions\MailerSendAssertException
     * @throws \MailerSend\Exceptions\MailerSendException
     * @throws \Psr\Http\Client\ClientExceptionInterface
     */
    public function checkReminders()
    {
        /**
         * Collect the guest and event information to send the reminder to
         * We access the database using Illuminate but you can use whatever you want.
         * The following is equivalent to this SQL query:
         *
         * SELECT guests.*, events.event_name, events.data FROM guests JOIN events ON events.id = guests.event_id
         * WHERE guests.reminded = 0 AND events.remind_date <= NOW();
         */
        $guestReminders = DB::table('guests')->select(['guests.*', 'events.event_name', 'events.date'])
            ->join('events', 'events.id', '=', 'guests.event_id')
            ->where('guests.reminded', '=', '0')
            ->where('events.remind_date', '<=', 'NOW()')
            ->get();


        // we'll store the emails we want to send in an array that we'll use in the bulk email request
        $emails = [];

        // we'll store the guest ids so that we can update the ones we sent the reminder to
        $sentToGuestIds = [];

        // we loop through the collected database records to setup the recipients and variables
        foreach ($guestReminders as $guestReminder) {

            $eventDate = DateTime::createFromFormat('Y-m-d H:i:s', $guestReminder->date);
            $currentDate = new DateTime();

            // we want to use a subject like "X days left for our event",
            // so we calculate the number of days between the event data and now
            $daysLeft = $currentDate->diff($eventDate);
            $daysLeft = (int)($daysLeft->days + ceil($daysLeft->h / 24));

            $recipients = [
                new Recipient($guestReminder->email, $guestReminder->name)
            ];

            // if the event is 1 day away, we don't want the subject to be '1 days left for the our event'
            // so we use a label for the word "days" which we change to "day" if the event is 1 day away
            $daysLabel = 'days';
            if ($daysLeft == 1) {

                $daysLabel = 'day';
            }

            // we setup the variables for our email
            $variables = [
                new Variable($guestReminder->email, [
                    'guest_name' => $guestReminder->name,
                    'event_name' => $guestReminder->event_name,
                    'date' => $eventDate->format('F jS'),
                    'time' => $eventDate->format('g:ia'),
                    'days_left' => (string)$daysLeft,
                    'days_label' => $daysLabel
                ])
            ];

            // setup the email
            $email = (new EmailParams())
                ->setFrom('your from email address')
                ->setFromName('your from name')
                ->setRecipients($recipients)
                ->setSubject('Only {$days_left} {$days_label} left for {$event_name}!')
                ->setTemplateId('your template id')
                ->setVariables($variables);

            // if we want to also include the schedule, we can do the following:
            /*
                $attachments = [
                    new Attachment(file_get_contents('schedule.jpg'), 'schedule.jpg')
                ];

                $email->setAttachments($attachments);
            */

            $emails[] = $email;

            // we store the id of each guest we sent the reminder to so that we can update its "reminded" field.
            // that way we prevent sending duplicate reminders to the guests
            $sentToGuestIds[] = $guestReminder->id;

            // there is a limit of 500 emails per bulk request, so if we reach 500 emails,
            // we need to break to send the emails we prepared and continue in the next run
            if (count($emails) == 500) {

                break;
            }
        }

        // we check that we have emails to send and then do a request to the bulk email MailerSend endpoint
        if (count($emails) > 0) {

            // init the MailerSend SDK
            $mailersend = new MailerSend(['api_key' => 'your MailerSend token']);

            // send the email
            $mailersend->bulkEmail->send($emails);

            // update the guests that have been reminded
            DB::table('guests')
                ->whereIn('id', $sentToGuestIds)
                ->update(['reminded' => 1]);
        }
    }
}