/*
 * SPDX-FileCopyrightText: Copyright (c) 1993-2024 NVIDIA CORPORATION &
 * AFFILIATES. All rights reserved. SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#pragma once

#include <flashinfer/trtllm/fmha/kernelParams.h>
// Helper to print Data_type
inline const char* dataTypeToString(Data_type dt) {
  switch (dt) {
    case DATA_TYPE_FP16:
      return "FP16";
    case DATA_TYPE_BF16:
      return "BF16";
    case DATA_TYPE_FP32:
      return "FP32";
    case DATA_TYPE_E4M3:
      return "E4M3";
    default:
      return "UNKNOWN";
  }
}

static const struct TllmGenFmhaKernelMetaInfo {
  Data_type mDataTypeQ;
  Data_type mDataTypeKv;
  Data_type mDataTypeO;
  int mTileSizeQ;
  int mTileSizeKv;
  int mStepQ;
  int mStepKv;
  int mHeadDimPerCtaV;
  int mHeadDimQk;
  int mHeadDimV;
  int mSM;
  const char* mFuncName;
  int mSharedMemBytes;
  int mThreadsPerCTA;
  int mQkvLayout;
  int mNumTokensPerPage;
  int mMaskType;
  int mKernelType;
  int mMaxNumHeadsQPerKvInCta;
  int mTileScheduler;
  int mMultiCtasKvMode;
  bool mGroupsHeadsQ;
  bool mReuseSmemKForV;
  bool m2CtaMma;
  const char* sha256;

  void print() const {
    std::cout << "TllmGenFmhaKernelMetaInfo {\n";
    std::cout << "  mDataTypeQ: " << dataTypeToString(mDataTypeQ) << "\n";
    std::cout << "  mDataTypeKv: " << dataTypeToString(mDataTypeKv) << "\n";
    std::cout << "  mDataTypeO: " << dataTypeToString(mDataTypeO) << "\n";
    std::cout << "  mTileSizeQ: " << mTileSizeQ << "\n";
    std::cout << "  mTileSizeKv: " << mTileSizeKv << "\n";
    std::cout << "  mStepQ: " << mStepQ << "\n";
    std::cout << "  mStepKv: " << mStepKv << "\n";
    std::cout << "  mHeadDimPerCtaV: " << mHeadDimPerCtaV << "\n";
    std::cout << "  mHeadDimQk: " << mHeadDimQk << "\n";
    std::cout << "  mHeadDimV: " << mHeadDimV << "\n";
    std::cout << "  mSM: " << mSM << "\n";
    std::cout << "  mFuncName: " << (mFuncName ? mFuncName : "null") << "\n";
    std::cout << "  mSharedMemBytes: " << mSharedMemBytes << "\n";
    std::cout << "  mThreadsPerCTA: " << mThreadsPerCTA << "\n";
    std::cout << "  mQkvLayout: " << mQkvLayout << "\n";
    std::cout << "  mNumTokensPerPage: " << mNumTokensPerPage << "\n";
    std::cout << "  mMaskType: " << mMaskType << "\n";
    std::cout << "  mKernelType: " << mKernelType << "\n";
    std::cout << "  mMaxNumHeadsQPerKvInCta: " << mMaxNumHeadsQPerKvInCta << "\n";
    std::cout << "  mTileScheduler: " << mTileScheduler << "\n";
    std::cout << "  mGroupsHeadsQ: " << std::boolalpha << mGroupsHeadsQ << "\n";
    std::cout << "  mMultiCtasKvMode: " << std::boolalpha << mMultiCtasKvMode << "\n";
    std::cout << "  mReuseSmemKForV: " << std::boolalpha << mReuseSmemKForV << "\n";
    std::cout << "  m2CtaMma: " << std::boolalpha << m2CtaMma << "\n";
    std::cout << "  sha256: " << (sha256 ? sha256 : "null") << "\n";
    std::cout << "}\n";
  }
} sTllmGenFmhaKernelMetaInfos[] = {
    // clang-format off
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "06fa6e5978cb0d5ed2ec654591b5bc7fe3fabfbc5bd29690dc039f8d2746a15f"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "c20701b2701dad7c0c8b10a5ad85d28f701ee1d432fc7298649fe1f2856cad92"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "31fa72526523245bdfb409115f3e8fb64f5b4670387336efd58bd69c17d83817"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "a19266f6b619a6ab1a92d6befe1b65283d56da8aa188470bc5f5a66549ff0e52"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "569e3f91b1c6e66981270a3ffc8e02432abc03d3ce12d888b2a0f297bd8e5738"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "0ef26a988cc53f3106f13d8c36aaee35d8add36ad3e6a9f8073715c20a9e6ac3"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "2618c62778666aa1ca523220ad434c43b8bcf0c9343be958c962e4dfebbd68f3"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "bc6c1a3d676a3eeebe82a6e3e510ac1a551e264027c9b962616cb22d762c31cb"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "1fee3d9a5184482a262ff77a3f4e1a7fa0467d5c4978d058e0fb3bf0da51ed93"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "34a4520e5e7fa4b5db84378e68e55ae14e5c27b8f6d3bfc57543565aceaf13cc"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "811d23e7802b145f4aae906c29365d17a81286d51ea7cea3d704570d69aff342"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "2d1b065c378e3199e727941b89b16ae3a4982e3d66a7ee072848cb57c76a6aa1"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "8590e436d65c2499a9d5840b3fa50d8b465669de94e6b2073f88a12be91c5c28"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "a978aed0b8623015b746062470309ecbc42c3a2eed59c44a66b850f0ff39565f"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "9b7ea05f7c454ba6c5f04e54cd1572f8b2cf86499fb344ded5eafb7b0f3fc47d"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "b14b1940167f1ed3a261fc529297be5c9b53776f493e69f03e05080148da300e"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "d7f6165de6ccfef24ce3a93c7a1b0c03413a0efbec264263235fbf7e03bbfa5a"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "f80baa13ec6d429c5e0c0f1289f5a30d0a63c3eb11a59d6518fcad9a537477f9"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "e897500a6b7143a596044fa367c78fe9535fd532941c20b76461587286ddd07d"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "57bd4edd182324bb2b1c49582388776162ef84c995466020be5065d81ca08470"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "dc3a6a8270d3a04893b8cb8b74a46f6a9a93cd75598fb5789390a7a0840a2d76"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "fd8f382ecec44ae071e091436338ffe584d29c84531792022e3b1780bb231940"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "239e86ba1f0baeeb0f3bf3b11bc6928ef79d572bd8827e7a3a3c12dfa144e80a"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "d158111df93a0238c159bba98606720590115ea52878591c2d7b292d749915d7"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "812f2a59cdb5cae44190c48b942d236d69cf8339b773988377678ca4a4a92cb2"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "bc39aeaae11aa020c83287160b1b6014e2b4fe021257675c572fdbfca05fcf10"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "9fd0b3aa97803ef16f22cbb81e3420674ff4ee615b3b598a27928b0f44f3b624"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "bbe9bc7e879587792db1853cf3fdf1948255aa8747b0ad44ce6ea68d5e7ca104"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "4cb2153897823b532f902716ffbef308a1ebe715f2492b91b7df569ecc6c29b6"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "5fc48366e358e41d2338ee9aef7ee6e9f6cf511c6acaf991471efdc2bae7cb28"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "fdbfff28d8edb6879b67c0ee57934d470385ab457a6530f05a6f410ad0ab2600"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "30cedef28f8f209bfafbceac05afacfec41edfd3440b9439aeb3064fc93c3bc2"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "1a414868ec41132c612339d8c74dc2fad35ebdd1f11b84eaecf80739c13ba022"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "037ca837ef5999fe9008971d0212e89b60db9ecddf7c87dce397400da1593953"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "27f2d359228edd8d7c2bfce7e27894e2ec61beccdfe02a781bf911f5ebe503a6"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "f74802234df6267c462b1297ab22f87548e2b6618f63cf9da55f5847307c122d"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "54f3ee9a91c0af8e1469e386ec38b873576d94f3bb562b3968ded615ea834e0c"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "5dbeb51a0b7b5ee98ce85ff99751319302eecd5082af15cf1d89dfdc41183920"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "59a27801752534cd8a36895bca7fabd0c51812f8ac9870e73d40a539aa692ca7"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "ee9764e7b108694342d43af8e7db6a55d92bf395df82fd5d1c5f313c6fcbe723"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "ff1e596a4f7d6a672564b55148a5a8b7734a4a05d3dcd601d78ccbe816102910"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "a9f93efd08025cd91d7c4de6a2ad65a9d4d922b05d73397df5b7dae0be7bb71e"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "6d11f5c93487b1942825441ab47c4002bc1eea3edf241d61b11fd74de3fa4d01"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "93064e75f33c39099644afe3f24a0eff33ab03e5046bcf699d8a68303e48727d"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "cb255fdd13369a7036afe2c64e9c0a3851e41475be5887f3bc8c8f544664a004"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "16c296b54189d02a2ca17035a0056127b2edf55e61f4e0107c50bb1f1695ea1a"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "da7336b69ec7528a4ff3a18315d883e1dd48a307837bf524781ea5c6e76ae3c6"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "245cf667f2b50c95b01c57af3c2fb007b9b4bb6badeb3b527e06add6c5b9ef69"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "f54bab2171824957f922a45e7b315410b53f3c06321a501c66f073c974b8cede"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "8152a76471a3482423e3b3c6447bfd35097edad393fd7f8a93ba7c18f318b46b"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "41cb62a923db56efa85eec333ff910b987949dc1f092ca58baaf7387a369e236"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "ab02d00ba50c0eb34d099a9ac675297b921b69625440af033124ed3cc10042d3"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "42908561922c0291982793623cb6c99ecbaa212e97de9c00036f496aa9ef7acd"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "99a8ced65f46dbdf8d13cf8b78624aabfeef644c65dc58c69a0bec000e3ce765"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "22aae8fdb51f2dab97480eef3d0b9f93eb9b5aaa2c66b46fbfaecd140c585428"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "4ba686ce4aebf97834158a730730ad801a6baa9cc97bc4ede3250c3009ca0b14"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "76e6237eff56ef4c4ddb62ee03031cf36097fe2ab60d7214dfe11c30ded2a3bb"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "ddd3574d8c9fb759eb2d386167917905d83e005ce2fc15f5159b54dbff6bd32e"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "0b0a711eae8380ea977c20a6f445c975be45a9ecc9aaed6b82191329c4de457f"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "c46ae569b8cdc1e19a7b2cdcaaf98efae89d46ac10de3a8dcaba7b7d3b7e1754"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "5337c4c3425402075c6b489a9d6fd7ae286132abbce1a1b642b38de2d3e34974"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "49eb2b6574ae6ed4e3d86ce8cd8a305e5e96a6d8a4e2bfe33a8248ebe2d2b91e"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "04e1c3a299cee66abd7cd89ab68dbd160b65937fd35c6ba662e42a485f03d051"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "e8932d29b53b01529bc212e9c4e7c9d9620f01630161b3782585882c53f03eaf"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "d2416b17fbaa2c795b60c16b062166bb82e3dca6998b2fdd3d16773ef48c466c"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "84b59bbb7e13f0905dcabea7e64a75762e011c5166c088eff3b342dfbae748ac"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "2ce2741355f5de6308fe5d61abd2d26eb40e09491233708dee0ec3987cec8ad0"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "5a1beb9c12500de35b4260cdd20412afcba66709c3da6083d47ad1a7b527aef7"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "bb0db052ff88bfbbf5074599d82ac594c954bc2668068b77260fbab578f7749d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "76d207a71b38034b8511a1644d4ee2f9432ac6b86ad1280caf7bd005852ff80e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "a3e47fe3064740984c9483c7e3f11b5f1c52b2b185b69ca14668d1a102ee2e11"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "1ac300079334d3858fceddad505e6557e6c35bee09b5ac5606af48ab1c1d4554"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "376893eb34eb3319bd37d049859311cd08dbd71de1cded07a9106342245586da"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "c6269026936043c4041130bb579dc31972ec36fa8c9af9b545d16aa2b303b996"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "cbec9b6543025e5a0cc0918f8fe1e644f1d4e97bb5d5b8b1c30e80a98db2251d"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "616cdc6d1f0ee5fa81f3aee7ec2fe2baab6effacca1844313a016f926431813e"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "5677403cfe8fb7d1223be9a0aadd03849f7863c2fb72b4708f5166a46d24b0db"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "cebfab6399c2feb9082278c1f9348135b8a33666b474a71f39c302233e985ffc"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "224321b048f3e34ca4d91645756dee72638971c228c711bd876a250487d6154e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "dc01a8706a732f184fe4a138231874b9b4a92d891b641a1be84c6688aa2a6919"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "4323c6f5b428600b37e84333616cb768c3f27b17fcd60f263ebdc767783fb4c7"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "c80e5d3524eabcdec602d001501ae63eb3399e2190b2a774b245f7e0baf30f55"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "109cf87c1d8b31d3bdf8880a56c1cd01f68054cb04befba635ffb707310c57be"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "afd97e66299276a460826f9edf95cd01d086d220ac87dcf72552c71bc10271e5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "d41a34801c44eaf0bf7c8b3489555d437a500dfd4d0b0543fcf61f8ce7b68c72"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "1018267bfafcb91b06e678fc6d42724d4ef79f1669b76720b09bfeda0c950329"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "d903fa968e37fa519f84fccca20b3f85899cfb4b2dd108ca82dba741b91b719e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "51c5c6a79e3b69e15ca03df8b6a9da24b9a4bea700538d0443e6f684d6cc6838"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "4c40df655f763a1b7182b15417f99540d5cbc0b2722b782bb5b1debc4214afdd"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "7853f10fb0aea65f847c050deceac0d9e049730ea4246484778b5fd1a2f0ae27"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "fbeb4f3d9cf50389f2093ef9e00604f358ed84917446b8d8609f3bfacd323f8b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "87d72bc1fe3b3391ecf32a2bddb579bd7ababe289791fbd700ac3014a4533875"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "5515784b3699722c476354af6909050a665b0549179cee0b1687971ec4457b14"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "daaaa9801978721a9b10386cef5a6ef28a2396dc3e74441855400864b0751add"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "4c647114210cd1a71bf204107f68958b1cbd29f3d7f688d4ed032771830ae856"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "5adb016db46ff70f0f7c06ea0ffb4c0da188f331bb975747dc3e422fe0faca11"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "e25d44ada0b07f42a34bffcb41784ba2b3d5569125654c12b7e4310c172c31ea"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "d3cfe16e0d0f70f7e69786ff451035e567bd8c11fa2ce214c9ae137bad43a98b"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "ac22a5b75a4134364f258012fbe5eeb71cedcc93646cb96937cb9b323699e0c7"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "0b722badeddb2186c97f563c6e84d1fe382394c0fec9f01ac95badf35f11b16d"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "b5b606dd330b92c13f2eb80376c76fbf9562244c90603737f54d233ccebe0b6d"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "7e723de9604b803cc9af1ecdbf739596c8bc10c3a9d228436d320c662f643086"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "469f817a1679bc7f86b7aedf299924747a914e252b1e5b31c41c3225d9c0ea49"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "1966fa6aad45352c8a027cbe399b5c2b60e69fc27814bbb182154ac9071ae0f8"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "d919ee41ad1ffd27b32155c11bfb3cfb7facdd38ceecb04d7cb612002dfebf53"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "eabe31d3fb4187ff327194456be8c529047d1fe58572f82dc114f51e509fb3d5"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "31604f918bb0341785e2fd980dc14d9778bf8dbd525022871d342e23729697d5"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "6df3e0e258056b31b0828b458c59a552d4efe8089f2aa7d4cd787bd852925fc1"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "8dc74c2981afbb88657524168d139b2d12417130fb24a8c241c9a6407e6764be"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "cd4ae09ed4ba9091714390bc72aaef59c23fb31ba8fcf1d9f477c7ca5c49e54a"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "52cc71524f96b6194c71fc3462a8e71ecbb45491ee963d80d082765e327add39"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "d8ad4d8a28df85aa17d77a05e27d4b4214210b56b15a7cbd15e8c165353f5a67"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "4676c28ab436b44947e123fec189f1a0519dacf59f30b82b9e744caf3d693440"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "d6df9b36e4ae2d7ab533ebd699bac6eb14a6bc2d39cd8492fe9721c46e5f6b55"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "eb6f03014cafca66df265fa6da2d4af7b9186f12c5c358adcdd0f05a94b6480e"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "7753c22b2fed9f0c5cd32fc9c236fac24fb593e1a3804e84fe6257d51e699305"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "48a6ee86ffcedacaf6a47b236383bcea0f42a8c5729ede9eab53db9e956224c0"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "8575ccf5f47dc296a0fa5bf3d74323fb2efee06f97600eb6cc3f6564d7f1896e"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "c9bd39581f7127b795a71cd9feb4899cb7f6677e58c62cb8cea5f73b1c825df5"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "b97ca758b697395aec4dfe45e365d5fe4f8c2fd3e4636a79c5db37b95f032486"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "2c754f3963b65524c2e51eb50c2b2345ca6de98cda062b464e43afe6f7eb54da"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "dd98e99786e6d829ae4c8fc9c4228c317bb637fdd4d7bdadfce2ac17b5dc6814"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "a3d158dbcbe95620440664edf17211a933c8be60ce8a90581c856c8d502ec7bf"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "0f6c06bb6f5b63928a7ca10302a4352b859aadab16945ee7c8ecd1f0c722bd94"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "6907631a161c313ad949407f218bfd40f63aee655a56cbb735bfe3ef1672f053"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "0b12536d03e6ef1429e23c00814c56ac146d3230dd9961541b3337b7536acf32"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "b311daf4d5c0626f2396bab6b2f983797f0607e4f9a133b069999e570e0eddf7"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "8207d12cb7655670d6ce190b41212fd1206f94c18dd524394fbecf45b6bb127c"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "af30fb7e994784b473f4136413d80fdf1b7303e30083c9015f719e9c4ab24df5"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "9fa93a60cb97a6260334ea5e48e9f5a1df5f007cb42c5bf95541386025714ce5"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "e02ec9f9fa2e5ebbb1f1a7a83b664bcad4396f0945d05b4a5dca3317bf2458c9"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "ab0aeae1e1ab9f51f3c0acf0d8d5ed0267f688d31f801c9d68d0e924416ad837"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "0f1120046c9d4f9e10dd0a2133bcb8223f5599f1fc079492770bd00033c50c4f"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "6118efac6b515008b2229ec40449e9cd4170e45a6505f8b46fdd8c944eb90098"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "6efa59d389bd4d4b6870f9ede5145d5f1a5b27d28b98cc4346fce14143f357d7"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "40592346bd2c7d0e7667e599ecb96813910154d97d2465fa92dd637c2a10e6ba"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "9a7622127a8f6dea8da48d6b3480343d8204ea566dfb95b9270791a237aca0fb"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "2b031c3df4b674b8ef3a3b8fa099b99680599853d46b6cbb65fd030e4585ebb9"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "a35a3356faf0c17b8d485361e03c47c90dfe3253073b92d39bfd525a69d2e8db"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "e784850c655ab4d9ab73b109b5d957737e590f4f8b289c0b2436f9ec6a498f92"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "726ebd60573a00de132636b77b0a24ff6608276c824a8f9e3cadc5581fca28aa"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "867f72233b2cf60ce0df46db4d57a7c8e363fe94a36609b210eba65448d753d6"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "78daf628e83527b9c50d1432ac236b6b8c9c0f3e65e94530f7fa31995e8cc5cb"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "0203068d68437fe8100f0ca89bed6b4710f11aa74771e8e7d7c1abe1810645f1"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "658cc34edf9eb2657ae7a18f72f91e2d4f381b31e9ea7a3bfa42b04812136a12"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "661be578c1d4195e0263251523f001ad02572e7aed3d69a7bd4b68ee63c95828"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "36a36ce05c6966a2d5d729ce723970a5080e917e2775079dd9c465f755a5e4ad"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "b1eb727007f9b23172bfdf122925e6dac628e687759ec73ee261481990e07299"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "ef29ee7957ab0c00f4e8a5f9a16cf4ca92e68f615d84dc2180562446778f9e7b"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "71ae7b217736a329abf992fbc8eb48766677d2ad605ed296bc0a20a217f0664c"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "54e8462156fad09311a25c5327635dc04e1425450355c4feb827e506c740f508"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "df50c9d738798db93b24cdf0850a6a463816b5c352f63abbcbe2d99819ed0505"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "e85a6781698f206d3d75398109fea670437a4c6828247810294edc8bbe18ad37"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "55af4d3f4a33d1262341e2cf986030aaf6fafc485f408a954e798af6811d6d64"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "8837c31dce89517d475af4804a67f1743ffe0eb5b072158bb116940ab10c6c6e"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "098e901827f616d82f319ea48fb12de23744498f91f0619bb421582779caf4ee"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "371059b4c0003fe5b11a2f43310d65c2c393ba45906f9cacede1698f86bc3182"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "532baf768c2f815153297c0355544e6f01304ab8c5f8b2956a2b207902feab59"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "536bc8a4569067f7a363a4ab1c2ee2033a3fc0fe942b1bac945bad102cf7fdef"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "129c6adbb74be751067a5388ed3c58d2fa17fce87f5f6440f3305e3049e32354"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "a51bda68bed8ad9d6d31163493519c5e238be00102b6e9f73288903c3bdcd61b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "0f1b1b50722b89b9189da966c758c20cedbc1ed8e5035eb16bc712bbd62f5f3b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "122f121da3fa5549ebbf7a7f57eaa6766fcc70194527ae89fc945604e142977f"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "79a6b1c79c7f0c965233155c1912bb76b9e09ef358483c878dc5a55786d8b3a3"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "7f427cda2b677acc7ff200db4c23a352ef86005f09ed7ea1c9f388ed4305994c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "f9030525e973c8c2eb57b7081757350158cdd96db878b529765c4736f9d7c1c1"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "952bab02f4f3a78956ef906baf83173c6093f14e1ad7cd01145df689bbdbfa0d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "2f2635860e42a69d67b74bb6eb5ea83cafbe13d719623ccc4e6e0334ed35a954"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "6a67e260208c21c5b374c631f25ffbfdf622e339674660fbcb43e29a1ac71d58"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "cec2ab69bac9adc0f0cf598daf5068474e186b1799a3693c3b8791357db56b8f"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "50bcf18d55e5f67075c10e5bb6ab20a92ca866bd601526eb17f5dd2f0c338920"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "39c38f44c38b94e9b328d767cab82fa7a37860966ccbcb9c35ae63ee2d610673"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "95d2320ae2bb255a520188fadf4fa3408107f3f6f67729a67e2f5393f51a5b50"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "61704184c62ea32b2d0b317a83375ababdcca1d247ab978163d5551c4dd46eae"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "19ac93434ad4ab6319183be977349039ab89a5757f894c87bc21b34f355d2aa7"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "e5a4102c8f14d7299babc8b42566c149f516a54b08f9f03e7990d0ff214db7ec"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "9f7ec23a4782b050ac0702a068cac2a0e42523ebc43cef5b0f8f5eb9d0d6c510"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "72b0851859f97c81d47f1bbba6932b272688095ad20cd9d014702b744b03dea0"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "55aeb28b36cb8574e0dac1be0295f018c443e51e621b6776588ed068ed7de2cf"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "38f2138c379600e22a56cfaec2b3903ccd4e5a6bb4b7a58fbea0694d2fe81c52"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "bd70a01eed4e207e76d60283bc59506596e76065239b43b21e9d9d800963f9a7"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "6fa03f57459c973ceb130acca022e9be3c88ea79d600a53178efdbf40cd5eddf"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "67990c64f07e01da376da901d9b7674842eeb6d18fba35ccb3892b154e02e10b"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "996c63ee13c2979173b5b755ec592b87dbcf8be4b7ff32234f2e95788b0c389e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "5cc1730f6ca989dd5ea88604eb58e1761bb2be8d56d7ce763bc25a5306ea453d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "4c1e7272bfaef0528c4f3b40a1a94fad9a44c11fcff0202f9a8e2aac515c06fe"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "cd33dda14afbb9bdd59474638f52eb38dbc20fc416cdae21d506be37df2f4034"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "043156f29a7cb05c3f1cbc511eb5072fb431e03cb5c6e525685be4193f47c169"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "5bf7a80fda35614ffb5ea4cbbbf814fd0cec0846a29920495335fe465110185c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "20244b8ed7a9c026f70c61788858cb83964add8eb7867717e2cbe4b835ec1067"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "98a9c0b2ed4fdacde0ff66e6be9c45c048e7f0ec4cb50af2c4a5b1181c5b3611"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "4f409bd224bdd24f95caec2e74a5dcfdf56b3e310375c093313d947c7beb9428"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "e0bda803883e96088a9c21b1d1144e17ebc4bad0eddcb90c791f99863e997919"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "451238af895fe1a702bc651b02728ab0b924dfb86b15531c89c4e33fc97e2671"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 64, 0, 3, 64, 0, 1, true, false, false, "72acb2876d48f2c2d363e136260e4519a963fa3f841089a84941149d7be39f26"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 64, 0, 3, 64, 1, 0, true, false, false, "2e586036d32ef9eddfcf83d7b328c928dc33bfcf2e9fe917a6474a415a292bcb"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 64, 0, 3, 64, 0, 0, true, false, false, "76dae756bfc8612231447ce7cf3f9535dadf6b492e111d5ed97c3b8d2cfc7aea"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "62d20be77ac50778a3fdccd678f03e6915ec26982fb2b659863ea29462f8dbc1"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ64Kv128Static2CtaKeepsAbForGen", 217768, 384, 2, 64, 0, 3, 64, 0, 2, true, false, true, "ed767f60bc080b9c17efea57d33b55613ccc49c877aa94153bde4cbe152bb346"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 217104, 384, 2, 16, 0, 3, 64, 0, 1, true, false, false, "8e76eb5b70cdfca4587b1d6171731de8256ff6cc462cd557d4d14498684e2089"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 114896, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "8dc4030ebc9855508ee4d7bdc46cbaed0555bd26edcebd6c1cc3708e086efdc2"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 32, 0, 3, 64, 0, 0, true, false, false, "7a0d5465283a6103061e1a45aefdd365dd353fd1a1cd7ce0ed05f0b5a9bdf7e7"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 64, 0, 3, 64, 1, 0, true, false, false, "4d479da0de3aef132d5a73ad166ef8c51f323359b24dc0171c6df47b1eb15470"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 32, 0, 3, 64, 0, 0, true, false, false, "027d87c91cdb5a9965ad8ae2e9bffae4036d760ca862f278c5d31870feaa6298"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "81ed6a52e1379e2ae6edec2392c327a89dc644bc5f76dc2696917dd5e04d5c83"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "8e9ac083854b843bcf3957420a6f1bb0b49ca76741c9ba1c391624f551134673"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "a73a1098652edc8e3582bc47d8368ba29374517909355ec3dcc80809e6a81801"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "042c99b161d2cb6e6019407d01a81dab4b19b3ef288fcc61d7f9aa9f1ec76315"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "e1c4e519baf9a46ecf84dc78a05d4a37ac05532a53adf7876a6db757abe10765"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "d86981405f6741053b23b827756376340bd8a79a34f26e3d2302f744c30c2f23"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "ee004dae956ea28802d1f53e28ea57e6b15ace6e507f9db58ebe6bbc7b81458d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "a3ac748d697c91bcf81db5162af468e12412473cd05ac82abbbfbd0d84e0f13e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "f1641fc046c507ca446f485980905c04687911249a4db5d97e6bc330c1f734ef"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "0d22ebe9db71520b146615995ec14e146c296e0de6f1a0522f84bed8f2116c00"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "8fef67f0174b11354022cbca055201509e853612edc4951cd06dc928fb5b94c8"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "48cb32afb8532fd45149b547e0a145bf88c03239d5504f2b323032104500ca9e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "367f9695e1a7e52e8ce8a3c104b24649dc51cd54f1652bd4c1bd7252f608f466"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "a1c06677295c78dad423da932119e6085679a8a1c8f0f44aaeb11df34365b3f8"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "022a9d9f008b3805517511a8170209b56ea879950c38c9709bb557e46c0831e6"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "c3aa8945fe6cfcc874d74a27f7261ee9f53b845850c1a600343355549d6463ae"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "8a952568e98074dee07feccddf340a4f1b268c73c0fde5b3dfd91aa578357c01"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "09094e1fd2233fe8dd9dff98a70b6d804c3022e225cca8654786546e3595aa9c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "fd0fd82c16a1f9a289c6fcfd54993d7bf56003641f80d535e1aa5deb72ac1d2e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "ae0489394137fcaa8c278187ca6aa3c32b0204d62757da5bcd94bd385eeb5242"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "aafabdaf969da2342c687eb0489c94d4f0d768f56d0da20158f1451ea323aaad"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "350510746ec71ff78c524ca6af803bb08107fc3b545384cde7ebc049f078f2cf"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "85aa6ebb73d72bb50e53f4261161cb71851394d2eb6d1697b92970f42f2847df"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "359c4efa07e3b9c4c570805949e5868a8427b35649394d4326ec9c3a381215ba"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "300dd82d1c6d503c16bd61d26eb6031c7b4d445bccab79df4d39adbc07c9f0ae"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "3ae6b6efb3470945c40ee34c0c30cdc55969a70d84604704c2defba933e34954"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "7d0485881f94a7ba67832da22f78fba22b46c44db781d6da2ad2ce5fb1288a1b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ8Kv64PersistentSwapsAbForGen", 92816, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "f2695aca341acc0755a69520285a342db39af2a90f4823711f3f40ac08675e59"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "eb8f20f675b9f73d687fa5b3b8e6ece6361c5e1c0c00b152677f8feaf24b6232"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "86b7f1153f3fd1f4d7041a3ce2080bf64aff5aee1ce9fac2d6eb6e1a2ba5ab61"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "7d81784d2ac3a772062ecdcb7d816645f09332697d83f0d6ad7246461b8d09dc"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "b28ded815872e11ca9368606be15df0f95fbceb83287c5fa507918c0f7cc921d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "20cb3fdb7e4482a3c603c3857b4d6b4a677bb45c91d34bcc01f3137b199bb57b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "d84f9010d6dd79299a773fdfd16169341cd5b1cc7609e364951f0b213c246e5a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "630d633262242bd3471a83d520fc073d4b9ed490a167031804b00cebce45c457"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "7fb71d591b65324ba840c88a9ef082a947359f4fa18e60cfcea080a775f31c40"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "54b782d441ad692a06989a2930c6548dba42b9ebb5823cb8333cc55218ffa285"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "468e4114120c1907c14fb515a8a678df15a09b3531fcffedaaef0aefab0af150"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "c5dde5b3039b5de77b18c2b30ff7cddde26c5d5e223988330a04abf316fac78e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "5e0eccb9eaa42a2efa1c68865402abb09c9a5fb050d8081cc6be8b768f322908"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "23333747d2b058fcfdbf7718fa2a893df9f0add8969cc1f3260625419d34d441"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "e8fd5561190dc1f2b4e3d0c62d063e51f1955575418b8db53994bad7403d2511"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "63cb72410d0e36752410ea75d455006ff925a55e5723f3741098922b60572b8d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "be25afdcc7d56f213211f2de8c0c9b80ea3fde26d854f4ef74ca09e852b3dd7a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "05ecd61d5eee4e08888eaf4fd992c268ad92811940253f384036d5008faee597"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "1a532c163c0934c7cb98547b25505fa16e8bcbeb3acb851b8fd1aa389248d340"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "e129b1cb87cde1a7ed2f053de5b47b45cd0fa7dc884b8c78c6419464c54e25f5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 160400, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "e3a2f9d60a88843a6ae5bad437914fa68b7c3db893c9d4d22ec4e54371fc2404"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "0002de9b6de73279d9e092744852164856c927938a885cd30ce01c1bdb093937"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "35419a0863d2f1c3fee344c0c03d148d766fb68c50c8ec33ca9e328faca0c5e7"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "d7987b0365c55a601240e799edce9686b57b99ff89addaf1f666f01c05961d3e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "263c305b1c8daeff6403661064c946ea6fd7c9b2bc1aad8a31d9b0ee5cbffe7a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "31de3c3d11ca35df69f24bc311806bd4a955510eaf6ecbf2f3b3ae450daa4ce0"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "701f6de46756203d21de00387058bd1305e1e46a3d223a12cf1acc731c9a496f"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "1e90724d64629f0900e963d5161bc9e877c12fe47780fe7673d147bd3b679952"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "48687baffec69e48dad3b55ab4e374445c2d1f5d7296c34a6e55c2a6140e010c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "0bb4ffee8dc2413cf267758e49430c9b8d543046e7a50a3c1ef5a019e9daf78e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "a5f3e3b913bd5778fe81232ef34c5191e9f25c07c551be10a04c104d6c5c295b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "1751dae43d135986ec1067e7bdfaeb781443069ac1ead70850384ad1b54ece3c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "8654558022257b20784a4cc7509e31dbe3952a28928a4a631e7b77a2feca1837"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "a9cd826b016650dc8486f18b143cd476575b1dee2b4c00a166630b889d9a7426"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "b8d5860e4df7d92e19212036baff676a3c48ef32afe474da39031afb7305e7e2"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ16Kv64PersistentSwapsAbForGen", 115856, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "d4fe916777301d491a02bf5b9ff5a2ffb0d51be07b3cf2fbaae3ef184c5982da"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "3bfdab59065d019a57b3985879d088747bb02633db78521f39cefe372147641a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "6980d227019f5c11dbf1310044d4caec7c1672aa6c7683b7caefc7e194276454"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "9206eaab0a344f1704191d91c2c8fa809fb3e214a392b8788f191d3587deb985"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "be2772af860ce47d2c6c4f9d85993ad569bd1d53b0fa338b3792280cf677acb6"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "6dde5f0794bc6ef102ec1adb2dcc6d459968b7de0691848fb91240f0d91e9ca3"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "17cae1872275a2a09e2b1902cf9413649d99c233cf4c351b163fdb9a748a82c6"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "da646abf165bbb2a8b2a1858af65348845626663e25229d034469a0724dc5a80"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "7d3a80d541cd9ce271d490fa278b9b07abfdebc50f361a23ffb3cecba5b91cb8"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "b23aa2d36bf495dc8ca500c98915d5e842d66fef249e9a3f2eef4efc4ed207a0"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "df7d9ecb8e2cf84b4fca4844105df1a3daca8f3ff8e44a27bc35adc45c569e53"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "fd0325aab5a38b3875298603fddb8fec1494898ad4ae44bf146f900f493a5f4c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "f68120d2dafc2c6f05021cbfa591ba30ec11bee068ca0930a07083e320f02c95"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "3ea425071e7cfa25ce85a4df6d18e5af70c0ca1a1fb48bb2d50dc87d60dfe105"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "af86859df865eee65672174353d8c027909d630252d344ff3217e62dd1a30208"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "33715735a3651f2c00132f202cb6a749faa93745883a7c0596ccd1b33051ae0d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "4476fcaa7e7561fd0bfd91fbe370934985128f83876e781a658037e119d3628a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ16Kv128PersistentSwapsAbForGen", 185488, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "5655fdcb23d4ca5cf95aa0136e80190082a130106c8180dbeab43aeb24ebbd1d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "eb424e9e2c1dd6910bb5165ba08e619b044325b643c4476896a1ac26c8cd274a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "5ecb6497242473da77f59f22c603de012446cf2893a13c516f83c286826c09a8"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "5474880cc4e670b94b682d95cf4ac66339be68518c5221f74c182fbcab7f4f80"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 81552, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "0b74af1e79ac889b9c0ee5917066788b8d37139fda8295382e53ff6b5d0adec5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "dec2ce9055debf96cbf5a0645ea5c0061baa9073aab68afa72fb73201c53d411"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "86b4b14b93c53308c5a370cb2f71bf0ddd133452e52a27b36519cbc90fd546f5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "c208f4fe14165a71c11fd988a09ca4cf9bedca9a6b9553d21994e4bd122ca0cd"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "dc09201b106ec6dff780a7d4d72a3f1e7cb3b1de1527cb50230526bbea9e264e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "f7a0e3202e101ca84d3665728f490a159324786580e3e223b54d9373bc461890"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 149136, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "f7f37ef3c3798e3ba186eb1dd9238f94686ef84bb5da153531fa2884bcd67449"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "c5556626fb088a833fe01ff7494a6e7588717ca256383bdf5b744bbb8caf0cc3"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "9444d4c740578a5e07b32a8372a86c884760120449e6c039f323ff32221efae5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "17fc0033dfb820929cb89d4c1497946d269ab4660f8d5f68140b738a843edfc9"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "d6cc3af599600b2be40a91a127de259f775eaa75d99cf711c4c9e84a6dacc379"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "7ef69eb63e28f11f014ed210242768dc6d984e2b571d8bde4b769cdb82ff1408"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "83abae1e81d7a9f5300d4d3878584d8871617fa0dbddf972c0bfe74b43194fca"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "5776b0b07bcfadb5f9b8ecff5cf3225f95aeedc714b138c8d515fe26e2e77222"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "72e9403c8dfddf5598b36a9a570e05fb7caa041c1f534a3a49f6770ed320ac44"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "4a32e4a5d7e532e22dcf47f3a4147dffeef1b60d886a6d145022f331a780db85"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "ffaaadef920b88aaad9c24d217773ff62a32cf47112d2b63de8fd3509e63479e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 114896, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "a0b8cf96f4c0bb6b562a156a9e9a21f3aae0f162c48813653221c49723900c3d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "2cb89b38374af352859e64cd9119b8bd8ed0b917a0b37c6972d2ccb815b6a446"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "b65501c0d2f8389d2f3fa2613a0bbff3ccebf46ad6b19a334e7aeb5f01e9479d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 115408, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "068aaa0c56cb1a1a9fe0233c65ee30706ff8816698f93d0bcec92d2530895a85"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 114896, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "6d13e90753370c2b64546bb4a48da322efd42ac77036009d112cde53a25b6f47"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 115408, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "5750f40f0706bb71ec2888a4e384586b05b994405d9d14d4ac85d9a08c205ff3"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 115408, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "e485b9ef1e91d8152b18848188717758433205ffdb8ae3beaf31c79efba0de7a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182992, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "089e676881690209f759b238df1f28dd09e61e709989454ce63e539b22c89731"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182992, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "360673d9d3b322aa6554b04e7d74ba2c370d7fb992ee6e47f2f86f7a6238b842"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "128d1082767ffc66e8619a0b4a4577cebd878b195c54e445d0de95d3e096ce17"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182992, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "d249c206088de5b5990c3b144382225fbba0ff8967861d4f1a207aacb958abcc"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "c826567da79a4aef820b2c9cd4e46a2598e61eb28d6bc29ddbde02687f2aa4c5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182480, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "fad5d66b26fd65b2e8cf101b39d75f0503e3ae63a8afe92bd8db0ef9ffe85cb1"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182480, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "7c59fb467a1c611f675375c20692d106648f99edb353ca798175b41927c729ac"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 114640, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "004e65bf692f4769155d723f2bb442c6ba36e4bdb9aa5650877f26f89c2cde0e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 114640, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "096ba789d4dbd0c2def8f958089e9fc88ce06d1c656850709f72ac5117154841"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 162960, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "06ccd93aee9f369fc4b5f8d2021259ccbaf539d77ca8a6d7b2818e47a40ca661"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182480, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "6725bec6700788705288fdd3e97bb47fe7f111c6bb31b537b3975dbce08bc6e0"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 114640, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "065192d45723d24dfb6d53eec88dd744b5370ad3fdd4b3b73dd249d0464d23a9"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182224, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "96e411394e0466246da8d2d27c81ec2442332ced0549447c189d25d9f4f94ffb"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 127184, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "58d625b6a2e0341b11a920b6a4d72261f010d73e67f0cdc0a5e220221f687a3a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182224, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "b55bdd3cf34c98b62119efcd1184aaeae6bf6e843088ba164acd917cf1c669a5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 182224, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "a98506715f8c426e7ee32664553af886d2e44e4625df046f2b8576d168937481"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 127184, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "f7f290a872b255e19fc94e144c251f24f51e1cbd262ea67619073c3981b7403e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 127184, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "a4588be00d33d43b86be854cf08e1296ae77c478922bff9a45f10af4d1b3c45c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196816, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "2d043ad0006eb54fbf40554dff616dd65a2fe93be81fd140953213290b054423"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196816, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "b00db2f282f104d3afec36a7d0fd6d1f4d877f885e5515f527fe75fd62c541b8"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196816, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "e42a169c201e3db4d09cfccd0d5823604a7d8e0ff8cc76240cfd7b2bd7bf5156"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126672, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "a2c77fdca646e910b7ca1e36c907aaddeab22d3adb08a9f65c964a73a924a12f"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126672, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "2a69f361fea570b5af661e4fc885d5fc2461b9fe7594ab7057882b73f48b2887"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126672, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "73663f154d015ad478368d15ea88b703ac43bdac908466571d11b78e19a2e987"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "2197adcc2674f8fa0833eac5f1fc9e9655498202553264a539b0e283ecb6207d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "cc0bd25ae1bfef747732501d354a7f894b7fbe762f347deda52a9c800c556af2"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196304, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "8d8c188a838c7eedcdbf927d53f6949ed4cf3ce62c59e99205590a970394ecfc"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "39d2e66925f8c9667f059ca00d7322a747dced2cbc0c6bb48f80903ecb9a30fd"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196304, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "32fec8ed7451ca0602f9b093cf4a768842c7e17831ce6c5d28b033167a93441d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196304, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "557d2185c6391b401afcc7729736b8f3a070480ebb8e20baa04762b28b3a310d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "0ac07499cec827c3c5f3ca455038ed312c818f559db99eafeeff6eb0c8334b22"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "b861be5167d62b6eef2267aaddba2b7e6fcbb8904f0f03d0eb5fe85e082eb832"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126416, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "fcb8db667d4dc3ccab35d1901b0db1d391b20bfa37c4bb1c8d0885b199400f4e"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "d6c3047e5c5b2ef9f2ec9f5cb34de2714fbb7162501b51da624ce60abd08da67"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126416, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "04601d5b7bbbbcb63aae7b8a5c68ab615ac4854e1f8d53860d28efeb8d542eea"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196048, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "fb2da469bcd88909d5d0c362a95e7f733cb001e0127b41f41fae52cc906da352"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "32601bf04311284d67ad1b895b292ac1292f8ed8e0802aac43b4526a43d71694"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126416, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "873da8f7398339731473f0725712e73b3640bcbad7ad2238c031386a052a9896"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "b4cd83d2a0baeb911bffec81181153cb258b8aa855a547f9f1871c6e7715c87e"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "cb39b3b4a87fa851d734365497ce78b02ff742c8dbd3379433ab227ff9ae18c1"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196048, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "11ff178e0ce650a334fb7c28f6754760101270416cabd42082218d59fc3f2230"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "e65cf86b15e03bf3ada70c3355afdab5c83d1a87ac211fe7748a33f0e2c4e807"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "f3c0e7c9cdf2202408f7b1803db0e1d976593812b010a91b3dc6fb99d3af68ad"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 196048, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "08dc593ac4574ad3e83c12bdb5486253793fe30c7147fc558505bb5657775af2"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "efe0b401304b71d83b278ef93b96bb47a12fb750444e39efd307ce28c14cecf5"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "6d5a8d8959754832ffba897a588eb5a5b79503f95ee09378c059aa4a5d0db004"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "98188e239dd0449c636fa1fdc3ff7f7e3c042b4a93a6c2de7287fc4dfc81d1a4"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "e1979d5f238840c9d9d7f470a8e038afa7eb0a83d6c5ad124cb7265ccc69147f"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "fecd27fcffb134cf8c72c9fcb0070eae2725b5e22bfbc2b09d266853b9566f43"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "cd44a7a71c42c54e16c37bdb16cd20361ba23e83896c0eafa39e6ab9496b12bd"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "e94ffcbaa828b6eafc843cc869b343c4e3f97b8b749fc7576e571756c95058fb"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "4623e68c7601e9336455e9008904847aa01ab0fe0a2f8d9a403ec8babd2c66dc"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "d09eb64b066559b9277fac0aedba917ffd8412302bba441f12bcf302249d81ae"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "35f9718f403f1d9620eea51cefbaa7224298e50775cff163c5ed627448511049"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "52148087064a685fa8bcf167fdf07304e9b772a9430d75ff2f713ec56bf4e77d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "c380cb8e18294ffd43190bd2df3d974ace51e0a2ed8c525b146ead2e34578160"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "c78fbe49b9ebd53dff2ec44f1f6f96c0b1c87a1c66bd700dd7daf67ef84a3f6f"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "8f93643f57bd52c134cb15cd13c8253062e33fd3dba16a9d266b640d7d363833"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "02ece50090c89b1e1a2caa521106697b8e723498992dfa0ace3e93070fb5f438"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "6c9f49b61f7a1ff38231a441b483cc4426f11c8e3eaefc7e7242fbea2c0190a5"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "c7dc23e650d86605a6bc5f8d5418ed8245fb6c60aeeca4e7176a367ee3e1472f"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "7a6ae587b6edf144eb862c9547b6d47e45077d15ecbb07eba77955b8ddd8a79b"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "0b063fc6d3906c52b2e5f3402e519b2d189c43e518047444c2fe494d7708d125"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "6a08b04433edc00ac48ccefb2ca61f5ab49f3332acd4a9396db015cf4b8c7320"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "6d9f687b8c35e32e026a6626b683382edc1162d899dd72ffe3c814c2c35127b1"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "a0d31686a690200de8b4df7786007b14a125bcbf558b6f7665fd396fe3f4b632"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "68d78a61e3e04dfa083207b24888b5733dce6a218ec08af1e0b67a3e26bdd49a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "806629e7d56a74f78c1433b7960bc8fd31f842b59853738706e200868bd10824"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "c96f2cf8e6aa52164e45819ce305a1e3788be7bdf1ed67816cc79f2818bedd4d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "f4f9e62424a65b71463d08a2e9f41f03ef1d5b200614c3fad8fe5b743c8922ca"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "a0b0ebd81390300720b7bcda389e2aaa48e89faa496ddb705e01e693338f6548"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "a8b1ab4794d76d949b927ab9e6c18d5e0f9443a344ad717b8d61a7cb63d33e53"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ8Kv64PersistentSwapsAbForGen", 91792, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "258ea7565eceeaaba95848c2e7a128f68c312de3479b0e8e1616fb0b75024e3f"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "0bea1ae29b5cfe51cb35e96d314dfc119e8b47a2097f7bb3c46085902f7bee72"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "effad3451df5bb3a6125a6122cadf8039593f8fa557870a4cbe52498aa88e530"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "db4cb7b7a301d75e7497d9b4e5f1b318729ea13d2b0334d9b396a513a658d060"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "a450e98c811b53986878a46e48dac95c4b9542c6f4441e00095bd922f0b6e4a7"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 166544, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "2bac04edcb23de6bfbef91fdf8b6a342a517f8a97c5123bf73b7b5f7f8d633f1"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "8d039c48a25d5569db24a65be77ddee8d890ce842c6644f1093b1fd4a98bf679"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "f60841b449c00c149d8115421594b1f97fbfa35d7c2c2979d56be273a68cd64a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "8011eb0f4038a2494c0edb3bc302ca87c6b15185de17fd9e8af8d9b18ac19574"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "ad2035eae5dc4da4f7d841a512e7c26356bc8d11b7901f9187beec1cfe2a2a9d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "d57f9fdbb9606ceba6028e0798637a0d65261062de88ea2bd54a6041eaa23afa"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "676d97f20315301ff67dd480c2fdf7494b4d6bf5fb5fd3f14e0940e04c809d91"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "f1b4aad84f53844293a2d7879522e0764c90745cf5e369118921125d58f96256"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "87b2fc894466e6d82a56b3f95a6850762670dfd0c4f429ef6edf49c79f49f681"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "1db1d4527c0494b858461f62ede79dfc1650c74d551b5758b662f1dea54f818f"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "39a3233f4f3082f811501dafc6b7bc15ae1fe65de1154175c7135f44064d6ba2"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "c1320e6281fc9078ce130ffd0a1af0a18fe6e684c62cc93f601459dc9f5813bc"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "533b40f90bdb85fcafe5b0670454c11aa0b63d38c4f530d56102ce6d51d08ddb"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "189328e01956fccbf69233bd746f9ceedae9bde2d08c3aff02e02c4c45156f77"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "4456fa1278c4e30504b86ec47aa9e26aec54b42a83842514da63fe087ba94d32"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "5e15a0b26dfb8b5d9e5b502c86867cc591f542de56edb6252efa3e2bcda70f4b"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "71d28db7de156aa508f391b678fa8a5e602ecb7616d4cb25b17d25eebae388c4"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "0239e2174dc7b5f5941b7ea96a1038fcbbf8ca453f2ddd7cde1e967b0edca362"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "6d113ec56b83daa64ccfd1b1350cabcc44574093a4daacdd821536819d5524a6"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "c4d2fda0b6b1c084ca62970cb938f7056a6c2141c4251b6b3f8ed624f3d8306f"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "bc0832041afe7dc26131ba0522848cb58adfa60c015bf51a928cca79e9781cc0"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 32, 0, 2, 16, 0, 0, true, false, false, "fc542855371d1d7ab782bacc58d307aea632c3d8c415da8e6e402b04397218ae"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "908fbf7e8a496d8564d636df3512b34f7a9c25ecb4ef5be92529bdc4f507295b"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "6f8a34437d9a25c4d2c9fbb09c4d6e68a0cc847ebe214d60d57caa9c2fa3f4f2"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "8eedf0bc817719bd81a1a80ec38accbec5f4670bd99344c34f02b1a99f039f3b"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ16Kv64PersistentSwapsAbForGen", 105616, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "34de9a8b012e7a87bcf0f1a3c1f281be1f311a6d0b5aa8ef2c5b6ff1536764a9"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "56273a7efc20d30467795aca8ae11ff374992d4d4550d2a0fe7189a90ad85ee9"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 16, 0, 2, 16, 0, 0, true, false, false, "7e83a9b5c3f501588cf1c40f113d5a2cef7f1b3a79a1fac0fb00017173e05ef7"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "a139df92cab2deac6d1bc01ebe8537e4056d86772644d6bade9b36ea91d7bed8"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 64, 0, 2, 16, 0, 0, true, false, false, "091ea064274534f288f2649c71a8d92f6caf56d355da76c08adda620114f2f5a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 64, 0, 2, 16, 1, 0, true, false, false, "db9506a17414b2b65ab7c8630b9c025bf466f211443bb8ff1f8b9c2ecfc31e8a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "ce334adfd348a1618b3ddd4564c073fc03e663c41d85ab7819002a99dda886f1"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 32, 0, 2, 16, 1, 0, true, false, false, "abd7353ccf7234fe782487fb6293c61ee1c1f112bf38fa9b6db69cc1214b45d1"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ16Kv128PersistentSwapsAbForGen", 181392, 512, 2, 16, 0, 2, 16, 1, 0, true, false, false, "a414d32c341b0e630e3ac85d2edd8d78a06d43efe467c1a950b99c9e61d0673c"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "457ab601df7d75f0e1f580f00d68d85a459df0120724a1777cabd32033a7e94b"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "6c8587f1afac8709623e0c374db68643a8c3156b196441f6737e8d7e8113c5d7"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "baf52406e8fc8f2de769791efc98b165e8cb26bbbf416a90f6bcb79a8dbeabe8"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "4c60469f7e80d1eead724500b9b9de11f8d1d51b7ccb11a0d25e8f74e6c86e4c"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "5ba2b8e29ad08b13b5125019d2866bd425f2c9bce192baf6bbc4fb5ac51009ef"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "592c45b361e3748509c6c273306e5700b7b9e4669769f21df45ae57024642f68"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "57cba6bbccb52a974f7d3d1869859ef717d9fe4ed13272b05b8c2a5919e9d7b0"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "d2714ae4bacda276e82f8bfdec973f3a11bb2f68ebd6ecf59376b0875b4235b2"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ8Kv64StaticSwapsAbForGen", 85648, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "fe7c45911298b8452e241d7e9cd22458c811bc6dc804caa30d2311bc28a67620"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "a60cae85ee4bdb52f95f343583268388571362ff881971470efed8a63419205c"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "ad95eb3df7994c8d7517aefefad4c593868607f56c9b6e6cd660bbb79e0bb0be"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "77aae88640756bcdb4d6f156be49b8a2ebcff5e28cd051fe2d8c3b6681f7833d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "036cc7dfc9c633c261d0b0571ac37b23c5297dbe0d210ee279386097a6f5d342"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "50652095a05c803a801e4a0808c0675ed73fe04eb04f6d17affb1e1997bc04af"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 159376, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "5bf964548131f6206dcef76cb5fc48368a8785876fa87282e6388db5191ad9f5"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "8a80cbf08c2a89f5a37c651de32342776f5239a946234811580a1c5c71e3dc19"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "2af6aaa2f4865a6f6717a00e7f35351e37844ad231c17f3b84df3e6c72be8fca"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "275d96aabd250941fc54b06165fa289dfde7de2b85d20fad7885241f76d8b544"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "3f70a89ebd8090b11fc5382a9708cbd9738e106000e49fa2f2ae40a3c0416ab7"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "197347a7061a4fa7abde0b34cdf8070702056a3b4314750abe3e5ca60340bb8d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "e8e8422054eb2b731475233db83be90616e05e0f19bb715979c9ea7c1922ae5a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "5ad002492b397135826152e90538b6819e9876aab47aa605b6433b06bf9c957a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "6cc3dc088595fe10354667768c235cfcf7c93cf6417dc6311f4f343bfb10ee0c"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "3cf40db2777e0a805d1d11eb3773d727e685d95decca7e2851b76a70dbf0a828"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "79bf66503a8c26c61d528005499e23554dd80ac3e30b31efe5fe53b05b4e1999"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "a4c8403904fb14ba6aef18703506278c631a14c04037628c4211dc136fc775f7"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 119504, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "a1b3bd68bc4d9d218b6c528dfa6a8f37ef8856a638435cc1fae5b389f86710bc"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ16Kv64StaticSwapsAbForGen", 93328, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "02242bce53d5ed684347af0a388eb6fd670c07f4b56ee64480c16c1bb293e273"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 119504, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "bb7893514e47eb95d2287a518a12f378741411cc40f2ba0ac97d8a21c20b8f3e"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 119504, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "128c6f596d17db5c2ef7b4adda2b825358e8afda4c382107502e496f03b4e941"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 16, 0, 2, 16, 0, 1, true, false, false, "2664469c0d148e2f8fbd645984b4855ec15e8d5639ac8bf16ecc48aede0399fa"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 32, 0, 2, 16, 0, 1, true, false, false, "e208d7db2a9dc0aae522a6b86d1f30e6364937bdbd118ec0d27c26eb504356dc"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ16Kv128StaticSwapsAbForGen", 167056, 512, 2, 64, 0, 2, 16, 0, 1, true, false, false, "4b2d6083046a766bf64b87bc9f663f81bb3f5b03878da43a50b5882c7933e9a9"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 193232, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "b1bd9787054c3dbe05a9a745b16156d7d58b4f83de310125803b88ce53e27297"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 193232, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "8d1cc81b6e38424c12eec9ecca5a81427bb9422976c8c18a6801cc1f199f5528"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 118992, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "72e1f3b36419102edc3e6c161ae99a678d486a5b2d13bbbf6f594b4002e25a79"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 193232, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "0c4634990aafc5a494695fd61feb3caad3cf8fe9368aaa089816d57258b628dc"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 118992, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "3aa96e82e172d2e3477bfce9bc3fbb2202f3dea132b6aba339fda9af9d23fed0"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 118992, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "a7cce326f0c834f0fc1cb23fcbdccea8a000e60cc7914c2356b19b1e99d91728"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 192720, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "637196ca68c648a8243a095885828be8e247677dc17b013a5cabeadfa9b25afb"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 192720, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "487fe226036ec97560f52ff7e722bd39b690a57222c3268ee72755aa41b7ed1c"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 192720, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "7df1c937597dcfca836db3b9aa4ae1e21f570034f0ac3572bbb03c1d96de117d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 118736, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "ec9b57a0fdc5a106e675f126e1d26bfbc3ab6eed3257f360cfb55037a25de2d0"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 118736, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "bb6e01b749cc937256f3614e612c3cb30cb64ca645982739e12589ff587a58e3"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 192464, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "c80939fedfba60fb4b03627093aab5754acf11d561f3db5631c49a019dfbb3a6"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 127184, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "0fb5cc4567e8d104d355623d083ab7da873ada15443c81c479eeff5bb623aaa9"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 192464, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "148e0a94bb8d7d78aa5ba8c163897498a2b75d1c440039016acad20cdf890a3b"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 127184, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "5e1e55e9360db250e42406e76ae8a7f7eef3c54bc3a9b42a2e934dd357a5e2af"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 64, 8, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv64StaticSwapsAbForGen", 118736, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "236e26658d7b20e5a80e7c9928b9213171c86b34dd2d16678204ddeb86b1093a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 127184, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "f006c595f8c21003c3df89db04270717ac391a4ad0fee07067cbbc321da09188"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 192464, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "8832cd7ee0dde047086a21171bba252e326e0658857f7ddbe2f37ffc7a168635"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200912, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "4232e5ee96708dabca6a98a412388c0c7376c49a92b27640a76c4ed4a03d3935"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126672, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "f6903eada3ba91d37f8c213c262907d3867acc426f4302d970adcc56f503bb29"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200912, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "76009b319b0e68ae6e3328bf97725d2db05957bbf0b7fc49824982314b30452e"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200912, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "e89e523aa5d397be6276cc680c532c01478d9ed2d15a922b7019c330dc1679e8"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126672, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "29202140a9e9534b62cdb02943857b7ca55e47a95bbdd2d43d7fbd47c8840c8a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126672, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "aaecff2b502951ec3f925a1ab8f8a0973d4e74eb960624de4811e54257447e5f"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200400, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "f45e8e5514ca7b85fc0152725e1495ac266532df91ff37cb3006fec461f40f3a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200400, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "e1d2e081db452a278085fa7f87d2d4614c61a5b7616add5b06c210c7ca530da9"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200400, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "3f02760afd8411185af09f98674faf41b1c54e1e73d659d51b1683bece84fd73"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126416, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "fdbca42a0a48ec5c61222d58b0faed499678532ade11711b719f1397e84b67ce"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126416, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "15ec0881409dffb7e5ce84a111e132f0df2f97b6342331719b16bbb1f405cf30"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 64, 16, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv64StaticSwapsAbForGen", 126416, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "d1a6aeaa84cebc3ea4bf517de98d1235d94d1d68ac915315fbccb5512f94ffee"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 16, 0, 3, 64, 0, 0, true, false, false, "02c289c211063d3cf421ead330f2a342ab7d7120c87e982bcf7bddd4011f3a06"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200144, 512, 2, 16, 0, 2, 16, 0, 2, true, false, false, "3aa0e5cc1dc810b5845b275880be8d3590b10211b796afa58a990d2c6aa7aed8"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200144, 512, 2, 32, 0, 2, 16, 0, 2, true, false, false, "4e5212736a71e25ab5ac4f6b269641582abc102819fe7f6e0f967d7e70f5c42e"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 16, 128, 16, 256, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ16Kv128StaticSwapsAbForGen", 200144, 512, 2, 64, 0, 2, 16, 0, 2, true, false, false, "dbc8b1e5581cffadc992681c5ce6f470e44cb2f03c02a71763f97831e4ca11d1"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 32, 0, 3, 64, 0, 0, true, false, false, "24a434b160db28f867b6136c3dd053fe8d04e4c377a9eaa9d96c981a638c928a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 64, 0, 3, 64, 0, 0, true, false, false, "72acb2876d48f2c2d363e136260e4519a963fa3f841089a84941149d7be39f26"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 16, 0, 3, 64, 1, 0, true, false, false, "31f1826fc01dc2506c5260ef5fecc4100a337e71bb1cd0ba1ef3d36521780b04"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 64, 0, 3, 64, 1, 0, true, false, false, "4f0b51268ca7bb955bab97735274866887c6cb0c16fad1bab3e964f623b346d7"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 64, 0, 3, 64, 0, 0, true, false, false, "2bca9a7864858ecc70d4ea1d33a6ad13af8d712ffc153f80cf868f36625de55a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 32, 0, 3, 64, 1, 0, true, false, false, "061bf735a065fdb949e0d9eb2e4c181a3731095caa46e7143addfcac57bbf402"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 16, 0, 3, 64, 0, 0, true, false, false, "a145da6ef8a3e94a817b74fb982b1fa0276cdfce1faca4cbd29e67f96d61a845"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 64, 0, 3, 64, 1, 0, true, false, false, "c9b8aae12bc34a785bd4cd9b371289acb1ddb801f85a977c592f7cada40c98b2"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 32, 0, 3, 64, 0, 1, true, false, false, "24a434b160db28f867b6136c3dd053fe8d04e4c377a9eaa9d96c981a638c928a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 16, 0, 3, 64, 1, 0, true, false, false, "47fa66584febf59067cc5b2737990996c0a628f5a2b85cceaa8817b54f71919a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 32, 0, 3, 64, 1, 0, true, false, false, "f05759fb1995839f3afaf53f763122e91a74667df09bda5757aa45aaea2c11ea"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 16, 0, 3, 64, 0, 1, true, false, false, "02c289c211063d3cf421ead330f2a342ab7d7120c87e982bcf7bddd4011f3a06"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 208912, 384, 2, 32, 0, 3, 64, 0, 1, true, false, false, "d5098833936e98e08001bbc3b8de4b0951bbbf6b88c47b05d271fbdbb35119ef"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 64, 0, 3, 64, 0, 0, true, false, false, "6af36ac2ebd2eb2b6ba3fa75da2dfe919dcc6da0914bbf6fb467e56633098260"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ64Kv128StaticKeepsAbForGen", 208896, 384, 2, 16, 0, 3, 64, 0, 0, true, false, false, "ded0c43ef6bd0c11d1cb975eb99f35febe36e7231fb954aa9bd992b7287ec342"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 32, 0, 3, 64, 1, 0, true, false, false, "0f65f8cf2d7f5976cf3ffc47285fb2da635a650dff5f3a90502d747847e2129b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32CgaVarSeqQ64Kv128StaticKeepsAbForGen", 208920, 384, 2, 32, 0, 3, 64, 0, 2, true, false, false, "fd23890a760618050e7084b9c50e9621c2a0b42efbc04aacdeb065fae9ef5e48"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 208912, 384, 2, 64, 0, 3, 64, 0, 1, true, false, false, "b0777430bdd3302e1e304d45bfa12fd569288ff594a3577b4394ac0b89ccafd7"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64CgaVarSeqQ64Kv128StaticKeepsAbForGen", 208920, 384, 2, 64, 0, 3, 64, 0, 2, true, false, false, "1fe9b2e234d6f9f1d828b00533915b1e639b593bbafcb159ddb3f07e0b101c80"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16CgaVarSeqQ64Kv128StaticKeepsAbForGen", 208920, 384, 2, 16, 0, 3, 64, 0, 2, true, false, false, "658f512da0f8d98b2ee501f9d488a59f20aecebbd800138590dd4e7c68cf70a8"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 208912, 384, 2, 64, 0, 3, 64, 0, 1, true, false, false, "30841a5cac07f0e2a732d37fcbdb845efdbe7c9343aa37f6334e5ea263b123e6"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 208912, 384, 2, 16, 0, 3, 64, 0, 1, true, false, false, "6e82badc12939d43b02cfb3819477e467ccb0bb9ad53b9f06dbde4267f1fcd4f"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ64Kv128PersistentKeepsAbForGen", 208896, 384, 2, 16, 0, 3, 64, 1, 0, true, false, false, "b12d7f45a50a6f38acfab8143a4c4c230697a0fb8113f27445878acd87c70a51"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 208912, 384, 2, 32, 0, 3, 64, 0, 1, true, false, false, "9ff0c8370529908f6461c4b892bbb8a697ce0a10c8d2554b394fea23bd3ffcca"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 208912, 384, 2, 16, 0, 3, 64, 0, 1, true, false, false, "3cdecf408d09e85bd369e0afcd838acef44feab230b61b9735397004bfd16024"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 208936, 384, 2, 16, 0, 3, 64, 0, 2, true, false, false, "01a16c8ed5c2445298491828b76fd98135fe224ec97b9d3f6d9b98549927cd8d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 208936, 384, 2, 32, 0, 3, 64, 0, 2, true, false, false, "2b9f8fd113411187a0d8d64c857ec7793d10f18b8b6c4afcd9edcfb664386bc2"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 208936, 384, 2, 64, 0, 3, 64, 0, 2, true, false, false, "ad9754c2baab40e6f9b441c1ff8b36556514b713e7ec9dd8d7936d89246fe703"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 32, 0, 3, 64, 0, 0, true, false, false, "c6448c04f4e3187cc5a9d45706eb667a4569359913d8a62c5716bd834a7de58d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 16, 0, 3, 64, 0, 0, true, false, false, "28358d08c4a1e5ac2d74f5a17d9c2e88a6281667b924b08b836d03f8eb82d958"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 208936, 384, 2, 64, 0, 3, 64, 0, 2, true, false, false, "aedafcb3f7a45f12fc12344631c636bb7b22e0b887a2569cc0bb6d72f4914107"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 208936, 384, 2, 16, 0, 3, 64, 0, 2, true, false, false, "38e79d9a1265d3d7d779b83381ed274c6491aee3bf00c6f44162e7f5b7b48716"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 64, 0, 3, 64, 0, 0, true, false, false, "f777e9ddf50b6b52958b07221ce840cb07f62dcbf37273e6bc51e447890bc614"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 208936, 384, 2, 32, 0, 3, 64, 0, 2, true, false, false, "25a109dae2701f03b0c94b84c6a002970cf3d5afb4e9d998b92fb470ff0bb72a"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 32, 0, 3, 64, 1, 0, true, false, false, "b6add9b2be8d9f32906ca8671c2b2da6c4f76c03623b95966471cec7927f434b"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 16, 0, 3, 64, 1, 0, true, false, false, "52143cd6d0f57b06d29deb7fc7064d016d799482a390596afed7581328f4f065"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 16, 0, 3, 64, 0, 0, true, false, false, "1bf1dbad0d145f3dfbc0fd62b84ac385cf1754ba57dbe366ad28c407c1f8ac62"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 32, 0, 3, 64, 0, 0, true, false, false, "fd48e8b62c7c523e93aeada4d9d82584f70e5ed632f5dab3701958aa79bbd29d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 16, 0, 3, 64, 1, 0, true, false, false, "56dba1388244102411159b3aad7b1b1bda2c7a1917229bf91639ce96d60cdd47"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 32, 0, 3, 64, 1, 0, true, false, false, "a9f913dd7888066835c3e36621beff437c32cd726e1c01063b93ecb4c240afbd"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 64, 0, 3, 64, 1, 0, true, false, false, "7d5d98386e77a2d2d27c8a568c4a8cd94ac6a361b778fb8c3458a5ed255854e2"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 64, 0, 3, 64, 0, 1, true, false, false, "f777e9ddf50b6b52958b07221ce840cb07f62dcbf37273e6bc51e447890bc614"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 32, 0, 3, 64, 0, 0, true, false, false, "f87993e358a5227824d0c7e424d5606d4c74e5309a2ece0e00c5e556ab46f257"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 32, 0, 3, 64, 0, 1, true, false, false, "c6448c04f4e3187cc5a9d45706eb667a4569359913d8a62c5716bd834a7de58d"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 16, 0, 3, 64, 0, 0, true, false, false, "bd8748320063a8969998544859886fe4d70de55bb76324501bfd76c58ffbb729"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 16, 0, 3, 64, 0, 1, true, false, false, "28358d08c4a1e5ac2d74f5a17d9c2e88a6281667b924b08b836d03f8eb82d958"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP32CgaVarSeqQ64Kv128StaticKeepsAbForGen", 217112, 384, 2, 32, 0, 3, 64, 0, 2, true, false, false, "fa360dfc0be768cc4d34d1b8ffafe7eadf4f34b9aed24ec57974acba1a475f9e"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 16, 0, 3, 64, 1, 0, true, false, false, "262bff2cdef00e50a541e8454e6bfb6f7da7c09071749aaa808141a6951e9bed"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ64Kv128StaticKeepsAbForGen", 217088, 384, 2, 64, 0, 3, 64, 0, 0, true, false, false, "e3a590c698489fa430ec5b16bd01140536c561927b4164a8970131e322ac54c0"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 32, 0, 3, 64, 1, 0, true, false, false, "f295231c55aad27b4c6808217ee710ab5b5a6183427efda86879759c77c76202"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP16CgaVarSeqQ64Kv128StaticKeepsAbForGen", 217112, 384, 2, 16, 0, 3, 64, 0, 2, true, false, false, "83adae5608830ab4a9a17c3a509e63a985bb2c8e7526b4b4e09851b423936da8"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 217104, 384, 2, 64, 0, 3, 64, 0, 1, true, false, false, "08b684be55982adda4bee9dcf3991365da5661ca8bb9215738efe7793bb8c4aa"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 217104, 384, 2, 32, 0, 3, 64, 0, 1, true, false, false, "5d23d901c98d187d6eeb34ea2af7fd0741dc256c3adfe075991afe4e97fcc2c7"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64VarSeqQ64Kv128PersistentKeepsAbForGen", 217088, 384, 2, 64, 0, 3, 64, 1, 0, true, false, false, "00e269428ce5d03849c773caefd4fbfbf75886dd3d559d4675f077cd96795b94"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 128, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta128PagedKvDenseP64CgaVarSeqQ64Kv128StaticKeepsAbForGen", 217112, 384, 2, 64, 0, 3, 64, 0, 2, true, false, false, "09903159b2d61f0afd0fbbe2cd7d18aa5cf866e34baf5770b414b898d7156f74"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 217104, 384, 2, 32, 0, 3, 64, 0, 1, true, false, false, "6b0be444707cc5753e19efb1135d2e554bcc0081ed03986aa8e399ee72dec3d6"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 217104, 384, 2, 64, 0, 3, 64, 0, 1, true, false, false, "1c437091fe09541048b35fc3dbda8bf83dd90347cd9b0604171bf1ae1af98409"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvVarSeqQ64Kv128StaticKeepsAbForGen", 217104, 384, 2, 16, 0, 3, 64, 0, 1, true, false, false, "928d4669b16bd936b56dbffffbd27e57772ddff083637c74379ade058f5f11fa"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 217128, 384, 2, 32, 0, 3, 64, 0, 2, true, false, false, "bf977ab144620ae9fb9b45abfc41ba0cbb70714eb4219608fab589a32e4a3a42"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 217128, 384, 2, 16, 0, 3, 64, 0, 2, true, false, false, "37dee01a6abca38fb6118498ba5f06d07a524e5cff0320c7aab05cff58560134"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 217128, 384, 2, 64, 0, 3, 64, 0, 2, true, false, false, "faa963070b2a55f8a4e3d73a964fb52811fbb3e3bd71700013e4f1a14a4454f6"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP16MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 217128, 384, 2, 16, 0, 3, 64, 0, 2, true, false, false, "8b2196d03a8573af020e931aa5474877a9f8323026bdb9daf7a6db762377e888"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP32MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 217128, 384, 2, 32, 0, 3, 64, 0, 2, true, false, false, "3b6ea99476bebea16d71ca333e97ff2459861ac58bb696223281b23fab96fa12"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ64Kv128Static2CtaKeepsAbForGen", 225936, 384, 2, 32, 0, 3, 64, 0, 0, true, false, true, "683ed28b48fff28d403d512da92a684ea11d9130b3f8987abf4f98cb9e65f6b5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ64Kv128Static2CtaKeepsAbForGen", 225936, 384, 2, 16, 0, 3, 64, 0, 0, true, false, true, "4c7979ebb085a6683aad06fc5cc6b93ac07a89e7a55108b5bb3214d79767e001"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 512, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta512PagedKvDenseP64MultiCtasKvCgaVarSeqQ64Kv128StaticKeepsAbForGen", 217128, 384, 2, 64, 0, 3, 64, 0, 2, true, false, false, "9bf48cdca6b97f337244ac0272ba76c6505ddcc8c75f74f54f58c3cfd96a818d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ64Kv128Static2CtaKeepsAbForGen", 225936, 384, 2, 64, 0, 3, 64, 0, 0, true, false, true, "2c3868462610e2b1540e630ca2d9dade6449df858be070a0186298ac8741aac1"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ64Kv128Persistent2CtaKeepsAbForGen", 225936, 384, 2, 16, 0, 3, 64, 1, 0, true, false, true, "478122ebafe81ed528b5bea4f83a477ea74a6fb43f024f8b1a38a291c1927636"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ64Kv128Persistent2CtaKeepsAbForGen", 225936, 384, 2, 64, 0, 3, 64, 1, 0, true, false, true, "a066ef035ced984085b4ec28680dbbe553515e31d5adf244e4196e754d3ca46a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ64Kv128Persistent2CtaKeepsAbForGen", 225936, 384, 2, 32, 0, 3, 64, 1, 0, true, false, true, "523a6ce8834cae91ae324ec565bedb97f8b4feca93f4f93a66c56e869e9d2992"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ64Kv128Static2CtaKeepsAbForGen", 225936, 384, 2, 32, 0, 3, 64, 0, 1, true, false, true, "e7cf2adf8324c2155745dcf022d0263ecde408c6e46ba21799bdd690c10576e3"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ64Kv128Static2CtaKeepsAbForGen", 225936, 384, 2, 16, 0, 3, 64, 0, 1, true, false, true, "43a77d94825433ec9d37dd3dde84b3f000da546d470d4b1a2ace9e05d44d6c03"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ64Kv128Static2CtaKeepsAbForGen", 225936, 384, 2, 64, 0, 3, 64, 0, 1, true, false, true, "7af35d8c1827b7fdac4b9c2b3ec9ac92793ff41f4b1ed5e49bc87b6d092f9705"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ64Kv128Static2CtaKeepsAbForGen", 217744, 384, 2, 32, 0, 3, 64, 0, 0, true, false, true, "bc8e239cd4c900959be68286a043e80167c3b74fe51b13c6652dcee87bf05613"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ64Kv128Static2CtaKeepsAbForGen", 217744, 384, 2, 16, 0, 3, 64, 0, 0, true, false, true, "4163bd0b25d824f9851e499331dbbc5b762aa915fb5fd8c15db2bb94adc531fc"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ64Kv128Static2CtaKeepsAbForGen", 217744, 384, 2, 64, 0, 3, 64, 0, 0, true, false, true, "b5d9b9bcbd1c584c22b2ec1ec87e9591d86cf8b788b63ce6cd506093b632a00a"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ64Kv128Static2CtaKeepsAbForGen", 225960, 384, 2, 32, 0, 3, 64, 0, 2, true, false, true, "77de78cb9e34c9d3af0be4317460b7432a4660b8be3436e0a780fdf4f1412454"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ64Kv128Static2CtaKeepsAbForGen", 225960, 384, 2, 16, 0, 3, 64, 0, 2, true, false, true, "8666d867ce36438aee8fafac3d5970b0f9894314d1cd3b1e30e53806a7a389aa"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvCgaVarSeqQ64Kv128Static2CtaKeepsAbForGen", 225960, 384, 2, 64, 0, 3, 64, 0, 2, true, false, true, "6e0ff4b15de2f6007a362907786dff2fef3947f26c064f8265a8e79d5c0439cb"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32VarSeqQ64Kv128Persistent2CtaKeepsAbForGen", 217744, 384, 2, 32, 0, 3, 64, 1, 0, true, false, true, "0653bcb0f6c42f7588b6fe3b6910519d36b033581bba5625353653ab2675c1f1"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64MultiCtasKvVarSeqQ64Kv128Static2CtaKeepsAbForGen", 217744, 384, 2, 64, 0, 3, 64, 0, 1, true, false, true, "bcf67ec38965d8d861d30858836202d9a42935c54c9c1ba9016d9fa376b91b44"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16VarSeqQ64Kv128Persistent2CtaKeepsAbForGen", 217744, 384, 2, 16, 0, 3, 64, 1, 0, true, false, true, "b98fd23239447905142a85a9bf504d24002d4a4c2d42f14e60c191cec6f695a8"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvVarSeqQ64Kv128Static2CtaKeepsAbForGen", 217744, 384, 2, 16, 0, 3, 64, 0, 1, true, false, true, "e80a158f633cd698f3aa104c64ed5804a22dfd0df9095e0beb4f7b3fce333d21"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP64VarSeqQ64Kv128Persistent2CtaKeepsAbForGen", 217744, 384, 2, 64, 0, 3, 64, 1, 0, true, false, true, "d1aeb4b7c95be7c394a65a1334a2308c1d8947cf3df51f494e8251a9c0de94b3"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvVarSeqQ64Kv128Static2CtaKeepsAbForGen", 217744, 384, 2, 32, 0, 3, 64, 0, 1, true, false, true, "ba528e65ed40ec08aaae009ccb021669f5491a7a6dd06e54f2c6e71e5812d242"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP16MultiCtasKvCgaVarSeqQ64Kv128Static2CtaKeepsAbForGen", 217768, 384, 2, 16, 0, 3, 64, 0, 2, true, false, true, "10c35f4e62fbdbaed4b4207ed623f97d2e4613b2c41dcbf4b2869531c1c2dad8"},
{ DATA_TYPE_E4M3, DATA_TYPE_E4M3, DATA_TYPE_BF16, 64, 128, 64, 128, 256, 576, 512, kSM_100, "fmhaSm100Kernel_QkvE4m3OBfloat16HQk576HV512HVPerCta256PagedKvDenseP32MultiCtasKvCgaVarSeqQ64Kv128Static2CtaKeepsAbForGen", 217768, 384, 2, 32, 0, 3, 64, 0, 2, true, false, true, "8ac98b851eeb31ec09fc7d709451c7d65fb1ec6fa077715c1f0cbf33c517d846"},

    // clang-format on
};
