import importlib
import FreeCAD
import Part
from FreeCAD import Vector
import types
import math

def createDoc(docname):
    d = []
    for n in FreeCAD.listDocuments():
        doc = FreeCAD.getDocument(n)
        if doc.Label == docname:
            d.append(n)
    for n in d:
        FreeCAD.closeDocument(n)

    doc = FreeCAD.newDocument()
    doc.Label = docname
    print("created "+docname)
    return doc

def circle2D(radius):
    origin = Vector(0)
    zAxis = Vector(0, 0, 1)
    return Part.Face(Part.Wire(Part.Circle(origin, zAxis, radius).toShape()))

def rect2D(wx, wy):
    return Part.Face(Part.Wire(Part.makePolygon([Vector(0, 0), Vector(wx, 0), Vector(wx, wy), Vector(0, wy), Vector(0, 0)])))

def chamferedAxleCutout(r, h, flat, chamfer):
    if flat is None:
        flat = 0
    assert(flat >= 0)
    
    nomXSection = circle2D(r)
    if flat > 0:
        b = rect2D(2*r, flat).translate(Vector(-r, -flat/2))
        nomXSection = nomXSection.common(b).removeSplitter()

    extXSection = circle2D(r+chamfer)
    if flat > 0:
        b = rect2D(2*r+2*chamfer, flat+2*chamfer).translate(Vector(-r-chamfer, -flat/2-chamfer))
        extXSection = extXSection.common(b).removeSplitter()

    ll = [
        extXSection.copy().translate(Vector(0, 0, 0)),
        nomXSection.copy().translate(Vector(0, 0, abs(chamfer))),
        nomXSection.copy().translate(Vector(0, 0, h-abs(chamfer))),
        extXSection.copy().translate(Vector(0, 0, h))]
    return Part.makeLoft(ll, solid=True, ruled=True)       

def circSpring(gb):
    origin = Vector(0)
    zAxis = Vector(0, 0, 1)

    l2 = []
    for ix in range(gb.n):
        currentR = gb.ri-gb.w # adjust offset as needed for chamfer
        x1 = -gb.w
        x2 = 0
        y = 0
        z1 = 0
        z2 = gb.h
        f = Part.Face(Part.Wire(Part.makePolygon([Vector(x1, y, z1), Vector(x2, y, z1), Vector(x2, y, z2), Vector(x1, y, z2), Vector(x1, y, z1)])))
        ll = []
        phi = 0
        
        while (currentR < gb.ro+2.0*gb.w): # adjust end as needed
            ll.append(f.copy().translate(Vector(currentR, 0, 0)).rotate(origin, zAxis, phi))
            currentR += gb.dPhi/360 * gb.ww
            phi += gb.dPhi
        
        q = Part.makeLoft(ll, solid=True, ruled=True)
        l2.append(q.rotate(origin, zAxis, ix/gb.n*360))    

    #cylOuter = Part.makeCylinder(gb.ro, gb.h)
    cylOuter = chamferedAxleCutout(gb.ro, gb.h, 0, gb.chamfer) # chamfer outwards

    #cylInner = Part.makeCylinder(gb.ri, gb.h)
    cylInner = chamferedAxleCutout(gb.ri, gb.h, 0, -gb.chamfer) # chamfer inwards

    neg = cylOuter.cut(cylInner)    
    pos = Part.makeCompound(l2).common(neg).removeSplitter()

    return {"pos": pos, "neg": neg}
        
# === wheel: axle ===

def stepperShaft():    
    lFlat = 6.0 # flat area on stepper motor shaft is 6 mm long
    chamfer = 0.3
    c = chamferedAxleCutout(geom.axleDiam/2, lFlat, geom.axleFlatT, chamfer).translate(Vector(0, 0, -lFlat/2))
    lCut = 25 # arbitrary length of circular cutout
    # extend round hole by chamfer so it continues seamlessly from the flat section's chamfered end
    c = c.fuse(Part.makeCylinder(geom.axleDiam/2+chamfer, lCut).translate(Vector(0, 0, lFlat/2)))
    c = c.fuse(Part.makeCylinder(geom.axleDiam/2+chamfer, lCut).translate(Vector(0, 0, -lFlat/2-lCut)))
    return c.removeSplitter()

if __name__ == "__main__":
    geom = types.SimpleNamespace()
    geom.roller = types.SimpleNamespace()
    geom.ring = types.SimpleNamespace()
    geom.rollerCut = types.SimpleNamespace()
    geom.cast = types.SimpleNamespace()
    geom.wheel = types.SimpleNamespace()
    geom.spring = types.SimpleNamespace()

    geom.axleDiam = 5+0.15   # motor axle round section (28BYJ-48). 0.2 for both was a bit too loose but workable (PLA)
    geom.axleFlatT = 3+0.15  # motor axle flat section (28BYJ-48). 0.2 for both was a bit too loose but workable (PLA)

    geom.wheel.h=6
    geom.wheel.d = 22

    # === wheel: spring ===
    geom.spring.ri = geom.axleDiam/2+1.2 # start diameter
    geom.spring.ro = geom.wheel.d/2-1.8 # end diameter
    geom.spring.h = geom.wheel.h # spring height
    geom.spring.w = 0.5 # spring thickness
    geom.spring.ww = 5 # increment per turn
    geom.spring.dPhi = 10 # angular resolution
    geom.spring.n = 5 # number of elements
    geom.spring.chamfer = 0.3

    doc = createDoc("frictionWheel")

    # === wheel: body ===
    if False: # no benefit in non-cylinder shape (keep it simple and stupid...)
        ll = [Part.Wire(c.toShape()) for c in [
            Part.Circle(Vector(0, 0, -geom.wheel.h/2), zAxis, geom.wheel.d1/2), 
            Part.Circle(Vector(0, 0, 0), zAxis, geom.wheel.d2/2), 
            Part.Circle(Vector(0, 0, geom.wheel.h/2), zAxis, geom.wheel.d1/2)
        ]]
        w = Part.makeLoft(ll, solid=True, ruled=True) # wheel body
    else:
        w = Part.makeCylinder(geom.wheel.d/2, geom.wheel.h).translate(Vector(0, 0, -geom.wheel.h/2))
    w = w.makeChamfer(0.3, w.Edges)

    # === wheel: spring ===
    rs = circSpring(geom.spring)
    for name, obj in rs.items():
        obj.translate(Vector(0, 0, -geom.wheel.h/2))
    w = w.cut(rs["neg"]).removeSplitter()
    w = w.fuse(rs["pos"]).removeSplitter()

    w = w.cut(stepperShaft()).removeSplitter()
    Part.show(w)

