/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * Copyright (c) 2002,2007-2021, The Linux Foundation. All rights reserved.
 */

#define ANY_ID (~0)

#define DEFINE_ADRENO_REV(_rev, _core, _major, _minor, _patchid) \
	.gpurev = _rev, .core = _core, .major = _major, .minor = _minor, \
	.patchid = _patchid

#define DEFINE_DEPRECATED_CORE(_name, _rev, _core, _major, _minor, _patchid) \
static const struct adreno_gpu_core adreno_gpu_core_##_name = { \
	DEFINE_ADRENO_REV(_rev, _core, _major, _minor, _patchid), \
	.features = ADRENO_DEPRECATED, \
}

static const struct adreno_reglist a306_vbif_regs[] = {
	{ A3XX_VBIF_ROUND_ROBIN_QOS_ARB, 0x0003 },
	{ A3XX_VBIF_OUT_RD_LIM_CONF0, 0x0000000A },
	{ A3XX_VBIF_OUT_WR_LIM_CONF0, 0x0000000A },
};

static const struct adreno_a3xx_core adreno_gpu_core_a306 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A306, 3, 0, 6, 0),
		.features = ADRENO_SOFT_FAULT_DETECT,
		.gpudev = &adreno_a3xx_gpudev,
		.perfcounters = &adreno_a3xx_perfcounters,
		.gmem_size = SZ_128K,
		.bus_width = 0,
		.snapshot_size = 600 * SZ_1K,
	},
	.pm4fw_name = "a300_pm4.fw",
	.pfpfw_name = "a300_pfp.fw",
	.vbif = a306_vbif_regs,
	.vbif_count = ARRAY_SIZE(a306_vbif_regs),
};

static const struct adreno_reglist a306a_vbif_regs[] = {
	{ A3XX_VBIF_ROUND_ROBIN_QOS_ARB, 0x0003 },
	{ A3XX_VBIF_OUT_RD_LIM_CONF0, 0x00000010 },
	{ A3XX_VBIF_OUT_WR_LIM_CONF0, 0x00000010 },
};

static const struct adreno_a3xx_core adreno_gpu_core_a306a = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A306A, 3, 0, 6, 0x20),
		.features = ADRENO_SOFT_FAULT_DETECT,
		.gpudev = &adreno_a3xx_gpudev,
		.perfcounters = &adreno_a3xx_perfcounters,
		.gmem_size = SZ_128K,
		.bus_width = 16,
		.snapshot_size = 600 * SZ_1K,
	},
	.pm4fw_name = "a300_pm4.fw",
	.pfpfw_name = "a300_pfp.fw",
	.vbif = a306a_vbif_regs,
	.vbif_count = ARRAY_SIZE(a306a_vbif_regs),
};

static const struct adreno_reglist a304_vbif_regs[] = {
	{ A3XX_VBIF_ROUND_ROBIN_QOS_ARB, 0x0003 },
};

static const struct adreno_a3xx_core adreno_gpu_core_a304 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A304, 3, 0, 4, 0),
		.features = ADRENO_SOFT_FAULT_DETECT,
		.gpudev = &adreno_a3xx_gpudev,
		.perfcounters = &adreno_a3xx_perfcounters,
		.gmem_size = (SZ_64K + SZ_32K),
		.bus_width = 0,
		.snapshot_size = 600 * SZ_1K,
	},
	.pm4fw_name = "a300_pm4.fw",
	.pfpfw_name = "a300_pfp.fw",
	.vbif = a304_vbif_regs,
	.vbif_count = ARRAY_SIZE(a304_vbif_regs),
};

DEFINE_DEPRECATED_CORE(a405, ADRENO_REV_A405, 4, 0, 5, ANY_ID);
DEFINE_DEPRECATED_CORE(a418, ADRENO_REV_A418, 4, 1, 8, ANY_ID);
DEFINE_DEPRECATED_CORE(a420, ADRENO_REV_A420, 4, 2, 0, ANY_ID);
DEFINE_DEPRECATED_CORE(a430, ADRENO_REV_A430, 4, 3, 0, ANY_ID);
DEFINE_DEPRECATED_CORE(a530v1, ADRENO_REV_A530, 5, 3, 0, 0);

static const struct adreno_reglist a530_hwcg_regs[] = {
	{A5XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL_SP1, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL_SP2, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL_SP3, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A5XX_RBBM_CLOCK_CNTL2_SP1, 0x02222220},
	{A5XX_RBBM_CLOCK_CNTL2_SP2, 0x02222220},
	{A5XX_RBBM_CLOCK_CNTL2_SP3, 0x02222220},
	{A5XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_HYST_SP1, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_HYST_SP2, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_HYST_SP3, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A5XX_RBBM_CLOCK_DELAY_SP1, 0x00000080},
	{A5XX_RBBM_CLOCK_DELAY_SP2, 0x00000080},
	{A5XX_RBBM_CLOCK_DELAY_SP3, 0x00000080},
	{A5XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_TP1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_TP2, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_TP3, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP2, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP3, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_TP0, 0x00002222},
	{A5XX_RBBM_CLOCK_CNTL3_TP1, 0x00002222},
	{A5XX_RBBM_CLOCK_CNTL3_TP2, 0x00002222},
	{A5XX_RBBM_CLOCK_CNTL3_TP3, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST_TP1, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST_TP2, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST_TP3, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP1, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP2, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP3, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST3_TP0, 0x00007777},
	{A5XX_RBBM_CLOCK_HYST3_TP1, 0x00007777},
	{A5XX_RBBM_CLOCK_HYST3_TP2, 0x00007777},
	{A5XX_RBBM_CLOCK_HYST3_TP3, 0x00007777},
	{A5XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY_TP1, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY_TP2, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY_TP3, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP1, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP2, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP3, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY3_TP0, 0x00001111},
	{A5XX_RBBM_CLOCK_DELAY3_TP1, 0x00001111},
	{A5XX_RBBM_CLOCK_DELAY3_TP2, 0x00001111},
	{A5XX_RBBM_CLOCK_DELAY3_TP3, 0x00001111},
	{A5XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL4_UCHE, 0x00222222},
	{A5XX_RBBM_CLOCK_HYST_UCHE, 0x00444444},
	{A5XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A5XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_RB1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_RB2, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_RB3, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB0, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB1, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB2, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB3, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL_CCU0, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_CCU1, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_CCU2, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_CCU3, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_RAC, 0x05522222},
	{A5XX_RBBM_CLOCK_CNTL2_RAC, 0x00505555},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU0, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU1, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU2, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU3, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RAC, 0x07444044},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_0, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_1, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_2, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_3, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RAC, 0x00010011},
	{A5XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A5XX_RBBM_CLOCK_MODE_GPC, 0x02222222},
	{A5XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A5XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A5XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A5XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A5XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
};

/* VBIF control registers for a530, a510, a508, a505 and a506 */
static const struct adreno_reglist a530_vbif_regs[] = {
	{A5XX_VBIF_ROUND_ROBIN_QOS_ARB, 0x00000003},
};

static const struct adreno_a5xx_core adreno_gpu_core_a530v2 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A530, 5, 3, 0, 1),
		.features = ADRENO_GPMU | ADRENO_SPTP_PC | ADRENO_LM |
			ADRENO_PREEMPTION |
			ADRENO_CONTENT_PROTECTION,
		.gpudev = &adreno_a5xx_gpudev,
		.perfcounters = &adreno_a5xx_perfcounters,
		.gmem_size = SZ_1M,
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.gpmu_tsens = 0x00060007,
	.max_power = 5448,
	.pm4fw_name = "a530_pm4.fw",
	.pfpfw_name = "a530_pfp.fw",
	.gpmufw_name = "a530_gpmu.fw2",
	.regfw_name = "a530v2_seq.fw2",
	.zap_name = "a530_zap",
	.hwcg = a530_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a530_hwcg_regs),
	.vbif = a530_vbif_regs,
	.vbif_count = ARRAY_SIZE(a530_vbif_regs),
	.highest_bank_bit = 15,
};

static const struct adreno_a5xx_core adreno_gpu_core_a530v3 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A530, 5, 3, 0, ANY_ID),
		.features = ADRENO_GPMU | ADRENO_SPTP_PC | ADRENO_LM |
			ADRENO_PREEMPTION |
			ADRENO_CONTENT_PROTECTION,
		.gpudev = &adreno_a5xx_gpudev,
		.perfcounters = &adreno_a5xx_perfcounters,
		.gmem_size = SZ_1M,
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.gpmu_tsens = 0x00060007,
	.max_power = 5448,
	.pm4fw_name = "a530_pm4.fw",
	.pfpfw_name = "a530_pfp.fw",
	.gpmufw_name = "a530v3_gpmu.fw2",
	.regfw_name = "a530v3_seq.fw2",
	.zap_name = "a530_zap",
	.hwcg = a530_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a530_hwcg_regs),
	.vbif = a530_vbif_regs,
	.vbif_count = ARRAY_SIZE(a530_vbif_regs),
	.highest_bank_bit = 15,
};

/* For a505, a506 and a508 */
static const struct adreno_reglist a50x_hwcg_regs[] = {
	{A5XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A5XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A5XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_TP0, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST3_TP0, 0x00007777},
	{A5XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY3_TP0, 0x00001111},
	{A5XX_RBBM_CLOCK_CNTL2_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL4_UCHE, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_HYST_UCHE, 0x00FFFFF4},
	{A5XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A5XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB0, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL_CCU0, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_RAC, 0x05522222},
	{A5XX_RBBM_CLOCK_CNTL2_RAC, 0x00505555},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU0, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RAC, 0x07444044},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_0, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RAC, 0x00010011},
	{A5XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A5XX_RBBM_CLOCK_MODE_GPC, 0x02222222},
	{A5XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A5XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A5XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A5XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A5XX_RBBM_CLOCK_DELAY_VFD, 0x00002222}
};

static const struct adreno_a5xx_core adreno_gpu_core_a505 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A505, 5, 0, 5, ANY_ID),
		.features = ADRENO_PREEMPTION,
		.gpudev = &adreno_a5xx_gpudev,
		.perfcounters = &adreno_a5xx_perfcounters,
		.gmem_size = (SZ_128K + SZ_8K),
		.bus_width = 16,
		.snapshot_size = SZ_1M,
	},
	.pm4fw_name = "a530_pm4.fw",
	.pfpfw_name = "a530_pfp.fw",
	.hwcg = a50x_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a50x_hwcg_regs),
	.vbif = a530_vbif_regs,
	.vbif_count = ARRAY_SIZE(a530_vbif_regs),
};

static const struct adreno_a5xx_core adreno_gpu_core_a506 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A506, 5, 0, 6, ANY_ID),
		.features = ADRENO_PREEMPTION |
			ADRENO_CONTENT_PROTECTION | ADRENO_CPZ_RETENTION,
		.gpudev = &adreno_a5xx_gpudev,
		.perfcounters = &adreno_a5xx_perfcounters,
		.gmem_size = (SZ_128K + SZ_8K),
		.bus_width = 16,
		.snapshot_size = SZ_1M,
	},
	.pm4fw_name = "a530_pm4.fw",
	.pfpfw_name = "a530_pfp.fw",
	.zap_name = "a506_zap",
	.hwcg = a50x_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a50x_hwcg_regs),
	.vbif = a530_vbif_regs,
	.vbif_count = ARRAY_SIZE(a530_vbif_regs),
	.highest_bank_bit = 14,
};

static const struct adreno_reglist a510_hwcg_regs[] = {
	{A5XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL_SP1, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A5XX_RBBM_CLOCK_CNTL2_SP1, 0x02222220},
	{A5XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_HYST_SP1, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A5XX_RBBM_CLOCK_DELAY_SP1, 0x00000080},
	{A5XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_TP1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_TP0, 0x00002222},
	{A5XX_RBBM_CLOCK_CNTL3_TP1, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST_TP1, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP1, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST3_TP0, 0x00007777},
	{A5XX_RBBM_CLOCK_HYST3_TP1, 0x00007777},
	{A5XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY_TP1, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP1, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY3_TP0, 0x00001111},
	{A5XX_RBBM_CLOCK_DELAY3_TP1, 0x00001111},
	{A5XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL4_UCHE, 0x00222222},
	{A5XX_RBBM_CLOCK_HYST_UCHE, 0x00444444},
	{A5XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A5XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_RB1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB0, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB1, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL_CCU0, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_CCU1, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_RAC, 0x05522222},
	{A5XX_RBBM_CLOCK_CNTL2_RAC, 0x00505555},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU0, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU1, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RAC, 0x07444044},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_0, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_1, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RAC, 0x00010011},
	{A5XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A5XX_RBBM_CLOCK_MODE_GPC, 0x02222222},
	{A5XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A5XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A5XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A5XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A5XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
};

static const struct adreno_a5xx_core adreno_gpu_core_a510 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A510, 5, 1, 0, ANY_ID),
		.gpudev = &adreno_a5xx_gpudev,
		.perfcounters = &adreno_a5xx_perfcounters,
		.gmem_size = SZ_256K,
		.bus_width = 16,
		.snapshot_size = SZ_1M,
	},
	.pm4fw_name = "a530_pm4.fw",
	.pfpfw_name = "a530_pfp.fw",
	.hwcg = a510_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a510_hwcg_regs),
	.vbif = a530_vbif_regs,
	.vbif_count = ARRAY_SIZE(a530_vbif_regs),
};

DEFINE_DEPRECATED_CORE(a540v1, ADRENO_REV_A540, 5, 4, 0, 0);

static const struct adreno_reglist a540_hwcg_regs[] = {
	{A5XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL_SP1, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL_SP2, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL_SP3, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A5XX_RBBM_CLOCK_CNTL2_SP1, 0x02222220},
	{A5XX_RBBM_CLOCK_CNTL2_SP2, 0x02222220},
	{A5XX_RBBM_CLOCK_CNTL2_SP3, 0x02222220},
	{A5XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_HYST_SP1, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_HYST_SP2, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_HYST_SP3, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A5XX_RBBM_CLOCK_DELAY_SP1, 0x00000080},
	{A5XX_RBBM_CLOCK_DELAY_SP2, 0x00000080},
	{A5XX_RBBM_CLOCK_DELAY_SP3, 0x00000080},
	{A5XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_TP1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_TP2, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_TP3, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP2, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP3, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_TP0, 0x00002222},
	{A5XX_RBBM_CLOCK_CNTL3_TP1, 0x00002222},
	{A5XX_RBBM_CLOCK_CNTL3_TP2, 0x00002222},
	{A5XX_RBBM_CLOCK_CNTL3_TP3, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST_TP1, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST_TP2, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST_TP3, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP1, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP2, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP3, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST3_TP0, 0x00007777},
	{A5XX_RBBM_CLOCK_HYST3_TP1, 0x00007777},
	{A5XX_RBBM_CLOCK_HYST3_TP2, 0x00007777},
	{A5XX_RBBM_CLOCK_HYST3_TP3, 0x00007777},
	{A5XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY_TP1, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY_TP2, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY_TP3, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP1, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP2, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP3, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY3_TP0, 0x00001111},
	{A5XX_RBBM_CLOCK_DELAY3_TP1, 0x00001111},
	{A5XX_RBBM_CLOCK_DELAY3_TP2, 0x00001111},
	{A5XX_RBBM_CLOCK_DELAY3_TP3, 0x00001111},
	{A5XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL4_UCHE, 0x00222222},
	{A5XX_RBBM_CLOCK_HYST_UCHE, 0x00444444},
	{A5XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A5XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_RB1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_RB2, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_RB3, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB0, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB1, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB2, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB3, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL_CCU0, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_CCU1, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_CCU2, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_CCU3, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_RAC, 0x05522222},
	{A5XX_RBBM_CLOCK_CNTL2_RAC, 0x00505555},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU0, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU1, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU2, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU3, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RAC, 0x07444044},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_0, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_1, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_2, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_3, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RAC, 0x00010011},
	{A5XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A5XX_RBBM_CLOCK_MODE_GPC, 0x02222222},
	{A5XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A5XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A5XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A5XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A5XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_GPMU, 0x00000222},
	{A5XX_RBBM_CLOCK_DELAY_GPMU, 0x00000770},
	{A5XX_RBBM_CLOCK_HYST_GPMU, 0x00000004},
};

static const struct adreno_reglist a540_vbif_regs[] = {
	{A5XX_VBIF_ROUND_ROBIN_QOS_ARB, 0x00000003},
	{A5XX_VBIF_GATE_OFF_WRREQ_EN, 0x00000009},
};

static const struct adreno_a5xx_core adreno_gpu_core_a540v2 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A540, 5, 4, 0, ANY_ID),
		.features = ADRENO_PREEMPTION |
			ADRENO_CONTENT_PROTECTION |
			ADRENO_GPMU | ADRENO_SPTP_PC,
		.gpudev = &adreno_a5xx_gpudev,
		.perfcounters = &adreno_a5xx_perfcounters,
		.gmem_size = SZ_1M,
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.gpmu_tsens = 0x000c000d,
	.max_power = 5448,
	.pm4fw_name = "a530_pm4.fw",
	.pfpfw_name = "a530_pfp.fw",
	.gpmufw_name = "a540_gpmu.fw2",
	.zap_name = "a540_zap",
	.hwcg = a540_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a540_hwcg_regs),
	.vbif = a540_vbif_regs,
	.vbif_count = ARRAY_SIZE(a540_vbif_regs),
	.highest_bank_bit = 15,
};

static const struct adreno_reglist a512_hwcg_regs[] = {
	{A5XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL_SP1, 0x02222222},
	{A5XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A5XX_RBBM_CLOCK_CNTL2_SP1, 0x02222220},
	{A5XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_HYST_SP1, 0x0000F3CF},
	{A5XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A5XX_RBBM_CLOCK_DELAY_SP1, 0x00000080},
	{A5XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_TP1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_TP1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_TP0, 0x00002222},
	{A5XX_RBBM_CLOCK_CNTL3_TP1, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST_TP1, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST2_TP1, 0x77777777},
	{A5XX_RBBM_CLOCK_HYST3_TP0, 0x00007777},
	{A5XX_RBBM_CLOCK_HYST3_TP1, 0x00007777},
	{A5XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY_TP1, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY2_TP1, 0x11111111},
	{A5XX_RBBM_CLOCK_DELAY3_TP0, 0x00001111},
	{A5XX_RBBM_CLOCK_DELAY3_TP1, 0x00001111},
	{A5XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL3_UCHE, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL4_UCHE, 0x00222222},
	{A5XX_RBBM_CLOCK_HYST_UCHE, 0x00444444},
	{A5XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A5XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL_RB1, 0x22222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB0, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL2_RB1, 0x00222222},
	{A5XX_RBBM_CLOCK_CNTL_CCU0, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_CCU1, 0x00022220},
	{A5XX_RBBM_CLOCK_CNTL_RAC, 0x05522222},
	{A5XX_RBBM_CLOCK_CNTL2_RAC, 0x00505555},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU0, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RB_CCU1, 0x04040404},
	{A5XX_RBBM_CLOCK_HYST_RAC, 0x07444044},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_0, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RB_CCU_L1_1, 0x00000002},
	{A5XX_RBBM_CLOCK_DELAY_RAC, 0x00010011},
	{A5XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A5XX_RBBM_CLOCK_MODE_GPC, 0x02222222},
	{A5XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A5XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A5XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A5XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A5XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A5XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A5XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
};

static const struct adreno_a5xx_core adreno_gpu_core_a512 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A512, 5, 1, 2, ANY_ID),
		.features = ADRENO_PREEMPTION |
			ADRENO_CONTENT_PROTECTION | ADRENO_CPZ_RETENTION,
		.gpudev = &adreno_a5xx_gpudev,
		.perfcounters = &adreno_a5xx_perfcounters,
		.gmem_size = (SZ_256K + SZ_16K),
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.pm4fw_name = "a530_pm4.fw",
	.pfpfw_name = "a530_pfp.fw",
	.zap_name = "a512_zap",
	.hwcg = a512_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a512_hwcg_regs),
	.highest_bank_bit = 14,
};

static const struct adreno_a5xx_core adreno_gpu_core_a508 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A508, 5, 0, 8, ANY_ID),
		.features = ADRENO_PREEMPTION |
			ADRENO_CONTENT_PROTECTION | ADRENO_CPZ_RETENTION,
		.gpudev = &adreno_a5xx_gpudev,
		.perfcounters = &adreno_a5xx_perfcounters,
		.gmem_size = (SZ_128K + SZ_8K),
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.pm4fw_name = "a530_pm4.fw",
	.pfpfw_name = "a530_pfp.fw",
	.zap_name = "a508_zap",
	.hwcg = a50x_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a50x_hwcg_regs),
	.vbif = a530_vbif_regs,
	.vbif_count = ARRAY_SIZE(a530_vbif_regs),
	.highest_bank_bit = 14,
};

DEFINE_DEPRECATED_CORE(a630v1, ADRENO_REV_A630, 6, 3, 0, 0);

static const struct adreno_reglist a630_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL_SP1, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL_SP2, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL_SP3, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02022220},
	{A6XX_RBBM_CLOCK_CNTL2_SP1, 0x02022220},
	{A6XX_RBBM_CLOCK_CNTL2_SP2, 0x02022220},
	{A6XX_RBBM_CLOCK_CNTL2_SP3, 0x02022220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A6XX_RBBM_CLOCK_DELAY_SP1, 0x00000080},
	{A6XX_RBBM_CLOCK_DELAY_SP2, 0x00000080},
	{A6XX_RBBM_CLOCK_DELAY_SP3, 0x00000080},
	{A6XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_HYST_SP1, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_HYST_SP2, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_HYST_SP3, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL_TP1, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL_TP2, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL_TP3, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP1, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP2, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP3, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP1, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP2, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP3, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_CNTL4_TP1, 0x00022222},
	{A6XX_RBBM_CLOCK_CNTL4_TP2, 0x00022222},
	{A6XX_RBBM_CLOCK_CNTL4_TP3, 0x00022222},
	{A6XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST_TP1, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST_TP2, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST_TP3, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP1, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP2, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP3, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP1, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP2, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP3, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00077777},
	{A6XX_RBBM_CLOCK_HYST4_TP1, 0x00077777},
	{A6XX_RBBM_CLOCK_HYST4_TP2, 0x00077777},
	{A6XX_RBBM_CLOCK_HYST4_TP3, 0x00077777},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY_TP1, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY_TP2, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY_TP3, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP1, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP2, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP3, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP1, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP2, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP3, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_DELAY4_TP1, 0x00011111},
	{A6XX_RBBM_CLOCK_DELAY4_TP2, 0x00011111},
	{A6XX_RBBM_CLOCK_DELAY4_TP3, 0x00011111},
	{A6XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_UCHE, 0x00222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE, 0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL_RB1, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL_RB2, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL_RB3, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x00002222},
	{A6XX_RBBM_CLOCK_CNTL2_RB1, 0x00002222},
	{A6XX_RBBM_CLOCK_CNTL2_RB2, 0x00002222},
	{A6XX_RBBM_CLOCK_CNTL2_RB3, 0x00002222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_CNTL_CCU1, 0x00002220},
	{A6XX_RBBM_CLOCK_CNTL_CCU2, 0x00002220},
	{A6XX_RBBM_CLOCK_CNTL_CCU3, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU1, 0x00040F00},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU2, 0x00040F00},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU3, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x05022022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x00222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555},
};

static const struct adreno_reglist a630_vbif_regs[] = {
	{A6XX_VBIF_GATE_OFF_WRREQ_EN, 0x00000009},
	{A6XX_RBBM_VBIF_CLIENT_QOS_CNTL, 0x3},
};


/* For a615, a616, a618, A619, a630, a640 and a680 */
static const struct a6xx_protected_regs a630_protected_regs[] = {
	{ A6XX_CP_PROTECT_REG + 0, 0x00000, 0x004ff, 0 },
	{ A6XX_CP_PROTECT_REG + 1, 0x00501, 0x00506, 0 },
	{ A6XX_CP_PROTECT_REG + 2, 0x0050b, 0x007ff, 0 },
	{ A6XX_CP_PROTECT_REG + 3, 0x0050e, 0x0050e, 1 },
	{ A6XX_CP_PROTECT_REG + 4, 0x00510, 0x00510, 1 },
	{ A6XX_CP_PROTECT_REG + 5, 0x00534, 0x00534, 1 },
	{ A6XX_CP_PROTECT_REG + 6, 0x00800, 0x00882, 1 },
	{ A6XX_CP_PROTECT_REG + 7, 0x008a0, 0x008a8, 1 },
	{ A6XX_CP_PROTECT_REG + 8, 0x008ab, 0x008cf, 1 },
	{ A6XX_CP_PROTECT_REG + 9, 0x008d0, 0x0098c, 0 },
	{ A6XX_CP_PROTECT_REG + 10, 0x00900, 0x0094d, 1 },
	{ A6XX_CP_PROTECT_REG + 11, 0x0098d, 0x00bff, 1 },
	{ A6XX_CP_PROTECT_REG + 12, 0x00e00, 0x00e01, 1 },
	{ A6XX_CP_PROTECT_REG + 13, 0x00e03, 0x00e0f, 1 },
	{ A6XX_CP_PROTECT_REG + 14, 0x03c00, 0x03cc3, 1 },
	{ A6XX_CP_PROTECT_REG + 15, 0x03cc4, 0x05cc3, 0 },
	{ A6XX_CP_PROTECT_REG + 16, 0x08630, 0x087ff, 1 },
	{ A6XX_CP_PROTECT_REG + 17, 0x08e00, 0x08e00, 1 },
	{ A6XX_CP_PROTECT_REG + 18, 0x08e08, 0x08e08, 1 },
	{ A6XX_CP_PROTECT_REG + 19, 0x08e50, 0x08e6f, 1 },
	{ A6XX_CP_PROTECT_REG + 20, 0x09624, 0x097ff, 1 },
	{ A6XX_CP_PROTECT_REG + 21, 0x09e70, 0x09e71, 1 },
	{ A6XX_CP_PROTECT_REG + 22, 0x09e78, 0x09fff, 1 },
	{ A6XX_CP_PROTECT_REG + 23, 0x0a630, 0x0a7ff, 1 },
	{ A6XX_CP_PROTECT_REG + 24, 0x0ae02, 0x0ae02, 1 },
	{ A6XX_CP_PROTECT_REG + 25, 0x0ae50, 0x0b17f, 1 },
	{ A6XX_CP_PROTECT_REG + 26, 0x0b604, 0x0b604, 1 },
	{ A6XX_CP_PROTECT_REG + 27, 0x0be02, 0x0be03, 1 },
	{ A6XX_CP_PROTECT_REG + 28, 0x0be20, 0x0d5ff, 1 },
	{ A6XX_CP_PROTECT_REG + 29, 0x0f000, 0x0fbff, 1 },
	{ A6XX_CP_PROTECT_REG + 30, 0x0fc00, 0x11bff, 0 },
	{ A6XX_CP_PROTECT_REG + 31, 0x11c00, 0x11c00, 1 },
	{ 0 },
};

static const struct adreno_a6xx_core adreno_gpu_core_a630v2 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A630, 6, 3, 0, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_IFPC |
			ADRENO_GPMU | ADRENO_CONTENT_PROTECTION |
			ADRENO_IOCOHERENT | ADRENO_PREEMPTION,
		.gpudev = &adreno_a630_gpudev,
		.perfcounters = &adreno_a630_perfcounters,
		.gmem_size = SZ_1M,
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.prim_fifo_threshold = 0x0018000,
	.gmu_major = 1,
	.gmu_minor = 3,
	.sqefw_name = "a630_sqe.fw",
	.gmufw_name = "a630_gmu.bin",
	.zap_name = "a630_zap",
	.hwcg = a630_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a630_hwcg_regs),
	.vbif = a630_vbif_regs,
	.vbif_count = ARRAY_SIZE(a630_vbif_regs),
	.hang_detect_cycles = 0xcfffff,
	.protected_regs = a630_protected_regs,
	.highest_bank_bit = 15,
};

/* For a615, a616, a618 and a619 */
static const struct adreno_reglist a615_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0,  0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A6XX_RBBM_CLOCK_HYST_SP0,  0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0,  0x02222222},
	{A6XX_RBBM_CLOCK_CNTL_TP1,  0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP1, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP1, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_CNTL4_TP1, 0x00022222},
	{A6XX_RBBM_CLOCK_HYST_TP0,  0x77777777},
	{A6XX_RBBM_CLOCK_HYST_TP1,  0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP1, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP1, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00077777},
	{A6XX_RBBM_CLOCK_HYST4_TP1, 0x00077777},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY_TP1, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP1, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP1, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_DELAY4_TP1, 0x00011111},
	{A6XX_RBBM_CLOCK_CNTL_UCHE,  0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_UCHE, 0x00222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE,  0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x00002222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_CNTL_CCU1, 0x00002220},
	{A6XX_RBBM_CLOCK_CNTL_CCU2, 0x00002220},
	{A6XX_RBBM_CLOCK_CNTL_CCU3, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU1, 0x00040F00},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU2, 0x00040F00},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU3, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x05022022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x00222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555}
};

/* For a615, a616, a618 and a619 */
static const struct adreno_reglist a615_gbif_regs[] = {
	{A6XX_RBBM_VBIF_CLIENT_QOS_CNTL, 0x3},
};

static const struct adreno_a6xx_core adreno_gpu_core_a615 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A615, 6, 1, 5, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_PREEMPTION |
			ADRENO_GPMU | ADRENO_CONTENT_PROTECTION | ADRENO_IFPC |
			ADRENO_IOCOHERENT,
		.gpudev = &adreno_a630_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = SZ_512K,
		.bus_width = 32,
		.snapshot_size = 600 * SZ_1K,
	},
	.prim_fifo_threshold = 0x0018000,
	.gmu_major = 1,
	.gmu_minor = 3,
	.sqefw_name = "a630_sqe.fw",
	.gmufw_name = "a630_gmu.bin",
	.zap_name = "a615_zap",
	.hwcg = a615_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a615_hwcg_regs),
	.vbif = a615_gbif_regs,
	.vbif_count = ARRAY_SIZE(a615_gbif_regs),
	.hang_detect_cycles = 0xcfffff,
	.protected_regs = a630_protected_regs,
	.highest_bank_bit = 14,
};

static const struct adreno_a6xx_core adreno_gpu_core_a618 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A618, 6, 1, 8, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_PREEMPTION |
			ADRENO_GPMU | ADRENO_CONTENT_PROTECTION | ADRENO_IFPC |
			ADRENO_IOCOHERENT,
		.gpudev = &adreno_a630_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.prim_fifo_threshold = 0x0018000,
	.gmu_major = 1,
	.gmu_minor = 7,
	.sqefw_name = "a630_sqe.fw",
	.gmufw_name = "a630_gmu.bin",
	.zap_name = "a615_zap",
	.hwcg = a615_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a615_hwcg_regs),
	.vbif = a615_gbif_regs,
	.vbif_count = ARRAY_SIZE(a615_gbif_regs),
	.hang_detect_cycles = 0x3fffff,
	.protected_regs = a630_protected_regs,
	.highest_bank_bit = 14,
};

static const struct adreno_a6xx_core adreno_gpu_core_a619 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A619, 6, 1, 9, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_PREEMPTION |
			ADRENO_GPMU | ADRENO_CONTENT_PROTECTION | ADRENO_IFPC |
			ADRENO_IOCOHERENT,
		.gpudev = &adreno_a630_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_2M,
	},
	.prim_fifo_threshold = 0x0018000,
	.gmu_major = 1,
	.gmu_minor = 9,
	.sqefw_name = "a630_sqe.fw",
	.gmufw_name = "a619_gmu.bin",
	.zap_name = "a615_zap",
	.hwcg = a615_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a615_hwcg_regs),
	.vbif = a615_gbif_regs,
	.vbif_count = ARRAY_SIZE(a615_gbif_regs),
	.hang_detect_cycles = 0x3fffff,
	.protected_regs = a630_protected_regs,
	.highest_bank_bit = 14,
};

static const struct adreno_a6xx_core adreno_gpu_core_a619_variant = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A619, 6, 1, 9, ANY_ID),
		.compatible = "qcom,adreno-gpu-a619-holi",
		.features =  ADRENO_PREEMPTION | ADRENO_CONTENT_PROTECTION,
		.gpudev = &adreno_a619_holi_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_2M,
	},
	.prim_fifo_threshold = 0x0018000,
	.sqefw_name = "a630_sqe.fw",
	.zap_name = "a615_zap",
	.hwcg = a615_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a615_hwcg_regs),
	.vbif = a615_gbif_regs,
	.vbif_count = ARRAY_SIZE(a615_gbif_regs),
	.hang_detect_cycles = 0x3fffff,
	.protected_regs = a630_protected_regs,
	.gx_cpr_toggle = true,
	.highest_bank_bit = 14,
};

static const struct adreno_reglist a620_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A6XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00077777},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x01002222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x25222022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x00222222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_TEX_FCHE, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_TEX_FCHE, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_TEX_FCHE, 0x00000777},
	{A6XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE, 0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_ISDB_CNT, 0x00000182},
	{A6XX_RBBM_RAC_THRESHOLD_CNT, 0x00000000},
	{A6XX_RBBM_SP_HYST_CNT, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555},
};

/* a620 and a650 */
static const struct adreno_reglist a650_gbif_regs[] = {
	{A6XX_GBIF_QSB_SIDE0, 0x00071620},
	{A6XX_GBIF_QSB_SIDE1, 0x00071620},
	{A6XX_GBIF_QSB_SIDE2, 0x00071620},
	{A6XX_GBIF_QSB_SIDE3, 0x00071620},
	{A6XX_RBBM_GBIF_CLIENT_QOS_CNTL, 0x3},
};

/* These are for a620 and a650 */
static const struct a6xx_protected_regs a620_protected_regs[] = {
	{ A6XX_CP_PROTECT_REG + 0, 0x00000, 0x004ff, 0 },
	{ A6XX_CP_PROTECT_REG + 1, 0x00501, 0x00506, 0 },
	{ A6XX_CP_PROTECT_REG + 2, 0x0050b, 0x007ff, 0 },
	{ A6XX_CP_PROTECT_REG + 3, 0x0050e, 0x0050e, 1 },
	{ A6XX_CP_PROTECT_REG + 4, 0x00510, 0x00510, 1 },
	{ A6XX_CP_PROTECT_REG + 5, 0x00534, 0x00534, 1 },
	{ A6XX_CP_PROTECT_REG + 6, 0x00800, 0x00882, 1 },
	{ A6XX_CP_PROTECT_REG + 7, 0x008a0, 0x008a8, 1 },
	{ A6XX_CP_PROTECT_REG + 8, 0x008ab, 0x008cf, 1 },
	{ A6XX_CP_PROTECT_REG + 9, 0x008d0, 0x0098c, 0 },
	{ A6XX_CP_PROTECT_REG + 10, 0x00900, 0x0094d, 1 },
	{ A6XX_CP_PROTECT_REG + 11, 0x0098d, 0x00bff, 1 },
	{ A6XX_CP_PROTECT_REG + 12, 0x00e00, 0x00e01, 1 },
	{ A6XX_CP_PROTECT_REG + 13, 0x00e03, 0x00e0f, 1 },
	{ A6XX_CP_PROTECT_REG + 14, 0x03c00, 0x03cc3, 1 },
	{ A6XX_CP_PROTECT_REG + 15, 0x03cc4, 0x05cc3, 0 },
	{ A6XX_CP_PROTECT_REG + 16, 0x08630, 0x087ff, 1 },
	{ A6XX_CP_PROTECT_REG + 17, 0x08e00, 0x08e00, 1 },
	{ A6XX_CP_PROTECT_REG + 18, 0x08e08, 0x08e08, 1 },
	{ A6XX_CP_PROTECT_REG + 19, 0x08e50, 0x08e6f, 1 },
	{ A6XX_CP_PROTECT_REG + 20, 0x08e80, 0x090ff, 1 },
	{ A6XX_CP_PROTECT_REG + 21, 0x09624, 0x097ff, 1 },
	{ A6XX_CP_PROTECT_REG + 22, 0x09e60, 0x09e71, 1 },
	{ A6XX_CP_PROTECT_REG + 23, 0x09e78, 0x09fff, 1 },
	{ A6XX_CP_PROTECT_REG + 24, 0x0a630, 0x0a7ff, 1 },
	{ A6XX_CP_PROTECT_REG + 25, 0x0ae02, 0x0ae02, 1 },
	{ A6XX_CP_PROTECT_REG + 26, 0x0ae50, 0x0b17f, 1 },
	{ A6XX_CP_PROTECT_REG + 27, 0x0b604, 0x0b604, 1 },
	{ A6XX_CP_PROTECT_REG + 28, 0x0b608, 0x0b60f, 1 },
	{ A6XX_CP_PROTECT_REG + 29, 0x0be02, 0x0be03, 1 },
	{ A6XX_CP_PROTECT_REG + 30, 0x0be20, 0x0d5ff, 1 },
	{ A6XX_CP_PROTECT_REG + 31, 0x0f000, 0x0fbff, 1 },
	{ A6XX_CP_PROTECT_REG + 32, 0x0fc00, 0x11bff, 0 },
	{ A6XX_CP_PROTECT_REG + 33, 0x18400, 0x1a3ff, 1 },
	{ A6XX_CP_PROTECT_REG + 34, 0x1a800, 0x1c7ff, 1 },
	{ A6XX_CP_PROTECT_REG + 35, 0x1c800, 0x1e7ff, 1 },
	{ A6XX_CP_PROTECT_REG + 36, 0x1f400, 0x1f843, 1 },
	{ A6XX_CP_PROTECT_REG + 37, 0x1f844, 0x1f8bf, 0 },
	{ A6XX_CP_PROTECT_REG + 38, 0x1f887, 0x1f8a2, 1 },
	{ A6XX_CP_PROTECT_REG + 47, 0x1f8c0, 0x1f8c0, 1 },
	{ 0 },
};

static const struct adreno_a6xx_core adreno_gpu_core_a620 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A620, 6, 2, 0, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_GPMU |
			ADRENO_CONTENT_PROTECTION | ADRENO_IOCOHERENT |
			ADRENO_IFPC | ADRENO_PREEMPTION | ADRENO_ACD |
			ADRENO_APRIV,
		.gpudev = &adreno_a630_gpudev,
		.perfcounters = &adreno_a6xx_perfcounters,
		.gmem_size = SZ_512K,
		.bus_width = 32,
		.snapshot_size = 2 * SZ_1M,
	},
	.prim_fifo_threshold = 0x0010000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a650_sqe.fw",
	.gmufw_name = "a650_gmu.bin",
	.zap_name = "a620_zap",
	.hwcg = a620_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a620_hwcg_regs),
	.vbif = a650_gbif_regs,
	.vbif_count = ARRAY_SIZE(a650_gbif_regs),
	.veto_fal10 = true,
	.hang_detect_cycles = 0x3ffff,
	.protected_regs = a620_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 14,
};

static const struct adreno_reglist a640_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A6XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00077777},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x01002222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x05222022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x00222222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_TEX_FCHE, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_TEX_FCHE, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_TEX_FCHE, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE, 0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_ISDB_CNT, 0x00000182},
	{A6XX_RBBM_RAC_THRESHOLD_CNT, 0x00000000},
	{A6XX_RBBM_SP_HYST_CNT, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555},
};

static const struct adreno_reglist a680_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A6XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_HYST_TP0, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x01002222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x05222022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x00222222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE, 0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_CLOCK_CNTL, 0x8AA8AA82},
	{A6XX_RBBM_ISDB_CNT, 0x00000182},
	{A6XX_RBBM_RAC_THRESHOLD_CNT, 0x00000000},
	{A6XX_RBBM_SP_HYST_CNT, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555},
	{A6XX_GMUAO_GMU_CGC_MODE_CNTL, 0x00020202},
	{A6XX_GMUAO_GMU_CGC_DELAY_CNTL, 0x00010111},
	{A6XX_GMUAO_GMU_CGC_HYST_CNTL, 0x00005555},
	{A6XX_GMUGX_GMU_SP_RF_CONTROL_0, 0x00000001},
	{A6XX_GMUGX_GMU_SP_RF_CONTROL_1, 0x00000001},
};

/* These apply to a640, a680, a612, a610 and a702 */
static const struct adreno_reglist a640_vbif_regs[] = {
	{A6XX_GBIF_QSB_SIDE0, 0x00071620},
	{A6XX_GBIF_QSB_SIDE1, 0x00071620},
	{A6XX_GBIF_QSB_SIDE2, 0x00071620},
	{A6XX_GBIF_QSB_SIDE3, 0x00071620},
	{A6XX_RBBM_GBIF_CLIENT_QOS_CNTL, 0x3},
};

static const struct adreno_a6xx_core adreno_gpu_core_a640 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A640, 6, 4, 0, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_GPMU |
			ADRENO_CONTENT_PROTECTION | ADRENO_IOCOHERENT |
			ADRENO_IFPC | ADRENO_PREEMPTION,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = SZ_1M, //Verified 1MB
		.bus_width = 32,
		.snapshot_size = 2 * SZ_1M,
	},
	.prim_fifo_threshold = 0x00200000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a630_sqe.fw",
	.gmufw_name = "a640_gmu.bin",
	.zap_name = "a640_zap",
	.hwcg = a640_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a640_hwcg_regs),
	.vbif = a640_vbif_regs,
	.vbif_count = ARRAY_SIZE(a640_vbif_regs),
	.hang_detect_cycles = 0xcfffff,
	.protected_regs = a630_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 15,
};

static const struct adreno_reglist a650_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A6XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00077777},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x01002222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x25222022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x00222222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_TEX_FCHE, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_TEX_FCHE, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_TEX_FCHE, 0x00000777},
	{A6XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE, 0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_ISDB_CNT, 0x00000182},
	{A6XX_RBBM_RAC_THRESHOLD_CNT, 0x00000000},
	{A6XX_RBBM_SP_HYST_CNT, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555},
};

static const struct adreno_a6xx_core adreno_gpu_core_a650 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A650, 6, 5, 0, 0),
		.features = ADRENO_RPMH | ADRENO_GPMU |
			ADRENO_IOCOHERENT | ADRENO_CONTENT_PROTECTION |
			ADRENO_IFPC | ADRENO_APRIV,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_perfcounters,
		.gmem_size = SZ_1M + SZ_128K, /* verified 1152kB */
		.bus_width = 32,
		.snapshot_size = 2 * SZ_1M,
	},
	.prim_fifo_threshold = 0x00300000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a650_sqe.fw",
	.gmufw_name = "a650_gmu.bin",
	.zap_name = "a650_zap",
	.vbif = a650_gbif_regs,
	.vbif_count = ARRAY_SIZE(a650_gbif_regs),
	.veto_fal10 = true,
	.pdc_in_aop = true,
	.hang_detect_cycles = 0xcfffff,
	.protected_regs = a620_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 16,
};

static const struct adreno_a6xx_core adreno_gpu_core_a650v2 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A650, 6, 5, 0, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_GPMU |
			ADRENO_IOCOHERENT | ADRENO_CONTENT_PROTECTION |
			ADRENO_IFPC | ADRENO_PREEMPTION | ADRENO_ACD |
			ADRENO_LM | ADRENO_APRIV,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_perfcounters,
		.gmem_size = SZ_1M + SZ_128K, /* verified 1152kB */
		.bus_width = 32,
		.snapshot_size = 2 * SZ_1M,
	},
	.prim_fifo_threshold = 0x00300000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a650_sqe.fw",
	.gmufw_name = "a650_gmu.bin",
	.zap_name = "a650_zap",
	.vbif = a650_gbif_regs,
	.vbif_count = ARRAY_SIZE(a650_gbif_regs),
	.veto_fal10 = true,
	.pdc_in_aop = true,
	.hang_detect_cycles = 0x3ffff,
	.protected_regs = a620_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 16,
};

static const struct adreno_a6xx_core adreno_gpu_core_a680 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A680, 6, 8, 0, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_GPMU |
			ADRENO_IOCOHERENT | ADRENO_CONTENT_PROTECTION |
			ADRENO_IFPC | ADRENO_PREEMPTION,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = SZ_2M,
		.bus_width = 32,
		.snapshot_size = SZ_2M,
	},
	.prim_fifo_threshold = 0x00400000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a630_sqe.fw",
	.gmufw_name = "a640_gmu.bin",
	.zap_name = "a640_zap",
	.hwcg = a680_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a680_hwcg_regs),
	.vbif = a640_vbif_regs,
	.vbif_count = ARRAY_SIZE(a640_vbif_regs),
	.hang_detect_cycles = 0xcfffff,
	.protected_regs = a630_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 16,
};

static const struct adreno_reglist a612_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000081},
	{A6XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00077777},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x01202222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x05522022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x02222222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE, 0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_ISDB_CNT, 0x00000182},
	{A6XX_RBBM_RAC_THRESHOLD_CNT, 0x00000000},
	{A6XX_RBBM_SP_HYST_CNT, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555},
};

static const struct adreno_a6xx_core adreno_gpu_core_a612 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A612, 6, 1, 2, ANY_ID),
		.features = ADRENO_CONTENT_PROTECTION |
			ADRENO_IOCOHERENT | ADRENO_PREEMPTION | ADRENO_GPMU |
			ADRENO_IFPC,
		.gpudev = &adreno_a6xx_rgmu_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = (SZ_128K + SZ_4K),
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.prim_fifo_threshold = 0x00080000,
	.sqefw_name = "a630_sqe.fw",
	.gmufw_name = "a612_rgmu.bin",
	.zap_name = "a612_zap",
	.hwcg = a612_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a612_hwcg_regs),
	.vbif = a640_vbif_regs,
	.vbif_count = ARRAY_SIZE(a640_vbif_regs),
	.hang_detect_cycles = 0x3fffff,
	.protected_regs = a630_protected_regs,
	.highest_bank_bit = 14,
};

static const struct adreno_a6xx_core adreno_gpu_core_a616 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A616, 6, 1, 6, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_PREEMPTION |
			ADRENO_GPMU | ADRENO_CONTENT_PROTECTION | ADRENO_IFPC |
			ADRENO_IOCOHERENT,
		.gpudev = &adreno_a630_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_1M,
	},
	.prim_fifo_threshold = 0x0018000,
	.gmu_major = 1,
	.gmu_minor = 3,
	.sqefw_name = "a630_sqe.fw",
	.gmufw_name = "a630_gmu.bin",
	.zap_name = "a615_zap",
	.hwcg = a615_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a615_hwcg_regs),
	.vbif = a615_gbif_regs,
	.vbif_count = ARRAY_SIZE(a615_gbif_regs),
	.hang_detect_cycles = 0xcfffff,
	.protected_regs = a630_protected_regs,
	.highest_bank_bit = 14,
};

static const struct adreno_a6xx_core adreno_gpu_core_a610 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A610, 6, 1, 0, ANY_ID),
		.features = ADRENO_CONTENT_PROTECTION |
			ADRENO_PREEMPTION,
		.gpudev = &adreno_a6xx_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = (SZ_128K + SZ_4K),
		.bus_width = 32,
	},
	.prim_fifo_threshold = 0x00080000,
	.sqefw_name = "a630_sqe.fw",
	.zap_name = "a610_zap",
	.hwcg = a612_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a612_hwcg_regs),
	.vbif = a640_vbif_regs,
	.vbif_count = ARRAY_SIZE(a640_vbif_regs),
	.hang_detect_cycles = 0x3ffff,
	.protected_regs = a630_protected_regs,
	.highest_bank_bit = 14,
};

static const struct adreno_reglist a660_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0, 0x02222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000080},
	{A6XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00077777},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x01002222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x25222022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x00222222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_TEX_FCHE, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_TEX_FCHE, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_TEX_FCHE, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE, 0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_ISDB_CNT, 0x00000182},
	{A6XX_RBBM_RAC_THRESHOLD_CNT, 0x00000000},
	{A6XX_RBBM_SP_HYST_CNT, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555},
};

/* A660 protected register list */
static const struct a6xx_protected_regs a660_protected_regs[] = {
	{ A6XX_CP_PROTECT_REG + 0, 0x00000, 0x004ff, 0 },
	{ A6XX_CP_PROTECT_REG + 1, 0x00501, 0x00506, 0 },
	{ A6XX_CP_PROTECT_REG + 2, 0x0050b, 0x007ff, 0 },
	{ A6XX_CP_PROTECT_REG + 3, 0x0050e, 0x0050e, 1 },
	{ A6XX_CP_PROTECT_REG + 4, 0x00510, 0x00510, 1 },
	{ A6XX_CP_PROTECT_REG + 5, 0x00534, 0x00534, 1 },
	{ A6XX_CP_PROTECT_REG + 6, 0x00800, 0x00882, 1 },
	{ A6XX_CP_PROTECT_REG + 7, 0x008a0, 0x008a8, 1 },
	{ A6XX_CP_PROTECT_REG + 8, 0x008ab, 0x008cf, 1 },
	{ A6XX_CP_PROTECT_REG + 9, 0x008d0, 0x0098c, 0 },
	{ A6XX_CP_PROTECT_REG + 10, 0x00900, 0x0094d, 1 },
	{ A6XX_CP_PROTECT_REG + 11, 0x0098d, 0x00bff, 1 },
	{ A6XX_CP_PROTECT_REG + 12, 0x00e00, 0x00e01, 1 },
	{ A6XX_CP_PROTECT_REG + 13, 0x00e03, 0x00e0f, 1 },
	{ A6XX_CP_PROTECT_REG + 14, 0x03c00, 0x03cc3, 1 },
	{ A6XX_CP_PROTECT_REG + 15, 0x03cc4, 0x05cc3, 0 },
	{ A6XX_CP_PROTECT_REG + 16, 0x08630, 0x087ff, 1 },
	{ A6XX_CP_PROTECT_REG + 17, 0x08e00, 0x08e00, 1 },
	{ A6XX_CP_PROTECT_REG + 18, 0x08e08, 0x08e08, 1 },
	{ A6XX_CP_PROTECT_REG + 19, 0x08e50, 0x08e6f, 1 },
	{ A6XX_CP_PROTECT_REG + 20, 0x08e80, 0x090ff, 1 },
	{ A6XX_CP_PROTECT_REG + 21, 0x09624, 0x097ff, 1 },
	{ A6XX_CP_PROTECT_REG + 22, 0x09e60, 0x09e71, 1 },
	{ A6XX_CP_PROTECT_REG + 23, 0x09e78, 0x09fff, 1 },
	{ A6XX_CP_PROTECT_REG + 24, 0x0a630, 0x0a7ff, 1 },
	{ A6XX_CP_PROTECT_REG + 25, 0x0ae02, 0x0ae02, 1 },
	{ A6XX_CP_PROTECT_REG + 26, 0x0ae50, 0x0af7f, 1 },
	{ A6XX_CP_PROTECT_REG + 27, 0x0b604, 0x0b604, 1 },
	{ A6XX_CP_PROTECT_REG + 28, 0x0b608, 0x0b60e, 1 },
	{ A6XX_CP_PROTECT_REG + 29, 0x0be02, 0x0be03, 1 },
	{ A6XX_CP_PROTECT_REG + 30, 0x0be20, 0x0bf7f, 1 },
	{ A6XX_CP_PROTECT_REG + 31, 0x0d000, 0x0d5ff, 1 },
	{ A6XX_CP_PROTECT_REG + 32, 0x0f000, 0x0fbff, 1 },
	{ A6XX_CP_PROTECT_REG + 33, 0x0fc00, 0x11bff, 0 },
	{ A6XX_CP_PROTECT_REG + 34, 0x18400, 0x1a3ff, 1 },
	{ A6XX_CP_PROTECT_REG + 35, 0x1a400, 0x1c3ff, 1 },
	{ A6XX_CP_PROTECT_REG + 36, 0x1c400, 0x1e3ff, 1 },
	{ A6XX_CP_PROTECT_REG + 37, 0x1f400, 0x1f843, 1 },
	{ A6XX_CP_PROTECT_REG + 38, 0x1f844, 0x1f8bf, 0 },
	{ A6XX_CP_PROTECT_REG + 39, 0x1f860, 0x1f860, 1 },
	{ A6XX_CP_PROTECT_REG + 40, 0x1f887, 0x1f8a2, 1 },
	{ A6XX_CP_PROTECT_REG + 47, 0x1f8c0, 0x1f8c0, 1 },
	{ 0 },
};

static const struct adreno_a6xx_core adreno_gpu_core_a660 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A660, 6, 6, 0, 0),
		.features = ADRENO_RPMH | ADRENO_GPMU | ADRENO_APRIV |
				ADRENO_IOCOHERENT | ADRENO_CONTENT_PROTECTION |
				ADRENO_IFPC | ADRENO_PREEMPTION,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_perfcounters,
		.gmem_size = SZ_1M + SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_2M,
	},
	.prim_fifo_threshold = 0x00300000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a660_sqe.fw",
	.gmufw_name = "a660_gmu.bin",
	.zap_name = "a660_zap",
	.hwcg = a660_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a660_hwcg_regs),
	.vbif = a650_gbif_regs,
	.vbif_count = ARRAY_SIZE(a650_gbif_regs),
	.hang_detect_cycles = 0xcfffff,
	.veto_fal10 = true,
	.protected_regs = a660_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 16,
	.pdc_in_aop = true,
	.ctxt_record_size = 2496 * 1024,
};

static const struct adreno_a6xx_core adreno_gpu_core_a660v2 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A660, 6, 6, 0, ANY_ID),
		.features = ADRENO_RPMH | ADRENO_GPMU | ADRENO_APRIV |
				ADRENO_IOCOHERENT | ADRENO_CONTENT_PROTECTION |
				ADRENO_IFPC | ADRENO_PREEMPTION | ADRENO_ACD |
				ADRENO_BCL,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_perfcounters,
		.gmem_size = SZ_1M + SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_2M,
	},
	.prim_fifo_threshold = 0x00300000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a660_sqe.fw",
	.gmufw_name = "a660_gmu.bin",
	.zap_name = "a660_zap",
	.hwcg = a660_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a660_hwcg_regs),
	.vbif = a650_gbif_regs,
	.vbif_count = ARRAY_SIZE(a650_gbif_regs),
	.hang_detect_cycles = 0xcfffff,
	.veto_fal10 = true,
	.protected_regs = a660_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 16,
	.pdc_in_aop = true,
	.ctxt_record_size = 2496 * 1024,
};

static const struct adreno_a6xx_core adreno_gpu_core_a642 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A642, ANY_ID, ANY_ID, ANY_ID, ANY_ID),
		.compatible = "qcom,adreno-gpu-a642",
		.features = ADRENO_RPMH | ADRENO_GPMU | ADRENO_APRIV |
				ADRENO_IOCOHERENT | ADRENO_CONTENT_PROTECTION |
				ADRENO_IFPC | ADRENO_PREEMPTION | ADRENO_ACD |
				ADRENO_BCL,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_perfcounters,
		.gmem_size = SZ_1M + SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_2M,
	},
	.prim_fifo_threshold = 0x00300000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a660_sqe.fw",
	.gmufw_name = "a660_gmu.bin",
	.zap_name = "a660_zap",
	.hwcg = a660_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a660_hwcg_regs),
	.vbif = a650_gbif_regs,
	.vbif_count = ARRAY_SIZE(a650_gbif_regs),
	.hang_detect_cycles = 0x3ffff,
	.veto_fal10 = true,
	.protected_regs = a660_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 15,
	.pdc_in_aop = true,
	.ctxt_record_size = 2496 * 1024,
};

static const struct adreno_a6xx_core adreno_gpu_core_a642l = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A642, ANY_ID, ANY_ID, ANY_ID, ANY_ID),
		.compatible = "qcom,adreno-gpu-a642l",
		.features = ADRENO_RPMH | ADRENO_GPMU | ADRENO_APRIV |
				ADRENO_IOCOHERENT | ADRENO_CONTENT_PROTECTION |
				ADRENO_PREEMPTION | ADRENO_IFPC | ADRENO_BCL |
				ADRENO_ACD,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_perfcounters,
		.gmem_size = SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_2M,
	},
	.prim_fifo_threshold = 0x00200000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a660_sqe.fw",
	.gmufw_name = "a660_gmu.bin",
	.zap_name = "a660_zap",
	.hwcg = a660_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a660_hwcg_regs),
	.vbif = a650_gbif_regs,
	.vbif_count = ARRAY_SIZE(a650_gbif_regs),
	.hang_detect_cycles = 0x3ffff,
	.veto_fal10 = true,
	.protected_regs = a660_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 15,
	.pdc_in_aop = true,
	.ctxt_record_size = 2496 * 1024,
};

static const struct adreno_a6xx_core adreno_gpu_core_a643 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A643, ANY_ID, ANY_ID, ANY_ID, ANY_ID),
		.compatible = "qcom,adreno-gpu-a643",
		.features = ADRENO_RPMH | ADRENO_GPMU | ADRENO_APRIV |
				ADRENO_IOCOHERENT | ADRENO_CONTENT_PROTECTION |
				ADRENO_PREEMPTION | ADRENO_IFPC | ADRENO_BCL |
				ADRENO_ACD,
		.gpudev = &adreno_a6xx_gmu_gpudev,
		.perfcounters = &adreno_a6xx_perfcounters,
		.gmem_size = SZ_512K,
		.bus_width = 32,
		.snapshot_size = SZ_2M,
	},
	.prim_fifo_threshold = 0x00200000,
	.gmu_major = 2,
	.gmu_minor = 0,
	.sqefw_name = "a660_sqe.fw",
	.gmufw_name = "a660_gmu.bin",
	.zap_name = "a660_zap",
	.hwcg = a660_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a660_hwcg_regs),
	.vbif = a650_gbif_regs,
	.vbif_count = ARRAY_SIZE(a650_gbif_regs),
	.hang_detect_cycles = 0x3ffff,
	.veto_fal10 = true,
	.protected_regs = a660_protected_regs,
	.disable_tseskip = true,
	.highest_bank_bit = 15,
	.pdc_in_aop = true,
	.ctxt_record_size = 2496 * 1024,
};

static const struct adreno_reglist a702_hwcg_regs[] = {
	{A6XX_RBBM_CLOCK_CNTL_SP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_SP0, 0x02222220},
	{A6XX_RBBM_CLOCK_DELAY_SP0, 0x00000081},
	{A6XX_RBBM_CLOCK_HYST_SP0, 0x0000F3CF},
	{A6XX_RBBM_CLOCK_CNTL_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL3_TP0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL4_TP0, 0x00022222},
	{A6XX_RBBM_CLOCK_DELAY_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY2_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY3_TP0, 0x11111111},
	{A6XX_RBBM_CLOCK_DELAY4_TP0, 0x00011111},
	{A6XX_RBBM_CLOCK_HYST_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST2_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST3_TP0, 0x77777777},
	{A6XX_RBBM_CLOCK_HYST4_TP0, 0x00077777},
	{A6XX_RBBM_CLOCK_CNTL_RB0, 0x22222222},
	{A6XX_RBBM_CLOCK_CNTL2_RB0, 0x01202222},
	{A6XX_RBBM_CLOCK_CNTL_CCU0, 0x00002220},
	{A6XX_RBBM_CLOCK_HYST_RB_CCU0, 0x00040F00},
	{A6XX_RBBM_CLOCK_CNTL_RAC, 0x05522022},
	{A6XX_RBBM_CLOCK_CNTL2_RAC, 0x00005555},
	{A6XX_RBBM_CLOCK_DELAY_RAC, 0x00000011},
	{A6XX_RBBM_CLOCK_HYST_RAC, 0x00445044},
	{A6XX_RBBM_CLOCK_CNTL_TSE_RAS_RBBM, 0x04222222},
	{A6XX_RBBM_CLOCK_MODE_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_MODE_GPC, 0x02222222},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ_2, 0x00000002},
	{A6XX_RBBM_CLOCK_MODE_HLSQ, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_TSE_RAS_RBBM, 0x00004000},
	{A6XX_RBBM_CLOCK_DELAY_VFD, 0x00002222},
	{A6XX_RBBM_CLOCK_DELAY_GPC, 0x00000200},
	{A6XX_RBBM_CLOCK_DELAY_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_TSE_RAS_RBBM, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_VFD, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GPC, 0x04104004},
	{A6XX_RBBM_CLOCK_HYST_HLSQ, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_UCHE, 0x22222222},
	{A6XX_RBBM_CLOCK_HYST_UCHE, 0x00000004},
	{A6XX_RBBM_CLOCK_DELAY_UCHE, 0x00000002},
	{A6XX_RBBM_ISDB_CNT, 0x00000182},
	{A6XX_RBBM_RAC_THRESHOLD_CNT, 0x00000000},
	{A6XX_RBBM_SP_HYST_CNT, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_GMU_GX, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_GMU_GX, 0x00000111},
	{A6XX_RBBM_CLOCK_HYST_GMU_GX, 0x00000555},
	{A6XX_RBBM_CLOCK_CNTL_FCHE, 0x00000222},
	{A6XX_RBBM_CLOCK_DELAY_FCHE, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_FCHE, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_GLC, 0x00222222},
	{A6XX_RBBM_CLOCK_DELAY_GLC, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_GLC, 0x00000000},
	{A6XX_RBBM_CLOCK_CNTL_MHUB, 0x00000002},
	{A6XX_RBBM_CLOCK_DELAY_MHUB, 0x00000000},
	{A6XX_RBBM_CLOCK_HYST_MHUB, 0x00000000},
};

static const struct adreno_a6xx_core adreno_gpu_core_a702 = {
	.base = {
		DEFINE_ADRENO_REV(ADRENO_REV_A702, 7, 0, 2, ANY_ID),
		.features = ADRENO_CONTENT_PROTECTION |
					ADRENO_APRIV | ADRENO_PREEMPTION,
		.gpudev = &adreno_a6xx_gpudev,
		.perfcounters = &adreno_a6xx_legacy_perfcounters,
		.gmem_size = SZ_128K,
		.bus_width = 16,
		.snapshot_size = SZ_1M,
	},
	.prim_fifo_threshold = 0x0000c000,
	.sqefw_name = "a702_sqe.fw",
	.zap_name = "a702_zap",
	.hwcg = a702_hwcg_regs,
	.hwcg_count = ARRAY_SIZE(a702_hwcg_regs),
	.vbif = a640_vbif_regs,
	.vbif_count = ARRAY_SIZE(a640_vbif_regs),
	.hang_detect_cycles = 0x3ffff,
	.protected_regs = a620_protected_regs,
	.highest_bank_bit = 14,
};

static const struct adreno_gpu_core *adreno_gpulist[] = {
	&adreno_gpu_core_a306.base,
	&adreno_gpu_core_a306a.base,
	&adreno_gpu_core_a304.base,
	&adreno_gpu_core_a405,		/* Deprecated */
	&adreno_gpu_core_a418,		/* Deprecated */
	&adreno_gpu_core_a420,		/* Deprecated */
	&adreno_gpu_core_a430,		/* Deprecated */
	&adreno_gpu_core_a530v1,	/* Deprecated */
	&adreno_gpu_core_a530v2.base,
	&adreno_gpu_core_a530v3.base,
	&adreno_gpu_core_a505.base,
	&adreno_gpu_core_a506.base,
	&adreno_gpu_core_a510.base,
	&adreno_gpu_core_a540v1,	/* Deprecated */
	&adreno_gpu_core_a540v2.base,
	&adreno_gpu_core_a512.base,
	&adreno_gpu_core_a508.base,
	&adreno_gpu_core_a630v1,	/* Deprecated */
	&adreno_gpu_core_a630v2.base,
	&adreno_gpu_core_a615.base,
	&adreno_gpu_core_a618.base,
	&adreno_gpu_core_a619.base,
	&adreno_gpu_core_a619_variant.base,
	&adreno_gpu_core_a620.base,
	&adreno_gpu_core_a640.base,
	&adreno_gpu_core_a650.base,
	&adreno_gpu_core_a650v2.base,
	&adreno_gpu_core_a660.base,
	&adreno_gpu_core_a660v2.base,
	&adreno_gpu_core_a680.base,
	&adreno_gpu_core_a612.base,
	&adreno_gpu_core_a616.base,
	&adreno_gpu_core_a610.base,
	&adreno_gpu_core_a642.base,
	&adreno_gpu_core_a642l.base,
	&adreno_gpu_core_a643.base,
	&adreno_gpu_core_a702.base,
};
