/*
 * (C) 2022 FCNT LIMITED
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
//==============================================================================
// include file
//==============================================================================
#include "sec_ts.h"
#include <linux/debugfs.h>

//==============================================================================
// define
//==============================================================================
#define TX_RX_DATA_SIZE(txcnt, rx_cnt) (txcnt * rx_cnt)
#define SELFTEST_SHORT_DATA_SIZE(txcnt, rx_cnt) (SEC_TS_SELFTEST_REPORT_SIZE + (TX_RX_DATA_SIZE(txcnt, rx_cnt) * 2))
#define SELFTEST_OTHER_DATA_SIZE(txcnt, rx_cnt) (SEC_TS_SELFTEST_REPORT_SIZE + (TX_RX_DATA_SIZE(txcnt, rx_cnt) * 2) + (txcnt * 2) + (rx_cnt * 2))
#define NOISE_READ_DATA_SIZE(txcnt, rx_cnt) (TX_RX_DATA_SIZE(txcnt, rx_cnt) * 2)
#define NOISE_DATA_SIZE(txcnt, rx_cnt) (TX_RX_DATA_SIZE(txcnt, rx_cnt) * sizeof(short))
#define NOISE_INT_DATA_SIZE(txcnt, rx_cnt) (TX_RX_DATA_SIZE(txcnt, rx_cnt) * sizeof(int))
#define DEBUGFS_STR_SIZE (SEC_CMD_BUF_SIZE * 2)

//==============================================================================
// typedef
//==============================================================================

//==============================================================================
// functions prototype
//==============================================================================
static ssize_t sec_ts_vendor_cust_version_show(struct device *dev,
					struct device_attribute *attr, char *buf);
static ssize_t sec_ts_vendor_cust_self_short_show(struct device *dev,
					struct device_attribute *attr, char *buf);
static ssize_t sec_ts_vendor_cust_state_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count);
static ssize_t sec_ts_vendor_cust_wakeup_gesture_show(struct device *dev,
		struct device_attribute *attr, char *buf);
static ssize_t sec_ts_vendor_cust_wakeup_gesture_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size);
static ssize_t sec_ts_vendor_cust_wakeup_event_show(struct device *dev,
		struct device_attribute *attr, char *buf);
static ssize_t sec_ts_vendor_cust_glove_mode_show(struct device *dev,
		struct device_attribute *attr, char *buf);
static ssize_t sec_ts_vendor_cust_glove_mode_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count);
#ifdef CONFIG_DEBUG_FS
static int sec_ts_vendor_cust_debugfs_open(struct inode *inode,
					struct file *filp);
static int sec_ts_vendor_cust_debugfs_close(struct inode *inode,
					struct file *filp);
static ssize_t sec_ts_vendor_cust_self_other_read(struct file *filp,
					char __user *buf, size_t count, loff_t *ppos);
static ssize_t sec_ts_vendor_cust_offsetcal_calib_read(struct file *filp,
					char __user *buf, size_t count, loff_t *ppos);
static int sec_ts_vendor_p2p_tmode(struct sec_ts_data *ts);
static int sec_ts_vendor_execute_p2ptest(struct sec_ts_data *ts);
static ssize_t sec_ts_vendor_cust_noise_read(struct file *filp,
					char __user *buf, size_t count, loff_t *ppos);
static ssize_t sec_ts_vendor_cust_noise_write(struct file *filp,
		const char __user *buf, size_t count, loff_t *ppos);
#endif
//==============================================================================
// private valuable
//==============================================================================
static DEVICE_ATTR(version, S_IRUGO, sec_ts_vendor_cust_version_show, NULL);
static DEVICE_ATTR(touch_state, S_IWUSR, NULL, sec_ts_vendor_cust_state_store);
static DEVICE_ATTR(self_short, S_IRUSR, sec_ts_vendor_cust_self_short_show, NULL);
static DEVICE_ATTR(wakeup_gesture, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP,
				 sec_ts_vendor_cust_wakeup_gesture_show, sec_ts_vendor_cust_wakeup_gesture_store);
static DEVICE_ATTR(wake_event, S_IRUSR | S_IRGRP, sec_ts_vendor_cust_wakeup_event_show, NULL);
static DEVICE_ATTR(glove_mode, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP,
				 sec_ts_vendor_cust_glove_mode_show, sec_ts_vendor_cust_glove_mode_store);

static struct attribute *vendor_cust_attributes[] = {
	&dev_attr_version.attr,
	&dev_attr_touch_state.attr,
	&dev_attr_self_short.attr,
	&dev_attr_wakeup_gesture.attr,
	&dev_attr_wake_event.attr,
	&dev_attr_glove_mode.attr,
	NULL,
};

static struct attribute_group vendor_cust_attr_group = {
	.attrs = vendor_cust_attributes,
};

#ifdef CONFIG_DEBUG_FS
static const struct file_operations self_other_debugfs_fops = {
	.open = sec_ts_vendor_cust_debugfs_open,
	.release = sec_ts_vendor_cust_debugfs_close,
	.read = sec_ts_vendor_cust_self_other_read,
	.write = NULL,
};

static const struct file_operations offsetcal_calib_debugfs_fops = {
	.open = sec_ts_vendor_cust_debugfs_open,
	.release = sec_ts_vendor_cust_debugfs_close,
	.read = sec_ts_vendor_cust_offsetcal_calib_read,
	.write = NULL,
};

static const struct file_operations noise_debugfs_fops = {
	.open = sec_ts_vendor_cust_debugfs_open,
	.release = sec_ts_vendor_cust_debugfs_close,
	.read = sec_ts_vendor_cust_noise_read,
	.write = sec_ts_vendor_cust_noise_write,
};
#endif
//==============================================================================
// functions
//==============================================================================
/* SYSFS */
static ssize_t sec_ts_vendor_cust_version_show(struct device *dev,
					struct device_attribute *attr, char *buf)
{
	struct sec_ts_data *ts = dev_get_drvdata(dev);
	u8 img_ver[8] = {0};
	u8 core_ver[4] = {0};
	u8 config_ver[4] = {0};
	int ret;

	if (ts->power_status == SEC_TS_STATE_POWER_OFF) {
		input_err(true, &ts->client->dev, "%s: [ERROR] Touch is stopped\n",
				__func__);
		return -EIO;
	}

	/* Image Version Read */
	ret = ts->sec_ts_i2c_read(ts, SEC_TS_READ_IMG_VERSION, img_ver, 8);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: Image version read error\n", __func__);
		return -EIO;
	}
	input_info(true, &ts->client->dev, "%s: IC Image version info : %x.%x.%x.%x\n",
			__func__, img_ver[0], img_ver[1], img_ver[2], img_ver[3]);

	/* Core Version Read */
	ret = ts->sec_ts_i2c_read(ts, SEC_TS_READ_FW_VERSION, core_ver, 4);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: core version read error\n", __func__);
		return -EIO;
	}
	input_info(true, &ts->client->dev, "%s: IC Core version info : %x.%x.%x.%x,\n",
			__func__, core_ver[0], core_ver[1], core_ver[2], core_ver[3]);

	/* Config Version Read */
	ret = ts->sec_ts_i2c_read(ts, SEC_TS_READ_PARA_VERSION, config_ver, 4);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: config version read error\n", __func__);
		return -EIO;
	}
	input_info(true, &ts->client->dev, "%s: IC config version info : %x.%x.%x.%x\n",
			__func__, config_ver[0], config_ver[1], config_ver[2], config_ver[3]);

	return snprintf(buf, SEC_CMD_BUF_SIZE,
			"IMG_VER:%02X.%02X.%02X.%02X.%02X.%02X.%02X.%02X\nCORE_VER:%02X.%02X.%02X.%02X\nCFG_VER:%02X.%02X.%02X.%02X\n",
			img_ver[0], img_ver[1], img_ver[2], img_ver[3], img_ver[4], img_ver[5], img_ver[6], img_ver[7],
			core_ver[0], core_ver[1], core_ver[2], core_ver[3],
			config_ver[0], config_ver[1], config_ver[2], config_ver[3]);
}

static ssize_t sec_ts_vendor_cust_self_short_show(struct device *dev,
					struct device_attribute *attr, char *buf)
{
	struct sec_ts_data *ts = dev_get_drvdata(dev);
	ssize_t ret = 0;
	int rc;
	u8   *rBuff = NULL;
	int size = SELFTEST_SHORT_DATA_SIZE(ts->tx_count, ts->rx_count);
	char para = TO_TOUCH_MODE;
	u8 data[2] = { 0 };
	u32 testret, rx_gnd_short, tx_gnd_short, rx_rx_short, tx_tx_short, rx_tx_short, tx_rx_short;

	testret = 0xFFFFFFFF;
	rx_gnd_short = tx_gnd_short = rx_rx_short = tx_tx_short = rx_tx_short = tx_rx_short = 0;
	disable_irq(ts->client->irq);
	if (ts->power_status == SEC_TS_STATE_POWER_OFF) {
		input_err(true, &ts->client->dev, "%s: [ERROR] Touch is stopped\n",
				__func__);
		goto error;
	}
	rBuff = vmalloc(size);
	if (!rBuff) {
		input_err(true, &ts->client->dev, "%s: Vmalloc Error\n",
				__func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: set power mode to test mode\n", __func__);
	para = TO_SELFTEST_MODE;
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_SET_POWER_MODE, &para, 1);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: set test mode failed\n", __func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: clear event stack\n", __func__);
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_CLEAR_EVENT_STACK, NULL, 0);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: clear event stack failed\n", __func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: self test short start\n", __func__);
	data[0] = 0x04;
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_SELFTEST, data, 1);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: Send selftest cmd failed!\n", __func__);
		goto error;
	}

	sec_ts_delay(700);

	rc = sec_ts_wait_for_ready(ts, SEC_TS_VENDOR_ACK_SELF_TEST_DONE);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: Selftest execution time out!\n", __func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: self test done\n", __func__);

	rc = ts->sec_ts_i2c_read(ts, SEC_TS_READ_SELFTEST_RESULT, rBuff, size);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: Selftest execution time out!\n", __func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: Test Result %02X, %02X, %02X, %02X\n",
			__func__, rBuff[16], rBuff[17], rBuff[18], rBuff[19]);

	testret = (rBuff[19] << 24) | (rBuff[18] << 16) | (rBuff[17] << 8) | rBuff[16];
	rx_gnd_short = (rBuff[59] << 24) | (rBuff[58] << 16) | (rBuff[57] << 8) | rBuff[56];
	tx_gnd_short = (rBuff[63] << 24) | (rBuff[62] << 16) | (rBuff[61] << 8) | rBuff[60];
	rx_rx_short = (rBuff[67] << 24) | (rBuff[66] << 16) | (rBuff[65] << 8) | rBuff[64];
	tx_tx_short = (rBuff[71] << 24) | (rBuff[70] << 16) | (rBuff[69] << 8) | rBuff[68];
	rx_tx_short = (rBuff[75] << 24) | (rBuff[74] << 16) | (rBuff[73] << 8) | rBuff[72];
	tx_rx_short = (rBuff[79] << 24) | (rBuff[78] << 16) | (rBuff[77] << 8) | rBuff[76];

error:
	para = TO_TOUCH_MODE;
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_SET_POWER_MODE, &para, 1);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: mode changed failed!\n", __func__);
	}

	sec_ts_reinit(ts);
	enable_irq(ts->client->irq);

	if (rBuff) {
		vfree(rBuff);
	}

	ret = snprintf(buf, SEC_CMD_BUF_SIZE,
			"STS:0x%08X\nTX_GND:0x%08X\nRX_GND:0x%08X\nRX_RX:0x%08X\nTX_TX:0x%08X\nRX_TX:0x%08X\nTX_RX:0x%08X\n",
			testret, tx_gnd_short, rx_gnd_short, rx_rx_short, tx_tx_short, rx_tx_short, tx_rx_short );

	return ret;
}

static ssize_t sec_ts_vendor_cust_state_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	unsigned long value;
	struct sec_ts_data *ts = dev_get_drvdata(dev);

	ret = kstrtoul(buf, 10, &value);
	if (ret < 0 || (value > 1)) {
		input_err(true, &ts->client->dev, "%s: Param Error\n", __func__);
		return -EINVAL;
	}

	input_info(true, &ts->client->dev, "%s: Set Touch State:%d\n", __func__, (int)value);
	if (value) {
		sec_ts_set_lowpowermode(ts, TO_TOUCH_MODE);
	} else {
		sec_ts_set_lowpowermode(ts, TO_LOWPOWER_MODE);
	}

	return count;
}

static ssize_t sec_ts_vendor_cust_wakeup_gesture_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct sec_ts_data *ts = dev_get_drvdata(dev);
	int ret = 0;

	ret = snprintf(buf, PAGE_SIZE, "%u\n", ts->wakeupgeture_en);

	return ret;
}

static ssize_t sec_ts_vendor_cust_wakeup_gesture_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct sec_ts_data *ts = dev_get_drvdata(dev);
	unsigned long value;
	int ret = 0;

	ret = kstrtoul(buf, 10, &value);
	if (ret < 0)
		return ret;

	if ((value != 0) && (value != 1)) {
		return -EINVAL;
	}

	pm_runtime_get_sync(&ts->client->dev);
	ts->wakeupgeture_en = (u8)value;
	input_info(true, &ts->client->dev, "%s: Updated wakeup_gesture = 0x%02X\n\n", __func__, ts->wakeupgeture_en);
	pm_runtime_put(&ts->client->dev);

	return count;
}

static ssize_t sec_ts_vendor_cust_wakeup_event_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct sec_ts_data *ts = dev_get_drvdata(dev);
	int ret = 0;

	ret = snprintf(buf, PAGE_SIZE, "%u\n", ts->wake_event);
	if (ret > 0) {
		ts->wake_event = false;
	}

	return ret;
}

static ssize_t sec_ts_vendor_cust_glove_mode_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct sec_ts_data *ts = dev_get_drvdata(dev);
	int ret = 0;

	ret = snprintf(buf, PAGE_SIZE, "%d\n", ts->glove_mode);

	return ret;
}

static ssize_t sec_ts_vendor_cust_glove_mode_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct sec_ts_data *ts = dev_get_drvdata(dev);
	unsigned long value;
	int ret = 0;

	ret = kstrtoul(buf, 10, &value);
	if (ret < 0)
		return ret;

	if ((value != 0) && (value != 1)) {
		input_err(true, &ts->client->dev, "%s: Param Error\n", __func__);
		return -EINVAL;
	}

	mutex_lock(&ts->modechange);
	ts->glove_mode = (int)value;
	ret = sec_ts_glove_mode_enables(ts, ts->glove_mode);
	mutex_unlock(&ts->modechange);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: glove_mode_enables failed\n", __func__);
		return -EIO;
	} else {
		input_info(true, &ts->client->dev, "%s: Updated glove_mode = 0x%02X\n\n", __func__, ts->glove_mode);
	}

	return count;
}

/* DEBUG FS */
#ifdef CONFIG_DEBUG_FS
static int sec_ts_vendor_cust_debugfs_open(struct inode *inode,
					struct file *filp)
{
	struct sec_ts_data *ts = inode->i_private;

	filp->private_data = inode->i_private;
	if (ts->selftest_str_buff) {
		vfree(ts->selftest_str_buff);
	}
	ts->selftest_str_buff = vmalloc(DEBUGFS_STR_SIZE);
	if (!ts->selftest_str_buff) {
		input_err(true, &ts->client->dev, "%s: Buffer Alloc Error\n", __func__);
		return -ENOMEM;
	}

	return 0;
}

static int sec_ts_vendor_cust_debugfs_close(struct inode *inode,
					struct file *filp)
{
	struct sec_ts_data *ts = inode->i_private;
	if (ts->selftest_str_buff) {
		vfree(ts->selftest_str_buff);
		ts->selftest_str_buff = NULL;
	}
	filp->private_data = NULL;

	return 0;
}

static ssize_t sec_ts_vendor_cust_self_other_read(struct file *filp,
					char __user *buf, size_t count, loff_t *ppos)
{
	struct sec_ts_data *ts = filp->private_data;
	ssize_t str_len = 0;
	u8   *rBuff = NULL;
	int rc, size, i, j;
	char para = TO_TOUCH_MODE;
	u8 tpara[2] = {0x23};
	u32 testret = 0;

	mutex_lock(&ts->debugfs_lock);
	if (*ppos) {
		goto release_mutex;
	}

	disable_irq(ts->client->irq);
	memset(ts->selftest_str_buff, 0, sizeof(char) * DEBUGFS_STR_SIZE);
	ts->selftest_str_idx = 0;
	if (ts->power_status == SEC_TS_STATE_POWER_OFF) {
		input_err(true, &ts->client->dev, "%s: [ERROR] Touch is stopped\n",
				__func__);
		goto error;
	}

	size = SELFTEST_OTHER_DATA_SIZE(ts->tx_count, ts->rx_count);
	rBuff = vmalloc(size);
	if (!rBuff) {
		input_err(true, &ts->client->dev, "%s: Vmalloc Error\n",
				__func__);
		goto error;
	}
	input_info(true, &ts->client->dev, "%s: set power mode to test mode\n", __func__);
	para = TO_SELFTEST_MODE;
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_SET_POWER_MODE, &para, 1);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: set test mode failed\n", __func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: clear event stack\n", __func__);
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_CLEAR_EVENT_STACK, NULL, 0);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: clear event stack failed\n", __func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: self test otherf start\n", __func__);
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_SELFTEST, tpara, 1);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: Send selftest cmd failed!\n", __func__);
		goto error;
	}

	sec_ts_delay(350);

	rc = sec_ts_wait_for_ready(ts, SEC_TS_VENDOR_ACK_SELF_TEST_DONE);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: Selftest execution time out!\n", __func__);
		goto error;
	}
	input_info(true, &ts->client->dev, "%s: self test done\n", __func__);

	rc = ts->sec_ts_i2c_read(ts, SEC_TS_READ_SELFTEST_RESULT, rBuff, size);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: Selftest execution time out!\n", __func__);
		goto error;
	}

	/* Set ResultData */
	ts->selftest_str_idx += scnprintf(ts->selftest_str_buff, DEBUGFS_STR_SIZE, "STS:OK\n");
	ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "CAP_MUT(0x%x):", ts->tx_count * ts->rx_count * 2);
	for (i = 0, j = SEC_TS_SELFTEST_REPORT_SIZE;i < (ts->tx_count * ts->rx_count * 2);i++) {
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "0x%02X,", rBuff[j + i]);
	}
	ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "\n");
	ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "CAP_TX(0x%x):",ts->tx_count * 2);
	for (i = 0, j = (SEC_TS_SELFTEST_REPORT_SIZE + (ts->tx_count * ts->rx_count * 2));i < (ts->tx_count * 2);i++) {
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "0x%02X,", rBuff[j + i]);
	}
	ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "\n");
	ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "CAP_RX(0x%x):",ts->rx_count * 2);
	for (i = 0, j = (SEC_TS_SELFTEST_REPORT_SIZE + (ts->tx_count * ts->rx_count * 2) + (ts->tx_count * 2));i < (ts->rx_count * 2);i++) {
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "0x%02X,", rBuff[j + i]);
	}
	ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx, DEBUGFS_STR_SIZE - ts->selftest_str_idx, "\n");
	testret = 1;
error:
	para = TO_TOUCH_MODE;
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_SET_POWER_MODE, &para, 1);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: mode changed failed!\n", __func__);
	}

	sec_ts_reinit(ts);
	enable_irq(ts->client->irq);
	if (rBuff) {
		vfree(rBuff);
	}

	if (testret == 0) {
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff, DEBUGFS_STR_SIZE, "STS:ERROR\n");
	}

release_mutex:
	str_len = simple_read_from_buffer(buf, count, ppos,
								ts->selftest_str_buff,
								ts->selftest_str_idx);
	mutex_unlock(&ts->debugfs_lock);

	return str_len;
}

static ssize_t sec_ts_vendor_cust_offsetcal_calib_read(struct file *filp,
					char __user *buf, size_t count, loff_t *ppos)
{
	struct sec_ts_data *ts = filp->private_data;
	int rc = 0, testret = 0;
	ssize_t str_len = 0;

	mutex_lock(&ts->debugfs_lock);
	if (*ppos) {
		goto release_mutex;
	}

	memset(ts->selftest_str_buff, 0, sizeof(char) * DEBUGFS_STR_SIZE);
	ts->selftest_str_idx = 0;
	disable_irq(ts->client->irq);
	if (ts->power_status == SEC_TS_STATE_POWER_OFF) {
		input_err(true, &ts->client->dev, "%s: [ERROR] Touch is stopped\n",
				__func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: clear event stack\n", __func__);
	rc = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_CLEAR_EVENT_STACK, NULL, 0);
	if (rc < 0) {
		input_err(true, &ts->client->dev, "%s: clear event stack failed\n", __func__);
		goto error;
	}
	sec_ts_delay(50);

	/* calibration(offset cal) */
	input_info(true, &ts->client->dev, "%s: calibration\n", __func__);
	rc = sec_ts_execute_force_calibration(ts, OFFSET_CAL_SET);
	if (rc == 0) {
		testret = 1;
	}

error:
	sec_ts_reinit(ts);
	enable_irq(ts->client->irq);

	if (testret == 1) {
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff, DEBUGFS_STR_SIZE, "STS:OK\n");
	} else {
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff, DEBUGFS_STR_SIZE, "STS:ERROR\n");
	}

release_mutex:
	str_len = simple_read_from_buffer(buf, count, ppos,
								ts->selftest_str_buff,
								ts->selftest_str_idx);
	mutex_unlock(&ts->debugfs_lock);

	return str_len;
}

static int sec_ts_vendor_p2p_tmode(struct sec_ts_data *ts)
{
	int ret = 0;
	u8 mode[3] = {0x2F,0x11,0x00};

	ret = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_CLEAR_EVENT_STACK, NULL, 0);
	ret = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_P2P_MODE, mode, sizeof(mode));
	sec_ts_delay(30);

	return ret;
}

static int sec_ts_vendor_execute_p2ptest(struct sec_ts_data *ts)
{
	int ret = -1;
	u8 test[2] = {0x00, 0x64};
	u8 tBuff[10] = {0};
	int retry = 0;

	if (ts->noise_sample_count != 0) {
		test[0] = (ts->noise_sample_count >> 8) & 0x00FF;
		test[1] =  ts->noise_sample_count & 0x00FF;
	}
	ret = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_P2P_TEST, test, sizeof(test));
	sec_ts_delay(2000);

	while (ts->sec_ts_i2c_read(ts, SEC_TS_READ_ONE_EVENT, tBuff, 8)) {
		if (((tBuff[0] >> 2) & 0xF) == TYPE_STATUS_EVENT_VENDOR_INFO) {
			if (tBuff[1] == SEC_TS_VENDOR_ACK_CMR_TEST_DONE) {
				ts->noise_p2p_max = (tBuff[2] & 0xFF) << 8 | (tBuff[3] & 0xFF);
				ts->noise_p2p_min = (tBuff[4] & 0xFF) << 8 | (tBuff[5] & 0xFF);
				ts->noise_p2p_diff = (tBuff[7] & 0xC0) << 2 | (tBuff[6] & 0xFF);
			}
		}

		if ((tBuff[7] & 0x3F) == 0x00) {
			input_info(true, &ts->client->dev, "%s: left event is 0\n", __func__);
			ret = 0;
			break;
		}

		if (retry++ > SEC_TS_WAIT_RETRY_CNT) {
			input_err(true, &ts->client->dev, "%s: Time Over\n", __func__);
			break;
		}
		sec_ts_delay(20);
	}
	return ret;
}

static ssize_t sec_ts_vendor_cust_noise_read(struct file *filp,
					char __user *buf, size_t count, loff_t *ppos)
{
	struct sec_ts_data *ts = filp->private_data;
	ssize_t str_len = 0;
	int testret = 0, ret, i, max_pos;
	char para = TO_TOUCH_MODE;
	u8 *rBuff = NULL;
	short *max_val = NULL;
	short *min_val = NULL;
	u8 type;
	int max_val_ret;
	int min_val_ret;

	mutex_lock(&ts->debugfs_lock);
	if (*ppos) {
		goto release_mutex;
	}

	disable_irq(ts->client->irq);
	memset(ts->selftest_str_buff, 0, sizeof(char) * DEBUGFS_STR_SIZE);
	ts->selftest_str_idx = 0;
	if (ts->power_status == SEC_TS_STATE_POWER_OFF) {
		input_err(true, &ts->client->dev, "%s: [ERROR] Touch is stopped\n",
				__func__);
		goto error;
	}

	rBuff = vmalloc(NOISE_READ_DATA_SIZE(ts->tx_count, ts->rx_count));
	if (!rBuff) {
		input_err(true, &ts->client->dev, "%s: Vmalloc read buffer Error\n",
				__func__);
		goto error;
	}

	max_val = vmalloc(NOISE_DATA_SIZE(ts->tx_count, ts->rx_count));
	if (!max_val) {
		input_err(true, &ts->client->dev, "%s: Vmalloc max buffer Error\n",
				__func__);
		goto error;
	}

	min_val = vmalloc(NOISE_DATA_SIZE(ts->tx_count, ts->rx_count));
	if (!min_val) {
		input_err(true, &ts->client->dev, "%s: Vmalloc min buffer Error\n",
				__func__);
		goto error;
	}

	memset(rBuff, 0, NOISE_READ_DATA_SIZE(ts->tx_count, ts->rx_count));
	memset(max_val, 0, NOISE_DATA_SIZE(ts->tx_count, ts->rx_count));
	memset(min_val, 0, NOISE_DATA_SIZE(ts->tx_count, ts->rx_count));

	input_info(true, &ts->client->dev, "%s: set p2p test mode\n", __func__);
	ret = sec_ts_vendor_p2p_tmode(ts);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: failed to fix p2p tmode\n",
				__func__);
		goto error;
	}

	input_info(true, &ts->client->dev, "%s: execute p2p test mode\n", __func__);
	ret = sec_ts_vendor_execute_p2ptest(ts);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: failed to fix p2p test\n",
				__func__);
		goto error;
	}

	/* P2P MIN Read */
	input_info(true, &ts->client->dev, "%s: raw min type\n", __func__);
	type = TYPE_RAW_DATA_P2P_MIN;
	ret = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_MUTU_RAW_TYPE, &type, 1);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: Set rawdata min type failed\n", __func__);
		goto error;
	}

	sec_ts_delay(50);

	input_info(true, &ts->client->dev, "%s: read min data\n", __func__);
	ret = ts->sec_ts_i2c_read(ts, SEC_TS_READ_TOUCH_RAWDATA, rBuff, NOISE_READ_DATA_SIZE(ts->tx_count, ts->rx_count));
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: read min rawdata failed!\n", __func__);
		goto error;
	}

	for (i = 0; i < NOISE_READ_DATA_SIZE(ts->tx_count, ts->rx_count); i += 2) {
		min_val[i / 2] = rBuff[i + 1] + (rBuff[i] << 8);
	}

	/* P2P MAX Read */
	input_info(true, &ts->client->dev, "%s: raw max type\n", __func__);
	type = TYPE_RAW_DATA_P2P_MAX;
	ret = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_MUTU_RAW_TYPE, &type, 1);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: Set rawdata type failed\n", __func__);
		goto error;
	}

	sec_ts_delay(50);

	input_info(true, &ts->client->dev, "%s: read max data\n", __func__);
	ret = ts->sec_ts_i2c_read(ts, SEC_TS_READ_TOUCH_RAWDATA, rBuff, NOISE_READ_DATA_SIZE(ts->tx_count, ts->rx_count));
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: read max rawdata failed!\n", __func__);
		goto error;
	}

	for (i = 0; i < NOISE_READ_DATA_SIZE(ts->tx_count, ts->rx_count); i += 2) {
		max_val[i / 2] = rBuff[i + 1] + (rBuff[i] << 8);
	}

	/* Check Max Noise */
	max_val_ret = max_val[0];
	max_pos = 0;
	for (i = 0; i < TX_RX_DATA_SIZE(ts->tx_count, ts->rx_count); i++) {
		if (max_val_ret < max_val[i]) {
			max_val_ret = max_val[i];
			max_pos = i;
		}
	}
	/* Check Min Noise */
	min_val_ret = min_val[0];
	for (i = 0; i < TX_RX_DATA_SIZE(ts->tx_count, ts->rx_count); i++) {
		if (min_val_ret > min_val[i]) {
			min_val_ret = min_val[i];
		}
	}
	testret = 1;

error:
	sec_ts_locked_release_all_finger(ts);
	sec_ts_delay(30);
	ret = ts->sec_ts_i2c_write(ts, SEC_TS_CMD_SET_POWER_MODE, &para, 1);
	enable_irq(ts->client->irq);

	if (testret == 1) {
		input_info(true, &ts->client->dev, "%s: noise test success. max;%d min:%d diff:%d\n",
					__func__, ts->noise_p2p_max, ts->noise_p2p_min, ts->noise_p2p_diff);
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff, DEBUGFS_STR_SIZE, "STS:OK\n");
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx,
												DEBUGFS_STR_SIZE - ts->selftest_str_idx,
												"MAX:%d\n", max_val_ret);
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx,
												DEBUGFS_STR_SIZE - ts->selftest_str_idx,
												"MIN:%d\n", min_val_ret);
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff + ts->selftest_str_idx,
												DEBUGFS_STR_SIZE - ts->selftest_str_idx,
												"POS:0x%x\n", max_pos);
	} else {
		input_info(true, &ts->client->dev, "%s: noise test error\n", __func__);
		ts->selftest_str_idx += scnprintf(ts->selftest_str_buff, DEBUGFS_STR_SIZE, "STS:ERROR\n");
	}

	if (rBuff) {
		vfree(rBuff);
	}
	if (max_val) {
		vfree(max_val);
	}
	if (min_val) {
		vfree(min_val);
	}
release_mutex:
	str_len = simple_read_from_buffer(buf, count, ppos,
								ts->selftest_str_buff,
								ts->selftest_str_idx);
	mutex_unlock(&ts->debugfs_lock);

	return str_len;
}

static ssize_t sec_ts_vendor_cust_noise_write(struct file *filp,
		const char __user *buf, size_t count, loff_t *ppos)
{
	ssize_t ret = -EINVAL;
	struct sec_ts_data *ts = filp->private_data;
	unsigned long val;
	char data[10];

	mutex_lock(&ts->debugfs_lock);
	while (1) {
		if (count >= 10) {
			break;
		}

		memset(data, 0, sizeof(data));
		if (!copy_from_user(data, buf, count)) {
			ret = kstrtoul(data, 10, &val);
			if (ret >= 0) {
				if ((val >= 1) && (val <= 65535)) {
					ts->noise_sample_count = val;
					input_info(true, &ts->client->dev, "%s: set noise count:%d\n", __func__, val);
					ret = count;
				}
			}
		}
		break;
	}
	mutex_unlock(&ts->debugfs_lock);

	return ret;
}
#endif

int sec_ts_vendor_cust_init(struct sec_ts_data *ts)
{
	int ret;

	input_info(true, &ts->client->dev, "%s: init\n", __func__);

	mutex_init(&ts->debugfs_lock);
	/* SYSFS */
	ts->vendor_class = class_create(THIS_MODULE, "touchscreen");
	if (IS_ERR(ts->vendor_class)) {
		input_err(true, &ts->client->dev, "%s: fail - class_create\n", __func__);
		ret = PTR_ERR(ts->vendor_class);
		goto err_cleate_class;
	}

	ts->vendor_dev = device_create(ts->vendor_class, NULL, 0, ts, "touch");
	if (IS_ERR(ts->vendor_dev)) {
		input_err(true, &ts->client->dev, "%s: fail - device_create\n", __func__);
		ret = PTR_ERR(ts->vendor_dev);
		goto err_cleate_device;
	}

	ret = sysfs_create_group(&ts->vendor_dev->kobj, &vendor_cust_attr_group);
	if (ret < 0) {
		input_err(true, &ts->client->dev, "%s: fail - sysfs_create_group\n", __func__);
		goto err_cleate_sysfs;
	}

#ifdef CONFIG_DEBUG_FS
	/* DEBUG FS */
	ts->debug_dentry = debugfs_create_dir("sec_ts_dbg", NULL);
	if (IS_ERR_OR_NULL(ts->debug_dentry)) {
		input_err(true, &ts->client->dev, "%s: fail - debugfs_create_dir\n", __func__);
		ret = -ENOMEM;
		goto err_debugfs_dir;
	}

	ts->self_other_dentry = debugfs_create_file("self_other", 0440, ts->debug_dentry,
													ts, &self_other_debugfs_fops);
	if (IS_ERR_OR_NULL(ts->self_other_dentry)) {
		input_err(true, &ts->client->dev, "%s: fail - debugfs_self_other\n", __func__);
		ret = -ENOMEM;
		goto err_debugfs_file;
	}
	ts->offsetcal_sdc_calib_dentry = debugfs_create_file("offsetcal_calib", 0440, ts->debug_dentry,
													ts, &offsetcal_calib_debugfs_fops);
	if (IS_ERR_OR_NULL(ts->offsetcal_sdc_calib_dentry)) {
		input_err(true, &ts->client->dev, "%s: fail - debugfs_offsetcal_sdc_calib\n", __func__);
		ret = -ENOMEM;
		goto err_debugfs_file;
	}
	ts->noise_dentry = debugfs_create_file("noise", 0660, ts->debug_dentry,
													ts, &noise_debugfs_fops);
	if (IS_ERR_OR_NULL(ts->offsetcal_sdc_calib_dentry)) {
		input_err(true, &ts->client->dev, "%s: fail - debugfs_noise_calib\n", __func__);
		ret = -ENOMEM;
		goto err_debugfs_file;
	}
#endif

	return 0;

#ifdef CONFIG_DEBUG_FS
err_debugfs_file:
	debugfs_remove_recursive(ts->debug_dentry);
err_debugfs_dir:
	sysfs_remove_group(&ts->vendor_dev->kobj, &vendor_cust_attr_group);
#endif
err_cleate_sysfs:
	device_destroy(ts->vendor_class, 0);
	ts->vendor_dev = NULL;
err_cleate_device:
	class_destroy(ts->vendor_class);
	ts->vendor_class = NULL;
err_cleate_class:
	input_err(true, &ts->client->dev, "%s: fail end\n", __func__);

	return ret;
}

void sec_ts_vendor_cust_remove(struct sec_ts_data *ts)
{
	if (!ts->vendor_class) {
		return;
	}
	input_info(true, &ts->client->dev, "%s: deinit\n", __func__);
#ifdef CONFIG_DEBUG_FS
	debugfs_remove_recursive(ts->debug_dentry);
#endif
	mutex_destroy(&ts->debugfs_lock);
	sysfs_remove_group(&ts->vendor_dev->kobj, &vendor_cust_attr_group);
	device_destroy(ts->vendor_class, 0);
	ts->vendor_dev = NULL;
	class_destroy(ts->vendor_class);
	ts->vendor_class = NULL;

	return;
}
