/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * FTS Capacitive touch screen controller (FingerTipS)
 *
 * Copyright (C) 2016-2019, STMicroelectronics Limited.
 * Authors: AMG(Analog Mems Group) <marco.cali@st.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef _LINUX_FTS_I2C_H_
#define _LINUX_FTS_I2C_H_

/*#include <linux/wakelock.h>*/
#include <linux/pm_wakeup.h>
#include <linux/timekeeping.h>
#include <linux/haven/hh_irq_lend.h>

#include "fts_lib/ftsSoftware.h"
#include "fts_lib/ftsHardware.h"
#include "fts_lib/ftsGesture.h"

#define FTS_POWER_ON     1
#define FTS_POWER_OFF    0

/****************** CONFIGURATION SECTION ******************/

/**** CODE CONFIGURATION ****/

#define FTS_TS_DRV_NAME      "fts"
#define FTS_TS_DRV_VERSION   "4.2.14" /* version */

/*#define SCRIPTLESS*/ /*allow to work in scriptless mode with the GUI*/
#ifdef SCRIPTLESS
#define SCRIPTLESS_DEBUG
/**
 * uncomment this macro definition to print debug
 * message for script less  support
 */
#endif

#define DRIVER_TEST

/* #define FW_H_FILE */ /*include the FW as header file*/
#ifdef FW_H_FILE
	#define FW_SIZE_NAME myArray_size
	#define FW_ARRAY_NAME myArray
#endif

/*#define LIMITS_H_FILE*/ /*include the limits file as header file*/
#ifdef LIMITS_H_FILE
	#define LIMITS_SIZE_NAME myArray2_size
	#define LIMITS_ARRAY_NAME myArray2
#endif

/**** END ****/


/**** FEATURES USED IN THE IC ***/
/* #define PHONE_KEY enable the keys */

#define PHONE_GESTURE /*allow to use the gestures*/
#ifdef PHONE_GESTURE
	#define USE_GESTURE_MASK
	#define USE_CUSTOM_GESTURES
#endif

#define USE_ONE_FILE_NODE
/*allow to enable/disable all the features just using one file node*/

#define EDGE_REJ
/*allow edge rej feature (comment to disable)*/

#define CORNER_REJ
/*allow corn rej feature (comment to disable)*/

#define EDGE_PALM_REJ
/*allow edge palm rej feature (comment to disable)*/

#define CHARGER_MODE
/*allow charger mode feature (comment to disable)*/

#define GLOVE_MODE
/*allow glove mode feature (comment to disable)*/

#define VR_MODE
/*allow vr mode feature (comment to disable)*/

#define COVER_MODE
/*allow cover mode feature (comment to disable)*/

#define STYLUS_MODE
/*allow stylus mode feature (comment to disable)*/

#define USE_NOISE_PARAM
/*set noise params during resume (comment to disable)*/

/**** END ****/


/**** PANEL SPECIFICATION ****/
#define X_AXIS_MAX           1440
#define X_AXIS_MIN           0
#define Y_AXIS_MAX           2880
#define Y_AXIS_MIN           0

#define PRESSURE_MIN         0
#define PRESSURE_MAX         127

#define TOUCH_ID_MAX         10

#define AREA_MIN             PRESSURE_MIN
#define AREA_MAX             PRESSURE_MAX
/**** END ****/

/*********************************************************/

/* Flash programming */

#define INIT_FLAG_CNT        3

/* KEYS */
#define KEY1                 0x02
#define KEY2                 0x01
#define KEY3                 0x04

/*
 * Configuration mode
 */
/**
 * bitmask which can assume the value defined as
 * features in ftsSoftware.h or the following values
 */

#define MODE_NOTHING         0x00000000
#define MODE_SENSEON         0x10000000
#define MODE_SENSEOFF        0x20000000
#define FEAT_GESTURE         0x40000000


/*
 * Status Event Field:
 * id of command that triggered the event
 */

#define FTS_FLASH_WRITE_CONFIG      0x03
#define FTS_FLASH_WRITE_COMP_MEMORY 0x04
#define FTS_FORCE_CAL_SELF_MUTUAL   0x05
#define FTS_FORCE_CAL_SELF          0x06
#define FTS_WATER_MODE_ON           0x07
#define FTS_WATER_MODE_OFF          0x08


#define EXP_FN_WORK_DELAY_MS 1000

#define CMD_STR_LEN		32
#define I2C_DATA_MAX_LEN	32

#ifdef SCRIPTLESS
/*
 * I2C Command Read/Write Function
 */

#define CMD_RESULT_STR_LEN   2048
#endif

#define TSP_BUF_SIZE         4096

#define PINCTRL_STATE_ACTIVE    "pmx_ts_active"
#define PINCTRL_STATE_SUSPEND   "pmx_ts_suspend"
#define PINCTRL_STATE_RELEASE   "pmx_ts_release"

/*add by guchong*/
#ifdef PHONE_GESTURE
extern u16 gesture_coordinates_x[GESTURE_COORDS_REPORT_MAX];
extern u16 gesture_coordinates_y[GESTURE_COORDS_REPORT_MAX];
extern int gesture_coords_reported;
extern struct mutex gestureMask_mutex;
#endif

struct fts_i2c_platform_data {
	bool x_flip;
	bool y_flip;
	int (*power)(bool on);
	int irq_gpio;
	int reset_gpio;
	const char *pwr_reg_name;
	const char *bus_reg_name;
	bool pwr_on_suspend;
};

/*
 * Forward declaration
 */
struct fts_ts_info;

/*
 * Dispatch event handler
 */
struct event_dispatch_handler_t {
	void (*handler)(struct fts_ts_info *info, unsigned char *data);
};

enum trusted_touch_mode_config {
	TRUSTED_TOUCH_VM_MODE,
	TRUSTED_TOUCH_MODE_NONE
};

enum trusted_touch_pvm_states {
	TRUSTED_TOUCH_PVM_INIT,
	PVM_I2C_RESOURCE_ACQUIRED,
	PVM_INTERRUPT_DISABLED,
	PVM_IOMEM_LENT,
	PVM_IOMEM_LENT_NOTIFIED,
	PVM_IRQ_LENT,
	PVM_IRQ_LENT_NOTIFIED,
	PVM_IOMEM_RELEASE_NOTIFIED,
	PVM_IRQ_RELEASE_NOTIFIED,
	PVM_ALL_RESOURCES_RELEASE_NOTIFIED,
	PVM_IRQ_RECLAIMED,
	PVM_IOMEM_RECLAIMED,
	PVM_INTERRUPT_ENABLED,
	PVM_I2C_RESOURCE_RELEASED,
	TRUSTED_TOUCH_PVM_STATE_MAX
};

enum trusted_touch_tvm_states {
	TRUSTED_TOUCH_TVM_INIT,
	TVM_IOMEM_LENT_NOTIFIED,
	TVM_IRQ_LENT_NOTIFIED,
	TVM_ALL_RESOURCES_LENT_NOTIFIED,
	TVM_IOMEM_ACCEPTED,
	TVM_I2C_SESSION_ACQUIRED,
	TVM_IRQ_ACCEPTED,
	TVM_INTERRUPT_ENABLED,
	TVM_INTERRUPT_DISABLED,
	TVM_IRQ_RELEASED,
	TVM_I2C_SESSION_RELEASED,
	TVM_IOMEM_RELEASED,
	TRUSTED_TOUCH_TVM_STATE_MAX
};

#ifdef CONFIG_ST_TRUSTED_TOUCH
#define TRUSTED_TOUCH_MEM_LABEL 0x7

#define TOUCH_RESET_GPIO_BASE 0xF116000
#define TOUCH_RESET_GPIO_SIZE 0x1000
#define TOUCH_RESET_GPIO_OFFSET 0x4
#define TOUCH_INTR_GPIO_BASE 0xF117000
#define TOUCH_INTR_GPIO_SIZE 0x1000
#define TOUCH_INTR_GPIO_OFFSET 0x8

#define TRUSTED_TOUCH_EVENT_LEND_FAILURE -1
#define TRUSTED_TOUCH_EVENT_LEND_NOTIFICATION_FAILURE -2
#define TRUSTED_TOUCH_EVENT_ACCEPT_FAILURE -3
#define	TRUSTED_TOUCH_EVENT_FUNCTIONAL_FAILURE -4
#define	TRUSTED_TOUCH_EVENT_RELEASE_FAILURE -5
#define	TRUSTED_TOUCH_EVENT_RECLAIM_FAILURE -6
#define	TRUSTED_TOUCH_EVENT_I2C_FAILURE -7
#define	TRUSTED_TOUCH_EVENT_NOTIFICATIONS_PENDING 5

struct trusted_touch_vm_info {
	enum hh_irq_label irq_label;
	enum hh_vm_names vm_name;
	u32 hw_irq;
	hh_memparcel_handle_t vm_mem_handle;
	u32 *iomem_bases;
	u32 *iomem_sizes;
	u32 iomem_list_size;
	void *mem_cookie;
#ifdef CONFIG_ARCH_QTI_VM
	struct mutex tvm_state_mutex;
	atomic_t tvm_state;
#else
	struct mutex pvm_state_mutex;
	atomic_t pvm_state;
#endif
};

#ifdef CONFIG_ARCH_QTI_VM
void fts_trusted_touch_tvm_i2c_failure_report(struct fts_ts_info *info);
#endif
#endif

/*
 * struct fts_ts_info - FTS capacitive touch screen device information
 * @dev:                  Pointer to the structure device
 * @client:               I2C client structure
 * @input_dev             Input device structure
 * @work                  Work thread
 * @event_wq              Event queue for work thread
 * @event_dispatch_table  Event dispatch table handlers
 * @attrs                 SysFS attributes
 * @mode                  Device operating mode (bitmask)
 * @touch_id              Bitmask for touch id (mapped to input slots)
 * @stylus_id             Bitmask for tracking the stylus touches
 * (mapped using the touchId)
 * @timer                 Timer when operating in polling mode
 * @power                 Power on/off routine
 * @bdata                 HW info retrived from device tree
 * @pwr_reg               DVDD power regulator
 * @bus_reg               AVDD power regulator
 * @resume_bit            Indicate if screen off/on
 * @fwupdate_stat         Store the result of a fw update triggered by the host
 * @notifier              Used for be notified from a suspend/resume event
 * @sensor_sleep          true susped was called, false resume was called
 * @wakelock              Wake Lock struct
 * @input_report_mutex    mutex for handling the pressure of keys
 * @series of switches    to store the enabling status of a particular
 * feature from the host
 */
struct fts_ts_info {
	struct device            *dev;
	struct i2c_client        *client;
	struct input_dev         *input_dev;

	struct work_struct       work;
	struct work_struct       suspend_work;
	struct work_struct       resume_work;
	struct workqueue_struct  *event_wq;

	struct delayed_work      fwu_work;
	struct workqueue_struct  *fwu_workqueue;
	struct completion        cmd_done;

	struct pinctrl *ts_pinctrl;
	struct pinctrl_state *pinctrl_state_active;
	struct pinctrl_state *pinctrl_state_suspend;
	struct pinctrl_state *pinctrl_state_release;

	struct event_dispatch_handler_t *event_dispatch_table;

	struct attribute_group    attrs;

	unsigned int              mode;
	unsigned long             touch_id;
#ifdef STYLUS_MODE
	unsigned long             stylus_id;
#endif


#ifdef FTS_USE_POLLING_MODE
	struct hrtimer            timer;
#endif


#ifdef SCRIPTLESS
	/*I2C cmd*/
	struct device             *i2c_cmd_dev;
	char cmd_read_result[CMD_RESULT_STR_LEN];
	char cmd_wr_result[CMD_RESULT_STR_LEN];
	char cmd_write_result[20];
#endif

#ifdef DRIVER_TEST
	struct device             *test_cmd_dev;
#endif
	int (*power)(bool on);

	struct fts_i2c_platform_data *bdata;
	struct regulator          *pwr_reg;
	struct regulator          *bus_reg;

	int resume_bit;
	int fwupdate_stat;

	struct notifier_block notifier;
	bool sensor_sleep;
	struct wakeup_source *wakeup_source;

	/* input lock */
	struct mutex input_report_mutex;

	/*switches for features*/
	unsigned int gesture_enabled;
	unsigned int glove_enabled;
	unsigned int charger_enabled;
	unsigned int stylus_enabled;
	unsigned int vr_enabled;
	unsigned int cover_enabled;
	unsigned int edge_rej_enabled;
	unsigned int corner_rej_enabled;
	unsigned int edge_palm_rej_enabled;

	uint8_t *i2c_data;
	uint8_t i2c_data_len;

	struct device *aoi_cmd_dev;
	bool aoi_notify_enabled;
	bool aoi_wake_on_suspend;

	/* aoi region */
	struct class *aoi_class;
	struct device *aoi_dev;
	int aoi_left;
	int aoi_top;
	int aoi_bottom;
	int aoi_right;

#ifdef CONFIG_ST_TRUSTED_TOUCH
	struct trusted_touch_vm_info *vm_info;
	struct mutex fts_clk_io_ctrl_mutex;
	const char *touch_environment;
	struct completion trusted_touch_powerdown;
	struct clk *core_clk;
	struct clk *iface_clk;
	atomic_t trusted_touch_initialized;
	atomic_t trusted_touch_underway;
	atomic_t trusted_touch_enabled;
	atomic_t trusted_touch_event;
	atomic_t trusted_touch_abort_status;
	atomic_t delayed_vm_probe_pending;
	atomic_t trusted_touch_mode;
#endif
};

extern struct chipInfo ftsInfo;

int fts_chip_powercycle(struct fts_ts_info *info);
int fts_chip_powercycle2(struct fts_ts_info *info, unsigned long sleep);
void release_all_touches(struct fts_ts_info *info);
/*int fts_get_fw_version(struct fts_ts_info *info);*/
/*extern unsigned int le_to_uint(const unsigned char *ptr);*/
/*extern unsigned int be_to_uint(const unsigned char *ptr);*/
extern int input_register_notifier_client(struct notifier_block *nb);
extern int input_unregister_notifier_client(struct notifier_block *nb);

extern struct attribute_group aoi_cmd_attr_group;
extern struct attribute_group aoi_enable_attr_group;

#ifdef SCRIPTLESS
extern struct attribute_group i2c_cmd_attr_group;
#endif

#ifdef DRIVER_TEST
extern struct attribute_group test_cmd_attr_group;
#endif


#endif

