// SPDX-License-Identifier: GPL-2.0
/*----------------------------------------------------------------------------*/
// (C) 2022 FCNT LIMITED
/*----------------------------------------------------------------------------*/

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <linux/mfd/core.h>
#include <linux/mfd/max77729.h>
#include <linux/mfd/max77729-private.h>
#include <linux/debugfs.h>

static ssize_t max77729_word_data_read_file(struct file *file, char __user *user_buf,
				    size_t count, loff_t *ppos)
{
	struct max77729_debugfs_data *data = file->private_data;
	int value, len;
	char buf[32];

	value = max77729_read_word(data->client, data->address);

	if (value >= 0) {
		len = snprintf(buf, sizeof(buf), "0x%04x\n", value);
	} else {
		len = snprintf(buf, sizeof(buf), "Error:%d\n", value);
	}

	return simple_read_from_buffer(user_buf, count, ppos, buf, len);
}

static ssize_t max77729_word_data_write_file(struct file *file,
				     const char __user *user_buf,
				     size_t count, loff_t *ppos)
{
	struct max77729_debugfs_data *data = file->private_data;
	u16 value;
	int ret;

	ret = kstrtou16_from_user(user_buf, count, 0, &value);
	if (ret)
		return ret;

	ret = max77729_write_word(data->client, data->address, value);
	if (ret)
		return ret;

	return count;
}

static const struct file_operations maxim_word_data_fops = {
	.open = simple_open,
	.read = max77729_word_data_read_file,
	.write = max77729_word_data_write_file,
	.llseek = default_llseek,
};

static ssize_t max77729_data_read_file(struct file *file, char __user *user_buf,
				    size_t count, loff_t *ppos)
{
	struct max77729_debugfs_data *data = file->private_data;
	u8 value, len;
	char buf[32];
	int ret;

	ret = max77729_read_reg(data->client, data->address, &value);

	if (!ret) {
		len = snprintf(buf, sizeof(buf), "0x%02x\n", value);
	} else {
		len = snprintf(buf, sizeof(buf), "Error:%d\n", ret);
	}

	return simple_read_from_buffer(user_buf, count, ppos, buf, len);
}

static ssize_t max77729_data_write_file(struct file *file,
				     const char __user *user_buf,
				     size_t count, loff_t *ppos)
{
	struct max77729_debugfs_data *data = file->private_data;
	u8 value;
	int ret;

	ret = kstrtou8_from_user(user_buf, count, 0, &value);
	if (ret)
		return ret;

	ret = max77729_write_reg(data->client, data->address, value);
	if (ret)
		return ret;

	return count;
}

static const struct file_operations maxim_data_fops = {
	.open = simple_open,
	.read = max77729_data_read_file,
	.write = max77729_data_write_file,
	.llseek = default_llseek,
};

static void max77729_debugfs_create_file(struct max77729_debugfs_data *data)
{
	debugfs_create_x8("address", 0600, data->node,
		    &data->address);
	debugfs_create_file("data", 0600, data->node,
		    data, &maxim_data_fops);
	debugfs_create_file("word_data", 0600, data->node,
		    data, &maxim_word_data_fops);

	return;
}

static void *max77729_debugfs_create(const char *name, struct dentry *root, struct i2c_client *client)
{
	struct max77729_debugfs_data *data;
	data = kzalloc(sizeof(struct max77729_debugfs_data), GFP_KERNEL);
	if (!data)
		return NULL;

	data->node = debugfs_create_dir(name, root);
	data->client = client;
	max77729_debugfs_create_file(data);

	return data;
}


void max77729_debugfs_init(struct max77729_dev *max77729)
{
	max77729->debugfs_root = debugfs_create_dir("max77729", NULL);

	max77729->debug_pmic = max77729_debugfs_create("pmic", max77729->debugfs_root, max77729->i2c);
	max77729->debug_charger = max77729_debugfs_create("charger", max77729->debugfs_root, max77729->charger);
	max77729->debug_fuelgauge = max77729_debugfs_create("fuelgauge", max77729->debugfs_root, max77729->fuelgauge);
	max77729->debug_muic = max77729_debugfs_create("usbc", max77729->debugfs_root, max77729->muic);

	return;
}

void max77729_debugfs_remove(struct max77729_dev *max77729)
{
	if (!max77729->debugfs_root) {
		debugfs_remove_recursive(max77729->debugfs_root);
		kfree(max77729->debug_pmic);
		kfree(max77729->debug_charger);
		kfree(max77729->debug_fuelgauge);
		kfree(max77729->debug_muic);
	}

	return;
}
