/*
 * (C) 2022 FCNT LIMITED
*/
// SPDX-License-Identifier: GPL-2.0

//==============================================================================
// include file
//==============================================================================
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/errno.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/power_supply.h>
#include <linux/printk.h>
#include <linux/moduleparam.h>
#include <linux/usb/typec/maxim/max77729_usbc.h>

#include "custom_det.h"
#include "max77729_charger.h"

#ifdef FEATURE_OEM_CHARGE
#include <linux/mfd/oem_charger.h>
#endif /* FEATURE_OEM_CHARGE */

//==============================================================================
// define & enum
//==============================================================================
#ifndef FEATURE_OEM_CHARGE
	#define OEM_CHG_ERROR_OVP_NONE			/* dummy */(0)
	#define OEM_CHG_ERROR_OVP_DETECT		/* dummy */(1)
	#define OEM_CHG_ERROR_USB_HOT_DETECT	/* dummy */(2)
	#define OEM_CHG_ERROR_USB_HOT_NONE		/* dummy */(3)
	#define OEM_CHG_USB_CURRENT				/* dummy */(1)
	#define OEM_CHG_OFF_CURRENT				/* dummy */(0)
	#define oem_chg_usb_vbus_draw(type)		/* nop */
	#define oem_chg_ac_vbus_draw(type)		/* nop */
	#define oem_chg_holder_vbus_draw(type)	/* nop */
	#define oem_chg_other_vbus_draw(type)	/* nop */
	#define oem_chg_apsd_vbus_draw(type)	/* nop */
	#define oem_chg_notify_error(type)		/* nop */
#endif /* FEATURE_OEM_CHARGE */

typedef enum {
	CUSTOM_DET_IRQ_FREE = 0,
	CUSTOM_DET_IRQ_ENABLE,
	CUSTOM_DET_IRQ_DISABLE
} CUSTOM_DET_IRQ_OPE;

/*
 * GPIO
 */
typedef enum {
	CUSTOM_DET_GPIO_OTG_EN,
	CUSTOM_DET_GPIO_TOTAL,
} CUSTOM_DET_GPIO_TYPE;

typedef enum {
	CUSTOM_DET_GPIO_FREE = 0,
	CUSTOM_DET_GPIO_IN,
	CUSTOM_DET_GPIO_OUT_L,
	CUSTOM_DET_GPIO_OUT_H,
	CUSTOM_DET_GPIO_GET
} CUSTOM_DET_GPIO_OPE;

#define CUSTOM_DET_GPIO(p, type)			((p)->gpios[(type)])
#define CUSTOM_DET_GPIO_IS_VALID(p, type)	gpio_is_valid((p)->gpios[(type)])

#define cls_dev_to_custom_det(d)	container_of(d, struct custom_det_platform_data, class_dev)

//==============================================================================
// struct
//==============================================================================
/*
 * GPIO Setting Information
 */
struct custom_det_gpio_setup_info {
	char * devtree_name;
	char * name;
	CUSTOM_DET_GPIO_OPE setting;
};


/*
 * CUSTOM_DET_DRIVER Platform Data
 */
struct custom_det_platform_data {
	struct max77729_charger_data *chg;			/* chg device */
	int		gpios[CUSTOM_DET_GPIO_TOTAL];
	struct device	class_dev;
	bool create_class;
};

//==============================================================================
// const
//==============================================================================
static const struct custom_det_gpio_setup_info custom_det_gpio_setup[CUSTOM_DET_GPIO_TOTAL] = {
	{"custom,otg_en-gpio",		"OEMDET_OTG_EN",		CUSTOM_DET_GPIO_OUT_L},
};

//==============================================================================
// static variable
//==============================================================================
static struct custom_det_platform_data custom_platform_data;

static int usb_conn_therm = 0;

//==============================================================================
// external reference
//==============================================================================
extern struct max77729_usbc_platform_data *g_usbc_data;

//==============================================================================
// functions
//==============================================================================
static void custom_det_classdev_release(struct device *dev)
{
	return;
}

static int custom_det_prepare(struct device *dev)
{
	return 0;
}

static const struct dev_pm_ops custom_det_ops = {
	.prepare = custom_det_prepare,
};

static struct class custom_det_class = {
	.name		= "custom_det",
	.dev_release	= custom_det_classdev_release,
	.pm		= &custom_det_ops,
};

static bool _custom_det_gpio_control_init(struct device_node *node, struct custom_det_platform_data *custom_pdata, CUSTOM_DET_GPIO_TYPE type, char* dt_name, char* gpio_name)
{
	bool ret = false;
	int rc;

	do {
		custom_pdata->gpios[type] = of_get_named_gpio_flags(node, dt_name, 0, NULL);

		if (!CUSTOM_DET_GPIO_IS_VALID(custom_pdata, type)) {
			CUSTOM_DET_LOG("%s invalid gpio %d\n", dt_name, custom_pdata->gpios[type]);
			break;
		}
		rc = gpio_request(custom_pdata->gpios[type], gpio_name);
		if (rc != 0) {
			CUSTOM_DET_LOG("gpio_request err %s rc=%d\n", gpio_name, rc);
			custom_pdata->gpios[type] = -1;
			break;
		}
		ret = true;
	} while (0);

	return ret;
}

static int _custom_det_gpio_control(struct custom_det_platform_data *custom_pdata, CUSTOM_DET_GPIO_TYPE type, CUSTOM_DET_GPIO_OPE ope)
{
	int ret = -1;

	if (CUSTOM_DET_GPIO_IS_VALID(custom_pdata, type)) {
		ret = 0;
		switch (ope) {
		case CUSTOM_DET_GPIO_FREE:
			gpio_free(custom_pdata->gpios[type]);
			custom_pdata->gpios[type] = -1;
			break;
		case CUSTOM_DET_GPIO_IN:
			gpio_direction_input(custom_pdata->gpios[type]);
			break;
		case CUSTOM_DET_GPIO_OUT_L:
			gpio_direction_output(custom_pdata->gpios[type], 0);
			break;
		case CUSTOM_DET_GPIO_OUT_H:
			gpio_direction_output(custom_pdata->gpios[type], 1);
			break;
		case CUSTOM_DET_GPIO_GET:
			ret = gpio_get_value(custom_pdata->gpios[type]);
		default:
			break;
		}
	}

	return ret;
}

void custom_det_otg_en_set(bool enable)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;

	CUSTOM_DET_LOG("%s:%d\n", __func__, enable);

	if (enable) {
		_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_OTG_EN, CUSTOM_DET_GPIO_OUT_H);
	} else {
		_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_OTG_EN, CUSTOM_DET_GPIO_OUT_L);
	}
}

void custom_det_charge_req(CUSTOM_DET_CHARGE_TYPE type, bool enable)
{
	switch (type) {
	case CUSTOM_DET_CHARGE_APSD:
		if (enable) {
			oem_chg_apsd_vbus_draw(OEM_CHG_APSD_CURRENT);
		} else {
			oem_chg_apsd_vbus_draw(OEM_CHG_OFF_CURRENT);
		}
		break;
	}

	CUSTOM_DET_LOG("%s value:%d\n", __func__, enable);

	return;
}

static ssize_t custom_det_water_det_store(struct device *dev,
		struct device_attribute *attr, const char *data, size_t len)
{
	int ret = 0;
	struct power_supply *psy_usb;
	int water_det;
	union power_supply_propval val = {0,};

	if (!g_usbc_data) {
		CUSTOM_DET_LOG("%s g_usbc_data null\n", __func__);
		return -ENODEV;
	}

	if (kstrtoint(data, 0, &water_det)) {
		CUSTOM_DET_LOG("%s invalid value\n", __func__);
		return -EINVAL;
	}

	psy_usb = power_supply_get_by_name("usb");

	if (!psy_usb) {
		pr_err("%s: fail to get psy usb\n", __func__);
		return -ENODEV;
	}

	if (water_det) {
		g_usbc_data->current_connstat = WATER;
	} else {
		g_usbc_data->current_connstat = DRY;
	}

	val.intval = g_usbc_data->current_connstat;
	ret = power_supply_set_property(psy_usb,
		(enum power_supply_property)POWER_SUPPLY_EXT_PROP_WATER_DETECT, &val);

	power_supply_changed(psy_usb);
	CUSTOM_DET_LOG("%s value:%d ret:%d\n", __func__, water_det, ret);

	return len;
}

static ssize_t custom_det_water_det_show(struct device *dev,
		struct device_attribute *attr, char *data)
{
	int water_det = DRY;

	if (!g_usbc_data) {
		CUSTOM_DET_LOG("%s g_usbc_data null\n", __func__);
		return -ENODEV;
	}

	if (g_usbc_data->current_connstat == WATER) {
		water_det = 1;
	} else if (g_usbc_data->current_connstat == DRY) {
		water_det = 0;
	}

	return sprintf(data, "%d\n", water_det);
}

static DEVICE_ATTR(water_det,  S_IWUSR | S_IRUGO, custom_det_water_det_show, custom_det_water_det_store);

static ssize_t custom_det_otg_stop_store(struct device *dev,
		struct device_attribute *attr, const char *data, size_t len)
{
	struct custom_det_platform_data *pdata = cls_dev_to_custom_det(dev);
	int ret = 0;
	struct power_supply *psy_otg;
	union power_supply_propval val;
	bool otg_stop;

	if (!pdata) {
		CUSTOM_DET_LOG("%s custom pdata null\n", __func__);
		return -ENODEV;
	}

	if (strtobool(data, &otg_stop)) {
		CUSTOM_DET_LOG("%s invalid value\n", __func__);
		return -EINVAL;
	}

	psy_otg = power_supply_get_by_name("otg");

	if (!psy_otg) {
		pr_err("%s: fail to get psy battery\n", __func__);
		return -ENODEV;
	}

	if (otg_stop) {
		val.intval = 0;
		ret = psy_otg->desc->set_property(psy_otg, POWER_SUPPLY_PROP_ONLINE, &val);
	}

	CUSTOM_DET_LOG("%s value:%d ret:%d\n", __func__, otg_stop, ret);

	return len;
}

static DEVICE_ATTR(otg_stop,  S_IWUSR, NULL, custom_det_otg_stop_store);

int custom_det_set_charge_control_limit(int set_val)
{
	if (set_val < 0 || set_val > CUSTOM_DET_HOT_DETECT)
		return -EINVAL;

	if (usb_conn_therm != set_val) {
		if (set_val == CUSTOM_DET_HOT_DETECT) {
			oem_chg_notify_error(OEM_CHG_ERROR_USB_HOT_DETECT);
			usb_conn_therm = CUSTOM_DET_HOT_DETECT;
		} else if (usb_conn_therm == CUSTOM_DET_HOT_DETECT
			&& set_val < CUSTOM_DET_HOT_DETECT) {
			oem_chg_notify_error(OEM_CHG_ERROR_USB_HOT_NONE);
			usb_conn_therm = CUSTOM_DET_HOT_NONE;
		} else {
			pr_debug("usb_conn_therm: skip error notification\n");
			return 0;
		}
		CUSTOM_DET_RECLOG("%s safety error usbhot=%d\n", __func__, usb_conn_therm);
	}
	return 0;
}
EXPORT_SYMBOL_GPL(custom_det_set_charge_control_limit);

int custom_det_get_charge_control_limit_max(void)
{
	return CUSTOM_DET_HOT_DETECT;
}
EXPORT_SYMBOL_GPL(custom_det_get_charge_control_limit_max);

int custom_det_get_charge_control_limit(void)
{
	return usb_conn_therm;
}
EXPORT_SYMBOL_GPL(custom_det_get_charge_control_limit);

#define CUSTOM_DET_GPIO_SETUP(node, pdata, type, dt_name, gpio_name, ope)		\
do {																		\
	if (!_custom_det_gpio_control_init(node, pdata, type, dt_name, gpio_name))	\
		break;																\
	_custom_det_gpio_control(custom_pdata, type, ope);								\
} while (0)

int custom_det_init(struct max77729_charger_data *chg)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	struct device_node *np;
	CUSTOM_DET_GPIO_TYPE gpio_i;
	int ret;

	memset(custom_pdata, 0, sizeof(struct custom_det_platform_data));

	custom_pdata->chg = chg;

	np = of_find_node_by_name(NULL, "max77729-charger");
	if (!np) {
		CUSTOM_DET_LOG("np(max77729-charger) NULL\n", __func__);
		return -ENODEV;
	}

	for (gpio_i = 0; gpio_i < CUSTOM_DET_GPIO_TOTAL; gpio_i++) {
		CUSTOM_DET_GPIO_SETUP(np, custom_pdata, gpio_i,
							custom_det_gpio_setup[gpio_i].devtree_name,
							custom_det_gpio_setup[gpio_i].name,
							custom_det_gpio_setup[gpio_i].setting);
	}

	ret = class_register(&custom_det_class);
	if (ret) {
		pr_err("%s: class register failed: %d\n", __func__);
		return ret;
	}

	device_initialize(&custom_pdata->class_dev);
	custom_pdata->class_dev.parent = custom_pdata->chg->dev;
	custom_pdata->class_dev.class = &custom_det_class;
	dev_set_drvdata(&custom_pdata->class_dev, custom_pdata);

	ret = dev_set_name(&custom_pdata->class_dev, "%s", "custom_det");
	if (ret) {
		CUSTOM_DET_LOG("Failed to set name %d\n", ret);
		goto classdev_put;
	}

	ret = device_add(&custom_pdata->class_dev);
	if (ret) {
		CUSTOM_DET_LOG("Failed to device add %d\n", ret);
		goto classdev_put;
	}

	ret = device_create_file(&custom_pdata->class_dev, &dev_attr_otg_stop);
	if (ret) {
		CUSTOM_DET_LOG("Failed to create setting sysfs entry %d\n", ret);
		goto device_del;
	}

	ret = device_create_file(&custom_pdata->class_dev, &dev_attr_water_det);
	if (ret) {
		CUSTOM_DET_LOG("Failed to create setting sysfs entry %d\n", ret);
		goto device_del;
	}

	custom_pdata->create_class = true;

	return 0;

device_del:
	device_del(&custom_pdata->class_dev);
classdev_put:
	put_device(&custom_pdata->class_dev);
	class_unregister(&custom_det_class);
	return ret;
}

void custom_det_shutdown(void)
{
	return;
}

int custom_det_remove(void)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	CUSTOM_DET_GPIO_TYPE gpio_i;

	if (custom_pdata->create_class) {
		device_remove_file(&custom_pdata->class_dev, &dev_attr_otg_stop);
		device_del(&custom_pdata->class_dev);
		put_device(&custom_pdata->class_dev);
		class_unregister(&custom_det_class);
		custom_pdata->create_class = false;
	}

	for (gpio_i = 0; gpio_i < CUSTOM_DET_GPIO_TOTAL; gpio_i++) {
		_custom_det_gpio_control(custom_pdata, gpio_i, CUSTOM_DET_GPIO_FREE);
	}

	return 0;
}

MODULE_DESCRIPTION("CUSTOM DET Driver");
MODULE_LICENSE("GPL v2");
