// SPDX-License-Identifier: GPL-2.0+
/*----------------------------------------------------------------------------*/
// (C) 2022 FCNT LIMITED
/*----------------------------------------------------------------------------*/
#include <linux/module.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/of.h>
#include <linux/nonvolatile_common.h>
#include <linux/freezer.h>
#include <linux/kthread.h>
#include <linux/pm_wakeup.h>
#include <linux/wait.h>
#include <linux/printk.h>
#include <linux/interrupt.h>
#include <linux/soc/qcom/smem_custom.h>
#include <linux/mfd/max77729.h>
#include <linux/mfd/max77729-private.h>
#include "max77729_fuelgauge.h"
#include "oem_battery_drv.h"

#include <linux/mfd/oem_charger.h>
#include "oem_charger_local.h"

#define BATT_DRV_DBGLOG(x, y...)				\
	if (unlikely(batt_drv_debug != 0)) {		\
		printk(KERN_ERR "[oem_batt] " x, ## y);	\
	}

#define BATT_DRV_INFOLOG(x, y...)				printk(KERN_INFO "[oem_batt] " x, ## y)
#define BATT_DRV_WARNLOG(x, y...)				printk(KERN_WARNING "[oem_batt] " x, ## y)
#define BATT_DRV_ERRLOG(x, y...)				printk(KERN_ERR "[oem_batt] " x, ## y)
#define BATT_DRV_RECLOG(x, y...)				printk("REC@REC@36[oem_batt] " x, ## y)

/* monitor time */
#define BATT_DRV_MONITOR_DELAY_100MS			msecs_to_jiffies(100)
#define BATT_DRV_MONITOR_DELAY_500MS			msecs_to_jiffies(500)
#define BATT_DRV_MONITOR_DELAY_1S				msecs_to_jiffies(1000)
#define BATT_DRV_MONITOR_DELAY_10S				msecs_to_jiffies(10000)

#define BATT_DRV_TIMER_10S						(10 * HZ)
#define BATT_DRV_TIMER_30S						(30 * HZ)
#define BATT_DRV_TIMER_60S						(60 * HZ)
#define BATT_DRV_TIMER_180S						(180 * HZ)

/* thread events */
#define BATT_DRV_EVENT_SUSPEND					0
#define BATT_DRV_EVENT_MONITOR					1

/* battery monitor time */
#define BATT_DRV_MONITOR_INTERVAL				BATT_DRV_MONITOR_DELAY_10S
#define BATT_DRV_RESUME_INTERVAL				BATT_DRV_MONITOR_DELAY_500MS
#define BATT_DRV_LOW_VOLTAGE_INTERVAL			BATT_DRV_MONITOR_DELAY_1S
#define BATT_DRV_SHUTDOWN_VOLTAGE_INTERVAL		BATT_DRV_MONITOR_DELAY_100MS

#define BATT_SOC_INIT							101
#define BATT_MSOC_INIT							0xfff
#define BATT_SOC_FULL							100
#define BATT_SOC_ZERO							0
#define BATT_AGE_OFFSET							100

#define BATT_CYCLE_COUNT_BACKUP_THRESHOLD		10
#define BATT_ADJUST_TEMP_TBL_MAX				10

#define BATT_CYCLE_COUNT_HI_SOC_DEFAULT			90
#define BATT_CYCLE_COUNT_HI_TEMP_DEFAULT		400
#define BATT_CYCLE_CHARGE_DEFAULT				100
#define BATT_CYCLE_FULLSOCTHR_DEFAULT			0x5f00

#define BATT_RECHARGE_SOC_DEFAULT				98
#define BATT_FULL_KEEP_DEFAULT					99
#define BATT_LONGEVITY_FULL_SOC_MIN_DEFAULT		85
#define BATT_LONGEVITY_FULL_SOC_MAX_DEFAULT		87
#define BATT_LONGEVITY_RECHARGE_SOC_DEFAULT		85
#define BATT_LOW_VOLTAGE_DEFAULT				3600
#define BATT_SHUTDOWN_VOLTAGE_DEFAULT			3200
#define BATT_SHUTDOWN_COUNT_DEFAULT				3
#define BATT_LONGEVITY_DESIGN_CAP_DEFAULT		4520
#define BATT_OOPS_DEFAULT						0x3

#define BATT_FAKE_STATUS_INIT					0xff
#define BATT_FAKE_HEALTH_INIT					0xff

#define BATT_LONGEVITY_CYCLE_CORRECTION(x)		(x * 100 / 85)

static char batt_drv_status_str[][16] = {
	"UNKNOWN",
	"CHARGING",
	"DISCHARGING",
	"NOT_CHARGING",
	"FULL",
};

static char batt_drv_health_str[][32] = {
	"UNKNOWN",
	"GOOD",
	"OVERHEAT",
	"DEAD",
	"OVERVOLTAGE",
	"UNSPEC_FAILURE",
	"COLD",
	"WATCHDOG_TIMER_EXPIRE",
	"SAFETY_TIMER_EXPIRE",
	"OVERCURRENT",
	"CALIBRATION_REQUIRED",
	"WARM",
	"COOL",
	"HOT",
};

static enum power_supply_property batt_drv_properties[] = {
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_HEALTH,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CAPACITY,
	POWER_SUPPLY_PROP_TEMP,
	POWER_SUPPLY_PROP_CYCLE_COUNT,
};

enum batt_drv_func_limit {
	BATT_DRV_FUNC_LIMIT_CHARGE = 0,			/*  0 */
	BATT_DRV_FUNC_LIMIT_BATTERY_PRESENT,	/*  1 */
	BATT_DRV_FUNC_LIMIT_LOW_BATTERY,		/*  2 */
	BATT_DRV_FUNC_LIMIT_RESERVE_01,			/*  3 */
	BATT_DRV_FUNC_LIMIT_BATTERY_TEMP,		/*  4 */
	BATT_DRV_FUNC_LIMIT_RESERVE_02,			/*  5 */
	BATT_DRV_FUNC_LIMIT_RESERVE_03,			/*  6 */
	BATT_DRV_FUNC_LIMIT_RESERVE_04,			/*  7 */
	BATT_DRV_FUNC_LIMIT_NUM,
};

enum batt_drv_keep_capacity {
	BATT_DRV_KEEP_CAPACITY_DISABLE = 0,
	BATT_DRV_KEEP_CAPACITY_STANDBY,
	BATT_DRV_KEEP_CAPACITY_ENABLE,
};

struct charging_setting {
	uint8_t		hi_capacity_cycle_ratio;
	uint8_t		hi_temp_cycle_ratio;
	uint8_t		unused1;
	uint8_t		unused2;
	uint8_t		unused3;
	uint8_t		unused4;
};

struct charging_data {
	uint32_t	charge_count;
	uint32_t	hi_capacity_count;
	uint32_t	hi_temp_count;
	uint8_t		unused1;
	uint8_t		unused2;
	uint16_t	unused3;
};

struct batt_drv_dt {
	int	recharge_soc_threshold;
	int	full_keep_threshold;

	int	low_voltage_threshold;
	int	shutdown_voltage_threshold;
	int	shutdown_count;

	int	longevity_full_soc_min_threshold;
	int	longevity_full_soc_max_threshold;
	int	longevity_recharge_soc_threshold;

	int	cycle_count_hi_soc_threshold;
	int	cycle_count_hi_temp_threshold;
	int	cycle_charge_threshold_lv1;
	int	cycle_charge_threshold_lv2;

	int	cycle_full_soc_threshold;

	int	temp_correction_num;
	int	temp_correction_values[BATT_ADJUST_TEMP_TBL_MAX];
	int	temp_correction_thresholds[BATT_ADJUST_TEMP_TBL_MAX];

	int design_cap;
};

struct oem_battery_drv_chip {
	struct max77729_fuelgauge_data		*fuelgauge;

	struct power_supply_desc			oem_battery_psy_desc;
	struct power_supply					*oem_battery_psy;

	struct device						*dev;
	struct device_node					*of_node;

	struct batt_drv_dt					dt_param;

	struct workqueue_struct				*batt_drv_wq;
	struct delayed_work					battery_monitor;

	int									status;
	int									health;
	int									soc;
	int									fake_soc;
	int									old_msoc;
	int									vcell;

	/* monitor event state */
	unsigned long						event_state;

	unsigned long						keep_capacity_flag;
	unsigned long						keep_capacity_time;

	unsigned int						shutdown_voltage_count;
	unsigned int						start_soc_value;
	bool								lowbatt_flag;

	/* charging data */
	struct charging_setting				chg_setting;
	struct charging_data				chg_data;
	uint32_t	longevity_cycle;
	uint32_t	charge_count_val;
	uint32_t	hi_capacity_count_val;
	uint32_t	hi_temp_count_val;
	uint32_t	longevity_count_val;
	int									total_cycle_count;
	enum chg_cycle_level				cycle_chg_lvl;

	unsigned char						soc_mask_flag;
	unsigned long						soc_mask_expire;
	unsigned char						soc_charging_mask_flag;
	unsigned long						soc_charging_mask_expire;

	unsigned int						incremental_soc;
	unsigned int						current_soc;
	bool								longevity_charge_flag;

	bool								check_disable[BATT_DRV_FUNC_LIMIT_NUM];

	struct mutex						lock;
	wait_queue_head_t					wq;
	void*								thread;

	struct device	class_dev;
	bool create_class;
};

const static int capacity_convert[101] =
{
	  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
	 10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
	 20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
	 30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
	 40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
	 50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
	 60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
	 70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
	 80,  81,  82,  83,  84,  85,  86,  87,  88,  89,
	 90,  91,  92,  93,  94,  95,  96,  97,  98,  99,
	100,
};

const static int capacity_convert_cycle_chg[101] =
{
	  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
	 10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
	 20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
	 30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
	 40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
	 50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
	 60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
	 70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
	 80,  81,  82,  83,  84,  85,  86,  87,  88,  89,
	 90,  91,  92,  93,  94,  95,  96,  97,  98,  99,
	100,
};

static struct oem_battery_drv_chip *oem_batt_drv_chip = NULL;
static int drv_initialized = 0; /* Driver init flag (0:no initialize, 1:initialized) */
static int battery_status = POWER_SUPPLY_STATUS_DISCHARGING;

extern int max77729_get_fuelgauge_value(struct max77729_fuelgauge_data *fuelgauge, int data);

static int fake_status = BATT_FAKE_STATUS_INIT;
static int batt_drv_set_fake_status(const char *val, const struct kernel_param *kp)
{
	int ret = 0;

	ret = param_set_int(val, kp);
	if ((ret) ||
		(fake_status < POWER_SUPPLY_STATUS_UNKNOWN) ||
		(fake_status > POWER_SUPPLY_STATUS_FULL)) {
		BATT_DRV_ERRLOG("[%s] error setting value %d\n",__func__, ret);
		fake_status = BATT_FAKE_STATUS_INIT;
	}

	return ret;
}
module_param_call(dbg_status, batt_drv_set_fake_status, param_get_uint, &fake_status, 0644);

static int fake_health = BATT_FAKE_HEALTH_INIT;
static int batt_drv_set_fake_health(const char *val, const struct kernel_param *kp)
{
	int ret = 0;

	ret = param_set_int(val, kp);
	if ((ret) ||
		(fake_health < POWER_SUPPLY_HEALTH_UNKNOWN) ||
		(fake_health > POWER_SUPPLY_HEALTH_HOT)) {
		BATT_DRV_ERRLOG("[%s] error setting value %d\n",__func__, ret);
		fake_health = BATT_FAKE_HEALTH_INIT;
	}

	return ret;
}
module_param_call(dbg_health, batt_drv_set_fake_health, param_get_uint, &fake_health, 0644);

static int batt_drv_debug = 0;
module_param_named(dbglog_battery, batt_drv_debug, int, S_IRUGO|S_IWUSR);

static int batt_drv_debug_cycle = 0;
module_param_named(dbg_count_up, batt_drv_debug_cycle, int, S_IRUGO|S_IWUSR);

static void batt_drv_function_limits_check(struct oem_battery_drv_chip *chip)
{
	int ret = 0;
	u16 val = 0;
	int i = 0;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	/*
	 * Limits function classification
	 * check_disable[0] : charge
	 * check_disable[1] : battery present
	 * check_disable[2] : low battery
	 * check_disable[3] : safety timer
	 * check_disable[4] : battery temperature
	 * check_disable[5] : reserve
	 * check_disable[6] : reserve
	 * check_disable[7] : reserve
	 */
	ret = get_nonvolatile((uint8_t*)&val, APNV_CHARGE_FG_FUNC_LIMITS_I, 2);
	if (unlikely(ret < 0)) {
		val = 0x0000;
		BATT_DRV_ERRLOG("[%s] NV read err : %d set value = 0x%x \n", __func__, ret, val);
	}
	if (val != 0x0000) {
		for (i = 0;i < BATT_DRV_FUNC_LIMIT_NUM;i++) {
			if (val & 0x0001) {
				chip->check_disable[i] = true;
			} else {
				chip->check_disable[i] = false;
			}
			val = val >> 1;
			BATT_DRV_DBGLOG("[%s] check_disable[%d] = %d\n", __func__, i, chip->check_disable[i]);
		}
	}
}

static int batt_drv_get_soc(struct oem_battery_drv_chip *chip,
									   int *battery_capacity)
{
	int ret = 0;
	int fg_soc = 0;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	fg_soc = max77729_get_fuelgauge_value(chip->fuelgauge, FG_LEVEL);

	if (fg_soc < 0) {
		/* Failure */
		BATT_DRV_ERRLOG("[%s] Get soc FAILURE! %d\n", __func__, fg_soc);
		ret = -1;
	} else {
		fg_soc = (fg_soc+9) / 10;
		if (chip->cycle_chg_lvl == CHG_CYCLE_LEVEL_02) {
			*battery_capacity = capacity_convert_cycle_chg[fg_soc];
		} else {
			*battery_capacity = capacity_convert[fg_soc];
		}

		if (chip->old_msoc != fg_soc) {
			BATT_DRV_INFOLOG("[%s] capacity:%d old_msoc:0x%03x -> msoc:0x%03x\n",
								__func__, *battery_capacity, chip->old_msoc, fg_soc);
			chip->old_msoc = fg_soc;
		}
	}

	return ret;
}

static int batt_drv_set_soc(struct oem_battery_drv_chip *chip,
									   int battery_capacity)
{
	int old_soc;
	unsigned long now = jiffies;
	int old_status;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	if (chip->cycle_chg_lvl != CHG_CYCLE_LEVEL_NONE) {
	  chip->soc = battery_capacity;
	}

	old_soc = chip->soc;
	old_status = chip->status;

	if (chip->lowbatt_flag) {
		chip->soc = BATT_SOC_ZERO;
	} else if ((chip->status == POWER_SUPPLY_STATUS_CHARGING) && (chip->soc_charging_mask_flag == 1)) {
		if (time_before(chip->soc_charging_mask_expire, now)) {
			chip->soc_charging_mask_flag = 0;
			chip->status = POWER_SUPPLY_STATUS_FULL;
		}
	} else if (chip->status == POWER_SUPPLY_STATUS_FULL) {
		if (chip->longevity_charge_flag) {
			if (battery_capacity < chip->dt_param.longevity_recharge_soc_threshold) {
				chip->soc = battery_capacity;
				chip->status = POWER_SUPPLY_STATUS_CHARGING;
			} else if (battery_capacity < chip->soc) {
				chip->soc--;
			} else if (battery_capacity > chip->dt_param.longevity_full_soc_max_threshold) {

			} else {
				chip->soc = chip->dt_param.longevity_full_soc_min_threshold;
			}
		} else {
			if (battery_capacity < chip->dt_param.recharge_soc_threshold) {
				chip->soc = battery_capacity;
				chip->status = POWER_SUPPLY_STATUS_CHARGING;
			} else {
				chip->soc = BATT_SOC_FULL;
			}
		}
	} else if (time_before(now, chip->soc_mask_expire) && chip->soc_mask_flag) {
		BATT_DRV_DBGLOG("[%s] not update soc indicator\n", __func__);
		if (time_before((now + BATT_DRV_TIMER_60S), chip->soc_mask_expire)) {
			BATT_DRV_RECLOG("[%s] Error!!! soc_mask_expire is Unknown Value. now:%lu, expire:%lu\n", __func__, now, chip->soc_mask_expire);
			chip->soc_mask_flag = 0;
			chip->soc_mask_expire = 0;
		}
	} else if (chip->status == POWER_SUPPLY_STATUS_CHARGING) {
		if (battery_capacity < chip->soc) {
			chip->soc--;
		} else {
			chip->soc = battery_capacity;
			if (chip->longevity_charge_flag) {
				if (chip->soc >= chip->dt_param.longevity_full_soc_min_threshold) {
					chip->status = POWER_SUPPLY_STATUS_FULL;
				}
			} else {
				if (chip->soc >= BATT_SOC_FULL) {
					chip->status = POWER_SUPPLY_STATUS_FULL;
				}
			}
		}
	} else {
		chip->soc_mask_flag = 0;
		chip->soc_mask_expire = 0;
		if (battery_capacity < chip->soc) {
			chip->soc--;
		} else {
			chip->soc = battery_capacity;
		}
	}
	if (chip->longevity_charge_flag) {
		if ((chip->soc >= chip->dt_param.longevity_full_soc_min_threshold) &&
			(chip->soc <= chip->dt_param.longevity_full_soc_max_threshold)) {
			chip->soc = chip->dt_param.longevity_full_soc_min_threshold;
		}
	}

	if ((old_soc != chip->soc) ||
		(old_status != chip->status)){
		BATT_DRV_INFOLOG("[%s] old_soc:%d new_soc:%d set_soc:%d old_status:%s new_status:%s volt:%dmV\n",
						   __func__, old_soc, chip->soc, battery_capacity,
						   batt_drv_status_str[old_status],
						   batt_drv_status_str[chip->status], chip->vcell);
	}

	return 0;
}

static int batt_drv_prop_age(struct oem_battery_drv_chip *chip)
{
	int age_val;
	int fullcaprep;
	int cycle_count;
	int longevity_cycle_count = 0;

	fullcaprep = max77729_get_fuelgauge_value(chip->fuelgauge, FG_FULLCAPREP);
	if (fullcaprep < 0) {
		BATT_DRV_ERRLOG("[%s] Get FULLCAPREP failed. ret = %d\n", __func__, fullcaprep);
		fullcaprep = chip->dt_param.design_cap;
	}

	age_val = ((fullcaprep * 100) + (chip->dt_param.design_cap - 1)) / chip->dt_param.design_cap;
	if (chip->longevity_charge_flag) {
		longevity_cycle_count = chip->longevity_cycle + BATT_LONGEVITY_CYCLE_CORRECTION(chip->longevity_count_val);
		longevity_cycle_count = longevity_cycle_count / 100;
		age_val = age_val - (longevity_cycle_count / 100);
	}

	cycle_count = oem_batt_cycle_count();

	if (cycle_count <= 50) {
		age_val = 100;
		goto age_result;
	}

	if (age_val > 100) {
		age_val = 100;
		goto age_result;
	}

age_result:
	BATT_DRV_DBGLOG("[%s] fullcaprep:%d designcap:%d cycle_count:%d"
					" longevity_cycle_count:%d result:%d\n",
					__func__, fullcaprep, chip->dt_param.design_cap,
					cycle_count, longevity_cycle_count, age_val);

	return age_val;
}

static int batt_drv_proc_cycle_chg(struct oem_battery_drv_chip *chip)
{
	enum chg_cycle_level val = CHG_CYCLE_LEVEL_NONE;
	struct device_node *node = chip->of_node;
	int ret = 0;
	int fullsocthr = 0;

	BATT_DRV_DBGLOG("[%s] in cycle_chg_level=%d \n", __func__, chip->cycle_chg_lvl);

	if (chip->cycle_chg_lvl == CHG_CYCLE_LEVEL_02) {
		goto batt_exit;
	}

	if (chip->status != POWER_SUPPLY_STATUS_DISCHARGING) {
		goto batt_exit;
	}

	if (drv_initialized == 0) {
		goto batt_exit;
	}

	chip->total_cycle_count =
		chip->chg_data.charge_count +
		(chip->chg_data.hi_capacity_count * chip->chg_setting.hi_capacity_cycle_ratio) +
		(chip->chg_data.hi_temp_count * chip->chg_setting.hi_temp_cycle_ratio);

	if (chip->longevity_charge_flag) {
		chip->total_cycle_count =
			chip->total_cycle_count +
			(BATT_LONGEVITY_CYCLE_CORRECTION(chip->charge_count_val)) +
			(BATT_LONGEVITY_CYCLE_CORRECTION(chip->hi_capacity_count_val) * chip->chg_setting.hi_capacity_cycle_ratio) +
			(BATT_LONGEVITY_CYCLE_CORRECTION(chip->hi_temp_count_val) * chip->chg_setting.hi_temp_cycle_ratio);
	} else {
		chip->total_cycle_count =
			chip->total_cycle_count +
			chip->charge_count_val +
			(chip->hi_capacity_count_val * chip->chg_setting.hi_capacity_cycle_ratio) +
			(chip->hi_temp_count_val * chip->chg_setting.hi_temp_cycle_ratio);
	}

	BATT_DRV_DBGLOG("[%s] in total_cycle_count=%d cycle_threshold_lv1=%d cycle_threshold_lv2=%d\n",
					__func__, chip->total_cycle_count, chip->dt_param.cycle_charge_threshold_lv1, chip->dt_param.cycle_charge_threshold_lv2);

#if 0
	if (chip->total_cycle_count >= (chip->dt_param.cycle_charge_threshold_lv1*BATT_SOC_FULL)) {
		val = CHG_CYCLE_LEVEL_01;
	}
	if (chip->total_cycle_count >= (chip->dt_param.cycle_charge_threshold_lv2*BATT_SOC_FULL)) {
		val = CHG_CYCLE_LEVEL_02;
	}
#else
	val = CHG_CYCLE_LEVEL_NONE;
#endif

	if (chip->cycle_chg_lvl != val) {
		switch (val) {
		case CHG_CYCLE_LEVEL_NONE:
			ret = of_property_read_u32(node, "fcnt,batt-cycle-none-full-soc-thr", &fullsocthr);
			if (ret < 0)
				fullsocthr = BATT_CYCLE_FULLSOCTHR_DEFAULT;
			break;

		case CHG_CYCLE_LEVEL_01:
			ret = of_property_read_u32(node, "fcnt,batt-cycle-lv1-full-soc-thr", &fullsocthr);
			if (ret < 0)
				fullsocthr = BATT_CYCLE_FULLSOCTHR_DEFAULT;
			break;

		case CHG_CYCLE_LEVEL_02:
		default:
			ret = of_property_read_u32(node, "fcnt,batt-cycle-lv2-full-soc-thr", &fullsocthr);
			if (ret < 0)
				fullsocthr = BATT_CYCLE_FULLSOCTHR_DEFAULT;
			break;
		}
		max77729_write_word(chip->fuelgauge->i2c, FULLSOCTHR_REG, fullsocthr);

		chip->dt_param.cycle_full_soc_threshold = fullsocthr;
		chip->cycle_chg_lvl = val;
	}

batt_exit:
	return chip->cycle_chg_lvl;
}

static int batt_drv_set_misc(struct oem_battery_drv_chip *chip)
{
	struct device_node *node = chip->of_node;
	u8 data[2];
	int ret;
	int oops = 0;

	ret = of_property_read_u32(node, "fcnt,batt-oops", &oops);
	if (ret < 0)
		oops = BATT_OOPS_DEFAULT;

	ret = max77729_bulk_read(chip->fuelgauge->i2c, MISCCFG_REG, 2, data);
	if (ret) {
		BATT_DRV_ERRLOG("%s: Failed to read MISCCFG_REG %d\n", __func__, ret);
		return ret;
	}

	data[1] &= ~(0xF0);
	data[1] |= (oops << 4);

	ret = max77729_bulk_write(chip->fuelgauge->i2c, MISCCFG_REG, 2, data);
	if (ret) {
		BATT_DRV_ERRLOG("%s: Failed to write MISCCFG_REG %d\n", __func__, ret);
		return ret;
	}

	BATT_DRV_INFOLOG("[%s] Set MISC_CFG_REG 0x%x%x\n", __func__, data[1],data[0]);

	return ret;
}

static void batt_drv_get_charge_cycle(struct oem_battery_drv_chip *chip)
{
	int ret = 0;
	struct charging_setting *chg_setting = &chip->chg_setting;
	struct charging_data *chg_data = &chip->chg_data;
	uint32_t *longevity_cycle = &chip->longevity_cycle;

	/* get the charging data from nv */
	ret = get_nonvolatile((uint8_t*)chg_data, APNV_CHARGING_DATA, sizeof(struct charging_data));
	if (unlikely(ret < 0)) {
		BATT_DRV_ERRLOG("[%s] get nonvolatile %d error\n", __func__, APNV_CHARGING_DATA);
	}

	ret = get_nonvolatile((uint8_t*)chg_setting, APNV_CHARGING_SETTING, sizeof(struct charging_setting));
	if (unlikely(ret < 0)) {
		BATT_DRV_ERRLOG("[%s] get nonvolatile %d error\n", __func__, APNV_CHARGING_SETTING);
	}

	ret = get_nonvolatile((uint8_t*)longevity_cycle, APNV_CHARGE_LONGEVITY_CYCLE, 4);
	if (unlikely(ret < 0)) {
		BATT_DRV_ERRLOG("[%s] get nonvolatile %d error\n", __func__, APNV_CHARGE_LONGEVITY_CYCLE);
	}

	BATT_DRV_RECLOG("[%s] charge_count:%d, hi_capacity_count:%d, hi_temp_count:%d\n", __func__,
						chg_data->charge_count, chg_data->hi_capacity_count, chg_data->hi_temp_count);
	BATT_DRV_RECLOG("[%s] hi_capacity_cycle_ratio:%d, hi_temp_cycle_ratio:%d\n", __func__,
						chg_setting->hi_capacity_cycle_ratio, chg_setting->hi_temp_cycle_ratio);

	batt_drv_proc_cycle_chg(chip);

	return;
}

static void batt_drv_longevity_count_clear(struct oem_battery_drv_chip *chip)
{
	int ret = 0;
	uint32_t longevity_cycle_data = 0;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	chip->longevity_count_val = 0;
	chip->longevity_cycle = 0;

	ret = set_nonvolatile((uint8_t*)&longevity_cycle_data, APNV_CHARGE_LONGEVITY_CYCLE, 4);
	if (unlikely(ret < 0)) {
		BATT_DRV_ERRLOG("[%s] set nonvolatile %d error\n", __func__, APNV_CHARGE_LONGEVITY_CYCLE);
	}
}

static void batt_drv_cycle_count_start(struct oem_battery_drv_chip *chip)
{
	int ret = 0;
	int battery_capacity = 0;
	int age = 100;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	if (drv_initialized == 0) {
		goto batt_exit;
	}

	if (chip->start_soc_value < BATT_SOC_INIT) {
		goto batt_exit;
	}

	ret = batt_drv_get_soc(chip, &battery_capacity);
	if (unlikely(ret < 0)) {
		goto batt_exit;
	}

	chip->incremental_soc = 0;
	chip->charge_count_val = 0;
	chip->hi_temp_count_val = 0;
	chip->hi_capacity_count_val = 0;
	chip->longevity_count_val = 0;
	chip->current_soc = battery_capacity;
	chip->start_soc_value = battery_capacity;
	age = batt_drv_prop_age(chip);
batt_exit:
	BATT_DRV_RECLOG("[chg_cycle] Charging, age=%d, charge_count=%d, hi_capacity_count=%d, hi_temp_count=%d\n",
					  age, chip->chg_data.charge_count, chip->chg_data.hi_capacity_count, chip->chg_data.hi_temp_count);
	BATT_DRV_INFOLOG("[chg_cycle] Charging, longevity_count=%d\n",
					  chip->longevity_cycle);

	return;

}

static void batt_drv_cycle_count_update(struct oem_battery_drv_chip *chip)
{
	int ret = 0;
	int count = 0;
	int batt_temp = 0;
	struct charging_data chg_data;
	uint32_t longevity_cycle_data;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	if (chip->current_soc == BATT_SOC_INIT) {
		goto batt_exit;
	}

	if (chip->current_soc == chip->soc) {
		goto batt_exit;
	}

	batt_temp = max77729_get_fuelgauge_value(chip->fuelgauge, FG_TEMPERATURE);
	if (batt_temp < 0) {
		/* Failure */
		BATT_DRV_ERRLOG("[%s] Get temp FAILURE %d\n", __func__, batt_temp);
		batt_temp = 0;
	}

	if (chip->current_soc < chip->soc) {
		count = (chip->soc - chip->current_soc);

		if (batt_drv_debug_cycle > 0) {
			count += batt_drv_debug_cycle;
			batt_drv_debug_cycle = 0;
		}

		chip->incremental_soc += count;
		chip->charge_count_val += count;
		if (chip->longevity_charge_flag)
			chip->longevity_count_val += count;

		if (batt_temp >= chip->dt_param.cycle_count_hi_temp_threshold) {
			chip->hi_temp_count_val += count;
		}

		if (chip->soc >= chip->dt_param.cycle_count_hi_soc_threshold) {
			chip->hi_capacity_count_val += count;
		}

		if (chip->incremental_soc >= BATT_CYCLE_COUNT_BACKUP_THRESHOLD) {
			memset(&chg_data, 0, sizeof(struct charging_data));

			if (chip->longevity_charge_flag) {
				chg_data.charge_count = chip->chg_data.charge_count + BATT_LONGEVITY_CYCLE_CORRECTION(chip->charge_count_val);
				chg_data.hi_capacity_count = chip->chg_data.hi_capacity_count + BATT_LONGEVITY_CYCLE_CORRECTION(chip->hi_capacity_count_val);
				chg_data.hi_temp_count = chip->chg_data.hi_temp_count + BATT_LONGEVITY_CYCLE_CORRECTION(chip->hi_temp_count_val);
				longevity_cycle_data = chip->longevity_cycle + BATT_LONGEVITY_CYCLE_CORRECTION(chip->longevity_count_val);
			} else {
				chg_data.charge_count = chip->chg_data.charge_count + chip->charge_count_val;
				chg_data.hi_capacity_count = chip->chg_data.hi_capacity_count + chip->hi_capacity_count_val;
				chg_data.hi_temp_count = chip->chg_data.hi_temp_count + chip->hi_temp_count_val;
				longevity_cycle_data = chip->longevity_cycle + chip->longevity_count_val;
			}

			ret = set_nonvolatile((uint8_t*)&chg_data, APNV_CHARGING_DATA, sizeof(struct charging_data));
			if (unlikely(ret < 0)) {
				BATT_DRV_ERRLOG("[%s] set nonvolatile %d error\n", __func__, APNV_CHARGING_DATA);
			}

			ret = set_nonvolatile((uint8_t*)&longevity_cycle_data, APNV_CHARGE_LONGEVITY_CYCLE, 4);
			if (unlikely(ret < 0)) {
				BATT_DRV_ERRLOG("[%s] set nonvolatile %d error\n", __func__, APNV_CHARGE_LONGEVITY_CYCLE);
			}
			chip->incremental_soc = 0;
		}
	}
	chip->current_soc = chip->soc;

batt_exit:
	return;
}

static void batt_drv_cycle_count_finish(struct oem_battery_drv_chip *chip)
{
	int ret = 0;
	struct charging_data chg_data;
	uint32_t longevity_cycle_data  = 0;
	int age = 100;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	if (drv_initialized == 0) {
		goto batt_exit;
	}

	if (chip->longevity_charge_flag) {
		BATT_DRV_INFOLOG("[%s] longevity_charge correction charge_count_val:%d -> %d hi_capacity_count_val:%d -> %d "
					"hi_temp_count_val:%d -> %d longevity_count_val:%d -> %d\n",
					__func__,
					chip->charge_count_val, BATT_LONGEVITY_CYCLE_CORRECTION(chip->charge_count_val),
					chip->hi_capacity_count_val, BATT_LONGEVITY_CYCLE_CORRECTION(chip->hi_capacity_count_val),
					chip->hi_temp_count_val, BATT_LONGEVITY_CYCLE_CORRECTION(chip->hi_temp_count_val),
					chip->longevity_count_val, BATT_LONGEVITY_CYCLE_CORRECTION(chip->longevity_count_val));
		chip->charge_count_val = BATT_LONGEVITY_CYCLE_CORRECTION(chip->charge_count_val);
		chip->hi_capacity_count_val = BATT_LONGEVITY_CYCLE_CORRECTION(chip->hi_capacity_count_val);
		chip->hi_temp_count_val = BATT_LONGEVITY_CYCLE_CORRECTION(chip->hi_temp_count_val);
		chip->longevity_count_val = BATT_LONGEVITY_CYCLE_CORRECTION(chip->longevity_count_val);
	}

	chip->chg_data.charge_count += chip->charge_count_val;
	chip->chg_data.hi_capacity_count += chip->hi_capacity_count_val;
	chip->chg_data.hi_temp_count += chip->hi_temp_count_val;
	chip->longevity_cycle += chip->longevity_count_val;

	chip->charge_count_val = 0;
	chip->hi_capacity_count_val = 0;
	chip->hi_temp_count_val = 0;
	chip->longevity_count_val = 0;

	/* save the charging data to nv */
	memset(&chg_data, 0, sizeof(struct charging_data));
	chg_data.charge_count = chip->chg_data.charge_count;
	chg_data.hi_capacity_count = chip->chg_data.hi_capacity_count;
	chg_data.hi_temp_count = chip->chg_data.hi_temp_count;

	longevity_cycle_data = chip->longevity_cycle;

	ret = set_nonvolatile((uint8_t*)&chg_data, APNV_CHARGING_DATA, sizeof(struct charging_data));
	if (unlikely(ret < 0)) {
		BATT_DRV_ERRLOG("[%s] set nonvolatile %d error\n", __func__, APNV_CHARGING_DATA);
	}

	ret = set_nonvolatile((uint8_t*)&longevity_cycle_data, APNV_CHARGE_LONGEVITY_CYCLE, 4);
	if (unlikely(ret < 0)) {
		BATT_DRV_ERRLOG("[%s] set nonvolatile %d error\n", __func__, APNV_CHARGE_LONGEVITY_CYCLE);
	}

	age = batt_drv_prop_age(chip);

batt_exit:
	BATT_DRV_RECLOG("[chg_cycle] Discharge, age=%d, charge_count=%d, hi_capacity_count=%d, hi_temp_count=%d\n",
					  age, chip->chg_data.charge_count, chip->chg_data.hi_capacity_count, chip->chg_data.hi_temp_count);
	BATT_DRV_INFOLOG("[chg_cycle] Discharge, longevity_count=%d\n",
					  chip->longevity_count_val);

	chip->start_soc_value = BATT_SOC_INIT;
	chip->current_soc = BATT_SOC_INIT;
	return;
}

static int batt_drv_set_status(struct oem_battery_drv_chip *chip, int status, int battery_capacity)
{
	int ret = 0;
	bool full_led_flag = false;
	unsigned long now = jiffies;
	int old_status;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	mutex_lock(&chip->lock);

	old_status = chip->status;

	if (chip->status != status) {
		if (status == POWER_SUPPLY_STATUS_CHARGING) {
			batt_drv_cycle_count_start(chip);
			if (chip->keep_capacity_flag == BATT_DRV_KEEP_CAPACITY_STANDBY) {
				chip->keep_capacity_time = jiffies;
				chip->keep_capacity_flag = BATT_DRV_KEEP_CAPACITY_ENABLE;
			}
			chip->soc_mask_flag = 0;
		} else {
			if (chip->keep_capacity_flag == BATT_DRV_KEEP_CAPACITY_ENABLE) {
				chip->keep_capacity_flag = BATT_DRV_KEEP_CAPACITY_DISABLE;
			}
		}

		if (status == POWER_SUPPLY_STATUS_DISCHARGING) {
			batt_drv_cycle_count_finish(chip);

			if (chip->soc >= BATT_SOC_FULL) {
				/* 30 sec */
				chip->soc_mask_expire = now + BATT_DRV_TIMER_30S;
				chip->soc_mask_flag = 1;
				BATT_DRV_DBGLOG("[%s] now:%lu, expire:%lu\n", __func__, now, chip->soc_mask_expire);
			}

			chip->soc_charging_mask_flag = 0;
		}

		if ((drv_initialized == 1) &&
			(status == POWER_SUPPLY_STATUS_CHARGING)) {
			if (chip->status == POWER_SUPPLY_STATUS_FULL) {
				/* Do not update state */
				goto batt_update_skip;
			} else {
				if (chip->longevity_charge_flag) {
					if ((battery_capacity >= chip->dt_param.longevity_full_soc_min_threshold) ||
						(chip->soc >= chip->dt_param.longevity_full_soc_min_threshold)){
						full_led_flag = true;
					}
				} else {
					if ((battery_capacity >= BATT_SOC_FULL) ||
						(chip->soc >= BATT_SOC_FULL)) {
						full_led_flag = true;
					}
				}
				if (full_led_flag) {
					/* 10 sec */
					chip->soc_charging_mask_expire = now + BATT_DRV_TIMER_10S;
					chip->soc_charging_mask_flag = 1;
					chip->status = POWER_SUPPLY_STATUS_CHARGING;
					goto batt_update_skip;
				}
			}
		}

		chip->status = status;

batt_update_skip:
		BATT_DRV_INFOLOG("[%s] old_status:%s new_status:%s set_status:%s capa:%d volt:%dmV\n",
					__func__,
					batt_drv_status_str[old_status],
					batt_drv_status_str[chip->status],
					batt_drv_status_str[status],
					chip->soc,
					chip->vcell);

		power_supply_changed(chip->oem_battery_psy);
	}

	mutex_unlock(&chip->lock);

	return ret;
}

static int batt_drv_set_health(struct oem_battery_drv_chip *chip, int health)
{
	int ret = 0;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	mutex_lock(&chip->lock);

	if (chip->health != health) {
		BATT_DRV_INFOLOG("[%s] old_health:%s new_status:%s capa:%d volt:%dmV\n", __func__,
								batt_drv_health_str[chip->health],
								batt_drv_health_str[health],
								chip->soc,
								chip->vcell);

		chip->health = health;
		ret = 1;

		power_supply_changed(chip->oem_battery_psy);
	}

	mutex_unlock(&chip->lock);

	return ret;
}

static int batt_drv_get_voltage(struct oem_battery_drv_chip *chip, int *battery_voltage)
{
	int ret = 0;
	int fg_vcell = 0;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	if (unlikely(battery_voltage == NULL)) {
		BATT_DRV_ERRLOG("[%s] battery_voltage pointer is NULL\n", __func__);
		ret = -1;
		goto batt_exit;
	}

	fg_vcell = max77729_get_fuelgauge_value(chip->fuelgauge, FG_VOLTAGE);
	if (fg_vcell < 0) {
		BATT_DRV_ERRLOG("[%s] get battery voltage FAILURE %d\n", __func__, fg_vcell);
		*battery_voltage = 0;
		ret = -1;
		goto batt_exit;
	} else {
		*battery_voltage = fg_vcell;
	}

batt_exit:
	return ret;
}

static int batt_drv_get_temperature(struct oem_battery_drv_chip *chip, int *battery_temperature)
{
	int ret = 0;
	int temp = 0;
	int i;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	if (unlikely(battery_temperature == NULL)) {
		BATT_DRV_ERRLOG("[%s] battery_temperature pointer is NULL\n", __func__);
		ret =  -1;
		goto batt_exit;
	}
	temp = max77729_get_fuelgauge_value(chip->fuelgauge, FG_TEMPERATURE);
	if (temp < 0) {
		BATT_DRV_ERRLOG("[%s] Get temp FAILURE %d\n", __func__, temp);
		*battery_temperature = 0;
		ret = -1;
		goto batt_exit;
	} else {
		for (i = 0; i < chip->dt_param.temp_correction_num; i++) {
			if (temp < chip->dt_param.temp_correction_thresholds[i]) {
				break;
			}
		}
		*battery_temperature = temp - chip->dt_param.temp_correction_values[i];
	}

batt_exit:
	return ret;
}

static int batt_drv_prop_status(struct oem_battery_drv_chip *chip)
{
	if (fake_status != BATT_FAKE_STATUS_INIT) {
		BATT_DRV_INFOLOG("[%s] in fake_status:%s\n", __func__, batt_drv_status_str[fake_status]);
		return fake_status;
	}
	BATT_DRV_DBGLOG("[%s] in status:%s\n", __func__, batt_drv_status_str[chip->status]);
	return chip->status;
}

static int batt_drv_prop_health(struct oem_battery_drv_chip *chip)
{
	if (fake_health != BATT_FAKE_HEALTH_INIT) {
		BATT_DRV_INFOLOG("[%s] in fake_health:%s\n", __func__, batt_drv_health_str[fake_health]);
		return fake_health;
	}
	BATT_DRV_DBGLOG("[%s] in health:%s\n", __func__, batt_drv_health_str[chip->health]);
	return chip->health;
}

static int batt_drv_prop_capacity(struct oem_battery_drv_chip *chip)
{
	int ret;
	int work_capacity;

	if (chip->check_disable[BATT_DRV_FUNC_LIMIT_LOW_BATTERY] == true) {
		ret = 90;
	} else if ((chip->fake_soc >= 0) && (chip->fake_soc <= 100)) {
		ret = chip->fake_soc;
	} else {
		batt_drv_get_soc(chip, &work_capacity);
		batt_drv_set_soc(chip, work_capacity);
		batt_drv_cycle_count_update(chip);
		if (chip->soc > BATT_SOC_FULL) {
			ret = BATT_SOC_FULL;
		} else {
			ret = chip->soc;
		}
	}

	if (chip->keep_capacity_flag != BATT_DRV_KEEP_CAPACITY_DISABLE) {
		if (ret == 0) {
			ret = 1;
		}
	}

	return ret;
}

static int batt_drv_prop_voltage_now(struct oem_battery_drv_chip *chip)
{
	int ret;
	int work_voltage;

	if (chip->check_disable[BATT_DRV_FUNC_LIMIT_LOW_BATTERY] == true) {
		ret = 3900;
	} else {
		batt_drv_get_voltage(chip, &work_voltage);
		ret = work_voltage;
	}

	return ret;
}

static int batt_drv_prop_temp(struct oem_battery_drv_chip *chip)
{
	int ret;
	int work_temp;

	if (chip->check_disable[BATT_DRV_FUNC_LIMIT_BATTERY_TEMP] == true) {
		ret = 250;
	} else if (drv_initialized == 0) {
		ret = 250;
	} else {
		batt_drv_get_temperature(chip, &work_temp);
		ret = work_temp;
	}

	return ret;
}

static int batt_drv_get_longevity_chg_mode(struct oem_battery_drv_chip *chip)
{
	int ret = 0;
	u8 val = 0;

	ret = get_nonvolatile((u8 *)&val, APNV_ITEM_LONGEVITYCHARGE, 1);
	if (unlikely(ret < 0)) {
		BATT_DRV_ERRLOG("[%s] get_nonvolatile failed. ret = %d\n", __func__, ret);
		chip->longevity_charge_flag = false;
	} else {
		if (val == 0x01) {
			chip->longevity_charge_flag = true;
		} else {
			chip->longevity_charge_flag = false;
		}
		ret = (int)chip->longevity_charge_flag;
	}

	return ret;
}

static int batt_drv_get_chg_ctrl_soc(struct oem_battery_drv_chip *chip)
{
	int work_capacity;

	batt_drv_get_soc(chip, &work_capacity);

	return work_capacity;
}

static int batt_drv_get_property(struct power_supply *psy,
									   enum power_supply_property psp,
									   union power_supply_propval *val)
{
	struct oem_battery_drv_chip	*chip = power_supply_get_drvdata(psy);
	enum power_supply_ext_property ext_psp = (enum power_supply_ext_property)psp;

	if (unlikely(psy == NULL)) {
		BATT_DRV_ERRLOG("[%s] psy pointer is NULL\n", __func__);
		return -EINVAL;
	}

	BATT_DRV_DBGLOG("[%s] in psp:%d\n", __func__, psp);

	switch ((int)psp) {
	case POWER_SUPPLY_PROP_STATUS:
		val->intval = batt_drv_prop_status(chip);
		break;

	case POWER_SUPPLY_PROP_HEALTH:
		val->intval = batt_drv_prop_health(chip);
		break;

	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = drv_initialized;
		break;

	case POWER_SUPPLY_PROP_CAPACITY:
		val->intval = batt_drv_prop_capacity(chip);
		break;

	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		val->intval = batt_drv_prop_voltage_now(chip);
		break;

	case POWER_SUPPLY_PROP_TEMP:
		val->intval = batt_drv_prop_temp(chip);
		break;

	case POWER_SUPPLY_PROP_CYCLE_COUNT:
		val->intval = batt_drv_prop_age(chip);
		break;

	case POWER_SUPPLY_EXT_PROP_MIN ... POWER_SUPPLY_EXT_PROP_MAX:
		switch (ext_psp) {
		case POWER_SUPPLY_EXT_PROP_LONGEVITY_CHG_MODE:
			val->intval = batt_drv_get_longevity_chg_mode(chip);
			break;

		case POWER_SUPPLY_EXT_PROP_CYCLE_CHG:
			val->intval = batt_drv_proc_cycle_chg(chip);
			break;

		case POWER_SUPPLY_EXT_PROP_CHG_CTRL_SOC:
			val->intval = batt_drv_get_chg_ctrl_soc(chip);
			break;

		default:
			return -EINVAL;
			break;
		}
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

static void batt_drv_set_prop_status(struct oem_battery_drv_chip *chip, int status)
{

	int work_capacity;

	batt_drv_get_soc(chip, &work_capacity);
	batt_drv_set_status(chip, status, work_capacity);

	return;
}

static int batt_drv_set_property(struct power_supply *psy,
									   enum power_supply_property psp,
									   const union power_supply_propval *val)
{
	struct oem_battery_drv_chip	*chip = power_supply_get_drvdata(psy);
	enum power_supply_ext_property ext_psp = (enum power_supply_ext_property)psp;

	if (unlikely(psy == NULL)) {
		BATT_DRV_ERRLOG("[%s] psy pointer is NULL\n", __func__);
		return -EINVAL;
	}

	BATT_DRV_DBGLOG("[%s] in psp:%d\n", __func__, psp);

	switch ((int)psp) {
	case POWER_SUPPLY_PROP_STATUS:
		batt_drv_set_prop_status(chip, val->intval);
		break;

	case POWER_SUPPLY_PROP_HEALTH:
		batt_drv_set_health(chip, val->intval);
		break;

	case POWER_SUPPLY_PROP_CAPACITY:
		chip->fake_soc = val->intval;
		break;

	case POWER_SUPPLY_EXT_PROP_MIN ... POWER_SUPPLY_EXT_PROP_MAX:
		switch (ext_psp) {
		case POWER_SUPPLY_EXT_PROP_LONGEVITY_CYCLE:
			if (val->intval == 0)
				batt_drv_longevity_count_clear(chip);
			break;

		default:
			return -EINVAL;
		}
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

static int batt_drv_property_is_writeable(struct power_supply *psy,
												enum power_supply_property psp)
{
	int ret = 0;

	if (unlikely(psy == NULL)) {
		BATT_DRV_ERRLOG("[%s] psy pointer is NULL\n", __func__);
		return -EINVAL;
	}

	BATT_DRV_DBGLOG("[%s] in psp:%d\n", __func__, psp);

	switch (psp) {
	case POWER_SUPPLY_PROP_CAPACITY:
		ret = 1;
		break;

	default:
		ret = 0;
		break;
	}

	return ret;
}

static void batt_drv_external_power_changed(struct power_supply *psy)
{
	if (unlikely(psy == NULL)) {
		BATT_DRV_ERRLOG("[%s] psy pointer is NULL\n", __func__);
		return;
	}
	return;
}

static void batt_drv_event_monitor(struct work_struct *work)
{
	struct oem_battery_drv_chip *chip = container_of(work,
					struct oem_battery_drv_chip, battery_monitor.work);

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	set_bit(BATT_DRV_EVENT_MONITOR, &chip->event_state);
	wake_up(&chip->wq);
}

static void batt_drv_work_monitor(struct oem_battery_drv_chip *chip)
{
	int ret;
	int work_volt;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	ret = batt_drv_get_voltage(chip, &work_volt);
	if (unlikely(ret < 0)) {
		queue_delayed_work(chip->batt_drv_wq, &chip->battery_monitor, BATT_DRV_MONITOR_INTERVAL);
		BATT_DRV_ERRLOG("[%s] Error get_voltage ret = %d\n", __func__, ret);
		goto batt_exit;
	}

	chip->vcell = work_volt;

	if (chip->vcell == 0) {
		/* fuel gauge driver initializing... */
		chip->shutdown_voltage_count = 0;
		queue_delayed_work(chip->batt_drv_wq, &chip->battery_monitor, BATT_DRV_LOW_VOLTAGE_INTERVAL);
	} else if (chip->vcell <= chip->dt_param.shutdown_voltage_threshold) {
		chip->shutdown_voltage_count++;
		if (chip->shutdown_voltage_count >= chip->dt_param.shutdown_count) {
			chip->lowbatt_flag = true;
			BATT_DRV_RECLOG("[%s] low battery detect volt:%dmV\n", __func__, chip->vcell);
		}
		queue_delayed_work(chip->batt_drv_wq, &chip->battery_monitor, BATT_DRV_SHUTDOWN_VOLTAGE_INTERVAL);
		chip->keep_capacity_flag = BATT_DRV_KEEP_CAPACITY_DISABLE;
	} else if (chip->vcell <= chip->dt_param.low_voltage_threshold) {
		chip->shutdown_voltage_count = 0;
		queue_delayed_work(chip->batt_drv_wq, &chip->battery_monitor, BATT_DRV_LOW_VOLTAGE_INTERVAL);
	} else {
		chip->shutdown_voltage_count = 0;
		queue_delayed_work(chip->batt_drv_wq, &chip->battery_monitor, BATT_DRV_MONITOR_INTERVAL);
	}

	if (chip->keep_capacity_flag == BATT_DRV_KEEP_CAPACITY_STANDBY) {
		if (time_before((chip->keep_capacity_time + BATT_DRV_TIMER_30S), jiffies)) {
			chip->keep_capacity_flag = BATT_DRV_KEEP_CAPACITY_DISABLE;
		}
	} else if (chip->keep_capacity_flag == BATT_DRV_KEEP_CAPACITY_ENABLE) {
		if (time_before((chip->keep_capacity_time + BATT_DRV_TIMER_180S), jiffies)) {
			chip->keep_capacity_flag = BATT_DRV_KEEP_CAPACITY_DISABLE;
		}
	}

	power_supply_changed(chip->oem_battery_psy);

batt_exit:
	return;
}

static int batt_drv_thread(void * ptr)
{
	struct oem_battery_drv_chip	*chip = ptr;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	if (unlikely(chip == NULL)) {
		BATT_DRV_WARNLOG("[%s] chip pointer is NULL\n", __func__);
		if (unlikely(oem_batt_drv_chip == NULL)) {
			BATT_DRV_ERRLOG("[%s] oem_batt_drv_chip pointer is NULL\n", __func__);
			return -EINVAL;
		}
		chip = oem_batt_drv_chip;
	}

	while (1) {
		batt_drv_get_soc(chip, &chip->soc);
		if (chip->soc <= BATT_SOC_FULL) {

			if ((chip->status == POWER_SUPPLY_STATUS_CHARGING) &&
				(chip->soc >= chip->dt_param.full_keep_threshold)) {
				chip->soc = BATT_SOC_FULL;
				chip->status = POWER_SUPPLY_STATUS_FULL;
			}
			break;
		}
		msleep(100);
	}

	chip->keep_capacity_time = jiffies;
	if (chip->status == POWER_SUPPLY_STATUS_CHARGING) {
		chip->keep_capacity_flag = BATT_DRV_KEEP_CAPACITY_ENABLE;
	} else {
		chip->keep_capacity_flag = BATT_DRV_KEEP_CAPACITY_STANDBY;
	}

	if (battery_status != POWER_SUPPLY_STATUS_DISCHARGING) {
		BATT_DRV_INFOLOG("[%s] old_status:%s new_status:%s\n",
						__func__,
						batt_drv_status_str[chip->status],
						batt_drv_status_str[battery_status]);
		chip->status = battery_status;
	}
	chip->health = POWER_SUPPLY_HEALTH_GOOD;

	batt_drv_function_limits_check(chip);
	batt_drv_get_charge_cycle(chip);
	batt_drv_set_misc(chip);

	drv_initialized = 1;
	if (battery_status == POWER_SUPPLY_STATUS_CHARGING) {
		batt_drv_cycle_count_start(chip);
	}

	set_bit(BATT_DRV_EVENT_MONITOR, &chip->event_state);
	set_freezable();

	do {
		wait_event_freezable(chip->wq, (chip->event_state || kthread_should_stop()));

		if (kthread_should_stop()) {
			BATT_DRV_INFOLOG("[%s] while loop break\n", __func__);
			break;
		}

		if (test_bit(BATT_DRV_EVENT_SUSPEND, &chip->event_state)) {
			test_and_clear_bit(BATT_DRV_EVENT_MONITOR, &chip->event_state);
		}

		if (test_and_clear_bit(BATT_DRV_EVENT_MONITOR, &chip->event_state)) {
			batt_drv_work_monitor(chip);
		}

	} while (1);

	return 0;
}

static int batt_drv_parse_dt(struct oem_battery_drv_chip *chip)
{
	struct batt_drv_dt *dt = &chip->dt_param;
	struct device_node *node = of_find_node_by_name(NULL, "max77729-fuelgauge");
	int ret;

	if (!node) {
		BATT_DRV_ERRLOG("[%s] device tree node missing\n", __func__);
		return -EINVAL;
	} else {
		chip->of_node = node;
	}

	ret = of_property_read_u32(node, "fcnt,batt-recharge-soc",
				&dt->recharge_soc_threshold);
	if (ret < 0)
		dt->recharge_soc_threshold = BATT_RECHARGE_SOC_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-full-keep-soc",
				&dt->full_keep_threshold);
	if (ret < 0)
		dt->full_keep_threshold = BATT_FULL_KEEP_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-low-voltage-mv",
				&dt->low_voltage_threshold);
	if (ret < 0)
		dt->low_voltage_threshold = BATT_LOW_VOLTAGE_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-shutdown-voltage-mv",
				&dt->shutdown_voltage_threshold);
	if (ret < 0)
		dt->shutdown_voltage_threshold = BATT_SHUTDOWN_VOLTAGE_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-shutdown-count",
				&dt->shutdown_count);
	if (ret < 0)
		dt->shutdown_count = BATT_SHUTDOWN_COUNT_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-longevity-full-soc-min",
				&dt->longevity_full_soc_min_threshold);
	if (ret < 0)
		dt->longevity_full_soc_min_threshold = BATT_LONGEVITY_FULL_SOC_MIN_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-longevity-full-soc-max",
				&dt->longevity_full_soc_max_threshold);
	if (ret < 0)
		dt->longevity_full_soc_max_threshold = BATT_LONGEVITY_FULL_SOC_MAX_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-longevity-recharge-soc",
				&dt->longevity_recharge_soc_threshold);
	if (ret < 0)
		dt->longevity_recharge_soc_threshold = BATT_LONGEVITY_RECHARGE_SOC_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-cycle-count-hi-soc",
				&dt->cycle_count_hi_soc_threshold);
	if (ret < 0)
		dt->cycle_count_hi_soc_threshold = BATT_CYCLE_COUNT_HI_SOC_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-cycle-count-hi-temp",
				&dt->cycle_count_hi_temp_threshold);
	if (ret < 0)
		dt->cycle_count_hi_temp_threshold = BATT_CYCLE_COUNT_HI_TEMP_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-cycle-charge-count-lv1",
				&dt->cycle_charge_threshold_lv1);
	if (ret < 0)
		dt->cycle_charge_threshold_lv1 = BATT_CYCLE_CHARGE_DEFAULT;

	ret = of_property_read_u32(node, "fcnt,batt-cycle-charge-count-lv2",
				&dt->cycle_charge_threshold_lv2);
	if (ret < 0)
		dt->cycle_charge_threshold_lv2 = BATT_CYCLE_CHARGE_DEFAULT;

	if ((of_property_count_u32_elems(node, "fcnt,batt-temp-correction-values") ==
		 of_property_count_u32_elems(node, "fcnt,batt-temp-correction-thresholds")) &&
		(of_property_count_u32_elems(node, "fcnt,batt-temp-correction-values") <=
													 BATT_ADJUST_TEMP_TBL_MAX) &&
		(of_property_count_u32_elems(node, "fcnt,batt-temp-correction-thresholds") <=
													 BATT_ADJUST_TEMP_TBL_MAX)) {
		dt->temp_correction_num = of_property_count_u32_elems(node,
										"fcnt,batt-temp-correction-values");

		ret = of_property_read_u32_array(node, "fcnt,batt-temp-correction-values",
				dt->temp_correction_values, dt->temp_correction_num);
		if (ret < 0) {
			dt->temp_correction_num = 0;
		}

		ret = of_property_read_u32_array(node, "fcnt,batt-temp-correction-thresholds",
				dt->temp_correction_thresholds, dt->temp_correction_num);
		if (ret < 0) {
			dt->temp_correction_num = 0;
		}
	} else {
		dt->temp_correction_num = 0;
	}

	ret = of_property_read_u32(node, "fcnt,batt-design-cap",
				&dt->design_cap);
	if (ret < 0)
		dt->design_cap = BATT_LONGEVITY_DESIGN_CAP_DEFAULT;

	return 0;
}

int oem_batt_charge_full(void)
{
	struct oem_battery_drv_chip	*chip = oem_batt_drv_chip;
	int age;
	int fcc;

	if (drv_initialized == 0) {
		return BATT_LONGEVITY_DESIGN_CAP_DEFAULT * 1000;
	}

	age = batt_drv_prop_age(chip);
	fcc = chip->dt_param.design_cap * age / 100;
	fcc *= 1000;

	return fcc;
}

int oem_batt_cycle_count(void)
{
	struct oem_battery_drv_chip	*chip = oem_batt_drv_chip;
	uint32_t cycle_count;

	if (drv_initialized == 0) {
		return 0;
	}

	if (chip->longevity_charge_flag)
		cycle_count = chip->chg_data.charge_count + BATT_LONGEVITY_CYCLE_CORRECTION(chip->charge_count_val);
	else
		cycle_count = chip->chg_data.charge_count + chip->charge_count_val;

	return cycle_count / 100;
}

static void oem_batt_classdev_release(struct device *dev)
{
	return;
}

static int oem_batt_prepare(struct device *dev)
{
	return 0;
}

static const struct dev_pm_ops oem_batt_ops = {
	.prepare = oem_batt_prepare,
};

static struct class oem_batt_class = {
	.name		= "oem_batt",
	.dev_release	= oem_batt_classdev_release,
	.pm		= &oem_batt_ops,
};

static ssize_t oem_batt_cycle_clear_show(struct device *dev,
		struct device_attribute *attr, char *data)
{
	struct oem_battery_drv_chip *chip = dev_get_drvdata(dev);
	int ret = 0;
	int result_value = 0;
	char *cmdline = (char *)smem_alloc_vendor0(SMEM_OEM_V0_022);
	char *boot_mode;
	struct charging_data chg_data;
	uint32_t longevity_cycle_data = 0;

	if (unlikely(cmdline == NULL)) {
		BATT_DRV_ERRLOG("[%s] smem_alloc_vendor0(SMEM_OEM_V0_022) error\n",__func__);
		return -ENOMEM;
	}
	if (unlikely(chip == NULL)) {
		BATT_DRV_WARNLOG("[%s] chip pointer is NULL\n", __func__);
		return -ENODEV;
	}

	boot_mode = strstr(cmdline, "androidboot.mode=");
	if (boot_mode != NULL) {
		boot_mode += strlen("androidboot.mode=");

		if (strncmp(boot_mode, "repair", strlen("repair")) == 0) {
			BATT_DRV_INFOLOG("[%s] Clear cycle count\n", __func__);

			memset(&chg_data, 0, sizeof(struct charging_data));
			chg_data.charge_count = 0;
			chip->chg_data.charge_count = 0;
			chip->chg_data.hi_capacity_count = 0;
			chip->chg_data.hi_temp_count = 0;
			ret = set_nonvolatile((uint8_t*)&chg_data, APNV_CHARGING_DATA, sizeof(struct charging_data));
			if (unlikely(ret < 0)) {
				BATT_DRV_ERRLOG("[%s] set nonvolatile %d error\n", __func__, APNV_CHARGING_DATA);
			}
			chip->longevity_count_val = 0;
			chip->longevity_cycle = 0;
			ret = set_nonvolatile((uint8_t*)&longevity_cycle_data, APNV_CHARGE_LONGEVITY_CYCLE, 4);
			if (unlikely(ret < 0)) {
				BATT_DRV_ERRLOG("[%s] set nonvolatile %d error\n", __func__, APNV_CHARGE_LONGEVITY_CYCLE);
			}

			result_value = 1;
		}
	}

	return sprintf(data, "%d\n", result_value);
}

static DEVICE_ATTR(cycle_clear, S_IRUGO, oem_batt_cycle_clear_show, NULL);

int oem_batt_drv_init(struct max77729_fuelgauge_data *fuelgauge)
{
	struct oem_battery_drv_chip	*chip;
	struct power_supply_config battery_drv_cfg = {};
	int ret = 0;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	chip = devm_kzalloc(fuelgauge->dev, sizeof(*chip), GFP_KERNEL);
	if (!chip) {
		return -ENOMEM;
	}

	chip->dev = fuelgauge->dev;
	chip->fuelgauge = fuelgauge;

	ret = class_register(&oem_batt_class);
	if (ret) {
		BATT_DRV_ERRLOG("%s: class register failed: %d\n", __func__, ret);
		return ret;
	}

	device_initialize(&chip->class_dev);
	chip->class_dev.parent = chip->dev;
	chip->class_dev.class = &oem_batt_class;
	dev_set_drvdata(&chip->class_dev, chip);

	ret = dev_set_name(&chip->class_dev, "%s", "oem_batt");
	if (ret) {
		BATT_DRV_ERRLOG("Failed to set name %d\n", ret);
		goto oem_batt_drv_err_classdev_put;
	}

	ret = device_add(&chip->class_dev);
	if (ret) {
		BATT_DRV_ERRLOG("Failed to device add %d\n", ret);
		goto oem_batt_drv_err_classdev_put;
	}

	ret = device_create_file(&chip->class_dev, &dev_attr_cycle_clear);
	if (ret) {
		BATT_DRV_ERRLOG("Failed to create setting schedule_chg entry %d\n", ret);
		goto oem_batt_drv_err_device_del;
	}

	chip->create_class = true;

	/* init wait queue */
	init_waitqueue_head(&chip->wq);

	ret = batt_drv_parse_dt(chip);
	if (unlikely(ret != 0)) {
		BATT_DRV_ERRLOG("[%s] dt perse failed ret = %d\n", __func__, ret);
		goto oem_batt_drv_err_cycle_clear_del;
	}

	mutex_init(&chip->lock);

	chip->health = POWER_SUPPLY_HEALTH_GOOD;
	chip->fake_soc = -EINVAL;
	chip->event_state = 0;
	chip->soc = BATT_SOC_INIT;
	chip->old_msoc = BATT_MSOC_INIT;
	chip->vcell = 0;
	chip->start_soc_value = BATT_SOC_INIT;
	chip->current_soc = BATT_SOC_INIT;
	chip->incremental_soc = 0;
	chip->cycle_chg_lvl = CHG_CYCLE_LEVEL_NUM;
	chip->soc_mask_flag = 0;
	chip->soc_mask_expire = 0;
	chip->soc_charging_mask_flag = 0;
	chip->soc_charging_mask_expire = 0;
	chip->status = POWER_SUPPLY_STATUS_DISCHARGING;
	chip->shutdown_voltage_count = 0;
	chip->keep_capacity_time = 0;
	chip->keep_capacity_flag = BATT_DRV_KEEP_CAPACITY_DISABLE;
	chip->longevity_charge_flag = false;
	chip->lowbatt_flag = false;

	chip->oem_battery_psy_desc.name 				  = OEM_BATTERY_PSY_NAME;
	chip->oem_battery_psy_desc.type 				  = POWER_SUPPLY_TYPE_MAINS;
	chip->oem_battery_psy_desc.properties			  = batt_drv_properties;
	chip->oem_battery_psy_desc.num_properties		  = ARRAY_SIZE(batt_drv_properties);
	chip->oem_battery_psy_desc.get_property 		  = batt_drv_get_property;
	chip->oem_battery_psy_desc.set_property 		  = batt_drv_set_property;
	chip->oem_battery_psy_desc.property_is_writeable  = batt_drv_property_is_writeable;
	chip->oem_battery_psy_desc.external_power_changed = batt_drv_external_power_changed;

	battery_drv_cfg.drv_data = chip;
	battery_drv_cfg.of_node = chip->of_node;
	chip->oem_battery_psy = devm_power_supply_register(chip->dev, &chip->oem_battery_psy_desc,
													  &battery_drv_cfg);
	if (IS_ERR(chip->oem_battery_psy)) {
		ret = PTR_ERR(chip->oem_battery_psy);
		BATT_DRV_ERRLOG("[%s] power_supply_register %s failed ret = %d\n", __func__, OEM_BATTERY_PSY_NAME, ret);
		goto oem_batt_drv_err_cycle_clear_del;
	}

	INIT_DELAYED_WORK(&chip->battery_monitor, batt_drv_event_monitor);
	chip->batt_drv_wq = create_singlethread_workqueue(OEM_BATTERY_PSY_NAME);
	if (!chip->batt_drv_wq) {
		BATT_DRV_ERRLOG("[%s] create_singlethread_workqueue failed\n", __func__);
		ret = -ESRCH;
		goto oem_batt_drv_err_cycle_clear_del;
	}

	oem_batt_drv_chip = chip;

	/* kthread start */
	chip->thread = kthread_run(batt_drv_thread, chip, "oem_batt_drv");

	BATT_DRV_INFOLOG("[%s] probe End\n", __func__);

	return 0;

oem_batt_drv_err_cycle_clear_del:
	device_remove_file(&chip->class_dev, &dev_attr_cycle_clear);
oem_batt_drv_err_device_del:
	device_del(&chip->class_dev);
oem_batt_drv_err_classdev_put:
	put_device(&chip->class_dev);
	class_unregister(&oem_batt_class);
	chip->create_class = false;

	return ret;
}

void oem_batt_drv_remove(struct max77729_fuelgauge_data *fuelgauge)
{
	struct oem_battery_drv_chip	*chip = oem_batt_drv_chip;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	cancel_delayed_work(&chip->battery_monitor);

	if (chip->batt_drv_wq)
		destroy_workqueue(chip->batt_drv_wq);

	if (chip->create_class) {
		device_remove_file(&chip->class_dev, &dev_attr_cycle_clear);
		device_del(&chip->class_dev);
		put_device(&chip->class_dev);
		class_unregister(&oem_batt_class);
		chip->create_class = false;
	}

	oem_batt_drv_chip = NULL;

	return;
}

void oem_batt_drv_suspend(struct max77729_fuelgauge_data *fuelgauge)
{
	struct oem_battery_drv_chip	*chip = oem_batt_drv_chip;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	set_bit(BATT_DRV_EVENT_SUSPEND, &chip->event_state);
	cancel_delayed_work(&chip->battery_monitor);

	return;
}

int oem_batt_drv_resume(struct max77729_fuelgauge_data *fuelgauge)
{
	struct oem_battery_drv_chip	*chip = oem_batt_drv_chip;

	BATT_DRV_DBGLOG("[%s] in\n", __func__);

	clear_bit(BATT_DRV_EVENT_SUSPEND, &chip->event_state);
	queue_delayed_work(chip->batt_drv_wq, &chip->battery_monitor, BATT_DRV_RESUME_INTERVAL);
	wake_up(&chip->wq);

	return 0;
}

MODULE_AUTHOR("FCNT");
MODULE_DESCRIPTION("OEM Fuel Gauge Driver");
MODULE_LICENSE("GPL");
