// SPDX-License-Identifier: GPL-2.0+
/*----------------------------------------------------------------------------*/
// (C) 2022 FCNT LIMITED
/*----------------------------------------------------------------------------*/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/kthread.h>
#include <linux/init.h>
#include <linux/moduleparam.h>
#include <linux/of.h>
#include <linux/timer.h>
#include <linux/power_supply.h>
#include <linux/seq_file.h>
#include <linux/pm_wakeup.h>
#include <linux/platform_device.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/reboot.h>
#include <linux/slab.h>
#include <linux/printk.h>
#include <linux/nonvolatile_common.h>
#include <linux/pmic-voter.h>
#include <linux/mfd/max77729.h>
#include <linux/mfd/max77729-private.h>
#include <linux/usb/typec/maxim/max77729-muic.h>
#include <linux/usb/typec/maxim/max77729_usbc.h>
#include "max77729_charger.h"

#include <linux/mfd/oem_charger.h>
#include "oem_charger_local.h"

extern int charging_mode;
extern bool max77729_charger_unlock(struct max77729_charger_data *charger);
extern void max77729_set_water_disable_reg(u8 data);
extern void max77729_set_ccx_open_set_reg(u8 data);
extern int max77729_get_cc_sts0_reg(u8 *value);

#define CHG_DRV_DBGLOG(x, y...)	   					\
	if (chg_drv_debug){ 							\
		printk(KERN_ERR "[oem_chg] " x, ## y);		\
	}
#define CHG_DRV_INFOLOG(x, y...)			printk(KERN_INFO "[oem_chg] " x, ## y)
#define CHG_DRV_WARNLOG(x, y...)			printk(KERN_WARNING "[oem_chg] " x, ## y)
#define CHG_DRV_ERRLOG(x, y...)				printk(KERN_ERR "[oem_chg] " x, ## y)
#define CHG_DRV_RECLOG(x, y...)				printk("REC@CHG@36[oem_chg] " x, ## y)

#define CHG_DRV_SOURCE_AC					(BIT(0) << OEM_CHG_SOURCE_AC)
#define CHG_DRV_SOURCE_USB					(BIT(0) << OEM_CHG_SOURCE_USB)
#define CHG_DRV_SOURCE_APSD					(BIT(0) << OEM_CHG_SOURCE_APSD)

#define CHG_DRV_SOURCE_USB_PORT				(CHG_DRV_SOURCE_AC | CHG_DRV_SOURCE_USB | CHG_DRV_SOURCE_APSD)
#define CHG_DRV_SOURCE(x)					(BIT(0) << x)
#define CHG_DRV_SOURCE_IS_USB_PORT(x)		((CHG_DRV_SOURCE_USB_PORT) & CHG_DRV_SOURCE(x))
#define CHG_DRV_SOURCE_MASK					(CHG_DRV_SOURCE_USB_PORT)
#define CHG_DRV_IS_USB_PORT_CHARGE(chip)	((chip->connect_state  == CHG_DRV_CONNECT_STATE_USB) ? true : false)

#define CHG_DRV_TYPE_LOG(chip)				("USB")

#define CHG_DRV_GET_SMALL(a ,b)				(a > b ? b : a)
#define CHG_DRV_GET_LARGE(a ,b)				(a > b ? a : b)

#define CHG_MA_TO_UA(mA)					(mA * 1000)
#define CHG_MIN_TO_MSEC(min)				(min * 60 * 1000)

#define APNV_CHG_ADJ_VMAX_I					(41041)
#define APNV_MC_CHARGE_FLAG					(49144)

#define CHG_DRV_CHARGE_MODE_INVALID			(-1)
#define CHG_DRV_CHARGE_MODE_NOT_CHARGE		(0)
#define CHG_DRV_CHARGE_MODE_MANUAL			(1)
#define CHG_DRV_CHARGE_MODE_NUM				(2)

#define CHG_DRV_CURRENT_OFF					(0)
#define CHG_DRV_CURRENT_500					(500)
#define CHG_DRV_CURRENT_1500				(1500)
#define CHG_DRV_CURRENT_1800				(1800)

#define CHG_DRV_SET_TIMER_300MS				(300)
#define CHG_DRV_SET_TIMER_500MS				(500)
#define CHG_DRV_SET_TIMER_750MS				(750)
#define CHG_DRV_SET_TIMER_1500MS			(1500)
#define CHG_DRV_SET_TIMER_1S				(  1 * 1000)
#define CHG_DRV_SET_TIMER_2S				(  2 * 1000)
#define CHG_DRV_SET_TIMER_10S				( 10 * 1000)
#define CHG_DRV_SET_TIMER_60S				( 60 * 1000)
#define CHG_DRV_SET_TIMER_300S				(300 * 1000)
#define CHG_DRV_SET_TIMER_600S				(600 * 1000)

#define CHG_DRV_ICL_NOT_LIMITED				(0)						/* Input Current Limit - not limited	  */
#define CHG_DRV_ICL_LIMITED_2000			2000					/* Input Current Limit -  limited 2000mA  */
#define CHG_DRV_ICL_LIMITED_1800			1800					/* Input Current Limit -  limited 1800mA  */
#define CHG_DRV_ICL_LIMITED_1500			1500					/* Input Current Limit -  limited 1500mA  */
#define CHG_DRV_ICL_LIMITED_900				900						/* Input Current Limit -  limited 900mA   */
#define CHG_DRV_ICL_LIMITED_500				500						/* Input Current Limit -  limited 500mA   */
#define CHG_DRV_ICL_LIMITED_300				300						/* Input Current Limit -  limited 300mA   */
#define CHG_DRV_ICL_LIMITED_150				150						/* Input Current Limit -  limited 150mA   */
#define CHG_DRV_ICL_LIMITED_100				100						/* Input Current Limit -  limited 100mA   */
#define CHG_DRV_ICL_LIMITED_MIN				300						/* Input Current Limit -  limited minimum */
#define CHG_DRV_ICL_DISABLE					(-1)					/* charge disable						  */

#define CHG_DRV_QUEUE_LOCK()				do { \
												unsigned long flags; \
												spin_lock_irqsave(&chg_drv_queue_lock, flags);

#define CHG_DRV_QUEUE_UNLOCK()					spin_unlock_irqrestore(&chg_drv_queue_lock, flags); \
											} while (0);

#define CHG_DRV_EVT_QUEUE_MAX				(30)
#define CHG_DRV_BATT_VOLT_INDEX				(5)
#define CHG_DRV_ADP_VOLT_CHK_CNT			(4)
#define CHG_DRV_CHECK_AICL_DETECT_CNT		(20)

/* charge err status bit */
#define CHG_DRV_PARAM_NO_ERROR				0x00000000
#define CHG_DRV_PARAM_BATT_TEMP_HOT			0x00000001
#define CHG_DRV_PARAM_BATT_TEMP_COLD		0x00000002
#define CHG_DRV_PARAM_BATT_VOLTAGE_OV		0x00000004
#define CHG_DRV_PARAM_ADAPTER_VOLTAGE_OV	0x00000008
#define CHG_DRV_PARAM_ADAPTER_VOLTAGE_UV	0x00000010
#define CHG_DRV_PARAM_OVP_DETECT_ERROR		0x00000020
#define CHG_DRV_PARAM_BATT_TEMP_WARM		0x00000040
#define CHG_DRV_PARAM_BATT_TEMP_COOL		0x00000080
#define CHG_DRV_PARAM_CHG_DISABLE			0x00000100
#define CHG_DRV_PARAM_UNSPEC_FAILURE		0x00000200
#define CHG_DRV_PARAM_USB_PORT_FAILURE		0x00000400
#define CHG_DRV_PARAM_USB_HOT_ERROR			0x00000800
#define CHG_DRV_PARAM_HOLDER_FAILURE		0x00001000
#define CHG_DRV_PARAM_WEAK_ADAPTER			0x00002000
#define CHG_DRV_PARAM_9V_UV_ERROR			0x00004000
#define CHG_DRV_PARAM_QC_UV_ERROR			CHG_DRV_PARAM_9V_UV_ERROR
#define CHG_DRV_PARAM_RECOVERY_ERROR		0x00008000
#define CHG_DRV_PARAM_CHG_TYPE_UNKNOWN		0x00010000
#define CHG_DRV_PARAM_FGIC_INITIALIZING		0x00020000
#define CHG_DRV_PARAM_WDOG_TIMEOUT			0x00040000
#define CHG_DRV_PARAM_CHGIC_ERROR			0x00080000
#define CHG_DRV_PARAM_AICL_ERROR			0x00100000
#define CHG_DRV_PARAM_BATT_ID_ERROR			0x00200000
#define CHG_DRV_PARAM_PDO_PROFILE_ERROR		0x00400000
#define CHG_DRV_PARAM_7V_UV_ERROR			0x00800000
#define CHG_DRV_PARAM_UNSPEC_CABLE			0x01000000
#define CHG_DRV_PARAM_EXT_FACTOR			0x02000000
#define CHG_DRV_PARAM_USB_SUSPEND			0x04000000
#define CHG_DRV_PARAM_SCHEDULE_CHARGE		0x08000000

#define CHG_DRV_BATT_VOLT_UV_DEFAULT		(3400000)
#define CHG_DRV_BATT_VOLT_OV_DEFAULT		(4450000)
#define CHG_DRV_BATT_SOC_FULL_DEFAULT		(100)
#define CHG_DRV_BATT_SOC_RECHG_DEFAULT		(98)
#define CHG_DRV_BATT_SOC_FULL_LONGEVITY_DEFAULT		(87)
#define CHG_DRV_BATT_SOC_RECHG_LONGEVITY_DEFAULT	(85)
#define CHG_DRV_BATT_TEMP_HOT_DEFAULT		(450)
#define CHG_DRV_BATT_TEMP_COLD_DEFAULT		(0)
#define CHG_DRV_BATT_TEMP_CANCEL_DEFAULT	(10)
#define CHG_DRV_FCC_DEFAULT					(500)
#define CHG_DRV_FCC_WARM_DEFAULT			(1200)
#define CHG_DRV_FCC_COOL_DEFAULT			(800)
#define CHG_DRV_CHARGE_VOLT_DEFAULT			(4340000)

#define CHG_DRV_REDUCTION_MONITOR_START		(95)
#define CHG_DRV_REDUCTION_MONITOR_COUNT		(3)
#define CHG_DRV_REDUCTION_MONITOR_CURRENT	(37500)
#define CHG_DRV_REDUCTION_MONITOR_AVG		(37500)
#define CHG_DRV_REDUCTION_DISABLED			(-1)

#define OEM_CHG_VOTER						"OEM_CHG_VOTER"
#define OEM_REDUCTION_VOTER					"OEM_REDUCTION_VOTER"

enum chg_drv_state_type {
	CHG_DRV_STATE_IDLE = 0,					/* idle						*/
	CHG_DRV_STATE_CHARGING,					/* charging/re-charging		*/
	CHG_DRV_STATE_FULL,						/* full						*/
	CHG_DRV_STATE_ERROR,					/* charge error occurred	*/
	CHG_DRV_STATE_NUM,

	CHG_DRV_STATE_NOCHANGE,
};

enum chg_drv_event_type {
	CHG_DRV_EVENT_MONITOR_PARAM = 0,		/* period monitor		*/
	CHG_DRV_EVENT_CHARGE_INFO_NOTICE,		/* charge info notice	*/
	CHG_DRV_EVENT_CHARGE_CTRL,				/* charge control		*/
	CHG_DRV_EVENT_CHARGE_TIMER,				/* timer expire			*/
	CHG_DRV_EVENT_PARAM_UPDATE,				/* parameter update		*/
	CHG_DRV_EVENT_NUM
};

enum chg_drv_connect_state_type {
	CHG_DRV_CONNECT_STATE_NONE = 0,
	CHG_DRV_CONNECT_STATE_USB,
	CHG_DRV_CONNECT_STATE_NUM,
	CHG_DRV_CONNECT_STATE_NOCHANGE,
};

enum chg_drv_connect_event_type {
	CHG_DRV_CONNECT_EVENT_USB_IN = 0,
	CHG_DRV_CONNECT_EVENT_USB_OUT,
	CHG_DRV_CONNECT_EVENT_NUM,
};

enum chg_drv_monitor_type {
	CHG_DRV_MONITOR_NORMAL = 0,
	CHG_DRV_MONITOR_ADAPTER_VOLTAGE,
	CHG_DRV_MONITOR_AICL,
	CHG_DRV_MONITOR_NUM,
};

enum chg_drv_notice_type {
	CHG_DRV_NOTICE_NON = 0,					/* 	*/
	CHG_DRV_NOTICE_OVP_NOTIFY_ERROR,		/* ovp notify error			*/
	CHG_DRV_NOTICE_USBHOT_NOTIFY_ERROR,		/* usb hot notify error		*/
	CHG_DRV_NOTICE_PDO_ERROR,				/* pdo profile notify error	*/
	CHG_DRV_NOTICE_CHG_ENABLE,				/* charge enable			*/
	CHG_DRV_NOTICE_CHANGE_CURRENT,			/* change of the current	*/
	CHG_DRV_NOTICE_NUM
};

enum chg_drv_timer_type {
	CHG_DRV_TIMER_NON = 0,					/* events other than a timer 	*/
	CHG_DRV_TIMER_FULL_1,					/* full charge detection timer1	*/
	CHG_DRV_TIMER_FULL_2,					/* full charge detection timer2	*/
	CHG_DRV_TIMER_PROTECTION,				/* protection timer				*/
	CHG_DRV_TIMER_NUM
};

enum chg_drv_ctrl_type {
	CHG_DRV_CONTROL_NONE = 0,
	CHG_DRV_CONTROL_USB_CHARGE,
	CHG_DRV_CONTROL_PRECHARGE,
	CHG_DRV_CONTROL_CHARGE_START,
	CHG_DRV_CONTROL_POWERPATH,
	CHG_DRV_CONTROL_RECHARGE,
	CHG_DRV_CONTROL_STOP,
	CHG_DRV_CONTROL_NUM,
};

enum chg_drv_device_type {
	CHG_DRV_DEVICE_TYPE_USB = 0,
	CHG_DRV_DEVICE_TYPE_NUM,
};

enum chg_drv_charge_type {
	CHG_DRV_CHARGE_TYPE_UNKNOWN = 0,
	CHG_DRV_CHARGE_TYPE_NORMAL,
	CHG_DRV_CHARGE_TYPE_PD,
	CHG_DRV_CHARGE_TYPE_NUM,
};

enum chg_drv_result_type {
	CHG_DRV_RESULT_SUCCESS = 0,
	CHG_DRV_RESULT_VOLTAGE_ERROR,
};

enum chg_drv_timer_state {
	CHG_DRV_CHARGE_TIMER_STATE_NONE = 0,
	CHG_DRV_CHARGE_TIMER_STATE_FULL1,
	CHG_DRV_CHARGE_TIMER_STATE_FULL2,
};

enum chg_drv_aicl_threshold {
	CHG_DRV_AICL_THRESHOLD_4P5  = 0x00,
	CHG_DRV_AICL_THRESHOLD_4P6  = 0x08,
	CHG_DRV_AICL_THRESHOLD_4P7  = 0x10,
	CHG_DRV_AICL_THRESHOLD_4P85 = 0x18,
};

struct chg_drv_monitor_param {
	int usbin_uv;
	int batt_volt_uv;
	int batt_temp;
	int soc;
	int fgic_initial_sts;
	int fcc_ua;
	int effective_icl;
	int effective_fcc;
	int usb_phy_icl;
	u8	vbadc[CHG_DRV_ADP_VOLT_CHK_CNT];
	u8	chgin_dtls;
	u8	chg_dtls;
	u8	bat_dtls;
	u8	cc_sts;
};

struct chg_drv_connect_ctrl {
	enum chg_drv_connect_state_type next_state;
	enum chg_drv_ctrl_type charge_ctrl;
};

struct chg_drv_monitor_info {
	enum chg_drv_monitor_type type;
};

struct chg_drv_connect_info {
	enum chg_drv_connect_event_type event;
	int chg_source;
	unsigned int max_current;
};

struct chg_drv_change_info {
	enum chg_drv_device_type device;
	int max_current;
};

struct chg_drv_notice_info {
	enum chg_drv_notice_type type;
	unsigned int ovp_detect;					/* type:OVP_DETECT parameter		*/
	unsigned int usb_hot_detect;				/* type:USB_HOT_DETECT parameter	*/
	unsigned int pdo_error_detect;				/* type:PDO_ERROR_DETECT parameter	*/
	int icl;									/* type:CHG_ENABLE parameter		*/
	int charge_mode;							/* type:CHANGE_MODE parameter		*/
	struct chg_drv_change_info change_info;		/* type:CHANGE_CURRENT parameter	*/
};

struct chg_drv_timer_info {
	enum chg_drv_timer_type type;
	enum chg_drv_device_type device;
};

union chg_drv_evt_data {
	struct chg_drv_monitor_info monitor_info;
	struct chg_drv_connect_info connect_info;
	struct chg_drv_notice_info notice_info;
	struct chg_drv_timer_info timer_info;
};

struct chg_drv_evt_queue {
	struct list_head head;				/* queue head	*/
	enum chg_drv_event_type event;		/* event id		*/
	union chg_drv_evt_data data;		/* event data	*/
};

struct chg_drv_err_ctrl {
	int set_status;
	int set_health;
	enum chg_drv_ctrl_type charge_ctrl;
};

struct chg_drv_device_info {
	unsigned int max_current;
};

struct chg_drv_err_info {
	bool safety_timer_expire;
	bool batt_temp_warm_restrict;
	bool batt_temp_cool_restrict;
	bool unspec_cable_err;
};

struct chg_drv_safety_err_info {
	bool ovp_detect;
	bool usb_port_failure;
	bool pdo_profile_error;
	bool usb_removal;
};

struct chg_drv_info {
	unsigned int	state;
	unsigned int	chg_source;
	unsigned int	chg_current;
	unsigned int	current_err;
};

struct chg_drv_batt_volt_info {
	int volt_data[CHG_DRV_BATT_VOLT_INDEX];
	int index;
	bool data_full;
};

struct chg_drv_dt {
	int batt_over_voltage;
	int batt_unspec_voltage;
	int batt_low_voltage_onchg;
	int batt_over_voltage_offchg;

	int chg_full_soc_lv1;
	int chg_full_soc_lv2;
	int chg_rechg_soc_lv1;
	int chg_rechg_soc_lv2;

	int chg_full_soc_longevity_lv1;
	int chg_full_soc_longevity_lv2;
	int chg_rechg_soc_longevity_lv1;
	int chg_rechg_soc_longevity_lv2;

	int chg_voltage;
	int chg_voltage_cycle_lv1;
	int chg_voltage_cycle_lv2;
	int chg_voltage_reduction;
	int chg_voltage_aicl_threshold;

	int chg_safety_timer_min;
	int chg_full1_timer_min;
	int chg_full2_timer_min;

	int batt_temp_hot_limit;
	int batt_temp_warm_limit;
	int batt_temp_cool_limit;
	int batt_temp_cold_limit;
	int batt_temp_start_limit;
	int batt_temp_limit_cancel;

	int wake_lock_timer_sec;

	int usb_normal_fcc_max;
	int usb_pd_fcc_max;
	int common_fcc_low_batt;
	int common_fcc_temp_warm_restrict;
	int common_fcc_temp_cool_restrict;

	int chg_consecutive_count;
	int chg_termcurrent_ma;
	int chg_cycle_termcurrent_ma;

	int *chg_icl_thermal_mitigation;
};

struct oem_charger_drv_chip {
	struct max77729_charger_data	*charger;

	struct task_struct				*chg_thread;
	wait_queue_head_t				chg_wq;			/* charger thread waitqueue head	*/

	struct device					*dev;
	struct device_node				*of_node;

	struct wakeup_source			*chg_drv_timer_wake_lock;
	struct wakeup_source			*chg_drv_charging_wake_lock;

	struct chg_drv_dt				dt_param;

	unsigned int					connect_state;
	int								max_icl_value;
	int								charge_type;
	int								heat_charge_mode;
	int								consecutive_count;
	bool							battery_low_flag;
	int								aicl_threshold;
	int								aicl_threshold_detect_count;
	int								schedule_chg;

	int								batt_temp_hot_limit;
	int								batt_temp_warm_limit;
	int								batt_temp_cold_limit;
	int								batt_temp_cool_limit;

	enum chg_drv_timer_state		full_timer_state;

	struct chg_drv_device_info		dev_usb;

	struct delayed_work				charge_monitor_work;
	struct delayed_work				full_timer_work1;
	struct delayed_work				full_timer_work2;
	struct delayed_work				safety_timer_work;
	struct delayed_work				adapter_volt_monitor_work;
	struct delayed_work				watchdog_work;
	struct delayed_work				reduction_work;
	struct delayed_work				usb_removal_wait_work;
	struct delayed_work				aicl_monitor_work;

	struct power_supply_desc		oem_chg_psy_desc;
	struct power_supply				*oem_chg_psy;

	struct power_supply				*battery_psy;
	struct power_supply				*usb_psy;
	struct power_supply				*bms_psy;
	struct power_supply				*oem_battery_psy;

	struct votable					*fv_votable;
	struct votable					*fcc_votable;
	struct votable					*usb_icl_votable;

	int								charge_mode;
	int								vmax_adj;
	enum chg_drv_ctrl_type 			charge_ctrl;
	struct chg_drv_err_info			err_info;
	struct chg_drv_safety_err_info	safety_err_info;
	struct chg_drv_info				charger_info;
	struct chg_drv_batt_volt_info	batt_volt_data;
	int								safety_timer_reduce_flag;
	bool 							chg_voltage_reduction_flag;
	bool							reduction_motinor_enable;
	int								reduction_disable_count;
	u8								mode_val;

	int								chg_enable;

	struct workqueue_struct			*chg_drv_wq;

	bool							longevity_charge_flag;
	enum chg_cycle_level			cycle_chg_lvl;

	int								thermal_level;
	int								thermal_level_max;

	struct device	class_dev;
	bool create_class;
};

#if defined(FJFEAT_AIRMONKEY)
static int chg_drv_debug = 1;
#else
static int chg_drv_debug = 0;
#endif
module_param_named(dbglog_charger, chg_drv_debug, int, S_IRUGO|S_IWUSR);

static struct oem_charger_drv_chip *oem_chg_drv_chip = NULL;
static unsigned long charger_src = 0;
static unsigned long charger_apsd_src = 0;
static int chg_enable = 0;
static int chg_mode = 0;

static DEFINE_SPINLOCK(chg_drv_queue_lock);

static LIST_HEAD(chg_drv_evt_list_free);
static LIST_HEAD(chg_drv_evt_list_exec);
static struct chg_drv_evt_queue charge_evt_queue[CHG_DRV_EVT_QUEUE_MAX];

static enum power_supply_property chg_drv_oem_chg_properties[] = {
	POWER_SUPPLY_PROP_CHARGE_CONTROL_LIMIT,
	POWER_SUPPLY_PROP_CHARGE_CONTROL_LIMIT_MAX,
};

enum chg_drv_err_type {
	CHG_DRV_ERR_NON = 0,					/* error not occurred		*/
	CHG_DRV_ERR_EXPIRE_TIME_OUT,			/* safe timer expire		*/
	CHG_DRV_ERR_USB_PORT_FAILURE,			/* usb port failure			*/
	CHG_DRV_ERR_OVP_DETECT,					/* ovp detect				*/
	CHG_DRV_ERR_CHARGE_DISABLE,				/* charge disable			*/
	CHG_DRV_ERR_BATT_TEMP_HOT_ERR,			/* batt temp hot 			*/
	CHG_DRV_ERR_BATT_TEMP_COLD_ERR,			/* batt temp cold			*/
	CHG_DRV_ERR_OVER_BATT_VOLTAGE,			/* over batt voltage		*/
	CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR,	/* over supply voltage		*/
	CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR,	/* under supply voltage		*/
	CHG_DRV_ERR_USB_HOT_ERR,				/* usb hot					*/
	CHG_DRV_ERR_HEAT_CHARGE,				/* heat charge				*/
	CHG_DRV_ERR_WEAK_ADAPTER,				/* weak adapter detect		*/
	CHG_DRV_ERR_OTHER_ERR,					/* other error				*/
	CHG_DRV_ERR_FGIC_INITIALIZING,			/* FGIC initializing		*/
	CHG_DRV_ERR_UNSPEC_CABLE,				/* unspecified cable		*/
	CHG_DRV_ERR_EXT_FACTOR,					/* external factors			*/
	CHG_DRV_ERR_USB_SUSPEND,				/* usb suspend				*/
	CHG_DRV_ERR_SCHEDULE_CHARGE,			/* schedule charge			*/
	CHG_DRV_ERR_NUM
};

static const struct chg_drv_err_ctrl err_ctrl_table[CHG_DRV_ERR_NUM] = {
/*	  set_status						set_health									charge_ctrl					*/
	{ POWER_SUPPLY_STATUS_CHARGING,		POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_NONE		},	/* CHG_DRV_ERR_NON						*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_SAFETY_TIMER_EXPIRE,	CHG_DRV_CONTROL_POWERPATH	},	/* CHG_DRV_ERR_EXPIRE_TIME_OUT			*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_OVERCURRENT,			CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_USB_PORT_FAILURE			*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_UNSPEC_FAILURE,			CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_OVP_DETECT				*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_CHARGE_DISABLE			*/
	{ POWER_SUPPLY_STATUS_CHARGING,		POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_POWERPATH	},	/* CHG_DRV_ERR_BATT_TEMP_HOT_ERR		*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_COLD,					CHG_DRV_CONTROL_POWERPATH	},	/* CHG_DRV_ERR_BATT_TEMP_COLD_ERR		*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_OVERVOLTAGE,			CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_OVER_BATT_VOLTAGE		*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_UNSPEC_FAILURE,			CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR	*/
	{ POWER_SUPPLY_STATUS_DISCHARGING,	POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR	*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_OVERCURRENT,			CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_USB_HOT_ERR				*/
	{ POWER_SUPPLY_STATUS_CHARGING,		POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_POWERPATH	},	/* CHG_DRV_ERR_HEAT_CHARGE				*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_WATCHDOG_TIMER_EXPIRE,	CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_WEAK_ADAPTER				*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_WATCHDOG_TIMER_EXPIRE,	CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_OTHER_ERR				*/
	{ POWER_SUPPLY_STATUS_CHARGING,		POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_POWERPATH	},	/* CHG_DRV_ERR_FGIC_INITIALIZING		*/
	{ POWER_SUPPLY_STATUS_DISCHARGING,	POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_UNSPEC_CABLE				*/
	{ POWER_SUPPLY_STATUS_DISCHARGING,	POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_POWERPATH	},	/* CHG_DRV_ERR_EXT_FACTOR				*/
	{ POWER_SUPPLY_STATUS_DISCHARGING,	POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_STOP		},	/* CHG_DRV_ERR_USB_SUSPEND				*/
	{ POWER_SUPPLY_STATUS_NOT_CHARGING,	POWER_SUPPLY_HEALTH_GOOD,					CHG_DRV_CONTROL_POWERPATH	},	/* CHG_DRV_ERR_SCHEDULE_CHARGE			*/
};

typedef enum chg_drv_state_type (*chg_drv_func)(struct oem_charger_drv_chip *chip, union chg_drv_evt_data *data);

static void chg_drv_queue_put(struct chg_drv_evt_queue *entry_p, struct list_head *head_p)
{
	CHG_DRV_QUEUE_LOCK();
	list_add_tail(&entry_p->head, head_p);
	CHG_DRV_QUEUE_UNLOCK();
}

static void *chg_drv_queue_get(struct list_head *head_p)
{
	void *queue_p = NULL;
	struct list_head *p, *n;

	CHG_DRV_QUEUE_LOCK();
	list_for_each_safe(p, n, head_p) {
		queue_p = list_entry(p, struct chg_drv_evt_queue, head);
		if (queue_p != NULL) {
			list_del(p);
			break;
		}
		queue_p = NULL;
	}
	CHG_DRV_QUEUE_UNLOCK();

	return queue_p;
}

static void chg_drv_event_req(struct oem_charger_drv_chip *chip,
								enum chg_drv_event_type event,
								union chg_drv_evt_data *evt_data)
{
	struct chg_drv_evt_queue *evt_queue_p = NULL;

	evt_queue_p = chg_drv_queue_get(&chg_drv_evt_list_free);
	if (unlikely(evt_queue_p == NULL)) {
		CHG_DRV_ERRLOG("[%s] charger event queue empty!!\n", __func__);
		return;
	}

	evt_queue_p->event = event;
	if (evt_data == NULL) {
		memset(&evt_queue_p->data, 0, sizeof(union chg_drv_evt_data));
	} else {
		memcpy(&evt_queue_p->data, evt_data, sizeof(union chg_drv_evt_data));
	}

	chg_drv_queue_put(evt_queue_p, &chg_drv_evt_list_exec);

	if (chip != NULL) {
		wake_up(&chip->chg_wq);
	}
}

static int chg_drv_reg_read(struct oem_charger_drv_chip *chip, u8 reg, u8 *val)
{
	struct max77729_charger_data *charger = chip->charger;
	u8 val_work = 0;

	int ret;

	ret = max77729_read_reg(charger->i2c, reg, &val_work);
	if (likely(ret == 0)) {
		*val = val_work;
	} else {
		CHG_DRV_ERRLOG("register read failure. addr:[0x%x]\n", reg);
	}

	CHG_DRV_DBGLOG("[%s] reg:[0x%02X] val:[0x%02X] ret:[%d]\n",
					__func__, reg, *val, ret);

	return ret;

}

static int chg_drv_reg_write_byte(struct oem_charger_drv_chip *chip, u8 reg, u8 mask, u8 val)
{
	struct max77729_charger_data *charger = chip->charger;
	int ret;

	(void)max77729_charger_unlock(charger);

	ret = max77729_update_reg(charger->i2c, reg, val, mask);
	if (unlikely(ret != 0)) {
		CHG_DRV_ERRLOG("register write failure. reg:[0x%02X] mask:[0x%02X] val:[0x%02X]\n",
						reg, mask, val);
	}

	CHG_DRV_DBGLOG("[%s] reg:[0x%02X] mask:[0x%02X] val:[0x%02X] ret:[%d]\n",
					__func__, reg, mask, val, ret);

	return ret;
}

static int chg_drv_reg_write_bit(struct oem_charger_drv_chip *chip, u8 reg, u8 mask, u8 set)
{
	struct max77729_charger_data *charger = chip->charger;
	int ret;

	(void)max77729_charger_unlock(charger);

	if (set == CHG_DRV_SET_HIGH) {
		ret = max77729_update_reg(charger->i2c, reg, mask, mask);
	} else {
		ret = max77729_update_reg(charger->i2c, reg, 0x00, mask);
	}
	if (unlikely(ret != 0)) {
		CHG_DRV_ERRLOG("register write failure. reg:[0x%02X] mask:[0x%02X] set:[%d]\n",
						reg, mask, set);
	}

	CHG_DRV_DBGLOG("[%s] reg:[0x%02X] mask:[0x%02X] set:[%d] ret:[%d]\n",
					__func__, reg, mask, set, ret);

	return ret;
}

static void chg_drv_check_current(struct oem_charger_drv_chip *chip, unsigned int *fcc, unsigned int *icl)
{
	unsigned int icl_work = CHG_DRV_ICL_NOT_LIMITED;
	unsigned int fcc_work = CHG_DRV_ICL_NOT_LIMITED;
	int charge_type = CHG_DRV_CHARGE_TYPE_UNKNOWN;

	icl_work = chip->dev_usb.max_current;
	charge_type = chip->charge_type;

	/* FCC */
	if (charge_type == CHG_DRV_CHARGE_TYPE_PD) {
		fcc_work = chip->dt_param.usb_pd_fcc_max;
	} else {
		if (charger_apsd_src == CHG_DRV_SOURCE_USB) {
			fcc_work = CHG_DRV_CURRENT_500;
		} else {
			fcc_work = chip->dt_param.usb_normal_fcc_max;
		}
	}

	if (chip->err_info.batt_temp_warm_restrict) {
		fcc_work = CHG_DRV_GET_SMALL(chip->dt_param.common_fcc_temp_warm_restrict, fcc_work);
	}

	if (chip->err_info.batt_temp_cool_restrict) {
		fcc_work = CHG_DRV_GET_SMALL(chip->dt_param.common_fcc_temp_cool_restrict, fcc_work);
	}

	if (chip->battery_low_flag) {
		fcc_work = CHG_DRV_GET_SMALL(chip->dt_param.common_fcc_low_batt, fcc_work);
	}

	*fcc = fcc_work;

	/* ICL */
	if (charge_type == CHG_DRV_CHARGE_TYPE_UNKNOWN) {
		icl_work = CHG_DRV_GET_SMALL(CHG_DRV_ICL_LIMITED_500, icl_work);
	}

	if (chip->max_icl_value != CHG_DRV_ICL_NOT_LIMITED) {
		if (chip->max_icl_value == CHG_DRV_ICL_DISABLE) {
			icl_work = CHG_DRV_ICL_LIMITED_MIN;
		} else {
			icl_work = CHG_DRV_GET_SMALL(chip->max_icl_value, icl_work);
		}
	}

	if ((chip->thermal_level != 0) &&
		(chip->dt_param.chg_icl_thermal_mitigation != NULL)) {
		if (chip->dt_param.chg_icl_thermal_mitigation[chip->thermal_level] == CHG_DRV_ICL_DISABLE) {
			icl_work = CHG_DRV_ICL_LIMITED_MIN;
		} else {
			icl_work = CHG_DRV_GET_SMALL(chip->dt_param.chg_icl_thermal_mitigation[chip->thermal_level],
											icl_work);
		}
	}

	*icl = icl_work;

	CHG_DRV_INFOLOG("[%s] %s FCC:[%d] ICL:[%d]\n",
					__func__, CHG_DRV_TYPE_LOG(chip), *fcc, *icl);
}

static int chg_drv_update_cycle_charge_setting(struct oem_charger_drv_chip *chip)
{
	int ret = 0;
	enum chg_cycle_level old_cycle_chg_lvl;
	union power_supply_propval pval = {0, };

	old_cycle_chg_lvl = chip->cycle_chg_lvl;

	ret = power_supply_get_property(chip->oem_battery_psy,
									(enum power_supply_property)POWER_SUPPLY_EXT_PROP_CYCLE_CHG, &pval);
	if (ret < 0) {
		CHG_DRV_ERRLOG("[%s] get CYCLE_CHG error %d\n", __func__, ret);
		return ret;
	} else {
		if (old_cycle_chg_lvl == pval.intval) {
			return ret;
		} else {
			chip->cycle_chg_lvl = pval.intval;
		}
	}

	switch (chip->cycle_chg_lvl) {
	case CHG_CYCLE_LEVEL_NONE:
	case CHG_CYCLE_LEVEL_01:
	case CHG_CYCLE_LEVEL_02:
	default:
		chip->vmax_adj = chip->dt_param.chg_voltage;
		ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_03,
					GENMASK(5, 3)|GENMASK(2, 0), BIT(3) | BIT(2) | BIT(0));
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("[%s] CHG_REG_CNFG_03 Failed. ret:[%d]\n", __func__, ret);
		}
		break;
	}

	if (old_cycle_chg_lvl != chip->cycle_chg_lvl) {
		ret = vote(chip->fv_votable, OEM_CHG_VOTER, true, (chip->vmax_adj/1000));
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("fv_votable Failed. ret:[%d]\n", ret);
		}
	}

	return ret;
}

static int chg_drv_get_longevity_chg_mode(struct oem_charger_drv_chip *chip)
{
	int ret = 0;
	union power_supply_propval pval = {0, };

	ret = power_supply_get_property(chip->oem_battery_psy,
									(enum power_supply_property)POWER_SUPPLY_EXT_PROP_LONGEVITY_CHG_MODE, &pval);
	if (ret < 0) {
		chip->longevity_charge_flag = false;
	} else {
		if (pval.intval) {
			chip->longevity_charge_flag = true;
		} else {
			chip->longevity_charge_flag = false;
		}
	}

	return ret;
}

static int chg_drv_pmic_slight_charge(struct oem_charger_drv_chip *chip, enum chg_drv_device_type device)
{
	int ret = 0;

	do {
		ret = vote(chip->usb_icl_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_500);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("usb_icl_votable Failed. ret:[%d]\n", ret);
			break;
		}
	} while (0);

	return ret;
}

static void chg_drv_update_charge_type(struct oem_charger_drv_chip *chip)
{
	enum chg_drv_charge_type chg_type = chip->charge_type;

#ifdef CHG_DRV_CONFIG_PD_ENABLE
	if (chip->charger->pd_active) {
		chip->dev_usb.max_current = CHG_DRV_ICL_LIMITED_1800;
		chg_type = CHG_DRV_CHARGE_TYPE_PD;

		if (chip->charge_type != chg_type) {
			CHG_DRV_INFOLOG("[%s] %s charge type %d -> %d [0:etc 1:NORMAL 2:PD]\n",
								__func__, CHG_DRV_TYPE_LOG(chip),
								chip->charge_type, chg_type);
			chip->charge_type = chg_type;
			chg_drv_event_req(chip, CHG_DRV_EVENT_PARAM_UPDATE, NULL);
		}
	}
#else
	*chg_type = CHG_DRV_CHARGE_TYPE_NORMAL;
#endif

	return;
}

static void chg_drv_pmic_watchdog_pet(struct oem_charger_drv_chip *chip)
{
	int ret;

	ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_06, GENMASK(1, 0), BIT(0));
	if (unlikely(ret != 0)) {
		CHG_DRV_ERRLOG("[%s] WDOG_PET Failed. ret:[%d]\n", __func__, ret);
	}
}

static int chg_drv_pmic_current_setting(struct oem_charger_drv_chip *chip)
{
	unsigned int fcc = 0;
	unsigned int icl = 0;
	int ret = 0;

	do {
		chg_drv_check_current(chip, &fcc, &icl);

		ret = vote(chip->fcc_votable, OEM_CHG_VOTER, true, fcc);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("fcc_votable Failed. ret:[%d]\n", ret);
			break;
		}

		ret = vote(chip->usb_icl_votable, OEM_CHG_VOTER, true, icl);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("usb_icl_votable Failed. ret:[%d]\n", ret);
			break;
		}
	} while(0);

	return ret;
}

static int chg_drv_pmic_initial_setting(struct oem_charger_drv_chip *chip)
{
	int ret;
	union power_supply_propval pval = {0,};

	do {
		ret = vote(chip->usb_icl_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_500);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("usb_icl_votable Failed. ret:[%d]\n", ret);
			break;
		}
		mdelay(10);
		ret = vote(chip->fcc_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_500);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("fcc_votable Failed. ret:[%d]\n", ret);
			break;
		}
		ret = vote(chip->fv_votable, OEM_CHG_VOTER, true, (chip->vmax_adj/1000));
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("fv_votable Failed. ret:[%d]\n", ret);
			break;
		}
		ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_01,
										GENMASK(5, 4), BIT(5)|BIT(4));
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_01 Failed. ret:[%d]\n", ret);
			break;
		}
		ret = chg_drv_reg_write_bit(chip, MAX77729_CHG_REG_CNFG_06,
									BIT(4), CHG_DRV_SET_LOW);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_06 Failed. ret:[%d]\n", ret);
			break;
		}

		ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_12,
										GENMASK(4, 3), 0x00);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_12 Failed. ret:[%d]\n", ret);
			break;
		}
		ret = chg_drv_reg_write_bit(chip, MAX77729_CHG_REG_CNFG_00,
									BIT(4), CHG_DRV_SET_LOW);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_00 Failed. ret:[%d]\n", ret);
			break;
		}
		ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_01,
										GENMASK(2, 0), 0x00);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_01 Failed. ret:[%d]\n", ret);
			break;
		}

		ret = power_supply_get_property(chip->usb_psy, (enum power_supply_property)POWER_SUPPLY_EXT_PROP_WATER_DETECT, &pval);

		if (unlikely(ret < 0)) {
			CHG_DRV_ERRLOG("[%s] get WATER_DETECT error %d\n", __func__, ret);
			pval.intval = 0;
		}

		if (!!pval.intval == true) {
			ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_00,
											GENMASK(3, 0), 0x00);
		} else {
			ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_00,
											GENMASK(3, 0), 0x05);
		}

		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_01 Failed. ret:[%d]\n", ret);
			break;
		}
		ret = chg_drv_reg_write_bit(chip, MAX77729_CHG_REG_CNFG_07,
									BIT(7), CHG_DRV_SET_HIGH);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_00 Failed. ret:[%d]\n", ret);
			break;
		}
	} while(0);

	return ret;
}

static int chg_drv_pmic_normal_charge(struct oem_charger_drv_chip *chip)
{
	int ret = 0;

	do {
		ret = chg_drv_pmic_current_setting(chip);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("[%s] Normal Charge Setting Failed. ret:[%d]\n", __func__, ret);
			break;
		}
	} while (0);

	return ret;
}

static int chg_drv_pmic_pd_charge(struct oem_charger_drv_chip *chip)
{
	int ret = 0;

	ret = chg_drv_pmic_current_setting(chip);
	if (unlikely(ret != 0)) {
		CHG_DRV_ERRLOG("[%s] PD Charge Setting Failed. ret:[%d]\n", __func__, ret);
	}

	return ret;
}

static int chg_drv_pmic_charge_start(struct oem_charger_drv_chip *chip)
{
	int ret;

	do {
		ret = chg_drv_reg_write_bit(chip, MAX77729_CHG_REG_CNFG_00,
									BIT(4), CHG_DRV_SET_HIGH);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_00 Failed. ret:[%d]\n", ret);
			break;
		}
		ret = vote(chip->usb_icl_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_500);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("usb_icl_votable Failed. ret:[%d]\n", ret);
			break;
		}
		ret = vote(chip->fcc_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_500);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("fcc_votable Failed. ret:[%d]\n", ret);
			break;
		}
		ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_00,
										GENMASK(3, 0), 0x05);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_01 Failed. ret:[%d]\n", ret);
			break;
		}
	} while(0);

	return ret;
}

static int chg_drv_pmic_charge_powerpath(struct oem_charger_drv_chip *chip)
{
	int ret;

	do {
		ret = chg_drv_reg_write_bit(chip, MAX77729_CHG_REG_CNFG_00,
									BIT(4), CHG_DRV_SET_HIGH);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_00 Failed. ret:[%d]\n", ret);
			break;
		}
		ret = vote(chip->usb_icl_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_500);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("usb_icl_votable Failed. ret:[%d]\n", ret);
			break;
		}
		ret = vote(chip->fcc_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_OFF);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("fcc_votable Failed. ret:[%d]\n", ret);
			break;
		}
		ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_00,
										GENMASK(3, 0), 0x04);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_01 Failed. ret:[%d]\n", ret);
			break;
		}
	} while (0);

	return ret;
}

static int chg_drv_pmic_charge_stop(struct oem_charger_drv_chip *chip)
{
	int ret;

	do {
		ret = chg_drv_reg_write_bit(chip, MAX77729_CHG_REG_CNFG_00,
									BIT(4), CHG_DRV_SET_LOW);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_00 Failed. ret:[%d]\n", ret);
			break;
		}
		ret = vote(chip->usb_icl_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_OFF);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("usb_icl_votable Failed. ret:[%d]\n", ret);
			break;
		}
		ret = vote(chip->fcc_votable, OEM_CHG_VOTER, true, CHG_DRV_CURRENT_OFF);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("fcc_votable Failed. ret:[%d]\n", ret);
			break;
		}
		ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_00,
										GENMASK(3, 0), 0x00);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_01 Failed. ret:[%d]\n", ret);
			break;
		}
	} while (0);

	return ret;
}

static void chg_drv_start_monitor(struct oem_charger_drv_chip *chip)
{
	unsigned long set_timer;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	set_timer = msecs_to_jiffies(CHG_DRV_SET_TIMER_10S);
	queue_delayed_work(chip->chg_drv_wq, &chip->charge_monitor_work, round_jiffies_relative(set_timer));
}

static void chg_drv_stop_monitor(struct oem_charger_drv_chip *chip)
{
	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	cancel_delayed_work(&chip->charge_monitor_work);
}

static void chg_drv_monitor_work(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, charge_monitor_work.work);
	union chg_drv_evt_data evt_data;

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	CHG_DRV_DBGLOG("[%s] state=0x%x\n", __func__, chip->charger_info.state);

	evt_data.monitor_info.type = CHG_DRV_MONITOR_NORMAL;
	chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);
}

static void chg_drv_start_full_timer1(struct oem_charger_drv_chip *chip)
{
	unsigned long set_timer;

	if (chip->full_timer_state == CHG_DRV_CHARGE_TIMER_STATE_NONE) {
		CHG_DRV_DBGLOG("[%s] in\n", __func__);
		chip->full_timer_state = CHG_DRV_CHARGE_TIMER_STATE_FULL1;
		set_timer = msecs_to_jiffies(CHG_MIN_TO_MSEC(chip->dt_param.chg_full1_timer_min));
		queue_delayed_work(chip->chg_drv_wq, &chip->full_timer_work1, round_jiffies_relative(set_timer));
	}
}

static void chg_drv_start_full_timer2(struct oem_charger_drv_chip *chip)
{
	unsigned long set_timer;

	if (chip->full_timer_state == CHG_DRV_CHARGE_TIMER_STATE_FULL1) {
		CHG_DRV_DBGLOG("[%s] in\n", __func__);
		chip->full_timer_state = CHG_DRV_CHARGE_TIMER_STATE_FULL2;
		set_timer = msecs_to_jiffies(CHG_MIN_TO_MSEC(chip->dt_param.chg_full2_timer_min));
		queue_delayed_work(chip->chg_drv_wq, &chip->full_timer_work2, round_jiffies_relative(set_timer));
	}
}

static void chg_drv_stop_full_timer(struct oem_charger_drv_chip *chip)
{
	CHG_DRV_DBGLOG("[%s] in timer state:[%d]\n", __func__, chip->full_timer_state);

	if (chip->full_timer_state != CHG_DRV_CHARGE_TIMER_STATE_NONE) {
		cancel_delayed_work(&chip->full_timer_work1);
		cancel_delayed_work(&chip->full_timer_work2);
		chip->full_timer_state = CHG_DRV_CHARGE_TIMER_STATE_NONE;
	}
}

static void chg_drv_full_timer_work1(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, full_timer_work1.work);
	union chg_drv_evt_data evt_data;

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	CHG_DRV_DBGLOG("[%s] state=0x%x\n", __func__, chip->charger_info.state);

	evt_data.timer_info.type = CHG_DRV_TIMER_FULL_1;
	chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_TIMER, &evt_data);
}

static void chg_drv_full_timer_work2(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, full_timer_work2.work);
	union chg_drv_evt_data evt_data;

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	CHG_DRV_DBGLOG("[%s] state=0x%x\n", __func__, chip->charger_info.state);

	evt_data.timer_info.type = CHG_DRV_TIMER_FULL_2;
	chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_TIMER, &evt_data);
}

static void chg_drv_start_safety_timer(struct oem_charger_drv_chip *chip)
{
	unsigned long set_timer;

	if (chip->dt_param.chg_safety_timer_min <= 0) {
		return;
	}

	if (!delayed_work_pending(&chip->safety_timer_work)) {
		if (chip->safety_timer_reduce_flag) {
			set_timer = msecs_to_jiffies(CHG_DRV_SET_TIMER_300S);
			CHG_DRV_INFOLOG("[%s] set safety_timer(reduce) = 5min\n", __func__);
		} else {
			set_timer = msecs_to_jiffies(CHG_MIN_TO_MSEC(chip->dt_param.chg_safety_timer_min));
			CHG_DRV_INFOLOG("[%s] set safety_timer = %dmin\n", __func__, chip->dt_param.chg_safety_timer_min);
		}
		queue_delayed_work(chip->chg_drv_wq, &chip->safety_timer_work, round_jiffies_relative(set_timer));
	}
}

static void chg_drv_stop_safety_timer(struct oem_charger_drv_chip *chip)
{
	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	cancel_delayed_work(&chip->safety_timer_work);
}

static void chg_drv_safety_timer_work(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, safety_timer_work.work);
	union chg_drv_evt_data evt_data;

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	CHG_DRV_DBGLOG("[%s] state=0x%x\n", __func__, chip->charger_info.state);

	evt_data.timer_info.type = CHG_DRV_TIMER_PROTECTION;
	chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_TIMER, &evt_data);
}

static void chg_drv_start_adapter_volt_monitor(struct oem_charger_drv_chip *chip, int set_period)
{
	unsigned long set_timer;

	if (!delayed_work_pending(&chip->adapter_volt_monitor_work)) {
		CHG_DRV_DBGLOG("[%s] in\n", __func__);
		set_timer = msecs_to_jiffies(set_period);
		queue_delayed_work(chip->chg_drv_wq, &chip->adapter_volt_monitor_work, set_timer);
	}
}

static void chg_drv_stop_adapter_volt_monitor(struct oem_charger_drv_chip *chip)
{
	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	cancel_delayed_work(&chip->adapter_volt_monitor_work);
}

static void chg_drv_adapter_volt_monitor_work(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, adapter_volt_monitor_work.work);
	union chg_drv_evt_data evt_data;

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	CHG_DRV_DBGLOG("[%s] state=0x%x\n", __func__, chip->charger_info.state);

	evt_data.monitor_info.type = CHG_DRV_MONITOR_ADAPTER_VOLTAGE;
	chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);
}

static void chg_drv_start_watchdog_timer(struct oem_charger_drv_chip *chip)
{
	unsigned long set_timer;

	if (!delayed_work_pending(&chip->watchdog_work)) {
		set_timer = msecs_to_jiffies(CHG_DRV_SET_TIMER_10S);
		queue_delayed_work(chip->chg_drv_wq, &chip->watchdog_work, round_jiffies_relative(set_timer));
	}
}

static void chg_drv_stop_watchdog_timer(struct oem_charger_drv_chip *chip)
{
	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	cancel_delayed_work(&chip->watchdog_work);
}

static void chg_drv_watchdog_work(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, watchdog_work.work);

	CHG_DRV_DBGLOG("[%s] WatchDog Pet.\n", __func__);

	chg_drv_pmic_watchdog_pet(chip);
	chg_drv_start_watchdog_timer(chip);
}

static void chg_drv_start_reduction_timer(struct oem_charger_drv_chip *chip)
{
	unsigned long set_timer;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	if (chip->reduction_motinor_enable == false) {
		chip->reduction_motinor_enable = true;
		chip->reduction_disable_count = 0;
		set_timer = msecs_to_jiffies(CHG_DRV_SET_TIMER_10S);
		queue_delayed_work(chip->chg_drv_wq, &chip->reduction_work, round_jiffies_relative(set_timer));
	}
}

static void chg_drv_stop_reduction_timer(struct oem_charger_drv_chip *chip)
{
	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	if (chip->reduction_motinor_enable == true) {
		cancel_delayed_work(&chip->reduction_work);
		chip->reduction_disable_count = 0;
		chip->reduction_motinor_enable = false;
	}
}

static void chg_drv_reduction_work(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, reduction_work.work);
	union power_supply_propval pval = {0, };
	int current_now;
	int current_avg;
	int ret;
	unsigned long set_timer;

	CHG_DRV_DBGLOG("[%s] reduction disable check.\n", __func__);

	if (!is_client_vote_enabled(chip->fv_votable, OEM_REDUCTION_VOTER))
		goto ret;

	ret = power_supply_get_property(chip->bms_psy,
									POWER_SUPPLY_PROP_CURRENT_NOW, &pval);

	if (ret < 0) {
		CHG_DRV_ERRLOG("[%s] get POWER_SUPPLY_PROP_CURRENT_NOW error %d\n", __func__, ret);
		goto ret;
	}

	current_now = pval.intval;

	pval.intval = SEC_BATTERY_CURRENT_UA;
	ret = power_supply_get_property(chip->bms_psy,
									POWER_SUPPLY_PROP_CURRENT_AVG, &pval);

	if (ret < 0) {
		CHG_DRV_ERRLOG("[%s] get POWER_SUPPLY_PROP_CURRENT_AVG error %d\n", __func__, ret);
		goto ret;
	}

	current_avg = pval.intval;

	if (current_now < CHG_DRV_REDUCTION_MONITOR_CURRENT && current_avg < CHG_DRV_REDUCTION_MONITOR_AVG) {
		chip->reduction_disable_count++;
	} else {
		chip->reduction_disable_count = 0;
	}

	CHG_DRV_DBGLOG("[%s] current:%d avg:%d count:%d\n", __func__, current_now, current_avg, chip->reduction_disable_count);

	if (chip->reduction_disable_count >= CHG_DRV_REDUCTION_MONITOR_COUNT) {
		CHG_DRV_INFOLOG("[%s] reduction disable current:%d avg:%d\n", __func__, current_now, current_avg);
		vote(chip->fv_votable, OEM_REDUCTION_VOTER, false, (chip->dt_param.chg_voltage_reduction / 1000));
	}

ret:
	set_timer = msecs_to_jiffies(CHG_DRV_SET_TIMER_10S);
	queue_delayed_work(chip->chg_drv_wq, &chip->reduction_work, round_jiffies_relative(set_timer));

	return;
}

static void chg_drv_start_usb_removal_wait_timer(struct oem_charger_drv_chip *chip)
{
	unsigned long set_timer;

	if (!delayed_work_pending(&chip->usb_removal_wait_work)) {
		set_timer = msecs_to_jiffies(CHG_DRV_SET_TIMER_2S);
		queue_delayed_work(chip->chg_drv_wq, &chip->usb_removal_wait_work, set_timer);
	}
}

static void chg_drv_stop_usb_removal_wait_timer(struct oem_charger_drv_chip *chip)
{
	cancel_delayed_work(&chip->usb_removal_wait_work);
}

static void chg_drv_usb_removal_wait_work(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, usb_removal_wait_work.work);
	union chg_drv_evt_data evt_data;

	chip->safety_err_info.usb_removal = false;

	if (unlikely(chip != NULL) && unlikely(chip->chg_drv_timer_wake_lock != NULL)) {
		__pm_wakeup_event(chip->chg_drv_timer_wake_lock, (chip->dt_param.wake_lock_timer_sec * HZ));
	}

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));
	evt_data.notice_info.type = CHG_DRV_NOTICE_USBHOT_NOTIFY_ERROR;
	evt_data.notice_info.usb_hot_detect = OEM_CHG_ERROR_NONE;
	chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_INFO_NOTICE, &evt_data);
}

static void chg_drv_start_aicl_monitor(struct oem_charger_drv_chip *chip)
{
	unsigned long set_timer;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	set_timer = msecs_to_jiffies(CHG_DRV_SET_TIMER_300MS);
	queue_delayed_work(chip->chg_drv_wq, &chip->aicl_monitor_work, round_jiffies_relative(set_timer));
}

static void chg_drv_stop_aicl_monitor(struct oem_charger_drv_chip *chip)
{
	int ret;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	chip->aicl_threshold_detect_count = 0;

	ret = vote(chip->fv_votable, OEM_CHG_VOTER, true, (chip->vmax_adj/1000));
	if (unlikely(ret != 0)) {
		CHG_DRV_ERRLOG("fv_votable Failed. ret:[%d]\n", ret);
	}

	cancel_delayed_work(&chip->aicl_monitor_work);
}

static void chg_drv_aicl_monitor_work(struct work_struct *work)
{
	struct oem_charger_drv_chip *chip = container_of(work,
								struct oem_charger_drv_chip, aicl_monitor_work.work);
	union chg_drv_evt_data evt_data;

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	CHG_DRV_DBGLOG("[%s] state=0x%x\n", __func__, chip->charger_info.state);

	evt_data.monitor_info.type = CHG_DRV_MONITOR_AICL;
	chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);
}

static void chg_drv_batt_set_status(struct oem_charger_drv_chip *chip, int status)
{
	union power_supply_propval pval = {0,};

	CHG_DRV_DBGLOG("[%s] in %d\n", __func__, status);

	pval.intval = status;
	power_supply_set_property(chip->oem_battery_psy, POWER_SUPPLY_PROP_STATUS, &pval);

	return;
}

static void chg_drv_batt_set_health(struct oem_charger_drv_chip *chip, int health)
{
	union power_supply_propval pval = {0,};

	CHG_DRV_DBGLOG("[%s] in %d\n", __func__, health);

	pval.intval = health;
	power_supply_set_property(chip->oem_battery_psy, POWER_SUPPLY_PROP_HEALTH, &pval);

	return;
}

static void chg_drv_get_charge_mode(struct oem_charger_drv_chip *chip)
{
	int charge_mode = CHG_DRV_CHARGE_MODE_MANUAL;
	int safety_timer = 0;
	u16 val;
	int ret;

	/*
	 * Limits function classification
	 * check_disable[0] : charge
	 * check_disable[1] : battery present
	 * check_disable[2] : low battery
	 * check_disable[3] : safety timer
	 * check_disable[4] : battery temperature
	 * check_disable[5] : reserve
	 * check_disable[6] : reserve
	 * check_disable[7] : reserve
	 */
	ret = get_nonvolatile((u8 *)&val, APNV_CHARGE_FG_FUNC_LIMITS_I, 2);
	if (likely(ret >= 0)) {
		if ((val & 0x0001) != 0) {
			charge_mode = CHG_DRV_CHARGE_MODE_NOT_CHARGE;
			CHG_DRV_RECLOG("[%s] charge_mode=%d\n", __func__, charge_mode);
		}
		if ((val & 0x0008) != 0) {
			safety_timer = 1;
		}
	} else {
		CHG_DRV_ERRLOG("[%s] NV read err result=%d\n", __func__, ret);
	}

	chip->charge_mode = charge_mode;
	chip->safety_timer_reduce_flag = safety_timer;

	return;
}

static int chg_drv_oem_chg_get_property(struct power_supply *psy,
										enum power_supply_property psp,
										union power_supply_propval *val)
{
	struct oem_charger_drv_chip *chip = power_supply_get_drvdata(psy);
	int rc = 0;

	switch (psp) {
	case POWER_SUPPLY_PROP_CHARGE_CONTROL_LIMIT:
		val->intval = chip->thermal_level;
		break;

	case POWER_SUPPLY_PROP_CHARGE_CONTROL_LIMIT_MAX:
		val->intval = chip->thermal_level_max;
		break;

	default:
		rc = -EINVAL;
	}

	return rc;
}

static int chg_drv_oem_chg_set_prop_chg_ctrl_limit(struct oem_charger_drv_chip *chip)
{
	CHG_DRV_INFOLOG("[%s] set chg_ctrl_limit %d\n",__func__, chip->thermal_level);
	chg_drv_event_req(chip, CHG_DRV_EVENT_PARAM_UPDATE, NULL);

	return 0;
}

static int chg_drv_oem_chg_set_property(struct power_supply *psy,
										enum power_supply_property psp,
										const union power_supply_propval *val)
{
	struct oem_charger_drv_chip *chip = power_supply_get_drvdata(psy);
	int rc = 0;

	switch (psp) {
	case POWER_SUPPLY_PROP_CHARGE_CONTROL_LIMIT:
		if (val->intval < 0 || val->intval > chip->thermal_level_max) {
			rc = -EINVAL;
			break;
		}
		if ((chip->thermal_level != val->intval) &&
			(chip->thermal_level_max > val->intval)) {
			chip->thermal_level = val->intval;
			rc = chg_drv_oem_chg_set_prop_chg_ctrl_limit(chip);
		}
		break;

	default:
		rc = -EINVAL;
	}

	return rc;
}

static int chg_drv_oem_chg_is_writeable(struct power_supply *psy,
										enum power_supply_property psp)
{
	int rc;

	switch (psp) {
	default:
		rc = 0;
		break;
	}

	return rc;
}

static int chg_drv_get_battery_voltage(struct oem_charger_drv_chip *chip)
{
	long volt_work = 0;
	int ave_vol = 0;
	int index_max = 0;
	int ret = 0;
	union power_supply_propval pval = {0,};
	int i;

	ret = power_supply_get_property(chip->battery_psy,
									POWER_SUPPLY_PROP_VOLTAGE_NOW, &pval);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get VOLTAGE_NOW error %d\n", __func__, ret);
	}

	CHG_DRV_DBGLOG("[%s] Batt_Voltage:[%d]\n", __func__, pval.intval);

	chip->batt_volt_data.volt_data[chip->batt_volt_data.index] = pval.intval;
	chip->batt_volt_data.index++;

	if (chip->batt_volt_data.index >= CHG_DRV_BATT_VOLT_INDEX) {
		chip->batt_volt_data.index = 0;
		chip->batt_volt_data.data_full = true;
	}

	if (chip->batt_volt_data.data_full) {
		index_max = CHG_DRV_BATT_VOLT_INDEX;
	} else {
		index_max = chip->batt_volt_data.index;
	}

	for (i = 0; i < index_max; i++) {
		volt_work += chip->batt_volt_data.volt_data[i];
	}
	ave_vol = (int)(volt_work / index_max);

	return ave_vol;
}

static enum chg_drv_result_type chg_drv_watch_usb_adapter_voltage(struct oem_charger_drv_chip *chip,
																	enum chg_drv_err_type *err_type,
																	struct chg_drv_monitor_param *param)
{
#if 0
	enum chg_drv_result_type result = CHG_DRV_RESULT_SUCCESS;
	int i;
	int ov_detect_count;
	int uv_detect_count;

	chg_drv_update_charge_type(chip);

	CHG_DRV_DBGLOG("[%s] chgin_dtls:[0x%x], vbadc:[0x%x][0x%x][0x%x][0x%x]\n",
					__func__, param->chgin_dtls,
					param->vbadc[0], param->vbadc[0], param->vbadc[2], param->vbadc[3]);

	do {
		if (param->chgin_dtls <= 0x01) {
			*err_type = CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR;
			result = CHG_DRV_RESULT_VOLTAGE_ERROR;
			break;
		}

		if (param->chgin_dtls == 0x02) {
			*err_type = CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR;
			result = CHG_DRV_RESULT_VOLTAGE_ERROR;
			break;
		}

		for (i = 0, ov_detect_count = 0, uv_detect_count = 0 ; i < CHG_DRV_ADP_VOLT_CHK_CNT; i++) {
			if (param->vbadc[i] >= 0x09) {
				ov_detect_count++;
			}
			if (param->vbadc[i] == 0x00) {
				uv_detect_count++;
			}
		}
		if (ov_detect_count == CHG_DRV_ADP_VOLT_CHK_CNT) {
			*err_type = CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR;
			result = CHG_DRV_RESULT_VOLTAGE_ERROR;
		} else if (uv_detect_count == CHG_DRV_ADP_VOLT_CHK_CNT) {
			*err_type = CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR;
			result = CHG_DRV_RESULT_VOLTAGE_ERROR;
		}
	} while (0);

	return result;
#else
	chg_drv_update_charge_type(chip);

	CHG_DRV_DBGLOG("[%s] chgin_dtls:[0x%x], vbadc:[0x%x][0x%x][0x%x][0x%x]\n",
					__func__, param->chgin_dtls,
					param->vbadc[0], param->vbadc[0], param->vbadc[2], param->vbadc[3]);

	return CHG_DRV_RESULT_SUCCESS;
#endif
}

static void chg_drv_check_aicl_threshold(struct oem_charger_drv_chip *chip)
{
	int ret;
	int aicl_threshold = CHG_DRV_AICL_THRESHOLD_4P5;
	int batt_volt_uv;
	u8 sts_work = 0;
	u8 chg_dtls = 0;

	batt_volt_uv = chg_drv_get_battery_voltage(chip);

	ret = chg_drv_reg_read(chip, MAX77729_CHG_REG_DETAILS_01, &sts_work);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get CHG_REG_DETAILS_01 error %d\n", __func__, ret);
		sts_work = 0;
	}
	chg_dtls = (sts_work & GENMASK(3, 0));

	if (batt_volt_uv < 4200000) {
		aicl_threshold = CHG_DRV_AICL_THRESHOLD_4P5;
	}
	if ((batt_volt_uv >= 4200000) &&
		(batt_volt_uv <  4300000)) {
		aicl_threshold = CHG_DRV_AICL_THRESHOLD_4P6;
	}
	if ((batt_volt_uv >= 4300000) &&
		(batt_volt_uv <  4500000)) {
		aicl_threshold = CHG_DRV_AICL_THRESHOLD_4P7;
	}

	if ((chg_dtls == 0x08) &&
		(chip->aicl_threshold_detect_count < CHG_DRV_CHECK_AICL_DETECT_CNT)) {
		chip->aicl_threshold_detect_count++;
		CHG_DRV_INFOLOG("[%s] Detect charge stop, count[%d]\n",
					__func__, chip->aicl_threshold_detect_count);
	}
	if (chip->aicl_threshold_detect_count >= CHG_DRV_CHECK_AICL_DETECT_CNT) {
		ret = vote(chip->fv_votable, OEM_CHG_VOTER, true, (chip->dt_param.chg_voltage_aicl_threshold/1000));
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("fv_votable Failed. ret:[%d]\n", ret);
		}
	}

	if (chip->aicl_threshold < aicl_threshold) {
		ret = chg_drv_reg_write_byte(chip, MAX77729_CHG_REG_CNFG_12,
										GENMASK(4, 3), (aicl_threshold));
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("CHG_REG_CNFG_12 Failed. ret:[%d]\n", ret);
		} else {
			CHG_DRV_INFOLOG("[%s] Updated aicl_threshold:[%d]->[%d]\n",
					__func__, (chip->aicl_threshold>>3), (aicl_threshold>>3));
			chip->aicl_threshold = aicl_threshold;
		}
	}

	return;
}

static void chg_drv_update_safety_err_info(struct oem_charger_drv_chip *chip, struct chg_drv_notice_info *notice_info)
{
	u8 cc_flag = 0x00;
	u8 water_disable_flag = 0x00;

	if (notice_info->type == CHG_DRV_NOTICE_OVP_NOTIFY_ERROR) {
		if (notice_info->ovp_detect == OEM_CHG_ERROR_NONE){
			chip->safety_err_info.ovp_detect = false;
		} else {
			chip->safety_err_info.ovp_detect = true;
		}
	}

	if (notice_info->type == CHG_DRV_NOTICE_USBHOT_NOTIFY_ERROR) {
		if (notice_info->usb_hot_detect == OEM_CHG_ERROR_NONE){
			CHG_DRV_RECLOG("[%s] set CC pin default\n", __func__);
			water_disable_flag = 0x00;
			cc_flag = 0x00;
			max77729_set_ccx_open_set_reg(cc_flag);
			max77729_set_water_disable_reg(water_disable_flag);
			if (chip->safety_err_info.usb_port_failure == true) {
				chip->safety_err_info.usb_removal = true;
				chg_drv_start_usb_removal_wait_timer(chip);
			}
			chip->safety_err_info.usb_port_failure = false;
		} else {
			CHG_DRV_RECLOG("[%s] set CC pin open\n", __func__);
			water_disable_flag = 0x01;
			cc_flag = 0x01;
			max77729_set_water_disable_reg(water_disable_flag);
			max77729_set_ccx_open_set_reg(cc_flag);
			chg_drv_stop_usb_removal_wait_timer(chip);
			chip->safety_err_info.usb_port_failure = true;
		}
	}

#ifdef CHG_DRV_CONFIG_PD_ENABLE
	if (notice_info->type == CHG_DRV_NOTICE_PDO_ERROR) {
		if (notice_info->pdo_error_detect == OEM_CHG_ERROR_NONE){
			chip->safety_err_info.pdo_profile_error = false;
		} else {
			chip->safety_err_info.pdo_profile_error = true;
		}
	}
#endif
}

static void chg_drv_get_parameters(struct oem_charger_drv_chip *chip,
									struct chg_drv_monitor_param *charger_param)
{
	union power_supply_propval pval = {0,};
	int ret = 0;
	int i;
	u8 sts_work = 0;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	/* Supply Voltage */
	for (i = 0; i < CHG_DRV_ADP_VOLT_CHK_CNT; i++) {
		if (i != 0) msleep(25);
		ret = power_supply_get_property(chip->usb_psy, POWER_SUPPLY_PROP_INPUT_VOLTAGE_LIMIT, &pval);
		if (unlikely(ret < 0)) {
			CHG_DRV_ERRLOG("[%s] get INPUT_VOLTAGE_LIMIT error %d\n", __func__, ret);
			pval.intval = 0;
		}
		charger_param->vbadc[i] = pval.intval;
	}
	ret = power_supply_get_property(chip->usb_psy, POWER_SUPPLY_PROP_VOLTAGE_NOW, &pval);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get VOLTAGE_NOW error %d\n", __func__, ret);
		pval.intval = 0;
	}
	charger_param->usbin_uv = pval.intval;

	/* Battery current */
	ret = power_supply_get_property(chip->bms_psy,
							POWER_SUPPLY_PROP_CURRENT_NOW, &pval);
	if (likely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get CURRENT_NOW error %d\n", __func__, ret);
		pval.intval = 0;
	}
	charger_param->fcc_ua = pval.intval;

	/* Battery Voltage */
	charger_param->batt_volt_uv = chg_drv_get_battery_voltage(chip);

	/* Battery Temp */
	ret = power_supply_get_property(chip->battery_psy, POWER_SUPPLY_PROP_TEMP, &pval);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get BATT_TEMP error %d\n", __func__, ret);
		pval.intval = 0;
	}
	charger_param->batt_temp = pval.intval;

	/* Status of Charge */
	ret = power_supply_get_property(chip->oem_battery_psy,
									(enum power_supply_property)POWER_SUPPLY_EXT_PROP_CHG_CTRL_SOC, &pval);
	if (ret < 0) {
		CHG_DRV_ERRLOG("[%s] get BATT_CAPACITY error %d\n", __func__, ret);
		pval.intval = 0;
	}
	charger_param->soc = pval.intval;

	/* charger ic status */
	ret = chg_drv_reg_read(chip, MAX77729_CHG_REG_DETAILS_00, &sts_work);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get CHG_REG_DETAILS_00 error %d\n", __func__, ret);
		sts_work = 0;
	}
	charger_param->chgin_dtls = ((sts_work & GENMASK(6, 5)) >> 5);

	ret = chg_drv_reg_read(chip, MAX77729_CHG_REG_DETAILS_01, &sts_work);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get CHG_REG_DETAILS_01 error %d\n", __func__, ret);
		sts_work = 0;
	}
	charger_param->chg_dtls = (sts_work & GENMASK(3, 0));
	charger_param->bat_dtls = ((sts_work & GENMASK(6, 4)) >> 4);

	ret = max77729_get_cc_sts0_reg(&sts_work);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get CC_STATUS0 error %d\n", __func__, ret);
		charger_param->cc_sts = 0;
	} else {
		charger_param->cc_sts = sts_work;
	}

	/* FGIC initial status */
	ret = power_supply_get_property(chip->oem_battery_psy, POWER_SUPPLY_PROP_PRESENT, &pval);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get FGIC_INITIAL_STATUS error %d\n", __func__, ret);
		pval.intval = 0;
	}
	charger_param->fgic_initial_sts = pval.intval;

	/* external factors */
	charger_param->effective_icl = get_effective_result(chip->usb_icl_votable);
	charger_param->effective_fcc = get_effective_result(chip->fcc_votable);

	/* usb phy */
	charger_param->usb_phy_icl = get_client_vote(chip->usb_icl_votable, USB_PSY_VOTER);
}

static enum chg_drv_err_type chg_drv_check_parameters(struct oem_charger_drv_chip *chip,
														struct chg_drv_monitor_param *param)
{
	enum chg_drv_err_type err_type = CHG_DRV_ERR_NON;
	unsigned int current_err = CHG_DRV_PARAM_NO_ERROR;
	enum chg_drv_err_type usb_err_type = CHG_DRV_ERR_NON;
	enum chg_drv_result_type usb_result = CHG_DRV_RESULT_SUCCESS;
	bool batt_temp_warm_flag = false;
	bool batt_temp_cool_flag = false;
	bool batt_low_flag = false;
	bool update_flag = false;
	int batt_low_volt_threshold;

	#define CHG_DRV_SET_ERR_TYPE(type, set)	if (type == CHG_DRV_ERR_NON) type = set;

	CHG_DRV_DBGLOG("[%s] USBIN:[%d] BATT_TEMP:[%d]\n",
				__func__, param->usbin_uv, param->batt_temp);

	/* Battery voltage check */
	if (((chip->dt_param.batt_over_voltage > 0) &&
		 (param->batt_volt_uv >= chip->dt_param.batt_over_voltage)) ||
		((chip->charger_info.current_err & CHG_DRV_PARAM_BATT_VOLTAGE_OV) == CHG_DRV_PARAM_BATT_VOLTAGE_OV)) {
		CHG_DRV_ERRLOG("[%s] [CHGDRVERR] BATT_VOLTAGE_OV(%d)\n", __func__, param->batt_volt_uv);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_OVER_BATT_VOLTAGE);
		current_err |= CHG_DRV_PARAM_BATT_VOLTAGE_OV;
	}

	if (charging_mode) {
		batt_low_volt_threshold = chip->dt_param.batt_over_voltage_offchg;
	} else {
		batt_low_volt_threshold = chip->dt_param.batt_low_voltage_onchg;
	}

	if (param->batt_volt_uv < batt_low_volt_threshold) {
		batt_low_flag = true;
	}
	if (chip->battery_low_flag != batt_low_flag) {
		chip->battery_low_flag = batt_low_flag;
		update_flag = true;
	}
	/* usb port failure check */
	if (chip->safety_err_info.usb_port_failure ||
		chip->safety_err_info.usb_removal ||
		((chip->charger_info.current_err & CHG_DRV_PARAM_USB_PORT_FAILURE) == CHG_DRV_PARAM_USB_PORT_FAILURE)) {
		CHG_DRV_ERRLOG("[%s] [CHGDRVERR] CHG_DRV_PARAM_USB_PORT_FAILURE\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_USB_PORT_FAILURE);
		current_err |= CHG_DRV_PARAM_USB_PORT_FAILURE;
	}
	/* safe timer expire check */
	if (chip->err_info.safety_timer_expire) {
		CHG_DRV_ERRLOG("[%s] [CHGDRVERR] CHG_DRV_PARAM_UNSPEC_FAILURE\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_EXPIRE_TIME_OUT);
		current_err |= CHG_DRV_PARAM_UNSPEC_FAILURE;
	}
	/* ovp detect check */
	if (chip->safety_err_info.ovp_detect) {
		CHG_DRV_ERRLOG("[%s] [CHGDRVERR] CHG_DRV_PARAM_OVP_DETECT_ERROR\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_OVP_DETECT);
		current_err |= CHG_DRV_PARAM_OVP_DETECT_ERROR;
	}

	/* Supply voltage check */
	if ((chip->charger_info.state == CHG_DRV_STATE_IDLE) ||
		(chip->charger_info.state == CHG_DRV_STATE_ERROR)) {
		if ((chip->charger_info.chg_source & CHG_DRV_SOURCE_USB_PORT) != 0) {
			if (chip->charge_type != CHG_DRV_CHARGE_TYPE_UNKNOWN) {
				usb_result = chg_drv_watch_usb_adapter_voltage(chip, &usb_err_type, param);

				if (usb_err_type == CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR) {
					CHG_DRV_ERRLOG("[%s] [CHGDRVERR] OVER SUPPLY VOLTAGE.\n", __func__);
					CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR);
					current_err |= CHG_DRV_PARAM_ADAPTER_VOLTAGE_OV;
				}
				if (usb_err_type == CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR) {
					CHG_DRV_ERRLOG("[%s] [CHGDRVERR] UNDER SUPPLY VOLTAGE.\n", __func__);
					CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR);
					current_err |= CHG_DRV_PARAM_ADAPTER_VOLTAGE_UV;
				}
			}
		}
	}

	/* charge disable check */
	if (chip->max_icl_value == CHG_DRV_ICL_DISABLE) {
		CHG_DRV_ERRLOG("[%s] CHG_DRV_PARAM_CHG_DISABLE\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_CHARGE_DISABLE);
		current_err |= CHG_DRV_PARAM_CHG_DISABLE;
	}

	if (chip->dt_param.chg_icl_thermal_mitigation[chip->thermal_level] == CHG_DRV_ICL_DISABLE) {
		CHG_DRV_ERRLOG("[%s] CHG_DRV_PARAM_CHG_DISABLE\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_CHARGE_DISABLE);
		current_err |= CHG_DRV_PARAM_CHG_DISABLE;
	}

	/* Battery temp cold check */
	if (param->batt_temp <= chip->batt_temp_cold_limit) {
		CHG_DRV_ERRLOG("[%s] BATT_TEMP_COLD(%d)\n", __func__, param->batt_temp);
		current_err |= CHG_DRV_PARAM_BATT_TEMP_COLD;
		batt_temp_cool_flag = true;
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_BATT_TEMP_COLD_ERR);
		chip->batt_temp_cold_limit = chip->dt_param.batt_temp_cold_limit + chip->dt_param.batt_temp_limit_cancel;
	} else {
		chip->batt_temp_cold_limit = chip->dt_param.batt_temp_cold_limit;
	}
	/* Battery temp hot check */
	if (param->batt_temp >= chip->batt_temp_hot_limit) {
		CHG_DRV_ERRLOG("[%s] temp_hot BATT_TEMP(%d)\n", __func__, param->batt_temp);
		current_err |= CHG_DRV_PARAM_BATT_TEMP_HOT;
		batt_temp_warm_flag = true;
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_BATT_TEMP_HOT_ERR);
		if (chip->batt_temp_hot_limit == chip->dt_param.batt_temp_start_limit) {
			chip->batt_temp_hot_limit = chip->dt_param.batt_temp_start_limit - chip->dt_param.batt_temp_limit_cancel;
		} else if (chip->batt_temp_hot_limit == chip->dt_param.batt_temp_hot_limit) {
			chip->batt_temp_hot_limit = chip->dt_param.batt_temp_hot_limit - chip->dt_param.batt_temp_limit_cancel;
		}
	} else {
		chip->batt_temp_hot_limit = chip->dt_param.batt_temp_hot_limit;
	}
	/* Battery temp warm check */
	if (param->batt_temp >= chip->batt_temp_warm_limit) {
		CHG_DRV_WARNLOG("[%s] BATT_TEMP_WARM(%d)\n", __func__, param->batt_temp);
		current_err |= CHG_DRV_PARAM_BATT_TEMP_WARM;
		batt_temp_warm_flag = true;
		chip->batt_temp_warm_limit = chip->dt_param.batt_temp_warm_limit - chip->dt_param.batt_temp_limit_cancel;
	} else {
		chip->batt_temp_warm_limit = chip->dt_param.batt_temp_warm_limit;
	}
	if (batt_temp_warm_flag != chip->err_info.batt_temp_warm_restrict) {
		chip->err_info.batt_temp_warm_restrict = batt_temp_warm_flag;
		update_flag = true;
	}
	/* Battery temp cool check */
	if (param->batt_temp <= chip->batt_temp_cool_limit) {
		CHG_DRV_WARNLOG("[%s] BATT_TEMP_COOL(%d)\n", __func__, param->batt_temp);
		current_err |= CHG_DRV_PARAM_BATT_TEMP_COOL;
		batt_temp_cool_flag = true;
		chip->batt_temp_cool_limit = chip->dt_param.batt_temp_cool_limit + chip->dt_param.batt_temp_limit_cancel;
	} else {
		chip->batt_temp_cool_limit = chip->dt_param.batt_temp_cool_limit;
	}
	if (batt_temp_cool_flag != chip->err_info.batt_temp_cool_restrict) {
		chip->err_info.batt_temp_cool_restrict = batt_temp_cool_flag;
		update_flag = true;
	}
	/* heat charge mode check */
	if (chip->heat_charge_mode == OEM_CHG_CHARGE_MODE_POWERPATH) {
		CHG_DRV_WARNLOG("[%s] HEAT_CHARGE_MODE [%d]\n",
												__func__, chip->heat_charge_mode);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_HEAT_CHARGE);
	}

	/* fgic initializing check */
	if (!param->fgic_initial_sts) {
		CHG_DRV_ERRLOG("[%s] FGIC INITIALIZING.\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_FGIC_INITIALIZING);
		current_err |= CHG_DRV_PARAM_FGIC_INITIALIZING;
	}

	/* Watch Dog Timer expire check */
	if ((param->chg_dtls == 0x0B) ||
		((chip->charger_info.current_err & CHG_DRV_PARAM_WDOG_TIMEOUT) == CHG_DRV_PARAM_WDOG_TIMEOUT)) {
		CHG_DRV_ERRLOG("[%s] [CHGDRVERR] WDOG TIMEOUT.\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_OTHER_ERR);
		current_err |= CHG_DRV_PARAM_WDOG_TIMEOUT;
	}

#ifdef CHG_DRV_CONFIG_PD_ENABLE
	/* PDO profile error check */
	if (chip->safety_err_info.pdo_profile_error) {
		CHG_DRV_ERRLOG("[%s] [CHGDRVERR] PDO PROFILE ERROR.\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_OTHER_ERR);
		current_err |= CHG_DRV_PARAM_PDO_PROFILE_ERROR;
	}
#endif

	if (chip->err_info.unspec_cable_err) {
		CHG_DRV_ERRLOG("[%s] [CHGDRVERR] UNSPEC CABLE.\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_UNSPEC_CABLE);
		current_err |= CHG_DRV_PARAM_UNSPEC_CABLE;
	}

	if (param->usb_phy_icl == 2) {
		CHG_DRV_ERRLOG("[%s] [CHGDRVERR] USB SUSPEND.\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_USB_SUSPEND);
		current_err |= CHG_DRV_PARAM_USB_SUSPEND;
	}

	if (chip->schedule_chg) {
		CHG_DRV_DBGLOG("[%s] SCHEDULE CHARGE.\n", __func__);
		CHG_DRV_SET_ERR_TYPE(err_type, CHG_DRV_ERR_SCHEDULE_CHARGE);
		current_err |= CHG_DRV_PARAM_SCHEDULE_CHARGE;
	}

	if (((strcmp(get_effective_client(chip->usb_icl_votable),OEM_CHG_VOTER)) &&
		 (param->effective_icl < 100)) ||
		((strcmp(get_effective_client(chip->fcc_votable),OEM_CHG_VOTER)) &&
		 (param->effective_fcc < 100))) {
			/* Log output only, no error occurs */
			CHG_DRV_INFOLOG("[%s] [CHGDRVERR] Charging stopped due to external factors. icl:%d, fcc:%d\n",
								 __func__, param->effective_icl, param->effective_fcc);
	}

	/* update check */
	if (update_flag) {
		chg_drv_event_req(chip, CHG_DRV_EVENT_PARAM_UPDATE, NULL);
	}

	if (chip->charger_info.current_err != current_err) {
		CHG_DRV_RECLOG("[%s] error sts update [0x%08x]->[0x%08x] err_type[%d]\n",
					__func__, chip->charger_info.current_err, current_err, err_type);
	}

	chip->charger_info.current_err = current_err;

	return err_type;
}

static bool chg_drv_is_battery_full(struct oem_charger_drv_chip *chip, struct chg_drv_monitor_param *param)
{
	bool batt_full = false;
	union power_supply_propval pval = {0,};

	if (param->chg_dtls == 0x04) {
		batt_full = true;
		pval.intval = 0;
		power_supply_set_property(chip->oem_battery_psy, (enum power_supply_property)POWER_SUPPLY_EXT_PROP_LONGEVITY_CYCLE, &pval);
	}

	return batt_full;
}

static void chg_drv_get_apsd_parameters(struct oem_charger_drv_chip *chip, int *apsd_result, unsigned int *max_current)
{
	union power_supply_propval pval = {0,};
	int ret = 0;

	ret = power_supply_get_property(chip->usb_psy, (enum power_supply_property)POWER_SUPPLY_EXT_PROP_REAL_TYPE, &pval);

	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get REAL_TYPE error %d\n", __func__, ret);
		pval.intval = POWER_SUPPLY_TYPE_UNKNOWN;
	}

	switch(pval.intval) {
		case POWER_SUPPLY_TYPE_USB:
			*apsd_result = CHG_DRV_SOURCE_USB;
			*max_current = CHG_DRV_CURRENT_500;
			break;
		case POWER_SUPPLY_TYPE_USB_CDP:
		case POWER_SUPPLY_TYPE_USB_HVDCP:
		case POWER_SUPPLY_TYPE_USB_DCP:
			*apsd_result = CHG_DRV_SOURCE_AC;
			*max_current = CHG_DRV_CURRENT_1500;
			break;
		case POWER_SUPPLY_TYPE_USB_PD:
			*apsd_result = CHG_DRV_SOURCE_AC;
			*max_current = CHG_DRV_CURRENT_1800;
			break;
		case POWER_SUPPLY_TYPE_USB_FLOAT:
		case POWER_SUPPLY_TYPE_UNKNOWN:
		default:
			*apsd_result = CHG_DRV_SOURCE_AC;
			*max_current = CHG_DRV_CURRENT_500;
			break;
	}
}

static void chg_drv_check_apsd_result(struct oem_charger_drv_chip *chip)
{
	int apsd_result = 0;
	unsigned int max_current = 0;
	struct chg_drv_device_info *dev_info = &chip->dev_usb;
	struct max77729_charger_data *charger = chip->charger;

	chg_drv_get_apsd_parameters(chip, &apsd_result, &max_current);

	if ((charger_apsd_src != apsd_result) ||
		(dev_info->max_current != max_current)) {
			charger_apsd_src = apsd_result;
			dev_info->max_current = max_current;
			chg_drv_event_req(chip, CHG_DRV_EVENT_PARAM_UPDATE, NULL);
			CHG_DRV_RECLOG("[%s] APSD:%d, mA=%d\n",
								__func__, charger->real_type, dev_info->max_current);

	}
}

static void chg_drv_terminate_usb(struct oem_charger_drv_chip *chip)
{
	chip->dev_usb.max_current = CHG_DRV_CURRENT_OFF;
}

static void chg_drv_terminate(struct oem_charger_drv_chip *chip, bool shutdown)
{
	int ret = 0;

	chg_drv_stop_monitor(chip);
	chg_drv_stop_adapter_volt_monitor(chip);
	chg_drv_stop_full_timer(chip);
	chg_drv_stop_safety_timer(chip);
	chg_drv_stop_aicl_monitor(chip);
	chg_drv_stop_watchdog_timer(chip);
	chg_drv_stop_reduction_timer(chip);

	chg_drv_terminate_usb(chip);

	if (!shutdown) {
		if ((chip->safety_err_info.usb_port_failure == true) ||
			(chip->safety_err_info.usb_removal == true)) {
			chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_OVERCURRENT);
			chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_NOT_CHARGING);
		} else if (chip->safety_err_info.ovp_detect == true) {
			chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_UNSPEC_FAILURE);
			chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_NOT_CHARGING);
#ifdef CHG_DRV_CONFIG_PD_ENABLE
		} else if (chip->safety_err_info.pdo_profile_error == true) {
			chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_WATCHDOG_TIMER_EXPIRE);
			chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_NOT_CHARGING);
#endif
		} else {
			chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_GOOD);
			chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_DISCHARGING);
		}
	}

	ret = chg_drv_update_cycle_charge_setting(chip);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] update_cycle_charge_setting failed. ret = %d\n", __func__, ret);
	}

	chg_drv_pmic_initial_setting(chip);

	chip->charger_info.chg_source  = 0;
	chip->charger_info.chg_current = 0;
	chip->charger_info.current_err = CHG_DRV_PARAM_NO_ERROR;

	chip->connect_state = CHG_DRV_CONNECT_STATE_NONE;
	chip->charge_type = CHG_DRV_CHARGE_TYPE_NORMAL;
	chip->consecutive_count = 0;
	chip->battery_low_flag = false;
	chip->aicl_threshold = CHG_DRV_AICL_THRESHOLD_4P5;
	chip->schedule_chg = 0;
	chip->aicl_threshold_detect_count = 0;

	memset(&chip->err_info, 0, sizeof(struct chg_drv_err_info));

	chip->batt_temp_hot_limit = chip->dt_param.batt_temp_start_limit;
	chip->batt_temp_warm_limit = chip->dt_param.batt_temp_warm_limit;
	chip->batt_temp_cold_limit = chip->dt_param.batt_temp_cold_limit;
	chip->batt_temp_cool_limit = chip->dt_param.batt_temp_cool_limit;

	chip->charge_ctrl = CHG_DRV_CONTROL_NONE;

	memset(chip->batt_volt_data.volt_data, 0, sizeof(chip->batt_volt_data.volt_data));
	chip->batt_volt_data.index = 0;
	chip->batt_volt_data.data_full = false;

	vote(chip->fv_votable, OEM_REDUCTION_VOTER, chip->chg_voltage_reduction_flag, (chip->dt_param.chg_voltage_reduction / 1000));
}

static int chg_drv_charge_control(struct oem_charger_drv_chip *chip,
									enum chg_drv_ctrl_type charge_ctrl)
{
	int ret = 0;
	u8 mode_val = 0;

	CHG_DRV_DBGLOG("[%s] in charge_ctrl = %d\n", __func__, charge_ctrl);

	ret = chg_drv_reg_read(chip, MAX77729_CHG_REG_CNFG_00, &mode_val);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get MAX77729_CHG_REG_CNFG_00 error %d\n", __func__, ret);
		mode_val = 0;
	}
	mode_val &= CHG_CNFG_00_MODE_MASK;

	if (chip->charge_ctrl == charge_ctrl && chip->mode_val == mode_val) {
		return ret;
	}

	switch (charge_ctrl) {
	case CHG_DRV_CONTROL_PRECHARGE:
		ret = chg_drv_pmic_charge_start(chip);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("[%s] PreCharge Failed. ret:[%d]\n", __func__, ret);
		}
		break;

	case CHG_DRV_CONTROL_RECHARGE:
	case CHG_DRV_CONTROL_CHARGE_START:
		ret = chg_drv_pmic_charge_start(chip);
		if (likely(ret == 0)) {
			if (chip->charge_type == CHG_DRV_CHARGE_TYPE_PD) {
				ret = chg_drv_pmic_pd_charge(chip);
			} else {
				ret = chg_drv_pmic_normal_charge(chip);
			}
		}
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("[%s] Charge Start Failed. ret:[%d]\n", __func__, ret);
		}
		break;

	case CHG_DRV_CONTROL_POWERPATH:
		ret = chg_drv_pmic_charge_powerpath(chip);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("[%s] PowerPath Failed. ret:[%d]\n", __func__, ret);
		}
		break;

	case CHG_DRV_CONTROL_STOP:
		ret = chg_drv_pmic_charge_stop(chip);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("[%s] Charge Stop Failed. ret:[%d]\n", __func__, ret);
		}
		break;

	default:
		ret = -EINVAL;
		break;
	}

	chip->charge_ctrl = charge_ctrl;
	ret = chg_drv_reg_read(chip, MAX77729_CHG_REG_CNFG_00, &mode_val);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get MAX77729_CHG_REG_CNFG_00 error %d\n", __func__, ret);
		mode_val = 0;
	}
	mode_val &= CHG_CNFG_00_MODE_MASK;
	chip->mode_val = mode_val;

	return ret;
}

static void chg_drv_err_control(struct oem_charger_drv_chip *chip, enum chg_drv_err_type err_type)
{
	struct chg_drv_err_ctrl err_ctrl = {0};

	CHG_DRV_DBGLOG("[%s] in err_type = %d\n", __func__, err_type);

	if (err_type >= CHG_DRV_ERR_NUM) {
		return;
	}

	chg_drv_stop_safety_timer(chip);
	chg_drv_stop_aicl_monitor(chip);
	chg_drv_stop_full_timer(chip);
	chg_drv_stop_reduction_timer(chip);

	memcpy(&err_ctrl, &err_ctrl_table[err_type], sizeof(struct chg_drv_err_ctrl));

	if (charging_mode && (err_type!=CHG_DRV_ERR_FGIC_INITIALIZING)) {
		if (err_ctrl.charge_ctrl == CHG_DRV_CONTROL_POWERPATH) {
			err_ctrl.charge_ctrl = CHG_DRV_CONTROL_STOP;
		}
		if ((err_ctrl.charge_ctrl == CHG_DRV_CONTROL_STOP)		   &&
			(err_ctrl.set_status == POWER_SUPPLY_STATUS_CHARGING)) {
			err_ctrl.set_status = POWER_SUPPLY_STATUS_NOT_CHARGING;
		}
	}

	chg_drv_batt_set_health(chip, err_ctrl.set_health);
	chg_drv_batt_set_status(chip, err_ctrl.set_status);
	chg_drv_charge_control(chip, err_ctrl.charge_ctrl);
}

static void chg_drv_charge_enable_control(struct oem_charger_drv_chip *chip, int icl)
{
	union chg_drv_evt_data evt_data;

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	if (chip->max_icl_value != icl) {
		CHG_DRV_DBGLOG("[%s] in chg_en:[%d]->[%d]\n",
									__func__, chip->max_icl_value, icl);
		if (icl != CHG_DRV_ICL_DISABLE) {
			chg_drv_event_req(chip, CHG_DRV_EVENT_PARAM_UPDATE, NULL);
		}
		if ((chip->max_icl_value == CHG_DRV_ICL_DISABLE) ||
			(icl == CHG_DRV_ICL_DISABLE)) {
			chg_drv_stop_monitor(chip);
			evt_data.monitor_info.type = CHG_DRV_MONITOR_NORMAL;
			chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);
		}
		chip->max_icl_value = icl;
	}
}

static void chg_drv_notify_error_control(struct oem_charger_drv_chip *chip, struct chg_drv_notice_info *notice_info)
{
	union chg_drv_evt_data evt_data;

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	chg_drv_stop_monitor(chip);

	chg_drv_update_safety_err_info(chip, notice_info);

	evt_data.monitor_info.type = CHG_DRV_MONITOR_NORMAL;
	chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);
}

static void chg_drv_change_current_control(struct oem_charger_drv_chip *chip,
											struct chg_drv_change_info *change_info)
{
	struct chg_drv_device_info *dev_info = &chip->dev_usb;

	dev_info->max_current = change_info->max_current;
	chg_drv_event_req(chip, CHG_DRV_EVENT_PARAM_UPDATE, NULL);
}

static void chg_drv_power_supply_update_all(struct oem_charger_drv_chip *chip)
{
	if (likely(chip != NULL)) {
		CHG_DRV_RECLOG("[%s] update online chg_src=0x%04x\n",
						__func__, (unsigned int)chip->charger_info.chg_source);
		power_supply_changed(chip->oem_chg_psy);
	}
}

static const struct chg_drv_connect_ctrl connect_ctrl_table[CHG_DRV_CONNECT_STATE_NUM][CHG_DRV_CONNECT_EVENT_NUM] = {
	/* CHG_DRV_CONNECT_STATE_NONE */ 			{
		/* CHG_DRV_CONNECT_EVENT_USB_IN		*/		{ CHG_DRV_CONNECT_STATE_USB,		CHG_DRV_CONTROL_USB_CHARGE		},
		/* CHG_DRV_CONNECT_EVENT_USB_OUT	*/		{ CHG_DRV_CONNECT_STATE_NOCHANGE,	CHG_DRV_CONTROL_NONE			},
	/* CHG_DRV_CONNECT_STATE_USB */				} , {
		/* CHG_DRV_CONNECT_EVENT_USB_IN		*/		{ CHG_DRV_CONNECT_STATE_NOCHANGE,	CHG_DRV_CONTROL_USB_CHARGE		},
		/* CHG_DRV_CONNECT_EVENT_USB_OUT	*/		{ CHG_DRV_CONNECT_STATE_NONE,		CHG_DRV_CONTROL_STOP			},
												}
};

static enum chg_drv_ctrl_type chg_drv_connect_info_update(struct oem_charger_drv_chip *chip,
															struct chg_drv_connect_info *connect_info)
{
	const struct chg_drv_connect_ctrl *connect_ctrl;
	unsigned long new_source = 0;
	int ret;
	u8 val;
	int apsd_result = 0;
	unsigned int max_current = 0;
	struct max77729_charger_data *charger = chip->charger;

	new_source = CHG_DRV_SOURCE(connect_info->chg_source);

	if (connect_info->max_current != CHG_DRV_CURRENT_OFF) {
		ret = chg_drv_reg_read(chip, MAX77729_CHG_REG_INT_OK, &val);
		if (unlikely(ret < 0)) {
			CHG_DRV_ERRLOG("[%s] Get CHG_REG_INT_OK Failed. ret:[%d]\n", __func__, ret);
		} else if ((val & BIT(6)) == 0) {
			connect_info->event = CHG_DRV_CONNECT_EVENT_USB_OUT;
			charger_src &= ~new_source;
			if (new_source == CHG_DRV_SOURCE_APSD) {
				charger_apsd_src = 0;
			}
		} else {
			if ((new_source & CHG_DRV_SOURCE_USB_PORT) != 0) {
				charger_src &= ~CHG_DRV_SOURCE_USB_PORT;
			}
			charger_src |= new_source;
			if (new_source == CHG_DRV_SOURCE_APSD) {
				chg_drv_get_apsd_parameters(chip, &apsd_result, &max_current);
				charger_apsd_src = apsd_result;
				connect_info->max_current = max_current;
				CHG_DRV_RECLOG("[%s] APSD:%d, mA=%d\n",
									__func__, charger->real_type, connect_info->max_current);

			}
		}
	} else {
		charger_src &= ~new_source;
		if (new_source == CHG_DRV_SOURCE_APSD) {
				charger_apsd_src = 0;
		}
	}
	chip->charger_info.chg_source = charger_src;

	connect_ctrl = &connect_ctrl_table[chip->connect_state][connect_info->event];

	switch(connect_info->event) {
	case CHG_DRV_CONNECT_EVENT_USB_IN:
		ret = chg_drv_get_longevity_chg_mode(chip);
		if (unlikely(ret < 0)) {
			CHG_DRV_ERRLOG("[%s] get_longevity_charge_mode failed. ret = %d\n", __func__, ret);
		}
		chip->dev_usb.max_current = connect_info->max_current;
		break;

	case CHG_DRV_CONNECT_EVENT_USB_OUT:
		chg_drv_terminate_usb(chip);
		chg_drv_pmic_slight_charge(chip, CHG_DRV_DEVICE_TYPE_USB);
		break;

	default:
		break;
	}

	switch(connect_ctrl->next_state) {
	case CHG_DRV_CONNECT_STATE_USB:
		chip->charger_info.chg_current = chip->dev_usb.max_current;
		chip->charge_type = CHG_DRV_CHARGE_TYPE_NORMAL;
		break;

	case CHG_DRV_CONNECT_STATE_NONE:
	default:
		break;
	}

	if (connect_ctrl->next_state != CHG_DRV_CONNECT_STATE_NOCHANGE) {
		if (chip->connect_state != connect_ctrl->next_state) {
			CHG_DRV_INFOLOG("[%s] charger connect state %d -> %d\n",
							__func__, chip->connect_state, connect_ctrl->next_state);
			chip->connect_state = connect_ctrl->next_state;
		}
	}

	if (connect_ctrl->charge_ctrl != CHG_DRV_CONTROL_NONE) {
		chg_drv_power_supply_update_all(chip);
	}

	return connect_ctrl->charge_ctrl;
}

static enum chg_drv_state_type chg_drv_func_charging_monitor(struct oem_charger_drv_chip *chip,
																union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;
	enum chg_drv_err_type err_type = CHG_DRV_ERR_NON;
	struct chg_drv_monitor_param param = {0};
	enum chg_drv_err_type usb_err_type = CHG_DRV_ERR_NON;
	enum chg_drv_result_type usb_result = CHG_DRV_RESULT_SUCCESS;
	union chg_drv_evt_data evt_data;

	CHG_DRV_DBGLOG("[%s] in type:[%d]\n", __func__, data->monitor_info.type);

	switch (data->monitor_info.type) {
	case CHG_DRV_MONITOR_NORMAL:
		chg_drv_get_parameters(chip, &param);
		err_type = chg_drv_check_parameters(chip, &param);

		switch (err_type) {
		case CHG_DRV_ERR_NON:
			CHG_DRV_INFOLOG("[%s] usb:[%d]uV chg_dtls:[%d] bat_dtls[%d] current:[%d]uA battvolt:[%d]uV\n",
				__func__, param.usbin_uv, param.chg_dtls, param.bat_dtls, param.fcc_ua, param.batt_volt_uv);
			if (chip->longevity_charge_flag) {
				if (param.soc <= chip->dt_param.chg_rechg_soc_longevity_lv2) {
					chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_CHARGING);
				}
				if (param.soc >= chip->dt_param.chg_full_soc_longevity_lv2) {
					chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_FULL);
				}
				if (param.soc >= chip->dt_param.chg_full_soc_longevity_lv1) {
					CHG_DRV_INFOLOG("[%s] battery full.\n", __func__);
					chg_drv_stop_safety_timer(chip);
					chg_drv_stop_reduction_timer(chip);
					chg_drv_stop_aicl_monitor(chip);
					chg_drv_charge_control(chip, CHG_DRV_CONTROL_POWERPATH);
					chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_FULL);
					next_state = CHG_DRV_STATE_FULL;
				} else {
					chg_drv_charge_control(chip, CHG_DRV_CONTROL_CHARGE_START);
				}
			} else {
				if (param.soc <= chip->dt_param.chg_rechg_soc_lv2) {
					chg_drv_stop_full_timer(chip);
					chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_CHARGING);
				} else if (param.soc >= chip->dt_param.chg_full_soc_lv1) {
					chg_drv_start_full_timer1(chip);
				}
				if (param.soc >= chip->dt_param.chg_full_soc_lv2 && chg_drv_is_battery_full(chip, &param)) {
					CHG_DRV_INFOLOG("[%s] battery full.\n", __func__);
					chg_drv_stop_full_timer(chip);
					chg_drv_stop_safety_timer(chip);
					chg_drv_stop_reduction_timer(chip);
					chg_drv_stop_aicl_monitor(chip);
					chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_FULL);
					next_state = CHG_DRV_STATE_FULL;
				} else {
					if (param.soc >= CHG_DRV_REDUCTION_MONITOR_START) {
						chg_drv_start_reduction_timer(chip);
					}
					chg_drv_charge_control(chip, CHG_DRV_CONTROL_CHARGE_START);
				}
			}
			break;

		default:
			chg_drv_err_control(chip, err_type);
			next_state = CHG_DRV_STATE_ERROR;
			break;
		}

		chg_drv_start_monitor(chip);
		break;

	case CHG_DRV_MONITOR_ADAPTER_VOLTAGE:
		/* check supply voltage */
		chg_drv_get_parameters(chip, &param);
		if ((chip->charger_info.chg_source & CHG_DRV_SOURCE_USB_PORT) != 0) {
			usb_result = chg_drv_watch_usb_adapter_voltage(chip, &usb_err_type, &param);
		}
		if (usb_result == CHG_DRV_RESULT_VOLTAGE_ERROR) {
			/* voltage error */
			if (usb_err_type == CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR) {
				CHG_DRV_ERRLOG("[%s] [CHGDRVERR] OVER SUPPLY VOLTAGE.\n", __func__);
				chg_drv_err_control(chip, CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR);
				next_state = CHG_DRV_STATE_ERROR;
			}
			if (usb_err_type == CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR) {
				CHG_DRV_ERRLOG("[%s] [CHGDRVERR] UNDER SUPPLY VOLTAGE.\n", __func__);
				chg_drv_err_control(chip, CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR);
				next_state = CHG_DRV_STATE_ERROR;
			}
		} else {
			chg_drv_charge_control(chip, CHG_DRV_CONTROL_CHARGE_START);
			chg_drv_check_apsd_result(chip);
			if ((param.cc_sts & GENMASK(2, 0)) == 0x07) {
				chip->err_info.unspec_cable_err = true;
				memset(&evt_data, 0, sizeof(union chg_drv_evt_data));
				evt_data.monitor_info.type = CHG_DRV_MONITOR_NORMAL;
				chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);
			}
		}
		chg_drv_start_adapter_volt_monitor(chip, CHG_DRV_SET_TIMER_1S);
		break;

	case CHG_DRV_MONITOR_AICL:
		chg_drv_check_aicl_threshold(chip);
		chg_drv_start_aicl_monitor(chip);
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_full_monitor(struct oem_charger_drv_chip *chip,
															union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;
	enum chg_drv_err_type err_type = CHG_DRV_ERR_NON;
	struct chg_drv_monitor_param param = {0};
	int ret = 0;
	int rechg_soc = 0;
	enum chg_drv_err_type usb_err_type = CHG_DRV_ERR_NON;
	enum chg_drv_result_type usb_result = CHG_DRV_RESULT_SUCCESS;
	union chg_drv_evt_data evt_data;

	CHG_DRV_DBGLOG("[%s] in type:[%d]\n", __func__, data->monitor_info.type);

	switch (data->monitor_info.type) {
	case CHG_DRV_MONITOR_NORMAL:
		chg_drv_get_parameters(chip, &param);
		err_type = chg_drv_check_parameters(chip, &param);

		switch (err_type) {
		case CHG_DRV_ERR_NON:
			if (chip->longevity_charge_flag) {
				rechg_soc = chip->dt_param.chg_rechg_soc_longevity_lv1;
			} else {
				rechg_soc = chip->dt_param.chg_rechg_soc_lv1;
			}
			if (param.soc <= rechg_soc) {
				CHG_DRV_RECLOG("[%s] soc changed from FULL to %d, start recharging\n",
									__func__, rechg_soc);
				ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_RECHARGE);
				if (ret == 0) {
					chg_drv_start_safety_timer(chip);
					chg_drv_start_aicl_monitor(chip);
					/* keep POWER_SUPPLY_STATUS_FULL status */
					next_state = CHG_DRV_STATE_CHARGING;
				} else {
					chg_drv_err_control(chip, CHG_DRV_ERR_OTHER_ERR);
					next_state = CHG_DRV_STATE_ERROR;
				}
			}
			break;

		default:
			chg_drv_err_control(chip, err_type);
			next_state = CHG_DRV_STATE_ERROR;
			break;
		}

		chg_drv_start_monitor(chip);
		break;

	case CHG_DRV_MONITOR_ADAPTER_VOLTAGE:
		/* check supply voltage */
		chg_drv_get_parameters(chip, &param);
		if ((chip->charger_info.chg_source & CHG_DRV_SOURCE_USB_PORT) != 0) {
			usb_result = chg_drv_watch_usb_adapter_voltage(chip, &usb_err_type, &param);
		}
		if (usb_result == CHG_DRV_RESULT_VOLTAGE_ERROR) {
			/* voltage error */
			if (usb_err_type == CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR) {
				CHG_DRV_ERRLOG("[%s] [CHGDRVERR] OVER SUPPLY VOLTAGE.\n", __func__);
				chg_drv_err_control(chip, CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR);
				next_state = CHG_DRV_STATE_ERROR;
			}
			if (usb_err_type == CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR) {
				CHG_DRV_ERRLOG("[%s] [CHGDRVERR] UNDER SUPPLY VOLTAGE.\n", __func__);
				chg_drv_err_control(chip, CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR);
				next_state = CHG_DRV_STATE_ERROR;
			}
		} else {
			chg_drv_check_apsd_result(chip);
			if ((param.cc_sts & GENMASK(2, 0)) == 0x07) {
				chip->err_info.unspec_cable_err = true;
				memset(&evt_data, 0, sizeof(union chg_drv_evt_data));
				evt_data.monitor_info.type = CHG_DRV_MONITOR_NORMAL;
				chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);
			}
		}
		chg_drv_start_adapter_volt_monitor(chip, CHG_DRV_SET_TIMER_1S);
		break;

	case CHG_DRV_MONITOR_AICL:
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_error_monitor(struct oem_charger_drv_chip *chip, union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;
	struct chg_drv_monitor_param param = {0};
	enum chg_drv_err_type err_type = CHG_DRV_ERR_NON;
	int ret = 0;
	enum chg_drv_err_type usb_err_type = CHG_DRV_ERR_NON;
	enum chg_drv_result_type usb_result = CHG_DRV_RESULT_SUCCESS;
	int full_soc = 0;
	union chg_drv_evt_data evt_data;

	CHG_DRV_DBGLOG("[%s] in type:[%d]\n", __func__, data->monitor_info.type);

	switch (data->monitor_info.type) {
	case CHG_DRV_MONITOR_NORMAL:
		chg_drv_get_parameters(chip, &param);
		err_type = chg_drv_check_parameters(chip, &param);

		switch (err_type) {
		case CHG_DRV_ERR_NON:
			if (chip->longevity_charge_flag) {
				full_soc = chip->dt_param.chg_full_soc_longevity_lv1;
			} else {
				full_soc = chip->dt_param.chg_full_soc_lv1;
			}
			if (param.soc > full_soc) {
				ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_POWERPATH);
				if (likely(ret == 0)) {
					chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_CHARGING);
					chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_GOOD);
					next_state = CHG_DRV_STATE_FULL;
				} else {
					chg_drv_err_control(chip, CHG_DRV_ERR_OTHER_ERR);
					next_state = CHG_DRV_STATE_ERROR;
				}
			} else {
				ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_RECHARGE);
				if (likely(ret == 0)) {
					chg_drv_start_safety_timer(chip);
					chg_drv_start_aicl_monitor(chip);
					chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_CHARGING);
					chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_GOOD);
					next_state = CHG_DRV_STATE_CHARGING;
				}
			}
			break;

		default:
			chg_drv_err_control(chip, err_type);
			break;
		}

		chg_drv_start_monitor(chip);
		break;

	case CHG_DRV_MONITOR_ADAPTER_VOLTAGE:
		/* check supply voltage */
		chg_drv_get_parameters(chip, &param);
		if ((chip->charger_info.chg_source & CHG_DRV_SOURCE_USB_PORT) != 0) {
			usb_result = chg_drv_watch_usb_adapter_voltage(chip, &usb_err_type, &param);
		}
		if (usb_result == CHG_DRV_RESULT_VOLTAGE_ERROR) {
			/* voltage error */
			if (usb_err_type == CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR) {
				chg_drv_err_control(chip, CHG_DRV_ERR_OVER_SUPPLY_VOLTAGE_ERR);
			}
			if (usb_err_type == CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR) {
				chg_drv_err_control(chip, CHG_DRV_ERR_UNDER_SUPPLY_VOLTAGE_ERR);
			}
		} else {
			if (chip->charge_ctrl != CHG_DRV_CONTROL_STOP) {
				chg_drv_check_apsd_result(chip);
			}
			if ((param.cc_sts & GENMASK(2, 0)) != 0x07) {
				chip->err_info.unspec_cable_err = false;
				memset(&evt_data, 0, sizeof(union chg_drv_evt_data));
				evt_data.monitor_info.type = CHG_DRV_MONITOR_NORMAL;
				chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);
			}
		}
		chg_drv_start_adapter_volt_monitor(chip, CHG_DRV_SET_TIMER_1S);
		break;

	case CHG_DRV_MONITOR_AICL:
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_idle_info_notice(struct oem_charger_drv_chip *chip, union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	switch (data->notice_info.type) {
	case CHG_DRV_NOTICE_OVP_NOTIFY_ERROR:
	case CHG_DRV_NOTICE_USBHOT_NOTIFY_ERROR:
#ifdef CHG_DRV_CONFIG_PD_ENABLE
	case CHG_DRV_NOTICE_PDO_ERROR:
#endif
		chg_drv_update_safety_err_info(chip, &data->notice_info);

		if ((chip->safety_err_info.usb_port_failure == true) ||
			(chip->safety_err_info.usb_removal == true)) {
			chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_OVERCURRENT);
			chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_NOT_CHARGING);
		} else if (chip->safety_err_info.ovp_detect == true) {
			chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_UNSPEC_FAILURE);
			chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_NOT_CHARGING);
#ifdef CHG_DRV_CONFIG_PD_ENABLE
		} else if (chip->safety_err_info.pdo_profile_error == true) {
			chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_WATCHDOG_TIMER_EXPIRE);
			chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_NOT_CHARGING);
#endif
		} else {
			chg_drv_batt_set_health(chip, POWER_SUPPLY_HEALTH_GOOD);
			chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_DISCHARGING);
		}
		break;

	case CHG_DRV_NOTICE_CHG_ENABLE:
		chip->max_icl_value = data->notice_info.icl;
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_info_notice(struct oem_charger_drv_chip *chip,
														union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;

	CHG_DRV_DBGLOG("[%s] in type = %d\n", __func__, data->notice_info.type);

	switch (data->notice_info.type) {
	case CHG_DRV_NOTICE_OVP_NOTIFY_ERROR:
	case CHG_DRV_NOTICE_USBHOT_NOTIFY_ERROR:
#ifdef CHG_DRV_CONFIG_PD_ENABLE
	case CHG_DRV_NOTICE_PDO_ERROR:
#endif
		chg_drv_notify_error_control(chip, &data->notice_info);
		break;

	case CHG_DRV_NOTICE_CHG_ENABLE:
		chg_drv_charge_enable_control(chip, data->notice_info.icl);
		break;

	case CHG_DRV_NOTICE_CHANGE_CURRENT:
		chg_drv_change_current_control(chip, &data->notice_info.change_info);
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_idle_control(struct oem_charger_drv_chip *chip,
															union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;
	enum chg_drv_ctrl_type charge_ctrl;
	enum chg_drv_err_type err_type = CHG_DRV_ERR_NON;
	struct chg_drv_monitor_param param = {0};
	int ret;
	int full_soc = 0;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	do {
		if (chip->charge_mode == CHG_DRV_CHARGE_MODE_NOT_CHARGE) {
			chg_drv_charge_control(chip, CHG_DRV_CONTROL_STOP);
			break;
		}

		charge_ctrl = chg_drv_connect_info_update(chip, &data->connect_info);

		switch (charge_ctrl) {
		case CHG_DRV_CONTROL_STOP:
			chg_drv_charge_control(chip, CHG_DRV_CONTROL_STOP);
			chg_drv_terminate(chip, false);
			break;

		case CHG_DRV_CONTROL_USB_CHARGE:
			chg_drv_get_parameters(chip, &param);
			err_type = chg_drv_check_parameters(chip, &param);
			if (err_type == CHG_DRV_ERR_NON) {
				chg_drv_check_aicl_threshold(chip);
				if (chip->longevity_charge_flag) {
					full_soc = chip->dt_param.chg_full_soc_longevity_lv1;
				} else {
					full_soc = chip->dt_param.chg_full_soc_lv1;
				}
				if (param.soc > full_soc) {
					ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_POWERPATH);
					if (likely(ret == 0)) {
						chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_CHARGING);
						next_state = CHG_DRV_STATE_FULL;
					} else {
						chg_drv_err_control(chip, CHG_DRV_ERR_OTHER_ERR);
						next_state = CHG_DRV_STATE_ERROR;
					}
				} else {
					ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_PRECHARGE);
					if (likely(ret == 0)) {
						chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_CHARGING);
						chg_drv_start_safety_timer(chip);
						chg_drv_start_aicl_monitor(chip);
						next_state = CHG_DRV_STATE_CHARGING;
					} else {
						chg_drv_err_control(chip, CHG_DRV_ERR_OTHER_ERR);
						next_state = CHG_DRV_STATE_ERROR;
					}
				}
			} else {
				chg_drv_err_control(chip, err_type);
				next_state = CHG_DRV_STATE_ERROR;
			}
			chg_drv_start_monitor(chip);
			chg_drv_start_adapter_volt_monitor(chip, CHG_DRV_SET_TIMER_1S);
			chg_drv_start_watchdog_timer(chip);
			break;

		default:
			break;
		}
	} while (0);

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_charging_control(struct oem_charger_drv_chip *chip,
																union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;
	enum chg_drv_ctrl_type charge_ctrl;
	enum chg_drv_err_type err_type = CHG_DRV_ERR_NON;
	struct chg_drv_monitor_param param = {0};
	int ret = 0;

	charge_ctrl = chg_drv_connect_info_update(chip, &data->connect_info);

	switch (charge_ctrl) {
	case CHG_DRV_CONTROL_STOP:
		chg_drv_charge_control(chip, CHG_DRV_CONTROL_STOP);
		chg_drv_terminate(chip, false);
		next_state = CHG_DRV_STATE_IDLE;
		break;

	case CHG_DRV_CONTROL_USB_CHARGE:
		chg_drv_stop_full_timer(chip);
		chg_drv_stop_safety_timer(chip);
		chg_drv_stop_reduction_timer(chip);
		chg_drv_stop_monitor(chip);

		chg_drv_get_parameters(chip, &param);
		err_type = chg_drv_check_parameters(chip, &param);

		if (err_type == CHG_DRV_ERR_NON) {
			chg_drv_check_aicl_threshold(chip);
			ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_PRECHARGE);
			if (likely(ret == 0)) {
				chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_CHARGING);
				chg_drv_start_safety_timer(chip);
			} else {
				chg_drv_err_control(chip, CHG_DRV_ERR_OTHER_ERR);
				next_state = CHG_DRV_STATE_ERROR;
			}
		} else {
			chg_drv_err_control(chip, err_type);
			next_state = CHG_DRV_STATE_ERROR;
		}
		chg_drv_start_monitor(chip);
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_connect_control(struct oem_charger_drv_chip *chip,
															union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;
	enum chg_drv_ctrl_type charge_ctrl;
	enum chg_drv_err_type err_type = CHG_DRV_ERR_NON;
	struct chg_drv_monitor_param param = {0};
	int ret = 0;

	charge_ctrl = chg_drv_connect_info_update(chip, &data->connect_info);

	switch (charge_ctrl) {
	case CHG_DRV_CONTROL_STOP:
		chg_drv_charge_control(chip, CHG_DRV_CONTROL_STOP);
		chg_drv_terminate(chip, false);
		next_state = CHG_DRV_STATE_IDLE;
		break;

	case CHG_DRV_CONTROL_USB_CHARGE:
		chg_drv_stop_monitor(chip);

		chg_drv_get_parameters(chip, &param);
		err_type = chg_drv_check_parameters(chip, &param);
		if (err_type == CHG_DRV_ERR_NON) {
			ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_POWERPATH);
			if (unlikely(ret != 0)) {
				chg_drv_err_control(chip, CHG_DRV_ERR_OTHER_ERR);
				next_state = CHG_DRV_STATE_ERROR;
			}
		} else {
			chg_drv_err_control(chip, err_type);
			next_state = CHG_DRV_STATE_ERROR;
		}
		chg_drv_start_monitor(chip);
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_charging_timer_expire(struct oem_charger_drv_chip *chip,
																	union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;
	int batt_voltage = 0;
	int ret = 0;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	switch (data->timer_info.type) {
	case CHG_DRV_TIMER_FULL_1:
		CHG_DRV_INFOLOG("[%s] additional charge\n", __func__);
		chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_FULL);
		chg_drv_start_full_timer2(chip);
		break;

	case CHG_DRV_TIMER_FULL_2:
		CHG_DRV_INFOLOG("[%s] battery full.\n", __func__);
		chg_drv_stop_full_timer(chip);
		chg_drv_stop_safety_timer(chip);
		chg_drv_stop_reduction_timer(chip);
		chg_drv_stop_aicl_monitor(chip);
		chg_drv_batt_set_status(chip, POWER_SUPPLY_STATUS_FULL);
		next_state = CHG_DRV_STATE_FULL;
		break;

	case CHG_DRV_TIMER_PROTECTION:
		ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_POWERPATH);
		if (unlikely(ret != 0)) {
			chip->err_info.safety_timer_expire = true;
			chg_drv_err_control(chip, CHG_DRV_ERR_EXPIRE_TIME_OUT);
			next_state = CHG_DRV_STATE_ERROR;
			break;
		}
		msleep(2000);
		batt_voltage = chg_drv_get_battery_voltage(chip);
		if (batt_voltage < chip->dt_param.batt_unspec_voltage) {
			chip->err_info.safety_timer_expire = true;
			chg_drv_err_control(chip, CHG_DRV_ERR_EXPIRE_TIME_OUT);
			next_state = CHG_DRV_STATE_ERROR;
			break;
		}

		ret = chg_drv_charge_control(chip, CHG_DRV_CONTROL_RECHARGE);
		if (unlikely(ret != 0)) {
			chip->err_info.safety_timer_expire = true;
			chg_drv_err_control(chip, CHG_DRV_ERR_EXPIRE_TIME_OUT);
			next_state = CHG_DRV_STATE_ERROR;
			break;
		}

		chg_drv_start_safety_timer(chip);
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_timer_expire(struct oem_charger_drv_chip *chip,
															union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	switch (data->timer_info.type) {
	case CHG_DRV_TIMER_FULL_1:
		chg_drv_stop_full_timer(chip);
		break;

	case CHG_DRV_TIMER_FULL_2:
		chg_drv_stop_full_timer(chip);
		break;

	default:
		break;
	}

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_parameter_update(struct oem_charger_drv_chip *chip,
																union chg_drv_evt_data *data)
{
	enum chg_drv_state_type next_state = CHG_DRV_STATE_NOCHANGE;
	int ret = 0;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	do {
		if (chip->charge_ctrl == CHG_DRV_CONTROL_STOP) {
			CHG_DRV_INFOLOG("[%s] charge_ctrl=STOP, Skip current setting\n", __func__);
			break;
		}

		ret = chg_drv_pmic_current_setting(chip);
		if (unlikely(ret != 0)) {
			CHG_DRV_ERRLOG("[%s] Parameter Update Failed. ret:[%d]\n", __func__, ret);
		}
	} while (0);

	return next_state;
}

static enum chg_drv_state_type chg_drv_func_null(struct oem_charger_drv_chip *chip, union chg_drv_evt_data *data)
{
	return CHG_DRV_STATE_NOCHANGE;
}

static const chg_drv_func chg_drv_state_func_table[CHG_DRV_STATE_NUM][CHG_DRV_EVENT_NUM] = {
	/* CHG_DRV_STATE_IDLE */					{
		/* CHG_DRV_EVENT_MONITOR_PARAM		*/		chg_drv_func_null,
		/* CHG_DRV_EVENT_CHARGE_INFO_NOTICE	*/		chg_drv_func_idle_info_notice,
		/* CHG_DRV_EVENT_CHARGE_CTRL		*/		chg_drv_func_idle_control,
		/* CHG_DRV_EVENT_CHARGE_TIMER		*/		chg_drv_func_null,
		/* CHG_DRV_EVENT_PARAM_UPDATE		*/		chg_drv_func_null,
	/* CHG_DRV_STATE_CHARGING */				} , {
		/* CHG_DRV_EVENT_MONITOR_PARAM		*/		chg_drv_func_charging_monitor,
		/* CHG_DRV_EVENT_CHARGE_INFO_NOTICE	*/		chg_drv_func_info_notice,
		/* CHG_DRV_EVENT_CHARGE_CTRL		*/		chg_drv_func_charging_control,
		/* CHG_DRV_EVENT_CHARGE_TIMER		*/		chg_drv_func_charging_timer_expire,
		/* CHG_DRV_EVENT_PARAM_UPDATE		*/		chg_drv_func_parameter_update,
	/* CHG_DRV_STATE_FULL */					} , {
		/* CHG_DRV_EVENT_MONITOR_PARAM		*/		chg_drv_func_full_monitor,
		/* CHG_DRV_EVENT_CHARGE_INFO_NOTICE	*/		chg_drv_func_info_notice,
		/* CHG_DRV_EVENT_CHARGE_CTRL		*/		chg_drv_func_connect_control,
		/* CHG_DRV_EVENT_CHARGE_TIMER		*/		chg_drv_func_timer_expire,
		/* CHG_DRV_EVENT_PARAM_UPDATE		*/		chg_drv_func_parameter_update,
	/* CHG_DRV_STATE_ERROR */				} , {
		/* CHG_DRV_EVENT_MONITOR_PARAM		*/		chg_drv_func_error_monitor,
		/* CHG_DRV_EVENT_CHARGE_INFO_NOTICE	*/		chg_drv_func_info_notice,
		/* CHG_DRV_EVENT_CHARGE_CTRL		*/		chg_drv_func_connect_control,
		/* CHG_DRV_EVENT_CHARGE_TIMER		*/		chg_drv_func_timer_expire,
		/* CHG_DRV_EVENT_PARAM_UPDATE		*/		chg_drv_func_parameter_update,
											}
};

static void chg_drv_source_req(int source, unsigned int mA)
{
	struct oem_charger_drv_chip *chip = oem_chg_drv_chip;
	unsigned long source_work = 0;
	union chg_drv_evt_data evt_data;
	static char chg_src_str[OEM_CHG_SOURCE_NUM][8] = {
		"USB", "AC", "APSD",
	};

	if (unlikely(chip == NULL)) {
		CHG_DRV_ERRLOG("[%s] CHG-DRV uninitialized\n", __func__);
	}

	CHG_DRV_RECLOG("[%s] %s mA=%d\n", __func__, chg_src_str[source], mA);

	if (unlikely(chip != NULL) && unlikely(chip->chg_drv_timer_wake_lock != NULL)) {
		__pm_wakeup_event(chip->chg_drv_timer_wake_lock, (chip->dt_param.wake_lock_timer_sec * HZ));
	}

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	source_work = CHG_DRV_SOURCE(source);

	if (mA != 0) {
		evt_data.connect_info.event = CHG_DRV_CONNECT_EVENT_USB_IN;
	} else {
		evt_data.connect_info.event = CHG_DRV_CONNECT_EVENT_USB_OUT;
	}

	evt_data.connect_info.chg_source = source;
	evt_data.connect_info.max_current = mA;

	chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_CTRL, &evt_data);
}

int oem_chg_drv_thread(void *ptr)
{
	struct oem_charger_drv_chip *chip = ptr;
	enum chg_drv_state_type new_state = CHG_DRV_STATE_NOCHANGE;
	struct chg_drv_evt_queue *evt_queue_p = NULL;
	const chg_drv_func *functbl;
	union power_supply_propval pval = {0,};
	int ret = 0;

	while (1) {
		ret = power_supply_get_property(chip->oem_battery_psy, POWER_SUPPLY_PROP_PRESENT, &pval);
		if (unlikely(ret < 0)) {
			CHG_DRV_ERRLOG("[%s] get FGIC_INITIAL_STATUS error %d\n", __func__, ret);
		}
		if (pval.intval) {
			break;
		} else {
			CHG_DRV_ERRLOG("[%s] oem_battery is initializing\n", __func__);
		}
		msleep(200);
	}

	ret = chg_drv_update_cycle_charge_setting(chip);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] update_cycle_charge_voltage failed. ret = %d\n", __func__, ret);
	}
	CHG_DRV_DBGLOG("[%s] vmax_adj = %d(uV)\n", __func__, chip->vmax_adj);

	do {
		wait_event(chip->chg_wq,
				  (!list_empty(&chg_drv_evt_list_exec) || kthread_should_stop()));
		if (kthread_should_stop()) {
			break;
		}

		evt_queue_p = chg_drv_queue_get(&chg_drv_evt_list_exec);
		if (evt_queue_p == NULL) {
			continue;
		}

		if ((chip->charger_info.state >= CHG_DRV_STATE_NUM) ||
			(evt_queue_p->event >= CHG_DRV_EVENT_NUM))	 {
			CHG_DRV_ERRLOG("[%s] thread Error:sts[%d] evt[%d] Fatal\n",
						__func__, chip->charger_info.state, evt_queue_p->event);
			emergency_restart();
		}

		if (chip->charger_info.state == CHG_DRV_STATE_IDLE) {
			if (unlikely(chip->chg_drv_timer_wake_lock != NULL)) {
				__pm_stay_awake(chip->chg_drv_charging_wake_lock);
			}
		}

		CHG_DRV_DBGLOG("[%s] charge_state = %d event = %d\n",
									__func__, chip->charger_info.state, evt_queue_p->event);

		functbl = &chg_drv_state_func_table[chip->charger_info.state][evt_queue_p->event];
		new_state = (*functbl)(chip, &evt_queue_p->data);

		if (new_state != CHG_DRV_STATE_NOCHANGE) {
			CHG_DRV_INFOLOG("[%s] event:%d old state:%d -> new state:%d\n",
					 __func__, evt_queue_p->event, chip->charger_info.state, new_state);
			chip->charger_info.state = new_state;
		}

		chg_drv_queue_put(evt_queue_p, &chg_drv_evt_list_free);

		if (chip->charger_info.state == CHG_DRV_STATE_IDLE) {
			if (unlikely(chip->chg_drv_timer_wake_lock != NULL)) {
				__pm_relax(chip->chg_drv_charging_wake_lock);
			}
		}
	} while (1);

	return 0;
}

void oem_chg_usb_vbus_draw(unsigned int onoff)
{
	unsigned int mA = 0;

	if (onoff != 0) {
		mA = OEM_CHG_USB_CURRENT;
	} else {
		mA = OEM_CHG_OFF_CURRENT;
	}
	chg_drv_source_req(OEM_CHG_SOURCE_USB, mA);
}
EXPORT_SYMBOL_GPL(oem_chg_usb_vbus_draw);

void oem_chg_ac_vbus_draw(unsigned int onoff)
{
	unsigned int mA = 0;

	if (onoff != 0) {
		mA = OEM_CHG_AC_CURRENT;
	} else {
		mA = OEM_CHG_OFF_CURRENT;
	}
	chg_drv_source_req(OEM_CHG_SOURCE_AC, mA);
}
EXPORT_SYMBOL_GPL(oem_chg_ac_vbus_draw);

void oem_chg_other_vbus_draw(unsigned int onoff)
{
	unsigned int mA = 0;

	if (onoff != 0) {
		mA = OEM_CHG_OTHER_CURRENT;
	} else {
		mA = OEM_CHG_OFF_CURRENT;
	}
	chg_drv_source_req(OEM_CHG_SOURCE_AC, mA);
}
EXPORT_SYMBOL_GPL(oem_chg_other_vbus_draw);

void oem_chg_apsd_vbus_draw(unsigned int onoff)
{
	unsigned int mA = 0;

	if (onoff != 0) {
		mA = OEM_CHG_APSD_CURRENT;
	} else {
		mA = OEM_CHG_OFF_CURRENT;
	}
	chg_drv_source_req(OEM_CHG_SOURCE_APSD, mA);
}
EXPORT_SYMBOL_GPL(oem_chg_apsd_vbus_draw);

void oem_chg_notify_error(oem_chg_err_type chg_err)
{
	struct oem_charger_drv_chip *chip = oem_chg_drv_chip;
	union chg_drv_evt_data evt_data;

	CHG_DRV_INFOLOG("[%s] in, err=%d\n", __func__, chg_err);

	if (unlikely(chip == NULL)) {
		CHG_DRV_ERRLOG("[%s] CHG-DRV uninitialized\n", __func__);
	}

	if (unlikely(chip != NULL) && unlikely(chip->chg_drv_timer_wake_lock != NULL)) {
		__pm_wakeup_event(chip->chg_drv_timer_wake_lock, (chip->dt_param.wake_lock_timer_sec * HZ));
	}

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));

	switch (chg_err) {
	case OEM_CHG_ERROR_OVP_NONE:
		evt_data.notice_info.type = CHG_DRV_NOTICE_OVP_NOTIFY_ERROR;
		evt_data.notice_info.ovp_detect = OEM_CHG_ERROR_NONE;
		chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_INFO_NOTICE, &evt_data);
		break;

	case OEM_CHG_ERROR_OVP_DETECT:
		evt_data.notice_info.type = CHG_DRV_NOTICE_OVP_NOTIFY_ERROR;
		evt_data.notice_info.ovp_detect = OEM_CHG_ERROR_DETECT;
		chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_INFO_NOTICE, &evt_data);
		break;

	case OEM_CHG_ERROR_USB_HOT_NONE:
		evt_data.notice_info.type = CHG_DRV_NOTICE_USBHOT_NOTIFY_ERROR;
		evt_data.notice_info.usb_hot_detect = OEM_CHG_ERROR_NONE;
		chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_INFO_NOTICE, &evt_data);
		break;

	case OEM_CHG_ERROR_USB_HOT_DETECT:
		evt_data.notice_info.type = CHG_DRV_NOTICE_USBHOT_NOTIFY_ERROR;
		evt_data.notice_info.usb_hot_detect = OEM_CHG_ERROR_DETECT;
		chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_INFO_NOTICE, &evt_data);
		break;

#ifdef CHG_DRV_CONFIG_PD_ENABLE
	case OEM_CHG_ERROR_PDO_ERROR_NONE:
		evt_data.notice_info.type = CHG_DRV_NOTICE_PDO_ERROR;
		evt_data.notice_info.pdo_error_detect = OEM_CHG_ERROR_NONE;
		chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_INFO_NOTICE, &evt_data);
		break;

	case OEM_CHG_ERROR_PDO_ERROR_DETECT:
		evt_data.notice_info.type = CHG_DRV_NOTICE_PDO_ERROR;
		evt_data.notice_info.pdo_error_detect = OEM_CHG_ERROR_DETECT;
		chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_INFO_NOTICE, &evt_data);
		break;
#endif

	default:
		CHG_DRV_ERRLOG("[%s] Error Unknown type %d\n", __func__, chg_err);
		break;
	}
}
EXPORT_SYMBOL_GPL(oem_chg_notify_error);

bool oem_chg_initialized(void)
{
	struct oem_charger_drv_chip *chip = oem_chg_drv_chip;

	if (unlikely(chip == NULL)) {
		CHG_DRV_ERRLOG("[%s] CHG-DRV uninitialized\n", __func__);
		return false;
	}

	CHG_DRV_INFOLOG("[%s] CHG-DRV initialized\n", __func__);
	return true;
}
EXPORT_SYMBOL_GPL(oem_chg_initialized);

uint8_t oem_chg_is_connect(void)
{
	uint8_t is_connect = 0;
	struct oem_charger_drv_chip *chip = oem_chg_drv_chip;

	if (chip != NULL && chip->connect_state != CHG_DRV_CONNECT_STATE_NONE) {
		is_connect = 1;
	}

	return is_connect;
}
EXPORT_SYMBOL_GPL(oem_chg_is_connect);

static int set_oem_chg_enable(const char *val, const struct kernel_param *kp)
{
	struct oem_charger_drv_chip *chip = oem_chg_drv_chip;
	union chg_drv_evt_data evt_data;
	int result = 0;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	if (unlikely(chip == NULL)) {
		CHG_DRV_ERRLOG("[%s] CHG-DRV uninitialized\n", __func__);
		goto exit;
	}

	result = param_set_int(val, kp);
	if (result) {
		CHG_DRV_ERRLOG("[%s] error setting value %d\n",__func__, result);
		goto exit;
	}

	CHG_DRV_INFOLOG("[%s] set chg_enable param to %d\n",__func__, chg_enable);

	if (unlikely(chip != NULL) && unlikely(chip->chg_drv_timer_wake_lock != NULL)) {
		__pm_wakeup_event(chip->chg_drv_timer_wake_lock, (chip->dt_param.wake_lock_timer_sec * HZ));
	}

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));
	chip->chg_enable = chg_enable;
	evt_data.notice_info.type = CHG_DRV_NOTICE_CHG_ENABLE;

	switch (chip->chg_enable) {
	case 0:
		evt_data.notice_info.icl = CHG_DRV_ICL_DISABLE;
		break;

	case 1:
		evt_data.notice_info.icl = CHG_DRV_ICL_NOT_LIMITED;
		break;

	case 2:
		evt_data.notice_info.icl = CHG_DRV_ICL_LIMITED_900;
		break;

	case 3:
		evt_data.notice_info.icl = CHG_DRV_ICL_LIMITED_500;
		break;

	case 4:
		evt_data.notice_info.icl = CHG_DRV_ICL_LIMITED_300;
		break;

	default:
		evt_data.notice_info.icl = CHG_DRV_ICL_NOT_LIMITED;
		break;
	}
	chg_drv_event_req(chip, CHG_DRV_EVENT_CHARGE_INFO_NOTICE, &evt_data);

exit:
	return 0;
}
module_param_call(chg_enable, set_oem_chg_enable, param_get_uint, &chg_enable, 0644);

static int set_oem_chg_mode(const char *val, const struct kernel_param *kp)
{
	struct oem_charger_drv_chip *chip = oem_chg_drv_chip;
	union chg_drv_evt_data evt_data;
	int result = 0;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	if (unlikely(chip == NULL)) {
		CHG_DRV_ERRLOG("[%s] CHG-DRV uninitialized\n", __func__);
		goto exit;
	}

	result = param_set_int(val, kp);
	if (result) {
		CHG_DRV_ERRLOG("[%s] error setting value %d\n",__func__, result);
		goto exit;
	}

	CHG_DRV_INFOLOG("[%s] in mode = %d\n", __func__, chg_mode);

	if (unlikely(chip != NULL) && unlikely(chip->chg_drv_timer_wake_lock != NULL)) {
		__pm_wakeup_event(chip->chg_drv_timer_wake_lock, (chip->dt_param.wake_lock_timer_sec * HZ));
	}

	memset(&evt_data, 0, sizeof(union chg_drv_evt_data));
	chip->heat_charge_mode = chg_mode;

	evt_data.monitor_info.type = CHG_DRV_MONITOR_NORMAL;
	chg_drv_stop_monitor(chip);
	chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);

exit:
	return 0;
}
module_param_call(chg_mode, set_oem_chg_mode, param_get_uint, &chg_mode, 0644);

static int chg_drv_parse_dt(struct oem_charger_drv_chip *chip)
{
	struct chg_drv_dt *dt = &chip->dt_param;
	struct device_node *node = of_find_node_by_name(NULL, "max77729-charger");
	int ret;
	int byte_len;

	if (!node) {
		CHG_DRV_ERRLOG("[%s] device tree node missing\n", __func__);
		return -EINVAL;
	} else {
		chip->of_node = node;
	}

	ret = of_property_read_u32(node,
				"fcnt,batt-over-volt-uv",
				&dt->batt_over_voltage);
	if (ret < 0)
		dt->batt_over_voltage = CHG_DRV_BATT_VOLT_OV_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,batt-unspec-volt-uv",
				&dt->batt_unspec_voltage);
	if (ret < 0)
		dt->batt_unspec_voltage = CHG_DRV_BATT_VOLT_UV_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,batt-low-volt-uv,onchg",
				&dt->batt_low_voltage_onchg);
	if (ret < 0)
		dt->batt_low_voltage_onchg = CHG_DRV_BATT_VOLT_UV_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,batt-low-volt-uv,offchg",
				&dt->batt_over_voltage_offchg);
	if (ret < 0)
		dt->batt_over_voltage_offchg = CHG_DRV_BATT_VOLT_UV_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-full-soc-lv1",
				&dt->chg_full_soc_lv1);
	if (ret < 0)
		dt->chg_full_soc_lv1 = CHG_DRV_BATT_SOC_FULL_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-full-soc-lv2",
				&dt->chg_full_soc_lv2);
	if (ret < 0)
		dt->chg_full_soc_lv2 = CHG_DRV_BATT_SOC_FULL_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-rechg-soc-lv1",
				&dt->chg_rechg_soc_lv1);
	if (ret < 0)
		dt->chg_rechg_soc_lv1 = CHG_DRV_BATT_SOC_RECHG_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-rechg-soc-lv2",
				&dt->chg_rechg_soc_lv2);
	if (ret < 0)
		dt->chg_rechg_soc_lv2 = CHG_DRV_BATT_SOC_RECHG_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-full-soc-longevity-lv1",
				&dt->chg_full_soc_longevity_lv1);
	if (ret < 0)
		dt->chg_full_soc_longevity_lv1 = CHG_DRV_BATT_SOC_FULL_LONGEVITY_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-full-soc-longevity-lv2",
				&dt->chg_full_soc_longevity_lv2);
	if (ret < 0)
		dt->chg_full_soc_longevity_lv2 = CHG_DRV_BATT_SOC_FULL_LONGEVITY_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-rechg-soc-longevity-lv1",
				&dt->chg_rechg_soc_longevity_lv1);
	if (ret < 0)
		dt->chg_rechg_soc_longevity_lv1 = CHG_DRV_BATT_SOC_RECHG_LONGEVITY_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-rechg-soc-longevity-lv2",
				&dt->chg_rechg_soc_longevity_lv2);
	if (ret < 0)
		dt->chg_rechg_soc_longevity_lv2 = CHG_DRV_BATT_SOC_RECHG_LONGEVITY_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-volt-uv",
				&dt->chg_voltage);
	if (ret < 0)
		dt->chg_voltage = CHG_DRV_CHARGE_VOLT_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-volt-cycle-uv-lv1",
				&dt->chg_voltage_cycle_lv1);
	if (ret < 0)
		dt->chg_voltage_cycle_lv1 = CHG_DRV_CHARGE_VOLT_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-volt-cycle-uv-lv2",
				&dt->chg_voltage_cycle_lv2);
	if (ret < 0)
		dt->chg_voltage_cycle_lv2 = CHG_DRV_CHARGE_VOLT_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-volt-reduction-uv",
				&dt->chg_voltage_reduction);
	if (ret < 0)
		dt->chg_voltage_reduction = CHG_DRV_REDUCTION_DISABLED;

	ret = of_property_read_u32(node,
				"fcnt,chg-volt-aicl-threshold-uv",
				&dt->chg_voltage_aicl_threshold);
	if (ret < 0)
		dt->chg_voltage_aicl_threshold = CHG_DRV_CHARGE_VOLT_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,chg-safety-timer-min",
				&dt->chg_safety_timer_min);
	if (ret < 0)
		dt->chg_safety_timer_min = 0;

	ret = of_property_read_u32(node,
				"fcnt,chg-full1-timer-min",
				&dt->chg_full1_timer_min);
	if (ret < 0)
		dt->chg_full1_timer_min = 0;

	ret = of_property_read_u32(node,
				"fcnt,chg-full2-timer-min",
				&dt->chg_full2_timer_min);
	if (ret < 0)
		dt->chg_full2_timer_min = 0;

	ret = of_property_read_u32(node,
				"fcnt,batt-temp-hot-limit",
				&dt->batt_temp_hot_limit);
	if (ret < 0)
		dt->batt_temp_hot_limit = CHG_DRV_BATT_TEMP_HOT_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,batt-temp-warm-limit",
				&dt->batt_temp_warm_limit);
	if (ret < 0)
		dt->batt_temp_warm_limit = CHG_DRV_BATT_TEMP_HOT_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,batt-temp-cool-limit",
				&dt->batt_temp_cool_limit);
	if (ret < 0)
		dt->batt_temp_cool_limit = CHG_DRV_BATT_TEMP_COLD_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,batt-temp-cold-limit",
				&dt->batt_temp_cold_limit);
	if (ret < 0)
		dt->batt_temp_cold_limit = CHG_DRV_BATT_TEMP_COLD_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,batt-temp-start-limit",
				&dt->batt_temp_start_limit);
	if (ret < 0)
		dt->batt_temp_start_limit = CHG_DRV_BATT_TEMP_HOT_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,batt-temp-limit-cancel",
				&dt->batt_temp_limit_cancel);
	if (ret < 0)
		dt->batt_temp_limit_cancel = CHG_DRV_BATT_TEMP_CANCEL_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,wake-lock-timer-sec",
				&dt->wake_lock_timer_sec);
	if (ret < 0)
		dt->wake_lock_timer_sec = 3;

	ret = of_property_read_u32(node,
				"fcnt,usb-normal-fcc-max-ma",
				&dt->usb_normal_fcc_max);
	if (ret < 0)
		dt->usb_normal_fcc_max = CHG_DRV_FCC_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,usb-pd-fcc-max-ma",
				&dt->usb_pd_fcc_max);
	if (ret < 0)
		dt->usb_pd_fcc_max = CHG_DRV_FCC_DEFAULT;

	ret = of_property_read_u32(node,
				"fcnt,fcc-low-batt-ma",
				&dt->common_fcc_low_batt);
	if (ret < 0) {
		dt->common_fcc_low_batt = CHG_DRV_FCC_DEFAULT;
	}

	ret = of_property_read_u32(node,
				"fcnt,fcc-temp-warm-restrict-ma",
				&dt->common_fcc_temp_warm_restrict);
	if (ret < 0) {
		dt->common_fcc_temp_warm_restrict = CHG_DRV_FCC_WARM_DEFAULT;
	}

	ret = of_property_read_u32(node,
				"fcnt,fcc-temp-cool-restrict-ma",
				&dt->common_fcc_temp_cool_restrict);
	if (ret < 0) {
		dt->common_fcc_temp_cool_restrict = CHG_DRV_FCC_COOL_DEFAULT;
	}

	ret = of_property_read_u32(node,
				"fcnt,chg-consecutive-count",
				&dt->chg_consecutive_count);
	if (ret < 0)
		dt->chg_consecutive_count = 0;

	ret = of_property_read_u32(node,
				"fcnt,chg-term-current-ma",
				&dt->chg_termcurrent_ma);
	if (ret < 0)
		dt->chg_termcurrent_ma = 0;

	ret = of_property_read_u32(node,
				"fcnt,chg-cycle-term-current-ma",
				&dt->chg_cycle_termcurrent_ma);
	if (ret < 0)
		dt->chg_cycle_termcurrent_ma = 0;

	if (of_find_property(node, "fcnt,thermal-mitigation", &byte_len)) {
		dt->chg_icl_thermal_mitigation = devm_kzalloc(chip->dev, byte_len,
			GFP_KERNEL);

		if (dt->chg_icl_thermal_mitigation == NULL) {
			return -ENOMEM;
		}

		chip->thermal_level_max = byte_len / sizeof(u32);
		ret = of_property_read_u32_array(node,
				"fcnt,thermal-mitigation",
				dt->chg_icl_thermal_mitigation,
				chip->thermal_level_max);
		if (ret < 0) {
			chip->thermal_level_max = 0;
		}
	}

	return 0;
}

static void oem_chg_classdev_release(struct device *dev)
{
	return;
}

static int oem_chg_prepare(struct device *dev)
{
	return 0;
}

static const struct dev_pm_ops oem_chg_ops = {
	.prepare = oem_chg_prepare,
};

static struct class oem_chg_class = {
	.name		= "oem_chg",
	.dev_release	= oem_chg_classdev_release,
	.pm		= &oem_chg_ops,
};

static ssize_t oem_chg_fv_reduction_store(struct device *dev,
		struct device_attribute *attr, const char *data, size_t len)
{
	struct oem_charger_drv_chip *chip = dev_get_drvdata(dev);
	int value;
	bool enable;

	if (!chip) {
		CHG_DRV_ERRLOG("%s oem_charger_drv_chip null\n", __func__);
		return -ENODEV;
	}

	if (chip->dt_param.chg_voltage_reduction == CHG_DRV_REDUCTION_DISABLED) {
		CHG_DRV_ERRLOG("%s Fv Reduction Disabled\n", __func__);
		return -ENODEV;
	}

	if (kstrtoint(data, 0, &value)) {
		CHG_DRV_ERRLOG("%s invalid value\n", __func__);
		return -EINVAL;
	}

	if (value)
		enable = true;
	else
		enable = false;

	if (chip->charger_info.state == CHG_DRV_STATE_IDLE) {
		vote(chip->fv_votable, OEM_REDUCTION_VOTER, enable, (chip->dt_param.chg_voltage_reduction / 1000));
	}

	chip->chg_voltage_reduction_flag = enable;

	CHG_DRV_INFOLOG("%s value:%d\n", __func__, enable);

	return len;
}

static ssize_t oem_chg_fv_reduction_show(struct device *dev,
		struct device_attribute *attr, char *data)
{
	struct oem_charger_drv_chip *chip = dev_get_drvdata(dev);
	bool enable;

	if (!chip) {
		CHG_DRV_ERRLOG("%s oem_charger_drv_chip null\n", __func__);
		return -ENODEV;
	}

	enable = is_client_vote_enabled(chip->fv_votable, OEM_REDUCTION_VOTER);
	return sprintf(data, "%d\n", enable);
}

static DEVICE_ATTR(fv_reduction,  S_IWUSR | S_IRUGO, oem_chg_fv_reduction_show, oem_chg_fv_reduction_store);


static ssize_t oem_chg_schedule_chg_store(struct device *dev,
		struct device_attribute *attr, const char *data, size_t len)
{
	struct oem_charger_drv_chip *chip = dev_get_drvdata(dev);
	int schedule_chg;
	union chg_drv_evt_data evt_data;

	if (!chip) {
		CHG_DRV_ERRLOG("%s oem_charger_drv_chip null\n", __func__);
		return -ENODEV;
	}

	if (kstrtoint(data, 0, &schedule_chg)) {
		CHG_DRV_ERRLOG("%s invalid value\n", __func__);
		return -EINVAL;
	}

	chip->schedule_chg = schedule_chg;
	evt_data.monitor_info.type = CHG_DRV_MONITOR_NORMAL;
	chg_drv_stop_monitor(chip);
	chg_drv_event_req(chip, CHG_DRV_EVENT_MONITOR_PARAM, &evt_data);

	CHG_DRV_INFOLOG("%s value:%d\n", __func__, schedule_chg);

	return len;
}

static ssize_t oem_chg_schedule_chg_show(struct device *dev,
		struct device_attribute *attr, char *data)
{
	struct oem_charger_drv_chip *chip = dev_get_drvdata(dev);

	if (!chip) {
		CHG_DRV_ERRLOG("%s oem_charger_drv_chip null\n", __func__);
		return -ENODEV;
	}

	return sprintf(data, "%d\n", chip->schedule_chg);
}

static DEVICE_ATTR(schedule_chg,  S_IWUSR | S_IRUGO, oem_chg_schedule_chg_show, oem_chg_schedule_chg_store);

int oem_chg_drv_init(struct max77729_charger_data *charger)
{
	struct oem_charger_drv_chip	*chip;
	struct power_supply_config chg_drv_cfg = {};
	int ret = 0;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	chip = devm_kzalloc(charger->dev, sizeof(*chip), GFP_KERNEL);
	if (unlikely(chip == NULL)) {
		CHG_DRV_ERRLOG("[%s] No Memory Error!!\n", __func__);
		return -ENOMEM;
	}

	chip->charger = charger;
	chip->dev = charger->dev;
	chg_drv_get_charge_mode(chip);

	init_waitqueue_head(&chip->chg_wq);

	ret = class_register(&oem_chg_class);
	if (ret) {
		CHG_DRV_ERRLOG("%s: class register failed: %d\n", __func__, ret);
		return ret;
	}

	device_initialize(&chip->class_dev);
	chip->class_dev.parent = chip->dev;
	chip->class_dev.class = &oem_chg_class;
	dev_set_drvdata(&chip->class_dev, chip);

	ret = dev_set_name(&chip->class_dev, "%s", "oem_chg");
	if (ret) {
		CHG_DRV_ERRLOG("Failed to set name %d\n", ret);
		goto oem_chg_drv_err_classdev_put;
	}

	ret = device_add(&chip->class_dev);
	if (ret) {
		CHG_DRV_ERRLOG("Failed to device add %d\n", ret);
		goto oem_chg_drv_err_classdev_put;
	}

	ret = device_create_file(&chip->class_dev, &dev_attr_schedule_chg);
	if (ret) {
		CHG_DRV_ERRLOG("Failed to create setting schedule_chg entry %d\n", ret);
		goto oem_chg_drv_err_device_del;
	}

	ret = device_create_file(&chip->class_dev, &dev_attr_fv_reduction);
	if (ret) {
		CHG_DRV_ERRLOG("Failed to create setting fv_reduction entry %d\n", ret);
		goto oem_chg_drv_err_schedule_chg_del;
	}

	chip->create_class = true;

	chip->chg_drv_timer_wake_lock = wakeup_source_register(chip->dev, "oem_chg_drv_timer");
	if (!chip->chg_drv_timer_wake_lock) {
		CHG_DRV_ERRLOG("[%s] dt perse failed ret = %d\n", __func__, ret);
		goto oem_chg_drv_err_wakeup_source_register_chg_drv_timer;
	}

	chip->chg_drv_charging_wake_lock = wakeup_source_register(chip->dev, "oem_charging");
	if (!chip->chg_drv_charging_wake_lock) {
		CHG_DRV_ERRLOG("[%s] dt perse failed ret = %d\n", __func__, ret);
		goto oem_chg_drv_err_wakeup_source_register_oem_chg;
	}

	ret = chg_drv_parse_dt(chip);
	if (unlikely(ret != 0)) {
		CHG_DRV_ERRLOG("[%s] dt perse failed ret = %d\n", __func__, ret);
		goto oem_chg_drv_err_parse_dt;
	}

	chip->connect_state = CHG_DRV_CONNECT_STATE_NONE;
	chip->charge_type = CHG_DRV_CHARGE_TYPE_NORMAL;
	chip->max_icl_value = CHG_DRV_ICL_NOT_LIMITED;
	chip->heat_charge_mode = OEM_CHG_CHARGE_MODE_NORMAL;
	chip->chg_enable = 1;
	chip->consecutive_count = 0;
	chip->battery_low_flag = false;
	chip->aicl_threshold = CHG_DRV_AICL_THRESHOLD_4P5;
	chip->chg_voltage_reduction_flag = false;
	chip->reduction_motinor_enable = false;
	chip->reduction_disable_count = 0;
	chip->aicl_threshold_detect_count = 0;

	chip->dev_usb.max_current = CHG_DRV_CURRENT_OFF;

	chip->charge_ctrl = CHG_DRV_CONTROL_NONE;
	chip->mode_val = 0xFF;

	memset(&chip->err_info, 0, sizeof(struct chg_drv_err_info));

	chip->batt_temp_hot_limit = chip->dt_param.batt_temp_start_limit;
	chip->batt_temp_warm_limit = chip->dt_param.batt_temp_warm_limit;
	chip->batt_temp_cold_limit = chip->dt_param.batt_temp_cold_limit;
	chip->batt_temp_cool_limit = chip->dt_param.batt_temp_cool_limit;

	INIT_DELAYED_WORK(&chip->charge_monitor_work, chg_drv_monitor_work);
	INIT_DELAYED_WORK(&chip->full_timer_work1, chg_drv_full_timer_work1);
	INIT_DELAYED_WORK(&chip->full_timer_work2, chg_drv_full_timer_work2);
	INIT_DELAYED_WORK(&chip->safety_timer_work, chg_drv_safety_timer_work);
	INIT_DELAYED_WORK(&chip->adapter_volt_monitor_work, chg_drv_adapter_volt_monitor_work);
	INIT_DELAYED_WORK(&chip->watchdog_work, chg_drv_watchdog_work);
	INIT_DELAYED_WORK(&chip->reduction_work, chg_drv_reduction_work);
	INIT_DELAYED_WORK(&chip->usb_removal_wait_work, chg_drv_usb_removal_wait_work);
	INIT_DELAYED_WORK(&chip->aicl_monitor_work, chg_drv_aicl_monitor_work);

	chip->full_timer_state = CHG_DRV_CHARGE_TIMER_STATE_NONE;

	memset(chip->batt_volt_data.volt_data, 0, sizeof(chip->batt_volt_data.volt_data));
	chip->batt_volt_data.index = 0;
	chip->batt_volt_data.data_full = false;
	chip->longevity_charge_flag = false;
	chip->thermal_level = 0;
	chip->cycle_chg_lvl = CHG_CYCLE_LEVEL_NUM;

	chip->battery_psy = power_supply_get_by_name(BATTERY_PSY_NAME);
	if (chip->battery_psy == NULL) {
		CHG_DRV_ERRLOG("[%s] power_supply_get_by_name %s failed\n", __func__, BATTERY_PSY_NAME);
		ret = -EPROBE_DEFER;
		goto oem_chg_drv_err_power_supply_get_by_name_battery;
	}

	chip->oem_battery_psy = power_supply_get_by_name(OEM_BATTERY_PSY_NAME);
	if (chip->oem_battery_psy == NULL) {
		CHG_DRV_ERRLOG("[%s] power_supply_get_by_name %s failed\n", __func__, OEM_BATTERY_PSY_NAME);
		ret = -EPROBE_DEFER;
		goto oem_chg_drv_err_power_supply_get_by_name_oem_battery;
	}

	chip->bms_psy = power_supply_get_by_name(BMS_PSY_NAME);
	if (chip->bms_psy == NULL) {
		CHG_DRV_ERRLOG("[%s] power_supply_get_by_name %s failed\n", __func__, BMS_PSY_NAME);
		ret = -EPROBE_DEFER;
		goto oem_chg_drv_err_power_supply_get_by_name_bms;
	}

	chip->usb_psy = power_supply_get_by_name(USB_PSY_NAME);
	if (chip->usb_psy == NULL) {
		CHG_DRV_ERRLOG("[%s] power_supply_get_by_name %s failed\n", __func__, USB_PSY_NAME);
		ret = -EPROBE_DEFER;
		goto oem_chg_drv_err_power_supply_get_by_name_usb;
	}

	chip->fcc_votable = find_votable("FCC");
	if (!chip->fcc_votable) {
		ret = -EPROBE_DEFER;
		goto oem_chg_drv_err_find_votable_fcc;
	}

	chip->fv_votable = find_votable("FV");
	if (!chip->fv_votable) {
		ret = -EPROBE_DEFER;
		goto oem_chg_drv_err_find_votable_fv;
	}

	chip->usb_icl_votable = find_votable("USB_ICL");
	if (!chip->usb_icl_votable) {
		ret = -EPROBE_DEFER;
		goto oem_chg_drv_err_find_votable_usb_icl;
	}

	chip->chg_drv_wq = create_singlethread_workqueue(OEM_CHARGER_PSY_NAME);
	if (!chip->chg_drv_wq) {
		CHG_DRV_ERRLOG("[%s] create_singlethread_workqueue failed\n", __func__);
		ret = -ESRCH;
		goto oem_chg_drv_err_create_singlethread_workqueue;
	}

	chip->oem_chg_psy_desc.name = OEM_CHARGER_PSY_NAME;
	chip->oem_chg_psy_desc.type = POWER_SUPPLY_TYPE_MAINS;
	chip->oem_chg_psy_desc.get_property = chg_drv_oem_chg_get_property;
	chip->oem_chg_psy_desc.set_property = chg_drv_oem_chg_set_property;
	chip->oem_chg_psy_desc.properties = chg_drv_oem_chg_properties;
	chip->oem_chg_psy_desc.num_properties = ARRAY_SIZE(chg_drv_oem_chg_properties);
	chip->oem_chg_psy_desc.property_is_writeable = chg_drv_oem_chg_is_writeable;
	chg_drv_cfg.drv_data = chip;
	chg_drv_cfg.of_node = chip->of_node;
	chip->oem_chg_psy = devm_power_supply_register(chip->dev, &chip->oem_chg_psy_desc,
											   &chg_drv_cfg);
	if (IS_ERR(chip->oem_chg_psy)) {
		ret = PTR_ERR(chip->oem_chg_psy);
		CHG_DRV_ERRLOG("[%s] power_supply_register %s failed ret = %d\n", __func__, OEM_CHARGER_PSY_NAME, ret);
		goto oem_chg_drv_err_power_supply_register_oem_chg;
	}

#ifdef CHG_DRV_CONFIG_USE_APNV_VMAX
	ret = get_nonvolatile((u8 *)&chip->vmax_adj, APNV_CHG_ADJ_VMAX_I, 1);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get_nonvolatile failed. ret = %d\n", __func__, ret);
		chip->vmax_adj = chip->dt_param.chg_voltage;
	}
#else
	chip->vmax_adj = chip->dt_param.chg_voltage;
#endif

	ret = chg_drv_get_longevity_chg_mode(chip);
	if (unlikely(ret < 0)) {
		CHG_DRV_ERRLOG("[%s] get_longevity_charge_mode failed. ret = %d\n", __func__, ret);
	}

	ret = chg_drv_pmic_initial_setting(chip);
	if (unlikely(ret != 0)) {
		CHG_DRV_ERRLOG("[%s] pmic initial setting failed ret = %d\n", __func__, ret);
		ret = -EPROBE_DEFER;
		goto oem_chg_drv_err_pmic_initial_setting;
	}

	oem_chg_drv_chip = chip;

	chip->chg_thread = kthread_run(oem_chg_drv_thread, chip, "oem_chg");

	CHG_DRV_INFOLOG("[%s] probe End\n", __func__);

	return 0;

oem_chg_drv_err_pmic_initial_setting:
oem_chg_drv_err_power_supply_register_oem_chg:

	if (chip->chg_drv_wq)
		destroy_workqueue(chip->chg_drv_wq);

oem_chg_drv_err_create_singlethread_workqueue:

oem_chg_drv_err_find_votable_usb_icl:
oem_chg_drv_err_find_votable_fv:
oem_chg_drv_err_find_votable_fcc:

oem_chg_drv_err_power_supply_get_by_name_usb:
oem_chg_drv_err_power_supply_get_by_name_bms:
oem_chg_drv_err_power_supply_get_by_name_oem_battery:
oem_chg_drv_err_power_supply_get_by_name_battery:

oem_chg_drv_err_parse_dt:
	if (chip->chg_drv_charging_wake_lock)
		wakeup_source_unregister(chip->chg_drv_charging_wake_lock);

oem_chg_drv_err_wakeup_source_register_oem_chg:
	if (chip->chg_drv_timer_wake_lock)
		wakeup_source_unregister(chip->chg_drv_timer_wake_lock);

oem_chg_drv_err_wakeup_source_register_chg_drv_timer:
	device_remove_file(&chip->class_dev, &dev_attr_fv_reduction);
oem_chg_drv_err_schedule_chg_del:
	device_remove_file(&chip->class_dev, &dev_attr_schedule_chg);
oem_chg_drv_err_device_del:
	device_del(&chip->class_dev);
oem_chg_drv_err_classdev_put:
	put_device(&chip->class_dev);
	class_unregister(&oem_chg_class);
	chip->create_class = false;

	return ret;
}

int oem_chg_drv_queue_init(void)
{
	int i = 0;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	INIT_LIST_HEAD(&chg_drv_evt_list_free);
	INIT_LIST_HEAD(&chg_drv_evt_list_exec);
	for (i = 0; i < CHG_DRV_EVT_QUEUE_MAX; i++) {
		list_add_tail(&charge_evt_queue[i].head, &chg_drv_evt_list_free);
	}

	return 0;
}

void oem_chg_drv_remove(struct max77729_charger_data *charger)
{
	struct oem_charger_drv_chip	*chip = oem_chg_drv_chip;

	if (chip->chg_drv_timer_wake_lock)
		wakeup_source_unregister(chip->chg_drv_timer_wake_lock);

	if (chip->chg_drv_charging_wake_lock)
		wakeup_source_unregister(chip->chg_drv_charging_wake_lock);

	kthread_stop(chip->chg_thread);

	destroy_workqueue(chip->chg_drv_wq);

	if (chip->create_class) {
		device_remove_file(&chip->class_dev, &dev_attr_schedule_chg);
		device_remove_file(&chip->class_dev, &dev_attr_fv_reduction);
		device_del(&chip->class_dev);
		put_device(&chip->class_dev);
		class_unregister(&oem_chg_class);
		chip->create_class = false;
	}

	oem_chg_drv_chip = NULL;

	return;
}

void oem_chg_drv_shutdown(struct max77729_charger_data *charger)
{
	struct oem_charger_drv_chip	*chip = oem_chg_drv_chip;

	CHG_DRV_DBGLOG("[%s] in\n", __func__);

	chg_drv_terminate(chip, true);

	return;
}

MODULE_AUTHOR("FCNT");
MODULE_DESCRIPTION("OEM Charger Driver");
MODULE_LICENSE("GPL");
