// SPDX-License-Identifier: GPL-2.0-only
/*
 * (C) 2022 FCNT LIMITED
 */
#define pr_fmt(fmt)	"[USBPD-PM]: %s: " fmt, __func__

#include <linux/slab.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/power_supply.h>
#include <linux/delay.h>
#include <linux/workqueue.h>
#include "pd_policy_manager.h"

#define PD_SRC_PDO_TYPE_FIXED		0
#define PD_SRC_PDO_TYPE_BATTERY		1
#define PD_SRC_PDO_TYPE_VARIABLE	2
#define PD_SRC_PDO_TYPE_AUGMENTED	3

#define BATT_MAX_CHG_VOLT			4460
#define BATT_FAST_CHG_CURR			6000
#define BUS_OVP_THRESHOLD			12000
#define BUS_OVP_ALARM_THRESHOLD		9500

#define BUS_VOLT_INIT_UP			400

#define BAT_VOLT_LOOP_LMT			BATT_MAX_CHG_VOLT
#define BAT_CURR_LOOP_LMT			BATT_FAST_CHG_CURR
#define BUS_VOLT_LOOP_LMT			BUS_OVP_THRESHOLD

#define PM_WORK_RUN_INTERVAL		600

static const struct pdpm_config pm_config = {
	.bat_volt_lp_lmt		= BAT_VOLT_LOOP_LMT,
	.bat_curr_lp_lmt		= BAT_CURR_LOOP_LMT,
	.bus_volt_lp_lmt		= BUS_VOLT_LOOP_LMT,
	.bus_curr_lp_lmt		= (BAT_CURR_LOOP_LMT >> 1),

	.min_adapter_volt_required	= 10000,
	.min_adapter_curr_required	= 2000,

	.min_vbat_for_cp		= 3500,

	.cp_sec_enable			= false,
};

static struct usbpd_pm *__pdpm;

extern int usbpd_select_pdo_maxim(int pdo, int mv, int ma);

static int usbpd_select_pdo(struct usbpd_pm *pdpm, u32 mV, u32 mA)
{
	if(!pdpm) {
		return usbpd_select_pdo_maxim(0, mV, mA);
	} else {
		return usbpd_select_pdo_maxim(pdpm->apdo_selected_pdo, mV, mA);
	}
}

static void usbpd_pm_update_cp_status(struct usbpd_pm *pdpm)
{
	int ret;
	union power_supply_propval val = {0,};

	if(pdpm->bms_psy){
		ret = power_supply_get_property(pdpm->bms_psy,
				POWER_SUPPLY_PROP_CURRENT_NOW, &val);
		if(!ret)
			pdpm->cp.ibat_curr = -(val.intval / 1000);
	}
}

/*
 * Enable charging of switching charger
 *
 * @en: enable/disable
 */
static int usbpd_pm_enable_sw(struct usbpd_pm *pdpm, bool en)
{
	union power_supply_propval val = {0,};
	int ret;

	pr_debug("usbpd_pm_enable_sw:en:%d\n", en);
	if (!pdpm->sw_psy) {
		pdpm->sw_psy = power_supply_get_by_name("battery");
		if (!pdpm->sw_psy) {
			return -ENODEV;
		}
	}

	pdpm->charge_enabled = en;
	val.intval = en;
	ret = power_supply_set_property(pdpm->sw_psy,
			(enum power_supply_property)POWER_SUPPLY_EXT_PROP_BATTERY_CHARGING_ENABLED, &val);

	return 0;
}

static void usbpd_check_charger_psy(struct usbpd_pm *pdpm)
{
	if (!pdpm->usb_psy) {
		pdpm->usb_psy = power_supply_get_by_name("usb");
		if (!pdpm->usb_psy)
			pr_err("usb psy not found!\n");
	}
}

static int usbpd_pm_check_sw_enabled(struct usbpd_pm *pdpm)
{
	int ret;
	union power_supply_propval val = {0,};

	if (!pdpm->sw_psy) {
		pdpm->sw_psy = power_supply_get_by_name("battery");
		if (!pdpm->sw_psy) {
			return -ENODEV;
		}
	}

	ret = power_supply_get_property(pdpm->sw_psy,
			(enum power_supply_property)POWER_SUPPLY_EXT_PROP_BATTERY_CHARGING_ENABLED, &val);
	if (!ret)
		pdpm->charge_enabled = !!val.intval;

	return ret;
}

static int usbpd_pm_evaluate_src_caps_maxim(struct usbpd_pm *pdpm)
{
	int i;

	pdpm->pdo = usbpd_fetch_pdo();
	if (!pdpm->pdo) {
		pr_err("pdo list is null\n");
		return -ENODEV;
	}

	pdpm->apdo_max_volt = pm_config.min_adapter_volt_required;
	pdpm->apdo_max_curr = pm_config.min_adapter_curr_required;
	pdpm->prev_fixed_selected_pdo =  pdpm->fixed_selected_pdo;

	for (i = 0; i < 7; i++) {
		pr_info("[SC manager] %d type %d\n", i, pdpm->pdo[i].apdo);

		if (pdpm->pdo[i].apdo == true) {
			if (pdpm->pdo[i].max_voltage >= pdpm->apdo_max_volt
					&& pdpm->pdo[i].max_current > pdpm->apdo_max_curr) {
				pdpm->apdo_max_volt = pdpm->pdo[i].max_voltage;
				pdpm->apdo_max_curr = pdpm->pdo[i].max_current;
				pdpm->apdo_selected_pdo = i;
				pdpm->pps_supported = true;
				pr_info("[SC manager] volt %d  curr %d\n",
						pdpm->apdo_max_volt, pdpm->apdo_max_curr);
			}
		} else {
			if(pdpm->pdo[i].max_voltage == 9000){
				pdpm->fixed_selected_pdo = i;
			}
		}
	}

	if (pdpm->pps_supported)
		pr_notice("PPS supported, preferred APDO pos:%d, max volt:%d, current:%d\n",
				pdpm->apdo_selected_pdo,
				pdpm->apdo_max_volt,
				pdpm->apdo_max_curr);
	else
		pr_notice("Not qualified PPS adapter\n");

	return 0;
}

static int usbpd_update_ibat_curr(struct usbpd_pm *pdpm)
{
	int ret;
	union power_supply_propval val = {0,};

	if (!pdpm->bms_psy) {
		pdpm->bms_psy = power_supply_get_by_name("bms");
		if (!pdpm->bms_psy) {
			return -ENODEV;
		}
	}

	ret = power_supply_get_property(pdpm->bms_psy,
			POWER_SUPPLY_PROP_CURRENT_NOW, &val);
	if (!ret)
		pdpm->cp.ibat_curr_sw= -(int)(val.intval/1000);

	ret = power_supply_get_property(pdpm->bms_psy,
			POWER_SUPPLY_PROP_VOLTAGE_NOW, &val);
	if (!ret)
		pdpm->cp.vbat_volt = (int)(val.intval/1000);

	pr_info("usbpd_update_ibat_curr: ibat_curr_fg:%d vbat_volt_fg:%d\n",
		pdpm->cp.ibat_curr_sw, pdpm->cp.vbat_volt);

	return ret;
}

static const unsigned char *pm_str[] = {
	"PD_PM_STATE_PPS_READY",
	"PD_PM_STATE_PPS_START",
	"PD_PM_STATE_PPS_ENTRY_1",
	"PD_PM_STATE_PPS_ENTRY_2",
	"PD_PM_STATE_PPS_STOP"
};

static void usbpd_pm_move_state(struct usbpd_pm *pdpm, enum pm_state state)
{
	pr_debug("state change:%s -> %s\n",
		pm_str[pdpm->state], pm_str[state]);
	pdpm->state = state;
}

static int usbpd_pm_sm(struct usbpd_pm *pdpm)
{
	int ret = 0;

	pr_debug("[PPS]pm_sm state phase :%d\n", pdpm->state);
	pr_debug("[PPS]pm_sm vbus_vol:%d vbat_vol:%d ibat_curr:%d\n",
		pdpm->cp.vbus_volt, pdpm->cp.vbat_volt, pdpm->cp.ibat_curr);
	switch (pdpm->state) {
	case PD_PM_STATE_PPS_READY:
		pdpm->request_voltage = pdpm->apdo_max_volt - 20;
		pdpm->request_current = pdpm->apdo_max_curr - 20;
		usbpd_pm_move_state(pdpm, PD_PM_STATE_PPS_START);
		break;
	case PD_PM_STATE_PPS_START:
		usbpd_select_pdo(pdpm,pdpm->request_voltage, pdpm->request_current);//request APDO PDO
		usbpd_pm_move_state(pdpm, PD_PM_STATE_PPS_STOP);
		break;
	case PD_PM_STATE_PPS_ENTRY_1:
		break;
	case PD_PM_STATE_PPS_ENTRY_2:
		break;
	case PD_PM_STATE_PPS_STOP:
		ret = 1;
		break;
	}

	return ret;
}

static void usbpd_pm_workfunc(struct work_struct *work)
{
	struct usbpd_pm *pdpm = container_of(work, struct usbpd_pm, pm_work.work);

	usbpd_update_ibat_curr(pdpm);
	usbpd_pm_update_cp_status(pdpm);

	if ((!usbpd_pm_sm(pdpm)) && pdpm->pd_active)
		schedule_delayed_work(&pdpm->pm_work, msecs_to_jiffies(PM_WORK_RUN_INTERVAL));

}

static void usbpd_pm_disconnect(struct usbpd_pm *pdpm)
{
	cancel_delayed_work(&pdpm->pm_work);

	if (!pdpm->charge_enabled) {
		usbpd_pm_enable_sw(pdpm, true);
		usbpd_pm_check_sw_enabled(pdpm);
	}

	pdpm->pps_supported = false;
	pdpm->apdo_selected_pdo = 0;
	pdpm->prev_fixed_selected_pdo = 0;
	pdpm->fixed_selected_pdo = 0;
	usbpd_pm_move_state(pdpm, PD_PM_STATE_PPS_READY);
}

static void usbpd_pd_contact(struct usbpd_pm *pdpm, bool connected)
{
	int ret = 0;
	union power_supply_propval propval;
	ret = power_supply_get_property(pdpm->usb_psy, (enum power_supply_property)POWER_SUPPLY_EXT_PROP_PD_ACTIVE, &propval);
	pdpm->pd_active = connected;
	pr_debug("usbpd_pd_contact pd_active %d\n", pdpm->pd_active);
	if (connected) {
		msleep(10);
		ret = usbpd_pm_evaluate_src_caps_maxim(pdpm);
		if (ret) {
			usbpd_pm_disconnect(pdpm);
			pdpm->pd_active = 0;
			return;
		}
		pr_info("start cp1 charging pps support %d\n", pdpm->pps_supported);
		if (pdpm->pps_supported && (propval.intval == PD_PPS_TA))
			schedule_delayed_work(&pdpm->pm_work, 0);
		else{
			if(pdpm->prev_fixed_selected_pdo != pdpm->fixed_selected_pdo){
				usbpd_select_fixed_pdo_maxim(pdpm->fixed_selected_pdo); //request 9V FIXED PDO
			}
		}
	} else {
		usbpd_pm_disconnect(pdpm);
	}
}

static int usbpd_check_plugout(struct usbpd_pm *pdpm)
{
	int ret;
	union power_supply_propval val = {0,};

	ret = power_supply_get_property(pdpm->usb_psy,
			POWER_SUPPLY_PROP_PRESENT, &val);

	return ret;
}

static void usb_psy_change_work(struct work_struct *work)
{
	int ret = 0;
	union power_supply_propval propval;
	struct usbpd_pm *pdpm = container_of(work, struct usbpd_pm,
					usb_psy_change_work);

	usbpd_check_plugout(pdpm);
	ret = power_supply_get_property(pdpm->usb_psy, (enum power_supply_property)POWER_SUPPLY_EXT_PROP_PD_ACTIVE, &propval);

	pr_debug("pre_pd_active %d,now:%d\n",pdpm->pd_active, propval.intval);

	if (!pdpm->pd_active && propval.intval == PD_NONPPS_TA)
		usbpd_pd_contact(pdpm, true);
	else if (!pdpm->pd_active && propval.intval == PD_PPS_TA)
		usbpd_pd_contact(pdpm, true);
	else if (pdpm->pd_active && !propval.intval)
		usbpd_pd_contact(pdpm, false);

	pdpm->psy_change_running = false;
}

static int usbpd_psy_notifier_cb(struct notifier_block *nb,
			unsigned long event, void *data)
{
	struct usbpd_pm *pdpm = container_of(nb, struct usbpd_pm, nb);
	struct power_supply *psy = data;
	unsigned long flags;

	pr_info("usbpd_psy_notifier_cb start\n");

	if (event != PSY_EVENT_PROP_CHANGED)
		return NOTIFY_OK;

	usbpd_check_charger_psy(pdpm);

	if (psy == pdpm->usb_psy) {
		spin_lock_irqsave(&pdpm->psy_change_lock, flags);
		pr_info("[SC manager] >>>pdpm->psy_change_running : %d\n", pdpm->psy_change_running);
		if (!pdpm->psy_change_running) {
			pdpm->psy_change_running = true;
				schedule_work(&pdpm->usb_psy_change_work);
		}
		spin_unlock_irqrestore(&pdpm->psy_change_lock, flags);
	}

	return NOTIFY_OK;
}

int usbpd_pm_init(void)
{
	struct usbpd_pm *pdpm;
	int ret = 0;
	static int probe_cnt = 0;
	pr_info("usbpd_pm_init start: probe_cnt:%d\n", ++probe_cnt);

	pdpm = kzalloc(sizeof(*pdpm), GFP_KERNEL);
	if (!pdpm)
		return -ENOMEM;

	__pdpm = pdpm;

	spin_lock_init(&pdpm->psy_change_lock);

	usbpd_check_charger_psy(pdpm);

	if(!pdpm->usb_psy){
		ret = -ENODEV;
		return ret;
	}

	INIT_WORK(&pdpm->usb_psy_change_work, usb_psy_change_work);
	INIT_DELAYED_WORK(&pdpm->pm_work, usbpd_pm_workfunc);

	pdpm->nb.notifier_call = usbpd_psy_notifier_cb;
	power_supply_reg_notifier(&pdpm->nb);
	pr_info("usbpd_pm_init end\n");
	return 0;
}

void usbpd_pm_exit(void)
{
	power_supply_unreg_notifier(&__pdpm->nb);
	cancel_delayed_work(&__pdpm->pm_work);
	cancel_work_sync(&__pdpm->usb_psy_change_work);
}
