/*
 * (C) 2022 FCNT LIMITED
 * Copyrights (C) 2021 Maxim Integrated Products, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/uaccess.h>
#include <linux/string.h>
#ifdef CONFIG_OF
#include <linux/of_gpio.h>
#endif
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/mfd/core.h>
#include <linux/mfd/max77729.h>
#include <linux/mfd/max77729-private.h>
#include <linux/usb/typec/maxim/max77729-muic.h>
#include <linux/usb/typec/maxim/max77729_usbc.h>
#include <linux/usb/typec/maxim/max77729_alternate.h>
#include <linux/firmware.h>

#define DRIVER_VER		"1.0VER"

#define MAX77729_MAX_APDCMD_TIME (10*HZ)

#define MAX77729_PMIC_REG_INTSRC_MASK 0x23
#define MAX77729_PMIC_REG_INTSRC 0x22

#define MAX77729_IRQSRC_CHG	(1 << 0)
#define MAX77729_IRQSRC_FG      (1 << 2)
#define MAX77729_IRQSRC_MUIC	(1 << 3)

static const unsigned int extcon_cable[] = {
	EXTCON_USB,
	EXTCON_USB_HOST,
	EXTCON_NONE,
};

struct max77729_usbc_platform_data *g_usbc_data;
EXPORT_SYMBOL_GPL(g_usbc_data);
int vdm_count;

#ifdef MAX77729_SYS_FW_UPDATE
#define MAXIM_DEFAULT_FW		"secure_max77729.bin"
#define MAXIM_SPU_FW			"/maxim/pdic_fw.bin"

struct pdic_fw_update {
	char id[10];
	char path[50];
	int need_verfy;
	int enforce_do;
};
#endif

static void max77729_usbc_mask_irq(struct max77729_usbc_platform_data *usbc_data);
static void max77729_usbc_umask_irq(struct max77729_usbc_platform_data *usbc_data);
static void max77729_get_version_info(struct max77729_usbc_platform_data *usbc_data);
void max77729_rprd_mode_change(struct max77729_usbc_platform_data *usbpd_data, u8 mode);

void max77729_set_water_disable_reg(u8 data)
{
	struct max77729_usbc_platform_data *usbc_data = g_usbc_data;
	usbc_cmd_data write_data;

	if (usbc_data->water_disable_state == data)
		return;

	init_usbc_cmd_data(&write_data);
	write_data.opcode = OPCODE_SET_WATER_DIS;
	write_data.write_data[0] = data;
	write_data.write_length = 0x1;
	write_data.read_length = 0x1;
	max77729_usbc_opcode_write(usbc_data, &write_data);

	usbc_data->water_disable_state = data;
}
EXPORT_SYMBOL(max77729_set_water_disable_reg);

void max77729_set_ccx_open_set_reg(u8 data)
{
	struct max77729_usbc_platform_data *usbc_data = g_usbc_data;
	usbc_cmd_data write_data;

	if (usbc_data->cc_open_state == data)
		return;

	init_usbc_cmd_data(&write_data);
	write_data.opcode = OPCODE_FW_CC_OPEN;
	write_data.write_data[0] = data;
	write_data.write_length = 0x1;
	write_data.read_length = 0x1;
	max77729_usbc_opcode_write(usbc_data, &write_data);

	usbc_data->cc_open_state = data;
}
EXPORT_SYMBOL(max77729_set_ccx_open_set_reg);

static void max77729_send_role_swap_message(struct max77729_usbc_platform_data *usbpd_data, u8 mode)
{
	usbc_cmd_data write_data;

	max77729_usbc_clear_queue(usbpd_data);
	init_usbc_cmd_data(&write_data);
	write_data.opcode = 0x37;
	/* 0x1 : DR_SWAP, 0x2 : PR_SWAP*/
	write_data.write_data[0] = mode;
	write_data.write_length = 0x1;
	write_data.read_length = 0x1;
	max77729_usbc_opcode_write(usbpd_data, &write_data);
}

void max77729_power_role_change(struct max77729_usbc_platform_data *usbpd_data, int power_role)
{
	msg_maxim("power_role = 0x%x", power_role);

	switch (power_role) {
	case TYPE_C_ATTACH_SRC:
	case TYPE_C_ATTACH_SNK:
		max77729_send_role_swap_message(usbpd_data, POWER_ROLE_SWAP);
		break;
	};
}

void max77729_data_role_change(struct max77729_usbc_platform_data *usbpd_data, int data_role)
{
	msg_maxim("data_role = 0x%x", data_role);

	switch (data_role) {
	case TYPE_C_ATTACH_DFP:
	case TYPE_C_ATTACH_UFP:
		max77729_send_role_swap_message(usbpd_data, DATA_ROLE_SWAP);
		break;
	};
}
static int max77729_pr_set(struct typec_port *port, enum typec_role role)
{
	struct max77729_usbc_platform_data *usbpd_data = (struct max77729_usbc_platform_data *)typec_get_drvdata(port);

	if (!usbpd_data)
		return -EINVAL;

	msg_maxim("typec_power_role=%d, typec_data_role=%d, role=%d",
		usbpd_data->typec_power_role, usbpd_data->typec_data_role, role);

	if (usbpd_data->typec_power_role != TYPEC_SINK
	    && usbpd_data->typec_power_role != TYPEC_SOURCE)
		return -EPERM;
	else if (usbpd_data->typec_power_role == role)
		return -EPERM;

	reinit_completion(&usbpd_data->typec_reverse_completion);
	if (role == TYPEC_SINK) {
		msg_maxim("try reversing, from Source to Sink");
		usbpd_data->typec_try_state_change = TRY_ROLE_SWAP_PR;
		max77729_power_role_change(usbpd_data, TYPE_C_ATTACH_SNK);
	} else if (role == TYPEC_SOURCE) {
		msg_maxim("try reversing, from Sink to Source");
		usbpd_data->typec_try_state_change = TRY_ROLE_SWAP_PR;
		max77729_power_role_change(usbpd_data, TYPE_C_ATTACH_SRC);
	} else {
		msg_maxim("invalid typec_role");
		return -EIO;
	}
	if (!wait_for_completion_timeout(&usbpd_data->typec_reverse_completion,
				msecs_to_jiffies(TRY_ROLE_SWAP_WAIT_MS))) {
		usbpd_data->typec_try_state_change = TRY_ROLE_SWAP_NONE;
		if (usbpd_data->typec_power_role != role)
		return -ETIMEDOUT;
	}
	return 0;
}

static int max77729_dr_set(struct typec_port *port, enum typec_data_role role)
{
	struct max77729_usbc_platform_data *usbpd_data = (struct max77729_usbc_platform_data *)typec_get_drvdata(port);

	if (!usbpd_data)
		return -EINVAL;
	msg_maxim("typec_power_role=%d, typec_data_role=%d, role=%d",
			usbpd_data->typec_power_role, usbpd_data->typec_data_role, role);

	if (usbpd_data->typec_data_role != TYPEC_DEVICE
			&& usbpd_data->typec_data_role != TYPEC_HOST)
		return -EPERM;
	else if (usbpd_data->typec_data_role == role)
		return -EPERM;

	reinit_completion(&usbpd_data->typec_reverse_completion);
	if (role == TYPEC_DEVICE) {
		msg_maxim("try reversing, from DFP to UFP");
		usbpd_data->typec_try_state_change = TRY_ROLE_SWAP_DR;
		max77729_data_role_change(usbpd_data, TYPE_C_ATTACH_UFP);
	} else if (role == TYPEC_HOST) {
		msg_maxim("try reversing, from UFP to DFP");
		usbpd_data->typec_try_state_change = TRY_ROLE_SWAP_DR;
		max77729_data_role_change(usbpd_data, TYPE_C_ATTACH_DFP);
	} else {
		msg_maxim("invalid typec_role");
		return -EIO;
	}
	if (!wait_for_completion_timeout(&usbpd_data->typec_reverse_completion,
				msecs_to_jiffies(TRY_ROLE_SWAP_WAIT_MS))) {
		usbpd_data->typec_try_state_change = TRY_ROLE_SWAP_NONE;
		return -ETIMEDOUT;
	}
	return 0;
}

static void max77729_request_response(struct max77729_usbc_platform_data *usbc_data)
{
	usbc_cmd_data value;

	init_usbc_cmd_data(&value);
	value.opcode = OPCODE_READ_RESPONSE_FOR_GET_REQUEST;
	value.write_data[0]= 0x00;
	value.write_length = 0x01;
	value.read_length = 31;
	max77729_usbc_opcode_push(usbc_data, &value);
	pr_debug("%s : OPCODE(0x%02x) W_LENGTH(%d) R_LENGTH(%d)\n",
		__func__, value.opcode, value.write_length, value.read_length);
}

void max77729_extend_msg_process(struct max77729_usbc_platform_data *usbc_data, unsigned char *data,
		unsigned char len)
{
	unsigned short vid=0x0;
	unsigned short pid=0x0;
	unsigned short xid=0x0;
	vid = *(unsigned short *)(data + 2);
	pid = *(unsigned short *)(data + 4);
	xid = *(unsigned int *)(data + 6);
	usbc_data->adapter_svid	= vid;
	usbc_data->adapter_id	= pid;
	usbc_data->xid	= xid;
	msg_maxim("%s, %04x, %04x, %08x",
		__func__, vid, pid, xid);
}

void max77729_read_response(struct max77729_usbc_platform_data *usbc_data, unsigned char *data)
{
	switch (data[1] >> 5) {
	case OPCODE_GET_SRC_CAP_EXT:
		max77729_extend_msg_process(usbc_data, data+2, data[1] & 0x1F);
		break;
	case OPCODE_GET_STATUS:
	case OPCODE_GET_BAT_CAP:
	case OPCODE_GET_BAT_STS:
		break;
	default:
		break;
	}
}

int max77729_current_pr_state(struct max77729_usbc_platform_data *usbc_data)
{
	int current_pr = usbc_data->cc_data->current_pr;
	return current_pr;
}

static void vbus_control_hard_reset(struct work_struct *work)
{
	struct max77729_usbc_platform_data *usbpd_data = g_usbc_data;

	msg_maxim("current_pr=%d", usbpd_data->cc_data->current_pr);

	if (usbpd_data->cc_data->current_pr == SRC)
		max77729_vbus_turn_on_ctrl(usbpd_data, ON, false);
}

static void max77729_usb_accessory_check_work(struct work_struct *work)
{
	struct max77729_usbc_platform_data *usbpd_data = g_usbc_data;
	union power_supply_propval pval={0, };
	int rc = 0;

	if (!usbpd_data->usb_psy) {
		usbpd_data->usb_psy = power_supply_get_by_name("usb");

		if (!usbpd_data->usb_psy) {
			goto error;
		}
	}

	if (usbpd_data->usb_psy) {
		pval.intval = 0;
		rc = power_supply_get_property(usbpd_data->usb_psy,
			(enum power_supply_property)POWER_SUPPLY_EXT_PROP_USB_ACCESSORY, &pval);

		if (pval.intval < 0 || rc < 0)
			goto error;

		if (pval.intval != usbpd_data->accessory) {
			pval.intval = usbpd_data->accessory;
			rc = power_supply_set_property(usbpd_data->usb_psy,
				(enum power_supply_property)POWER_SUPPLY_EXT_PROP_USB_ACCESSORY, &pval);

			if (rc < 0)
				goto error;

			power_supply_changed(usbpd_data->usb_psy);
			pr_info("%s: Set usb accessory val:%d\n", __func__, pval.intval);
		}
	}

	return;
error:
	pr_err("%s: usb_psy error val:%d rc:%d\n", __func__, pval.intval, rc);

	schedule_delayed_work(&usbpd_data->usb_accessory_set_work, msecs_to_jiffies(1000));
	return;
}


void max77729_usbc_enable_audio(struct max77729_usbc_platform_data *usbc_data)
{
	usbc_cmd_data write_data;

	usbc_data->op_ctrl1_w |= (BIT_CCDbgEn | BIT_CCAudEn);

	init_usbc_cmd_data(&write_data);
	write_data.opcode = OPCODE_CCCTRL1_W;
	write_data.write_data[0] = usbc_data->op_ctrl1_w;
	write_data.write_length = 0x1;
	write_data.read_length = 0x1;
	max77729_usbc_opcode_write(usbc_data, &write_data);
	msg_maxim("Enable Audio Detect");
}

int max77729_get_pd_support(struct max77729_usbc_platform_data *usbc_data)
{
	bool support_pd_role_swap = false;
	struct device_node *np = NULL;

	np = of_find_compatible_node(NULL, NULL, "maxim,max77729-usbc");

	if (np)
		support_pd_role_swap = of_property_read_bool(np, "support_pd_role_swap");
	else{
		msg_maxim("np is null so set the pd_role_swap==false");
	}
	msg_maxim("TYPEC_CLASS: support_pd_role_swap is %d, usbc_data->pd_support : %d",
		support_pd_role_swap, usbc_data->pd_support);

	if (support_pd_role_swap && usbc_data->pd_support)
		return TYPEC_PWR_MODE_PD;

	return usbc_data->pwr_opmode;
}

static int max77729_firmware_update_sys(struct max77729_usbc_platform_data *data, int fw_dir)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	const struct firmware *fw_entry;
	int fw_size, ret = 0;

	if (!usbc_data) {
		msg_maxim("usbc_data is null!!");
		return -ENODEV;
	}

	ret = request_firmware(&fw_entry, MAXIM_SPU_FW, usbc_data->dev);
	if (ret) {
		pr_info("%s: firmware is not available %d\n", __func__, ret);
		return ret;
	}

	fw_size = (int)fw_entry->size;
	ret = max77729_usbc_fw_update(usbc_data->max77729, fw_entry->data,
				fw_size, 1);
	release_firmware(fw_entry);
	return ret;
}

#if defined(MAX77729_SYS_FW_UPDATE)
static int max77729_firmware_update_sysfs(struct max77729_usbc_platform_data *usbpd_data, int fw_dir)
{
	int ret = 0;
	usbpd_data->fw_update = 1;
	max77729_usbc_mask_irq(usbpd_data);
	max77729_write_reg(usbpd_data->muic, REG_PD_INT_M, 0xFF);
	max77729_write_reg(usbpd_data->muic, REG_CC_INT_M, 0xFF);
	max77729_write_reg(usbpd_data->muic, REG_UIC_INT_M, 0xFF);
	max77729_write_reg(usbpd_data->muic, REG_VDM_INT_M, 0xFF);
	ret = max77729_firmware_update_sys(usbpd_data, fw_dir);
	max77729_write_reg(usbpd_data->muic, REG_UIC_INT_M, REG_UIC_INT_M_INIT);
	max77729_write_reg(usbpd_data->muic, REG_CC_INT_M, REG_CC_INT_M_INIT);
	max77729_write_reg(usbpd_data->muic, REG_PD_INT_M, REG_PD_INT_M_INIT);
	max77729_write_reg(usbpd_data->muic, REG_VDM_INT_M, REG_VDM_INT_M_INIT);
	max77729_set_enable_alternate_mode(ALTERNATE_MODE_START);
	max77729_usbc_umask_irq(usbpd_data);
	if (ret)
		usbpd_data->fw_update = 2;
	else
		usbpd_data->fw_update = 0;
	return ret;
}
#endif

#if defined(MAX77729_SYS_FW_UPDATE)
static void max77729_firmware_update_sysfs_work(struct work_struct *work)
{
	struct max77729_usbc_platform_data *usbpd_data = container_of(work,
			struct max77729_usbc_platform_data, fw_update_work);

	max77729_firmware_update_sysfs(usbpd_data, 1);
}
#endif

static ssize_t max77729_get_fw_version(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct max77729_usbc_platform_data *usbc_data = g_usbc_data;
	int result;

	max77729_get_version_info(usbc_data);
	result = sprintf(buf, "%02X.%02X\n", usbc_data->FW_Revision, usbc_data->FW_Minor_Revision);
	return result;
}
static DEVICE_ATTR(fw_version, S_IRUSR | S_IRGRP,
		max77729_get_fw_version, NULL);

static ssize_t max77729_fw_update(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t size)
{
	unsigned int start_fw_update = 0;

	if (kstrtou32(buf, 0, &start_fw_update)) {
		dev_err(dev,
			"%s: Failed converting from str to u32.", __func__);
	}

	msg_maxim("start_fw_update %d", start_fw_update);
	switch (start_fw_update) {
	case 1:
		max77729_firmware_update_sysfs(g_usbc_data, 1);
		break;
	}
	return size;
}
static DEVICE_ATTR(fw_update, S_IRUGO | S_IWUSR | S_IWGRP,
		NULL, max77729_fw_update);

static ssize_t pdo_n_show(struct device *dev, struct device_attribute *attr,
		char *buf);

#define PDO_ATTR(n) {					\
	.attr	= { .name = __stringify(pdo##n), .mode = 0444 },	\
	.show	= pdo_n_show,				\
}

static struct device_attribute dev_attr_pdos[] = {
	PDO_ATTR(1),
	PDO_ATTR(2),
	PDO_ATTR(3),
	PDO_ATTR(4),
	PDO_ATTR(5),
	PDO_ATTR(6),
	PDO_ATTR(7),
};
static ssize_t pdo_n_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct max77729_usbc_platform_data *usbpd_data = g_usbc_data;
	int i;

	for (i = 0; i < ARRAY_SIZE(dev_attr_pdos); i++) {
		if (attr == &dev_attr_pdos[i])
			/* dump the PDO as a hex string */
			return snprintf(buf, PAGE_SIZE, "%08x\n",
				usbpd_data->received_pdos[i]);
	}

	dev_err(dev, "%s: Invalid PDO index\n", __func__);
	return -EINVAL;
}

static struct attribute *max77729_attr[] = {
	&dev_attr_pdos[0].attr,
	&dev_attr_pdos[1].attr,
	&dev_attr_pdos[2].attr,
	&dev_attr_pdos[3].attr,
	&dev_attr_pdos[4].attr,
	&dev_attr_pdos[5].attr,
	&dev_attr_pdos[6].attr,
	&dev_attr_fw_update.attr,
	&dev_attr_fw_version.attr,
	NULL,
};

static struct attribute_group max77729_groups = {
	.attrs = max77729_attr,
};

static void max77729_get_version_info(struct max77729_usbc_platform_data *usbc_data)
{
	u8 hw_rev[4] = {0, };
	u8 sw_main[3] = {0, };

	max77729_read_reg(usbc_data->muic, REG_UIC_HW_REV, &hw_rev[0]);
	max77729_read_reg(usbc_data->muic, REG_UIC_FW_MINOR, &sw_main[1]);
	max77729_read_reg(usbc_data->muic, REG_UIC_FW_REV, &sw_main[0]);

	usbc_data->HW_Revision = hw_rev[0];
	usbc_data->FW_Minor_Revision = sw_main[1] & MINOR_VERSION_MASK;
	usbc_data->FW_Revision = sw_main[0];

	/* H/W, Minor, Major, Boot */
	msg_maxim("HW rev is %02Xh, FW rev is %02X.%02X!",
			usbc_data->HW_Revision, usbc_data->FW_Revision, usbc_data->FW_Minor_Revision);
}

void max77729_usbc_disable_auto_vbus(struct max77729_usbc_platform_data *usbc_data)
{
	usbc_cmd_data write_data;

	init_usbc_cmd_data(&write_data);
	write_data.opcode = 0x54;
	write_data.write_data[0] = 0x0;
	write_data.write_length = 0x1;
	write_data.read_length = 0x1;
	max77729_usbc_opcode_write(usbc_data, &write_data);
	msg_maxim("TURN OFF THE AUTO VBUS");
	usbc_data->auto_vbus_en = false;
}

static void max77729_init_opcode
		(struct max77729_usbc_platform_data *usbc_data, int reset)
{
	struct max77729_platform_data *pdata = usbc_data->max77729_data;

	max77729_usbc_disable_auto_vbus(usbc_data);

	if (pdata && pdata->support_audio)
		max77729_usbc_enable_audio(usbc_data);
	if (reset)
		max77729_set_enable_alternate_mode(ALTERNATE_MODE_START | ALTERNATE_MODE_READY);
}

static bool max77729_check_recover_opcode(u8 opcode)
{
	bool ret = false;

	switch (opcode) {
	case OPCODE_SET_ALTERNATEMODE:
		ret = true;
		break;
	default:
		ret = false;
		break;
	}
	return ret;
}

static void max77729_recover_opcode
		(struct max77729_usbc_platform_data *usbc_data, bool opcode_list[])
{
	int i;

	for (i = 0; i < OPCODE_NONE; i++) {
		if (opcode_list[i]) {
			msg_maxim("opcode = 0x%02x", i);
			switch (i) {
			case OPCODE_SET_ALTERNATEMODE:
				max77729_set_enable_alternate_mode
					(usbc_data->set_altmode);
				break;
			default:
				break;
			}
			opcode_list[i] = false;
		}
	}
}

void init_usbc_cmd_data(usbc_cmd_data *cmd_data)
{
	cmd_data->opcode = OPCODE_NONE;
	cmd_data->prev_opcode = OPCODE_NONE;
	cmd_data->response = OPCODE_NONE;
	cmd_data->val = REG_NONE;
	cmd_data->mask = REG_NONE;
	cmd_data->reg = REG_NONE;
	cmd_data->noti_cmd = OPCODE_NOTI_NONE;
	cmd_data->write_length = 0;
	cmd_data->read_length = 0;
	cmd_data->seq = 0;
	cmd_data->is_uvdm = 0;
	cmd_data->debug_command_flg = false;
	memset(cmd_data->write_data, REG_NONE, OPCODE_DATA_LENGTH);
	memset(cmd_data->read_data, REG_NONE, OPCODE_DATA_LENGTH);
}

static void init_usbc_cmd_node(usbc_cmd_node *usbc_cmd_node)
{
	usbc_cmd_data *cmd_data = &(usbc_cmd_node->cmd_data);

	pr_debug("%s:%s\n", "MAX77729", __func__);

	usbc_cmd_node->next = NULL;

	init_usbc_cmd_data(cmd_data);
}

static void copy_usbc_cmd_data(usbc_cmd_data *from, usbc_cmd_data *to)
{
	to->opcode = from->opcode;
	to->response = from->response;
	memcpy(to->read_data, from->read_data, OPCODE_DATA_LENGTH);
	memcpy(to->write_data, from->write_data, OPCODE_DATA_LENGTH);
	to->reg = from->reg;
	to->mask = from->mask;
	to->val = from->val;
	to->seq = from->seq;
	to->read_length = from->read_length;
	to->write_length = from->write_length;
	to->prev_opcode = from->prev_opcode;
	to->is_uvdm = from->is_uvdm;
	to->debug_command_flg = from->debug_command_flg;
}

bool is_empty_usbc_cmd_queue(usbc_cmd_queue_t *usbc_cmd_queue)
{
	bool ret = false;

	if (usbc_cmd_queue->front == NULL)
		ret = true;

	if (ret)
		msg_maxim("usbc_cmd_queue Empty(%c)", ret ? 'T' : 'F');

	return ret;
}

void enqueue_usbc_cmd(usbc_cmd_queue_t *usbc_cmd_queue, usbc_cmd_data *cmd_data)
{
	usbc_cmd_node	*temp_node = kzalloc(sizeof(usbc_cmd_node), GFP_KERNEL);

	if (!temp_node) {
		msg_maxim("failed to allocate usbc command queue");
		return;
	}

	init_usbc_cmd_node(temp_node);

	copy_usbc_cmd_data(cmd_data, &(temp_node->cmd_data));

	if (is_empty_usbc_cmd_queue(usbc_cmd_queue)) {
		usbc_cmd_queue->front = temp_node;
		usbc_cmd_queue->rear = temp_node;
	} else {
		usbc_cmd_queue->rear->next = temp_node;
		usbc_cmd_queue->rear = temp_node;
	}

	if (g_usbc_data && g_usbc_data->max77729)
		g_usbc_data->max77729->is_usbc_queue = 1;
}

static void dequeue_usbc_cmd
	(usbc_cmd_queue_t *usbc_cmd_queue, usbc_cmd_data *cmd_data)
{
	usbc_cmd_node *temp_node;

	if (is_empty_usbc_cmd_queue(usbc_cmd_queue)) {
		msg_maxim("Queue, Empty!");
		return;
	}

	temp_node = usbc_cmd_queue->front;
	copy_usbc_cmd_data(&(temp_node->cmd_data), cmd_data);

	msg_maxim("Opcode(0x%02x) Response(0x%02x)", cmd_data->opcode, cmd_data->response);

	if (usbc_cmd_queue->front->next == NULL) {
		msg_maxim("front->next = NULL");
		usbc_cmd_queue->front = NULL;
	} else
		usbc_cmd_queue->front = usbc_cmd_queue->front->next;

	if (is_empty_usbc_cmd_queue(usbc_cmd_queue))
		usbc_cmd_queue->rear = NULL;

	kfree(temp_node);
}

static bool front_usbc_cmd
	(usbc_cmd_queue_t *cmd_queue, usbc_cmd_data *cmd_data)
{
	if (is_empty_usbc_cmd_queue(cmd_queue)) {
		msg_maxim("Queue, Empty!");
		return false;
	}

	copy_usbc_cmd_data(&(cmd_queue->front->cmd_data), cmd_data);
	msg_maxim("Opcode(0x%02x)", cmd_data->opcode);
	return true;
}

static bool is_usbc_notifier_opcode(u8 opcode)
{
	bool noti = false;

	return noti;
}

bool check_usbc_opcode_queue(void)
{
	struct max77729_usbc_platform_data *usbpd_data = g_usbc_data;
	usbc_cmd_queue_t *cmd_queue = NULL;
	bool ret = true;

	if (usbpd_data == NULL)
		goto err;

	cmd_queue = &(usbpd_data->usbc_cmd_queue);

	if (cmd_queue == NULL)
		goto err;

	ret = is_empty_usbc_cmd_queue(cmd_queue);

err:
	return ret;
}
EXPORT_SYMBOL(check_usbc_opcode_queue);

int max77729_i2c_opcode_write(struct max77729_usbc_platform_data *usbc_data,
		u8 opcode, u8 length, u8 *values)
{
	u8 write_values[OPCODE_MAX_LENGTH] = { 0, };
	int ret = 0;

	if (length > OPCODE_DATA_LENGTH)
		return -EMSGSIZE;

	write_values[0] = opcode;
	if (length)
		memcpy(&write_values[1], values, length);

	msg_maxim("opcode 0x%x, write_length %d",
			opcode, length + OPCODE_SIZE);
	print_hex_dump(KERN_ERR, "max77729: opcode_write: ",
			DUMP_PREFIX_OFFSET, 16, 1, write_values,
			length + OPCODE_SIZE, false);
	/* Write opcode and data */
	ret = max77729_bulk_write(usbc_data->muic, OPCODE_WRITE,
			length + OPCODE_SIZE, write_values);
	/* Write end of data by 0x00 */
	if (length < OPCODE_DATA_LENGTH)
		max77729_write_reg(usbc_data->muic, OPCODE_WRITE_END, 0x00);

	if (opcode == OPCODE_SET_ALTERNATEMODE)
		usbc_data->set_altmode_error = ret;

	if (ret == 0)
		usbc_data->opcode_stamp = jiffies;

	return ret;
}

int max77729_i2c_opcode_read(struct max77729_usbc_platform_data *usbc_data,
		u8 opcode, u8 length, u8 *values)
{
	int size = 0;

	if (length > OPCODE_DATA_LENGTH)
		return -EMSGSIZE;

	/* Read opcode data */
	size = max77729_bulk_read(usbc_data->muic, OPCODE_READ,
			length + OPCODE_SIZE, values);

	msg_maxim("opcode 0x%x, read_length %d, ret_error %d",
			opcode, length + OPCODE_SIZE, size);
	print_hex_dump(KERN_ERR, "max77729: opcode_read: ",
			DUMP_PREFIX_OFFSET, 16, 1, values,
			length + OPCODE_SIZE, false);
	return size;
}

static void max77729_notify_execute(struct max77729_usbc_platform_data *usbc_data,
		const usbc_cmd_data *cmd_data)
{

}

static void max77729_handle_update_opcode(struct max77729_usbc_platform_data *usbc_data,
		const usbc_cmd_data *cmd_data, unsigned char *data)
{
	usbc_cmd_data write_data;
	u8 read_value = data[1];
	u8 write_value = (read_value & (~cmd_data->mask)) | (cmd_data->val & cmd_data->mask);
	u8 opcode = cmd_data->response + 1; /* write opcode = read opocde + 1 */

	pr_debug("%s: value update [0x%x]->[0x%x] at OPCODE(0x%x)\n", __func__,
			read_value, write_value, opcode);

	init_usbc_cmd_data(&write_data);
	write_data.opcode = opcode;
	write_data.write_length = 1;
	write_data.write_data[0] = write_value;
	write_data.read_length = 0;

	max77729_usbc_opcode_push(usbc_data, &write_data);
}

void max77729_handle_qc_result(struct max77729_muic_data *muic_data, unsigned char *data)
{
	int result = data[1];
	union power_supply_propval pvalue ={0,};
	pr_debug("%s:%s result:0x%x vbadc:0x%x\n", MUIC_DEV_NAME,
			__func__, data[1], data[2]);

	switch (result) {
	case 0:
		pr_info("%s:%s QC2.0 Success\n", MUIC_DEV_NAME, __func__);
		g_usbc_data->is_hvdcp = true;
		pvalue.intval = POWER_SUPPLY_TYPE_USB_HVDCP;
		psy_do_property("usb", set, (enum power_supply_property)POWER_SUPPLY_EXT_PROP_REAL_TYPE, pvalue);
		break;
	case 1:
		pr_info("%s:%s No CHGIN\n", MUIC_DEV_NAME, __func__);
		break;
	case 2:
		pr_info("%s:%s Not High Voltage DCP\n",
				MUIC_DEV_NAME, __func__);
		break;
	case 3:
		pr_info("%s:%s Not DCP\n", MUIC_DEV_NAME, __func__);
		break;
	case 6:
		pr_info("%s:%s Vbus is not changed with 3 continuous ping\n",
				MUIC_DEV_NAME, __func__);
		break;
	case 7:
		pr_info("%s:%s Vbus is not changed in 1 sec\n",
				MUIC_DEV_NAME, __func__);
		break;
	default:
		pr_info("%s:%s QC2.0 error(%d)\n", MUIC_DEV_NAME, __func__, result);
		break;
	}
}

static void max77729_irq_execute(struct max77729_usbc_platform_data *usbc_data,
		const usbc_cmd_data *cmd_data)
{
	int len = cmd_data->read_length;
	unsigned char data[OPCODE_MAX_LENGTH] = {0,};
	u8 response = 0xff;
	u8 vdm_opcode_header = 0x0;
	UND_DATA_MSG_VDM_HEADER_Type vdm_header;
	u8 vdm_command = 0x0;
	u8 vdm_type = 0x0;
	u8 vdm_response = 0x0;
	u8 reqd_vdm_command = 0;
	uint8_t W_DATA = 0x0;
	u8 result = 0x0;

	memset(&vdm_header, 0, sizeof(UND_DATA_MSG_VDM_HEADER_Type));
	max77729_i2c_opcode_read(usbc_data, cmd_data->opcode,
			len, data);

	/* opcode identifying the messsage type. (0x51)*/
	response = data[0];

	if (response != cmd_data->response) {
		msg_maxim("Response [0x%02x] != [0x%02x]",
			response, cmd_data->response);
		if (cmd_data->response == OPCODE_FW_OPCODE_CLEAR) {
			msg_maxim("Response after FW opcode cleared, just return");
			return;
		}
	}

	if (cmd_data->debug_command_flg) {
		pr_debug("%s: Debug Command Response [0x%02x]", __func__, response);
		memcpy(usbc_data->debug_read_data, data, OPCODE_DATA_LENGTH);
	}

	switch (response) {
	case OPCODE_BCCTRL1_R:
	case OPCODE_BCCTRL2_R:
	case OPCODE_CTRL1_R:
	case OPCODE_CTRL2_R:
	case OPCODE_CTRL3_R:
	case OPCODE_CCCTRL1_R:
	case OPCODE_CCCTRL2_R:
	case OPCODE_CCCTRL3_R:
	case OPCODE_HVCTRL_R:
	case OPCODE_OPCODE_VCONN_ILIM_R:
	case OPCODE_CHGIN_ILIM_R:
	case OPCODE_CHGIN_ILIM2_R:
		if (cmd_data->seq == OPCODE_UPDATE_SEQ)
			max77729_handle_update_opcode(usbc_data, cmd_data, data);
		break;
	case OPCODE_CURRENT_SRCCAP:
		max77729_current_pdo(usbc_data, data);
		break;
	case OPCODE_GET_SRCCAP:
		max77729_current_pdo(usbc_data, data);
		break;
	case OPCODE_READ_RESPONSE_FOR_GET_REQUEST:
		max77729_read_response(usbc_data, data);
		break;
	case OPCODE_SEND_GET_REQUEST:
		result = data[1];
		switch(result){
			case SENT_REQ_MSG:
			    msg_maxim("sucess to send");
				max77729_request_response(usbc_data);
				break;
			case ERR_SNK_RDY:
			    msg_maxim(" Not in Snk Ready");
				break;
			case ERR_PD20:
				msg_maxim(" PD 2.0");
				break;
			case ERR_SNKTXNG:
				msg_maxim(" SinkTxNG");
				break;
			default:
				msg_maxim("OPCODE_SEND_GET_REQUEST = [%x]",result);
				break;
		}
		break;

	case OPCODE_SRCCAP_REQUEST:
		/*
		 * If response of Source_Capablities message is SinkTxNg(0xFE) or Not in Ready State(0xFF)
		 * It means that the message can not be sent to Port Partner.
		 * After Attaching Rp 3.0A, send again the message.
		 */
		if (data[1] == 0xfe || data[1] == 0xff){
			usbc_data->srcccap_request_retry = true;
			pr_info("%s : srcccap_request_retry is set\n", __func__);
		}
		break;
	case OPCODE_APDO_SRCCAP_REQUEST:
		max77729_response_apdo_request(usbc_data, data);
		break;
	case OPCODE_SET_PPS:
		max77729_response_set_pps(usbc_data, data);
		break;
	case OPCODE_READ_MESSAGE:
		pr_info("@TA_ALERT: %s : OPCODE[%x] Data[1] = 0x%x Data[7] = 0x%x Data[9] = 0x%x\n",
			__func__, OPCODE_READ_MESSAGE, data[1], data[7], data[9]);

		break;
	case OPCODE_VDM_DISCOVER_GET_VDM_RESP:
		max77729_vdm_message_handler(usbc_data, data, len + OPCODE_SIZE);
		break;
	case OPCODE_VDM_DISCOVER_SET_VDM_REQ:
		vdm_opcode_header = data[1];
		switch (vdm_opcode_header) {
		case 0xFF:
			msg_maxim("This isn't invalid response(OPCODE : 0x48, HEADER : 0xFF)");
			break;
		default:
			memcpy(&vdm_header, &data[2], sizeof(vdm_header));
			vdm_type = vdm_header.BITS.VDM_Type;
			vdm_command = vdm_header.BITS.VDM_command;
			vdm_response = vdm_header.BITS.VDM_command_type;
			msg_maxim("vdm_type[%x], vdm_command[%x], vdm_response[%x]",
				vdm_type, vdm_command, vdm_response);
			switch (vdm_type) {
			case STRUCTURED_VDM:
				if (vdm_response == SEC_UVDM_RESPONDER_ACK) {
					switch (vdm_command) {
					case Discover_Identity:
						msg_maxim("ignore Discover_Identity");
						break;
					case Discover_SVIDs:
						msg_maxim("ignore Discover_SVIDs");
						break;
					case Discover_Modes:
						msg_maxim("ignore Discover_Modes");
						break;
					case Enter_Mode:
						msg_maxim("ignore Enter_Mode");
						break;
					case Exit_Mode:
						msg_maxim("ignore Exit_Mode");
						break;
					case Attention:
						msg_maxim("ignore Attention");
						break;
					case Configure:
						break;
					default:
						msg_maxim("vdm_command isn't valid[%x]", vdm_command);
						break;
					};
				} else if (vdm_response == SEC_UVDM_ININIATOR) {
					switch (vdm_command) {
					case Attention:
						/* Attention message is not able to be received via 0x48 OPCode */
						/* Check requested vdm command and responded vdm command */
						{
							/* Read requested vdm command */
							max77729_read_reg(usbc_data->muic, 0x23, &reqd_vdm_command);
							reqd_vdm_command &= 0x1F; /* Command bit, b4...0 */

							if (reqd_vdm_command == Configure) {
								W_DATA = 1 << (usbc_data->dp_selected_pin - 1);
								/* Retry Configure message */
								msg_maxim("Retry Configure message, W_DATA = %x, dp_selected_pin = %d",
										W_DATA, usbc_data->dp_selected_pin);
								max77729_vdm_process_set_DP_configure_mode_req(usbc_data, W_DATA);
							}
						}
						break;
					case Discover_Identity:
					case Discover_SVIDs:
					case Discover_Modes:
					case Enter_Mode:
					case Configure:
					default:
						/* Nothing */
						break;
					};
				} else
					msg_maxim("vdm_response is error value[%x]", vdm_response);
				break;
            case SEC_UVDM_UNSTRUCTURED_VDM :
                msg_maxim("SEC_UVDM_UNSTRUCTURED_VDM : %x", usbc_data->adapter_svid);
                break;
			default:
				msg_maxim("vdm_type isn't valid error");
				break;
			};
			break;
		};
		break;
	case OPCODE_SET_ALTERNATEMODE:
		usbc_data->max77729->set_altmode_en = 1;
		msg_maxim("set altmode en to 1");
		break;
	case OPCODE_QC_2_0_SET:
		max77729_handle_qc_result(usbc_data->muic_data, data);
		break;

 	case OPCODE_FW_OPCODE_CLEAR:
		msg_maxim("Cleared FW OPCODE");
	default:
		break;
	}
}

void max77729_usbc_dequeue_queue(struct max77729_usbc_platform_data *usbc_data)
{
	usbc_cmd_data cmd_data;
	usbc_cmd_queue_t *cmd_queue = NULL;

	cmd_queue = &(usbc_data->usbc_cmd_queue);

	init_usbc_cmd_data(&cmd_data);

	if (is_empty_usbc_cmd_queue(cmd_queue)) {
		msg_maxim("Queue, Empty");
		return;
	}

	dequeue_usbc_cmd(cmd_queue, &cmd_data);
	msg_maxim("!! Dequeue queue : opcode : %x, 1st data : %x. 2st data : %x",
		cmd_data.write_data[0],
		cmd_data.read_data[0],
		cmd_data.val);
}

static void max77729_usbc_clear_fw_queue(struct max77729_usbc_platform_data *usbc_data)
{
	usbc_cmd_data write_data;

	msg_maxim("called");

	init_usbc_cmd_data(&write_data);
	write_data.opcode = OPCODE_FW_OPCODE_CLEAR;
	max77729_usbc_opcode_write(usbc_data, &write_data);
}

void max77729_usbc_clear_queue(struct max77729_usbc_platform_data *usbc_data)
{
	usbc_cmd_data cmd_data;
	usbc_cmd_queue_t *cmd_queue = NULL;

	mutex_lock(&usbc_data->op_lock);
	msg_maxim("IN");
	cmd_queue = &(usbc_data->usbc_cmd_queue);

	while (!is_empty_usbc_cmd_queue(cmd_queue)) {
		init_usbc_cmd_data(&cmd_data);
		dequeue_usbc_cmd(cmd_queue, &cmd_data);
		if (max77729_check_recover_opcode(cmd_data.opcode)){
			usbc_data->recover_opcode_list[cmd_data.opcode] = true;
			usbc_data->need_recover = true;
		}
	}
	usbc_data->opcode_stamp = 0;
	msg_maxim("OUT");
	mutex_unlock(&usbc_data->op_lock);
	/* also clear fw opcode queue to sync with driver */
	max77729_usbc_clear_fw_queue(usbc_data);
}

static void max77729_usbc_cmd_run(struct max77729_usbc_platform_data *usbc_data)
{
	usbc_cmd_queue_t *cmd_queue = NULL;
	usbc_cmd_node *run_node;
	usbc_cmd_data cmd_data;
	int ret = 0;

	cmd_queue = &(usbc_data->usbc_cmd_queue);

	run_node = kzalloc(sizeof(usbc_cmd_node), GFP_KERNEL);
	if (!run_node) {
		msg_maxim("failed to allocate muic command queue");
		return;
	}

	init_usbc_cmd_node(run_node);

	init_usbc_cmd_data(&cmd_data);

	if (is_empty_usbc_cmd_queue(cmd_queue)) {
		msg_maxim("Queue, Empty");
		kfree(run_node);
		return;
	}

	dequeue_usbc_cmd(cmd_queue, &cmd_data);

	if (is_usbc_notifier_opcode(cmd_data.opcode)) {
		max77729_notify_execute(usbc_data, &cmd_data);
		max77729_usbc_cmd_run(usbc_data);
	} else if (cmd_data.opcode == OPCODE_NONE) {/* Apcmdres isr */
		msg_maxim("Apcmdres ISR !!!");
		max77729_irq_execute(usbc_data, &cmd_data);
		usbc_data->opcode_stamp = 0;
		max77729_usbc_cmd_run(usbc_data);
	} else { /* No ISR */
		msg_maxim("No ISR");
		copy_usbc_cmd_data(&cmd_data, &(usbc_data->last_opcode));
		ret = max77729_i2c_opcode_write(usbc_data, cmd_data.opcode,
				cmd_data.write_length, cmd_data.write_data);
		if (ret < 0) {
			msg_maxim("i2c write fail. dequeue opcode");
			max77729_usbc_dequeue_queue(usbc_data);
		}
	}
	kfree(run_node);
}

void max77729_usbc_opcode_write(struct max77729_usbc_platform_data *usbc_data,
	usbc_cmd_data *write_op)
{
	usbc_cmd_queue_t *cmd_queue = &(usbc_data->usbc_cmd_queue);
	usbc_cmd_data execute_cmd_data;
	usbc_cmd_data current_cmd;

	mutex_lock(&usbc_data->op_lock);
	init_usbc_cmd_data(&current_cmd);

	/* the messages sent to USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.opcode = write_op->opcode;
	execute_cmd_data.write_length = write_op->write_length;
	execute_cmd_data.is_uvdm = write_op->is_uvdm;
	memcpy(execute_cmd_data.write_data, write_op->write_data, OPCODE_DATA_LENGTH);
	execute_cmd_data.seq = OPCODE_WRITE_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	/* the messages recevied From USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.response = write_op->opcode;
	execute_cmd_data.read_length = write_op->read_length;
	execute_cmd_data.is_uvdm = write_op->is_uvdm;
	execute_cmd_data.seq = OPCODE_WRITE_SEQ;
	execute_cmd_data.debug_command_flg = write_op->debug_command_flg;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	msg_maxim("W->W opcode[0x%02x] write_length[%d] read_length[%d]",
		write_op->opcode, write_op->write_length, write_op->read_length);

	front_usbc_cmd(cmd_queue, &current_cmd);
	if (current_cmd.opcode == write_op->opcode)
		max77729_usbc_cmd_run(usbc_data);
	else {
		msg_maxim("!!!current_cmd.opcode [0x%02x][0x%02x], read_op->opcode[0x%02x]",
			current_cmd.opcode, current_cmd.response, write_op->opcode);
		if (usbc_data->opcode_stamp != 0 && current_cmd.opcode == OPCODE_NONE) {
			if (time_after(jiffies,
					usbc_data->opcode_stamp + MAX77729_MAX_APDCMD_TIME)) {
				usbc_data->opcode_stamp = 0;
				msg_maxim("error. we will dequeue response data");
				max77729_usbc_dequeue_queue(usbc_data);
				max77729_usbc_cmd_run(usbc_data);
			}
		}
	}
	mutex_unlock(&usbc_data->op_lock);
}

void max77729_usbc_opcode_read(struct max77729_usbc_platform_data *usbc_data,
	usbc_cmd_data *read_op)
{
	usbc_cmd_queue_t *cmd_queue = &(usbc_data->usbc_cmd_queue);
	usbc_cmd_data execute_cmd_data;
	usbc_cmd_data current_cmd;

	mutex_lock(&usbc_data->op_lock);
	init_usbc_cmd_data(&current_cmd);

	/* the messages sent to USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.opcode = read_op->opcode;
	execute_cmd_data.write_length = read_op->write_length;
	execute_cmd_data.is_uvdm = read_op->is_uvdm;
	memcpy(execute_cmd_data.write_data, read_op->write_data, read_op->write_length);
	execute_cmd_data.seq = OPCODE_READ_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	/* the messages recevied From USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.response = read_op->opcode;
	execute_cmd_data.read_length = read_op->read_length;
	execute_cmd_data.is_uvdm = read_op->is_uvdm;
	execute_cmd_data.seq = OPCODE_READ_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	msg_maxim("R->R opcode[0x%02x] write_length[%d] read_length[%d]",
		read_op->opcode, read_op->write_length, read_op->read_length);

	front_usbc_cmd(cmd_queue, &current_cmd);
	if (current_cmd.opcode == read_op->opcode)
		max77729_usbc_cmd_run(usbc_data);
	else {
		msg_maxim("!!!current_cmd.opcode [0x%02x][0x%02x], read_op->opcode[0x%02x]",
			current_cmd.opcode, current_cmd.response, read_op->opcode);
		if (usbc_data->opcode_stamp != 0 && current_cmd.opcode == OPCODE_NONE) {
			if (time_after(jiffies,
					usbc_data->opcode_stamp + MAX77729_MAX_APDCMD_TIME)) {
				usbc_data->opcode_stamp = 0;
				msg_maxim("error. we will dequeue response data");
				max77729_usbc_dequeue_queue(usbc_data);
				max77729_usbc_cmd_run(usbc_data);
			}
		}
	}

	mutex_unlock(&usbc_data->op_lock);
}

void max77729_usbc_opcode_update(struct max77729_usbc_platform_data *usbc_data,
	usbc_cmd_data *update_op)
{
	usbc_cmd_queue_t *cmd_queue = &(usbc_data->usbc_cmd_queue);
	usbc_cmd_data execute_cmd_data;
	usbc_cmd_data current_cmd;

	switch (update_op->opcode) {
	case OPCODE_BCCTRL1_R:
	case OPCODE_BCCTRL2_R:
	case OPCODE_CTRL1_R:
	case OPCODE_CTRL2_R:
	case OPCODE_CTRL3_R:
	case OPCODE_CCCTRL1_R:
	case OPCODE_CCCTRL2_R:
	case OPCODE_CCCTRL3_R:
	case OPCODE_HVCTRL_R:
	case OPCODE_OPCODE_VCONN_ILIM_R:
	case OPCODE_CHGIN_ILIM_R:
	case OPCODE_CHGIN_ILIM2_R:
		break;
	default:
		pr_err("%s: invalid usage(0x%x), return\n", __func__, update_op->opcode);
		return;
	}

	mutex_lock(&usbc_data->op_lock);
	init_usbc_cmd_data(&current_cmd);

	/* the messages sent to USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.opcode = update_op->opcode;
	execute_cmd_data.write_length = 0;
	execute_cmd_data.is_uvdm = update_op->is_uvdm;
	memcpy(execute_cmd_data.write_data, update_op->write_data, update_op->write_length);
	execute_cmd_data.seq = OPCODE_UPDATE_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	/* the messages recevied From USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.response = update_op->opcode;
	execute_cmd_data.read_length = 1;
	execute_cmd_data.seq = OPCODE_UPDATE_SEQ;
	execute_cmd_data.val = update_op->val;
	execute_cmd_data.mask = update_op->mask;
	execute_cmd_data.is_uvdm = update_op->is_uvdm;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	msg_maxim("U->U opcode[0x%02x] write_length[%d] read_length[%d]",
		update_op->opcode, update_op->write_length, update_op->read_length);

	front_usbc_cmd(cmd_queue, &current_cmd);
	if (current_cmd.opcode == update_op->opcode)
		max77729_usbc_cmd_run(usbc_data);
	else {
		msg_maxim("!!! current_cmd.opcode [0x%02x], update_op->opcode[0x%02x]",
			current_cmd.opcode, update_op->opcode);
		if (usbc_data->opcode_stamp != 0 && current_cmd.opcode == OPCODE_NONE) {
			if (time_after(jiffies,
					usbc_data->opcode_stamp + MAX77729_MAX_APDCMD_TIME)) {
				usbc_data->opcode_stamp = 0;
				msg_maxim("error. we will dequeue response data");
				max77729_usbc_dequeue_queue(usbc_data);
				max77729_usbc_cmd_run(usbc_data);
			}
		}
	}

	mutex_unlock(&usbc_data->op_lock);
}

void max77729_usbc_opcode_push(struct max77729_usbc_platform_data *usbc_data,
	usbc_cmd_data *read_op)
{
	usbc_cmd_queue_t *cmd_queue = &(usbc_data->usbc_cmd_queue);
	usbc_cmd_data execute_cmd_data;
	usbc_cmd_data current_cmd;

	init_usbc_cmd_data(&current_cmd);

	/* the messages sent to USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.opcode = read_op->opcode;
	execute_cmd_data.write_length = read_op->write_length;
	execute_cmd_data.is_uvdm = read_op->is_uvdm;
	memcpy(execute_cmd_data.write_data, read_op->write_data, read_op->write_length);
	execute_cmd_data.seq = OPCODE_PUSH_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	/* the messages recevied From USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.response = read_op->opcode;
	execute_cmd_data.read_length = read_op->read_length;
	execute_cmd_data.is_uvdm = read_op->is_uvdm;
	execute_cmd_data.seq = OPCODE_PUSH_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	msg_maxim("P->P opcode[0x%02x] write_length[%d] read_length[%d]",
		read_op->opcode, read_op->write_length, read_op->read_length);
}

void max77729_usbc_opcode_rw(struct max77729_usbc_platform_data *usbc_data,
	usbc_cmd_data *read_op, usbc_cmd_data *write_op)
{
	usbc_cmd_queue_t *cmd_queue = &(usbc_data->usbc_cmd_queue);
	usbc_cmd_data execute_cmd_data;
	usbc_cmd_data current_cmd;

	mutex_lock(&usbc_data->op_lock);
	init_usbc_cmd_data(&current_cmd);

	/* the messages sent to USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.opcode = read_op->opcode;
	execute_cmd_data.write_length = read_op->write_length;
	execute_cmd_data.is_uvdm = read_op->is_uvdm;
	memcpy(execute_cmd_data.write_data, read_op->write_data, read_op->write_length);
	execute_cmd_data.seq = OPCODE_RW_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	/* the messages recevied From USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.response = read_op->opcode;
	execute_cmd_data.read_length = read_op->read_length;
	execute_cmd_data.is_uvdm = read_op->is_uvdm;
	execute_cmd_data.seq = OPCODE_RW_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	/* the messages sent to USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.opcode = write_op->opcode;
	execute_cmd_data.write_length = write_op->write_length;
	execute_cmd_data.is_uvdm = write_op->is_uvdm;
	memcpy(execute_cmd_data.write_data, write_op->write_data, OPCODE_DATA_LENGTH);
	execute_cmd_data.seq = OPCODE_RW_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	/* the messages recevied From USBC. */
	init_usbc_cmd_data(&execute_cmd_data);
	execute_cmd_data.response = write_op->opcode;
	execute_cmd_data.read_length = write_op->read_length;
	execute_cmd_data.is_uvdm = write_op->is_uvdm;
	execute_cmd_data.seq = OPCODE_RW_SEQ;
	enqueue_usbc_cmd(cmd_queue, &execute_cmd_data);

	msg_maxim("RW->R opcode[0x%02x] write_length[%d] read_length[%d]",
		read_op->opcode, read_op->write_length, read_op->read_length);
	msg_maxim("RW->W opcode[0x%02x] write_length[%d] read_length[%d]",
		write_op->opcode, write_op->write_length, write_op->read_length);

	front_usbc_cmd(cmd_queue, &current_cmd);
	if (current_cmd.opcode == read_op->opcode)
		max77729_usbc_cmd_run(usbc_data);
	else {
		msg_maxim("!!! current_cmd.opcode [0x%02x], read_op->opcode[0x%02x]",
			current_cmd.opcode, read_op->opcode);
		if (usbc_data->opcode_stamp != 0 && current_cmd.opcode == OPCODE_NONE) {
			if (time_after(jiffies,
					usbc_data->opcode_stamp + MAX77729_MAX_APDCMD_TIME)) {
				usbc_data->opcode_stamp = 0;
				msg_maxim("error. we will dequeue response data");
				max77729_usbc_dequeue_queue(usbc_data);
				max77729_usbc_cmd_run(usbc_data);
			}
		}
	}

	mutex_unlock(&usbc_data->op_lock);
}

static void max77729_reset_ic(struct max77729_usbc_platform_data *usbc_data)
{
	struct max77729_dev *max77729 = usbc_data->max77729;

	//gurantee to block i2c trasaction during ccic reset
	mutex_lock(&max77729->i2c_lock);
	max77729_write_reg_nolock(usbc_data->muic, 0x80, 0x0F);
	msleep(100); /* need 100ms delay */
	mutex_unlock(&max77729->i2c_lock);
}

void max77729_usbc_check_sysmsg(struct max77729_usbc_platform_data *usbc_data, u8 sysmsg)
{
	usbc_cmd_queue_t *cmd_queue = &(usbc_data->usbc_cmd_queue);
	bool is_empty_queue = is_empty_usbc_cmd_queue(cmd_queue);
	usbc_cmd_data cmd_data;
	usbc_cmd_data next_cmd_data;
	u8 next_opcode = 0xFF;
	u8 interrupt;

	int ret = 0;

	if (usbc_data->shut_down) {
		msg_maxim("IGNORE SYSTEM_MSG IN SHUTDOWN MODE!!");
		return;
	}

	switch (sysmsg) {
	case SYSERROR_NONE:
		break;
	case SYSERROR_BOOT_WDT:
		usbc_data->watchdog_count++;
		msg_maxim("SYSERROR_BOOT_WDT: %d", usbc_data->watchdog_count);
		max77729_usbc_mask_irq(usbc_data);
		max77729_write_reg(usbc_data->muic, REG_UIC_INT_M, REG_UIC_INT_M_INIT);
		max77729_write_reg(usbc_data->muic, REG_CC_INT_M, REG_CC_INT_M_INIT);
		max77729_write_reg(usbc_data->muic, REG_PD_INT_M, REG_PD_INT_M_INIT);
		max77729_write_reg(usbc_data->muic, REG_VDM_INT_M, REG_VDM_INT_M_INIT);
		/* clear UIC_INT to prevent infinite sysmsg irq*/
		g_usbc_data->max77729->enable_nested_irq = 1;
		max77729_read_reg(usbc_data->muic, MAX77729_USBC_REG_UIC_INT, &interrupt);
		g_usbc_data->max77729->usbc_irq = interrupt & 0xBF; //clear the USBC SYSTEM IRQ
		max77729_usbc_clear_queue(usbc_data);
		usbc_data->is_first_booting = 1;
		max77729_init_opcode(usbc_data, 1);
		max77729_usbc_umask_irq(usbc_data);
		break;
	case SYSERROR_BOOT_SWRSTREQ:
		break;
	case SYSMSG_BOOT_POR:
		usbc_data->por_count++;
		max77729_usbc_mask_irq(usbc_data);
		max77729_reset_ic(usbc_data);
		max77729_write_reg(usbc_data->muic, REG_UIC_INT_M, REG_UIC_INT_M_INIT);
		max77729_write_reg(usbc_data->muic, REG_CC_INT_M, REG_CC_INT_M_INIT);
		max77729_write_reg(usbc_data->muic, REG_PD_INT_M, REG_PD_INT_M_INIT);
		max77729_write_reg(usbc_data->muic, REG_VDM_INT_M, REG_VDM_INT_M_INIT);
		/* clear UIC_INT to prevent infinite sysmsg irq*/
	        g_usbc_data->max77729->enable_nested_irq = 1;
		max77729_read_reg(usbc_data->muic, MAX77729_USBC_REG_UIC_INT, &interrupt);
		g_usbc_data->max77729->usbc_irq = interrupt & 0xBF; //clear the USBC SYSTEM IRQ
		msg_maxim("SYSERROR_BOOT_POR: %d, UIC_INT:0x%02x", usbc_data->por_count, interrupt);
		max77729_usbc_clear_queue(usbc_data);
		usbc_data->is_first_booting = 1;
		max77729_init_opcode(usbc_data, 1);
		max77729_usbc_umask_irq(usbc_data);
		break;
	case SYSERROR_APCMD_UNKNOWN:
		break;
	case SYSERROR_APCMD_INPROGRESS:
		break;
	case SYSERROR_APCMD_FAIL:

		init_usbc_cmd_data(&cmd_data);
		init_usbc_cmd_data(&next_cmd_data);

		if (front_usbc_cmd(cmd_queue, &next_cmd_data))
			next_opcode = next_cmd_data.response;

		if (!is_empty_queue) {
			copy_usbc_cmd_data(&(usbc_data->last_opcode), &cmd_data);

 			if (next_opcode == OPCODE_VDM_DISCOVER_SET_VDM_REQ) {
				usbc_data->opcode_stamp = 0;
				max77729_usbc_dequeue_queue(usbc_data);
				cmd_data.opcode = OPCODE_NONE;
			}

			if ((cmd_data.opcode != OPCODE_NONE) && (cmd_data.opcode == next_opcode)) {
				if (next_opcode != OPCODE_VDM_DISCOVER_SET_VDM_REQ) {
					ret = max77729_i2c_opcode_write(usbc_data,
						cmd_data.opcode,
						cmd_data.write_length,
						cmd_data.write_data);
					if (ret) {
						msg_maxim("i2c write fail. dequeue opcode");
						max77729_usbc_dequeue_queue(usbc_data);
					} else
						msg_maxim("RETRY SUCCESS : %x, %x", cmd_data.opcode, next_opcode);
				} else
					msg_maxim("IGNORE COMMAND : %x, %x", cmd_data.opcode, next_opcode);
			} else {
				msg_maxim("RETRY FAILED : %x, %x", cmd_data.opcode, next_opcode);
			}

		}

		break;
	default:
		break;
	}
}

static irqreturn_t max77729_apcmd_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	u8 sysmsg = 0;

	msg_maxim("IRQ(%d)_IN", irq);
	max77729_read_reg(usbc_data->muic, REG_USBC_STATUS2, &usbc_data->usbc_status2);
	sysmsg = usbc_data->usbc_status2;
	msg_maxim(" [IN] sysmsg : %d", sysmsg);

	mutex_lock(&usbc_data->op_lock);
	max77729_usbc_cmd_run(usbc_data);
	mutex_unlock(&usbc_data->op_lock);

	if (usbc_data->need_recover) {
		max77729_recover_opcode(usbc_data,
			usbc_data->recover_opcode_list);
		usbc_data->need_recover = false;
	}

	msg_maxim("IRQ(%d)_OUT", irq);

	return IRQ_HANDLED;
}

static irqreturn_t max77729_sysmsg_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	u8 sysmsg = 0;
	u8 i = 0;
	u8 raw_data[3] = {0, };
	u8 usbc_status2 = 0;
	u8 dump_reg[10] = {0, };

	for (i = 0; i < 3; i++) {
		usbc_status2 = 0;
		max77729_read_reg(usbc_data->muic, REG_USBC_STATUS2, &usbc_status2);
		raw_data[i] = usbc_status2;
	}
	if((raw_data[0] == raw_data[1]) && (raw_data[0] == raw_data[2])){
		sysmsg = raw_data[0];
	} else {
		max77729_bulk_read(usbc_data->muic, REG_USBC_STATUS1,
				8, dump_reg);
		msg_maxim("[DEBUG ]sys_reg, %x, %x, %x", raw_data[0], raw_data[1],raw_data[2]);
		msg_maxim("[DEBUG ]dump_reg, %x, %x, %x, %x, %x, %x, %x, %x\n", dump_reg[0], dump_reg[1],
			dump_reg[2], dump_reg[3], dump_reg[4], dump_reg[5], dump_reg[6], dump_reg[7]);
		sysmsg = 0x6D;
	}
	msg_maxim("IRQ(%d)_IN sysmsg: %x", irq, sysmsg);
	max77729_usbc_check_sysmsg(usbc_data, sysmsg);
	usbc_data->sysmsg = sysmsg;
	msg_maxim("IRQ(%d)_OUT sysmsg: %x", irq, sysmsg);

	return IRQ_HANDLED;
}

static irqreturn_t max77729_vdm_identity_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	MAX77729_VDM_MSG_IRQ_STATUS_Type VDM_MSG_IRQ_State;

	memset(&VDM_MSG_IRQ_State, 0, sizeof(VDM_MSG_IRQ_State));
	msg_maxim("IRQ(%d)_IN", irq);
	VDM_MSG_IRQ_State.BITS.Vdm_Flag_Discover_ID = 1;
	max77729_receive_alternate_message(usbc_data, &VDM_MSG_IRQ_State);
	msg_maxim("IRQ(%d)_OUT", irq);

	return IRQ_HANDLED;
}

static irqreturn_t max77729_vdm_svids_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	MAX77729_VDM_MSG_IRQ_STATUS_Type VDM_MSG_IRQ_State;

	memset(&VDM_MSG_IRQ_State, 0, sizeof(VDM_MSG_IRQ_State));
	msg_maxim("IRQ(%d)_IN", irq);
	VDM_MSG_IRQ_State.BITS.Vdm_Flag_Discover_SVIDs = 1;
	max77729_receive_alternate_message(usbc_data, &VDM_MSG_IRQ_State);
	msg_maxim("IRQ(%d)_OUT", irq);
	return IRQ_HANDLED;
}

static irqreturn_t max77729_vdm_discover_mode_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	MAX77729_VDM_MSG_IRQ_STATUS_Type VDM_MSG_IRQ_State;

	memset(&VDM_MSG_IRQ_State, 0, sizeof(VDM_MSG_IRQ_State));
	msg_maxim("IRQ(%d)_IN", irq);
	VDM_MSG_IRQ_State.BITS.Vdm_Flag_Discover_MODEs = 1;
	max77729_receive_alternate_message(usbc_data, &VDM_MSG_IRQ_State);
	msg_maxim("IRQ(%d)_OUT", irq);
	return IRQ_HANDLED;
}

static irqreturn_t max77729_vdm_enter_mode_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	MAX77729_VDM_MSG_IRQ_STATUS_Type VDM_MSG_IRQ_State;

	memset(&VDM_MSG_IRQ_State, 0, sizeof(VDM_MSG_IRQ_State));
	msg_maxim("IRQ(%d)_IN", irq);
	VDM_MSG_IRQ_State.BITS.Vdm_Flag_Enter_Mode = 1;
	max77729_receive_alternate_message(usbc_data, &VDM_MSG_IRQ_State);
	msg_maxim("IRQ(%d)_OUT", irq);
	return IRQ_HANDLED;
}

static irqreturn_t max77729_vdm_dp_status_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	MAX77729_VDM_MSG_IRQ_STATUS_Type VDM_MSG_IRQ_State;

	memset(&VDM_MSG_IRQ_State, 0, sizeof(VDM_MSG_IRQ_State));
	msg_maxim("IRQ(%d)_IN", irq);
	VDM_MSG_IRQ_State.BITS.Vdm_Flag_DP_Status_Update = 1;
	max77729_receive_alternate_message(usbc_data, &VDM_MSG_IRQ_State);
	msg_maxim("IRQ(%d)_OUT", irq);
	return IRQ_HANDLED;
}

static irqreturn_t max77729_vdm_dp_configure_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	MAX77729_VDM_MSG_IRQ_STATUS_Type VDM_MSG_IRQ_State;

	memset(&VDM_MSG_IRQ_State, 0, sizeof(VDM_MSG_IRQ_State));
	msg_maxim("IRQ(%d)_IN", irq);
	VDM_MSG_IRQ_State.BITS.Vdm_Flag_DP_Configure = 1;
	max77729_receive_alternate_message(usbc_data, &VDM_MSG_IRQ_State);
	msg_maxim("IRQ(%d)_OUT", irq);
	return IRQ_HANDLED;
}

static irqreturn_t max77729_vdm_attention_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;
	MAX77729_VDM_MSG_IRQ_STATUS_Type VDM_MSG_IRQ_State;

	memset(&VDM_MSG_IRQ_State, 0, sizeof(VDM_MSG_IRQ_State));
	msg_maxim("IRQ(%d)_IN", irq);
	VDM_MSG_IRQ_State.BITS.Vdm_Flag_Attention = 1;
	max77729_receive_alternate_message(usbc_data, &VDM_MSG_IRQ_State);
	msg_maxim("IRQ(%d)_OUT", irq);
	return IRQ_HANDLED;
}

static irqreturn_t max77729_vir_altmode_irq(int irq, void *data)
{
	struct max77729_usbc_platform_data *usbc_data = data;

	msg_maxim("max77729_vir_altmode_irq");

	if (usbc_data->shut_down) {
		msg_maxim("%s doing shutdown. skip set alternate mode", __func__);
		goto skip;
	}

	max77729_set_enable_alternate_mode
		(usbc_data->set_altmode);

skip:
	return IRQ_HANDLED;
}

int max77729_init_irq_handler(struct max77729_usbc_platform_data *usbc_data)
{
	int ret = 0;

	usbc_data->irq_apcmd = usbc_data->irq_base + MAX77729_USBC_IRQ_APC_INT;
	pr_info("%s: irq_base(%d) (%d)!!!\n",__func__, usbc_data->irq_base,usbc_data->irq_apcmd );
	if (usbc_data->irq_apcmd) {
		ret = request_threaded_irq(usbc_data->irq_apcmd,
			   NULL, max77729_apcmd_irq,
			   0,
			   "usbc-apcmd-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_sysmsg = usbc_data->irq_base + MAX77729_USBC_IRQ_SYSM_INT;
	if (usbc_data->irq_sysmsg) {
		ret = request_threaded_irq(usbc_data->irq_sysmsg,
			   NULL, max77729_sysmsg_irq,
			   0,
			   "usbc-sysmsg-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_vdm0 = usbc_data->irq_base + MAX77729_IRQ_VDM_DISCOVER_ID_INT;
	if (usbc_data->irq_vdm0) {
		ret = request_threaded_irq(usbc_data->irq_vdm0,
			   NULL, max77729_vdm_identity_irq,
			   0,
			   "usbc-vdm0-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_vdm1 = usbc_data->irq_base + MAX77729_IRQ_VDM_DISCOVER_SVIDS_INT;
	if (usbc_data->irq_vdm1) {
		ret = request_threaded_irq(usbc_data->irq_vdm1,
			   NULL, max77729_vdm_svids_irq,
			   0,
			   "usbc-vdm1-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_vdm2 = usbc_data->irq_base + MAX77729_IRQ_VDM_DISCOVER_MODES_INT;
	if (usbc_data->irq_vdm2) {
		ret = request_threaded_irq(usbc_data->irq_vdm2,
			   NULL, max77729_vdm_discover_mode_irq,
			   0,
			   "usbc-vdm2-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_vdm3 = usbc_data->irq_base + MAX77729_IRQ_VDM_ENTER_MODE_INT;
	if (usbc_data->irq_vdm3) {
		ret = request_threaded_irq(usbc_data->irq_vdm3,
			   NULL, max77729_vdm_enter_mode_irq,
			   0,
			   "usbc-vdm3-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_vdm4 = usbc_data->irq_base + MAX77729_IRQ_VDM_DP_STATUS_UPDATE_INT;
	if (usbc_data->irq_vdm4) {
		ret = request_threaded_irq(usbc_data->irq_vdm4,
			   NULL, max77729_vdm_dp_status_irq,
			   0,
			   "usbc-vdm4-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_vdm5 = usbc_data->irq_base + MAX77729_IRQ_VDM_DP_CONFIGURE_INT;
	if (usbc_data->irq_vdm5) {
		ret = request_threaded_irq(usbc_data->irq_vdm5,
			   NULL, max77729_vdm_dp_configure_irq,
			   0,
			   "usbc-vdm5-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_vdm6 = usbc_data->irq_base + MAX77729_IRQ_VDM_ATTENTION_INT;
	if (usbc_data->irq_vdm6) {
		ret = request_threaded_irq(usbc_data->irq_vdm6,
			   NULL, max77729_vdm_attention_irq,
			   0,
			   "usbc-vdm6-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	usbc_data->irq_vir0 = usbc_data->irq_base + MAX77729_VIR_IRQ_ALTERROR_INT;
	if (usbc_data->irq_vir0) {
		ret = request_threaded_irq(usbc_data->irq_vir0,
			   NULL, max77729_vir_altmode_irq,
			   0,
			   "usbc-vir0-irq", usbc_data);
		if (ret) {
			pr_err("%s: Failed to Request IRQ (%d)\n", __func__, ret);
			return ret;
		}
	}

	return ret;
}

static void max77729_usbc_umask_irq(struct max77729_usbc_platform_data *usbc_data)
{
	int ret = 0;
	u8 i2c_data = 0;
	/* Unmask max77729 interrupt */
	ret = max77729_read_reg(usbc_data->i2c, 0x23,
			  &i2c_data);
	if (ret) {
		pr_err("%s fail to read muic reg\n", __func__);
		return;
	}

	i2c_data &= ~((1 << 3));	/* Unmask muic interrupt */
	max77729_write_reg(usbc_data->i2c, 0x23,
			   i2c_data);
}
static void max77729_usbc_mask_irq(struct max77729_usbc_platform_data *usbc_data)
{
	int ret = 0;
	u8 i2c_data = 0;
	/* Unmask max77729 interrupt */
	ret = max77729_read_reg(usbc_data->i2c, 0x23,
			  &i2c_data);
	if (ret) {
		pr_err("%s fail to read muic reg\n", __func__);
		return;
	}

	i2c_data |= ((1 << 3));	/* Unmask muic interrupt */
	max77729_write_reg(usbc_data->i2c, 0x23,
			   i2c_data);
}

const struct typec_operations max77729_typec_ops = {
	.dr_set = max77729_dr_set,
	.pr_set = max77729_pr_set,
	.port_type_set = NULL,
};

#ifdef CONFIG_DEBUG_FS
#define DEBUG_OPCODE_BUF_SIZE 256
static ssize_t max77729_debug_write_data_show(struct file *file, char __user *user_buf,
				    size_t count, loff_t *ppos)
{
	struct max77729_usbc_platform_data *data = file->private_data;
	int i;
	int len = 0;
	char buf[DEBUG_OPCODE_BUF_SIZE];

	if (data->debug_write_count > OPCODE_MAX_LENGTH) {
		len += snprintf(buf, DEBUG_OPCODE_BUF_SIZE, "Send size error\n");
		goto out;
	}

	for (i = 0; i < data->debug_write_count; i++) {
		if (data->debug_write_data[i] <= 0xff)
			len += snprintf(&buf[len], DEBUG_OPCODE_BUF_SIZE - len, "0x%02x", data->debug_write_data[i]);
		else
			len += snprintf(&buf[len], DEBUG_OPCODE_BUF_SIZE - len, "0xXX");

		if (i + 1 < data->debug_write_count)
			len += snprintf(&buf[len], DEBUG_OPCODE_BUF_SIZE - len, ",");
		else
			len += snprintf(&buf[len], DEBUG_OPCODE_BUF_SIZE - len, "\n");
	}

out:
	return simple_read_from_buffer(user_buf, count, ppos, buf, len);
}

static ssize_t max77729_debug_write_data_set(struct file *file,
				     const char __user *user_buf,
				     size_t count, loff_t *ppos)
{
	struct max77729_usbc_platform_data *data = file->private_data;
	unsigned int option_data[OPCODE_MAX_LENGTH + 1];
	char buf[DEBUG_OPCODE_BUF_SIZE];

	if (count >= DEBUG_OPCODE_BUF_SIZE)
		return -EINVAL;

	if (copy_from_user(buf, user_buf, count) != 0) {
		return -EFAULT;
	}

	memset(option_data, 0, sizeof(unsigned int) * (OPCODE_MAX_LENGTH + 1));

	get_options(buf, OPCODE_MAX_LENGTH, option_data);

	if (option_data[0] > OPCODE_MAX_LENGTH)
		return -EINVAL;

	data->debug_write_count = option_data[0];
	memcpy(data->debug_write_data, &option_data[1], sizeof(unsigned int) * OPCODE_MAX_LENGTH);

	return count;
}

static const struct file_operations maxim_debug_write_data_fops = {
	.open = simple_open,
	.read = max77729_debug_write_data_show,
	.write = max77729_debug_write_data_set,
	.llseek = default_llseek,
};

static ssize_t max77729_debug_read_data_show(struct file *file, char __user *user_buf,
				    size_t count, loff_t *ppos)
{
	struct max77729_usbc_platform_data *data = file->private_data;
	int i;
	int len = 0;
	char buf[DEBUG_OPCODE_BUF_SIZE];

	if (data->debug_read_count > OPCODE_MAX_LENGTH) {
		len += snprintf(buf, DEBUG_OPCODE_BUF_SIZE, "Response size error\n");
		goto out;
	}

	for (i = 0; i < data->debug_read_count; i++) {
		len += snprintf(&buf[len], DEBUG_OPCODE_BUF_SIZE - len, "0x%02x", data->debug_read_data[i]);

		if (i + 1 < data->debug_read_count)
			len += snprintf(&buf[len], DEBUG_OPCODE_BUF_SIZE - len, ",");
		else
			len += snprintf(&buf[len], DEBUG_OPCODE_BUF_SIZE - len, "\n");
	}

out:
	return simple_read_from_buffer(user_buf, count, ppos, buf, len);
}

static const struct file_operations maxim_debug_read_data_fops = {
	.open = simple_open,
	.read = max77729_debug_read_data_show,
	.llseek = default_llseek,
};

static ssize_t max77729_debug_run_command(struct file *file, char __user *user_buf,
				    size_t count, loff_t *ppos)
{
	struct max77729_usbc_platform_data *data = file->private_data;
	int i;
	int len = 0;
	char buf[DEBUG_OPCODE_BUF_SIZE];
	usbc_cmd_data write_data;

	init_usbc_cmd_data(&write_data);

	if (data->debug_read_count > OPCODE_MAX_LENGTH || data->debug_read_count == 0) {
		len += snprintf(buf, DEBUG_OPCODE_BUF_SIZE, "Response size error\n");
		goto out;
	}

	if (data->debug_write_count > OPCODE_MAX_LENGTH || data->debug_write_count == 0) {
		len += snprintf(buf, DEBUG_OPCODE_BUF_SIZE, "Send size error\n");
		goto out;
	}

	if (data->debug_write_data[0] >= OPCODE_NONE) {
		len += snprintf(buf, DEBUG_OPCODE_BUF_SIZE, "OPCODE error\n");
		goto out;
	}

	write_data.opcode = data->debug_write_data[0];

	for (i = 1; i < data->debug_write_count; i++) {
		if (data->debug_write_data[i] > 0xff) {
			len += snprintf(buf, DEBUG_OPCODE_BUF_SIZE, "Write data error\n");
			goto out;
		}

		write_data.write_data[i - 1] = (u8)data->debug_write_data[i];
	}

	write_data.write_length = data->debug_write_count - 1;
	write_data.read_length = data->debug_read_count - 1;
	write_data.debug_command_flg = true;

	max77729_usbc_opcode_write(data, &write_data);

out:
	return simple_read_from_buffer(user_buf, count, ppos, buf, len);
}

static const struct file_operations maxim_debug_run_command_fops = {
	.open = simple_open,
	.read = max77729_debug_run_command,
	.llseek = default_llseek,
};

static int max77729_debugfs_create(struct max77729_usbc_platform_data *usbc_data)
{
	if (usbc_data->max77729->debug_muic) {
		usbc_data->cmd_node = debugfs_create_dir("command", usbc_data->max77729->debug_muic->node);
		debugfs_create_x8("send_size", 0400, usbc_data->cmd_node,
				    &usbc_data->debug_write_count);
		debugfs_create_x8("response_size", 0600, usbc_data->cmd_node,
				    &usbc_data->debug_read_count);
		debugfs_create_file("send_data", 0600, usbc_data->cmd_node,
				    usbc_data, &maxim_debug_write_data_fops);
		debugfs_create_file("response_data", 0400, usbc_data->cmd_node,
				    usbc_data, &maxim_debug_read_data_fops);
		debugfs_create_file("run_command", 0400, usbc_data->cmd_node,
				    usbc_data, &maxim_debug_run_command_fops);
	}

	return 0;
}
#else
static inline int max77729_debugfs_create(struct max77729_usbc_platform_data *usbc_data)
{
	return 0;
}
#endif

static int max77729_usbc_probe(struct platform_device *pdev)
{
	struct max77729_dev *max77729 = dev_get_drvdata(pdev->dev.parent);
	struct max77729_platform_data *pdata = dev_get_platdata(max77729->dev);
	struct max77729_usbc_platform_data *usbc_data = NULL;
	int ret;

	msg_maxim("Probing : %d", max77729->irq);
	usbc_data =  kzalloc(sizeof(struct max77729_usbc_platform_data), GFP_KERNEL);
	if (!usbc_data)
		return -ENOMEM;

	max77729->check_usbc_opcode_queue = check_usbc_opcode_queue;
	usbc_data->dev = pdev->dev.parent;
	usbc_data->max77729 = max77729;
	usbc_data->muic = max77729->muic;
	usbc_data->charger = max77729->charger;
	usbc_data->i2c = max77729->i2c;
	usbc_data->max77729_data = pdata;
	usbc_data->irq_base = pdata->irq_base;

	pr_info("%s: irq_base(%d)!!!\n",__func__, pdata->irq_base);

	usbc_data->pd_data = kzalloc(sizeof(struct max77729_pd_data), GFP_KERNEL);
	if (!usbc_data->pd_data)
		return -ENOMEM;

	usbc_data->cc_data = kzalloc(sizeof(struct max77729_cc_data), GFP_KERNEL);
	if (!usbc_data->cc_data)
		return -ENOMEM;

	platform_set_drvdata(pdev, usbc_data);

	ret = sysfs_create_group(&max77729->dev->kobj, &max77729_groups);
	msg_maxim("created sysfs. ret=%d", ret);
	pr_info("hary: %s node %p, %p", __func__, usbc_data->dev->of_node, pdev->dev.parent->of_node);

	usbc_data->HW_Revision = 0x0;
	usbc_data->FW_Revision = 0x0;
	usbc_data->plug_attach_done = 0x0;
	usbc_data->cc_data->current_pr = 0xFF;
	usbc_data->pd_data->current_dr = 0xFF;
	usbc_data->cc_data->current_vcon = 0xFF;
	usbc_data->op_code_done = 0x0;
	usbc_data->usbc_cmd_queue.front = NULL;
	usbc_data->usbc_cmd_queue.rear = NULL;
	usbc_data->opcode_stamp = 0;
	mutex_init(&usbc_data->op_lock);
	usbc_data->vconn_en = 1;
	usbc_data->cc_pin_status = NO_DETERMINATION;

 	usbc_data->typec_cap.revision = 0x0130;
	usbc_data->typec_cap.pd_revision = 0x300;
	usbc_data->typec_cap.prefer_role = TYPEC_NO_PREFERRED_ROLE;

	usbc_data->typec_cap.ops = &max77729_typec_ops;
	usbc_data->typec_cap.driver_data = usbc_data;

	usbc_data->typec_cap.type = TYPEC_PORT_DRP;
	usbc_data->typec_cap.data = TYPEC_PORT_DRD;

	usbc_data->typec_power_role = TYPEC_SINK;
	usbc_data->typec_try_state_change = TRY_ROLE_SWAP_NONE;

	usbc_data->typec_try_pps_enable = TRY_PPS_NONE;

	usbc_data->port = typec_register_port(usbc_data->dev, &usbc_data->typec_cap);
	if (IS_ERR(usbc_data->port))
		pr_err("unable to register typec_register_port\n");
	else
		msg_maxim("success typec_register_port port=%pK", usbc_data->port);
	init_completion(&usbc_data->typec_reverse_completion);

	usbc_data->auto_vbus_en = false;
	usbc_data->is_first_booting = 1;
	usbc_data->pd_support = false;
	usbc_data->ccrp_state = 0;
	usbc_data->set_altmode = 0;
	usbc_data->set_altmode_error = 0;
	usbc_data->need_recover = false;
	usbc_data->op_ctrl1_w = (BIT_CCSrcSnk | BIT_CCSnkSrc | BIT_CCDetEn);
	usbc_data->srcccap_request_retry = false;
	init_completion(&usbc_data->op_completion);
	init_completion(&usbc_data->ccic_sysfs_completion);
	init_completion(&usbc_data->psrdy_wait);
	init_completion(&usbc_data->pps_in_wait);
	init_completion(&usbc_data->uvdm_longpacket_out_wait);
	INIT_WORK(&usbc_data->fw_update_work,
			max77729_firmware_update_sysfs_work);
	INIT_DELAYED_WORK(&usbc_data->usb_accessory_set_work,
				max77729_usb_accessory_check_work);

	g_usbc_data = usbc_data;
#if defined CONFIG_EXTCON_QC
 	/*
	 * associate extcon with the parent dev as it could have a DT
	 * node which will be useful for extcon_get_edev_by_phandle()
	 */
	usbc_data->extcon = devm_extcon_dev_allocate(usbc_data->dev, extcon_cable);
	if (IS_ERR(usbc_data->extcon)) {
		ret = PTR_ERR(usbc_data->extcon);

	}

	ret = devm_extcon_dev_register(usbc_data->dev, usbc_data->extcon);
	if (ret) {

	}

	/* Support reporting polarity and speed via properties */
	extcon_set_property_capability(usbc_data->extcon, EXTCON_USB,
			EXTCON_PROP_USB_TYPEC_POLARITY);
	extcon_set_property_capability(usbc_data->extcon, EXTCON_USB,
			EXTCON_PROP_USB_SS);
	extcon_set_property_capability(usbc_data->extcon, EXTCON_USB_HOST,
			EXTCON_PROP_USB_TYPEC_POLARITY);
	extcon_set_property_capability(usbc_data->extcon, EXTCON_USB_HOST,
			EXTCON_PROP_USB_SS);
#endif
	max77729_get_version_info(usbc_data);
	max77729_init_irq_handler(usbc_data);
	max77729_bc12_probe(usbc_data);
	max77729_cc_init(usbc_data);
	max77729_pd_init(usbc_data);
	max77729_write_reg(usbc_data->muic, REG_PD_INT_M, 0x1C);
	max77729_write_reg(usbc_data->muic, REG_VDM_INT_M, 0xFF);
	max77729_init_opcode(usbc_data, 1);                   //harry change it for alt mode enable
	usbc_data->water_disable_state = -1;
	max77729_set_water_disable_reg(0x0);
	INIT_DELAYED_WORK(&usbc_data->vbus_hard_reset_work,
				vbus_control_hard_reset);
	/* turn on the VBUS automatically. */
	max77729->cc_booting_complete = 1;
	max77729_usbc_umask_irq(usbc_data);
	init_waitqueue_head(&usbc_data->host_turn_on_wait_q);
	init_waitqueue_head(&usbc_data->device_add_wait_q);
	usbc_data->host_turn_on_wait_time = 3;
	usbc_data->cc_open_req = 1;
	usbc_data->cc_open_state = 0;
	max77729_debugfs_create(usbc_data);
	msg_maxim("probing Complete..");
	return 0;
}

static int max77729_usbc_remove(struct platform_device *pdev)
{
	struct max77729_usbc_platform_data *usbc_data =
		platform_get_drvdata(pdev);
	struct max77729_dev *max77729 = usbc_data->max77729;

	sysfs_remove_group(&max77729->dev->kobj, &max77729_groups);
	kfree(usbc_data->hmd_list);
	usbc_data->hmd_list = NULL;
	mutex_destroy(&usbc_data->hmd_power_lock);
	mutex_destroy(&usbc_data->op_lock);
	free_irq(usbc_data->irq_apcmd, usbc_data);
	free_irq(usbc_data->irq_sysmsg, usbc_data);
	free_irq(usbc_data->irq_vdm0, usbc_data);
	free_irq(usbc_data->irq_vdm1, usbc_data);
	free_irq(usbc_data->irq_vdm2, usbc_data);
	free_irq(usbc_data->irq_vdm3, usbc_data);
	free_irq(usbc_data->irq_vdm4, usbc_data);
	free_irq(usbc_data->irq_vdm5, usbc_data);
	free_irq(usbc_data->irq_vdm6, usbc_data);
	free_irq(usbc_data->irq_vdm7, usbc_data);
	free_irq(usbc_data->pd_data->irq_pdmsg, usbc_data);
	free_irq(usbc_data->pd_data->irq_datarole, usbc_data);
	free_irq(usbc_data->pd_data->irq_ssacc, usbc_data);
	free_irq(usbc_data->pd_data->irq_fct_id, usbc_data);
	free_irq(usbc_data->cc_data->irq_vconncop, usbc_data);
	free_irq(usbc_data->cc_data->irq_vsafe0v, usbc_data);
	free_irq(usbc_data->cc_data->irq_detabrt, usbc_data);
	free_irq(usbc_data->cc_data->irq_moisture, usbc_data);
	free_irq(usbc_data->cc_data->irq_ccpinstat, usbc_data);
	free_irq(usbc_data->cc_data->irq_ccistat, usbc_data);
	free_irq(usbc_data->cc_data->irq_ccvcnstat, usbc_data);
	free_irq(usbc_data->cc_data->irq_ccstat, usbc_data);
	kfree(usbc_data->cc_data);
	kfree(usbc_data->pd_data);
	kfree(usbc_data);
	return 0;
}

#if defined CONFIG_PM_QCOM
static int max77729_usbc_suspend(struct device *dev)
{
	struct max77729_usbc_platform_data *usbc_data =
		dev_get_drvdata(dev);

	max77729_muic_suspend(usbc_data);

	return 0;
}

static int max77729_usbc_resume(struct device *dev)
{
	struct max77729_usbc_platform_data *usbc_data =
		dev_get_drvdata(dev);

	max77729_muic_resume(usbc_data);
	if (usbc_data->set_altmode_error) {
		msg_maxim("set alternate mode");
		max77729_set_enable_alternate_mode
			(usbc_data->set_altmode);
	}

	return 0;
}
#else
#define max77729_usbc_suspend NULL
#define max77729_usbc_resume NULL
#endif

static void max77729_usbc_disable_irq(struct max77729_usbc_platform_data *usbc_data)
{
	disable_irq(usbc_data->irq_apcmd);
	disable_irq(usbc_data->irq_sysmsg);
	disable_irq(usbc_data->irq_vdm0);
	disable_irq(usbc_data->irq_vdm1);
	disable_irq(usbc_data->irq_vdm2);
	disable_irq(usbc_data->irq_vdm3);
	disable_irq(usbc_data->irq_vdm4);
	disable_irq(usbc_data->irq_vdm5);
	disable_irq(usbc_data->irq_vdm6);
	disable_irq(usbc_data->irq_vir0);
	disable_irq(usbc_data->pd_data->irq_pdmsg);
	disable_irq(usbc_data->pd_data->irq_psrdy);
	disable_irq(usbc_data->pd_data->irq_datarole);
	disable_irq(usbc_data->pd_data->irq_ssacc);
	disable_irq(usbc_data->pd_data->irq_fct_id);
	disable_irq(usbc_data->cc_data->irq_vconncop);
	disable_irq(usbc_data->cc_data->irq_vsafe0v);
	disable_irq(usbc_data->cc_data->irq_moisture);
	disable_irq(usbc_data->cc_data->irq_ccpinstat);
	disable_irq(usbc_data->cc_data->irq_ccistat);
	disable_irq(usbc_data->cc_data->irq_ccvcnstat);
	disable_irq(usbc_data->cc_data->irq_ccstat);
}

static void max77729_usbc_shutdown(struct platform_device *pdev)
{
	struct max77729_usbc_platform_data *usbc_data =
		platform_get_drvdata(pdev);
	u8 uic_int = 0;
	u8 uid = 0;

	msg_maxim("max77729 usbc driver shutdown++++");
	if (!usbc_data->muic) {
		msg_maxim("no max77729 i2c client");
		return;
	}
	usbc_data->shut_down = 1;
	max77729_usbc_mask_irq(usbc_data);
	/* unmask */
	max77729_write_reg(usbc_data->muic, REG_PD_INT_M, 0xFF);
	max77729_write_reg(usbc_data->muic, REG_CC_INT_M, 0xFF);
	max77729_write_reg(usbc_data->muic, REG_UIC_INT_M, 0xFF);
	max77729_write_reg(usbc_data->muic, REG_VDM_INT_M, 0xFF);

	max77729_usbc_disable_irq(usbc_data);

	max77729_read_reg(usbc_data->muic, REG_USBC_STATUS1, &uid);
	uid = (uid & BIT_UIDADC) >> FFS(BIT_UIDADC);
    /* send the reset command */

	max77729_reset_ic(usbc_data);
	max77729_write_reg(usbc_data->muic, REG_PD_INT_M, 0xFF);
	max77729_write_reg(usbc_data->muic, REG_CC_INT_M, 0xFF);
	max77729_write_reg(usbc_data->muic, REG_UIC_INT_M, 0xFF);
	max77729_write_reg(usbc_data->muic, REG_VDM_INT_M, 0xFF);
	max77729_read_reg(usbc_data->muic,
			MAX77729_USBC_REG_UIC_INT, &uic_int);
	msg_maxim("max77729 usbc driver shutdown----");
}

static SIMPLE_DEV_PM_OPS(max77729_usbc_pm_ops, max77729_usbc_suspend,
			 max77729_usbc_resume);

static struct platform_driver max77729_usbc_driver = {
	.driver = {
		.name = "max77729-usbc",
		.owner = THIS_MODULE,
#ifdef CONFIG_PM
		.pm = &max77729_usbc_pm_ops,
#endif
	},
	.shutdown = max77729_usbc_shutdown,
	.probe = max77729_usbc_probe,
	.remove = max77729_usbc_remove,
};

static int __init max77729_usbc_init(void)
{
	msg_maxim("init");
	return platform_driver_register(&max77729_usbc_driver);
}
device_initcall(max77729_usbc_init);

static void __exit max77729_usbc_exit(void)
{
	platform_driver_unregister(&max77729_usbc_driver);
}
module_exit(max77729_usbc_exit);

MODULE_DESCRIPTION("max77729 USBPD driver");
MODULE_LICENSE("GPL");
