// SPDX-License-Identifier: GPL-2.0-only
/* Copyright (c) 2012-2021, The Linux Foundation. All rights reserved.
 * Copyright (c) 2022 Qualcomm Innovation Center, Inc. All rights reserved.
 */
/*
* Add support for 24 and 32bit format for ASM loopback and playback session.
*/

#include <linux/init.h>
#include <linux/err.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/bitops.h>
#include <linux/mutex.h>
#include <linux/of_device.h>
#include <linux/slab.h>
#include <sound/core.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/pcm.h>
#include <sound/initval.h>
#include <sound/control.h>
#include <sound/tlv.h>
#include <sound/pcm_params.h>
#include <sound/hwdep.h>
#include <audio/sound/audio_effects.h>
#include <dsp/msm-dts-srs-tm-config.h>
#include <dsp/q6voice.h>
#include <dsp/q6adm-v2.h>
#include <dsp/q6asm-v2.h>
#include <dsp/q6afe-v2.h>
#include <dsp/q6lsm.h>
#include <dsp/q6core.h>
#include <dsp/q6common.h>
#include <dsp/audio_cal_utils.h>

#include "msm-pcm-routing-v2.h"
#include "msm-pcm-routing-devdep.h"
#include "msm-qti-pp-config.h"
#include "msm-dolby-dap-config.h"
#include "msm-ds2-dap-config.h"

#define DRV_NAME "msm-pcm-routing-v2"

#ifndef CONFIG_DOLBY_DAP
#undef DOLBY_ADM_COPP_TOPOLOGY_ID
#define DOLBY_ADM_COPP_TOPOLOGY_ID 0xFFFFFFFE
#endif

#ifndef CONFIG_DOLBY_DS2
#undef DS2_ADM_COPP_TOPOLOGY_ID
#define DS2_ADM_COPP_TOPOLOGY_ID 0xFFFFFFFF
#endif

#define STRING_LENGTH_OF_INT 12
#define MAX_USR_CTRL_CNT 128

#ifdef RX_TO_TX_LOOPBACK
#define RX_TO_TX_LOOPBACK_DUMMY_TX_PORT  AFE_PORT_ID_QUINARY_TDM_TX_7
#define RX_TO_TX_LOOPBACK_RX_PORT  AFE_PORT_ID_QUINARY_TDM_RX
#endif

static struct mutex routing_lock;

static struct cal_type_data *cal_data[MAX_ROUTING_CAL_TYPES];

static int hfp_switch_enable;
static int a2dp_switch_enable;
static int sco_switch_enable;
#ifndef CONFIG_MI2S_DISABLE
static int int0_mi2s_switch_enable;
static int int4_mi2s_switch_enable;
static int pri_mi2s_switch_enable;
static int sec_mi2s_switch_enable;
static int tert_mi2s_switch_enable;
static int quat_mi2s_switch_enable;
static int quin_mi2s_switch_enable;
static int sen_mi2s_switch_enable;
#endif
static int fm_pcmrx_switch_enable;
static int usb_switch_enable;
static int lsm_port_index[MAX_LSM_SESSIONS];
static int msm_route_ec_ref_rx;
static int msm_ec_ref_ch = 4;
static int msm_ec_ref_ch_downmixed = 4;
static int msm_ec_ref_bit_format = SNDRV_PCM_FORMAT_S16_LE;
static int msm_ec_ref_sampling_rate = 48000;
static uint16_t msm_ec_ref_ch_weights[PCM_FORMAT_MAX_NUM_CHANNEL_V8]
				[PCM_FORMAT_MAX_NUM_CHANNEL_V8];

static uint32_t voc_session_id = ALL_SESSION_VSID;
static int vi_lch_port;
static int vi_rch_port;
static int msm_route_ext_ec_ref;
static bool is_custom_stereo_on;
static bool is_ds2_on;
static bool hifi_filter_enabled;
static int num_app_cfg_types;
static int msm_ec_ref_port_id;
static int afe_loopback_tx_port_index;
static int afe_loopback_tx_port_id = -1;
static struct msm_pcm_channel_mixer ec_ref_chmix_cfg[MSM_FRONTEND_DAI_MAX];
static struct msm_ec_ref_port_cfg ec_ref_port_cfg;

#define WEIGHT_0_DB 0x4000
/* all the FEs which can support channel mixer */
static struct msm_pcm_channel_mixer channel_mixer[MSM_FRONTEND_DAI_MM_SIZE];

/* all the FES which can support channel mixer for bidirection */
static struct msm_pcm_channel_mixer
	channel_mixer_v2[MSM_FRONTEND_DAI_MM_SIZE][2];

/* input BE for each FE */
static int channel_input[MSM_FRONTEND_DAI_MM_SIZE][ADM_MAX_CHANNELS];

enum {
	MADNONE,
	MADAUDIO,
	MADBEACON,
	MADULTRASOUND,
	MADSWAUDIO,
};

#define ADM_LSM_PORT_INDEX 9

#define SLIMBUS_0_TX_TEXT "SLIMBUS_0_TX"
#define SLIMBUS_1_TX_TEXT "SLIMBUS_1_TX"
#define SLIMBUS_2_TX_TEXT "SLIMBUS_2_TX"
#define SLIMBUS_3_TX_TEXT "SLIMBUS_3_TX"
#define SLIMBUS_4_TX_TEXT "SLIMBUS_4_TX"
#define SLIMBUS_5_TX_TEXT "SLIMBUS_5_TX"
#define TERT_MI2S_TX_TEXT "TERT_MI2S_TX"
#define QUAT_MI2S_TX_TEXT "QUAT_MI2S_TX"
#define ADM_LSM_TX_TEXT "ADM_LSM_TX"
#define INT3_MI2S_TX_TEXT "INT3_MI2S_TX"
#define VA_CDC_DMA_TX_0_TEXT "VA_CDC_DMA_TX_0"
#define VA_CDC_DMA_TX_1_TEXT "VA_CDC_DMA_TX_1"
#define VA_CDC_DMA_TX_2_TEXT "VA_CDC_DMA_TX_2"
#define TX_CDC_DMA_TX_3_TEXT "TX_CDC_DMA_TX_3"
#define QUIN_TDM_TX_TEXT "QUIN_TDM_TX_0"
#define TERT_TDM_TX_TEXT "TERT_TDM_TX_0"
#define AFE_PCM_TX_TEXT "AFE_PCM_TX"

#define LSM_FUNCTION_TEXT "LSM Function"
static const char * const lsm_port_text[] = {
	"None",
	SLIMBUS_0_TX_TEXT, SLIMBUS_1_TX_TEXT, SLIMBUS_2_TX_TEXT,
	SLIMBUS_3_TX_TEXT, SLIMBUS_4_TX_TEXT, SLIMBUS_5_TX_TEXT,
	TERT_MI2S_TX_TEXT, QUAT_MI2S_TX_TEXT, ADM_LSM_TX_TEXT,
	INT3_MI2S_TX_TEXT, VA_CDC_DMA_TX_0_TEXT, VA_CDC_DMA_TX_1_TEXT,
	VA_CDC_DMA_TX_2_TEXT, TX_CDC_DMA_TX_3_TEXT, QUIN_TDM_TX_TEXT,
	TERT_TDM_TX_TEXT, AFE_PCM_TX_TEXT
};

struct msm_pcm_route_bdai_pp_params {
	u16 port_id; /* AFE port ID */
	unsigned long pp_params_config;
	bool mute_on;
	int latency;
};

static struct msm_pcm_route_bdai_pp_params
	msm_bedais_pp_params[MSM_BACKEND_DAI_PP_PARAMS_REQ_MAX] = {
	{HDMI_RX, 0, 0, 0},
	{HDMI_RX_MS, 0, 0, 0},
	{DISPLAY_PORT_RX, 0, 0, 0},
};

/*
 * The be_dai_name_table is passed to HAL so that it can specify the
 * BE ID for the BE it wants to enable based on the name. Thus there
 * is a matching table and structure in HAL that need to be updated
 * if any changes to these are made.
 */
struct msm_pcm_route_bdai_name {
	unsigned int be_id;
	char be_name[LPASS_BE_NAME_MAX_LENGTH];
};
static struct msm_pcm_route_bdai_name be_dai_name_table[MSM_BACKEND_DAI_MAX];

static int msm_routing_send_device_pp_params(int port_id,  int copp_idx,
					     int fe_id);

static void msm_routing_load_topology(size_t data_size, void *data);
static void msm_routing_unload_topology(uint32_t topology_id);

static int pcm_soft_volume_ctl_info(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = SOFT_VOLUME_CURVE_ENUM_MAX;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = INT_MAX;
	return 0;
}

static void pcm_soft_volume_ctl_private_free(struct snd_kcontrol *kcontrol)
{
	struct snd_pcm_volume *info = snd_kcontrol_chip(kcontrol);

	kfree(info);
}

int snd_pcm_add_soft_volume_ctls(struct snd_pcm *pcm, int stream,
			   const struct snd_pcm_soft_vol_usr_elem *soft_vol_params,
			   unsigned long private_value,
			   struct snd_pcm_soft_volume **info_ret)
{
	int err = 0;
	int size = 0;
	struct snd_pcm_soft_volume *info;
	struct snd_kcontrol_new knew = {
		.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
		.access = SNDRV_CTL_ELEM_ACCESS_TLV_READ |
			SNDRV_CTL_ELEM_ACCESS_READWRITE,
		.info = pcm_soft_volume_ctl_info,
	};

	info = kzalloc(sizeof(*info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;
	info->pcm = pcm;
	info->stream = stream;
	info->usr_val = soft_vol_params;
	size = sizeof("Playback ") + sizeof("Soft Vol Params") +
		STRING_LENGTH_OF_INT*sizeof(char) + 1;
	knew.name = kzalloc(size, GFP_KERNEL);
	if (!knew.name) {
		kfree(info);
		return -ENOMEM;
	}
	if (stream == SNDRV_PCM_STREAM_PLAYBACK)
		snprintf((char *)knew.name, size, "%s %d %s",
			"Playback", pcm->device, "Soft Vol Params");
	else
		snprintf((char *)knew.name, size, "%s %d %s",
			"Capture", pcm->device, "Soft Vol Params");
	knew.device = pcm->device;
	knew.count = pcm->streams[stream].substream_count;
	knew.private_value = private_value;
	info->kctl = snd_ctl_new1(&knew, info);
	if (!info->kctl) {
		kfree(info);
		kfree(knew.name);
		return -ENOMEM;
	}
	info->kctl->private_free = pcm_soft_volume_ctl_private_free;
	err = snd_ctl_add(pcm->card, info->kctl);
	if (err < 0) {
		kfree(info);
		kfree(knew.name);
		return -ENOMEM;
	}
	if (info_ret)
		*info_ret = info;
	kfree(knew.name);

	return 0;
}
EXPORT_SYMBOL(snd_pcm_add_soft_volume_ctls);

#ifndef SND_PCM_ADD_VOLUME_CTL
static int pcm_volume_ctl_info(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 0x2000;
	return 0;
}

static void pcm_volume_ctl_private_free(struct snd_kcontrol *kcontrol)
{
	struct snd_pcm_volume *info = snd_kcontrol_chip(kcontrol);

	kfree(info);
}

/**
 * snd_pcm_add_volume_ctls - create volume control elements
 * @pcm: the assigned PCM instance
 * @stream: stream direction
 * @max_length: the max length of the volume parameter of stream
 * @private_value: the value passed to each kcontrol's private_value field
 * @info_ret: store struct snd_pcm_volume instance if non-NULL
 *
 * Create volume control elements assigned to the given PCM stream(s).
 * Returns zero if succeed, or a negative error value.
 */
int snd_pcm_add_volume_ctls(struct snd_pcm *pcm, int stream,
			   const struct snd_pcm_volume_elem *volume,
			   int max_length,
			   unsigned long private_value,
			   struct snd_pcm_volume **info_ret)
{
	int err = 0;
	int size = 0;
	struct snd_pcm_volume *info;
	struct snd_kcontrol_new knew = {
		.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
		.access = SNDRV_CTL_ELEM_ACCESS_TLV_READ |
			SNDRV_CTL_ELEM_ACCESS_READWRITE,
		.info = pcm_volume_ctl_info,
	};

	info = kzalloc(sizeof(*info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;
	info->pcm = pcm;
	info->stream = stream;
	info->volume = volume;
	info->max_length = max_length;
	size = sizeof("Playback ") + sizeof(" Volume") +
		STRING_LENGTH_OF_INT*sizeof(char) + 1;
	knew.name = kzalloc(size, GFP_KERNEL);
	if (!knew.name) {
		kfree(info);
		return -ENOMEM;
	}
	if (stream == SNDRV_PCM_STREAM_PLAYBACK)
		snprintf((char *)knew.name, size, "%s %d %s",
			"Playback", pcm->device, "Volume");
	else
		snprintf((char *)knew.name, size, "%s %d %s",
			"Capture", pcm->device, "Volume");
	knew.device = pcm->device;
	knew.count = pcm->streams[stream].substream_count;
	knew.private_value = private_value;
	info->kctl = snd_ctl_new1(&knew, info);
	if (!info->kctl) {
		kfree(info);
		kfree(knew.name);
		return -ENOMEM;
	}
	info->kctl->private_free = pcm_volume_ctl_private_free;
	err = snd_ctl_add(pcm->card, info->kctl);
	if (err < 0) {
		kfree(info);
		kfree(knew.name);
		return -ENOMEM;
	}
	if (info_ret)
		*info_ret = info;
	kfree(knew.name);
	return 0;
}
EXPORT_SYMBOL(snd_pcm_add_volume_ctls);
#endif

#ifndef SND_PCM_ADD_USR_CTL
static int pcm_usr_ctl_info(struct snd_kcontrol *kcontrol,
			    struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = MAX_USR_CTRL_CNT;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = INT_MAX;
	return 0;
}

static void pcm_usr_ctl_private_free(struct snd_kcontrol *kcontrol)
{
	struct snd_pcm_usr *info = snd_kcontrol_chip(kcontrol);

	kfree(info);
}

/**
 * snd_pcm_add_usr_ctls - create user control elements
 * @pcm: the assigned PCM instance
 * @stream: stream direction
 * @max_length: the max length of the user parameter of stream
 * @private_value: the value passed to each kcontrol's private_value field
 * @info_ret: store struct snd_pcm_usr instance if non-NULL
 *
 * Create usr control elements assigned to the given PCM stream(s).
 * Returns zero if succeed, or a negative error value.
 */
int snd_pcm_add_usr_ctls(struct snd_pcm *pcm, int stream,
			 const struct snd_pcm_usr_elem *usr,
			 int max_length, int max_kctrl_str_len,
			 unsigned long private_value,
			 struct snd_pcm_usr **info_ret)
{
	int err = 0;
	char *buf = NULL;
	struct snd_pcm_usr *info;
	struct snd_kcontrol_new knew = {
		.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
		.access = SNDRV_CTL_ELEM_ACCESS_READWRITE,
		.info = pcm_usr_ctl_info,
	};

	info = kzalloc(sizeof(*info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;

	info->pcm = pcm;
	info->stream = stream;
	info->usr = usr;
	info->max_length = max_length;
	buf = kzalloc(max_kctrl_str_len, GFP_KERNEL);
	if (!buf) {
		pr_err("%s: buffer allocation failed\n", __func__);
		kfree(info);
		return -ENOMEM;
	}
	knew.name = buf;
	if (stream == SNDRV_PCM_STREAM_PLAYBACK)
		snprintf(buf, max_kctrl_str_len, "%s %d %s",
			"Playback", pcm->device, "User kcontrol");
	else
		snprintf(buf, max_kctrl_str_len, "%s %d %s",
			"Capture", pcm->device, "User kcontrol");
	knew.device = pcm->device;
	knew.count = pcm->streams[stream].substream_count;
	knew.private_value = private_value;
	info->kctl = snd_ctl_new1(&knew, info);
	if (!info->kctl) {
		kfree(info);
		kfree(knew.name);
		pr_err("%s: snd_ctl_new failed\n", __func__);
		return -ENOMEM;
	}
	info->kctl->private_free = pcm_usr_ctl_private_free;
	err = snd_ctl_add(pcm->card, info->kctl);
	if (err < 0) {
		kfree(info);
		kfree(knew.name);
		pr_err("%s: snd_ctl_add failed:%d\n", __func__,
			err);
		return -ENOMEM;
	}
	if (info_ret)
		*info_ret = info;
	kfree(knew.name);
	return 0;
}
EXPORT_SYMBOL(snd_pcm_add_usr_ctls);
#endif

static int msm_routing_get_bit_width(unsigned int format)
{
	int bit_width;

	switch (format) {
	case SNDRV_PCM_FORMAT_S32_LE:
		bit_width = 32;
		break;
	case SNDRV_PCM_FORMAT_S24_LE:
	case SNDRV_PCM_FORMAT_S24_3LE:
		bit_width = 24;
		break;
	case SNDRV_PCM_FORMAT_S16_LE:
	default:
		bit_width = 16;
	}
	return bit_width;
}

static bool msm_is_resample_needed(int input_sr, int output_sr)
{
	bool rc = false;

	if (input_sr != output_sr)
		rc = true;

	pr_debug("perform resampling (%s) for copp rate (%d)afe rate (%d)",
		(rc ? "oh yes" : "not really"),
		input_sr, output_sr);

	return rc;
}

static void msm_pcm_routing_cfg_pp(int port_id, int copp_idx, int topology,
				   int channels)
{
	int rc = 0;

	switch (topology) {
	case SRS_TRUMEDIA_TOPOLOGY_ID:
		pr_debug("%s: SRS_TRUMEDIA_TOPOLOGY_ID\n", __func__);
		msm_dts_srs_tm_init(port_id, copp_idx);
		break;
	case DS2_ADM_COPP_TOPOLOGY_ID:
		pr_debug("%s: DS2_ADM_COPP_TOPOLOGY %d\n",
			 __func__, DS2_ADM_COPP_TOPOLOGY_ID);
		rc = msm_ds2_dap_init(port_id, copp_idx, channels,
				      is_custom_stereo_on);
		if (rc < 0)
			pr_err("%s: DS2 topo_id 0x%x, port %d, CS %d rc %d\n",
				__func__, topology, port_id,
				is_custom_stereo_on, rc);
		break;
	case DOLBY_ADM_COPP_TOPOLOGY_ID:
		if (is_ds2_on) {
			pr_debug("%s: DS2_ADM_COPP_TOPOLOGY\n", __func__);
			rc = msm_ds2_dap_init(port_id, copp_idx, channels,
				is_custom_stereo_on);
			if (rc < 0)
				pr_err("%s:DS2 topo_id 0x%x, port %d, rc %d\n",
					__func__, topology, port_id, rc);
		} else {
			pr_debug("%s: DOLBY_ADM_COPP_TOPOLOGY_ID\n", __func__);
			rc = msm_dolby_dap_init(port_id, copp_idx, channels,
						is_custom_stereo_on);
			if (rc < 0)
				pr_err("%s: DS1 topo_id 0x%x, port %d, rc %d\n",
					__func__, topology, port_id, rc);
		}
		break;
	case ADM_CMD_COPP_OPEN_TOPOLOGY_ID_AUDIOSPHERE:
		pr_debug("%s: TOPOLOGY_ID_AUDIOSPHERE\n", __func__);
		rc = msm_qti_pp_asphere_init(port_id, copp_idx);
		if (rc < 0)
			pr_err("%s: topo_id 0x%x, port %d, copp %d, rc %d\n",
				__func__, topology, port_id, copp_idx, rc);
		break;
	default:
		/* custom topology specific feature param handlers */
		break;
	}
}

static void msm_pcm_routing_deinit_pp(int port_id, int topology)
{
	switch (topology) {
	case SRS_TRUMEDIA_TOPOLOGY_ID:
		pr_debug("%s: SRS_TRUMEDIA_TOPOLOGY_ID\n", __func__);
		msm_dts_srs_tm_deinit(port_id);
		break;
	case DS2_ADM_COPP_TOPOLOGY_ID:
		pr_debug("%s: DS2_ADM_COPP_TOPOLOGY_ID %d\n",
			 __func__, DS2_ADM_COPP_TOPOLOGY_ID);
		msm_ds2_dap_deinit(port_id);
		break;
	case DOLBY_ADM_COPP_TOPOLOGY_ID:
		if (is_ds2_on) {
			pr_debug("%s: DS2_ADM_COPP_TOPOLOGY_ID\n", __func__);
			msm_ds2_dap_deinit(port_id);
		} else {
			pr_debug("%s: DOLBY_ADM_COPP_TOPOLOGY_ID\n", __func__);
			msm_dolby_dap_deinit(port_id);
		}
		break;
	case ADM_CMD_COPP_OPEN_TOPOLOGY_ID_AUDIOSPHERE:
		pr_debug("%s: TOPOLOGY_ID_AUDIOSPHERE\n", __func__);
		msm_qti_pp_asphere_deinit(port_id);
		break;
	default:
		/* custom topology specific feature deinit handlers */
		break;
	}
}

static void msm_pcm_routng_cfg_matrix_map_pp(struct route_payload payload,
					     int path_type, int perf_mode)
{
	int itr = 0, rc = 0;

	if ((path_type == ADM_PATH_PLAYBACK) &&
	    (perf_mode == LEGACY_PCM_MODE) &&
	    is_custom_stereo_on) {
		for (itr = 0; itr < payload.num_copps; itr++) {
			if ((payload.port_id[itr] != SLIMBUS_0_RX) &&
			    (payload.port_id[itr] != RT_PROXY_PORT_001_RX) &&
			    (payload.port_id[itr] != RT_PROXY_PORT_002_RX)) {
				continue;
			}

			rc = msm_qti_pp_send_stereo_to_custom_stereo_cmd(
				payload.port_id[itr],
				payload.copp_idx[itr],
				payload.session_id,
				Q14_GAIN_ZERO_POINT_FIVE,
				Q14_GAIN_ZERO_POINT_FIVE,
				Q14_GAIN_ZERO_POINT_FIVE,
				Q14_GAIN_ZERO_POINT_FIVE);
			if (rc < 0)
				pr_err("%s: err setting custom stereo\n",
					__func__);
		}
	}
}

#define SLIMBUS_EXTPROC_RX AFE_PORT_INVALID
struct msm_pcm_routing_bdai_data msm_bedais[MSM_BACKEND_DAI_MAX] = {
	{ PRIMARY_I2S_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_PRI_I2S_RX},
	{ PRIMARY_I2S_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_PRI_I2S_TX},
	{ SLIMBUS_0_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_0_RX},
	{ SLIMBUS_0_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_0_TX},
	{ HDMI_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_HDMI},
	{ INT_BT_SCO_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_INT_BT_SCO_RX},
	{ INT_BT_SCO_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_INT_BT_SCO_TX},
	{ INT_FM_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_INT_FM_RX},
	{ INT_FM_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_INT_FM_TX},
	{ RT_PROXY_PORT_001_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_AFE_PCM_RX},
	{ RT_PROXY_PORT_001_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_AFE_PCM_TX},
	{ AFE_PORT_ID_PRIMARY_PCM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_AUXPCM_RX},
	{ AFE_PORT_ID_PRIMARY_PCM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_AUXPCM_TX},
	{ VOICE_PLAYBACK_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_VOICE_PLAYBACK_TX},
	{ VOICE2_PLAYBACK_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_VOICE2_PLAYBACK_TX},
	{ VOICE_RECORD_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INCALL_RECORD_RX},
	{ VOICE_RECORD_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INCALL_RECORD_TX},
	{ MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_MI2S_RX},
	{ MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_MI2S_TX},
	{ SECONDARY_I2S_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SEC_I2S_RX},
	{ SLIMBUS_1_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_1_RX},
	{ SLIMBUS_1_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_1_TX},
	{ SLIMBUS_2_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_2_RX},
	{ SLIMBUS_2_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_2_TX},
	{ SLIMBUS_3_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_3_RX},
	{ SLIMBUS_3_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_3_TX},
	{ SLIMBUS_4_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_4_RX},
	{ SLIMBUS_4_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_4_TX},
	{ SLIMBUS_5_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_5_RX},
	{ SLIMBUS_5_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_5_TX},
	{ SLIMBUS_6_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_6_RX},
	{ SLIMBUS_6_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_6_TX},
	{ SLIMBUS_7_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_7_RX},
	{ SLIMBUS_7_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_7_TX},
	{ SLIMBUS_8_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_8_RX},
	{ SLIMBUS_8_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_8_TX},
	{ SLIMBUS_EXTPROC_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_STUB_RX},
	{ SLIMBUS_EXTPROC_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_STUB_TX},
	{ SLIMBUS_EXTPROC_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_STUB_1_TX},
	{ AFE_PORT_ID_QUATERNARY_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_MI2S_RX},
	{ AFE_PORT_ID_QUATERNARY_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_MI2S_TX},
	{ AFE_PORT_ID_SECONDARY_MI2S_RX,  0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_MI2S_RX},
	{ AFE_PORT_ID_SECONDARY_MI2S_TX,  0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_MI2S_TX},
	{ AFE_PORT_ID_PRIMARY_MI2S_RX,    0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_MI2S_RX},
	{ AFE_PORT_ID_PRIMARY_MI2S_TX,    0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_MI2S_TX},
	{ AFE_PORT_ID_TERTIARY_MI2S_RX,   0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_MI2S_RX},
	{ AFE_PORT_ID_TERTIARY_MI2S_TX,   0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_MI2S_TX},
	{ AUDIO_PORT_ID_I2S_RX,           0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_AUDIO_I2S_RX},
	{ AFE_PORT_ID_SECONDARY_PCM_RX,	  0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_AUXPCM_RX},
	{ AFE_PORT_ID_SECONDARY_PCM_TX,   0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_AUXPCM_TX},
	{ AFE_PORT_ID_PRIMARY_SPDIF_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_SPDIF_RX},
	{ AFE_PORT_ID_SECONDARY_MI2S_RX_SD1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_MI2S_RX_SD1},
	{ AFE_PORT_ID_QUINARY_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_MI2S_RX},
	{ AFE_PORT_ID_QUINARY_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_MI2S_TX},
	{ AFE_PORT_ID_SENARY_MI2S_TX,   0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SENARY_MI2S_TX},
	{ AFE_PORT_ID_PRIMARY_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_RX_0},
	{ AFE_PORT_ID_PRIMARY_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_TX_0},
	{ AFE_PORT_ID_PRIMARY_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_RX_1},
	{ AFE_PORT_ID_PRIMARY_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_TX_1},
	{ AFE_PORT_ID_PRIMARY_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_RX_2},
	{ AFE_PORT_ID_PRIMARY_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_TX_2},
	{ AFE_PORT_ID_PRIMARY_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_RX_3},
	{ AFE_PORT_ID_PRIMARY_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_TX_3},
	{ AFE_PORT_ID_PRIMARY_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_RX_4},
	{ AFE_PORT_ID_PRIMARY_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_TX_4},
	{ AFE_PORT_ID_PRIMARY_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_RX_5},
	{ AFE_PORT_ID_PRIMARY_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_TX_5},
	{ AFE_PORT_ID_PRIMARY_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_RX_6},
	{ AFE_PORT_ID_PRIMARY_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_TX_6},
	{ AFE_PORT_ID_PRIMARY_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_RX_7},
	{ AFE_PORT_ID_PRIMARY_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_TDM_TX_7},
	{ AFE_PORT_ID_SECONDARY_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_RX_0},
	{ AFE_PORT_ID_SECONDARY_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_TX_0},
	{ AFE_PORT_ID_SECONDARY_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_RX_1},
	{ AFE_PORT_ID_SECONDARY_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_TX_1},
	{ AFE_PORT_ID_SECONDARY_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_RX_2},
	{ AFE_PORT_ID_SECONDARY_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_TX_2},
	{ AFE_PORT_ID_SECONDARY_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_RX_3},
	{ AFE_PORT_ID_SECONDARY_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_TX_3},
	{ AFE_PORT_ID_SECONDARY_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_RX_4},
	{ AFE_PORT_ID_SECONDARY_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_TX_4},
	{ AFE_PORT_ID_SECONDARY_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_RX_5},
	{ AFE_PORT_ID_SECONDARY_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_TX_5},
	{ AFE_PORT_ID_SECONDARY_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_RX_6},
	{ AFE_PORT_ID_SECONDARY_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_TX_6},
	{ AFE_PORT_ID_SECONDARY_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_RX_7},
	{ AFE_PORT_ID_SECONDARY_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_TDM_TX_7},
	{ AFE_PORT_ID_TERTIARY_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_RX_0},
	{ AFE_PORT_ID_TERTIARY_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_TX_0},
	{ AFE_PORT_ID_TERTIARY_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_RX_1},
	{ AFE_PORT_ID_TERTIARY_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_TX_1},
	{ AFE_PORT_ID_TERTIARY_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_RX_2},
	{ AFE_PORT_ID_TERTIARY_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_TX_2},
	{ AFE_PORT_ID_TERTIARY_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_RX_3},
	{ AFE_PORT_ID_TERTIARY_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_TX_3},
	{ AFE_PORT_ID_TERTIARY_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_RX_4},
	{ AFE_PORT_ID_TERTIARY_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_TX_4},
	{ AFE_PORT_ID_TERTIARY_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_RX_5},
	{ AFE_PORT_ID_TERTIARY_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_TX_5},
	{ AFE_PORT_ID_TERTIARY_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_RX_6},
	{ AFE_PORT_ID_TERTIARY_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_TX_6},
	{ AFE_PORT_ID_TERTIARY_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_RX_7},
	{ AFE_PORT_ID_TERTIARY_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_TDM_TX_7},
	{ AFE_PORT_ID_QUATERNARY_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_RX_0},
	{ AFE_PORT_ID_QUATERNARY_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_TX_0},
	{ AFE_PORT_ID_QUATERNARY_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_RX_1},
	{ AFE_PORT_ID_QUATERNARY_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_TX_1},
	{ AFE_PORT_ID_QUATERNARY_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_RX_2},
	{ AFE_PORT_ID_QUATERNARY_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_TX_2},
	{ AFE_PORT_ID_QUATERNARY_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_RX_3},
	{ AFE_PORT_ID_QUATERNARY_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_TX_3},
	{ AFE_PORT_ID_QUATERNARY_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_RX_4},
	{ AFE_PORT_ID_QUATERNARY_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_TX_4},
	{ AFE_PORT_ID_QUATERNARY_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_RX_5},
	{ AFE_PORT_ID_QUATERNARY_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_TX_5},
	{ AFE_PORT_ID_QUATERNARY_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_RX_6},
	{ AFE_PORT_ID_QUATERNARY_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_TX_6},
	{ AFE_PORT_ID_QUATERNARY_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_RX_7},
	{ AFE_PORT_ID_QUATERNARY_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_TDM_TX_7},
	{ AFE_PORT_ID_QUINARY_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_RX_0},
	{ AFE_PORT_ID_QUINARY_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_TX_0},
	{ AFE_PORT_ID_QUINARY_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_RX_1},
	{ AFE_PORT_ID_QUINARY_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_TX_1},
	{ AFE_PORT_ID_QUINARY_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_RX_2},
	{ AFE_PORT_ID_QUINARY_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_TX_2},
	{ AFE_PORT_ID_QUINARY_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_RX_3},
	{ AFE_PORT_ID_QUINARY_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_TX_3},
	{ AFE_PORT_ID_QUINARY_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_RX_4},
	{ AFE_PORT_ID_QUINARY_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_TX_4},
	{ AFE_PORT_ID_QUINARY_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_RX_5},
	{ AFE_PORT_ID_QUINARY_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_TX_5},
	{ AFE_PORT_ID_QUINARY_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_RX_6},
	{ AFE_PORT_ID_QUINARY_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_TX_6},
	{ AFE_PORT_ID_QUINARY_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_RX_7},
	{ AFE_PORT_ID_QUINARY_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_TDM_TX_7},
	{ AFE_PORT_ID_SENARY_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_RX_0},
	{ AFE_PORT_ID_SENARY_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_TX_0},
	{ AFE_PORT_ID_SENARY_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_RX_1},
	{ AFE_PORT_ID_SENARY_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_TX_1},
	{ AFE_PORT_ID_SENARY_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_RX_2},
	{ AFE_PORT_ID_SENARY_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_TX_2},
	{ AFE_PORT_ID_SENARY_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_RX_3},
	{ AFE_PORT_ID_SENARY_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_TX_3},
	{ AFE_PORT_ID_SENARY_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_RX_4},
	{ AFE_PORT_ID_SENARY_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_TX_4},
	{ AFE_PORT_ID_SENARY_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_RX_5},
	{ AFE_PORT_ID_SENARY_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_TX_5},
	{ AFE_PORT_ID_SENARY_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_RX_6},
	{ AFE_PORT_ID_SENARY_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_TX_6},
	{ AFE_PORT_ID_SENARY_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_RX_7},
	{ AFE_PORT_ID_SENARY_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_TDM_TX_7},
	{ INT_BT_A2DP_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT_BT_A2DP_RX},
	{ AFE_PORT_ID_USB_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_USB_AUDIO_RX},
	{ AFE_PORT_ID_USB_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_USB_AUDIO_TX},
	{ DISPLAY_PORT_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_DISPLAY_PORT},
	{ DISPLAY_PORT_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_DISPLAY_PORT1},
	{ AFE_PORT_ID_TERTIARY_PCM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_AUXPCM_RX},
	{ AFE_PORT_ID_TERTIARY_PCM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TERT_AUXPCM_TX},
	{ AFE_PORT_ID_QUATERNARY_PCM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_AUXPCM_RX},
	{ AFE_PORT_ID_QUATERNARY_PCM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUAT_AUXPCM_TX},
	{ AFE_PORT_ID_QUINARY_PCM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_AUXPCM_RX},
	{ AFE_PORT_ID_QUINARY_PCM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_QUIN_AUXPCM_TX},
	{ AFE_PORT_ID_INT0_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT0_MI2S_RX},
	{ AFE_PORT_ID_INT0_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT0_MI2S_TX},
	{ AFE_PORT_ID_INT1_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT1_MI2S_RX},
	{ AFE_PORT_ID_INT1_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT1_MI2S_TX},
	{ AFE_PORT_ID_INT2_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT2_MI2S_RX},
	{ AFE_PORT_ID_INT2_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT2_MI2S_TX},
	{ AFE_PORT_ID_INT3_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT3_MI2S_RX},
	{ AFE_PORT_ID_INT3_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT3_MI2S_TX},
	{ AFE_PORT_ID_INT4_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT4_MI2S_RX},
	{ AFE_PORT_ID_INT4_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT4_MI2S_TX},
	{ AFE_PORT_ID_INT5_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT5_MI2S_RX},
	{ AFE_PORT_ID_INT5_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT5_MI2S_TX},
	{ AFE_PORT_ID_INT6_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT6_MI2S_RX},
	{ AFE_PORT_ID_INT6_MI2S_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_INT6_MI2S_TX},
	{ AFE_PORT_ID_SENARY_PCM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_AUXPCM_RX},
	{ AFE_PORT_ID_SENARY_PCM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEN_AUXPCM_TX},
	{ AFE_PORT_ID_SENARY_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SENARY_MI2S_RX},
	{ AFE_PORT_ID_WSA_CODEC_DMA_RX_0, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_WSA_CDC_DMA_RX_0},
	{ AFE_PORT_ID_WSA_CODEC_DMA_TX_0, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_WSA_CDC_DMA_TX_0},
	{ AFE_PORT_ID_WSA_CODEC_DMA_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_WSA_CDC_DMA_RX_1},
	{ AFE_PORT_ID_WSA_CODEC_DMA_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_WSA_CDC_DMA_TX_1},
	{ AFE_PORT_ID_WSA_CODEC_DMA_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_WSA_CDC_DMA_TX_2},
	{ AFE_PORT_ID_VA_CODEC_DMA_TX_0, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_VA_CDC_DMA_TX_0},
	{ AFE_PORT_ID_VA_CODEC_DMA_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_VA_CDC_DMA_TX_1},
	{ AFE_PORT_ID_VA_CODEC_DMA_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_VA_CDC_DMA_TX_2},
	{ AFE_PORT_ID_RX_CODEC_DMA_RX_0, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_RX_CDC_DMA_RX_0},
	{ AFE_PORT_ID_TX_CODEC_DMA_TX_0, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TX_CDC_DMA_TX_0},
	{ AFE_PORT_ID_RX_CODEC_DMA_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_RX_CDC_DMA_RX_1},
	{ AFE_PORT_ID_TX_CODEC_DMA_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TX_CDC_DMA_TX_1},
	{ AFE_PORT_ID_RX_CODEC_DMA_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_RX_CDC_DMA_RX_2},
	{ AFE_PORT_ID_TX_CODEC_DMA_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TX_CDC_DMA_TX_2},
	{ AFE_PORT_ID_RX_CODEC_DMA_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_RX_CDC_DMA_RX_3},
	{ AFE_PORT_ID_TX_CODEC_DMA_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TX_CDC_DMA_TX_3},
	{ AFE_PORT_ID_RX_CODEC_DMA_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_RX_CDC_DMA_RX_4},
	{ AFE_PORT_ID_TX_CODEC_DMA_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TX_CDC_DMA_TX_4},
	{ AFE_PORT_ID_RX_CODEC_DMA_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_RX_CDC_DMA_RX_5},
	{ AFE_PORT_ID_TX_CODEC_DMA_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_TX_CDC_DMA_TX_5},
	{ AFE_PORT_ID_RX_CODEC_DMA_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_RX_CDC_DMA_RX_6},
	{ AFE_PORT_ID_RX_CODEC_DMA_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_RX_CDC_DMA_RX_7},
	{ AFE_PORT_ID_PRIMARY_SPDIF_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_SPDIF_TX},
	{ AFE_PORT_ID_SECONDARY_SPDIF_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_SPDIF_RX},
	{ AFE_PORT_ID_SECONDARY_SPDIF_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_SPDIF_TX},
	{ SLIMBUS_9_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_9_RX},
	{ SLIMBUS_9_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_SLIMBUS_9_TX},
	{ AFE_LOOPBACK_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_AFE_LOOPBACK_TX},
	{ AFE_PORT_ID_PRIMARY_META_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_PRI_META_MI2S_RX},
	{ AFE_PORT_ID_SECONDARY_META_MI2S_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEC_META_MI2S_RX},
	{ RT_PROXY_PORT_002_RX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_PROXY_RX},
	{ RT_PROXY_PORT_002_TX, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_PROXY_TX},
	{ HDMI_RX_MS, 0, {0}, {0}, 0, 0, 0, 0, LPASS_BE_HDMI_MS},
	{ RT_PROXY_PORT_002_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_AFE_PCM_RX1},
	{ AFE_PORT_ID_SEPTENARY_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_RX_0},
	{ AFE_PORT_ID_SEPTENARY_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_TX_0},
	{ AFE_PORT_ID_SEPTENARY_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_RX_1},
	{ AFE_PORT_ID_SEPTENARY_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_TX_1},
	{ AFE_PORT_ID_SEPTENARY_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_RX_2},
	{ AFE_PORT_ID_SEPTENARY_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_TX_2},
	{ AFE_PORT_ID_SEPTENARY_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_RX_3},
	{ AFE_PORT_ID_SEPTENARY_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_TX_3},
	{ AFE_PORT_ID_SEPTENARY_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_RX_4},
	{ AFE_PORT_ID_SEPTENARY_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_TX_4},
	{ AFE_PORT_ID_SEPTENARY_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_RX_5},
	{ AFE_PORT_ID_SEPTENARY_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_TX_5},
	{ AFE_PORT_ID_SEPTENARY_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_RX_6},
	{ AFE_PORT_ID_SEPTENARY_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_TX_6},
	{ AFE_PORT_ID_SEPTENARY_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_RX_7},
	{ AFE_PORT_ID_SEPTENARY_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_SEP_TDM_TX_7},
	{ AFE_PORT_ID_HSIF0_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_RX_0},
	{ AFE_PORT_ID_HSIF0_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_TX_0},
	{ AFE_PORT_ID_HSIF0_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_RX_1},
	{ AFE_PORT_ID_HSIF0_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_TX_1},
	{ AFE_PORT_ID_HSIF0_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_RX_2},
	{ AFE_PORT_ID_HSIF0_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_TX_2},
	{ AFE_PORT_ID_HSIF0_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_RX_3},
	{ AFE_PORT_ID_HSIF0_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_TX_3},
	{ AFE_PORT_ID_HSIF0_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_RX_4},
	{ AFE_PORT_ID_HSIF0_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_TX_4},
	{ AFE_PORT_ID_HSIF0_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_RX_5},
	{ AFE_PORT_ID_HSIF0_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_TX_5},
	{ AFE_PORT_ID_HSIF0_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_RX_6},
	{ AFE_PORT_ID_HSIF0_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_TX_6},
	{ AFE_PORT_ID_HSIF0_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_RX_7},
	{ AFE_PORT_ID_HSIF0_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF0_TDM_TX_7},
	{ AFE_PORT_ID_HSIF1_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_RX_0},
	{ AFE_PORT_ID_HSIF1_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_TX_0},
	{ AFE_PORT_ID_HSIF1_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_RX_1},
	{ AFE_PORT_ID_HSIF1_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_TX_1},
	{ AFE_PORT_ID_HSIF1_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_RX_2},
	{ AFE_PORT_ID_HSIF1_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_TX_2},
	{ AFE_PORT_ID_HSIF1_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_RX_3},
	{ AFE_PORT_ID_HSIF1_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_TX_3},
	{ AFE_PORT_ID_HSIF1_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_RX_4},
	{ AFE_PORT_ID_HSIF1_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_TX_4},
	{ AFE_PORT_ID_HSIF1_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_RX_5},
	{ AFE_PORT_ID_HSIF1_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_TX_5},
	{ AFE_PORT_ID_HSIF1_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_RX_6},
	{ AFE_PORT_ID_HSIF1_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_TX_6},
	{ AFE_PORT_ID_HSIF1_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_RX_7},
	{ AFE_PORT_ID_HSIF1_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF1_TDM_TX_7},
	{ AFE_PORT_ID_HSIF2_TDM_RX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_RX_0},
	{ AFE_PORT_ID_HSIF2_TDM_TX, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_TX_0},
	{ AFE_PORT_ID_HSIF2_TDM_RX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_RX_1},
	{ AFE_PORT_ID_HSIF2_TDM_TX_1, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_TX_1},
	{ AFE_PORT_ID_HSIF2_TDM_RX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_RX_2},
	{ AFE_PORT_ID_HSIF2_TDM_TX_2, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_TX_2},
	{ AFE_PORT_ID_HSIF2_TDM_RX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_RX_3},
	{ AFE_PORT_ID_HSIF2_TDM_TX_3, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_TX_3},
	{ AFE_PORT_ID_HSIF2_TDM_RX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_RX_4},
	{ AFE_PORT_ID_HSIF2_TDM_TX_4, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_TX_4},
	{ AFE_PORT_ID_HSIF2_TDM_RX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_RX_5},
	{ AFE_PORT_ID_HSIF2_TDM_TX_5, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_TX_5},
	{ AFE_PORT_ID_HSIF2_TDM_RX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_RX_6},
	{ AFE_PORT_ID_HSIF2_TDM_TX_6, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_TX_6},
	{ AFE_PORT_ID_HSIF2_TDM_RX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_RX_7},
	{ AFE_PORT_ID_HSIF2_TDM_TX_7, 0, {0}, {0}, 0, 0, 0, 0,
	  LPASS_BE_HSIF2_TDM_TX_7},
};

/* Track ASM playback & capture sessions of DAI
 * Track LSM listen sessions
 */
static struct msm_pcm_routing_fdai_data
	fe_dai_map[MSM_FRONTEND_DAI_MAX][2] = {
	/* MULTIMEDIA1 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA2 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA3 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA4 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA5 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA6 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA7*/
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA8 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA9 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA10 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA11 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA12 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA13 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA14 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA15 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA16 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA17 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA18 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA19 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA20 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA21 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA22 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA23 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA24 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA25 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA26 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA27 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA28 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA29 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA30 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA31 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA32 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA33 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* MULTIMEDIA34 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* VOIP */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* AFE_RX */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* AFE_TX */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* VOICE_STUB */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* DTMF_RX */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* QCHAT */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* VOLTE_STUB */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* LSM1 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* LSM2 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* LSM3 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* LSM4 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* LSM5 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* LSM6 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* LSM7 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* LSM8 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* VOICE2_STUB */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* VOICEMMODE1 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
	/* VOICEMMODE2 */
	{{0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM},
	 {0, INVALID_SESSION, LEGACY_PCM_MODE, {NULL, NULL}, LEGACY_PCM} },
};

static unsigned long session_copp_map[MSM_FRONTEND_DAI_MAX][2]
				     [MSM_BACKEND_DAI_MAX];
static struct msm_pcm_routing_app_type_data app_type_cfg[MAX_APP_TYPES];
static struct msm_pcm_routing_app_type_data lsm_app_type_cfg[MAX_APP_TYPES];
static struct msm_pcm_stream_app_type_cfg
	fe_dai_app_type_cfg[MSM_FRONTEND_DAI_MAX][2][MSM_BACKEND_DAI_MAX];

static int last_be_id_configured[MSM_FRONTEND_DAI_MAX][MAX_SESSION_TYPES];

/* The caller of this should acquire routing lock */
void msm_pcm_routing_get_bedai_info(int be_idx,
				    struct msm_pcm_routing_bdai_data *be_dai)
{
	if (be_idx >= 0 && be_idx < MSM_BACKEND_DAI_MAX)
		memcpy(be_dai, &msm_bedais[be_idx],
		       sizeof(struct msm_pcm_routing_bdai_data));
}

/* The caller of this should acquire routing lock */
void msm_pcm_routing_get_fedai_info(int fe_idx, int sess_type,
				    struct msm_pcm_routing_fdai_data *fe_dai)
{
	if ((sess_type == SESSION_TYPE_TX) || (sess_type == SESSION_TYPE_RX))
		memcpy(fe_dai, &fe_dai_map[fe_idx][sess_type],
		       sizeof(struct msm_pcm_routing_fdai_data));
}

void msm_pcm_routing_acquire_lock(void)
{
	mutex_lock(&routing_lock);
}

void msm_pcm_routing_release_lock(void)
{
	mutex_unlock(&routing_lock);
}

static int msm_pcm_routing_get_app_type_idx(int app_type)
{
	int idx;

	pr_debug("%s: app_type: %d\n", __func__, app_type);
	for (idx = 0; idx < MAX_APP_TYPES; idx++) {
		if (app_type_cfg[idx].app_type == app_type)
			return idx;
	}
	pr_info("%s: App type not available, fallback to default\n", __func__);
	return 0;
}

static int msm_pcm_routing_get_lsm_app_type_idx(int app_type)
{
	int idx;

	pr_debug("%s: app_type: %d\n", __func__, app_type);
	for (idx = 0; idx < MAX_APP_TYPES; idx++) {
		if (lsm_app_type_cfg[idx].app_type == app_type)
			return idx;
	}
	pr_debug("%s: App type not available, fallback to default\n", __func__);
	return 0;
}

static int get_port_id(int port_id)
{
	int ret = port_id;

	if (port_id == AFE_LOOPBACK_TX) {
		/*
		 * Return afe_loopback_tx_port_id if set. Else return
		 * msm_ec_ref_port_id to maintain backward compatibility.
		 */
		if (afe_loopback_tx_port_id != -1)
			ret = afe_loopback_tx_port_id;
		else
			ret = msm_ec_ref_port_id;
	}

	return ret;
}

static bool is_mm_lsm_fe_id(int fe_id)
{
	bool rc = true;

	if (fe_id >= MSM_FRONTEND_DAI_MM_MAX_ID &&
		((fe_id < MSM_FRONTEND_DAI_LSM1) ||
		 (fe_id > MSM_FRONTEND_DAI_LSM8))) {
		rc = false;
	}
	return rc;
}

static int get_copp_perf_mode(int fe_id, int sess_type, int port)
{
	int fdai_mode, copp_perf_mode, flag, value;
	int rc = 0;

	if ((fe_id < 0) || (fe_id >= MSM_FRONTEND_DAI_MAX) ||
		(port < 0) || (port >= MSM_BACKEND_DAI_MAX) ||
		(sess_type != SESSION_TYPE_RX && sess_type != SESSION_TYPE_TX)) {
			pr_err("%s: Input out of bounds\n", __func__);
			return rc;
		}
	fdai_mode = fe_dai_map[fe_id][sess_type].perf_mode;
	copp_perf_mode = fe_dai_app_type_cfg[fe_id][sess_type][port].copp_perf_mode;
	flag = (copp_perf_mode & PP_PERF_MODE_FLAG_MASK);
	value = (copp_perf_mode & PP_PERF_MODE_VALUE_MASK);
	rc = fdai_mode;
	if ((flag != 0) && (value >= LEGACY_PCM_MODE) &&
		(value <= ULL_POST_PROCESSING_PCM_MODE)) {
			rc = value;
	}
	pr_debug("%s: Get copp_perf_mode = %d", __func__, rc);
	return rc;
}

/*
 * msm_pcm_routing_send_chmix_cfg:
 *	send the channel mixer command to mix the input channels
 *	into output channels.
 *
 * @fe_id: front end id
 * @ip_channel_cnt: input channel count
 * @op_channel_cnt: output channel count
 * @ch_wght_coeff: channel weight co-efficients for channel mixing
 * @session_type: indicates session is of type TX or RX
 * @stream_type: indicates either Audio or Listen stream type
 */
int msm_pcm_routing_send_chmix_cfg(int fe_id, int ip_channel_cnt,
				int op_channel_cnt, int *ch_wght_coeff,
				int session_type, int stream_type)
{

	int rc = 0, idx = 0;
	int be_index = 0, port_id;
	unsigned int session_id = 0;

	pr_debug("%s:fe_id[%d] ip_ch[%d] op_ch[%d] sess_type [%d], stream_type[%d]",
		 __func__, fe_id, ip_channel_cnt, op_channel_cnt, session_type,
		 stream_type);
	if (!is_mm_lsm_fe_id(fe_id)) {
		/* bad ID assigned in machine driver */
		pr_err("%s: bad MM ID %d\n", __func__, fe_id);
		return -EINVAL;
	}

	if (ch_wght_coeff == NULL) {
		pr_err("%s: Null channel weightage coefficients passed\n",
			__func__);
		return -EINVAL;
	}

	for (be_index = 0; be_index < MSM_BACKEND_DAI_MAX; be_index++) {
		port_id = msm_bedais[be_index].port_id;
		if (!msm_bedais[be_index].active ||
		    !test_bit(fe_id, &msm_bedais[be_index].fe_sessions[0]))
			continue;

		session_id = fe_dai_map[fe_id][session_type].strm_id;

		for (idx = 0; idx < MAX_COPPS_PER_PORT; idx++) {
			unsigned long copp =
				session_copp_map[fe_id][session_type][be_index];
			if (!test_bit(idx, &copp))
				continue;
			msm_qti_pp_send_chmix_cfg_cmd(port_id, idx,
						session_id, ip_channel_cnt,
						op_channel_cnt, ch_wght_coeff,
						session_type, stream_type);
			if (rc < 0)
				pr_err("%s: err setting channel mix config\n",
					__func__);
		}
	}

	return 0;
}
EXPORT_SYMBOL(msm_pcm_routing_send_chmix_cfg);

/**
 * msm_pcm_routing_set_channel_mixer_cfg - cache channel mixer
 * setting before use case start.
 *
 * @fe_id: frontend idx
 * @type: stream direction type
 * @params: parameters of channel mixer setting
 *
 * Return 0 for success
 */
int msm_pcm_routing_set_channel_mixer_cfg(
	int fe_id, int type,
	struct msm_pcm_channel_mixer *params)
{
	int i, j = 0;
	int num_ch_input_map = 0;
	int num_ch_out_map = 0;

	channel_mixer_v2[fe_id][type].enable = params->enable;
	channel_mixer_v2[fe_id][type].rule = params->rule;
	channel_mixer_v2[fe_id][type].input_channel =
		params->input_channel;
	channel_mixer_v2[fe_id][type].output_channel =
		params->output_channel;
	channel_mixer_v2[fe_id][type].port_idx = params->port_idx;

	for (i = 0; i < ADM_MAX_CHANNELS; i++) {
		channel_mixer_v2[fe_id][type].in_ch_map[i] =
			params->in_ch_map[i];
		if (channel_mixer_v2[fe_id][type].in_ch_map[i] > 0)
			num_ch_input_map++;
	}
	for (i = 0; i < ADM_MAX_CHANNELS; i++) {
		channel_mixer_v2[fe_id][type].out_ch_map[i] =
			params->out_ch_map[i];
		if (channel_mixer_v2[fe_id][type].out_ch_map[i] > 0)
			num_ch_out_map++;
	}

	for (i = 0; i < ADM_MAX_CHANNELS; i++)
		for (j = 0; j < ADM_MAX_CHANNELS; j++)
			channel_mixer_v2[fe_id][type].channel_weight[i][j] =
				params->channel_weight[i][j];

	channel_mixer_v2[fe_id][type].override_in_ch_map =
			params->override_in_ch_map;
	channel_mixer_v2[fe_id][type].override_out_ch_map =
			params->override_out_ch_map;
	if (channel_mixer_v2[fe_id][type].input_channel != num_ch_input_map && channel_mixer_v2[fe_id][type].override_in_ch_map) {
			channel_mixer_v2[fe_id][type].override_in_ch_map =false;
			pr_info("%s: mismatched with num_ch_input_map %d override_in_ch_map set to false \n", __func__,num_ch_input_map);
	}
	if (channel_mixer_v2[fe_id][type].output_channel != num_ch_out_map && channel_mixer_v2[fe_id][type].override_out_ch_map) {
			channel_mixer_v2[fe_id][type].override_out_ch_map =false;
			pr_info("%s: mismatched with num_ch_out_map %d override_out_ch_map set to false \n", __func__,num_ch_out_map);
	}
	return 0;
}
EXPORT_SYMBOL(msm_pcm_routing_set_channel_mixer_cfg);


/**
 * msm_pcm_routing_set_stream_ec_ref_chmix_cfg
 *
 * Receives fedai_id, ec_ref chmix configuration.
 * Returns 0 on success. On failure returns
 * -EINVAL and does not alter passed values.
 *
 * fedai_id - Passed value, front end ID for which ec_ref config is wanted
 * cfg_data - Passed value, configuration data used by ec_ref
 */
int msm_pcm_routing_set_stream_ec_ref_chmix_cfg(
	int fedai_id, struct msm_pcm_channel_mixer *cfg_data)
{
	int i, j, ret = 0;

	if (cfg_data == NULL) {
		pr_err("%s: Received NULL pointer for cfg_data\n", __func__);
		ret = -EINVAL;
		goto done;
	}

	pr_debug("%s: fedai_id %d, input_channel %d output_channel %d\n",
		__func__, fedai_id,
		 cfg_data->input_channel, cfg_data->output_channel);

	for (i = 0; i < cfg_data->output_channel; i++)
		for (j = 0; j < cfg_data->input_channel; j++)
			pr_debug("%s: ch[%d][%d] weight = %d",
				 __func__, i, j, cfg_data->channel_weight[i][j]);

	if (fedai_id < 0 || fedai_id >= MSM_FRONTEND_DAI_MAX) {
		pr_err("%s: Received out of bounds fedai_id %d\n",
			__func__, fedai_id);
		ret = -EINVAL;
		goto done;
	}

	mutex_lock(&routing_lock);

	ec_ref_chmix_cfg[fedai_id].input_channel =
						cfg_data->input_channel;
	ec_ref_chmix_cfg[fedai_id].output_channel =
						cfg_data->output_channel;
	for (i = 0; i < cfg_data->output_channel; i++)
		for (j = 0; j < cfg_data->input_channel; j++)
			ec_ref_chmix_cfg[fedai_id].channel_weight[i][j] =
						cfg_data->channel_weight[i][j];

	ec_ref_chmix_cfg[fedai_id].override_in_ch_map =
			cfg_data->override_in_ch_map;
	if (ec_ref_chmix_cfg[fedai_id].override_in_ch_map) {
		for (i = 0; i < ec_ref_chmix_cfg[fedai_id].input_channel; i++)
			ec_ref_chmix_cfg[fedai_id].in_ch_map[i] =
				cfg_data->in_ch_map[i];
	}

	ec_ref_chmix_cfg[fedai_id].override_out_ch_map =
			cfg_data->override_out_ch_map;
	if (ec_ref_chmix_cfg[fedai_id].override_out_ch_map) {
		for (i = 0; i < ec_ref_chmix_cfg[fedai_id].output_channel; i++)
			ec_ref_chmix_cfg[fedai_id].out_ch_map[i] =
				cfg_data->out_ch_map[i];
	}

	mutex_unlock(&routing_lock);

done:
	return ret;
}
EXPORT_SYMBOL(msm_pcm_routing_set_stream_ec_ref_chmix_cfg);

int msm_pcm_routing_reg_stream_app_type_cfg(
	int fedai_id, int session_type, int be_id,
	struct msm_pcm_stream_app_type_cfg *cfg_data)
{
	int ret = 0;

	if (cfg_data == NULL) {
		pr_err("%s: Received NULL pointer for cfg_data\n", __func__);
		ret = -EINVAL;
		goto done;
	}

	pr_debug("%s: fedai_id %d, session_type %d, be_id %d, app_type %d, acdb_dev_id %d,"
		"sample_rate %d, copp_token %d, bit_width %d, copp_perf_mode %d\n",
		__func__, fedai_id, session_type, be_id,
		cfg_data->app_type, cfg_data->acdb_dev_id,
		cfg_data->sample_rate, cfg_data->copp_token,
		cfg_data->bit_width, cfg_data->copp_perf_mode);

	if (!is_mm_lsm_fe_id(fedai_id)) {
		pr_err("%s: Invalid machine driver ID %d\n",
			__func__, fedai_id);
		ret = -EINVAL;
		goto done;
	}
	if (session_type != SESSION_TYPE_RX &&
		session_type != SESSION_TYPE_TX) {
		pr_err("%s: Invalid session type %d\n",
			__func__, session_type);
		ret = -EINVAL;
		goto done;
	}
	if (be_id < 0 || be_id >= MSM_BACKEND_DAI_MAX) {
		pr_err("%s: Received out of bounds be_id %d\n",
			__func__, be_id);
		ret = -EINVAL;
		goto done;
	}

	fe_dai_app_type_cfg[fedai_id][session_type][be_id] = *cfg_data;

	/*
	 * Store the BE ID of the configuration information set as the latest so
	 * the get mixer control knows what to return.
	 */
	last_be_id_configured[fedai_id][session_type] = be_id;

done:
	return ret;
}
EXPORT_SYMBOL(msm_pcm_routing_reg_stream_app_type_cfg);

/**
 * msm_pcm_routing_get_stream_app_type_cfg
 *
 * Receives fedai_id, session_type, be_id, and populates app_type,
 * acdb_dev_id, & sample rate. Returns 0 on success. On failure returns
 * -EINVAL and does not alter passed values.
 *
 * fedai_id - Passed value, front end ID for which app type config is wanted
 * session_type - Passed value, session type for which app type config
 *                is wanted
 * be_id - Returned value, back end device id the app type config data is for
 * cfg_data - Returned value, configuration data used by app type config
 */
int msm_pcm_routing_get_stream_app_type_cfg(
	int fedai_id, int session_type, int *bedai_id,
	struct msm_pcm_stream_app_type_cfg *cfg_data)
{
	int be_id;
	int ret = 0;

	if (bedai_id == NULL) {
		pr_err("%s: Received NULL pointer for backend ID\n", __func__);
		ret = -EINVAL;
		goto done;
	} else if (cfg_data == NULL) {
		pr_err("%s: NULL pointer sent for cfg_data\n", __func__);
		ret = -EINVAL;
		goto done;
	} else if (!is_mm_lsm_fe_id(fedai_id)) {
		pr_err("%s: Invalid FE ID %d\n", __func__, fedai_id);
		ret = -EINVAL;
		goto done;
	} else if (session_type != SESSION_TYPE_RX &&
		   session_type != SESSION_TYPE_TX) {
		pr_err("%s: Invalid session type %d\n", __func__, session_type);
		ret = -EINVAL;
		goto done;
	}

	be_id = last_be_id_configured[fedai_id][session_type];
	if (be_id < 0 || be_id >= MSM_BACKEND_DAI_MAX) {
		pr_err("%s: Invalid BE ID %d\n", __func__, be_id);
		ret = -EINVAL;
		goto done;
	}

	*bedai_id = be_id;
	*cfg_data = fe_dai_app_type_cfg[fedai_id][session_type][be_id];
	pr_debug("%s: fedai_id %d, session_type %d, be_id %d, app_type %d, acdb_dev_id %d,"
		"sample_rate %d, copp_token %d, bit_width %d, copp_perf_mode %d\n",
		__func__, fedai_id, session_type, *bedai_id,
		cfg_data->app_type, cfg_data->acdb_dev_id,
		cfg_data->sample_rate, cfg_data->copp_token,
		cfg_data->bit_width, cfg_data->copp_perf_mode);
done:
	return ret;
}
EXPORT_SYMBOL(msm_pcm_routing_get_stream_app_type_cfg);

static struct cal_block_data *msm_routing_find_topology_by_path(int path,
								int cal_index)
{
	struct list_head		*ptr, *next;
	struct cal_block_data		*cal_block = NULL;
	pr_debug("%s\n", __func__);

	list_for_each_safe(ptr, next,
		&cal_data[cal_index]->cal_blocks) {

		cal_block = list_entry(ptr,
			struct cal_block_data, list);

		if (cal_utils_is_cal_stale(cal_block))
			continue;

		if (((struct audio_cal_info_adm_top *)cal_block
			->cal_info)->path == path) {
			return cal_block;
		}
	}
	pr_debug("%s: Can't find topology for path %d\n", __func__, path);
	return NULL;
}

static struct cal_block_data *msm_routing_find_topology(int path,
							int app_type,
							int acdb_id,
							int cal_index,
							bool exact)
{
	struct list_head *ptr, *next;
	struct cal_block_data *cal_block = NULL;
	struct audio_cal_info_adm_top *cal_info;

	pr_debug("%s\n", __func__);

	list_for_each_safe(ptr, next,
		&cal_data[cal_index]->cal_blocks) {

		cal_block = list_entry(ptr,
			struct cal_block_data, list);

		if (cal_utils_is_cal_stale(cal_block))
			continue;

		cal_info = (struct audio_cal_info_adm_top *)
			cal_block->cal_info;
		if ((cal_info->path == path)  &&
			(cal_info->app_type == app_type) &&
			(cal_info->acdb_id == acdb_id)) {
			return cal_block;
		}
	}
	pr_debug("%s: Can't find topology for path %d, app %d, "
		 "acdb_id %d %s\n",  __func__, path, app_type, acdb_id,
		 exact ? "fail" : "defaulting to search by path");
	return exact ? NULL : msm_routing_find_topology_by_path(path,
								cal_index);
}

static struct cal_block_data *msm_routing_find_topology_by_buf_number(int usecase, int path,
							int app_type,
							int acdb_id,
							int cal_index,
							bool exact)
{
	struct list_head *ptr, *next;
	struct cal_block_data *cal_block = NULL;
	struct audio_cal_info_adm_top *cal_info;
	int buffer_idx_w_path;

	pr_debug("%s\n", __func__);

	buffer_idx_w_path = path + MAX_SESSION_TYPES * usecase;

	list_for_each_safe(ptr, next,
		&cal_data[cal_index]->cal_blocks) {

		cal_block = list_entry(ptr,
			struct cal_block_data, list);

		if (cal_utils_is_cal_stale(cal_block))
			continue;

		cal_info = (struct audio_cal_info_adm_top *)
			cal_block->cal_info;
		if ((cal_block->buffer_number == buffer_idx_w_path) &&
			(cal_info->path == path)  &&
			(cal_info->app_type == app_type) &&
			(cal_info->acdb_id == acdb_id)) {
			return cal_block;
		}
	}
	pr_debug("%s: Can't find topology for buffer_number %d, path %d, app %d, acdb_id %d %s\n",
		__func__, buffer_idx_w_path, path, app_type, acdb_id,
		exact ? "fail" : "defaulting to search by path, app_type and acdb_id");
	return exact ? NULL : msm_routing_find_topology(path, app_type,
							      acdb_id, cal_index, exact);
}

static int msm_routing_find_topology_on_index(int fedai_id, int session_type, int app_type,
					      int acdb_dev_id,  int idx,
					      bool exact)
{
	int topology = -EINVAL;
	struct cal_block_data *cal_block = NULL;

	mutex_lock(&cal_data[idx]->lock);
	if (idx == ADM_TOPOLOGY_CAL_TYPE_IDX)
		cal_block = msm_routing_find_topology_by_buf_number(fedai_id,
						     session_type,
						     app_type,
						     acdb_dev_id,
						     idx, exact);
	else
		cal_block = msm_routing_find_topology(session_type, app_type,
							      acdb_dev_id, idx, exact);

	if (cal_block != NULL) {
		topology = ((struct audio_cal_info_adm_top *)
			    cal_block->cal_info)->topology;
	}
	mutex_unlock(&cal_data[idx]->lock);
	return topology;
}

/*
 * Retrieving cal_block will mark cal_block as stale.
 * Hence it cannot be reused or resent unless the flag
 * is reset.
 */
static int msm_routing_get_adm_topology(int fedai_id, int session_type,
					int be_id)
{
	int topology = NULL_COPP_TOPOLOGY;
	int app_type = 0, acdb_dev_id = 0;

	pr_debug("%s: fedai_id %d, session_type %d, be_id %d\n",
	       __func__, fedai_id, session_type, be_id);

	if (cal_data == NULL)
		goto done;

	app_type = fe_dai_app_type_cfg[fedai_id][session_type][be_id].app_type;
	acdb_dev_id =
		fe_dai_app_type_cfg[fedai_id][session_type][be_id].acdb_dev_id;

	pr_debug("%s: Check for exact LSM topology\n", __func__);
	topology = msm_routing_find_topology_on_index(fedai_id,
					       session_type,
					       app_type,
					       acdb_dev_id,
					       ADM_LSM_TOPOLOGY_CAL_TYPE_IDX,
					       true /*exact*/);
	if (topology < 0) {
		pr_debug("%s: Check for compatible topology\n", __func__);
		topology = msm_routing_find_topology_on_index(fedai_id,
						      session_type,
						      app_type,
						      acdb_dev_id,
						      ADM_TOPOLOGY_CAL_TYPE_IDX,
						      false /*exact*/);
		if (topology < 0)
			topology = NULL_COPP_TOPOLOGY;
	}
done:
	pr_debug("%s: Using topology %d\n", __func__, topology);
	return topology;
}

static uint8_t is_be_dai_extproc(int be_dai)
{
	if (be_dai == MSM_BACKEND_DAI_EXTPROC_RX ||
	   be_dai == MSM_BACKEND_DAI_EXTPROC_TX ||
	   be_dai == MSM_BACKEND_DAI_EXTPROC_EC_TX)
		return 1;
	else
		return 0;
}

static void msm_pcm_routing_build_matrix(int fedai_id, int sess_type,
					 int path_type, int perf_mode,
					 uint32_t passthr_mode)
{
	int i, port_type, j, num_copps = 0;
	struct route_payload payload;

	port_type = ((path_type == ADM_PATH_PLAYBACK ||
		      path_type == ADM_PATH_COMPRESSED_RX) ?
		MSM_AFE_PORT_TYPE_RX : MSM_AFE_PORT_TYPE_TX);

	for (i = 0; i < MSM_BACKEND_DAI_MAX; i++) {
		if (!is_be_dai_extproc(i) &&
		   (afe_get_port_type(msm_bedais[i].port_id) == port_type) &&
		   (msm_bedais[i].active) &&
		   (test_bit(fedai_id, &msm_bedais[i].fe_sessions[0]))) {
			int port_id = get_port_id(msm_bedais[i].port_id);
			for (j = 0; j < MAX_COPPS_PER_PORT; j++) {
				unsigned long copp =
				      session_copp_map[fedai_id][sess_type][i];
				if (test_bit(j, &copp)) {
					payload.port_id[num_copps] = port_id;
					payload.copp_idx[num_copps] = j;
					payload.app_type[num_copps] =
						fe_dai_app_type_cfg
							[fedai_id][sess_type][i]
								.app_type;
					payload.acdb_dev_id[num_copps] =
						fe_dai_app_type_cfg
							[fedai_id][sess_type][i]
								.acdb_dev_id;
					payload.sample_rate[num_copps] =
						fe_dai_app_type_cfg
							[fedai_id][sess_type][i]
								.sample_rate;
					num_copps++;
				}
			}
		}
	}

	if (num_copps) {
		payload.num_copps = num_copps;
		payload.session_id = fe_dai_map[fedai_id][sess_type].strm_id;
		adm_matrix_map(fedai_id, path_type, payload, perf_mode, passthr_mode);
		msm_pcm_routng_cfg_matrix_map_pp(payload, path_type, perf_mode);
	}
}

void msm_pcm_routing_reg_psthr_stream(int fedai_id, int dspst_id,
				      int stream_type)
{
	int i, session_type, path_type, port_type;
	u32 mode = 0;

	if (fedai_id >= MSM_FRONTEND_DAI_MM_MAX_ID) {
		/* bad ID assigned in machine driver */
		pr_err("%s: bad MM ID\n", __func__);
		return;
	}

	if (stream_type == SNDRV_PCM_STREAM_PLAYBACK) {
		session_type = SESSION_TYPE_RX;
		path_type = ADM_PATH_PLAYBACK;
		port_type = MSM_AFE_PORT_TYPE_RX;
	} else {
		session_type = SESSION_TYPE_TX;
		path_type = ADM_PATH_LIVE_REC;
		port_type = MSM_AFE_PORT_TYPE_TX;
	}

	mutex_lock(&routing_lock);

	fe_dai_map[fedai_id][session_type].strm_id = dspst_id;
	for (i = 0; i < MSM_BACKEND_DAI_MAX; i++) {
		if (!is_be_dai_extproc(i) &&
		    (afe_get_port_type(msm_bedais[i].port_id) == port_type) &&
		    (msm_bedais[i].active) &&
		    (test_bit(fedai_id, &msm_bedais[i].fe_sessions[0]))) {
			mode = afe_get_port_type(msm_bedais[i].port_id);
			adm_connect_afe_port(mode, dspst_id,
					     msm_bedais[i].port_id);
			break;
		}
	}
	mutex_unlock(&routing_lock);
}

static bool route_check_fe_id_adm_support(int fe_id)
{
	bool rc = true;

	if ((fe_id >= MSM_FRONTEND_DAI_LSM1) &&
		 (fe_id <= MSM_FRONTEND_DAI_LSM8)) {
		/* fe id is listen while port is set to afe */
		if (lsm_port_index[fe_id - MSM_FRONTEND_DAI_LSM1] !=
				ADM_LSM_PORT_INDEX) {
			pr_debug("%s: fe_id %d, lsm mux slim port %d\n",
				__func__, fe_id,
				lsm_port_index[fe_id - MSM_FRONTEND_DAI_LSM1]);
			rc = false;
		}
	}

	return rc;
}

/*
 * msm_pcm_routing_get_pp_ch_cnt:
 *	Read the processed channel count
 *
 * @fe_id: Front end ID
 * @session_type: Inidicates RX or TX session type
 */
int msm_pcm_routing_get_pp_ch_cnt(int fe_id, int session_type)
{
	struct msm_pcm_stream_app_type_cfg cfg_data;
	int be_id = 0, app_type_idx = 0, app_type = 0;
	int ret = -EINVAL;

	memset(&cfg_data, 0, sizeof(cfg_data));

	if (!is_mm_lsm_fe_id(fe_id)) {
		pr_err("%s: bad MM ID\n", __func__);
		return -EINVAL;
	}

	ret = msm_pcm_routing_get_stream_app_type_cfg(fe_id, session_type,
						      &be_id, &cfg_data);
	if (ret) {
		pr_err("%s: cannot get stream app type cfg\n", __func__);
		return ret;
	}

	app_type = cfg_data.app_type;
	app_type_idx = msm_pcm_routing_get_lsm_app_type_idx(app_type);
	return lsm_app_type_cfg[app_type_idx].num_out_channels;
}
EXPORT_SYMBOL(msm_pcm_routing_get_pp_ch_cnt);

int msm_pcm_routing_reg_phy_compr_stream(int fe_id, int perf_mode,
					  int dspst_id, int stream_type,
					  uint32_t passthr_mode)
{
	int i, j, session_type, path_type, port_type, topology;
	int num_copps = 0;
	struct route_payload payload;
	u32 channels, sample_rate;
	u16 bit_width = 16, be_bit_width;
	bool is_lsm;
	uint32_t copp_token = 0;
	int copp_perf_mode = 0;

	pr_debug("%s:fe_id[%d] perf_mode[%d] id[%d] stream_type[%d] passt[%d]",
		 __func__, fe_id, perf_mode, dspst_id,
		 stream_type, passthr_mode);
	if (!is_mm_lsm_fe_id(fe_id)) {
		/* bad ID assigned in machine driver */
		pr_err("%s: bad MM ID %d\n", __func__, fe_id);
		return -EINVAL;
	}

	if (stream_type == SNDRV_PCM_STREAM_PLAYBACK) {
		session_type = SESSION_TYPE_RX;
		if (passthr_mode != LEGACY_PCM)
			path_type = ADM_PATH_COMPRESSED_RX;
		else
			path_type = ADM_PATH_PLAYBACK;
		port_type = MSM_AFE_PORT_TYPE_RX;
	} else if (stream_type == SNDRV_PCM_STREAM_CAPTURE) {
		session_type = SESSION_TYPE_TX;
		if ((passthr_mode != LEGACY_PCM) && (passthr_mode != LISTEN))
			path_type = ADM_PATH_COMPRESSED_TX;
		else
			path_type = ADM_PATH_LIVE_REC;
		port_type = MSM_AFE_PORT_TYPE_TX;
	} else {
		pr_err("%s: invalid stream type %d\n", __func__, stream_type);
		return -EINVAL;
	}

	is_lsm = (fe_id >= MSM_FRONTEND_DAI_LSM1) &&
			 (fe_id <= MSM_FRONTEND_DAI_LSM8);
	mutex_lock(&routing_lock);

	fe_dai_map[fe_id][session_type].strm_id = dspst_id;
	fe_dai_map[fe_id][session_type].perf_mode = perf_mode;
	fe_dai_map[fe_id][session_type].passthr_mode = passthr_mode;
	if (!route_check_fe_id_adm_support(fe_id)) {
		/* ignore adm open if not supported for fe_id */
		pr_debug("%s: No ADM support for fe id %d\n", __func__, fe_id);
		mutex_unlock(&routing_lock);
		return 0;
	}

	payload.num_copps = 0; /* only RX needs to use payload */
	/* re-enable EQ if active */
	msm_qti_pp_send_eq_values(fe_id);
	for (i = 0; i < MSM_BACKEND_DAI_MAX; i++) {
		if (!is_be_dai_extproc(i) &&
			(afe_get_port_type(msm_bedais[i].port_id) ==
			port_type) &&
			(msm_bedais[i].active) &&
			(test_bit(fe_id, &msm_bedais[i].fe_sessions[0]))) {
			int app_type, app_type_idx, copp_idx, acdb_dev_id;
			int port_id = get_port_id(msm_bedais[i].port_id);

			/*
			 * check if ADM needs to be configured with different
			 * channel mapping than backend
			 */
			if (!msm_bedais[i].adm_override_ch)
				channels = msm_bedais[i].channel;
			else
				channels = msm_bedais[i].adm_override_ch;

			bit_width = msm_routing_get_bit_width(
						msm_bedais[i].format);
			app_type =
			fe_dai_app_type_cfg[fe_id][session_type][i].app_type;
			if (app_type && is_lsm) {
				app_type_idx =
				msm_pcm_routing_get_lsm_app_type_idx(app_type);
				sample_rate =
				fe_dai_app_type_cfg[fe_id][session_type][i]
					.sample_rate;
				bit_width =
				lsm_app_type_cfg[app_type_idx].bit_width;
			} else if (app_type) {
				app_type_idx =
					msm_pcm_routing_get_app_type_idx(
						app_type);
				sample_rate =
			fe_dai_app_type_cfg[fe_id][session_type][i].sample_rate;
				bit_width =
					(fe_dai_app_type_cfg[fe_id][session_type][i].bit_width) ?
					fe_dai_app_type_cfg[fe_id][session_type][i].bit_width :
					app_type_cfg[app_type_idx].bit_width;
				copp_token =
			fe_dai_app_type_cfg[fe_id][session_type][i].copp_token;
			} else {
				sample_rate = msm_bedais[i].sample_rate;
			}
			acdb_dev_id =
			fe_dai_app_type_cfg[fe_id][session_type][i].acdb_dev_id;
			copp_perf_mode = get_copp_perf_mode(fe_id, session_type, i);
			topology = msm_routing_get_adm_topology(fe_id,
								session_type,
								i);
			if ((passthr_mode == COMPRESSED_PASSTHROUGH_DSD)
			     || (passthr_mode ==
				COMPRESSED_PASSTHROUGH_GEN)
			     || (passthr_mode ==
				COMPRESSED_PASSTHROUGH_IEC61937))
				topology = COMPRESSED_PASSTHROUGH_NONE_TOPOLOGY;
			pr_debug("%s: Before adm open topology %d\n", __func__,
				topology);

			be_bit_width = msm_routing_get_bit_width(
                                                msm_bedais[i].format);
			if (hifi_filter_enabled && (msm_bedais[i].sample_rate
				== 384000 || msm_bedais[i].sample_rate ==
				352800) && be_bit_width == 32)
				bit_width = msm_routing_get_bit_width(
						SNDRV_PCM_FORMAT_S32_LE);
			copp_idx =
				adm_open(port_id, path_type, sample_rate,
					 channels, topology, copp_perf_mode,
					 bit_width, app_type, acdb_dev_id,
					 session_type, passthr_mode, copp_token);
			if ((copp_idx < 0) ||
				(copp_idx >= MAX_COPPS_PER_PORT)) {
				pr_err("%s:adm open failed coppid:%d\n",
				__func__, copp_idx);
				mutex_unlock(&routing_lock);
				return -EINVAL;
			}
			pr_debug("%s: set idx bit of fe:%d, type: %d, be:%d\n",
				 __func__, fe_id, session_type, i);
			set_bit(copp_idx,
				&session_copp_map[fe_id][session_type][i]);

			if (msm_is_resample_needed(
				sample_rate,
				msm_bedais[i].sample_rate))
				adm_copp_mfc_cfg(port_id, copp_idx,
					msm_bedais[i].sample_rate);

			for (j = 0; j < MAX_COPPS_PER_PORT; j++) {
				unsigned long copp =
				session_copp_map[fe_id][session_type][i];
				if (test_bit(j, &copp)) {
					payload.port_id[num_copps] = port_id;
					payload.copp_idx[num_copps] = j;
					payload.app_type[num_copps] =
						fe_dai_app_type_cfg
							[fe_id][session_type][i]
								.app_type;
					payload.acdb_dev_id[num_copps] =
						fe_dai_app_type_cfg
							[fe_id][session_type][i]
								.acdb_dev_id;
					payload.sample_rate[num_copps] =
						fe_dai_app_type_cfg
							[fe_id][session_type][i]
								.sample_rate;
					num_copps++;
				}
			}
			if (passthr_mode != COMPRESSED_PASSTHROUGH_DSD
			    && passthr_mode != COMPRESSED_PASSTHROUGH_GEN
			    && passthr_mode != COMPRESSED_PASSTHROUGH_IEC61937)
				msm_routing_send_device_pp_params(port_id,
				copp_idx, fe_id);
		}
	}
	if (num_copps) {
		payload.num_copps = num_copps;
		payload.session_id = fe_dai_map[fe_id][session_type].strm_id;
		adm_matrix_map(fe_id, path_type, payload, perf_mode, passthr_mode);
		msm_pcm_routng_cfg_matrix_map_pp(payload, path_type, perf_mode);
	}
	mutex_unlock(&routing_lock);
	return 0;
}

static u32 msm_pcm_routing_get_voc_sessionid(u16 val)
{
	u32 session_id;

	switch (val) {
	case MSM_FRONTEND_DAI_QCHAT:
		session_id = voc_get_session_id(QCHAT_SESSION_NAME);
		break;
	case MSM_FRONTEND_DAI_VOIP:
		session_id = voc_get_session_id(VOIP_SESSION_NAME);
		break;
	case MSM_FRONTEND_DAI_VOICEMMODE1:
		session_id = voc_get_session_id(VOICEMMODE1_NAME);
		break;
	case MSM_FRONTEND_DAI_VOICEMMODE2:
		session_id = voc_get_session_id(VOICEMMODE2_NAME);
		break;
	default:
		session_id = 0;
	}

	pr_debug("%s session_id 0x%x", __func__, session_id);
	return session_id;
}

static int msm_pcm_routing_channel_mixer_v2(int fe_id, bool perf_mode,
				int dspst_id, int stream_type)
{
	int copp_idx = 0;
	int sess_type = 0;
	int j = 0, be_id = 0;
	int ret = 0;

	if (fe_id >= MSM_FRONTEND_DAI_MAX) {
		pr_err("%s: invalid FE %d\n", __func__, fe_id);
		return 0;
	}

	if (stream_type == SNDRV_PCM_STREAM_PLAYBACK)
		sess_type = SESSION_TYPE_RX;
	else
		sess_type = SESSION_TYPE_TX;

	if (!(channel_mixer_v2[fe_id][sess_type].enable)) {
		pr_debug("%s: channel mixer not enabled for FE %d direction %d\n",
			__func__, fe_id, sess_type);
		return 0;
	}

	be_id = channel_mixer_v2[fe_id][sess_type].port_idx - 1;
        if (be_id < 0 || be_id >= MSM_BACKEND_DAI_MAX) {
		pr_err("%s: Received out of bounds be_id %d\n",
				__func__, be_id);
		return -EINVAL;
	}
	channel_mixer_v2[fe_id][sess_type].input_channels[0] =
		channel_mixer_v2[fe_id][sess_type].input_channel;

	pr_debug("%s sess type %d,fe_id %d,override in:%d out:%d,be active %d\n",
			__func__, sess_type, fe_id,
			channel_mixer_v2[fe_id][sess_type].override_in_ch_map,
			channel_mixer_v2[fe_id][sess_type].override_out_ch_map,
			msm_bedais[be_id].active);

	if ((msm_bedais[be_id].active) &&
		test_bit(fe_id, &msm_bedais[be_id].fe_sessions[0])) {
		unsigned long copp =
			session_copp_map[fe_id][sess_type][be_id];
		for (j = 0; j < MAX_COPPS_PER_PORT; j++) {
			if (test_bit(j, &copp)) {
				copp_idx = j;
				break;
			}
		}

		ret = adm_programable_channel_mixer(
			msm_bedais[be_id].port_id,
			copp_idx, dspst_id, sess_type,
			&channel_mixer_v2[fe_id][sess_type], 0);
	}

	return ret;
}

static int msm_pcm_routing_channel_mixer(int fe_id, bool perf_mode,
				int dspst_id, int stream_type)
{
	int copp_idx = 0;
	int sess_type = 0;
	int i = 0, j = 0, be_id;
	int ret = 0;

	ret = msm_pcm_routing_channel_mixer_v2(fe_id, perf_mode,
				dspst_id, stream_type);
	if (ret) {
		pr_err("%s channel mixer v2 cmd  set failure%d\n", __func__,
				fe_id);
		return ret;
	}

	if (fe_id >= MSM_FRONTEND_DAI_MAX) {
		pr_err("%s: invalid FE %d\n", __func__, fe_id);
		return 0;
	}

	if (!(channel_mixer[fe_id].enable)) {
		pr_debug("%s: channel mixer not enabled for FE %d\n",
			__func__, fe_id);
		return 0;
	}

	if (stream_type == SNDRV_PCM_STREAM_PLAYBACK)
		sess_type = SESSION_TYPE_RX;
	else
		sess_type = SESSION_TYPE_TX;

	for (i = 0; i < ADM_MAX_CHANNELS && channel_input[fe_id][i] > 0;
		++i) {
		be_id = channel_input[fe_id][i] - 1;
		if (be_id < 0 || be_id >= MSM_BACKEND_DAI_MAX) {
			pr_err("%s: Received out of bounds be_id %d\n",
					__func__, be_id);
			return -EINVAL;
		}
		channel_mixer[fe_id].input_channels[i] =
						msm_bedais[be_id].channel;

		if ((msm_bedais[be_id].active) &&
			test_bit(fe_id,
			&msm_bedais[be_id].fe_sessions[0])) {
			unsigned long copp =
				session_copp_map[fe_id][sess_type][be_id];
			for (j = 0; j < MAX_COPPS_PER_PORT; j++) {
				if (test_bit(j, &copp)) {
					copp_idx = j;
					break;
				}
			}

			pr_debug("%s: fe %d, be %d, channel %d, copp %d\n",
				__func__,
				fe_id, be_id, msm_bedais[be_id].channel,
				copp_idx);
			ret = adm_programable_channel_mixer(
					get_port_id(msm_bedais[be_id].port_id),
					copp_idx, dspst_id, sess_type,
					channel_mixer + fe_id, i);
		}
	}

	return ret;
}

/**
 * msm_pcm_routing_set_channel_mixer_runtime - apply channel mixer
 * setting during runtime.
 *
 * @be_id: backend index
 * @session_id: session index
 * @session_type: session type
 * @params: parameters for channel mixer
 *
 * Retuen: 0 for success, else error
 */
int msm_pcm_routing_set_channel_mixer_runtime(int be_id, int session_id,
			int session_type,
			struct msm_pcm_channel_mixer *params)
{
	int rc = 0;
	int port_id, copp_idx = 0;

	be_id--;
	if (be_id < 0 || be_id >= MSM_BACKEND_DAI_MAX) {
		pr_err("%s: invalid backend id %d\n", __func__,
				be_id);
		return -EINVAL;
	}

	port_id = msm_bedais[be_id].port_id;
	copp_idx = adm_get_default_copp_idx(port_id);
	pr_debug("%s: port_id - %d, copp_idx %d session id - %d\n",
		 __func__, port_id, copp_idx, session_id);

	if ((params->input_channel < 0) ||
		(params->input_channel > ADM_MAX_CHANNELS)) {
		pr_err("%s: invalid input channel %d\n", __func__,
				params->input_channel);
		return -EINVAL;
	}

	if ((params->output_channel < 0) ||
		(params->output_channel > ADM_MAX_CHANNELS)) {
		pr_err("%s: invalid output channel %d\n", __func__,
				params->output_channel);
		return -EINVAL;
	}

	params->input_channels[0] = params->input_channel;

	pr_debug("%s sess type %d,override in:%d out:%d,be active %d\n",
			__func__, session_type,
			params->override_in_ch_map,
			params->override_out_ch_map,
			msm_bedais[be_id].active);

	rc = adm_programable_channel_mixer(port_id,
					copp_idx,
					session_id,
					session_type,
					params,
					0);
	if (rc) {
		pr_err("%s: send params failed rc=%d\n", __func__, rc);
		rc = -EINVAL;
	}
	return rc;
}
EXPORT_SYMBOL(msm_pcm_routing_set_channel_mixer_runtime);

int msm_pcm_routing_reg_phy_stream(int fedai_id, int perf_mode,
					int dspst_id, int stream_type)
{
	int i, j, session_type, path_type, port_type, topology, num_copps = 0;
	struct route_payload payload;
	u32 channels, sample_rate;
	uint16_t bits_per_sample = 16, be_bit_width;
	uint32_t passthr_mode = LEGACY_PCM;
	int ret = 0;
	uint32_t copp_token = 0;
	int copp_perf_mode = 0;

	if (fedai_id >= MSM_FRONTEND_DAI_MM_MAX_ID) {
		/* bad ID assigned in machine driver */
		pr_err("%s: bad MM ID %d\n", __func__, fedai_id);
		return -EINVAL;
	}

	if (stream_type == SNDRV_PCM_STREAM_PLAYBACK) {
		session_type = SESSION_TYPE_RX;
		path_type = ADM_PATH_PLAYBACK;
		port_type = MSM_AFE_PORT_TYPE_RX;
	} else {
		session_type = SESSION_TYPE_TX;
		path_type = ADM_PATH_LIVE_REC;
		port_type = MSM_AFE_PORT_TYPE_TX;
	}

	mutex_lock(&routing_lock);

	payload.num_copps = 0; /* only RX needs to use payload */
	fe_dai_map[fedai_id][session_type].strm_id = dspst_id;
	fe_dai_map[fedai_id][session_type].perf_mode = perf_mode;
	fe_dai_map[fedai_id][session_type].passthr_mode = LEGACY_PCM;

	/* re-enable EQ if active */
	msm_qti_pp_send_eq_values(fedai_id);
	for (i = 0; i < MSM_BACKEND_DAI_MAX; i++) {
		if (!is_be_dai_extproc(i) &&
		   (afe_get_port_type(msm_bedais[i].port_id) == port_type) &&
		   (msm_bedais[i].active) &&
		   (test_bit(fedai_id, &msm_bedais[i].fe_sessions[0]))) {
			int app_type, app_type_idx, copp_idx, acdb_dev_id;
			int port_id = get_port_id(msm_bedais[i].port_id);

			/*
			 * check if ADM needs to be configured with different
			 * channel mapping than backend
			 */
			if (!msm_bedais[i].adm_override_ch)
				channels = msm_bedais[i].channel;
			else
				channels = msm_bedais[i].adm_override_ch;

			bits_per_sample = msm_routing_get_bit_width(
						msm_bedais[i].format);

			topology = msm_routing_get_adm_topology(fedai_id,
								session_type,
								i);
			app_type =
			fe_dai_app_type_cfg[fedai_id][session_type][i].app_type;
			if (app_type) {
				app_type_idx =
				msm_pcm_routing_get_app_type_idx(app_type);
				if ((fe_dai_app_type_cfg[fedai_id][session_type][i].sample_rate != msm_bedais[i].sample_rate) &&
					(topology == AUDIO_COPP_MFC)) {
					pr_debug("%s: FE and BE SR is not equal", __func__);
					sample_rate = msm_bedais[i].sample_rate;
				} else
					sample_rate = fe_dai_app_type_cfg[fedai_id][session_type][i].sample_rate;
				bits_per_sample =
					(fe_dai_app_type_cfg[fedai_id][session_type][i].bit_width) ?
					fe_dai_app_type_cfg[fedai_id][session_type][i].bit_width :
					app_type_cfg[app_type_idx].bit_width;
				copp_token =
				fe_dai_app_type_cfg[fedai_id][session_type][i]
					.copp_token;
			} else
				sample_rate = msm_bedais[i].sample_rate;

			acdb_dev_id =
			fe_dai_app_type_cfg[fedai_id][session_type][i]
				.acdb_dev_id;

			be_bit_width = msm_routing_get_bit_width(
                                                msm_bedais[i].format);
			copp_perf_mode = get_copp_perf_mode(fedai_id, session_type, i);
			if (hifi_filter_enabled && (msm_bedais[i].sample_rate ==
                                384000 ||msm_bedais[i].sample_rate == 352800)
				&& be_bit_width == 32)
				bits_per_sample = msm_routing_get_bit_width(
							SNDRV_PCM_FORMAT_S32_LE);
			if ((session_type == SESSION_TYPE_TX) &&
				 (ec_ref_chmix_cfg[fedai_id].output_channel)) {
				/* per-session ec_ref configuration */
				ec_ref_port_cfg.rx = msm_route_ec_ref_rx;
				ec_ref_port_cfg.port_id = msm_ec_ref_port_id;
				ec_ref_port_cfg.ch = msm_ec_ref_ch;
				ec_ref_port_cfg.sampling_rate =
						msm_ec_ref_sampling_rate;
				if (msm_ec_ref_bit_format ==
						SNDRV_PCM_FORMAT_S16_LE)
					ec_ref_port_cfg.bit_width = 16;
				else if (msm_ec_ref_bit_format ==
						SNDRV_PCM_FORMAT_S24_LE)
					ec_ref_port_cfg.bit_width = 24;

				copp_idx = adm_open_v2(port_id, path_type,
						sample_rate, channels, topology,
						copp_perf_mode, bits_per_sample,
						app_type, acdb_dev_id,
						session_type, passthr_mode,
						copp_token,
						&ec_ref_port_cfg,
						&ec_ref_chmix_cfg[fedai_id]);
				/* reset ec_ref config */
				ec_ref_chmix_cfg[fedai_id].output_channel = 0;
			}
#ifdef RX_TO_TX_LOOPBACK
			else if(session_type == SESSION_TYPE_TX &&
					port_id == RX_TO_TX_LOOPBACK_DUMMY_TX_PORT) {
				port_id = RX_TO_TX_LOOPBACK_RX_PORT;
				copp_idx = adm_open(port_id, path_type,
				sample_rate, channels, topology,
				copp_perf_mode, bits_per_sample,
				app_type, acdb_dev_id,
				session_type, passthr_mode, copp_token);
			}
#endif
			else
				copp_idx = adm_open(port_id, path_type,
					    sample_rate, channels, topology,
					    copp_perf_mode, bits_per_sample,
					    app_type, acdb_dev_id,
					    session_type, passthr_mode, copp_token);
			if ((copp_idx < 0) ||
				(copp_idx >= MAX_COPPS_PER_PORT)) {
				pr_err("%s: adm open failed copp_idx:%d\n",
				       __func__, copp_idx);
				mutex_unlock(&routing_lock);
				return -EINVAL;
			}
			pr_debug("%s: setting idx bit of fe:%d, type: %d, be:%d\n",
				 __func__, fedai_id, session_type, i);
			set_bit(copp_idx,
				&session_copp_map[fedai_id][session_type][i]);

			if (msm_is_resample_needed(
				fe_dai_app_type_cfg[fedai_id][session_type][i].sample_rate,
				msm_bedais[i].sample_rate)) {
				if ((session_type == SESSION_TYPE_TX) && (topology == AUDIO_COPP_MFC)) {
					adm_copp_mfc_cfg(port_id, copp_idx,
						fe_dai_app_type_cfg[fedai_id][session_type][i].sample_rate);
				} else {
					adm_copp_mfc_cfg(port_id, copp_idx,
						msm_bedais[i].sample_rate);
				}
			}
			for (j = 0; j < MAX_COPPS_PER_PORT; j++) {
				unsigned long copp =
				    session_copp_map[fedai_id][session_type][i];
				if (test_bit(j, &copp)) {
					payload.port_id[num_copps] = port_id;
					payload.copp_idx[num_copps] = j;
					payload.app_type[num_copps] =
						fe_dai_app_type_cfg
							[fedai_id][session_type]
							[i].app_type;
					payload.acdb_dev_id[num_copps] =
						fe_dai_app_type_cfg
							[fedai_id][session_type]
							[i].acdb_dev_id;
					payload.sample_rate[num_copps] =
						fe_dai_app_type_cfg
							[fedai_id][session_type]
							[i].sample_rate;
					num_copps++;
				}
			}
			if (perf_mode == LEGACY_PCM_MODE)
				msm_pcm_routing_cfg_pp(port_id, copp_idx,
						       topology, channels);
		}
	}
	if (num_copps) {
		payload.num_copps = num_copps;
		payload.session_id = fe_dai_map[fedai_id][session_type].strm_id;
		adm_matrix_map(fedai_id, path_type, payload, perf_mode, passthr_mode);
		msm_pcm_routng_cfg_matrix_map_pp(payload, path_type, perf_mode);
	}

	ret = msm_pcm_routing_channel_mixer(fedai_id, perf_mode,
				dspst_id, stream_type);
	mutex_unlock(&routing_lock);
	return ret;
}

int msm_pcm_routing_reg_phy_stream_v2(int fedai_id, int perf_mode,
				      int dspst_id, int stream_type,
				      struct msm_pcm_routing_evt event_info)
{
	if (msm_pcm_routing_reg_phy_stream(fedai_id, perf_mode, dspst_id,
				       stream_type)) {
		pr_err("%s: failed to reg phy stream\n", __func__);
		return -EINVAL;
	}

	if (stream_type == SNDRV_PCM_STREAM_PLAYBACK)
		fe_dai_map[fedai_id][SESSION_TYPE_RX].event_info = event_info;
	else
		fe_dai_map[fedai_id][SESSION_TYPE_TX].event_info = event_info;
	return 0;
}

void msm_pcm_routing_dereg_phy_stream(int fedai_id, int stream_type)
{
	int i, port_type, session_type, path_type, topology, port_id;
	struct msm_pcm_routing_fdai_data *fdai;
	int copp_perf_mode;

	if (!is_mm_lsm_fe_id(fedai_id)) {
		/* bad ID assigned in machine driver */
		pr_err("%s: bad MM ID\n", __func__);
		return;
	}

	if (stream_type == SNDRV_PCM_STREAM_PLAYBACK) {
		port_type = MSM_AFE_PORT_TYPE_RX;
		session_type = SESSION_TYPE_RX;
		path_type = ADM_PATH_PLAYBACK;
	} else {
		port_type = MSM_AFE_PORT_TYPE_TX;
		session_type = SESSION_TYPE_TX;
		path_type = ADM_PATH_LIVE_REC;
	}

	mutex_lock(&routing_lock);
	for (i = 0; i < MSM_BACKEND_DAI_MAX; i++) {
		if (!is_be_dai_extproc(i) &&
		   (afe_get_port_type(msm_bedais[i].port_id) == port_type) &&
		   (msm_bedais[i].active) &&
		   (test_bit(fedai_id, &msm_bedais[i].fe_sessions[0]))) {
			int idx;
			unsigned long copp =
				session_copp_map[fedai_id][session_type][i];
			fdai = &fe_dai_map[fedai_id][session_type];

			for (idx = 0; idx < MAX_COPPS_PER_PORT; idx++)
				if (test_bit(idx, &copp))
					break;

			if (idx >= MAX_COPPS_PER_PORT || idx < 0) {
				pr_debug("%s: copp idx is invalid, exiting\n",
								__func__);
				continue;
			}
			port_id = get_port_id(msm_bedais[i].port_id);
			topology = adm_get_topology_for_port_copp_idx(
					port_id, idx);
			msm_routing_unload_topology(topology);
			copp_perf_mode = get_copp_perf_mode(fedai_id, session_type, i);
			adm_close(port_id, copp_perf_mode, idx);
			pr_debug("%s:copp:%ld,idx bit fe:%d,type:%d,be:%d\n",
				 __func__, copp, fedai_id, session_type, i);
			clear_bit(idx,
				  &session_copp_map[fedai_id][session_type][i]);
			if ((topology == DOLBY_ADM_COPP_TOPOLOGY_ID ||
				topology == DS2_ADM_COPP_TOPOLOGY_ID) &&
			    (fdai->perf_mode == LEGACY_PCM_MODE) &&
			    (fdai->passthr_mode == LEGACY_PCM))
				msm_pcm_routing_deinit_pp(port_id, topology);
		}
	}

	fe_dai_map[fedai_id][session_type].strm_id = INVALID_SESSION;
	fe_dai_map[fedai_id][session_type].be_srate = 0;
	mutex_unlock(&routing_lock);
}

/* Check if FE/BE route is set */
static bool msm_pcm_routing_route_is_set(u16 be_id, u16 fe_id)
{
	bool rc = false;

	if (!is_mm_lsm_fe_id(fe_id)) {
		/* recheck FE ID in the mixer control defined in this file */
		pr_err("%s: bad MM ID\n", __func__);
		return rc;
	}

	if (test_bit(fe_id, &msm_bedais[be_id].fe_sessions[0]))
		rc = true;

	return rc;
}

static void msm_pcm_routing_process_audio(u16 reg, u16 val, int set)
{
	int session_type, path_type, topology;
	u32 channels, sample_rate;
	uint16_t bits_per_sample = 16, be_bit_width;
	struct msm_pcm_routing_fdai_data *fdai;
	uint32_t passthr_mode;
	bool is_lsm;
	uint32_t copp_token = 0;
	int copp_perf_mode = 0;

	pr_debug("%s: reg %x val %x set %x\n", __func__, reg, val, set);

	if (val == MSM_FRONTEND_DAI_DTMF_RX &&
		afe_get_port_type(msm_bedais[reg].port_id) ==
			MSM_AFE_PORT_TYPE_RX) {
		pr_debug("%s(): set=%d port id=0x%x for dtmf generation\n",
			__func__, set, msm_bedais[reg].port_id);
		afe_set_dtmf_gen_rx_portid(msm_bedais[reg].port_id, set);
	} else if (!is_mm_lsm_fe_id(val)) {
		/* recheck FE ID in the mixer control defined in this file */
		pr_err("%s: bad MM ID\n", __func__);
		return;
	}

	if (!route_check_fe_id_adm_support(val)) {
		/* ignore adm open if not supported for fe_id */
		pr_debug("%s: No ADM support for fe id %d\n", __func__, val);
		if (set)
			set_bit(val, &msm_bedais[reg].fe_sessions[0]);
		else
			clear_bit(val, &msm_bedais[reg].fe_sessions[0]);
		return;
	}

	session_type =
		(afe_get_port_type(msm_bedais[reg].port_id) == MSM_AFE_PORT_TYPE_RX) ?
		SESSION_TYPE_RX : SESSION_TYPE_TX;
	fdai = &fe_dai_map[val][session_type];
	passthr_mode = fdai->passthr_mode;
	if (session_type == SESSION_TYPE_RX) {
		if (passthr_mode != LEGACY_PCM)
			path_type = ADM_PATH_COMPRESSED_RX;
		else
			path_type = ADM_PATH_PLAYBACK;
	} else {
		if ((passthr_mode != LEGACY_PCM) && (passthr_mode != LISTEN))
			path_type = ADM_PATH_COMPRESSED_TX;
		else
			path_type = ADM_PATH_LIVE_REC;
	}
	is_lsm = (val >= MSM_FRONTEND_DAI_LSM1) &&
			 (val <= MSM_FRONTEND_DAI_LSM8);

	mutex_lock(&routing_lock);
	if (set) {
		if (!test_bit(val, &msm_bedais[reg].fe_sessions[0]) &&
			((msm_bedais[reg].port_id == VOICE_PLAYBACK_TX) ||
			(msm_bedais[reg].port_id == VOICE2_PLAYBACK_TX)))
			voc_start_playback(set, msm_bedais[reg].port_id);

		set_bit(val, &msm_bedais[reg].fe_sessions[0]);
		if (msm_bedais[reg].active && fdai->strm_id !=
			INVALID_SESSION) {
			int app_type, app_type_idx, copp_idx, acdb_dev_id;
			int port_id = get_port_id(msm_bedais[reg].port_id);
			/*
			 * check if ADM needs to be configured with different
			 * channel mapping than backend
			 */
			if (!msm_bedais[reg].adm_override_ch)
				channels = msm_bedais[reg].channel;
			else
				channels = msm_bedais[reg].adm_override_ch;
			if (session_type == SESSION_TYPE_TX &&
			    fdai->be_srate &&
			    (fdai->be_srate != msm_bedais[reg].sample_rate)) {
				pr_debug("%s: flush strm %d diff BE rates\n",
					__func__, fdai->strm_id);

				if (fdai->event_info.event_func)
					fdai->event_info.event_func(
						MSM_PCM_RT_EVT_BUF_RECFG,
						fdai->event_info.priv_data);
				fdai->be_srate = 0; /* might not need it */
			}

			bits_per_sample = msm_routing_get_bit_width(
						msm_bedais[reg].format);

			app_type =
			fe_dai_app_type_cfg[val][session_type][reg].app_type;
			if (app_type && is_lsm) {
				app_type_idx =
				msm_pcm_routing_get_lsm_app_type_idx(app_type);
				sample_rate =
				fe_dai_app_type_cfg[val][session_type][reg]
					.sample_rate;
				bits_per_sample =
				lsm_app_type_cfg[app_type_idx].bit_width;
			} else if (app_type) {
				app_type_idx =
				msm_pcm_routing_get_app_type_idx(app_type);
				sample_rate =
				fe_dai_app_type_cfg[val][session_type][reg]
					.sample_rate;
				bits_per_sample =
					(fe_dai_app_type_cfg[val][session_type][reg].bit_width) ?
					fe_dai_app_type_cfg[val][session_type][reg].bit_width :
					app_type_cfg[app_type_idx].bit_width;
				copp_token =
				fe_dai_app_type_cfg[val][session_type][reg]
					.copp_token;
			} else
				sample_rate = msm_bedais[reg].sample_rate;

			topology = msm_routing_get_adm_topology(val,
								session_type,
								reg);
			acdb_dev_id =
			fe_dai_app_type_cfg[val][session_type][reg].acdb_dev_id;

			be_bit_width = msm_routing_get_bit_width(
                                                msm_bedais[reg].format);
			copp_perf_mode = get_copp_perf_mode(val, session_type, reg);
			if (hifi_filter_enabled && (msm_bedais[reg].sample_rate
				== 384000 ||msm_bedais[reg].sample_rate ==
				352800) && be_bit_width == 32)
				bits_per_sample = msm_routing_get_bit_width(
							SNDRV_PCM_FORMAT_S32_LE);
			copp_idx = adm_open(port_id, path_type,
					    sample_rate, channels, topology,
					    copp_perf_mode, bits_per_sample,
					    app_type, acdb_dev_id,
					    session_type, passthr_mode, copp_token);
			if ((copp_idx < 0) ||
			    (copp_idx >= MAX_COPPS_PER_PORT)) {
				pr_err("%s: adm open failed\n", __func__);
				mutex_unlock(&routing_lock);
				return;
			}
			pr_debug("%s: setting idx bit of fe:%d, type: %d, be:%d\n",
				 __func__, val, session_type, reg);
			set_bit(copp_idx,
				&session_copp_map[val][session_type][reg]);

			if (msm_is_resample_needed(
				sample_rate,
				msm_bedais[reg].sample_rate))
				adm_copp_mfc_cfg(port_id, copp_idx,
					msm_bedais[reg].sample_rate);

			if (session_type == SESSION_TYPE_RX &&
			    fdai->event_info.event_func)
				fdai->event_info.event_func(
					MSM_PCM_RT_EVT_DEVSWITCH,
					fdai->event_info.priv_data);

			msm_pcm_routing_build_matrix(val, session_type,
						     path_type,
						     fdai->perf_mode,
						     passthr_mode);
			if ((fdai->perf_mode == LEGACY_PCM_MODE) &&
				(passthr_mode == LEGACY_PCM))
				msm_pcm_routing_cfg_pp(port_id, copp_idx,
						       topology, channels);
		}
	} else {
		if (test_bit(val, &msm_bedais[reg].fe_sessions[0]) &&
			((msm_bedais[reg].port_id == VOICE_PLAYBACK_TX) ||
			(msm_bedais[reg].port_id == VOICE2_PLAYBACK_TX)))
			voc_start_playback(set, msm_bedais[reg].port_id);
		clear_bit(val, &msm_bedais[reg].fe_sessions[0]);
		if (msm_bedais[reg].active && fdai->strm_id !=
			INVALID_SESSION) {
			int idx;
			int port_id;
			unsigned long copp =
				session_copp_map[val][session_type][reg];
			for (idx = 0; idx < MAX_COPPS_PER_PORT; idx++)
				if (test_bit(idx, &copp))
					break;

			if (idx >= MAX_COPPS_PER_PORT) {
				pr_debug("%s: copp idx is invalid, exiting\n",
								__func__);
				mutex_unlock(&routing_lock);
				return;
			}
			port_id = get_port_id(msm_bedais[reg].port_id);
			topology = adm_get_topology_for_port_copp_idx(port_id,
								      idx);
			msm_routing_unload_topology(topology);
			copp_perf_mode = get_copp_perf_mode(val, session_type, reg);
			adm_close(port_id, copp_perf_mode, idx);
			pr_debug("%s: copp: %ld, reset idx bit fe:%d, type: %d, be:%d topology=0x%x\n",
				 __func__, copp, val, session_type, reg,
				 topology);
			clear_bit(idx,
				  &session_copp_map[val][session_type][reg]);
			if ((topology == DOLBY_ADM_COPP_TOPOLOGY_ID ||
				topology == DS2_ADM_COPP_TOPOLOGY_ID) &&
			    (fdai->perf_mode == LEGACY_PCM_MODE) &&
			    (passthr_mode == LEGACY_PCM))
				msm_pcm_routing_deinit_pp(port_id, topology);
			msm_pcm_routing_build_matrix(val, session_type,
						     path_type,
						     fdai->perf_mode,
						     passthr_mode);
		}
	}
	if ((msm_bedais[reg].port_id == VOICE_RECORD_RX)
			|| (msm_bedais[reg].port_id == VOICE_RECORD_TX))
		voc_start_record(msm_bedais[reg].port_id, set, voc_session_id);

	mutex_unlock(&routing_lock);
}

static int msm_routing_get_audio_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct soc_mixer_control *mc =
	(struct soc_mixer_control *)kcontrol->private_value;

	if (test_bit(mc->rshift, &msm_bedais[mc->shift].fe_sessions[0]))
		ucontrol->value.integer.value[0] = 1;
	else
		ucontrol->value.integer.value[0] = 0;

	pr_debug("%s: shift %x rshift %x val %ld\n", __func__, mc->shift, mc->rshift,
	ucontrol->value.integer.value[0]);

	return 0;
}

static int msm_routing_put_audio_mixer(struct snd_kcontrol *kcontrol,
			struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct soc_mixer_control *mc =
		(struct soc_mixer_control *)kcontrol->private_value;
	struct snd_soc_dapm_update *update = NULL;

	if (ucontrol->value.integer.value[0] &&
	   msm_pcm_routing_route_is_set(mc->shift, mc->rshift) == false) {
		msm_pcm_routing_process_audio(mc->shift, mc->rshift, 1);
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
			update);
	} else if (!ucontrol->value.integer.value[0] &&
		  msm_pcm_routing_route_is_set(mc->shift, mc->rshift) == true) {
		msm_pcm_routing_process_audio(mc->shift, mc->rshift, 0);
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
			update);
	}

	return 1;
}

static int msm_routing_get_listen_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct soc_mixer_control *mc =
	(struct soc_mixer_control *)kcontrol->private_value;

	if (test_bit(mc->rshift, &msm_bedais[mc->shift].fe_sessions[0]))
		ucontrol->value.integer.value[0] = 1;
	else
		ucontrol->value.integer.value[0] = 0;

	pr_debug("%s: shift %x rshift %x val %ld\n", __func__, mc->shift, mc->rshift,
		ucontrol->value.integer.value[0]);

	return 0;
}

static int msm_routing_put_listen_mixer(struct snd_kcontrol *kcontrol,
			struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct soc_mixer_control *mc =
		(struct soc_mixer_control *)kcontrol->private_value;
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: shift %x rshift %x val %ld\n", __func__, mc->shift, mc->rshift,
		ucontrol->value.integer.value[0]);

	if (ucontrol->value.integer.value[0]) {
		if (msm_pcm_routing_route_is_set(mc->shift, mc->rshift) == false)
			msm_pcm_routing_process_audio(mc->shift, mc->rshift, 1);
		snd_soc_dapm_mixer_update_power(widget->dapm,
						kcontrol, 1, update);
	} else if (!ucontrol->value.integer.value[0]) {
		if (msm_pcm_routing_route_is_set(mc->shift, mc->rshift) == true)
			msm_pcm_routing_process_audio(mc->shift, mc->rshift, 0);
		snd_soc_dapm_mixer_update_power(widget->dapm,
						kcontrol, 0, update);
	}
	return 1;
}

/*
 * Return the mapping between port ID and backend ID to enable the AFE callback
 * to determine the acdb_dev_id from the port id
 */
int msm_pcm_get_be_id_from_port_id(int port_id)
{
	int i;
	int be_id = -EINVAL;

	for (i = 0; i < MSM_BACKEND_DAI_MAX; i++) {
		if (msm_bedais[i].port_id == port_id) {
			be_id = i;
			break;
		}
	}

	return be_id;
}

/*
 * Return the registered dev_acdb_id given a port ID to enable identifying the
 * correct AFE calibration information by comparing the header information.
 */
static int msm_pcm_get_dev_acdb_id_by_port_id(int port_id)
{
	int acdb_id = -EINVAL;
	int i = 0;
	int session;
	int port_type = afe_get_port_type(port_id);
	int be_id = msm_pcm_get_be_id_from_port_id(port_id);

	pr_debug("%s:port_id %d be_id %d, port_type 0x%x\n",
		  __func__, port_id, be_id, port_type);

	if (port_type == MSM_AFE_PORT_TYPE_TX) {
		session = SESSION_TYPE_TX;
	} else if (port_type == MSM_AFE_PORT_TYPE_RX) {
		session = SESSION_TYPE_RX;
	} else {
		pr_err("%s: Invalid port type %d\n", __func__, port_type);
		acdb_id = -EINVAL;
		goto exit;
	}

	if (be_id < 0) {
		pr_err("%s: Error getting backend id %d\n", __func__, be_id);
		goto exit;
	}

	mutex_lock(&routing_lock);
	i = find_first_bit(&msm_bedais[be_id].fe_sessions[0],
			   MSM_FRONTEND_DAI_MAX);
	if (i < MSM_FRONTEND_DAI_MAX)
		acdb_id = fe_dai_app_type_cfg[i][session][be_id].acdb_dev_id;

	pr_debug("%s: FE[%d] session[%d] BE[%d] acdb_id(%d)\n",
		 __func__, i, session, be_id, acdb_id);
	mutex_unlock(&routing_lock);
exit:
	return acdb_id;
}

static int msm_routing_get_hfp_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = hfp_switch_enable;
	pr_debug("%s: HFP Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_hfp_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: HFP Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol,
						1, update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol,
						0, update);
	hfp_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_a2dp_switch_mixer_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = a2dp_switch_enable;
	pr_debug("%s: A2DP Switch enable %ld\n", __func__,
		  ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_a2dp_switch_mixer_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: A2DP Switch enable %ld\n", __func__,
		  ucontrol->value.integer.value[0]);
	a2dp_switch_enable = ucontrol->value.integer.value[0];
	if (a2dp_switch_enable)
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol,
						1, update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol,
						0, update);
	return 1;
}

static int msm_routing_sco_switch_mixer_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = sco_switch_enable;
	pr_debug("%s: SCO Switch enable %ld\n", __func__,
		  ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_sco_switch_mixer_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: SCO Switch enable %ld\n", __func__,
		  ucontrol->value.integer.value[0]);
	sco_switch_enable = ucontrol->value.integer.value[0];
	if (sco_switch_enable)
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol,
						1, update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol,
						0, update);
	return 1;
}

#ifndef CONFIG_MI2S_DISABLE
static int msm_routing_get_int0_mi2s_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = int0_mi2s_switch_enable;
	pr_debug("%s: INT0 MI2S Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_int0_mi2s_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: INT0 MI2S Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	int0_mi2s_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_get_int4_mi2s_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = int4_mi2s_switch_enable;
	pr_debug("%s: INT4 MI2S Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_int4_mi2s_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: INT4 MI2S Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	int4_mi2s_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_get_pri_mi2s_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = pri_mi2s_switch_enable;
	pr_debug("%s: PRI MI2S Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_pri_mi2s_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: PRI MI2S Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	pri_mi2s_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_get_sec_mi2s_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = sec_mi2s_switch_enable;
	pr_debug("%s: SEC MI2S Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_sec_mi2s_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: SEC MI2S Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	sec_mi2s_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_get_tert_mi2s_switch_mixer(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = tert_mi2s_switch_enable;
	pr_debug("%s: TERT MI2S Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_tert_mi2s_switch_mixer(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: TERT MI2S Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	tert_mi2s_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_get_quat_mi2s_switch_mixer(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = quat_mi2s_switch_enable;
	pr_debug("%s: QUAT MI2S Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_quat_mi2s_switch_mixer(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: QUAT MI2S Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	quat_mi2s_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_get_quin_mi2s_switch_mixer(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = quin_mi2s_switch_enable;
	pr_debug("%s: QUIN MI2S Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_quin_mi2s_switch_mixer(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: QUIN MI2S Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	quin_mi2s_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_get_sen_mi2s_switch_mixer(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = sen_mi2s_switch_enable;
	pr_debug("%s: SEN MI2S Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_sen_mi2s_switch_mixer(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: SEN MI2S Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	sen_mi2s_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}
#endif

static int msm_routing_get_usb_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = usb_switch_enable;
	pr_debug("%s: HFP Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_usb_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: USB Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol,
						1, update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol,
						0, update);
	usb_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static int msm_routing_get_fm_pcmrx_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = fm_pcmrx_switch_enable;
	pr_debug("%s: FM Switch enable %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_routing_put_fm_pcmrx_switch_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct snd_soc_dapm_update *update = NULL;

	pr_debug("%s: FM Switch enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	if (ucontrol->value.integer.value[0])
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 1,
						update);
	else
		snd_soc_dapm_mixer_update_power(widget->dapm, kcontrol, 0,
						update);
	fm_pcmrx_switch_enable = ucontrol->value.integer.value[0];
	return 1;
}

static void msm_routing_get_lsm_fe_idx(struct snd_kcontrol *kcontrol,
						u8 *fe_idx)
{
	int fe_id = MSM_FRONTEND_DAI_LSM1;

	if (strnstr(kcontrol->id.name, "LSM1", sizeof("LSM1"))) {
		fe_id = MSM_FRONTEND_DAI_LSM1;
	} else if (strnstr(kcontrol->id.name, "LSM2", sizeof("LSM2"))) {
		fe_id = MSM_FRONTEND_DAI_LSM2;
	} else if (strnstr(kcontrol->id.name, "LSM3", sizeof("LSM3"))) {
		fe_id = MSM_FRONTEND_DAI_LSM3;
	} else if (strnstr(kcontrol->id.name, "LSM4", sizeof("LSM4"))) {
		fe_id = MSM_FRONTEND_DAI_LSM4;
	} else if (strnstr(kcontrol->id.name, "LSM5", sizeof("LSM5"))) {
		fe_id = MSM_FRONTEND_DAI_LSM5;
	} else if (strnstr(kcontrol->id.name, "LSM6", sizeof("LSM6"))) {
		fe_id = MSM_FRONTEND_DAI_LSM6;
	} else if (strnstr(kcontrol->id.name, "LSM7", sizeof("LSM7"))) {
		fe_id = MSM_FRONTEND_DAI_LSM7;
	} else if (strnstr(kcontrol->id.name, "LSM8", sizeof("LSM8"))) {
		fe_id = MSM_FRONTEND_DAI_LSM8;
	} else {
		pr_err("%s: Invalid kcontrol name:%s\n", __func__,
				kcontrol->id.name);
		return;
	}

	*fe_idx = fe_id - MSM_FRONTEND_DAI_LSM1;
	pr_debug("%s: fe_id: %d, fe_idx:%d\n", __func__, fe_id, *fe_idx);
}

static int msm_routing_lsm_port_get(struct snd_kcontrol *kcontrol,
				   struct snd_ctl_elem_value *ucontrol)
{
	u8 fe_idx = 0;

	msm_routing_get_lsm_fe_idx(kcontrol, &fe_idx);
	ucontrol->value.integer.value[0] = lsm_port_index[fe_idx];
	return 0;
}

static int msm_routing_lsm_port_put(struct snd_kcontrol *kcontrol,
				   struct snd_ctl_elem_value *ucontrol)
{
	struct soc_enum *e = (struct soc_enum *)kcontrol->private_value;
	int mux = ucontrol->value.enumerated.item[0];
	int lsm_port = AFE_PORT_ID_SLIMBUS_MULTI_CHAN_5_TX;
	int lsm_port_idx = 0;
	u8 fe_idx = 0;

	if (mux >= e->items) {
		pr_err("%s: Invalid mux value %d\n", __func__, mux);
		return -EINVAL;
	}

	pr_debug("%s: LSM enable %ld\n", __func__,
			ucontrol->value.integer.value[0]);
	lsm_port_idx = ucontrol->value.integer.value[0];
	switch (ucontrol->value.integer.value[0]) {
	case 1:
		lsm_port = AFE_PORT_ID_SLIMBUS_MULTI_CHAN_0_TX;
		break;
	case 2:
		lsm_port = AFE_PORT_ID_SLIMBUS_MULTI_CHAN_1_TX;
		break;
	case 3:
		lsm_port = AFE_PORT_ID_SLIMBUS_MULTI_CHAN_2_TX;
		break;
	case 4:
		lsm_port = AFE_PORT_ID_SLIMBUS_MULTI_CHAN_3_TX;
		break;
	case 5:
		lsm_port = AFE_PORT_ID_SLIMBUS_MULTI_CHAN_4_TX;
		break;
	case 6:
		lsm_port = AFE_PORT_ID_SLIMBUS_MULTI_CHAN_5_TX;
		break;
	case 7:
		lsm_port = AFE_PORT_ID_TERTIARY_MI2S_TX;
		break;
	case 8:
		lsm_port = AFE_PORT_ID_QUATERNARY_MI2S_TX;
		break;
	case 9:
		lsm_port = ADM_LSM_PORT_ID;
		break;
	case 10:
		lsm_port = AFE_PORT_ID_INT3_MI2S_TX;
		break;
	case 11:
		lsm_port = AFE_PORT_ID_VA_CODEC_DMA_TX_0;
		break;
	case 12:
		lsm_port = AFE_PORT_ID_VA_CODEC_DMA_TX_1;
		break;
	case 13:
		lsm_port = AFE_PORT_ID_VA_CODEC_DMA_TX_2;
		break;
	case 14:
		lsm_port = AFE_PORT_ID_TX_CODEC_DMA_TX_3;
		break;
	case 15:
		lsm_port = AFE_PORT_ID_QUINARY_TDM_TX;
		break;
	case 16:
		lsm_port = AFE_PORT_ID_TERTIARY_TDM_TX;
		break;
	case 17:
		lsm_port = RT_PROXY_PORT_001_TX;
		break;
	default:
		pr_err("Default lsm port");
		break;
	}
	set_lsm_port(lsm_port);
	msm_routing_get_lsm_fe_idx(kcontrol, &fe_idx);
	lsm_port_index[fe_idx] = ucontrol->value.integer.value[0];
        /* Set the default AFE LSM Port to 0xffff */
	if(lsm_port_idx <= 0 || lsm_port_idx >= ARRAY_SIZE(lsm_port_text))
		lsm_port = 0xffff;
	afe_set_lsm_afe_port_id(fe_idx, lsm_port);

	return 0;
}

static int msm_routing_lsm_func_get(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *ucontrol)
{
	int i;
	u16 port_id;
	enum afe_mad_type mad_type;

	pr_debug("%s: id name %s\n", __func__, kcontrol->id.name);
	for (i = 0; i < ARRAY_SIZE(lsm_port_text); i++)
		if (!strnstr(kcontrol->id.name, lsm_port_text[i],
			    strlen(lsm_port_text[i])))
			break;

	if (i-- == ARRAY_SIZE(lsm_port_text)) {
		pr_warn("%s: Invalid id name %s\n", __func__,
			kcontrol->id.name);
		return -EINVAL;
	}

	port_id = i * 2 + 1 + SLIMBUS_0_RX;

	/*Check for Tertiary/Quaternary/INT3 TX port*/
	if (strnstr(kcontrol->id.name, lsm_port_text[7],
			strlen(lsm_port_text[7])))
		port_id = AFE_PORT_ID_TERTIARY_MI2S_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[8],
			strlen(lsm_port_text[8])))
		port_id = AFE_PORT_ID_QUATERNARY_MI2S_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[10],
			strlen(lsm_port_text[10])))
		port_id = AFE_PORT_ID_INT3_MI2S_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[14],
			strlen(lsm_port_text[14])))
		port_id = AFE_PORT_ID_TX_CODEC_DMA_TX_3;

	if (strnstr(kcontrol->id.name, lsm_port_text[15],
			strlen(lsm_port_text[15])))
		port_id = AFE_PORT_ID_QUINARY_TDM_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[16],
			strlen(lsm_port_text[16])))
		port_id = AFE_PORT_ID_TERTIARY_TDM_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[17],
			strlen(lsm_port_text[17])))
		port_id = RT_PROXY_PORT_001_TX;

	mad_type = afe_port_get_mad_type(port_id);
	pr_debug("%s: port_id 0x%x, mad_type %d\n", __func__, port_id,
		 mad_type);
	switch (mad_type) {
	case MAD_HW_NONE:
		ucontrol->value.integer.value[0] = MADNONE;
		break;
	case MAD_HW_AUDIO:
		ucontrol->value.integer.value[0] = MADAUDIO;
		break;
	case MAD_HW_BEACON:
		ucontrol->value.integer.value[0] = MADBEACON;
		break;
	case MAD_HW_ULTRASOUND:
		ucontrol->value.integer.value[0] = MADULTRASOUND;
		break;
	case MAD_SW_AUDIO:
		ucontrol->value.integer.value[0] = MADSWAUDIO;
	break;
	default:
		pr_warn("%s: Unknown\n", __func__);
		return -EINVAL;
	}
	return 0;
}

static int msm_routing_lsm_func_put(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *ucontrol)
{
	int i;
	u16 port_id;
	enum afe_mad_type mad_type;

	pr_debug("%s: id name %s\n", __func__, kcontrol->id.name);
	for (i = 0; i < ARRAY_SIZE(lsm_port_text); i++)
		if (strnstr(kcontrol->id.name, lsm_port_text[i],
			    strlen(lsm_port_text[i])))
			break;

	if (i-- == ARRAY_SIZE(lsm_port_text)) {
		pr_warn("%s: Invalid id name %s\n", __func__,
			kcontrol->id.name);
		return -EINVAL;
	}

	port_id = i * 2 + 1 + SLIMBUS_0_RX;
	switch (ucontrol->value.integer.value[0]) {
	case MADNONE:
		mad_type = MAD_HW_NONE;
		break;
	case MADAUDIO:
		mad_type = MAD_HW_AUDIO;
		break;
	case MADBEACON:
		mad_type = MAD_HW_BEACON;
		break;
	case MADULTRASOUND:
		mad_type = MAD_HW_ULTRASOUND;
		break;
	case MADSWAUDIO:
		mad_type = MAD_SW_AUDIO;
		break;
	default:
		pr_warn("%s: Unknown\n", __func__);
		return -EINVAL;
	}

	/*Check for Tertiary/Quaternary/INT3 TX port*/
	if (strnstr(kcontrol->id.name, lsm_port_text[7],
			strlen(lsm_port_text[7])))
		port_id = AFE_PORT_ID_TERTIARY_MI2S_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[8],
			strlen(lsm_port_text[8])))
		port_id = AFE_PORT_ID_QUATERNARY_MI2S_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[10],
			strlen(lsm_port_text[10])))
		port_id = AFE_PORT_ID_INT3_MI2S_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[14],
			strlen(lsm_port_text[14])))
		port_id = AFE_PORT_ID_TX_CODEC_DMA_TX_3;

	if (strnstr(kcontrol->id.name, lsm_port_text[15],
			strlen(lsm_port_text[15])))
		port_id = AFE_PORT_ID_QUINARY_TDM_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[16],
			strlen(lsm_port_text[16])))
		port_id = AFE_PORT_ID_TERTIARY_TDM_TX;

	if (strnstr(kcontrol->id.name, lsm_port_text[17],
			strlen(lsm_port_text[17])))
		port_id = RT_PROXY_PORT_001_TX;

	pr_debug("%s: port_id 0x%x, mad_type %d\n", __func__, port_id,
		 mad_type);
	return afe_port_set_mad_type(port_id, mad_type);
}

static const char *const adm_override_chs_text[] = {"Zero", "One", "Two", "Three", "Four"};

static SOC_ENUM_SINGLE_EXT_DECL(adm_override_chs,
				adm_override_chs_text);

static int msm_routing_adm_get_backend_idx(struct snd_kcontrol *kcontrol)
{
	int backend_id;

	if (strnstr(kcontrol->id.name, "SLIM7_RX", sizeof("SLIM7_RX"))) {
		backend_id = MSM_BACKEND_DAI_SLIMBUS_7_RX;
	} else if (strnstr(kcontrol->id.name, "SLIM9_TX", sizeof("SLIM9_TX"))) {
		backend_id = MSM_BACKEND_DAI_SLIMBUS_9_TX;
	} else if (strnstr(kcontrol->id.name, "SLIM7_TX", sizeof("SLIM7_TX"))) {
		backend_id = MSM_BACKEND_DAI_SLIMBUS_7_TX;
	} else if (strnstr(kcontrol->id.name, "TERT_TDM_TX_0", sizeof("TERT_TDM_TX_0"))) {
		backend_id = MSM_BACKEND_DAI_TERT_TDM_TX_0;
	} else if (strnstr(kcontrol->id.name, "TERT_TDM_RX_0", sizeof("TERT_TDM_RX_0"))) {
		backend_id = MSM_BACKEND_DAI_TERT_TDM_RX_0;
	} else if (strnstr(kcontrol->id.name, "SEC_TDM_TX_0", sizeof("SEC_TDM_TX_0"))) {
		backend_id = MSM_BACKEND_DAI_SEC_TDM_TX_0;
	} else if (strnstr(kcontrol->id.name, "SEC_TDM_RX_0", sizeof("SEC_TDM_RX_0"))) {
		backend_id = MSM_BACKEND_DAI_SEC_TDM_RX_0;
	} else {
		pr_err("%s: unsupported backend id: %s",
			__func__, kcontrol->id.name);
		return -EINVAL;
	}

	return backend_id;
}

static int msm_routing_adm_channel_config_get(
					struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *ucontrol)
{
	int backend_id = msm_routing_adm_get_backend_idx(kcontrol);

	if (backend_id >= 0) {
		mutex_lock(&routing_lock);
		ucontrol->value.integer.value[0] =
			 msm_bedais[backend_id].adm_override_ch;
		pr_debug("%s: adm channel count %ld for BE:%d\n", __func__,
			 ucontrol->value.integer.value[0], backend_id);
		 mutex_unlock(&routing_lock);
	}

	return 0;
}

static int msm_routing_adm_channel_config_put(
					struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *ucontrol)
{
	int backend_id = msm_routing_adm_get_backend_idx(kcontrol);

	if (backend_id >= 0) {
		mutex_lock(&routing_lock);
		msm_bedais[backend_id].adm_override_ch =
				 ucontrol->value.integer.value[0];
		pr_debug("%s:updating BE :%d  adm channels: %d\n",
			  __func__, backend_id,
			  msm_bedais[backend_id].adm_override_ch);
		mutex_unlock(&routing_lock);
	}

	return 0;
}

static const struct snd_kcontrol_new adm_channel_config_controls[] = {
	SOC_ENUM_EXT("TERT_TDM_TX_0 ADM Channels", adm_override_chs,
			msm_routing_adm_channel_config_get,
			msm_routing_adm_channel_config_put),
	SOC_ENUM_EXT("TERT_TDM_RX_0 ADM Channels", adm_override_chs,
			msm_routing_adm_channel_config_get,
			msm_routing_adm_channel_config_put),
	SOC_ENUM_EXT("SEC_TDM_TX_0 ADM Channels", adm_override_chs,
			msm_routing_adm_channel_config_get,
			msm_routing_adm_channel_config_put),
	SOC_ENUM_EXT("SEC_TDM_RX_0 ADM Channels", adm_override_chs,
			msm_routing_adm_channel_config_get,
			msm_routing_adm_channel_config_put),
};

static int msm_routing_get_port_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	int idx = 0, shift = 0;
	struct soc_mixer_control *mc =
	(struct soc_mixer_control *)kcontrol->private_value;

	idx = mc->rshift/(sizeof(msm_bedais[mc->shift].port_sessions[0]) * 8);
	shift = mc->rshift%(sizeof(msm_bedais[mc->shift].port_sessions[0]) * 8);

	if (idx >= BE_DAI_PORT_SESSIONS_IDX_MAX) {
		pr_err("%s: Invalid idx = %d\n", __func__, idx);
		return -EINVAL;
	}

	if (test_bit(shift,
		(unsigned long *)&msm_bedais[mc->shift].port_sessions[idx]))
		ucontrol->value.integer.value[0] = 1;
	else
		ucontrol->value.integer.value[0] = 0;

	pr_debug("%s: shift %x rshift %x val %ld\n", __func__, mc->shift, mc->rshift,
	ucontrol->value.integer.value[0]);

	return 0;
}

static int msm_routing_put_port_mixer(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	int idx = 0, shift = 0;
	struct soc_mixer_control *mc =
		(struct soc_mixer_control *)kcontrol->private_value;

	idx = mc->rshift/(sizeof(msm_bedais[mc->shift].port_sessions[0]) * 8);
	shift = mc->rshift%(sizeof(msm_bedais[mc->shift].port_sessions[0]) * 8);

	if (idx >= BE_DAI_PORT_SESSIONS_IDX_MAX) {
		pr_err("%s: Invalid idx = %d\n", __func__, idx);
		return -EINVAL;
	}

	pr_debug("%s: shift 0x%x rshift 0x%x val %ld idx %d reminder shift %d\n",
		 __func__, mc->shift, mc->rshift,
		 ucontrol->value.integer.value[0], idx, shift);

	if (ucontrol->value.integer.value[0]) {
		afe_loopback(1, msm_bedais[mc->shift].port_id,
			    msm_bedais[mc->rshift].port_id);
		set_bit(shift,
		(unsigned long *)&msm_bedais[mc->shift].port_sessions[idx]);
	} else {
		afe_loopback(0, msm_bedais[mc->shift].port_id,
			    msm_bedais[mc->rshift].port_id);
		clear_bit(shift,
		(unsigned long *)&msm_bedais[mc->shift].port_sessions[idx]);
	}

	return 1;
}

static const char *const ch_mixer[] = {"Disable", "Enable"};

/* If new backend is added, need update this array */
static const char *const be_name[] = {
"ZERO", "PRI_I2S_RX", "PRI_I2S_TX", "SLIM_0_RX",
"SLIM_0_TX", "HDMI_RX", "INT_BT_SCO_RX", "INT_BT_SCO_TX",
"INT_FM_RX", "INT_FM_TX", "AFE_PCM_RX", "AFE_PCM_TX",
"AUXPCM_RX", "AUXPCM_TX", "VOICE_PLAYBACK_TX", "VOICE2_PLAYBACK_TX",
"INCALL_RECORD_RX", "INCALL_RECORD_TX", "MI2S_RX", "MI2S_TX",
"SEC_I2S_RX", "SLIM_1_RX", "SLIM_1_TX", "SLIM_2_RX",
"SLIM_2_TX", "SLIM_3_RX", "SLIM_3_TX", "SLIM_4_RX",
"SLIM_4_TX", "SLIM_5_RX", "SLIM_5_TX", "SLIM_6_RX",
"SLIM_6_TX", "SLIM_7_RX", "SLIM_7_TX", "SLIM_8_RX",
"SLIM_8_TX", "EXTPROC_RX", "EXTPROC_TX", "EXPROC_EC_TX",
"QUAT_MI2S_RX", "QUAT_MI2S_TX", "SEC_MI2S_RX", "SEC_MI2S_TX",
"PRI_MI2S_RX", "PRI_MI2S_TX", "TERT_MI2S_RX", "TERT_MI2S_TX",
"AUDIO_I2S_RX", "SEC_AUXPCM_RX", "SEC_AUXPCM_TX", "PRI_SPDIF_RX",
"SEC_MI2S_RX_SD1", "QUIN_MI2S_RX", "QUIN_MI2S_TX", "SENARY_MI2S_TX",
"PRI_TDM_RX_0", "PRI_TDM_TX_0", "PRI_TDM_RX_1", "PRI_TDM_TX_1",
"PRI_TDM_RX_2", "PRI_TDM_TX_2", "PRI_TDM_RX_3", "PRI_TDM_TX_3",
"PRI_TDM_RX_4", "PRI_TDM_TX_4", "PRI_TDM_RX_5", "PRI_TDM_TX_5",
"PRI_TDM_RX_6", "PRI_TDM_TX_6", "PRI_TDM_RX_7", "PRI_TDM_TX_7",
"SEC_TDM_RX_0", "SEC_TDM_TX_0", "SEC_TDM_RX_1", "SEC_TDM_TX_1",
"SEC_TDM_RX_2", "SEC_TDM_TX_2", "SEC_TDM_RX_3", "SEC_TDM_TX_3",
"SEC_TDM_RX_4", "SEC_TDM_TX_4", "SEC_TDM_RX_5", "SEC_TDM_TX_5",
"SEC_TDM_RX_6", "SEC_TDM_TX_6", "SEC_TDM_RX_7", "SEC_TDM_TX_7",
"TERT_TDM_RX_0", "TERT_TDM_TX_0", "TERT_TDM_RX_1", "TERT_TDM_TX_1",
"TERT_TDM_RX_2", "TERT_TDM_TX_2", "TERT_TDM_RX_3", "TERT_TDM_TX_3",
"TERT_TDM_RX_4", "TERT_TDM_TX_4", "TERT_TDM_RX_5", "TERT_TDM_TX_5",
"TERT_TDM_RX_6", "TERT_TDM_TX_6", "TERT_TDM_RX_7", "TERT_TDM_TX_7",
"QUAT_TDM_RX_0", "QUAT_TDM_TX_0", "QUAT_TDM_RX_1", "QUAT_TDM_TX_1",
"QUAT_TDM_RX_2", "QUAT_TDM_TX_2", "QUAT_TDM_RX_3", "QUAT_TDM_TX_3",
"QUAT_TDM_RX_4", "QUAT_TDM_TX_4", "QUAT_TDM_RX_5", "QUAT_TDM_TX_5",
"QUAT_TDM_RX_6", "QUAT_TDM_TX_6", "QUAT_TDM_RX_7", "QUAT_TDM_TX_7",
"QUIN_TDM_RX_0", "QUIN_TDM_TX_0", "QUIN_TDM_RX_1", "QUIN_TDM_TX_1",
"QUIN_TDM_RX_2", "QUIN_TDM_TX_2", "QUIN_TDM_RX_3", "QUIN_TDM_TX_3",
"QUIN_TDM_RX_4", "QUIN_TDM_TX_4", "QUIN_TDM_RX_5", "QUIN_TDM_TX_5",
"QUIN_TDM_RX_6", "QUIN_TDM_TX_6", "QUIN_TDM_RX_7", "QUIN_TDM_TX_7",
"SEN_TDM_RX_0", "SEN_TDM_TX_0", "SEN_TDM_RX_1", "SEN_TDM_TX_1",
"SEN_TDM_RX_2", "SEN_TDM_TX_2", "SEN_TDM_RX_3", "SEN_TDM_TX_3",
"SEN_TDM_RX_4", "SEN_TDM_TX_4", "SEN_TDM_RX_5", "SEN_TDM_TX_5",
"SEN_TDM_RX_6", "SEN_TDM_TX_6", "SEN_TDM_RX_7", "SEN_TDM_TX_7",
"INT_BT_A2DP_RX", "USB_RX", "USB_TX", "DISPLAY_PORT_RX",
"DISPLAY_PORT_RX1", "TERT_AUXPCM_RX", "TERT_AUXPCM_TX", "QUAT_AUXPCM_RX",
"QUAT_AUXPCM_TX", "QUIN_AUXPCM_RX", "QUIN_AUXPCM_TX", "INT0_MI2S_RX",
"INT0_MI2S_TX", "INT1_MI2S_RX", "INT1_MI2S_TX", "INT2_MI2S_RX",
"INT2_MI2S_TX", "INT3_MI2S_RX", "INT3_MI2S_TX", "INT4_MI2S_RX",
"INT4_MI2S_TX", "INT5_MI2S_RX", "INT5_MI2S_TX", "INT6_MI2S_RX",
"INT6_MI2S_TX", "SEN_AUXPCM_RX", "SEN_AUXPCM_TX", "SENARY_MI2S_RX",
"WSA_CDC_DMA_RX_0", "WSA_CDC_DMA_TX_0", "WSA_CDC_DMA_RX_1","WSA_CDC_DMA_TX_1",
"WSA_CDC_DMA_TX_2", "VA_CDC_DMA_TX_0", "VA_CDC_DMA_TX_1", "VA_CDC_DMA_TX_2",
"RX_CDC_DMA_RX_0", "TX_CDC_DMA_TX_0", "RX_CDC_DMA_RX_1", "TX_CDC_DMA_TX_1",
"RX_CDC_DMA_RX_2", "TX_CDC_DMA_TX_2", "RX_CDC_DMA_RX_3", "TX_CDC_DMA_TX_3",
"RX_CDC_DMA_RX_4", "TX_CDC_DMA_TX_4", "RX_CDC_DMA_RX_5", "TX_CDC_DMA_TX_5",
"RX_CDC_DMA_RX_6", "RX_CDC_DMA_RX_7",
"PRI_SPDIF_TX", "SEC_SPDIF_RX", "SEC_SPDIF_TX",
"SLIM_9_RX", "SLIM_9_TX", "AFE_LOOPBACK_TX", "PRI_META_MI2S_RX",
"SEC_META_MI2S_RX", "PROXY_RX", "PROXY_TX", "HDMI_RX_MS", "AFE_PCM_RX1",
"SEP_TDM_RX_0", "SEP_TDM_TX_0", "SEP_TDM_RX_1", "SEP_TDM_TX_1",
"SEP_TDM_RX_2", "SEP_TDM_TX_2", "SEP_TDM_RX_3", "SEP_TDM_TX_3",
"SEP_TDM_RX_4", "SEP_TDM_TX_4", "SEP_TDM_RX_5", "SEP_TDM_TX_5",
"SEP_TDM_RX_6", "SEP_TDM_TX_6", "SEP_TDM_RX_7", "SEP_TDM_TX_7",
"HSIF0_TDM_RX_0", "HSIF0_TDM_TX_0", "HSIF0_TDM_RX_1", "HSIF0_TDM_TX_1",
"HSIF0_TDM_RX_2", "HSIF0_TDM_TX_2", "HSIF0_TDM_RX_3", "HSIF0_TDM_TX_3",
"HSIF0_TDM_RX_4", "HSIF0_TDM_TX_4", "HSIF0_TDM_RX_5", "HSIF0_TDM_TX_5",
"HSIF0_TDM_RX_6", "HSIF0_TDM_TX_6", "HSIF0_TDM_RX_7", "HSIF0_TDM_TX_7",
"HSIF1_TDM_RX_0", "HSIF1_TDM_TX_0", "HSIF1_TDM_RX_1", "HSIF1_TDM_TX_1",
"HSIF1_TDM_RX_2", "HSIF1_TDM_TX_2", "HSIF1_TDM_RX_3", "HSIF1_TDM_TX_3",
"HSIF1_TDM_RX_4", "HSIF1_TDM_TX_4", "HSIF1_TDM_RX_5", "HSIF1_TDM_TX_5",
"HSIF1_TDM_RX_6", "HSIF1_TDM_TX_6", "HSIF1_TDM_RX_7", "HSIF1_TDM_TX_7",
"HSIF2_TDM_RX_0", "HSIF2_TDM_TX_0", "HSIF2_TDM_RX_1", "HSIF2_TDM_TX_1",
"HSIF2_TDM_RX_2", "HSIF2_TDM_TX_2", "HSIF2_TDM_RX_3", "HSIF2_TDM_TX_3",
"HSIF2_TDM_RX_4", "HSIF2_TDM_TX_4", "HSIF2_TDM_RX_5", "HSIF2_TDM_TX_5",
"HSIF2_TDM_RX_6", "HSIF2_TDM_TX_6", "HSIF2_TDM_RX_7", "HSIF2_TDM_TX_7"
};

static SOC_ENUM_SINGLE_DECL(mm1_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, ch_mixer);
static SOC_ENUM_SINGLE_DECL(mm2_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA2, ch_mixer);
static SOC_ENUM_SINGLE_DECL(mm3_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA3, ch_mixer);
static SOC_ENUM_SINGLE_DECL(mm4_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA4, ch_mixer);
static SOC_ENUM_SINGLE_DECL(mm18_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA18, ch_mixer);
static SOC_ENUM_SINGLE_DECL(mm19_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA19, ch_mixer);
static SOC_ENUM_SINGLE_DECL(mm28_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA28, ch_mixer);
static SOC_ENUM_SINGLE_DECL(mm29_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA29, ch_mixer);
static SOC_ENUM_SINGLE_DECL(mm30_channel_mux,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA30, ch_mixer);

static SOC_ENUM_DOUBLE_DECL(mm1_ch1_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, 0, be_name);
static SOC_ENUM_DOUBLE_DECL(mm1_ch2_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, 1, be_name);
static SOC_ENUM_DOUBLE_DECL(mm1_ch3_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, 2, be_name);
static SOC_ENUM_DOUBLE_DECL(mm1_ch4_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, 3, be_name);
static SOC_ENUM_DOUBLE_DECL(mm1_ch5_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, 4, be_name);
static SOC_ENUM_DOUBLE_DECL(mm1_ch6_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, 5, be_name);
static SOC_ENUM_DOUBLE_DECL(mm1_ch7_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, 6, be_name);
static SOC_ENUM_DOUBLE_DECL(mm1_ch8_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA1, 7, be_name);
static SOC_ENUM_DOUBLE_DECL(mm18_ch1_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA18, 0, be_name);
static SOC_ENUM_DOUBLE_DECL(mm18_ch2_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA18, 1, be_name);
static SOC_ENUM_DOUBLE_DECL(mm19_ch1_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA19, 0, be_name);
static SOC_ENUM_DOUBLE_DECL(mm19_ch2_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA19, 1, be_name);
static SOC_ENUM_DOUBLE_DECL(mm28_ch1_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA28, 0, be_name);
static SOC_ENUM_DOUBLE_DECL(mm28_ch2_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA28, 1, be_name);
static SOC_ENUM_DOUBLE_DECL(mm29_ch1_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA29, 0, be_name);
static SOC_ENUM_DOUBLE_DECL(mm29_ch2_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA29, 1, be_name);
static SOC_ENUM_DOUBLE_DECL(mm30_ch1_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA30, 0, be_name);
static SOC_ENUM_DOUBLE_DECL(mm30_ch2_enum,
	SND_SOC_NOPM, MSM_FRONTEND_DAI_MULTIMEDIA30, 1, be_name);


static int msm_ec_ref_ch_get(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = msm_ec_ref_ch;
	pr_debug("%s: msm_ec_ref_ch = %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_ec_ref_ch_put(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	msm_ec_ref_ch = ucontrol->value.integer.value[0];
	pr_debug("%s: msm_ec_ref_ch = %d\n", __func__, msm_ec_ref_ch);
	adm_num_ec_ref_rx_chans(msm_ec_ref_ch);
	return 0;
}

static int msm_ec_ref_ch_downmixed_get(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = msm_ec_ref_ch_downmixed;
	pr_debug("%s: msm_ec_ref_downmixed_ch = %ld\n", __func__,
		ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_ec_ref_ch_downmixed_put(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	msm_ec_ref_ch_downmixed = ucontrol->value.integer.value[0];
	pr_debug("%s: msm_ec_ref_downmixed_ch = %d\n",
		__func__, msm_ec_ref_ch_downmixed);
	adm_num_ec_ref_rx_chans_downmixed(msm_ec_ref_ch_downmixed);
	return 0;
}

static int msm_ec_ref_chmixer_weights_get(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	int i = 0, ret = 0;
	int out_channel_idx = ((struct soc_multi_mixer_control *)
				kcontrol->private_value)->shift;

	for (; i < PCM_FORMAT_MAX_NUM_CHANNEL_V8; i++)
		ucontrol->value.integer.value[i] =
			msm_ec_ref_ch_weights[out_channel_idx][i];

	return ret;
}

static int msm_ec_ref_chmixer_weights_put(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	int i = 0, ret = 0;
	int out_channel_idx = ((struct soc_multi_mixer_control *)
				kcontrol->private_value)->shift;

	for (; i < PCM_FORMAT_MAX_NUM_CHANNEL_V8; i++)
		msm_ec_ref_ch_weights[out_channel_idx][i] =
			ucontrol->value.integer.value[i];

	ret = adm_ec_ref_chmixer_weights(out_channel_idx,
					msm_ec_ref_ch_weights[out_channel_idx],
					PCM_FORMAT_MAX_NUM_CHANNEL_V8);
	pr_debug("%s: ch_index = %d, ret = %d\n", __func__, out_channel_idx, ret);
	return ret;
}

static const char *const ec_ref_ch_text[] = {"Zero", "One", "Two", "Three",
	"Four", "Five", "Six", "Seven", "Eight", "Nine", "Ten", "Eleven",
	"Twelve", "Thirteen", "Fourteen", "Fifteen", "Sixteen"};

static int msm_ec_ref_bit_format_get(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	switch (msm_ec_ref_bit_format) {
	case SNDRV_PCM_FORMAT_S24_LE:
		ucontrol->value.integer.value[0] = 2;
		break;
	case SNDRV_PCM_FORMAT_S16_LE:
		ucontrol->value.integer.value[0] = 1;
		break;
	default:
		ucontrol->value.integer.value[0] = 0;
		break;
	}
	pr_debug("%s: msm_ec_ref_bit_format = %ld\n",
		 __func__, ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_ec_ref_bit_format_put(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	u16 bit_width = 0;

	switch (ucontrol->value.integer.value[0]) {
	case 2:
		msm_ec_ref_bit_format = SNDRV_PCM_FORMAT_S24_LE;
		break;
	case 1:
		msm_ec_ref_bit_format = SNDRV_PCM_FORMAT_S16_LE;
		break;
	default:
		msm_ec_ref_bit_format = 0;
		break;
	}

	if (msm_ec_ref_bit_format == SNDRV_PCM_FORMAT_S16_LE)
		bit_width = 16;
	else if (msm_ec_ref_bit_format == SNDRV_PCM_FORMAT_S24_LE)
		bit_width = 24;

	pr_debug("%s: msm_ec_ref_bit_format = %d\n",
		 __func__, msm_ec_ref_bit_format);
	adm_ec_ref_rx_bit_width(bit_width);
	return 0;
}

static char const *ec_ref_bit_format_text[] = {"0", "S16_LE", "S24_LE"};

static int msm_ec_ref_rate_get(struct snd_kcontrol *kcontrol,
				      struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = msm_ec_ref_sampling_rate;
	pr_debug("%s: msm_ec_ref_sampling_rate = %ld\n",
		 __func__, ucontrol->value.integer.value[0]);
	return 0;
}

static int msm_ec_ref_rate_put(struct snd_kcontrol *kcontrol,
				      struct snd_ctl_elem_value *ucontrol)
{
	switch (ucontrol->value.integer.value[0]) {
	case 0:
		msm_ec_ref_sampling_rate = 0;
		break;
	case 1:
		msm_ec_ref_sampling_rate = 8000;
		break;
	case 2:
		msm_ec_ref_sampling_rate = 16000;
		break;
	case 3:
		msm_ec_ref_sampling_rate = 24000;
		break;
	case 4:
		msm_ec_ref_sampling_rate = 32000;
		break;
	case 5:
		msm_ec_ref_sampling_rate = 44100;
		break;
	case 6:
		msm_ec_ref_sampling_rate = 48000;
		break;
	case 7:
		msm_ec_ref_sampling_rate = 96000;
		break;
	case 8:
		msm_ec_ref_sampling_rate = 192000;
		break;
	case 9:
		msm_ec_ref_sampling_rate = 384000;
		break;
	default:
		msm_ec_ref_sampling_rate = 48000;
		break;
	}
	pr_debug("%s: msm_ec_ref_sampling_rate = %d\n",
		 __func__, msm_ec_ref_sampling_rate);
	adm_ec_ref_rx_sampling_rate(msm_ec_ref_sampling_rate);
	return 0;
}

static int get_ec_ref_port_id(int value, int *index)
{
	int port_id;

	switch (value) {
	case 0:
		*index = 0;
		port_id = AFE_PORT_INVALID;
		break;
	case 1:
		*index = 1;
		port_id = SLIMBUS_0_RX;
		break;
	case 2:
		*index = 2;
		port_id = AFE_PORT_ID_PRIMARY_MI2S_RX;
		break;
	case 3:
		*index = 3;
		port_id = AFE_PORT_ID_PRIMARY_MI2S_TX;
		break;
	case 4:
		*index = 4;
		port_id = AFE_PORT_ID_SECONDARY_MI2S_TX;
		break;
	case 5:
		*index = 5;
		port_id = AFE_PORT_ID_TERTIARY_MI2S_TX;
		break;
	case 6:
		*index = 6;
		port_id = AFE_PORT_ID_QUATERNARY_MI2S_TX;
		break;
	case 7:
		*index = 7;
		port_id = AFE_PORT_ID_SECONDARY_MI2S_RX;
		break;
	case 9:
		*index = 9;
		port_id = SLIMBUS_5_RX;
		break;
	case 10:
		*index = 10;
		port_id = SLIMBUS_1_TX;
		break;
	case 11:
		*index = 11;
		port_id = AFE_PORT_ID_QUATERNARY_TDM_TX_1;
		break;
	case 12:
		*index = 12;
		port_id = AFE_PORT_ID_QUATERNARY_TDM_RX;
		break;
	case 13:
		*index = 13;
		port_id = AFE_PORT_ID_QUATERNARY_TDM_RX_1;
		break;
	case 14:
		*index = 14;
		port_id = AFE_PORT_ID_QUATERNARY_TDM_RX_2;
		break;
	case 15:
		*index = 15;
		port_id = SLIMBUS_6_RX;
		break;
	case 16:
		*index = 16;
		port_id = AFE_PORT_ID_TERTIARY_MI2S_RX;
		break;
	case 17:
		*index = 17;
		port_id = AFE_PORT_ID_QUATERNARY_MI2S_RX;
		break;
	case 18:
		*index = 18;
		port_id = AFE_PORT_ID_TERTIARY_TDM_TX;
		break;
	case 19:
		*index = 19;
		port_id = AFE_PORT_ID_USB_RX;
		break;
	case 20:
		*index = 20;
		port_id = AFE_PORT_ID_INT0_MI2S_RX;
		break;
	case 21:
		*index = 21;
		port_id = AFE_PORT_ID_INT4_MI2S_RX;
		break;
	case 22:
		*index = 22;
		port_id = AFE_PORT_ID_INT3_MI2S_TX;
		break;
	case 23:
		*index = 23;
		port_id = AFE_PORT_ID_HDMI_OVER_DP_RX;
		break;
	case 24:
		*index = 24;
		port_id = AFE_PORT_ID_WSA_CODEC_DMA_RX_0;
		break;
	case 25:
		*index = 25;
		port_id = AFE_PORT_ID_WSA_CODEC_DMA_RX_1;
		break;
	case 26:
		*index = 26;
		port_id = AFE_PORT_ID_WSA_CODEC_DMA_TX_0;
		break;
	case 27:
		*index = 27;
		port_id = AFE_PORT_ID_WSA_CODEC_DMA_TX_1;
		break;
	case 28:
		*index = 28;
		port_id = AFE_PORT_ID_WSA_CODEC_DMA_TX_2;
		break;
	case 29:
		*index = 29;
		port_id = SLIMBUS_7_RX;
		break;
	case 30:
		*index = 30;
		port_id = AFE_PORT_ID_RX_CODEC_DMA_RX_0;
		break;
	case 31:
		*index = 31;
		port_id = AFE_PORT_ID_RX_CODEC_DMA_RX_1;
		break;
	case 32:
		*index = 32;
		port_id = AFE_PORT_ID_RX_CODEC_DMA_RX_2;
		break;
	case 33:
		*index = 33;
		port_id = AFE_PORT_ID_RX_CODEC_DMA_RX_3;
		break;
	case 34:
		*index = 34;
		port_id = AFE_PORT_ID_TX_CODEC_DMA_TX_0;
		break;
	case 35:
		*index = 35;
		port_id = AFE_PORT_ID_TERTIARY_TDM_RX_2;
		break;
	case 36:
		*index = 36;
		port_id = AFE_PORT_ID_SECONDARY_TDM_TX;
		break;
	case 37:
		*index = 37;
		port_id = AFE_PORT_ID_HDMI_OVER_DP_RX;
		break;
	case 38:
		*index = 38;
		port_id = AFE_PORT_ID_SENARY_MI2S_RX;
		break;
	case 39:
		*index = 39;
		port_id = AFE_PORT_ID_QUINARY_TDM_TX;
		break;
	case 40:
		*index = 40;
		port_id = AFE_PORT_ID_SENARY_MI2S_TX;
		break;
	case 41:
		*index = 41;
		port_id = AFE_PORT_ID_PRIMARY_TDM_RX;
		break;
	case 42:
		*index = 42;
		port_id = AFE_PORT_ID_PRIMARY_TDM_TX;
		break;
	case 43:
		*index = 43;
		port_id = AFE_PORT_ID_QUINARY_MI2S_RX;
		break;
	case 44:
		*index = 44;
		port_id = RT_PROXY_PORT_001_TX;
		break;
	case 45:
		*index = 45;
		port_id = AFE_PORT_ID_PRIMARY_TDM_RX_2;
		break;
	case 46:
		*index = 46;
		port_id = AFE_PORT_ID_TERTIARY_TDM_RX;
		break;
	case 47:
		*index = 47;
		port_id = AFE_PORT_ID_SECONDARY_TDM_TX_1;
		break;
	default:
		*index = 0; /* NONE */
		pr_err("%s: Invalid value %d\n", __func__, value);
		port_id = AFE_PORT_INVALID;
		break;
	}

	return port_id;
}

static int msm_routing_afe_lb_tx_port_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	pr_debug("%s: port index = %d", __func__, afe_loopback_tx_port_index);
	mutex_lock(&routing_lock);
	ucontrol->value.integer.value[0] = afe_loopback_tx_port_index;
	mutex_unlock(&routing_lock);

	return 0;
}

static int msm_routing_afe_lb_tx_port_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	int value = ucontrol->value.integer.value[0];

	mutex_lock(&routing_lock);
	afe_loopback_tx_port_id = get_ec_ref_port_id(value,
			&afe_loopback_tx_port_index);
	pr_debug("%s: afe_loopback_tx_port_index = %d\n",
	    __func__, afe_loopback_tx_port_index);
	mutex_unlock(&routing_lock);

	return 0;
}

#define ASM_CFG_PARAM_MAX 4
static int asm_cfg_params[ASM_CFG_PARAM_MAX];

struct msm_asm_config loopback_cfg[] = {
	[MSM_FRONTEND_DAI_MULTIMEDIA1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA3] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA4] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA5] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA6] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA7] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA8] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA9] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA10] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA11] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA12] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA13] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA14] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA15] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA16] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA17] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA18] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA19] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA20] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA21] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA22] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA23] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA24] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA25] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA26] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA27] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA28] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA29] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA30] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA31] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA32] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA33] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA34] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOIP] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_AFE_RX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_AFE_TX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICE_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_DTMF_RX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_QCHAT] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOLTE_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM3] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM4] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM5] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM6] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM7] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM8] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICE2_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICEMMODE1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICEMMODE2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
};

struct msm_asm_config playback_cfg[] = {
	[MSM_FRONTEND_DAI_MULTIMEDIA1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA3] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA4] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA5] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA6] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA7] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA8] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA9] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA10] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA11] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA12] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA13] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA14] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA15] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA16] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA17] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA18] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA19] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA20] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA21] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA22] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA23] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA24] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA25] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA26] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA27] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA28] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA29] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA30] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA31] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA32] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA33] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA34] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOIP] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_AFE_RX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_AFE_TX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICE_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_DTMF_RX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_QCHAT] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOLTE_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM3] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM4] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM5] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM6] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM7] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM8] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICE2_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICEMMODE1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICEMMODE2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
};

struct msm_asm_config capture_cfg[] = {
	[MSM_FRONTEND_DAI_MULTIMEDIA1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA3] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA4] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA5] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA6] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA7] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA8] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA9] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA10] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA11] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA12] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA13] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA14] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA15] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA16] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA17] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA18] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA19] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA20] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA21] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA22] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA23] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA24] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA25] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA26] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA27] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA28] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA29] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA30] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA31] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA32] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA33] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_MULTIMEDIA34] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOIP] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_AFE_RX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_AFE_TX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICE_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_DTMF_RX] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_QCHAT] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOLTE_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM3] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM4] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM5] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM6] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM7] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_LSM8] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICE2_STUB] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICEMMODE1] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
	[MSM_FRONTEND_DAI_VOICEMMODE2] = {0, 0, SNDRV_PCM_FORMAT_S16_LE},
};

int msm_pcm_asm_cfg_get(int fe_id, int mode)
{
	int format = 0;
	if (mode == MSM_ASM_PLAYBACK_MODE) {
		format = playback_cfg[fe_id].bit_format;
	}
	else if(mode == MSM_ASM_CAPTURE_MODE) {
		format = capture_cfg[fe_id].bit_format;
	}
	else if (mode == MSM_ASM_LOOPBACK_MODE) {
		format = loopback_cfg[fe_id].bit_format;
	}
	return format;
}
EXPORT_SYMBOL(msm_pcm_asm_cfg_get);

static int msm_pcm_get_format(int value)
{
	int format = 0;
	switch (value) {
	case 16:
		format = SNDRV_PCM_FORMAT_S16_LE;
		break;
	case 24:
		format = SNDRV_PCM_FORMAT_S24_LE;
		break;
	case 32:
		format = SNDRV_PCM_FORMAT_S32_LE;
		break;
	default:
		break;
	}
	return format;
}

static int msm_pcm_asm_format_put(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	int ret = 0;
	int fe_id = 0, mode = 0, format = 0, set = 0;

	mutex_lock(&routing_lock);
	asm_cfg_params[0] = ucontrol->value.integer.value[0];
	asm_cfg_params[1] = ucontrol->value.integer.value[1];
	asm_cfg_params[2] = ucontrol->value.integer.value[2];
	set =  ucontrol->value.integer.value[3];
	mutex_unlock(&routing_lock);

	if(set) {
		fe_id  = asm_cfg_params[0];
		mode   = asm_cfg_params[1];
		format = msm_pcm_get_format(asm_cfg_params[2]);

		pr_debug("%s: fe_id:%d, mode:%d, format:%d, set:%d\n",
				__func__,fe_id,mode,format,set);

		if((fe_id < MSM_FRONTEND_DAI_MULTIMEDIA1) ||
		   (fe_id >= MSM_FRONTEND_DAI_MAX)) {
			pr_err("%s: Received invalid fe_id %lu\n", __func__, fe_id);
			return -EINVAL;
		}
		if((mode < MSM_ASM_PLAYBACK_MODE) ||
		   (mode >= MSM_ASM_MAX_MODE)) {
			pr_err("%s: Received invalid mode %lu\n", __func__, mode);
			return -EINVAL;
		}
		if((format == SNDRV_PCM_FORMAT_S16_LE) ||
			(format == SNDRV_PCM_FORMAT_S24_LE) ||
			(format == SNDRV_PCM_FORMAT_S32_LE)) {
				if(mode == MSM_ASM_PLAYBACK_MODE) {
					playback_cfg[fe_id].fe_id = fe_id;
					playback_cfg[fe_id].mode = mode;
					playback_cfg[fe_id].bit_format = format;
				} else if(mode == MSM_ASM_CAPTURE_MODE) {
					capture_cfg[fe_id].fe_id = fe_id;
					capture_cfg[fe_id].mode = mode;
					capture_cfg[fe_id].bit_format = format;
				} else if(mode == MSM_ASM_LOOPBACK_MODE) {
					loopback_cfg[fe_id].fe_id = fe_id;
					loopback_cfg[fe_id].mode = mode;
					loopback_cfg[fe_id].bit_format = format;
			}
		}else {
			pr_err("%s: Received invalid foramt %lu\n", __func__,
					asm_cfg_params[2]);
			return -EINVAL;
		}
	}
	return ret;
}

static int msm_pcm_asm_format_get(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	mutex_lock(&routing_lock);
	ucontrol->value.integer.value[0] = asm_cfg_params[0];
	ucontrol->value.integer.value[1] = asm_cfg_params[1];
	ucontrol->value.integer.value[2] = asm_cfg_params[2];
	ucontrol->value.integer.value[3] = 0;
	mutex_unlock(&routing_lock);

	pr_debug("%s: fe_id:%d, mode:%d, format:%d, set:%d\n",__func__,
			ucontrol->value.integer.value[0],
			ucontrol->value.integer.value[1],
			ucontrol->value.integer.value[2],
			ucontrol->value.integer.value[3]);
	return 0;
}

static const struct snd_kcontrol_new msm_asm_format_controls[] = {
		SOC_SINGLE_MULTI_EXT("ASM Format", SND_SOC_NOPM,
		0, 0xFFFF, 0, 4,
		msm_pcm_asm_format_get,
		msm_pcm_asm_format_put),
};

static const char *const ec_ref_rate_text[] = {"0", "8000", "16000",
	"24000", "32000", "44100", "48000", "96000", "192000", "384000"};

static const struct soc_enum msm_route_ec_ref_params_enum[] = {
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(ec_ref_ch_text), ec_ref_ch_text),
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(ec_ref_bit_format_text),
				ec_ref_bit_format_text),
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(ec_ref_rate_text), ec_ref_rate_text),
};

static const char *const ec_ref_rx[] = { "None", "SLIM_RX", "I2S_RX",
	"PRI_MI2S_TX", "SEC_MI2S_TX",
	"TERT_MI2S_TX", "QUAT_MI2S_TX", "SEC_I2S_RX", "PROXY_RX",
	"SLIM_5_RX", "SLIM_1_TX", "QUAT_TDM_TX_1",
	"QUAT_TDM_RX_0", "QUAT_TDM_RX_1", "QUAT_TDM_RX_2", "SLIM_6_RX",
	"TERT_MI2S_RX", "QUAT_MI2S_RX", "TERT_TDM_TX_0", "USB_AUDIO_RX",
	"INT0_MI2S_RX", "INT4_MI2S_RX", "INT3_MI2S_TX", "DISPLAY_PORT",
	"WSA_CDC_DMA_RX_0", "WSA_CDC_DMA_RX_1",
	"WSA_CDC_DMA_TX_0", "WSA_CDC_DMA_TX_1", "WSA_CDC_DMA_TX_2",
	"SLIM_7_RX", "RX_CDC_DMA_RX_0", "RX_CDC_DMA_RX_1", "RX_CDC_DMA_RX_2",
	"RX_CDC_DMA_RX_3", "TX_CDC_DMA_TX_0", "TERT_TDM_RX_2", "SEC_TDM_TX_0",
	"DISPLAY_PORT1", "SEN_MI2S_RX", "QUIN_TDM_TX_0", "SENARY_MI2S_TX",
	"PRI_TDM_RX_0", "PRI_TDM_TX_0", "QUIN_MI2S_RX", "AFE_PCM_TX",
	"PRI_TDM_RX_2", "TERT_TDM_RX_0", "SEC_TDM_TX_1",
};

static const struct soc_enum msm_route_ec_ref_rx_enum[] = {
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(ec_ref_rx), ec_ref_rx),
};

static const struct snd_kcontrol_new ec_ref_param_controls[] = {
	SOC_ENUM_EXT("EC Reference Channels", msm_route_ec_ref_params_enum[0],
		msm_ec_ref_ch_get, msm_ec_ref_ch_put),
	SOC_ENUM_EXT("EC Reference Bit Format", msm_route_ec_ref_params_enum[1],
		msm_ec_ref_bit_format_get, msm_ec_ref_bit_format_put),
	SOC_ENUM_EXT("EC Reference SampleRate", msm_route_ec_ref_params_enum[2],
		msm_ec_ref_rate_get, msm_ec_ref_rate_put),
	SOC_ENUM_EXT("EC Reference Downmixed Channels",
		msm_route_ec_ref_params_enum[0],
		msm_ec_ref_ch_downmixed_get, msm_ec_ref_ch_downmixed_put),
	SOC_SINGLE_MULTI_EXT("EC Reference ChMixer Weights Ch1", SND_SOC_NOPM,
		0, 16384, 0, PCM_FORMAT_MAX_NUM_CHANNEL_V8,
		msm_ec_ref_chmixer_weights_get,
		msm_ec_ref_chmixer_weights_put),
	SOC_SINGLE_MULTI_EXT("EC Reference ChMixer Weights Ch2", SND_SOC_NOPM,
		1, 16384, 0, PCM_FORMAT_MAX_NUM_CHANNEL_V8,
		msm_ec_ref_chmixer_weights_get,
		msm_ec_ref_chmixer_weights_put),
	SOC_SINGLE_MULTI_EXT("EC Reference ChMixer Weights Ch3", SND_SOC_NOPM,
		2, 16384, 0, PCM_FORMAT_MAX_NUM_CHANNEL_V8,
		msm_ec_ref_chmixer_weights_get,
		msm_ec_ref_chmixer_weights_put),
	SOC_SINGLE_MULTI_EXT("EC Reference ChMixer Weights Ch4", SND_SOC_NOPM,
		3, 16384, 0, PCM_FORMAT_MAX_NUM_CHANNEL_V8,
		msm_ec_ref_chmixer_weights_get,
		msm_ec_ref_chmixer_weights_put),
	SOC_SINGLE_MULTI_EXT("EC Reference ChMixer Weights Ch5", SND_SOC_NOPM,
		4, 16384, 0, PCM_FORMAT_MAX_NUM_CHANNEL_V8,
		msm_ec_ref_chmixer_weights_get,
		msm_ec_ref_chmixer_weights_put),
	SOC_SINGLE_MULTI_EXT("EC Reference ChMixer Weights Ch6", SND_SOC_NOPM,
		5, 16384, 0, PCM_FORMAT_MAX_NUM_CHANNEL_V8,
		msm_ec_ref_chmixer_weights_get,
		msm_ec_ref_chmixer_weights_put),
	SOC_SINGLE_MULTI_EXT("EC Reference ChMixer Weights Ch7", SND_SOC_NOPM,
		6, 16384, 0, PCM_FORMAT_MAX_NUM_CHANNEL_V8,
		msm_ec_ref_chmixer_weights_get,
		msm_ec_ref_chmixer_weights_put),
	SOC_SINGLE_MULTI_EXT("EC Reference ChMixer Weights Ch8", SND_SOC_NOPM,
		7, 16384, 0, PCM_FORMAT_MAX_NUM_CHANNEL_V8,
		msm_ec_ref_chmixer_weights_get,
		msm_ec_ref_chmixer_weights_put),
	SOC_ENUM_EXT("AFE_LOOPBACK_TX Port", msm_route_ec_ref_rx_enum[0],
		msm_routing_afe_lb_tx_port_get, msm_routing_afe_lb_tx_port_put),
};

static int msm_routing_ec_ref_rx_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	pr_debug("%s: ec_ref_rx  = %d", __func__, msm_route_ec_ref_rx);
	mutex_lock(&routing_lock);
	ucontrol->value.integer.value[0] = msm_route_ec_ref_rx;
	mutex_unlock(&routing_lock);

	return 0;
}

static int msm_routing_ec_ref_rx_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	int value = ucontrol->value.integer.value[0];
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	struct soc_enum *e = (struct soc_enum *)kcontrol->private_value;
	struct snd_soc_dapm_update *update = NULL;

	mutex_lock(&routing_lock);
	msm_ec_ref_port_id = get_ec_ref_port_id(value, &msm_route_ec_ref_rx);
	adm_ec_ref_rx_id(msm_ec_ref_port_id);
	pr_debug("%s: msm_route_ec_ref_rx = %d\n",
	    __func__, msm_route_ec_ref_rx);
	mutex_unlock(&routing_lock);

	snd_soc_dapm_mux_update_power(widget->dapm, kcontrol,
					msm_route_ec_ref_rx, e, update);
	return 0;
}

static const struct snd_kcontrol_new ext_ec_ref_mux_ul1 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL1 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul2 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL2 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul3 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL3 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul4 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL4 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul5 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL5 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul6 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL6 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul8 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL8 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul9 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL9 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul16 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL16 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul10 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL10 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul17 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL17 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul18 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL18 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul19 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL19 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul28 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL28 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul29 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL29 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);

static const struct snd_kcontrol_new ext_ec_ref_mux_ul30 =
	SOC_DAPM_ENUM_EXT("AUDIO_REF_EC_UL30 MUX Mux",
		msm_route_ec_ref_rx_enum[0],
		msm_routing_ec_ref_rx_get, msm_routing_ec_ref_rx_put);


static int msm_routing_ext_ec_get(struct snd_kcontrol *kcontrol,
				  struct snd_ctl_elem_value *ucontrol)
{
	pr_debug("%s: ext_ec_ref_rx  = %x\n", __func__, msm_route_ext_ec_ref);

	mutex_lock(&routing_lock);
	ucontrol->value.integer.value[0] = msm_route_ext_ec_ref;
	mutex_unlock(&routing_lock);
	return 0;
}

static int msm_routing_ext_ec_put(struct snd_kcontrol *kcontrol,
				  struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dapm_widget *widget =
		snd_soc_dapm_kcontrol_widget(kcontrol);
	int mux = ucontrol->value.enumerated.item[0];
	struct soc_enum *e = (struct soc_enum *)kcontrol->private_value;
	int ret = 1;
	bool state = true;
	uint16_t ext_ec_ref_port_id;
	struct snd_soc_dapm_update *update = NULL;

	if (mux >= e->items) {
		pr_err("%s: Invalid mux value %d\n", __func__, mux);
		return -EINVAL;
	}

	mutex_lock(&routing_lock);
	msm_route_ext_ec_ref = ucontrol->value.integer.value[0];

	switch (msm_route_ext_ec_ref) {
	case EXT_EC_REF_PRI_MI2S_TX:
		ext_ec_ref_port_id = AFE_PORT_ID_PRIMARY_MI2S_TX;
		break;
	case EXT_EC_REF_SEC_MI2S_TX:
		ext_ec_ref_port_id = AFE_PORT_ID_SECONDARY_MI2S_TX;
		break;
	case EXT_EC_REF_TERT_MI2S_TX:
		ext_ec_ref_port_id = AFE_PORT_ID_TERTIARY_MI2S_TX;
		break;
	case EXT_EC_REF_QUAT_MI2S_TX:
		ext_ec_ref_port_id = AFE_PORT_ID_QUATERNARY_MI2S_TX;
		break;
	case EXT_EC_REF_QUIN_MI2S_TX:
		ext_ec_ref_port_id = AFE_PORT_ID_QUINARY_MI2S_TX;
		break;
	case EXT_EC_REF_SLIM_1_TX:
		ext_ec_ref_port_id = SLIMBUS_1_TX;
		break;
	case EXT_EC_REF_PRI_TDM_TX:
		ext_ec_ref_port_id = AFE_PORT_ID_PRIMARY_TDM_TX;
		break;
	case EXT_EC_REF_SEC_TDM_TX:
		ext_ec_ref_port_id = AFE_PORT_ID_SECONDARY_TDM_TX;
		break;
	case EXT_EC_REF_SENARY_MI2S_TX:
		ext_ec_ref_port_id = AFE_PORT_ID_SENARY_MI2S_TX;
		break;
	case EXT_EC_REF_NONE:
	default:
		ext_ec_ref_port_id = AFE_PORT_INVALID;
		state = false;
		break;
	}

	pr_debug("%s: val = %d ext_ec_ref_port_id = 0x%0x state = %d\n",
		 __func__, msm_route_ext_ec_ref, ext_ec_ref_port_id, state);

	if (!voc_set_ext_ec_ref_port_id(ext_ec_ref_port_id, state)) {
		mutex_unlock(&routing_lock);
		snd_soc_dapm_mux_update_power(widget->dapm, kcontrol, mux, e,
						update);
	} else {
		ret = -EINVAL;
		mutex_unlock(&routing_lock);
	}
	return ret;
}

static const char * const ext_ec_ref_rx[] = {"NONE", "PRI_MI2S_TX",
					"SEC_MI2S_TX", "TERT_MI2S_TX",
					"QUAT_MI2S_TX", "QUIN_MI2S_TX",
					"SLIM_1_TX", "PRI_TDM_TX",
					"SEC_TDM_TX", "SENARY_MI2S_TX"};

static const struct soc_enum msm_route_ext_ec_ref_rx_enum[] = {
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(ext_ec_ref_rx), ext_ec_ref_rx),
};

static const struct snd_kcontrol_new voc_ext_ec_mux =
	SOC_DAPM_ENUM_EXT("VOC_EXT_EC MUX Mux", msm_route_ext_ec_ref_rx_enum[0],
			  msm_routing_ext_ec_get, msm_routing_ext_ec_put);

#ifndef CONFIG_MI2S_DISABLE
static const struct snd_kcontrol_new pri_i2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sec_i2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quaternary_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quinary_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),

};

static const struct snd_kcontrol_new senary_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),

};

static const struct snd_kcontrol_new tertiary_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new secondary_mi2s_rx2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX_SD1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new secondary_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new primary_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("DTMF", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_FRONTEND_DAI_DTMF_RX, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new int0_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new int4_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new pri_meta_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),

};

static const struct snd_kcontrol_new sec_meta_mi2s_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_META_MI2S_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new int0_mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_BACKEND_DAI_SENARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_BACKEND_DAI_INT3_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT0_MI2S_RX,
	MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new int4_mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_BACKEND_DAI_SENARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_BACKEND_DAI_INT3_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT4_MI2S_RX,
	MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_RX,
	MSM_BACKEND_DAI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new primary_mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_BACKEND_DAI_SENARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_RX,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quat_mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_BACKEND_DAI_AUXPCM_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_RX,
	MSM_BACKEND_DAI_SENARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quin_mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_BACKEND_DAI_SENARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sen_mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_BACKEND_DAI_SENARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new tert_mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_BACKEND_DAI_SENARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sec_mi2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_BACKEND_DAI_SENARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_RX,
	MSM_BACKEND_DAI_AUXPCM_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};


static const struct snd_kcontrol_new int0_mi2s_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_int0_mi2s_switch_mixer,
	msm_routing_put_int0_mi2s_switch_mixer);

static const struct snd_kcontrol_new int4_mi2s_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_int4_mi2s_switch_mixer,
	msm_routing_put_int4_mi2s_switch_mixer);

static const struct snd_kcontrol_new pri_mi2s_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_pri_mi2s_switch_mixer,
	msm_routing_put_pri_mi2s_switch_mixer);

static const struct snd_kcontrol_new sec_mi2s_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_sec_mi2s_switch_mixer,
	msm_routing_put_sec_mi2s_switch_mixer);

static const struct snd_kcontrol_new tert_mi2s_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_tert_mi2s_switch_mixer,
	msm_routing_put_tert_mi2s_switch_mixer);

static const struct snd_kcontrol_new quat_mi2s_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_quat_mi2s_switch_mixer,
	msm_routing_put_quat_mi2s_switch_mixer);

static const struct snd_kcontrol_new quin_mi2s_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_quin_mi2s_switch_mixer,
	msm_routing_put_quin_mi2s_switch_mixer);

static const struct snd_kcontrol_new sen_mi2s_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_sen_mi2s_switch_mixer,
	msm_routing_put_sen_mi2s_switch_mixer);
#endif


static const struct snd_kcontrol_new int_bt_sco_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new int_bt_a2dp_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_A2DP_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new int_fm_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new afe_pcm_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new afe_pcm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

#ifndef CONFIG_AUXPCM_DISABLE
static const struct snd_kcontrol_new auxpcm_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sec_auxpcm_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia17", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia30", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new tert_auxpcm_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quat_auxpcm_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quin_auxpcm_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sen_auxpcm_rx_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia26", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA26, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};
#endif

#ifndef CONFIG_TDM_DISABLE
static const struct snd_kcontrol_new pri_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new pri_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new pri_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new pri_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_4_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quin_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quin_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),

};

static const struct snd_kcontrol_new quin_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new quin_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia22", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia23", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia24", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia25", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia31", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia32", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia33", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia34", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};
//Gen4 specific mixer controls start
static const struct snd_kcontrol_new sen_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sen_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sen_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sen_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia18", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia19", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia28", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia29", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_4_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_5_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_6_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_7_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_4_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_5_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_6_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_7_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_4_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_5_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_6_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_7_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_0_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_1_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_2_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_3_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_4_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_5_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_6_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_7_mixer_controls[] = {
	SOC_DOUBLE_EXT("MultiMedia1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA7, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia8", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia9", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia10", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia11", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA11, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia12", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA12, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia13", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA13, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia14", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA14, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia15", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA15, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia16", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia20", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MultiMedia21", SND_SOC_NOPM,
	MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};
//Gen4 specific mixer controls ends

static const struct snd_kcontrol_new pri_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new pri_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new pri_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new pri_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sec_tdm_rx_7_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("TERT_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_7, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new tert_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quat_tdm_rx_7_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("TERT_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_7, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_7, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_7, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quin_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quin_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quin_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new quin_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};
//Gen4 specific mixer controls start
static const struct snd_kcontrol_new sen_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_0,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sen_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_1,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sen_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_2,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sen_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_RX_3,
		MSM_BACKEND_DAI_SEN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_SEP_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_SEP_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_SEP_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_0,
		MSM_BACKEND_DAI_SEP_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_SEP_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_SEP_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_SEP_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_1,
		MSM_BACKEND_DAI_SEP_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_SEP_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_SEP_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_SEP_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_2,
		MSM_BACKEND_DAI_SEP_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_SEP_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_SEP_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_SEP_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_3,
		MSM_BACKEND_DAI_SEP_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_4_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_SEP_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_SEP_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_SEP_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_4,
		MSM_BACKEND_DAI_SEP_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_5_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_SEP_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_SEP_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_SEP_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_5,
		MSM_BACKEND_DAI_SEP_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_6_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_SEP_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_SEP_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_SEP_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_6,
		MSM_BACKEND_DAI_SEP_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new sep_tdm_rx_7_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_INT_BT_SCO_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_SEP_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_SEP_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_SEP_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_RX_7,
		MSM_BACKEND_DAI_SEP_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_4_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_5_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_6_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif0_tdm_rx_7_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_4_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_5_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_6_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif1_tdm_rx_7_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_0_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_0,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_1_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_1,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_2_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_2,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_3_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_3,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_4_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_4,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_5_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_5,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_6_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_6,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new hsif2_tdm_rx_7_port_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_PRI_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_INT_FM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_AFE_PCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
#endif
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_RX_7,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};
//Gen4 specific mixer controls ends

static const struct snd_kcontrol_new quin_tdm_rx_7_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("TERT_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_7,
		MSM_BACKEND_DAI_TERT_TDM_TX_7, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_7,
		MSM_BACKEND_DAI_QUAT_TDM_TX_7, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_RX_7,
		MSM_BACKEND_DAI_QUIN_TDM_TX_7, 1, 0,
		msm_routing_get_port_mixer,
		msm_routing_put_port_mixer),
};
#endif

static const struct snd_kcontrol_new mmul1_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_I2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT2_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SLIMBUS_0_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SLIMBUS_1_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AUXPCM_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_AUXPCM_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_AUXPCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_AUXPCM_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_AUXPCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_AUXPCM_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_AUXPCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_AUXPCM_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_AUXPCM_UL_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_AUXPCM_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT_BT_SCO_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT_FM_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INCALL_RECORD_RX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INCALL_RECORD_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_4_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SLIMBUS_4_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_6_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SLIMBUS_6_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUINARY_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SENARY_MI2S_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUAT_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
#ifdef RX_TO_TX_LOOPBACK
	SOC_DOUBLE_EXT("QUIN_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEN_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("SLIM_7_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SLIMBUS_7_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_8_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SLIMBUS_8_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_USB_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_PRI_SPDIF_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SEC_SPDIF_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_SLIMBUS_9_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
		MSM_FRONTEND_DAI_MULTIMEDIA1, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul2_mixer_controls[] = {
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT2_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_6_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_6_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("SLIM_8_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_8_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("SEC_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA2, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};

static const struct snd_kcontrol_new mmul3_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT2_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA3, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul4_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT2_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA4, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul5_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT2_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("SLIM_7_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_7_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_8_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_8_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA5, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul6_mixer_controls[] = {
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT2_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_AUXPCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_AUXPCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA6, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul8_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT2_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_6_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_6_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("SLIM_7_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_7_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA8, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul16_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT2_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_6_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_6_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("SLIM_7_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_7_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA16, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul9_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_6_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_6_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA9, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul10_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT2_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT2_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_6_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_6_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_7_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_7_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_AUX_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA10, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};
static const struct snd_kcontrol_new mmul17_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_7_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_7_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA17, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul18_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA18, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul19_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA19, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul20_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA20, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul21_mixer_controls[] = {
#ifndef CONFIG_AUXPCM_DISABLE
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_AUXPCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_AUXPCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_6", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_6,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEP_TDM_TX_7", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEP_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF0_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF0_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF1_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF1_TDM_TX_7,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_6", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_6,
		MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("HSIF2_TDM_TX_7", SND_SOC_NOPM,
		MSM_BACKEND_DAI_HSIF2_TDM_TX_7,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA21, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul22_mixer_controls[] = {
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA22, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};

static const struct snd_kcontrol_new mmul23_mixer_controls[] = {
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA23, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};

static const struct snd_kcontrol_new mmul24_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA24, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};

static const struct snd_kcontrol_new mmul25_mixer_controls[] = {
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA25, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};

static const struct snd_kcontrol_new mmul27_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_6_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_6_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_7_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_7_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUINARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SENARY_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SENARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA27, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul28_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA28, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul29_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_FM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_9_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_9_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA29, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul30_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_0_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SLIMBUS_1_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("PRI_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT3_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
	SOC_DOUBLE_EXT("INTERNAL_BT_SCO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_DL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_RX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VOC_REC_UL", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INCALL_RECORD_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("WSA_CDC_DMA_TX_2", SND_SOC_NOPM,
			MSM_BACKEND_DAI_WSA_CDC_DMA_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_0", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("VA_CDC_DMA_TX_1", SND_SOC_NOPM,
			MSM_BACKEND_DAI_VA_CDC_DMA_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_0,
		MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_1", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_1,
		MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_2", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_2,
		MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_3", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_3,
		MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_4", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_4,
		MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TX_CDC_DMA_TX_5", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TX_CDC_DMA_TX_5,
		MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0,
		msm_routing_get_audio_mixer,
		msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_SPDIF_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_SPDIF_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("AFE_LOOPBACK_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_LOOPBACK_TX,
	MSM_FRONTEND_DAI_MULTIMEDIA30, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
};

static const struct snd_kcontrol_new mmul31_mixer_controls[] = {
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA31, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};

static const struct snd_kcontrol_new mmul32_mixer_controls[] = {
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA32, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};

static const struct snd_kcontrol_new mmul33_mixer_controls[] = {
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA33, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};

static const struct snd_kcontrol_new mmul34_mixer_controls[] = {
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("PRI_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("PRI_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_PRI_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("SEC_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_TERT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUAT_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_0,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_1", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_1,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_2", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_2,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
	SOC_DOUBLE_EXT("QUIN_TDM_TX_3", SND_SOC_NOPM,
	MSM_BACKEND_DAI_QUIN_TDM_TX_3,
	MSM_FRONTEND_DAI_MULTIMEDIA34, 1, 0, msm_routing_get_audio_mixer,
	msm_routing_put_audio_mixer),
#endif
};


#ifndef CONFIG_AUXPCM_DISABLE
static const struct snd_kcontrol_new aux_pcm_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_BACKEND_DAI_AUXPCM_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_BACKEND_DAI_SLIMBUS_0_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_BACKEND_DAI_SLIMBUS_1_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_AUX_PCM_UL_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_BACKEND_DAI_SEC_AUXPCM_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("QUAT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_BACKEND_DAI_QUAT_TDM_TX_0, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SEC_MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_BACKEND_DAI_SECONDARY_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AUXPCM_RX,
	MSM_BACKEND_DAI_TERT_TDM_TX_0, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};
#endif

static const struct snd_kcontrol_new bt_sco_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("SLIM_1_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_BACKEND_DAI_SLIMBUS_1_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
	SOC_DOUBLE_EXT("SLIM_0_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_INT_BT_SCO_RX,
	MSM_BACKEND_DAI_SLIMBUS_0_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new afe_pcm_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("INTERNAL_FM_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_AFE_PCM_RX,
	MSM_BACKEND_DAI_INT_FM_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};



static const struct snd_kcontrol_new sec_i2s_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("MI2S_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_SEC_I2S_RX,
	MSM_BACKEND_DAI_MI2S_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new usb_rx_port_mixer_controls[] = {
	SOC_DOUBLE_EXT("USB_AUDIO_TX", SND_SOC_NOPM,
	MSM_BACKEND_DAI_USB_RX,
	MSM_BACKEND_DAI_USB_TX, 1, 0, msm_routing_get_port_mixer,
	msm_routing_put_port_mixer),
};

static const struct snd_kcontrol_new lsm1_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM1, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM1, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_LSM1, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM1, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM1, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
	SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_LSM1, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
};

static const struct snd_kcontrol_new lsm2_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM2, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM2, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_LSM2, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM2, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM2, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
        SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_LSM2, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
};

static const struct snd_kcontrol_new lsm3_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM3, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM3, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_LSM3, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM3, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM3, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
        SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_LSM3, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
};

static const struct snd_kcontrol_new lsm4_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM4, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM4, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_LSM4, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM4, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM4, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
        SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_LSM4, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
};

static const struct snd_kcontrol_new lsm5_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM5, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM5, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_LSM5, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM5, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM5, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
        SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_LSM5, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
};

static const struct snd_kcontrol_new lsm6_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM6, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM6, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_LSM6, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM6, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM6, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
        SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_LSM6, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
};

static const struct snd_kcontrol_new lsm7_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM7, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM7, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_LSM7, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM7, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM7, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
        SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_LSM7, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
};

static const struct snd_kcontrol_new lsm8_mixer_controls[] = {
#ifndef CONFIG_MI2S_DISABLE
	SOC_DOUBLE_EXT("TERT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERTIARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM8, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("QUAT_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUATERNARY_MI2S_TX,
		MSM_FRONTEND_DAI_LSM8, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("INT3_MI2S_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_INT3_MI2S_TX,
		MSM_FRONTEND_DAI_LSM8, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
#ifndef CONFIG_TDM_DISABLE
	SOC_DOUBLE_EXT("QUIN_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_QUIN_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM8, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
	SOC_DOUBLE_EXT("TERT_TDM_TX_0", SND_SOC_NOPM,
		MSM_BACKEND_DAI_TERT_TDM_TX_0,
		MSM_FRONTEND_DAI_LSM8, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
#endif
        SOC_DOUBLE_EXT("AFE_PCM_TX", SND_SOC_NOPM,
		MSM_BACKEND_DAI_AFE_PCM_TX,
		MSM_FRONTEND_DAI_LSM8, 1, 0, msm_routing_get_listen_mixer,
		msm_routing_put_listen_mixer),
};


static const struct snd_kcontrol_new pcm_rx_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_fm_pcmrx_switch_mixer,
	msm_routing_put_fm_pcmrx_switch_mixer);

static const struct snd_kcontrol_new hfp_pri_aux_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_hfp_switch_mixer,
	msm_routing_put_hfp_switch_mixer);

static const struct snd_kcontrol_new hfp_aux_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_hfp_switch_mixer,
	msm_routing_put_hfp_switch_mixer);

static const struct snd_kcontrol_new hfp_int_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_hfp_switch_mixer,
	msm_routing_put_hfp_switch_mixer);

static const struct snd_kcontrol_new hfp_slim7_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_hfp_switch_mixer,
	msm_routing_put_hfp_switch_mixer);

static const struct snd_kcontrol_new usb_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_get_usb_switch_mixer,
	msm_routing_put_usb_switch_mixer);

static const struct snd_kcontrol_new a2dp_slim7_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_a2dp_switch_mixer_get,
	msm_routing_a2dp_switch_mixer_put);

static const struct snd_kcontrol_new sco_slim7_switch_mixer_controls =
	SOC_SINGLE_EXT("Switch", SND_SOC_NOPM,
	0, 1, 0, msm_routing_sco_switch_mixer_get,
	msm_routing_sco_switch_mixer_put);

static const struct soc_enum lsm_port_enum =
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(lsm_port_text), lsm_port_text);

static const char * const lsm_func_text[] = {
	"None", "AUDIO", "BEACON", "ULTRASOUND", "SWAUDIO",
};
static const struct soc_enum lsm_func_enum =
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(lsm_func_text), lsm_func_text);

static const struct snd_kcontrol_new lsm_controls[] = {
	/* kcontrol of lsm_function */
	SOC_ENUM_EXT(TERT_MI2S_TX_TEXT" "LSM_FUNCTION_TEXT, lsm_func_enum,
		    msm_routing_lsm_func_get, msm_routing_lsm_func_put),
	SOC_ENUM_EXT(QUAT_MI2S_TX_TEXT" "LSM_FUNCTION_TEXT, lsm_func_enum,
		    msm_routing_lsm_func_get, msm_routing_lsm_func_put),
	SOC_ENUM_EXT(INT3_MI2S_TX_TEXT" "LSM_FUNCTION_TEXT, lsm_func_enum,
		    msm_routing_lsm_func_get, msm_routing_lsm_func_put),
	SOC_ENUM_EXT(QUIN_TDM_TX_TEXT" "LSM_FUNCTION_TEXT, lsm_func_enum,
		    msm_routing_lsm_func_get, msm_routing_lsm_func_put),
	SOC_ENUM_EXT(TERT_TDM_TX_TEXT" "LSM_FUNCTION_TEXT, lsm_func_enum,
		    msm_routing_lsm_func_get, msm_routing_lsm_func_put),
        SOC_ENUM_EXT(AFE_PCM_TX_TEXT" "LSM_FUNCTION_TEXT, lsm_func_enum,
		    msm_routing_lsm_func_get, msm_routing_lsm_func_put),
	/* kcontrol of lsm_port */
	SOC_ENUM_EXT("LSM1 Port", lsm_port_enum,
			  msm_routing_lsm_port_get,
			  msm_routing_lsm_port_put),
	SOC_ENUM_EXT("LSM2 Port", lsm_port_enum,
			  msm_routing_lsm_port_get,
			  msm_routing_lsm_port_put),
	SOC_ENUM_EXT("LSM3 Port", lsm_port_enum,
			  msm_routing_lsm_port_get,
			  msm_routing_lsm_port_put),
	SOC_ENUM_EXT("LSM4 Port", lsm_port_enum,
			  msm_routing_lsm_port_get,
			  msm_routing_lsm_port_put),
	SOC_ENUM_EXT("LSM5 Port", lsm_port_enum,
			  msm_routing_lsm_port_get,
			  msm_routing_lsm_port_put),
	SOC_ENUM_EXT("LSM6 Port", lsm_port_enum,
			  msm_routing_lsm_port_get,
			  msm_routing_lsm_port_put),
	SOC_ENUM_EXT("LSM7 Port", lsm_port_enum,
			  msm_routing_lsm_port_get,
			  msm_routing_lsm_port_put),
	SOC_ENUM_EXT("LSM8 Port", lsm_port_enum,
			  msm_routing_lsm_port_get,
			  msm_routing_lsm_port_put),
};


static int msm_routing_get_app_type_cfg_control(struct snd_kcontrol *kcontrol,
					  struct snd_ctl_elem_value *ucontrol)
{
	return 0;
}

static int msm_routing_put_app_type_cfg_control(struct snd_kcontrol *kcontrol,
					  struct snd_ctl_elem_value *ucontrol)
{
	int i = 0, j;
	int num_app_types = ucontrol->value.integer.value[i++];

	pr_debug("%s\n", __func__);

	memset(app_type_cfg, 0, MAX_APP_TYPES*
				sizeof(struct msm_pcm_routing_app_type_data));
	if (num_app_types > MAX_APP_TYPES || num_app_types < 0) {
		pr_err("%s: number of app types %d is invalid\n",
			__func__, num_app_types);
		return -EINVAL;
	}
	for (j = 0; j < num_app_types; j++) {
		app_type_cfg[j].app_type =
				ucontrol->value.integer.value[i++];
		app_type_cfg[j].sample_rate =
				ucontrol->value.integer.value[i++];
		app_type_cfg[j].bit_width =
				ucontrol->value.integer.value[i++];
	}

	return 0;
}

static int msm_routing_put_app_type_gain_control(struct snd_kcontrol *kcontrol,
					  struct snd_ctl_elem_value *ucontrol)
{
	int j, fe_id, be_id, port_type;
	int ret = 0;
	unsigned long copp;
	struct msm_pcm_routing_bdai_data *bedai;
	int dir = ucontrol->value.integer.value[0] ? SESSION_TYPE_TX :
						     SESSION_TYPE_RX;
	int app_type = ucontrol->value.integer.value[1];
	int gain = (ucontrol->value.integer.value[2] +
		    ucontrol->value.integer.value[3])/2;

	port_type = (dir == SESSION_TYPE_RX) ? MSM_AFE_PORT_TYPE_RX :
					       MSM_AFE_PORT_TYPE_TX;

	mutex_lock(&routing_lock);
	for (be_id = 0; be_id < MSM_BACKEND_DAI_MAX; be_id++) {
		if (is_be_dai_extproc(be_id))
			continue;

		bedai = &msm_bedais[be_id];
		if (afe_get_port_type(bedai->port_id) != port_type)
			continue;

		if (!bedai->active)
			continue;

		for (fe_id = 0; fe_id < MSM_FRONTEND_DAI_MAX; fe_id++) {
			if (!test_bit(fe_id, &bedai->fe_sessions[0]))
				continue;

			if (app_type !=
			    fe_dai_app_type_cfg[fe_id][dir][be_id].app_type)
				continue;

			copp = session_copp_map[fe_id][dir][be_id];
			for (j = 0; j < MAX_COPPS_PER_PORT; j++) {
				if (!test_bit(j, &copp))
					continue;
				ret |= adm_set_volume(bedai->port_id, j, gain);
			}
		}
	}
	mutex_unlock(&routing_lock);
	return ret ? -EINVAL : 0;
}

static const struct snd_kcontrol_new app_type_cfg_controls[] = {
	SOC_SINGLE_MULTI_EXT("App Type Config", SND_SOC_NOPM, 0,
	0x7FFFFFFF, 0, 128, msm_routing_get_app_type_cfg_control,
	msm_routing_put_app_type_cfg_control),
	SOC_SINGLE_MULTI_EXT("App Type Gain", SND_SOC_NOPM, 0,
	0x7FFFFFFF, 0, 4, NULL, msm_routing_put_app_type_gain_control)
};

static int msm_routing_put_module_cfg_control(struct snd_kcontrol *kcontrol,
					  struct snd_ctl_elem_value *ucontrol)
{
	int copp_idx, fe_id, be_id, port_type;
	int ret = 0;
	unsigned long copp;
	struct msm_pcm_routing_bdai_data *bedai;
	u8 *packed_params = NULL;
	struct param_hdr_v3 param_hdr;
	u32 packed_param_size = (sizeof(struct param_hdr_v3) +
				 sizeof(uint32_t));

	int dir = ucontrol->value.integer.value[0] ? SESSION_TYPE_TX :
						     SESSION_TYPE_RX;
	int app_type = ucontrol->value.integer.value[1];
	int module_id = ucontrol->value.integer.value[2];
	int instance_id = ucontrol->value.integer.value[3];
	int param_id = ucontrol->value.integer.value[4];
	int param_value = ucontrol->value.integer.value[5];

	port_type = (dir == SESSION_TYPE_RX) ? MSM_AFE_PORT_TYPE_RX :
					       MSM_AFE_PORT_TYPE_TX;
	pr_debug("%s app_type:%d mod_id:%d instance_id:%d param_id:%d value:%d\n",
		  __func__, app_type, module_id,
		  instance_id, param_id, param_value);

	packed_params = kzalloc(packed_param_size, GFP_KERNEL);
	if (!packed_params)
		return -ENOMEM;

	memset(&param_hdr, 0, sizeof(param_hdr));
	param_hdr.module_id = module_id;
	param_hdr.instance_id = instance_id;
	param_hdr.param_id = param_id;
	param_hdr.param_size = sizeof(uint32_t);

	packed_param_size = 0;

	mutex_lock(&routing_lock);
	for (be_id = 0; be_id < MSM_BACKEND_DAI_MAX; be_id++) {
		if (is_be_dai_extproc(be_id))
			continue;

		bedai = &msm_bedais[be_id];
		if (afe_get_port_type(bedai->port_id) != port_type)
			continue;

		if (!bedai->active)
			continue;

		for (fe_id = 0; fe_id < MSM_FRONTEND_DAI_MAX; fe_id++) {
			if (!test_bit(fe_id, &bedai->fe_sessions[0]))
				continue;

			if (app_type !=
			    fe_dai_app_type_cfg[fe_id][dir][be_id].app_type)
				continue;

			copp = session_copp_map[fe_id][dir][be_id];
			for (copp_idx = 0; copp_idx < MAX_COPPS_PER_PORT;
			     copp_idx++) {
				if (!test_bit(copp_idx, &copp))
					continue;

				ret = q6common_pack_pp_params(packed_params,
							&param_hdr,
							(u8 *) &param_value,
							&packed_param_size);
				if (ret) {
					pr_err("%s: Failed to pack params, error %d\n",
					       __func__, ret);
					goto done;
				}

				ret = adm_set_pp_params(bedai->port_id,
							 copp_idx, NULL,
							 packed_params,
							 packed_param_size);
				if (ret) {
					pr_err("%s: Setting param failed with err=%d\n",
						__func__, ret);
					ret = -EINVAL;
					goto done;
				}
			}
		}
	}
done:
	mutex_unlock(&routing_lock);
	kfree(packed_params);
	return ret;
}

static const struct snd_kcontrol_new module_cfg_controls[] = {
	SOC_SINGLE_MULTI_EXT("Audio Effect", SND_SOC_NOPM, 0,
	0x2000, 0, 6, NULL, msm_routing_put_module_cfg_control)
};

static int msm_routing_get_lsm_app_type_cfg_control(
					struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *ucontrol)
{
	int shift = ((struct soc_multi_mixer_control *)
				kcontrol->private_value)->shift;
	int i = 0, j = 0;

	mutex_lock(&routing_lock);
	ucontrol->value.integer.value[i] = num_app_cfg_types;

	for (j = 0; j < num_app_cfg_types; ++j) {
		ucontrol->value.integer.value[++i] =
				lsm_app_type_cfg[j].app_type;
		ucontrol->value.integer.value[++i] =
				lsm_app_type_cfg[j].sample_rate;
		ucontrol->value.integer.value[++i] =
				lsm_app_type_cfg[j].bit_width;
		if (shift == 1)
			ucontrol->value.integer.value[++i] =
				lsm_app_type_cfg[j].num_out_channels;
	}
	mutex_unlock(&routing_lock);
	return 0;
}

static int msm_routing_put_lsm_app_type_cfg_control(
					struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *ucontrol)
{
	int shift = ((struct soc_multi_mixer_control *)
				kcontrol->private_value)->shift;
	int i = 0, j;

	mutex_lock(&routing_lock);
	if (ucontrol->value.integer.value[0] < 0 ||
	    ucontrol->value.integer.value[0] > MAX_APP_TYPES) {
		pr_err("%s: number of app types %ld is invalid\n",
			__func__, ucontrol->value.integer.value[0]);
		mutex_unlock(&routing_lock);
		return -EINVAL;
	}

	num_app_cfg_types = ucontrol->value.integer.value[i++];
	memset(lsm_app_type_cfg, 0, MAX_APP_TYPES*
	       sizeof(struct msm_pcm_routing_app_type_data));

	for (j = 0; j < num_app_cfg_types; j++) {
		lsm_app_type_cfg[j].app_type =
				ucontrol->value.integer.value[i++];
		lsm_app_type_cfg[j].sample_rate =
				ucontrol->value.integer.value[i++];
		lsm_app_type_cfg[j].bit_width =
				ucontrol->value.integer.value[i++];
		/* Shift of 1 indicates this is V2 mixer control */
		if (shift == 1)
			lsm_app_type_cfg[j].num_out_channels =
				ucontrol->value.integer.value[i++];
	}
	mutex_unlock(&routing_lock);
	return 0;
}

static const struct snd_kcontrol_new lsm_app_type_cfg_controls[] = {
	SOC_SINGLE_MULTI_EXT("Listen App Type Config", SND_SOC_NOPM, 0,
	0xFFFFFFFF, 0, 128, msm_routing_get_lsm_app_type_cfg_control,
	msm_routing_put_lsm_app_type_cfg_control),
	SOC_SINGLE_MULTI_EXT("Listen App Type Config V2", SND_SOC_NOPM, 1,
	0xFFFFFFFF, 0, 128, msm_routing_get_lsm_app_type_cfg_control,
	msm_routing_put_lsm_app_type_cfg_control),
};


int msm_routing_get_rms_value_control(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol) {
	int rc = 0;
	int be_idx = 0;
	char *param_value;
	int *update_param_value;
	uint32_t param_size = (RMS_PAYLOAD_LEN + 1) * sizeof(uint32_t);
	struct param_hdr_v3 param_hdr;

	param_value = kzalloc(param_size, GFP_KERNEL);
	if (!param_value)
		return -ENOMEM;

	memset(&param_hdr, 0, sizeof(param_hdr));
	for (be_idx = 0; be_idx < MSM_BACKEND_DAI_MAX; be_idx++)
		if (msm_bedais[be_idx].port_id == SLIMBUS_0_TX)
			break;
	if ((be_idx < MSM_BACKEND_DAI_MAX) && msm_bedais[be_idx].active) {
		param_hdr.module_id = RMS_MODULEID_APPI_PASSTHRU;
		param_hdr.instance_id = INSTANCE_ID_0;
		param_hdr.param_id = RMS_PARAM_FIRST_SAMPLE;
		param_hdr.param_size = param_size;
		rc = adm_get_pp_params(SLIMBUS_0_TX, 0, ADM_CLIENT_ID_DEFAULT,
				       NULL, &param_hdr, (u8 *) param_value);
		if (rc) {
			pr_err("%s: get parameters failed:%d\n", __func__, rc);
			kfree(param_value);
			return -EINVAL;
		}
		update_param_value = (int *)param_value;
		ucontrol->value.integer.value[0] = update_param_value[0];

		pr_debug("%s: FROM DSP value[0] 0x%x\n",
			  __func__, update_param_value[0]);
	}
	kfree(param_value);
	return 0;
}





static int spkr_prot_put_vi_lch_port(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	int ret = 0;
	struct soc_enum *e = (struct soc_enum *)kcontrol->private_value;

	pr_debug("%s item is %d\n", __func__,
		   ucontrol->value.enumerated.item[0]);
	mutex_lock(&routing_lock);
	vi_lch_port = ucontrol->value.enumerated.item[0];
	if (vi_lch_port < e->items) {
		pr_debug("%s RX DAI ID %d TX DAI id %d\n",
			__func__, e->shift_l, e->values[vi_lch_port]);
		if (e->shift_l < MSM_BACKEND_DAI_MAX &&
			e->values[vi_lch_port] < MSM_BACKEND_DAI_MAX)
			/* Enable feedback TX path */
			ret = afe_spk_prot_feed_back_cfg(
			   msm_bedais[e->values[vi_lch_port]].port_id,
			   msm_bedais[e->shift_l].port_id, 1, 0, 1);
		else {
			pr_debug("%s values are out of range item %d\n",
			__func__, e->values[vi_lch_port]);
			/* Disable feedback TX path */
			if (e->values[vi_lch_port] == MSM_BACKEND_DAI_MAX)
				ret = afe_spk_prot_feed_back_cfg(0, 0, 0, 0, 0);
			else
				ret = -EINVAL;
		}
	} else {
		pr_err("%s item value is out of range items = %d\n", __func__,
			vi_lch_port);
		ret = -EINVAL;
	}
	mutex_unlock(&routing_lock);
	return ret;
}

static int spkr_prot_put_vi_rch_port(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	int ret = 0;
	struct soc_enum *e = (struct soc_enum *)kcontrol->private_value;

	pr_debug("%s item is %d\n", __func__,
			ucontrol->value.enumerated.item[0]);
	mutex_lock(&routing_lock);
	vi_rch_port = ucontrol->value.enumerated.item[0];
	if (vi_rch_port < e->items) {
		pr_debug("%s RX DAI ID %d TX DAI id %d\n",
				__func__, e->shift_l, e->values[vi_rch_port]);
		if (e->shift_l < MSM_BACKEND_DAI_MAX &&
				e->values[vi_rch_port] < MSM_BACKEND_DAI_MAX)
			/* Enable feedback TX path */
			ret = afe_spk_prot_feed_back_cfg(
					msm_bedais[e->values[vi_rch_port]].port_id,
					msm_bedais[e->shift_l].port_id,
					1, 1, 1);
		else {
			pr_debug("%s values are out of range item %d\n",
					__func__, e->values[vi_rch_port]);
			/* Disable feedback TX path */
			if (e->values[vi_rch_port] == MSM_BACKEND_DAI_MAX)
				ret = afe_spk_prot_feed_back_cfg(0,
						0, 0, 0, 0);
			else
				ret = -EINVAL;
		}
	} else {
		pr_err("%s item value is out of range items = %d\n", __func__,
			vi_rch_port);
		ret = -EINVAL;
	}
	mutex_unlock(&routing_lock);
	return ret;
}

static int spkr_prot_get_vi_lch_port(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.enumerated.item[0] = vi_lch_port;
	pr_debug("%s\n", __func__);
	return 0;
}

static int spkr_prot_get_vi_rch_port(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	pr_debug("%s\n", __func__);
	ucontrol->value.enumerated.item[0] = vi_rch_port;
	return 0;
}

static const char * const slim0_rx_vi_fb_tx_lch_mux_text[] = {
	"ZERO", "SLIM4_TX"
};

static const char * const slim0_rx_vi_fb_tx_rch_mux_text[] = {
	"ZERO", "SLIM4_TX"
};

static const char * const wsa_rx_0_vi_fb_tx_lch_mux_text[] = {
	"ZERO", "WSA_CDC_DMA_TX_0"
};

static const char * const wsa_rx_0_vi_fb_tx_rch_mux_text[] = {
	"ZERO", "WSA_CDC_DMA_TX_0"
};

static const char * const mi2s_rx_vi_fb_tx_mux_text[] = {
	"ZERO", "SENARY_TX"
};

static const char * const int4_mi2s_rx_vi_fb_tx_mono_mux_text[] = {
	"ZERO", "INT5_MI2S_TX"
};

static const char * const int4_mi2s_rx_vi_fb_tx_stereo_mux_text[] = {
	"ZERO", "INT5_MI2S_TX"
};

static const int slim0_rx_vi_fb_tx_lch_value[] = {
	MSM_BACKEND_DAI_MAX, MSM_BACKEND_DAI_SLIMBUS_4_TX
};

static const int slim0_rx_vi_fb_tx_rch_value[] = {
	MSM_BACKEND_DAI_MAX, MSM_BACKEND_DAI_SLIMBUS_4_TX
};

static const int wsa_rx_0_vi_fb_tx_lch_value[] = {
	MSM_BACKEND_DAI_MAX, MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0
};

static const int wsa_rx_0_vi_fb_tx_rch_value[] = {
	MSM_BACKEND_DAI_MAX, MSM_BACKEND_DAI_WSA_CDC_DMA_TX_0
};


static const int mi2s_rx_vi_fb_tx_value[] = {
	MSM_BACKEND_DAI_MAX, MSM_BACKEND_DAI_SENARY_MI2S_TX
};

static const int int4_mi2s_rx_vi_fb_tx_mono_ch_value[] = {
	MSM_BACKEND_DAI_MAX, MSM_BACKEND_DAI_INT5_MI2S_TX
};

static const int int4_mi2s_rx_vi_fb_tx_stereo_ch_value[] = {
	MSM_BACKEND_DAI_MAX, MSM_BACKEND_DAI_INT5_MI2S_TX
};

static const struct soc_enum slim0_rx_vi_fb_lch_mux_enum =
	SOC_VALUE_ENUM_DOUBLE(0, MSM_BACKEND_DAI_SLIMBUS_0_RX, 0, 0,
	ARRAY_SIZE(slim0_rx_vi_fb_tx_lch_mux_text),
	slim0_rx_vi_fb_tx_lch_mux_text, slim0_rx_vi_fb_tx_lch_value);

static const struct soc_enum slim0_rx_vi_fb_rch_mux_enum =
	SOC_VALUE_ENUM_DOUBLE(0, MSM_BACKEND_DAI_SLIMBUS_0_RX, 0, 0,
	ARRAY_SIZE(slim0_rx_vi_fb_tx_rch_mux_text),
	slim0_rx_vi_fb_tx_rch_mux_text, slim0_rx_vi_fb_tx_rch_value);

static const struct soc_enum wsa_rx_0_vi_fb_lch_mux_enum =
	SOC_VALUE_ENUM_DOUBLE(0, MSM_BACKEND_DAI_WSA_CDC_DMA_RX_0, 0, 0,
	ARRAY_SIZE(wsa_rx_0_vi_fb_tx_lch_mux_text),
	wsa_rx_0_vi_fb_tx_lch_mux_text, wsa_rx_0_vi_fb_tx_lch_value);

static const struct soc_enum wsa_rx_0_vi_fb_rch_mux_enum =
	SOC_VALUE_ENUM_DOUBLE(0, MSM_BACKEND_DAI_WSA_CDC_DMA_RX_0, 0, 0,
	ARRAY_SIZE(wsa_rx_0_vi_fb_tx_rch_mux_text),
	wsa_rx_0_vi_fb_tx_rch_mux_text, wsa_rx_0_vi_fb_tx_rch_value);

static const struct soc_enum mi2s_rx_vi_fb_mux_enum =
	SOC_VALUE_ENUM_DOUBLE(0, MSM_BACKEND_DAI_PRI_MI2S_RX, 0, 0,
	ARRAY_SIZE(mi2s_rx_vi_fb_tx_mux_text),
	mi2s_rx_vi_fb_tx_mux_text, mi2s_rx_vi_fb_tx_value);

static const struct soc_enum int4_mi2s_rx_vi_fb_mono_ch_mux_enum =
	SOC_VALUE_ENUM_DOUBLE(0, MSM_BACKEND_DAI_INT4_MI2S_RX, 0, 0,
	ARRAY_SIZE(int4_mi2s_rx_vi_fb_tx_mono_mux_text),
	int4_mi2s_rx_vi_fb_tx_mono_mux_text,
	int4_mi2s_rx_vi_fb_tx_mono_ch_value);

static const struct soc_enum int4_mi2s_rx_vi_fb_stereo_ch_mux_enum =
	SOC_VALUE_ENUM_DOUBLE(0, MSM_BACKEND_DAI_INT4_MI2S_RX, 0, 0,
	ARRAY_SIZE(int4_mi2s_rx_vi_fb_tx_stereo_mux_text),
	int4_mi2s_rx_vi_fb_tx_stereo_mux_text,
	int4_mi2s_rx_vi_fb_tx_stereo_ch_value);

static const struct snd_kcontrol_new slim0_rx_vi_fb_lch_mux =
	SOC_DAPM_ENUM_EXT("SLIM0_RX_VI_FB_LCH_MUX",
	slim0_rx_vi_fb_lch_mux_enum, spkr_prot_get_vi_lch_port,
	spkr_prot_put_vi_lch_port);

static const struct snd_kcontrol_new slim0_rx_vi_fb_rch_mux =
	SOC_DAPM_ENUM_EXT("SLIM0_RX_VI_FB_RCH_MUX",
	slim0_rx_vi_fb_rch_mux_enum, spkr_prot_get_vi_rch_port,
	spkr_prot_put_vi_rch_port);

static const struct snd_kcontrol_new wsa_rx_0_vi_fb_lch_mux =
	SOC_DAPM_ENUM_EXT("WSA_RX_0_VI_FB_LCH_MUX",
	wsa_rx_0_vi_fb_lch_mux_enum, spkr_prot_get_vi_lch_port,
	spkr_prot_put_vi_lch_port);

static const struct snd_kcontrol_new wsa_rx_0_vi_fb_rch_mux =
	SOC_DAPM_ENUM_EXT("WSA_RX_0_VI_FB_RCH_MUX",
	wsa_rx_0_vi_fb_rch_mux_enum, spkr_prot_get_vi_rch_port,
	spkr_prot_put_vi_rch_port);

static const struct snd_kcontrol_new mi2s_rx_vi_fb_mux =
	SOC_DAPM_ENUM_EXT("PRI_MI2S_RX_VI_FB_MUX",
	mi2s_rx_vi_fb_mux_enum, spkr_prot_get_vi_lch_port,
	spkr_prot_put_vi_lch_port);

static const struct snd_kcontrol_new int4_mi2s_rx_vi_fb_mono_ch_mux =
	SOC_DAPM_ENUM_EXT("INT4_MI2S_RX_VI_FB_MONO_CH_MUX",
	int4_mi2s_rx_vi_fb_mono_ch_mux_enum, spkr_prot_get_vi_lch_port,
	spkr_prot_put_vi_lch_port);

static const struct snd_kcontrol_new int4_mi2s_rx_vi_fb_stereo_ch_mux =
	SOC_DAPM_ENUM_EXT("INT4_MI2S_RX_VI_FB_STEREO_CH_MUX",
	int4_mi2s_rx_vi_fb_stereo_ch_mux_enum, spkr_prot_get_vi_rch_port,
	spkr_prot_put_vi_rch_port);

static const struct snd_soc_dapm_widget msm_qdsp6_widgets[] = {
	/* Frontend AIF */
	/* Widget name equals to Front-End DAI name<Need confirmation>,
	 * Stream name must contains substring of front-end dai name
	 */
	SND_SOC_DAPM_AIF_IN("MM_DL1", "MultiMedia1 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL2", "MultiMedia2 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL3", "MultiMedia3 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL4", "MultiMedia4 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL5", "MultiMedia5 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL6", "MultiMedia6 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL7", "MultiMedia7 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL8", "MultiMedia8 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL9", "MultiMedia9 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL10", "MultiMedia10 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL11", "MultiMedia11 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL12", "MultiMedia12 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL13", "MultiMedia13 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL14", "MultiMedia14 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL15", "MultiMedia15 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL16", "MultiMedia16 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL20", "MultiMedia20 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL21", "MultiMedia21 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL22", "MultiMedia22 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL23", "MultiMedia23 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL24", "MultiMedia24 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL25", "MultiMedia25 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL26", "MultiMedia26 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL31", "MultiMedia31 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL32", "MultiMedia32 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL33", "MultiMedia33 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("MM_DL34", "MultiMedia34 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("VOIP_DL", "VoIP Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL1", "MultiMedia1 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL2", "MultiMedia2 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL3", "MultiMedia3 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL4", "MultiMedia4 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL5", "MultiMedia5 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL6", "MultiMedia6 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL8", "MultiMedia8 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL9", "MultiMedia9 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL10", "MultiMedia10 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL16", "MultiMedia16 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL17", "MultiMedia17 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL18", "MultiMedia18 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL19", "MultiMedia19 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL20", "MultiMedia20 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL21", "MultiMedia21 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL22", "MultiMedia22 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL23", "MultiMedia23 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL24", "MultiMedia24 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL25", "MultiMedia25 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL27", "MultiMedia27 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL28", "MultiMedia28 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL29", "MultiMedia29 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL30", "MultiMedia30 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL31", "MultiMedia31 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL32", "MultiMedia32 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL33", "MultiMedia33 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("MM_UL34", "MultiMedia34 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("VOIP_UL", "VoIP Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("VOICEMMODE1_DL",
		"VoiceMMode1 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("VOICEMMODE1_UL",
		"VoiceMMode1 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("VOICEMMODE2_DL",
		"VoiceMMode2 Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("VOICEMMODE2_UL",
		"VoiceMMode2 Capture", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("INTHFP_DL_HL", "INT_HFP_BT_HOSTLESS Playback",
		0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("INTHFP_UL_HL", "INT_HFP_BT_HOSTLESS Capture",
	0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("USBAUDIO_DL_HL", "USBAUDIO_HOSTLESS Playback",
		0, 0, 0, 0),
	SND_SOC_DAPM_AIF_OUT("USBAUDIO_UL_HL", "USBAUDIO_HOSTLESS Capture",
		0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("HDMI_DL_HL", "HDMI_HOSTLESS Playback", 0, 0, 0, 0),
	SND_SOC_DAPM_AIF_IN("