// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.cronet_sample_apk;

import static androidx.test.espresso.Espresso.onView;
import static androidx.test.espresso.action.ViewActions.pressBack;
import static androidx.test.espresso.assertion.ViewAssertions.doesNotExist;
import static androidx.test.espresso.assertion.ViewAssertions.matches;
import static androidx.test.espresso.matcher.ViewMatchers.isDisplayed;
import static androidx.test.espresso.matcher.ViewMatchers.withId;

import android.content.ComponentName;
import android.content.Intent;
import android.net.Uri;
import android.os.ConditionVariable;
import android.text.Editable;
import android.text.TextWatcher;
import android.widget.TextView;

import androidx.test.InstrumentationRegistry;
import androidx.test.espresso.NoMatchingViewException;
import androidx.test.filters.SmallTest;
import androidx.test.rule.ActivityTestRule;
import androidx.test.runner.AndroidJUnit4;

import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;

import org.chromium.net.impl.CronetLogger.CronetSource;
import org.chromium.net.impl.CronetManifest;

/**
 * Base test class for all CronetSample based tests.
 */
@RunWith(AndroidJUnit4.class)
public class CronetSampleTest {
    private final String mUrl = "https://localhost";

    @Rule
    public ActivityTestRule<CronetSampleActivity> mActivityTestRule =
            new ActivityTestRule<>(CronetSampleActivity.class, false, false);

    @Test
    @SmallTest
    public void testLoadUrl() throws Exception {
        CronetSampleActivity activity = launchCronetSampleWithUrl(mUrl);

        // Make sure the activity was created as expected.
        Assert.assertNotNull(activity);

        // Verify successful fetch.
        final TextView textView = (TextView) activity.findViewById(R.id.resultView);
        final ConditionVariable done = new ConditionVariable();
        final TextWatcher textWatcher = new TextWatcher() {
            @Override
            public void afterTextChanged(Editable s) {}

            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {}

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                if (s.toString().startsWith("Failed " + mUrl
                            + " (Exception in CronetUrlRequest: net::ERR_CONNECTION_REFUSED")) {
                    done.open();
                }
            }
        };
        textView.addTextChangedListener(textWatcher);
        // Check current text in case it changed before |textWatcher| was added.
        textWatcher.onTextChanged(textView.getText(), 0, 0, 0);
        done.block();
    }

    @Test
    @SmallTest
    public void testOpenUrlPromptWhenDataViewClicked() {
        CronetSampleActivity activity = launchCronetSampleWithUrl(mUrl);
        // Make sure the activity was created as expected.
        Assert.assertNotNull(activity);

        // Blindly interacting with the view is racy as it is generated by a runnable spawned by the
        // onFailed or onSucceeded methods (which are also asynchronously called after returning
        // from launchCronetSampleWithUrl()). Unfortunately, there doesn't seem to be a simple way
        // to synchronize with a runnable on the UI thread (?), so all we can do is spin.
        boolean isWaitingForView = true;
        while (isWaitingForView) {
            isWaitingForView = false;
            try {
                onView(withId(R.id.urlView)).perform(pressBack());
                onView(withId(R.id.urlView)).check(doesNotExist());
            } catch (NoMatchingViewException e) {
                isWaitingForView = true;
                try {
                    final long sleepDurationMs = 200;
                    Thread.sleep(sleepDurationMs);
                } catch (Exception e2) {
                }
            }
        }

        // The activity begins with the url prompt showing.
        // Press back on the url prompt to clear it from the screen.
        final TextView dataView = (TextView) activity.findViewById(R.id.dataView);
        final ConditionVariable done = new ConditionVariable();
        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                dataView.performClick();
                done.open();
            }
        });
        done.block();
        onView(withId(R.id.urlView)).check(matches(isDisplayed()));
    }

    @Test
    @SmallTest
    public void testCronetLoggingManifestMetadataNotSet() {
        Assert.assertFalse(CronetManifest.isAppOptedInForTelemetry(
                InstrumentationRegistry.getTargetContext(), null));
        Assert.assertFalse(
                CronetManifest.isAppOptedInForTelemetry(InstrumentationRegistry.getTargetContext(),
                        CronetSource.CRONET_SOURCE_STATICALLY_LINKED));
        Assert.assertFalse(
                CronetManifest.isAppOptedInForTelemetry(InstrumentationRegistry.getTargetContext(),
                        CronetSource.CRONET_SOURCE_PLAY_SERVICES));
        Assert.assertFalse(CronetManifest.isAppOptedInForTelemetry(
                InstrumentationRegistry.getTargetContext(), CronetSource.CRONET_SOURCE_FALLBACK));
    }

    /**
     * Starts the CronetSample activity and loads the given URL.
     */
    protected CronetSampleActivity launchCronetSampleWithUrl(String url) {
        Intent intent = new Intent(Intent.ACTION_MAIN);
        intent.addCategory(Intent.CATEGORY_LAUNCHER);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.setData(Uri.parse(url));
        intent.setComponent(new ComponentName(
                InstrumentationRegistry.getTargetContext(), CronetSampleActivity.class));
        return mActivityTestRule.launchActivity(intent);
    }
}
