// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/cert/crl_set.h"

#include "base/files/file_util.h"
#include "crypto/sha2.h"
#include "net/cert/asn1_util.h"
#include "net/cert/x509_certificate.h"
#include "net/cert/x509_util.h"
#include "net/test/cert_test_util.h"
#include "net/test/test_data_directory.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace net {

// These data blocks were generated using a lot of code that is still in
// development. For now, if you need to update them, you have to contact agl.
static const uint8_t kGIACRLSet[] = {
  0x60, 0x00, 0x7b, 0x22, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a,
  0x30, 0x2c, 0x22, 0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x54, 0x79, 0x70,
  0x65, 0x22, 0x3a, 0x22, 0x43, 0x52, 0x4c, 0x53, 0x65, 0x74, 0x22, 0x2c, 0x22,
  0x53, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x63, 0x65, 0x22, 0x3a, 0x30, 0x2c, 0x22,
  0x44, 0x65, 0x6c, 0x74, 0x61, 0x46, 0x72, 0x6f, 0x6d, 0x22, 0x3a, 0x30, 0x2c,
  0x22, 0x4e, 0x75, 0x6d, 0x50, 0x61, 0x72, 0x65, 0x6e, 0x74, 0x73, 0x22, 0x3a,
  0x31, 0x2c, 0x22, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x65, 0x64, 0x53, 0x50, 0x4b,
  0x49, 0x73, 0x22, 0x3a, 0x5b, 0x5d, 0x7d, 0xb6, 0xb9, 0x54, 0x32, 0xab, 0xae,
  0x57, 0xfe, 0x02, 0x0c, 0xb2, 0xb7, 0x4f, 0x4f, 0x9f, 0x91, 0x73, 0xc8, 0xc7,
  0x08, 0xaf, 0xc9, 0xe7, 0x32, 0xac, 0xe2, 0x32, 0x79, 0x04, 0x7c, 0x6d, 0x05,
  0x0d, 0x00, 0x00, 0x00, 0x0a, 0x10, 0x0d, 0x7f, 0x30, 0x00, 0x03, 0x00, 0x00,
  0x23, 0xb0, 0x0a, 0x10, 0x0e, 0x37, 0x06, 0x00, 0x03, 0x00, 0x00, 0x23, 0xb1,
  0x0a, 0x16, 0x25, 0x42, 0x54, 0x00, 0x03, 0x00, 0x00, 0x14, 0x51, 0x0a, 0x16,
  0x69, 0xd1, 0xd7, 0x00, 0x03, 0x00, 0x00, 0x14, 0x52, 0x0a, 0x16, 0x70, 0x8c,
  0x22, 0x00, 0x03, 0x00, 0x00, 0x14, 0x53, 0x0a, 0x16, 0x71, 0x31, 0x2c, 0x00,
  0x03, 0x00, 0x00, 0x14, 0x54, 0x0a, 0x16, 0x7d, 0x75, 0x9d, 0x00, 0x03, 0x00,
  0x00, 0x14, 0x55, 0x0a, 0x1f, 0xee, 0xf9, 0x49, 0x00, 0x03, 0x00, 0x00, 0x23,
  0xae, 0x0a, 0x1f, 0xfc, 0xd1, 0x89, 0x00, 0x03, 0x00, 0x00, 0x23, 0xaf, 0x0a,
  0x61, 0xdd, 0xc7, 0x48, 0x00, 0x03, 0x00, 0x00, 0x18, 0x0e, 0x0a, 0x61, 0xe6,
  0x12, 0x64, 0x00, 0x03, 0x00, 0x00, 0x18, 0x0f, 0x0a, 0x61, 0xe9, 0x46, 0x56,
  0x00, 0x03, 0x00, 0x00, 0x18, 0x10, 0x0a, 0x64, 0x63, 0x49, 0xd2, 0x00, 0x03,
  0x00, 0x00, 0x1d, 0x77,
};

static const uint8_t kBlockedSPKICRLSet[] = {
  0x8e, 0x00, 0x7b, 0x22, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a,
  0x30, 0x2c, 0x22, 0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x54, 0x79, 0x70,
  0x65, 0x22, 0x3a, 0x22, 0x43, 0x52, 0x4c, 0x53, 0x65, 0x74, 0x22, 0x2c, 0x22,
  0x53, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x63, 0x65, 0x22, 0x3a, 0x30, 0x2c, 0x22,
  0x44, 0x65, 0x6c, 0x74, 0x61, 0x46, 0x72, 0x6f, 0x6d, 0x22, 0x3a, 0x30, 0x2c,
  0x22, 0x4e, 0x75, 0x6d, 0x50, 0x61, 0x72, 0x65, 0x6e, 0x74, 0x73, 0x22, 0x3a,
  0x30, 0x2c, 0x22, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x65, 0x64, 0x53, 0x50, 0x4b,
  0x49, 0x73, 0x22, 0x3a, 0x5b, 0x22, 0x34, 0x37, 0x44, 0x45, 0x51, 0x70, 0x6a,
  0x38, 0x48, 0x42, 0x53, 0x61, 0x2b, 0x2f, 0x54, 0x49, 0x6d, 0x57, 0x2b, 0x35,
  0x4a, 0x43, 0x65, 0x75, 0x51, 0x65, 0x52, 0x6b, 0x6d, 0x35, 0x4e, 0x4d, 0x70,
  0x4a, 0x57, 0x5a, 0x47, 0x33, 0x68, 0x53, 0x75, 0x46, 0x55, 0x3d, 0x22, 0x5d,
  0x7d,
};

static const uint8_t kExpiredCRLSet[] = {
  0x6d, 0x00, 0x7b, 0x22, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a,
  0x30, 0x2c, 0x22, 0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x54, 0x79, 0x70,
  0x65, 0x22, 0x3a, 0x22, 0x43, 0x52, 0x4c, 0x53, 0x65, 0x74, 0x22, 0x2c, 0x22,
  0x53, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x63, 0x65, 0x22, 0x3a, 0x31, 0x2c, 0x22,
  0x44, 0x65, 0x6c, 0x74, 0x61, 0x46, 0x72, 0x6f, 0x6d, 0x22, 0x3a, 0x30, 0x2c,
  0x22, 0x4e, 0x75, 0x6d, 0x50, 0x61, 0x72, 0x65, 0x6e, 0x74, 0x73, 0x22, 0x3a,
  0x30, 0x2c, 0x22, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x65, 0x64, 0x53, 0x50, 0x4b,
  0x49, 0x73, 0x22, 0x3a, 0x5b, 0x5d, 0x2c, 0x22, 0x4e, 0x6f, 0x74, 0x41, 0x66,
  0x74, 0x65, 0x72, 0x22, 0x3a, 0x31, 0x7d,
};

// kGIASPKISHA256 is the SHA256 digest the Google Internet Authority's
// SubjectPublicKeyInfo.
static const uint8_t kGIASPKISHA256[32] = {
  0xb6, 0xb9, 0x54, 0x32, 0xab, 0xae, 0x57, 0xfe, 0x02, 0x0c, 0xb2, 0xb7, 0x4f,
  0x4f, 0x9f, 0x91, 0x73, 0xc8, 0xc7, 0x08, 0xaf, 0xc9, 0xe7, 0x32, 0xac, 0xe2,
  0x32, 0x79, 0x04, 0x7c, 0x6d, 0x05,
};

TEST(CRLSetTest, Parse) {
  base::StringPiece s(reinterpret_cast<const char*>(kGIACRLSet),
                      sizeof(kGIACRLSet));
  scoped_refptr<CRLSet> set;
  EXPECT_TRUE(CRLSet::Parse(s, &set));
  ASSERT_TRUE(set.get() != nullptr);

  const CRLSet::CRLList& crls = set->CrlsForTesting();
  ASSERT_EQ(1u, crls.size());
  const std::vector<std::string>& serials = crls.begin()->second;
  static const unsigned kExpectedNumSerials = 13;
  ASSERT_EQ(kExpectedNumSerials, serials.size());
  EXPECT_EQ(std::string("\x10\x0D\x7F\x30\x00\x03\x00\x00\x23\xB0", 10),
            serials[0]);
  EXPECT_EQ(std::string("\x64\x63\x49\xD2\x00\x03\x00\x00\x1D\x77", 10),
            serials[kExpectedNumSerials - 1]);

  const std::string gia_spki_hash(reinterpret_cast<const char*>(kGIASPKISHA256),
                                  sizeof(kGIASPKISHA256));
  EXPECT_EQ(CRLSet::REVOKED,
            set->CheckSerial(
                std::string("\x16\x7D\x75\x9D\x00\x03\x00\x00\x14\x55", 10),
                gia_spki_hash));
  EXPECT_EQ(CRLSet::GOOD,
            set->CheckSerial(
                std::string("\x47\x54\x3E\x79\x00\x03\x00\x00\x14\xF5", 10),
                gia_spki_hash));

  EXPECT_FALSE(set->IsExpired());
}

TEST(CRLSetTest, BlockedSPKIs) {
  base::StringPiece s(reinterpret_cast<const char*>(kBlockedSPKICRLSet),
                      sizeof(kBlockedSPKICRLSet));
  scoped_refptr<CRLSet> set;
  EXPECT_TRUE(CRLSet::Parse(s, &set));
  ASSERT_TRUE(set.get() != nullptr);

  const uint8_t spki_hash[] = {
    227, 176, 196, 66, 152, 252, 28, 20, 154, 251, 244, 200, 153, 111, 185, 36,
    39, 174, 65, 228, 100, 155, 147, 76, 164, 149, 153, 27, 120, 82, 184, 85,
    0,
  };

  EXPECT_EQ(CRLSet::GOOD, set->CheckSPKI(""));
  EXPECT_EQ(CRLSet::REVOKED,
            set->CheckSPKI(reinterpret_cast<const char*>(spki_hash)));
}

TEST(CertVerifyProcTest, CRLSetIncorporatesStaticBlocklist) {
  // Test both the builtin CRLSet and a parsed CRLSet to be sure that both
  // include the block list.
  scoped_refptr<CRLSet> set1 = CRLSet::BuiltinCRLSet();
  ASSERT_TRUE(set1);
  base::StringPiece s(reinterpret_cast<const char*>(kGIACRLSet),
                      sizeof(kGIACRLSet));
  scoped_refptr<CRLSet> set2;
  EXPECT_TRUE(CRLSet::Parse(s, &set2));
  ASSERT_TRUE(set2);

  static const char* const kDigiNotarFilenames[] = {
      "diginotar_root_ca.pem",          "diginotar_cyber_ca.pem",
      "diginotar_services_1024_ca.pem", "diginotar_pkioverheid.pem",
      "diginotar_pkioverheid_g2.pem",   nullptr,
  };

  base::FilePath certs_dir = GetTestCertsDirectory();

  for (size_t i = 0; kDigiNotarFilenames[i]; i++) {
    scoped_refptr<X509Certificate> diginotar_cert =
        ImportCertFromFile(certs_dir, kDigiNotarFilenames[i]);
    ASSERT_TRUE(diginotar_cert);
    base::StringPiece spki;
    ASSERT_TRUE(asn1::ExtractSPKIFromDERCert(
        x509_util::CryptoBufferAsStringPiece(diginotar_cert->cert_buffer()),
        &spki));

    std::string spki_sha256 = crypto::SHA256HashString(spki);

    EXPECT_EQ(CRLSet::REVOKED, set1->CheckSPKI(spki_sha256))
        << "Public key not blocked for " << kDigiNotarFilenames[i];
    EXPECT_EQ(CRLSet::REVOKED, set2->CheckSPKI(spki_sha256))
        << "Public key not blocked for " << kDigiNotarFilenames[i];
  }
}

TEST(CRLSetTest, BlockedSubjects) {
  std::string crl_set_bytes;
  EXPECT_TRUE(base::ReadFileToString(
      GetTestCertsDirectory().AppendASCII("crlset_by_root_subject.raw"),
      &crl_set_bytes));
  scoped_refptr<CRLSet> set;
  EXPECT_TRUE(CRLSet::Parse(crl_set_bytes, &set));
  ASSERT_TRUE(set.get() != nullptr);

  scoped_refptr<X509Certificate> root = CreateCertificateChainFromFile(
      GetTestCertsDirectory(), "root_ca_cert.pem",
      X509Certificate::FORMAT_AUTO);
  base::StringPiece root_der =
      net::x509_util::CryptoBufferAsStringPiece(root->cert_buffer());

  base::StringPiece spki;
  ASSERT_TRUE(asn1::ExtractSPKIFromDERCert(root_der, &spki));
  SHA256HashValue spki_sha256;
  crypto::SHA256HashString(spki, spki_sha256.data, sizeof(spki_sha256.data));

  base::StringPiece subject;
  ASSERT_TRUE(asn1::ExtractSubjectFromDERCert(root_der, &subject));

  // Unrelated subjects are unaffected.
  EXPECT_EQ(CRLSet::GOOD, set->CheckSubject("abcdef", ""));

  // The subject in question is considered revoked if used with an unknown SPKI
  // hash.
  EXPECT_EQ(CRLSet::REVOKED,
            set->CheckSubject(
                subject,
                base::StringPiece(reinterpret_cast<const char*>(kGIASPKISHA256),
                                  sizeof(kGIASPKISHA256))));

  // When used with the correct hash, that subject should be accepted.
  EXPECT_EQ(CRLSet::GOOD,
            set->CheckSubject(
                subject, base::StringPiece(
                             reinterpret_cast<const char*>(spki_sha256.data),
                             sizeof(spki_sha256.data))));
}

TEST(CRLSetTest, Expired) {
  // This CRLSet has an expiry value set to one second past midnight, 1st Jan,
  // 1970.
  base::StringPiece s(reinterpret_cast<const char*>(kExpiredCRLSet),
                      sizeof(kExpiredCRLSet));
  scoped_refptr<CRLSet> set;
  EXPECT_TRUE(CRLSet::Parse(s, &set));
  ASSERT_TRUE(set.get() != nullptr);

  EXPECT_TRUE(set->IsExpired());
}

}  // namespace net
