/*
 * (C) 2021 FCNT LIMITED
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/version.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/pm_wakeup.h>
#include <linux/gpio.h>
#include <linux/gpio/consumer.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/time.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/irq.h>
#include <linux/of_gpio.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <linux/miscdevice.h>
#include <linux/soc/qcom/smem.h>
#include <linux/earphone.h>
#include "edc.h"

#define EDC_NOTFY_WL_TIME_MS            500
#define EDC_NOTFY_WL_TIME_JF            msecs_to_jiffies(EDC_NOTFY_WL_TIME_MS)
#define EDC_POLLING_TIME_MS             1000
#define EDC_POLLING_TIME_JF             msecs_to_jiffies(EDC_POLLING_TIME_MS)
#define EDC_WAKEUP_TIME_MS              100

static int earphone_pins_set_default(struct edc_data *edc_data);
static int earphone_pins_set_checking(struct edc_data *edc_data);
static int earphone_pins_set_inserted(struct edc_data *edc_data);

static int (*earphone_micbias_ctrl_func)(int) = NULL;
static int (*earphone_micbias_wakeup_func)(bool) = NULL;

static int jack_switch_types[] = {
	SW_MICROPHONE_INSERT,
	SW_HEADPHONE_INSERT,
	SW_UNSUPPORT_INSERT,

};

static int earphone_timer_init(struct edc_data *edc_data);
static int earphone_timer_start(struct edc_data *edc_data);
static int earphone_timer_stop(struct edc_data *edc_data);

void earphone_micbias_ctrl_unregister(void)
{
	pr_info("%s: called.\n", __func__);
	earphone_micbias_ctrl_func = NULL;
}
EXPORT_SYMBOL_GPL(earphone_micbias_ctrl_unregister);

void earphone_micbias_ctrl_register(void* func)
{
	pr_info("%s: called.\n", __func__);
	earphone_micbias_ctrl_func = func;
}
EXPORT_SYMBOL_GPL(earphone_micbias_ctrl_register);

void earphone_micbias_wakeup_unregister(void)
{
	pr_info("%s: called.\n", __func__);
	earphone_micbias_wakeup_func = NULL;
}
EXPORT_SYMBOL_GPL(earphone_micbias_wakeup_unregister);

void earphone_micbias_wakeup_register(void* func)
{
	pr_info("%s: called.\n", __func__);
	earphone_micbias_wakeup_func = func;
}
EXPORT_SYMBOL_GPL(earphone_micbias_wakeup_register);

/* audio driver function */
static int edc_headset_micbias_wakeup(bool on)
{
	if (!IS_ERR_OR_NULL(earphone_micbias_wakeup_func)) {
			return earphone_micbias_wakeup_func(on);
	}
	return -1;
}

/* audio driver function */
static int edc_headset_micbias_ctrl(int on)
{
	int ret = -1;
	int count = 0;
	pr_info("%s: called.\n", __func__);
	if (!IS_ERR_OR_NULL(earphone_micbias_ctrl_func)) {
		edc_headset_micbias_wakeup(true);
		for(count = 0; count < 20; count++) {
			ret = earphone_micbias_ctrl_func(on);
			if (ret) {
				EDC_WAIT_MS(10);
			} else {
				break;
			}
		}
		edc_headset_micbias_wakeup(false);
	}
	return ret;
}

int earphone_get_adc(struct edc_data *edc_data, enum edc_adc_type type, bool raw, int *adc_intval)
{
	int ret = 0;
	struct iio_channel *channel;

	if (type == EDC_ADC_XHEADSET_DET) {
		channel = edc_data->xheadset_det;
	} else {
		channel = edc_data->jmic_sw_adc;
	}

	if (raw == true) {
		ret = iio_read_channel_raw(channel, adc_intval);
	} else {
		ret = iio_read_channel_processed(channel, adc_intval);
	}
	if (ret < 0) {
		pr_err("%s: read channel failed. type = %d raw = %d ret = %d\n", __func__, type, raw, ret);
	} else {
		if (raw == true) {
			pr_debug("%s: read channel. type = %d raw = %d value = 0x%08X ret = %d\n", __func__, type, raw, *adc_intval, ret);
		} else {
			pr_debug("%s: read channel. type = %d raw = %d value = %d ret = %d\n", __func__, type, raw, *adc_intval, ret);
		}
		ret = 0;
	}

	return ret;
}

void earphone_get_adc_raw(struct edc_data *edc_data, enum edc_adc_type type, uint16_t *out)
{
	int adc_raw;

	if (!earphone_get_adc(edc_data, type, true, &adc_raw)) {
	    *out = adc_raw;
    } else {
	    *out = 0;
	}
}

enum edc_detect_type earphone_get_type_status(struct edc_data *edc_data)
{
	return edc_data->earphone_type;
}

static int earphone_set_type_status(struct edc_data *edc_data, enum edc_detect_type type)
{
	int prev_type;
	static int type_index = 0;
	static bool new_state = false;

	if (edc_data->earphone_type == type) {
		return 0;
	}

	if (new_state == true && !!type == true) {
		input_report_switch(edc_data->sw_dev, jack_switch_types[type_index], false);
		input_sync(edc_data->sw_dev);
	}

	prev_type = edc_data->earphone_type;
	edc_data->earphone_type = type;
	new_state = !!type;

	if (new_state) {
		switch (type) {
		case EDC_DET_TYPE_EARPHONE_MIC:
		case EDC_DET_TYPE_EARPHONE:
			type_index = type - 1;
			break;
		case EDC_DET_TYPE_NONE:
		case EDC_DET_TYPE_ANTENNA:
		default:
			type_index = 2;
			break;
		}
	} else {
		switch (prev_type) {
		case EDC_DET_TYPE_EARPHONE_MIC:
		case EDC_DET_TYPE_EARPHONE:
			type_index = prev_type - 1;
			break;
		case EDC_DET_TYPE_NONE:
		case EDC_DET_TYPE_ANTENNA:
		default:
			type_index = 2;
			break;
		}
	}
	input_report_switch(edc_data->sw_dev, jack_switch_types[type_index], new_state);
	input_sync(edc_data->sw_dev);
	__pm_wakeup_event(edc_data->notify_wake_src, EDC_NOTFY_WL_TIME_MS);

	pr_info("%s: earphone type is %s\n", __func__, 0 == type_index ? "microphone" : (1 == type_index ? "headphone" : "unsupported device"));

	return 0;
}

bool earphone_get_detect_status(struct edc_data *edc_data)
{
	return edc_data->earphone_detect;
}

static int earphone_set_detect_status(struct edc_data *edc_data, bool status)
{
	pr_debug("%s: (%d -> %d)\n", __func__, edc_data->earphone_detect, status);
	edc_data->earphone_detect = status;
	if (!status) {
		earphone_set_type_status(edc_data, EDC_DET_TYPE_NONE);
	}
	return 0;
}

#define EDC_EARPHONE_TYPE_CHECK_COUNT 5
static int earphone_check_type(struct edc_data *edc_data)
{
	int i, j, adc_phys, ear_type;
	int check_types[EDC_DET_TYPE_MAX] = {0};

	ear_type = earphone_get_type_status(edc_data);
	/* Is earphone type changed from previous type ? */
	if (ear_type != EDC_DET_TYPE_NONE) {
		if (!earphone_get_adc(edc_data, EDC_ADC_JMIC_SW_ADC, false, &adc_phys)) {
			adc_phys = adc_phys / 1000;
		} else {
			goto error;
		}
		for (i=0; i<edc_data->edc_type_num; i++) {
			if (edc_data->edc_type_th[i].adc_min <= adc_phys && adc_phys < edc_data->edc_type_th[i].adc_max) {
				if (ear_type == edc_data->edc_type_th[i].type) {
					/* earphone type is not changed */
					return ear_type;
				} else {
					break;
				}
			}
		}
	}

	for (i=0; i<EDC_EARPHONE_TYPE_CHECK_COUNT; i++) {
		if (!earphone_get_adc(edc_data, EDC_ADC_JMIC_SW_ADC, false, &adc_phys)) {
			adc_phys = adc_phys / 1000;
		} else {
			goto error;
		}
		for (j=0; j<edc_data->edc_type_num; j++) {
			if (edc_data->edc_type_th[j].adc_min <= adc_phys && adc_phys < edc_data->edc_type_th[j].adc_max) {
				check_types[j]++;
			}
		}
	}
	if (edc_data->test) {
		earphone_get_adc_raw(edc_data, EDC_ADC_JMIC_SW_ADC, &edc_data->insert_detect.jmic_sw_adc_raw);
	}
	for (i=0; i<edc_data->edc_type_num; i++) {
		if (check_types[i] == EDC_EARPHONE_TYPE_CHECK_COUNT) {
			pr_debug("%s: earphone_type is changed (%d -> %d)\n", __func__, ear_type, edc_data->edc_type_th[i].type);
			return edc_data->edc_type_th[i].type;
		}
	}

error:
	/* Not found earphone type. retry to check soon. */
	pr_err("%s: earphone kind unkown. adc_phy = %d\n", __func__, adc_phys);
	return ear_type;
}

static bool earphone_check_detect(struct edc_data *edc_data)
{
	int i, val, adc_phys;
	int det_st = earphone_get_detect_status(edc_data);

	if (det_st) {
		for (i=0; i<5; i++) {
			EDC_WAIT_MS(10);
			if (!gpiod_get_value(edc_data->detect_gpio)) { // Low
				pr_debug("%s: eject chattering detected\n", __func__);
				/* earphone detect status is not changed. */
				goto check_end;
			}
		}
		pr_info("%s: earphone is ejected \n", __func__);
		mutex_lock(&edc_data->edc_key_info->mute_mutex);
		earphone_mic_mute(false);
		mutex_unlock(&edc_data->edc_key_info->mute_mutex);
		det_st = false;
		earphone_timer_stop(edc_data);
	} else {
		EDC_WAIT_MS(10);
		val = 1;
		for (i=0; i<2; i++) {
			if (i != 0) {
				EDC_WAIT_MS(20);
			}
			if (!gpiod_get_value(edc_data->detect_gpio)) { // Low
				val = 0;
				break;
			}
		}
		if (val) { /* High 2 times */
			goto check_end;
		}

		for (i=0; i<5; i++) {
			if (gpiod_get_value(edc_data->detect_gpio)) { // High
				pr_info("%s: insert chattering detected\n", __func__);
				goto check_end;
			}
			EDC_WAIT_MS(10);
		}

		/* check earphone jack is wet? */
		if (edc_data->th_wet_check) {
			/* switching analog switch and read adc */
			if (!earphone_get_adc(edc_data, EDC_ADC_XHEADSET_DET, false, &adc_phys)) {
				adc_phys = adc_phys / 1000;
			} else {
				goto check_end;
			}
			if (edc_data->test) {
				earphone_get_adc_raw(edc_data, EDC_ADC_XHEADSET_DET, &edc_data->insert_detect.xheadset_det_raw);
			}
			if (edc_data->th_wet_check <= adc_phys) {
				pr_info("%s: earphone jack is wet. phys = %d[mV]\n", __func__, (int)adc_phys);
				goto check_end;
			}
		}
		pr_info("%s: earphone is inserted\n", __func__);
		det_st = true;
		earphone_timer_start(edc_data);
	}
check_end:

	return det_st;
}

static void earphone_check_work(struct work_struct *work)
{
	int det_st, prev_type, ear_type, ret;
	struct delayed_work *dwork = container_of(work, struct delayed_work, work);
	struct edc_data *edc_data  = container_of(dwork, struct edc_data, check_work);

	__pm_stay_awake(edc_data->check_wake_src);
	mutex_lock(&edc_data->check_mutex);

	//wait for modem finishing the current work
	det_st = earphone_get_detect_status(edc_data);
	if( edc_data->voice_call_flag && (!det_st) ) {
		EDC_WAIT_MS(100);
	}

	//Get current earphone state
	det_st = earphone_get_detect_status(edc_data);

	if (!det_st) {
		/* if plug status is off, checking plug detect */
		ret = earphone_pins_set_checking(edc_data);
		if (ret < 0 ) {
			pr_err("%s: earphone_pins_set_checking failed.\n", __func__);
			goto work_end;
		}
		det_st = earphone_check_detect(edc_data);
		if (det_st) {
			earphone_set_detect_status(edc_data, true);
		}
	}

	/* if plug status is on,still checking earphone type */
	if (det_st) { /* plug is enable */
		ret = earphone_pins_set_inserted(edc_data);
		if (ret < 0 ) {
			pr_err("%s: earphone_pins_set_inserted failed.\n", __func__);
			goto work_end;
		}
		if (!(edc_data->mic_bais_on)) {
			ret = edc_headset_micbias_ctrl(1);
			if (ret) {
				pr_err("%s: edc_headset_micbias_ctrl on failed\n", __func__);
				det_st = false;
				earphone_set_detect_status(edc_data, det_st);
				goto work_end;
			}
			edc_data->mic_bais_on = true;
			pr_debug("%s: micbias on\n", __func__);
			EDC_WAIT_MS(40);
		}

		earphone_key_mutex_lock(edc_data);
		if (earphone_key_get_down_keycode(edc_data) != EDC_KEY_NONE) {
			goto check_type_end;
		}
		prev_type = earphone_get_type_status(edc_data);

		ear_type = earphone_check_type(edc_data);
		if (ear_type == EDC_DET_TYPE_EARPHONE_MIC) {
			/* start checking earphone key when earphone type is "earphone mic" */
			earphone_key_check_start(edc_data, 1);
		} else {
			/* stop checking earphone key when earphone type changed to other from "earphone mic" */
			earphone_key_check_start(edc_data, 0);
		}

		if (ear_type == EDC_DET_TYPE_ANTENNA) {
			/* Re-check for antenna */
			ret = earphone_pins_set_checking(edc_data);
			if (ret < 0 ) {
				pr_err("%s: earphone_pins_set_checking failed.\n", __func__);
				goto check_type_end;
			}
			det_st = earphone_check_detect(edc_data);
			if (!det_st) {
				if (edc_data->mic_bais_on) {
					ret = edc_headset_micbias_ctrl(0);
					if (ret) {
						det_st = true;
						pr_err("%s: edc_headset_micbias_ctrl off failed\n", __func__);
						goto check_type_end;
					}
					edc_data->mic_bais_on = false;
					pr_debug("%s: micbias off\n", __func__);
				}
				earphone_set_detect_status(edc_data, false);
				ear_type = EDC_DET_TYPE_NONE;
			}
		}
		earphone_set_type_status(edc_data, ear_type);

		goto check_type_end;

check_type_end:
		earphone_key_mutex_unlock(edc_data);
	}

work_end:
	if (det_st) {
		ret = earphone_pins_set_inserted(edc_data);
		if (ret < 0 ) {
			pr_err("%s: earphone_pins_set_inserted failed.\n", __func__);
		}
	} else {
		ret = earphone_pins_set_default(edc_data);
		if (ret < 0 ) {
			pr_err("%s: earphone_pins_set_default failed.\n", __func__);
		}
	}

	mutex_unlock(&edc_data->check_mutex);
	__pm_relax(edc_data->check_wake_src);
	queue_delayed_work(edc_data->edc_wq, &edc_data->check_work, EDC_POLLING_TIME_JF);
}

static void earphone_status_get(struct edc_data *edc_data, struct earphone_status *edc_read_status)
{
	int adc_code, adc_phys;
	memset(edc_read_status, 0, sizeof(struct earphone_status));
	edc_read_status->plug_status = earphone_get_detect_status(edc_data);
	if (!earphone_get_adc(edc_data, EDC_ADC_XHEADSET_DET, true, &adc_code)) {
		/* adc raw value, do nothing */
		if (adc_code >= EDC_INVALID_ADC_CODE_MIN)
		{
			pr_err("%s: Error ADC values , adc_code = %d\n", __func__, adc_code);
		}
	} else {
		goto error;
	}
	edc_read_status->plug_vol[0] = (char)adc_code;
	edc_read_status->plug_vol[1] = (char)(adc_code >> 8);

	if (earphone_key_get_down_keycode(edc_data) == EDC_KEY_NONE) {
		edc_read_status->sw_status = 0;
	} else {
		edc_read_status->sw_status = 1;
	}
	if (!earphone_get_adc(edc_data, EDC_ADC_JMIC_SW_ADC, true, &adc_code)) {
		/* adc raw value, do nothing */
		if (adc_code >= EDC_INVALID_ADC_CODE_MIN)
		{
			pr_err("%s: Error ADC values , adc_code = %d\n", __func__, adc_code);
		}
	} else {
		goto error;
	}
	edc_read_status->sw_vol[0] = (char)adc_code;
	edc_read_status->sw_vol[1] = (char)(adc_code >> 8);

	if (!earphone_get_adc(edc_data, EDC_ADC_JMIC_SW_ADC, false, &adc_phys)) {
		adc_phys = adc_phys / 1000;
		if (adc_phys >= EDC_INVALID_ADC_PHYS_MV)
		{
			pr_err("%s: Error ADC values , adc_phys = %d[mV]\n", __func__, adc_phys);
		}
	} else {
		goto error;
	}
	edc_read_status->pierce_vol[0] = (char)adc_phys;
	edc_read_status->pierce_vol[1] = (char)(adc_phys >> 8);

	return;

error:
	pr_err("%s: get earphone status falied\n", __func__);
	return;
}

static long earphone_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct edc_data *edc_data = container_of(filp->private_data, struct edc_data, misc_dev);
	int ret;
	struct earphone_status earphone_read_status;

	pr_debug("%s: cmd = %x\n", __func__, cmd);

	switch (cmd) {
	case EARPHONE_VOICE_CALL_START:
		edc_data->voice_call_flag = true;
		break;
	case EARPHONE_VOICE_CALL_STOP:
		edc_data->voice_call_flag = false;
		break;
	case EARPHONE_DET_OUTPUT:
		/* none */
		break;
	case EARPHONE_STATUS_GET:
		earphone_status_get(edc_data, &earphone_read_status);
		ret = copy_to_user((char __user *)arg, &earphone_read_status, sizeof(earphone_read_status));
		break;
	default:
		pr_warn("%s: unknown cmd = %x\n", __func__, cmd);
		break;
	}
	return 0;
}

static long earphone_compat_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	arg = (unsigned long)compat_ptr(arg);
	return earphone_ioctl(filp, cmd, arg);
}

static int earphone_open(struct inode *inode, struct file *filp)
{
	return 0;
}

static int earphone_release(struct inode *inode, struct file *filp)
{
	return 0;
}

static struct file_operations earphone_fops = {
	.unlocked_ioctl = earphone_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl   = earphone_compat_ioctl,
#endif	/* CONFIG_COMPAT */
	.open		   = earphone_open,
	.release		= earphone_release,
};

/**
 * edc platform device
 *
 **/
static int earphone_resume(struct platform_device *pdev)
{
	struct edc_data *edc_data = platform_get_drvdata(pdev);

	pr_debug("%s: det_st = %d, ear_type = %d\n", __func__, earphone_get_detect_status(edc_data), earphone_get_type_status(edc_data));
	if (edc_data->smem_ptr) {
		*(edc_data->smem_ptr) = 0;
	}

	cancel_delayed_work_sync(&edc_data->check_work);
	queue_delayed_work(edc_data->edc_wq, &edc_data->check_work, msecs_to_jiffies(0));

	return 0;
}

static int earphone_suspend(struct platform_device *pdev, pm_message_t st)
{
	struct edc_data *edc_data = platform_get_drvdata(pdev);

	int det_st = earphone_get_detect_status(edc_data);
	if (edc_data->smem_ptr && edc_data->voice_call_flag) {
		if (!det_st)
			*(edc_data->smem_ptr) = 1;
		else
			*(edc_data->smem_ptr) = 2;
	}
	cancel_delayed_work_sync(&edc_data->check_work);
	pr_info("%s: *smem_ptr = %d\n", __func__, edc_data->smem_ptr ? *(edc_data->smem_ptr) : -1);
	pr_debug("%s: det_st = %d, ear_type = %d\n", __func__, earphone_get_detect_status(edc_data), earphone_get_type_status(edc_data));
	return 0;
}

static int earphone_parse_dt(struct edc_data *edc_data)
{
	int i, j, len, ret, tmp;
	const u32 *arr;
	u32 arr_32[(EDC_DET_TYPE_MAX-1) * 3] = {0};

	/* adc channel */
	ret = of_property_match_string(edc_data->dev_np, "io-channel-names", "xheadset_det_voltage");
	if (ret >= 0) {
		edc_data->xheadset_det = iio_channel_get(&edc_data->pdev->dev, "xheadset_det_voltage");
		if (IS_ERR(edc_data->xheadset_det)) {
			ret = PTR_ERR(edc_data->xheadset_det);
			if (ret != -EPROBE_DEFER)
				pr_err("%s: xheadset_det_voltage channel unavailable.\n", __func__);
			else
				pr_warn("%s: xheadset_det_voltage channel probe defer.\n", __func__);
			edc_data->xheadset_det = NULL;
			return ret;
		}
	} else {
		pr_err("%s: no match xheadset_det_voltage in io-channel-names.\n", __func__);
		return ret;
	}

	ret = of_property_match_string(edc_data->dev_np, "io-channel-names", "jmic_sw_voltage");
	if (ret >= 0) {
		edc_data->jmic_sw_adc = iio_channel_get(&edc_data->pdev->dev, "jmic_sw_voltage");
		if (IS_ERR(edc_data->jmic_sw_adc)) {
			ret = PTR_ERR(edc_data->jmic_sw_adc);
			if (ret != -EPROBE_DEFER)
				pr_err("%s: jmic_sw_voltage channel unavailable.\n", __func__);
			else
				pr_warn("%s: jmic_sw_voltage channel probe defer.\n", __func__);
			edc_data->jmic_sw_adc = NULL;
			return ret;
		}
	} else {
		pr_err("%s: no match jmic_sw_voltage in io-channel-names.\n", __func__);
		return ret;
	}

	/* earphone jack wet adc thresthold */
	ret = of_property_read_u32(edc_data->dev_np, "fj-edc,wet_threshold", &tmp);
	if (ret) {
		pr_err("%s: of_property_read_u32 failed.\n", __func__);
		edc_data->th_wet_check	= 0;
	} else {
		edc_data->th_wet_check	= tmp;
	}

	/* earphone jack type (earphone, mic, antenna) thresthold */
	arr = of_get_property(edc_data->dev_np, "fj-edc,edc_type_threshold", &len);
	if (IS_ERR_OR_NULL(arr)) {
		pr_err("%s: of_get_property failed.\n", __func__);
		return -1;
	}
	len = len / sizeof(u32);
	if ((len%3) != 0) {
		pr_err("%s: syntax error for fj-edc,edc_type_threshold.\n", __func__);
		return -1;
	}

	ret = of_property_read_u32_array(edc_data->dev_np, "fj-edc,edc_type_threshold", arr_32, len);
	if (ret) {
		pr_err("%s: of_property_read_u32_array failed.\n", __func__);
		return -1;
	}
	j = 0;
	for (i=0; i<len; i=i+3) {
		if (EDC_DET_TYPE_NONE < arr_32[i] && arr_32[i] < EDC_DET_TYPE_MAX) {
			edc_data->edc_type_th[j].type	= arr_32[i];
			edc_data->edc_type_th[j].adc_min = arr_32[i+1];
			edc_data->edc_type_th[j].adc_max = arr_32[i+2];
			pr_info("%s: detect type : idx = %d, type = %d, min = %d[mV], max = %d[mV]\n", __func__, j,
			        edc_data->edc_type_th[j].type,
			        edc_data->edc_type_th[j].adc_min,
			        edc_data->edc_type_th[j].adc_max);
			j++;
		}
	}
	edc_data->edc_type_num = j;
	return 0;
}

static int earphone_pinctrl_init(struct platform_device *pdev, struct edc_data *edc_data)
{
	/* get pinctrl */
	edc_data->pinctrl = devm_pinctrl_get(&pdev->dev);
	if (IS_ERR_OR_NULL(edc_data->pinctrl)) {
		pr_err("%s: get pinctrl failed.\n", __func__);
		return -1;
	}

	/* get edc detect default state*/
	edc_data->pin_state_info.detect_default = pinctrl_lookup_state(
	            edc_data->pinctrl,
	            "fj_edc_detect_default");
	if (IS_ERR_OR_NULL(edc_data->pin_state_info.detect_default)) {
		pr_err("%s: get fj_edc_detect_default failed.\n", __func__);
		return -1;
	}

	/* get edc detect checking state*/
	edc_data->pin_state_info.detect_checking = pinctrl_lookup_state(
	            edc_data->pinctrl,
	            "fj_edc_detect_checking");
	if (IS_ERR_OR_NULL(edc_data->pin_state_info.detect_checking)) {
		pr_err("%s: get detect_checking failed.\n", __func__);
		return -1;
	}

	/* get edc detect inserted state*/
	edc_data->pin_state_info.detect_inserted = pinctrl_lookup_state(
	            edc_data->pinctrl,
	            "fj_edc_detect_inserted");
	if (IS_ERR_OR_NULL(edc_data->pin_state_info.detect_inserted)) {
		pr_err("%s: get detect_inserted failed.\n", __func__);
		return -1;
	}

	return 0;
}

static int earphone_pins_set_default(struct edc_data *edc_data)
{
	int ret = -1;

	/* select detect as default state */
	ret = pinctrl_select_state(
	          edc_data->pinctrl,
	          edc_data->pin_state_info.detect_default);
	if (ret < 0 ) {
		pr_err("%s: set detect_default failed.\n", __func__);
		return -1;
	}

	return 0;
}

static int earphone_pins_set_checking(struct edc_data *edc_data)
{
	int ret = -1;

	/* select detect as checking state */
	ret = pinctrl_select_state(
	          edc_data->pinctrl,
	          edc_data->pin_state_info.detect_checking);
	if (ret < 0 ) {
		pr_err("%s: set detect_checking failed.\n", __func__);
		return -1;
	}

	return 0;
}


static int earphone_pins_set_inserted(struct edc_data *edc_data)
{
	int ret = -1;

	/* select detect as inserted state */
	ret = pinctrl_select_state(
	          edc_data->pinctrl,
	          edc_data->pin_state_info.detect_inserted);
	if (ret < 0 ) {
		pr_err("%s: set detect_inserted failed.\n", __func__);
		return -1;
	}

	return 0;
}

static int earphone_gpio_init(struct platform_device *pdev, struct edc_data *edc_data)
{
	/* detect gpiod init */
	edc_data->detect_gpio = devm_gpiod_get_optional(&pdev->dev, "fj-edc,edc_detect", GPIOD_OUT_LOW);
	if (IS_ERR_OR_NULL(edc_data->detect_gpio)) {
		pr_err("%s: devm_gpiod_get_optional failed.\n", __func__);
		edc_data->detect_gpio = NULL;
		return -1;
	}

	return 0;
}

static void earphone_resource_release(struct platform_device *pdev, struct edc_data *edc_data)
{
	if (!edc_data) {
		return;
	}

	debugfs_remove(edc_data->root);

	earphone_timer_stop(edc_data);

	if (edc_data->edc_wq) {
		cancel_delayed_work_sync(&edc_data->check_work);
		destroy_workqueue(edc_data->edc_wq);
	}
	mutex_destroy(&edc_data->check_mutex);
	wakeup_source_unregister(edc_data->check_wake_src);
	wakeup_source_unregister(edc_data->notify_wake_src);

	if (NULL != edc_data->detect_gpio) {
		gpiod_put(edc_data->detect_gpio);
	}

	if (edc_data->sw_dev) {
		input_unregister_device(edc_data->sw_dev);
	}

	if (edc_data->misc_dev.minor) {
		misc_deregister(&edc_data->misc_dev);
	}
	earphone_key_deinit(edc_data);

	if (!IS_ERR_OR_NULL(edc_data->xheadset_det))
		iio_channel_release(edc_data->xheadset_det);
	if (!IS_ERR_OR_NULL(edc_data->jmic_sw_adc))
		iio_channel_release(edc_data->jmic_sw_adc);

	kfree(edc_data);
	platform_set_drvdata(pdev, NULL);
	edc_data = NULL;
	return;
}

static void earphone_timer_cb(struct timer_list *t)
{
	struct edc_data *edc_data = from_timer(edc_data, t, earphone_timer);
	pr_info("%s: timer expire \n", __func__);

	/*  unset flag */
	edc_data->ignore_key_flag = false;
}

static int earphone_timer_init(struct edc_data *edc_data)
{
	pr_info("%s: timer init \n", __func__);
	/* initialize the timer */
	timer_setup(&edc_data->earphone_timer, earphone_timer_cb, 0);

	/* unset flag */
	edc_data->ignore_key_flag = false;
	return 0;
}

static int earphone_timer_start(struct edc_data *edc_data)
{
	pr_info("%s: timer start \n", __func__);
	/* set flag */
	edc_data->ignore_key_flag = true;

	if(timer_pending(&edc_data->earphone_timer)){
		pr_info("%s: pending timer exist. stop \n", __func__);
		del_timer(&edc_data->earphone_timer);
	}

	/* start the timer */
	edc_data->earphone_timer.expires = jiffies + msecs_to_jiffies(1000);
	add_timer(&edc_data->earphone_timer);
	return 0;
}

static int earphone_timer_stop(struct edc_data *edc_data)
{
	pr_info("%s: timer stop \n", __func__);
	del_timer_sync(&edc_data->earphone_timer);

	/* unset flag */
	edc_data->ignore_key_flag = false;
	return 0;
}

#define STRING_MAX_LEN 10

static ssize_t earphone_debugfs_output_decimal(
	 char __user *user_buf, size_t user_len, loff_t *ppos, int value)
{
	char *buf;
	u32 len = 0;
	size_t max_len = min_t(size_t, user_len, STRING_MAX_LEN);

	if (*ppos)
		return 0;

	buf = kzalloc(max_len, GFP_KERNEL);
	if (ZERO_OR_NULL_PTR(buf))
		return -ENOMEM;

	len = snprintf(buf, max_len, "%d\n", value);

	if (copy_to_user(user_buf, buf, len)) {
		kfree(buf);
		return -EFAULT;
	}

	*ppos += len;

	kfree(buf);
	return len;
}

static ssize_t earphone_debugfs_output_hexadecimal(
	 char __user *user_buf, size_t user_len, loff_t *ppos, int value)
{
	char *buf;
	u32 len = 0;
	size_t max_len = min_t(size_t, user_len, STRING_MAX_LEN);

	if (*ppos)
		return 0;

	buf = kzalloc(max_len, GFP_KERNEL);
	if (ZERO_OR_NULL_PTR(buf))
		return -ENOMEM;

	len = snprintf(buf, max_len, "0x%04X\n", (uint16_t)value);

	if (copy_to_user(user_buf, buf, len)) {
		kfree(buf);
		return -EFAULT;
	}

	*ppos += len;

	kfree(buf);
	return len;
}

static ssize_t earphone_debugfs_adc_read(struct file *file, enum edc_adc_type type, bool raw, char __user *user_buf, size_t user_len, loff_t *ppos)
{
	int ret, value = 0;
	struct edc_data *edc_data = file->private_data;

	mutex_lock(&edc_data->check_mutex);

	if (type == EDC_ADC_XHEADSET_DET) {
		ret = earphone_pins_set_checking(edc_data);
		if (ret < 0) {
			pr_err("%s: earphone_pins_set_checking failed.\n", __func__);
			goto error;
		}
	} else {
		ret = earphone_pins_set_inserted(edc_data);
		if (ret < 0) {
			pr_err("%s: earphone_pins_set_inserted failed.\n", __func__);
			goto error;
		}
		if (!(edc_data->mic_bais_on)) {
			ret = edc_headset_micbias_ctrl(1);
			if (ret) {
				pr_err("%s: edc_headset_micbias_ctrl on failed\n", __func__);
			}
			EDC_WAIT_MS(40);
		}
	}

	EDC_WAIT_MS(10);

	earphone_get_adc(edc_data, type, raw, &value);

	if (type == EDC_ADC_JMIC_SW_ADC) {
		if (!(edc_data->mic_bais_on)) {
			ret = edc_headset_micbias_ctrl(0);
			if (ret) {
				pr_err("%s: edc_headset_micbias_ctrl off failed\n", __func__);
			}
			EDC_WAIT_MS(40);
		}
	}
	if (earphone_get_detect_status(edc_data)) {
		ret = earphone_pins_set_inserted(edc_data);
		if (ret < 0 ) {
			pr_err("%s: earphone_pins_set_inserted failed.\n", __func__);
		}
	} else {
		ret = earphone_pins_set_default(edc_data);
		if (ret < 0 ) {
			pr_err("%s: earphone_pins_set_default failed.\n", __func__);
		}
	}

error:
	mutex_unlock(&edc_data->check_mutex);

	if (raw) {
		return earphone_debugfs_output_hexadecimal(user_buf, user_len, ppos, value);
	} else {
		return earphone_debugfs_output_decimal(user_buf, user_len, ppos, value);
	}
}

static ssize_t earphone_debugfs_xheadset_det_input_read(struct file *file,
	 char __user *user_buf, size_t user_len, loff_t *ppos)
{
	return earphone_debugfs_adc_read(file, EDC_ADC_XHEADSET_DET, false, user_buf, user_len, ppos);
}

static ssize_t earphone_debugfs_xheadset_det_raw_read(struct file *file,
	 char __user *user_buf, size_t user_len, loff_t *ppos)
{
	return earphone_debugfs_adc_read(file, EDC_ADC_XHEADSET_DET, true, user_buf, user_len, ppos);
}

static ssize_t earphone_debugfs_jmic_sw_adc_input_read(struct file *file,
	 char __user *user_buf, size_t user_len, loff_t *ppos)
{
	return earphone_debugfs_adc_read(file, EDC_ADC_JMIC_SW_ADC, false, user_buf, user_len, ppos);
}

static ssize_t earphone_debugfs_jmic_sw_adc_raw_read(struct file *file,
	 char __user *user_buf, size_t user_len, loff_t *ppos)
{
	return earphone_debugfs_adc_read(file, EDC_ADC_JMIC_SW_ADC, true, user_buf, user_len, ppos);
}

static const struct file_operations xheadset_det_input_fops = {
	.open = simple_open,
	.read = earphone_debugfs_xheadset_det_input_read,
};

static const struct file_operations xheadset_det_raw_fops = {
	.open = simple_open,
	.read = earphone_debugfs_xheadset_det_raw_read,
};

static const struct file_operations jmic_sw_adc_input_fops = {
	.open = simple_open,
	.read = earphone_debugfs_jmic_sw_adc_input_read,
};

static const struct file_operations jmic_sw_adc_raw_fops = {
	.open = simple_open,
	.read = earphone_debugfs_jmic_sw_adc_raw_read,
};

static ssize_t earphone_debugfs_detect_xheadset_det_raw_read(
	struct file *file, char __user *user_buf, size_t user_len, loff_t *ppos)
{
	struct adc_raws *raws = file->private_data;
	return earphone_debugfs_output_hexadecimal(user_buf, user_len, ppos, raws->xheadset_det_raw);
}

static ssize_t earphone_debugfs_detect_jmic_sw_adc_raw_read(
	struct file *file, char __user *user_buf, size_t user_len, loff_t *ppos)
{
	struct adc_raws *raws = file->private_data;
	return earphone_debugfs_output_hexadecimal(user_buf, user_len, ppos, raws->jmic_sw_adc_raw);
}

static const struct file_operations detect_xheadset_det_raw_fops = {
	.open = simple_open,
	.read = earphone_debugfs_detect_xheadset_det_raw_read,
};

static const struct file_operations detect_jmic_sw_adc_raw_fops = {
	.open = simple_open,
	.read = earphone_debugfs_detect_jmic_sw_adc_raw_read,
};

static int earphone_init_debugfs(struct edc_data *edc_data)
{
	int ret = 0;
	struct dentry *dir, *file;

	dir = debugfs_create_dir("earphone", NULL);
	if (IS_ERR_OR_NULL(dir)) {
		ret = PTR_ERR(dir);
		pr_err("%s: debugfs_create_dir failed, ret = %d\n", __func__, ret);
		goto error;
	}

	file = debugfs_create_file("xheadset_det_input",
					0400, dir, edc_data, &xheadset_det_input_fops);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create xheadset_det_input failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	file = debugfs_create_file("xheadset_det_raw",
					0400, dir, edc_data, &xheadset_det_raw_fops);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create xheadset_det_raw failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	file = debugfs_create_file("jmic_sw_adc_input",
					0400, dir, edc_data, &jmic_sw_adc_input_fops);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create jmic_sw_adc_input failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	file = debugfs_create_file("jmic_sw_adc_raw",
					0400, dir, edc_data, &jmic_sw_adc_raw_fops);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create jmic_sw_adc_raw failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	file = debugfs_create_bool("test",
					0600, dir, &edc_data->test);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create test failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	file = debugfs_create_file("insert_detect_xheadset_det_raw",
					0400, dir, &edc_data->insert_detect,
					&detect_xheadset_det_raw_fops);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create insert_detect_xheadset_det_raw failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	file = debugfs_create_file("insert_detect_jmic_sw_adc_raw",
					0400, dir, &edc_data->insert_detect,
					&detect_jmic_sw_adc_raw_fops);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create insert_detect_jmic_sw_adc_raw failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	file = debugfs_create_file("key_press_detect_xheadset_det_raw",
					0400, dir, &edc_data->key_press_detect,
					&detect_xheadset_det_raw_fops);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create key_press_detect_xheadset_det_raw failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	file = debugfs_create_file("key_press_detect_jmic_sw_adc_raw",
					0400, dir, &edc_data->key_press_detect,
					&detect_jmic_sw_adc_raw_fops);
	if (IS_ERR_OR_NULL(file)) {
		ret = PTR_ERR(file);
		pr_err("%s: debugfs create key_press_detect_jmic_sw_adc_raw failed, ret = %d\n",
				__func__, ret);
		goto error_remove_dir;
	}

	edc_data->root = dir;
	return ret;

error_remove_dir:
	debugfs_remove(dir);
error:
	return ret;
}

static int earphone_probe(struct platform_device *pdev)
{
	struct edc_data *edc_data;
	int i,ret;

	edc_data = kzalloc(sizeof(struct edc_data),  GFP_KERNEL);
	if (!edc_data) {
		pr_err("%s: allocate edc_data failed.\n", __func__);
		return -1;
	}

	platform_set_drvdata(pdev, edc_data);

	edc_data->name			= "edc";
	edc_data->sw_name		= "sw_edc";
	edc_data->earphone_detect	= false;
	edc_data->mic_bais_on		= false;
	edc_data->earphone_type		= EDC_DET_TYPE_NONE;
	edc_data->detect_gpio		= NULL;
	edc_data->pdev			= pdev;
	edc_data->dev_np		= pdev->dev.of_node;
	edc_data->check_wake_src	= wakeup_source_register(&pdev->dev, "edc_wl");
	edc_data->notify_wake_src	= wakeup_source_register(&pdev->dev, "edc_ntfy_wl");
	edc_data->ignore_key_flag = false;

	mutex_init(&edc_data->check_mutex);

	ret = earphone_parse_dt(edc_data);
	if (ret) {
		pr_err("%s: earphone_parse_dt failed.\n", __func__);
		goto error;
	}

	/* earphone detect pinctrl init */
	ret = earphone_pinctrl_init(pdev,edc_data);
	if (ret) {
		pr_err("%s: earphone_pinctrl_init failed.\n", __func__);
		goto error;
	}

	/* earphone detect_gpio setting init */
	ret = earphone_gpio_init(pdev,edc_data);
	if (ret) {
		pr_err("%s: earphone_gpio_init failed.\n", __func__);
		goto error;
	}

	/* earphone detect and check earphone kind workqueue */
	edc_data->edc_wq = alloc_workqueue("edc_wq", WQ_UNBOUND | WQ_MEM_RECLAIM, 0);
	if (!edc_data->edc_wq) {
		pr_err("%s: allocate edc_wq failed.\n", __func__);
		ret = -1;
		goto error;
	}
	INIT_DELAYED_WORK(&edc_data->check_work, earphone_check_work);

	/* register inputdev */
	edc_data->sw_dev = input_allocate_device();
	if (!edc_data->sw_dev) {
		pr_err("%s: input_allocate_device failed.\n", __func__);
		ret = -1;
		goto error;
	}

	set_bit(EV_SW, edc_data->sw_dev->evbit);
	edc_data->sw_dev->name        = edc_data->sw_name;

	for (i = 0; i < sizeof(jack_switch_types) / sizeof(int); i++) {
		input_set_capability(edc_data->sw_dev, EV_SW, jack_switch_types[i]);
	}

	ret = input_register_device(edc_data->sw_dev);
	if (ret) {
		input_free_device(edc_data->sw_dev);
		edc_data->sw_dev = NULL;
		pr_err("%s: input_register_device failed.\n", __func__);
		goto error;
	}

	/* alloc smem pointer for voice call */
	edc_data->smem_ptr = smem_alloc_vendor0(SMEM_OEM_V0_010);
	if (!edc_data->smem_ptr) {
		pr_warn("%s: alloc smem failed.\n", __func__);
	}

	edc_data->misc_dev.minor = MISC_DYNAMIC_MINOR;
	edc_data->misc_dev.name  = edc_data->name;
	edc_data->misc_dev.fops  = &earphone_fops;
	ret = misc_register(&edc_data->misc_dev);
	if (ret) {
		edc_data->misc_dev.minor = 0;
		pr_err("%s: misc_register failed.\n", __func__);
		goto error;
	}

	ret = earphone_key_init(edc_data);
	if (ret) {
		pr_err("%s: earphone_key_init failed.\n", __func__);
		goto error;
	}
	earphone_timer_init(edc_data);

	earphone_init_debugfs(edc_data);

	queue_delayed_work(edc_data->edc_wq, &edc_data->check_work, msecs_to_jiffies(5000));
	return 0;
error:
	earphone_resource_release(pdev, edc_data);
	return ret;
}

static int earphone_remove(struct platform_device *pdev)
{
	struct edc_data *edc_data = platform_get_drvdata(pdev);
	earphone_resource_release(pdev, edc_data);
	return 0;
}

static void earphone_shutdown(struct platform_device *pdev)
{
	earphone_remove(pdev);
}

static const struct of_device_id edc_dt_match[] = {
	{.compatible = "fj-edc",},
	{},
};
MODULE_DEVICE_TABLE(of, edc_dt_match);

static struct platform_driver platform_edc_driver = {
	.driver = {
		.name = "earphone-driver",
		.owner = THIS_MODULE,
		.of_match_table = edc_dt_match,
	},
	.probe	= earphone_probe,
	.remove   = earphone_remove,
	.shutdown = earphone_shutdown,
	.suspend  = earphone_suspend,
	.resume   = earphone_resume,
};

static __init int earphone_init(void)
{
	return platform_driver_register(&platform_edc_driver);
}
module_init(earphone_init);

static __exit void earphone_exit(void)
{
	return platform_driver_unregister(&platform_edc_driver);
}
module_exit(earphone_exit);

MODULE_DESCRIPTION("Earphone Device Driver");
MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("Fujitsu Connected Technologies");
