/*
  * (C) 2021 FCNT LIMITED
  *
  * This program is free software; you can redistribute it and/or
  * modify it under the terms of the GNU General Public License
  * as published by the Free Software Foundation; version 2
  * of the License.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  * GNU General Public License for more details.
  *
  * You should have received a copy of the GNU General Public License
  * along with this program; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef __EDC_H__
#define __EDC_H__

#include <linux/power_supply.h>
#include <linux/timer.h>
#include <linux/iio/consumer.h>
#include <linux/debugfs.h>

/* earphone detect type */
enum edc_detect_type {
	EDC_DET_TYPE_NONE            = 0,
	EDC_DET_TYPE_EARPHONE_MIC    = 1,
	EDC_DET_TYPE_EARPHONE        = 2,
	EDC_DET_TYPE_ANTENNA         = 3,
	EDC_DET_TYPE_MAX,
};

/* earphone adc type */
enum edc_adc_type {
	EDC_ADC_XHEADSET_DET,
	EDC_ADC_JMIC_SW_ADC,
};

struct edc_detect_threshold {
	enum edc_detect_type    type;
	unsigned int            adc_min;
	unsigned int            adc_max;
};

#define EDC_SUPPORT_KEY_MAX  4    /* KEYCODE_HEADSETHOOK, KEYCODE_VOLUME_UP, KEYCODE_VOLUME_DOWN, KEYCODE_VOICE_ASSIST */
#define EDC_KEY_NONE         (-1) /* KEY is UP */
#define EDC_INVALID_ADC_CODE_MIN	0XB4A0
#define EDC_INVALID_ADC_PHYS_MV		2550

#define EDC_WAIT_MS(n) {                       \
	if (n >= 20) {                             \
		msleep(n);                             \
	} else {                                   \
		udelay((unsigned long)(n * 1000));     \
	}                                          \
}

struct edc_key_threshold {
	int                             keycode;
	unsigned int                    adc_min;
	unsigned int                    adc_max;
};

struct edc_key_info {
	int                             down_key;
	struct gpio_desc *              key_gpio;
	int                             key_gpio_irq;
	struct mutex                    jmic_mutex;
	struct edc_key_threshold        edc_key_th[EDC_SUPPORT_KEY_MAX];
	int                             edc_key_num;
	struct wakeup_source            *notify_wake_src;
	struct input_dev *              indev;
	void *                          edc_data_ptr;
	struct workqueue_struct         *key_wq;
	struct delayed_work             check_key_work;
	struct workqueue_struct         *key_wq2;
	struct delayed_work             check_key_work2;
	struct workqueue_struct         *unmute_wq;
	struct delayed_work             unmute_work;
	bool                            is_press;
	struct mutex                    mute_mutex;
	bool                            is_irq_enable;
};

struct pinctrl_state_info {
	struct pinctrl_state *detect_default;
	struct pinctrl_state *detect_checking;
	struct pinctrl_state *detect_inserted;
};

struct adc_raws{
	int16_t                         xheadset_det_raw;
	int16_t                         jmic_sw_adc_raw;
};

struct edc_data {
	char *                          name;
	char *                          sw_name;
	struct platform_device *        pdev;
	struct device_node *            dev_np;
	bool                            earphone_detect;
	enum edc_detect_type            earphone_type;
	struct workqueue_struct         *edc_wq;
	bool                            mic_bais_on;
	struct gpio_desc *              detect_gpio;
	struct pinctrl *                pinctrl;
	struct pinctrl_state_info       pin_state_info;
	struct input_dev *              sw_dev;
	int                             th_wet_check;
	struct delayed_work             check_work;
	struct mutex                    check_mutex;
	struct wakeup_source            *check_wake_src;
	struct wakeup_source            *notify_wake_src;
	struct edc_detect_threshold     edc_type_th[EDC_DET_TYPE_MAX];
	int                             edc_type_num;
	struct miscdevice               misc_dev;
	bool                            voice_call_flag;
	unsigned int *                  smem_ptr;
	struct edc_key_info *           edc_key_info;
	bool                            ignore_key_flag;
	struct timer_list               earphone_timer;
	struct iio_channel              *xheadset_det;
	struct iio_channel              *jmic_sw_adc;
	struct dentry                   *root;
	bool                            test;
	struct adc_raws                 insert_detect;
	struct adc_raws                 key_press_detect;
};

/**
 *  earphone_get_adc - read adc value.
 *  @edc_data : earphone driver data
 *  @type     : ADC type.
 *  @adc_intval : the pointer to return the adc value,
 *                when raw is true, return adc raw value,
 *                when raw is false, return adc uv value.
 *
 *  return    :0 if success
 *             -1 if falied
 **/
int earphone_get_adc(struct edc_data *edc_data, enum edc_adc_type type, bool raw, int *adc_intval);

/**
 *  earphone_get_adc_raw - read adc raw value.
 *  @edc_data : earphone driver data
 *  @type     : ADC type.
 *  @out      : output raw data.
 **/
void earphone_get_adc_raw(struct edc_data *edc_data, enum edc_adc_type type, uint16_t *out);

/**
 *  earphone_get_type_status - get earphone type (earphone, mic, antenna,,)
 *  @edc_data : earphone driver data
 *  return    : enum edc_detect_type
 **/
enum edc_detect_type earphone_get_type_status(struct edc_data *edc_data);

/**
 *  earphone_get_detect_status - get earphone detect or not.
 *  @edc_data : earphone driver data
 *  return    : 0: Not detect / 1: detect
 **/
bool earphone_get_detect_status(struct edc_data *edc_data);

/**
 *  earphone_key_init - initilize earphone key resource
 *  @edc_data : earphone driver data
 *  return    : 0 - success / else - fail
 **/
int earphone_key_init(struct edc_data *edc_data);

/**
 *  earphone_key_deinit - release earphone key resource
 *  @edc_data : earphone driver data
 *  return    : 0 - success / else - fail
 **/
int earphone_key_deinit(struct edc_data *edc_data);

/**
 *  earphone_key_check_start - start or stop the checking earphone key.
 *  @edc_data : earphone driver data
 *  @start    : 1 - start / 0 - stop
 *  return    : 0 - success / else - fail
 **/
int earphone_key_check_start(struct edc_data *edc_data, bool start);

/**
 *  earphone_key_get_down_keycode - get keycode
 *  @edc_data : earphone driver data
 *  return    : EDC_KEY_NONE - any key is not down
 *              else         - keycode
 **/
int earphone_key_get_down_keycode(struct edc_data *edc_data);

/**
 *  earphone_key_mutex_lock/earphone_key_mutex_unlock
 *              - mutex of checking earphone kind and key.
 *  @edc_data : earphone driver data
 *  return    : 0
 *
 *  Same adc is used by both checking earphone kind and checking key.
 *  Then mutex is needed.
 **/
void earphone_key_mutex_lock(struct edc_data *edc_data);
void earphone_key_mutex_unlock(struct edc_data *edc_data);

/**
 *  earphone_mic_mute
 *              - earphone mic mute / unmute.
 *  @mute     : true - mute / false - unmute
 *  return    : 0
 **/
void earphone_mic_mute(bool mute);

#endif /* __EDC_H__ */
