/*
 * (C) 2021 FCNT LIMITED
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/version.h>
#include <linux/interrupt.h>
#include <linux/extcon.h>
#include <linux/pm_wakeup.h>
#include <linux/irq.h>
#include <linux/of_gpio.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <linux/miscdevice.h>
#include <linux/input.h>
#include <linux/earphone.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/gpio/consumer.h>
#include "edc.h"

#define EDC_KEY_NOTFY_WL_TIME_MS        500
#define EDC_KEY_NOTFY_WL_TIME_JF        msecs_to_jiffies(EDC_KEY_NOTFY_WL_TIME_MS)

#define EDC_MIC_MUTE_ENABLE             0

static int (*earphone_mic_mute_func)(bool) = NULL;

void earphone_mic_mute_unregister(void)
{
	pr_info("%s: called.\n", __func__);
	earphone_mic_mute_func = NULL;
}
EXPORT_SYMBOL_GPL(earphone_mic_mute_unregister);

void earphone_mic_mute_register(void* func)
{
	pr_info("%s: called.\n", __func__);
	earphone_mic_mute_func = func;
}
EXPORT_SYMBOL_GPL(earphone_mic_mute_register);

/* audio driver function */
static int edc_headset_mic_mute(bool on)
{
#if EDC_MIC_MUTE_ENABLE
	pr_info("%s: called.\n", __func__);
	if (!IS_ERR_OR_NULL(earphone_mic_mute_func)) {
			return earphone_mic_mute_func(on);
	}
	return -1;
#else
	return 0;
#endif
}

void earphone_mic_mute(bool mute)
{
	if (mute) {
		if (edc_headset_mic_mute(true) < 0) {
			pr_err("%s: mute failed\n", __func__);
		}
	} else {
		if (edc_headset_mic_mute(false) < 0) {
			pr_err("%s: unmute failed\n", __func__);
		}
	}
}

static void earphone_unmute_work(struct work_struct *work)
{
	struct delayed_work *dwork = container_of(work, struct delayed_work, work);
	struct edc_key_info *edc_key_info  = container_of(dwork, struct edc_key_info, unmute_work);

	mutex_lock(&edc_key_info->mute_mutex);

	pr_info("%s: %d\n", __func__, edc_key_info->is_press);

	if (!edc_key_info->is_press) {
		earphone_mic_mute(false);
	}

	mutex_unlock(&edc_key_info->mute_mutex);

	return;
}

static int earphone_key_mute(struct edc_data *edc_data)
{
	mutex_lock(&edc_data->edc_key_info->mute_mutex);

	cancel_delayed_work(&edc_data->edc_key_info->unmute_work);
	edc_data->edc_key_info->is_press = true;
	earphone_mic_mute(true);

	mutex_unlock(&edc_data->edc_key_info->mute_mutex);

	return 0;
}

static int earphone_key_unmute(struct edc_data *edc_data)
{
	mutex_lock(&edc_data->edc_key_info->mute_mutex);

	edc_data->edc_key_info->is_press = false;
	mod_delayed_work(edc_data->edc_key_info->unmute_wq, &edc_data->edc_key_info->unmute_work, msecs_to_jiffies(600));

	mutex_unlock(&edc_data->edc_key_info->mute_mutex);

	return 0;
}

int earphone_key_get_down_keycode(struct edc_data *edc_data)
{
	int key;
	key = edc_data->edc_key_info->down_key;
	return key;
}

static void earphone_key_down_report(struct edc_data *edc_data, int key_code)
{
	pr_info("%s: switch key %d down\n", __func__, key_code);
	earphone_key_mute(edc_data);
	input_report_key(edc_data->edc_key_info->indev, key_code, 1);
	input_sync(edc_data->edc_key_info->indev);
	__pm_wakeup_event(edc_data->edc_key_info->notify_wake_src, EDC_KEY_NOTFY_WL_TIME_MS);
}

static void earphone_key_up_report(struct edc_data *edc_data, int key_code)
{
	pr_info("%s: switch key %d up\n", __func__, key_code);
	earphone_key_unmute(edc_data);
	input_report_key(edc_data->edc_key_info->indev, key_code, 0);
	input_sync(edc_data->edc_key_info->indev);
	__pm_wakeup_event(edc_data->edc_key_info->notify_wake_src, EDC_KEY_NOTFY_WL_TIME_MS);
}

static bool earphone_key_set_down_key_voice_call(struct edc_data *edc_data, int key_code)
{
	int i = 0, counter = 0, adc_phys, gpio_val;
	struct edc_key_info *edc_key_info = edc_data->edc_key_info;
	bool cancel_wq2 = false;

	disable_irq_nosync(edc_data->edc_key_info->key_gpio_irq);
	/* check key really down and microphone exist */
	for (i = 0, counter = 0;i < 39;i++) {
		EDC_WAIT_MS(8);
		if (!earphone_get_adc(edc_data, EDC_ADC_JMIC_SW_ADC, false, &adc_phys)) {
			adc_phys = adc_phys / 1000;
			if (adc_phys > 1650) {
				counter++;
			} else {
				counter = 0;
			}
		}
	}
	if (counter > 5) {
		/* microphone is ejected */
		edc_key_info->down_key = EDC_KEY_NONE;
		goto end;
	}
	earphone_key_down_report(edc_data, key_code);

	/* check key up */
	for (i = 0,counter = 0;i < 20;i++) {
		EDC_WAIT_MS(20);
		gpio_val = !!gpiod_get_value(edc_key_info->key_gpio);
		if (!gpio_val) {
			counter++;
		} else {
			counter = 0;
		}
		if (counter >= 3) {
			/* check key up 3 times */
			cancel_wq2 = true;
			earphone_key_up_report(edc_data, key_code);
			edc_key_info->down_key = EDC_KEY_NONE;
			break;
		}
	}
	if (counter > 0 && counter < 3) {
		queue_delayed_work(edc_data->edc_key_info->key_wq2, &edc_data->edc_key_info->check_key_work2, 0);
	} else if (counter == 0) {
		queue_delayed_work(edc_data->edc_key_info->key_wq2, &edc_data->edc_key_info->check_key_work2, msecs_to_jiffies(500));
	}

end:
	enable_irq(edc_data->edc_key_info->key_gpio_irq);
	return cancel_wq2;
}

static bool earphone_key_set_down_keycode(struct edc_data *edc_data, int new_key)
{
	struct edc_key_info *edc_key_info = edc_data->edc_key_info;
	int prev_key = earphone_key_get_down_keycode(edc_data);
	bool cancel_wq2 = false;

	if (prev_key == new_key) {
		return cancel_wq2;
	}
	edc_key_info->down_key = new_key;

	if (new_key != EDC_KEY_NONE) {
		if (prev_key != EDC_KEY_NONE) {
			cancel_wq2 = true;
			pr_warn("%s: switch key %d up (up irq does not occur)\n", __func__, prev_key);
			earphone_key_up_report(edc_data, prev_key);
		}

		if (KEY_MEDIA == new_key && edc_data->voice_call_flag) {
			cancel_wq2 = earphone_key_set_down_key_voice_call(edc_data, new_key);
		} else {
			earphone_key_down_report(edc_data, new_key);
		}
	} else {
		cancel_wq2 = true;
		earphone_key_up_report(edc_data, prev_key);
	}

	return cancel_wq2;
}

/**
*  1:gpio hi,key down(press) state
*  0:gpio low,key up(release) state
*  -1: error
*/
static int earphone_read_key_gpio_value(struct edc_key_info *edc_key_info)
{
	int gpio_val, i;

	gpio_val = !!gpiod_get_value(edc_key_info->key_gpio);
	for (i=0; i < 3; i++) {
		EDC_WAIT_MS(2);
		if (gpio_val !=  (!!gpiod_get_value(edc_key_info->key_gpio))) {
			pr_debug("%s: gpio chattering detected\n", __func__);
			return -1;
		}
	}
	return gpio_val;
}

static int earphone_key_check(struct edc_data *edc_data, int key_down_status)
{
	int i, prev_key, new_key, adc_phys;
	struct edc_key_info *edc_key_info = edc_data->edc_key_info;

	prev_key = earphone_key_get_down_keycode(edc_data);

	new_key = prev_key;
	if (key_down_status && prev_key == EDC_KEY_NONE) {      /* H : down */
		if (!earphone_get_adc(edc_data, EDC_ADC_JMIC_SW_ADC, false, &adc_phys)) {
			adc_phys = adc_phys / 1000;
			if (adc_phys >= EDC_INVALID_ADC_PHYS_MV)
			{
				pr_err("%s: Error ADC values , adc_phys = %d[mV]\n", __func__, adc_phys);
			}
			if (edc_data->test) {
				earphone_get_adc_raw(edc_data, EDC_ADC_XHEADSET_DET, &edc_data->key_press_detect.xheadset_det_raw);
				earphone_get_adc_raw(edc_data, EDC_ADC_JMIC_SW_ADC, &edc_data->key_press_detect.jmic_sw_adc_raw);
			}
		} else {
			return EDC_KEY_NONE;
		}
		for (i=0; i<edc_key_info->edc_key_num; i++) {
			if ((edc_key_info->edc_key_th[i].adc_min <= adc_phys) &&
			    (adc_phys < edc_key_info->edc_key_th[i].adc_max)) {
				new_key = edc_key_info->edc_key_th[i].keycode;
				break;
			}
		}
	} else if (!key_down_status && prev_key != EDC_KEY_NONE) { /* L : Up */
		new_key = EDC_KEY_NONE;
	}

	return new_key;
}

void earphone_key_mutex_lock(struct edc_data *edc_data)
{
	mutex_lock(&edc_data->edc_key_info->jmic_mutex);
}

void earphone_key_mutex_unlock(struct edc_data *edc_data)
{
	mutex_unlock(&edc_data->edc_key_info->jmic_mutex);
}

static irqreturn_t earphone_key_irq_thread(int irq, void *data)
{
	struct edc_data *edc_data = data;
	int gpio_val = earphone_read_key_gpio_value(edc_data->edc_key_info);

	if (-1 == gpio_val) {
		pr_info("%s: read key gpio value error,do not check key status.\n", __func__);
	} else {
		queue_delayed_work(edc_data->edc_key_info->key_wq, &edc_data->edc_key_info->check_key_work, 0);
	}

	return IRQ_HANDLED;
}

static void earphone_check_key_work(struct work_struct *work)
{
	int new_key, key_down_status;
	struct delayed_work *dwork = container_of(work, struct delayed_work, work);
	struct edc_key_info *edc_key_info  = container_of(dwork, struct edc_key_info, check_key_work);
	struct edc_data *edc_data  = edc_key_info->edc_data_ptr;
	bool cancel_wq2 = false;

	earphone_key_mutex_lock(edc_data);
	if (earphone_get_type_status(edc_data) != EDC_DET_TYPE_EARPHONE_MIC) {
		pr_info("%s: earphone is not EDC_DET_TYPE_EARPHONE_MIC\n", __func__);
		new_key = EDC_KEY_NONE;
		goto check_key_end;
	}

	if(edc_data->ignore_key_flag){
		pr_info("%s: ignore_key flag is active \n", __func__);
		new_key = EDC_KEY_NONE;
		goto check_key_end;
	}
	key_down_status = earphone_read_key_gpio_value(edc_key_info);
	if (-1 == key_down_status) {
		new_key = EDC_KEY_NONE;
		goto check_key_end;
	}
	EDC_WAIT_MS(4);
	new_key = earphone_key_check(edc_data, key_down_status);

check_key_end:
	cancel_wq2 = earphone_key_set_down_keycode(edc_data, new_key);
	earphone_key_mutex_unlock(edc_data);
	if (cancel_wq2) {
		pr_info("%s: cancel check_key_work2\n", __func__);
		cancel_delayed_work_sync(&edc_data->edc_key_info->check_key_work2);
	}

	return;
}

/* check key up event only */
static void earphone_check_key_work2(struct work_struct *work)
{
	int key_down_status;
	struct delayed_work *dwork = container_of(work, struct delayed_work, work);
	struct edc_key_info *edc_key_info  = container_of(dwork, struct edc_key_info, check_key_work2);
	struct edc_data *edc_data  = edc_key_info->edc_data_ptr;
	int prev_key = earphone_key_get_down_keycode(edc_data);

	earphone_key_mutex_lock(edc_data);
	key_down_status = earphone_read_key_gpio_value(edc_key_info);
	if (0 == key_down_status && prev_key != EDC_KEY_NONE) {
		pr_warn("%s: switch key %d up (after voice call)\n", __func__, prev_key);
		earphone_key_up_report(edc_data, prev_key);
		edc_key_info->down_key = EDC_KEY_NONE;
	} else {
		if (edc_data->earphone_detect) {
			queue_delayed_work(edc_data->edc_key_info->key_wq2, &edc_data->edc_key_info->check_key_work2, msecs_to_jiffies(500));
		}
	}
	earphone_key_mutex_unlock(edc_data);

	return;
}

int earphone_key_check_start(struct edc_data *edc_data, bool start)
{
	if (start) {
		if (!edc_data->edc_key_info->is_irq_enable) {
			edc_data->edc_key_info->is_irq_enable = true;
			/* start check switch */
			enable_irq(edc_data->edc_key_info->key_gpio_irq);
			pr_debug("%s: key_gpio_irq : enable\n", __func__);
		}
	} else {
		if (edc_data->edc_key_info->is_irq_enable) {
			edc_data->edc_key_info->is_irq_enable = false;
			/* stop check switch */
			disable_irq_nosync(edc_data->edc_key_info->key_gpio_irq);
			pr_debug("%s: key_gpio_irq : disable\n", __func__);
			earphone_key_set_down_keycode(edc_data, EDC_KEY_NONE);
		}
	}
	return 0;
}

static int earphone_key_parse_dt(struct edc_data *edc_data)
{
	int i = 0, ret;
	u32 min, max;
	struct edc_key_info *edc_key_info = edc_data->edc_key_info;

	edc_key_info->edc_key_num = 0;
	ret  = of_property_read_u32(edc_data->dev_np, "fj-edc,headsethock_min", &min);
	ret |= of_property_read_u32(edc_data->dev_np, "fj-edc,headsethock_max", &max);
	if (ret == 0) {
		edc_key_info->edc_key_th[i].keycode = KEY_MEDIA;
		edc_key_info->edc_key_th[i].adc_min = min;
		edc_key_info->edc_key_th[i].adc_max = max;
		pr_info("%s: KEYCODE_HEADSETHOOK idx = %d: min = %d[mV], max = %d[mV]\n", __func__, i, min, max);
		i++;
	}
	ret  = of_property_read_u32(edc_data->dev_np, "fj-edc,voiceassist_min", &min);
	ret |= of_property_read_u32(edc_data->dev_np, "fj-edc,voiceassist_max", &max);
	if (ret == 0) {
		edc_key_info->edc_key_th[i].keycode = KEY_VOICECOMMAND;
		edc_key_info->edc_key_th[i].adc_min = min;
		edc_key_info->edc_key_th[i].adc_max = max;
		pr_info("%s: KEYCODE_VOICE_ASSIST idx = %d: min = %d[mV], max = %d[mV]\n", __func__, i, min, max);
		i++;
	}
	ret  = of_property_read_u32(edc_data->dev_np, "fj-edc,volumeup_min", &min);
	ret |= of_property_read_u32(edc_data->dev_np, "fj-edc,volumeup_max", &max);
	if (ret == 0) {
		edc_key_info->edc_key_th[i].keycode = KEY_VOLUMEUP;
		edc_key_info->edc_key_th[i].adc_min = min;
		edc_key_info->edc_key_th[i].adc_max = max;
		pr_info("%s: KEYCODE_VOLUME_UP   idx = %d: min = %d[mV], max = %d[mV]\n", __func__, i, min, max);
		i++;
	}
	ret  = of_property_read_u32(edc_data->dev_np, "fj-edc,volumedown_min", &min);
	ret |= of_property_read_u32(edc_data->dev_np, "fj-edc,volumedown_max", &max);
	if (ret == 0) {
		edc_key_info->edc_key_th[i].keycode = KEY_VOLUMEDOWN;
		edc_key_info->edc_key_th[i].adc_min = min;
		edc_key_info->edc_key_th[i].adc_max = max;
		pr_info("%s: KEYCODE_VOLUME_DOWN idx = %d: min = %d[mV], max = %d[mV]\n", __func__, i, min, max);
		i++;
	}
	edc_key_info->edc_key_num = i;

	return 0;
}

static int earphone_key_gpio_init(struct edc_data *edc_data)
{
	int ret;
	struct edc_key_info *edc_key_info = edc_data->edc_key_info;

	/* detect gpiod init */
	edc_key_info->key_gpio = devm_gpiod_get_optional(&edc_data->pdev->dev, "fj-edc,edc_key", GPIOD_IN);
	if (IS_ERR_OR_NULL(edc_key_info->key_gpio)) {
		pr_err("%s: devm_gpiod_get_optional failed.\n", __func__);
		edc_key_info->key_gpio = NULL;
		return -1;
	}

	/* earphone key irq init */
	edc_key_info->key_gpio_irq = gpiod_to_irq(edc_key_info->key_gpio);
	irq_set_status_flags(edc_key_info->key_gpio_irq, IRQ_NOAUTOEN);
	ret = request_threaded_irq(edc_key_info->key_gpio_irq, NULL,
	                           earphone_key_irq_thread,
	                           (IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING | IRQF_NO_SUSPEND | IRQF_ONESHOT),
	                           "earphone_switch", edc_data);
	if (ret) {
		pr_err("%s: request_threaded_irq failed.\n", __func__);
		edc_key_info->key_gpio_irq = -1;
		return -1;
	}

	return 0;
}

int earphone_key_deinit(struct edc_data *edc_data)
{
	if (!edc_data->edc_key_info) {
		return 0;
	}
	wakeup_source_unregister(edc_data->edc_key_info->notify_wake_src);
	mutex_destroy(&edc_data->edc_key_info->jmic_mutex);

	cancel_delayed_work_sync(&edc_data->edc_key_info->check_key_work);
	cancel_delayed_work_sync(&edc_data->edc_key_info->check_key_work2);
	cancel_delayed_work_sync(&edc_data->edc_key_info->unmute_work);
	if (edc_data->edc_key_info->key_wq) {
		destroy_workqueue(edc_data->edc_key_info->key_wq);
	}
	if (edc_data->edc_key_info->key_wq2) {
		destroy_workqueue(edc_data->edc_key_info->key_wq2);
	}
	if (edc_data->edc_key_info->unmute_wq) {
		destroy_workqueue(edc_data->edc_key_info->unmute_wq);
	}
	edc_data->edc_key_info->edc_data_ptr = NULL;

	mutex_destroy(&edc_data->edc_key_info->mute_mutex);

	if (0 <= edc_data->edc_key_info->key_gpio_irq) {
		free_irq(edc_data->edc_key_info->key_gpio_irq, edc_data);
	}
	if (0 <= edc_data->edc_key_info->key_gpio) {
		gpiod_put(edc_data->edc_key_info->key_gpio);
	}

	if (edc_data->edc_key_info->indev) {
		input_unregister_device(edc_data->edc_key_info->indev);
	}
	kfree(edc_data->edc_key_info);
	edc_data->edc_key_info = NULL;
	return 0;
}

int earphone_key_init(struct edc_data *edc_data)
{
	int i, ret;
	struct edc_key_info *edc_key_info = NULL;

	edc_key_info = kzalloc(sizeof(struct edc_key_info),  GFP_KERNEL);
	edc_data->edc_key_info = edc_key_info;
	if (!edc_data->edc_key_info) {
		pr_err("%s: allocate edc_key_info failed.\n", __func__);
		edc_data->edc_key_info = NULL;
		return -1;
	}

	edc_data->edc_key_info->down_key     = EDC_KEY_NONE;
	edc_data->edc_key_info->key_gpio     = NULL;
	edc_data->edc_key_info->key_gpio_irq = -1;
	edc_data->edc_key_info->notify_wake_src = wakeup_source_register(&edc_data->pdev->dev, "edc_key_ntfy_wl");
	mutex_init(&edc_data->edc_key_info->jmic_mutex);
	mutex_init(&edc_data->edc_key_info->mute_mutex);

	edc_key_info->edc_data_ptr = (void*)edc_data;

	/* earphone switch keys check workqueue */
	edc_key_info->key_wq = alloc_workqueue("key_wq", WQ_UNBOUND | WQ_MEM_RECLAIM, 0);
	if (!edc_key_info->key_wq) {
		pr_err("%s: allocate key_wq failed.\n", __func__);
		goto error;
	}
	INIT_DELAYED_WORK(&edc_key_info->check_key_work, earphone_check_key_work);
	/* earphone switch keys check workqueue2,a simple work to check key up event */
	edc_key_info->key_wq2 = alloc_workqueue("key_wq2", WQ_UNBOUND | WQ_MEM_RECLAIM, 0);
	if (!edc_key_info->key_wq2) {
		pr_err("%s: allocate key_wq2 failed.\n", __func__);
		goto error;
	}
	INIT_DELAYED_WORK(&edc_key_info->check_key_work2, earphone_check_key_work2);
	/* earphone switch keys unmute workqueue */
	edc_key_info->unmute_wq = alloc_workqueue("unmute_wq", WQ_UNBOUND | WQ_MEM_RECLAIM, 0);
	if (!edc_key_info->unmute_wq) {
		pr_err("%s: allocate unmute_wq failed.\n", __func__);
		goto error;
	}
	INIT_DELAYED_WORK(&edc_key_info->unmute_work, earphone_unmute_work);

	ret = earphone_key_parse_dt(edc_data);
	if (ret) {
		pr_err("%s: earphone_key_parse_dt failed.\n", __func__);
		goto error;
	}

	ret = earphone_key_gpio_init(edc_data);
	if (ret) {
		pr_err("%s: earphone_key_gpio_init failed.\n", __func__);
		goto error;
	}

	/* register inputdev */
	edc_key_info->indev = input_allocate_device();
	if (!edc_key_info->indev) {
		pr_err("%s: input_allocate_device failed.\n", __func__);
		goto error;
	}
	set_bit(EV_KEY, edc_key_info->indev->evbit);
	edc_key_info->indev->name        = edc_data->name;
	edc_key_info->indev->id.vendor   = 0x0001;
	edc_key_info->indev->id.product  = 1;
	edc_key_info->indev->id.version  = 1;
	for (i=0; i<edc_key_info->edc_key_num; i++) {
		input_set_capability(edc_key_info->indev, EV_KEY, edc_key_info->edc_key_th[i].keycode);
	}
	ret = input_register_device(edc_key_info->indev);
	if (ret) {
		input_free_device(edc_key_info->indev);
		edc_key_info->indev = NULL;
		pr_err("%s: input_register_device failed.\n", __func__);
		goto error;
	}

	return 0;

error:
	earphone_key_deinit(edc_data);
	return -1;
}
